// Copyright (c) 2016-2021 Fabian Schuiki
use std::sync::atomic::{AtomicUsize, Ordering};

/// A positive, small ID assigned to nodes in the AST and derived data
/// structures. Used as a lightweight way to refer to individual nodes, e.g.
/// during symbol table construction and name resolution.
#[derive(Clone, Copy, PartialEq, PartialOrd, Eq, Ord, Hash)]
pub struct NodeId(u32);

impl NodeId {
    pub fn new(x: usize) -> NodeId {
        use std::u32;
        assert!(x < (u32::MAX as usize));
        NodeId(x as u32)
    }

    /// Allocate a new unused ID. The IDs generated by this function are
    /// monotonically increasing.
    pub fn alloc() -> NodeId {
        static NEXT_ID: AtomicUsize = AtomicUsize::new(0);
        NodeId::new(NEXT_ID.fetch_add(1, Ordering::SeqCst))
    }

    pub fn from_u32(x: u32) -> NodeId {
        NodeId(x)
    }

    pub fn as_usize(&self) -> usize {
        self.0 as usize
    }

    pub fn as_u32(&self) -> u32 {
        self.0
    }
}

impl std::fmt::Debug for NodeId {
    fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
        write!(f, "n{}", self.0)
    }
}
