use std::collections::{btree_map, BTreeMap};

use super::{database::Database, Error, Result};
use crate::proto::*;

use tokio::sync::Mutex;

pub struct Universe {
    inner: Mutex<Inner>,
}

struct Inner {
    databases: BTreeMap<u64, Database>,
    database_specs: BTreeMap<String, DatabaseSpec>,
    next_database_id: u64,
}

impl Universe {
    pub fn new() -> Self {
        let inner = Inner {
            databases: BTreeMap::new(),
            database_specs: BTreeMap::new(),
            next_database_id: 0,
        };
        Self {
            inner: Mutex::new(inner),
        }
    }

    pub async fn database(&self, id: u64) -> Option<Database> {
        let inner = self.inner.lock().await;
        inner.databases.get(&id).cloned()
    }

    pub async fn create_database(&self, spec: DatabaseSpec) -> Result<DatabaseDesc> {
        let mut inner = self.inner.lock().await;
        match inner.database_specs.entry(spec.name.clone()) {
            btree_map::Entry::Vacant(ent) => {
                ent.insert(spec.clone());
                let id = inner.next_database_id;
                inner.next_database_id += 1;
                let db = Database::new(id, spec);
                inner.databases.insert(id, db.clone());
                Ok(db.desc().await)
            }
            btree_map::Entry::Occupied(ent) => {
                Err(Error::AlreadyExists(format!("database '{}'", ent.key())))
            }
        }
    }
}
