use std::net::SocketAddr;

use crate::proto::*;

mod collection;
mod database;
mod error;
mod universe;

pub use self::error::{Error, Result};

use self::universe::Universe;

pub struct Server {
    uv: Universe,
}

type TonicResult<T> = std::result::Result<T, tonic::Status>;

#[tonic::async_trait]
impl monsta_server::Monsta for Server {
    async fn call(
        &self,
        request: tonic::Request<Request>,
    ) -> TonicResult<tonic::Response<Response>> {
        let request = request.into_inner();
        if let Some(req) = request.universe {
            let res = self.handle_universe(req).await?;
            Ok(tonic::Response::new(Response {
                universe: Some(res),
                ..Default::default()
            }))
        } else if let Some(req) = request.database {
            let res = self.handle_database(req).await?;
            Ok(tonic::Response::new(Response {
                database: Some(res),
                ..Default::default()
            }))
        } else if let Some(co) = request.collection {
            todo!()
        } else {
            Err(Error::InvalidArgument("missing request".to_owned()).into())
        }
    }
}

impl Server {
    pub fn new() -> Self {
        Self {
            uv: Universe::new(),
        }
    }

    pub async fn serve(self, addr: SocketAddr) -> std::result::Result<(), tonic::transport::Error> {
        let service = monsta_server::MonstaServer::new(self);
        tonic::transport::Server::builder()
            .add_service(service)
            .serve(addr)
            .await
    }

    async fn handle_universe(&self, request: UniverseRequest) -> Result<UniverseResponse> {
        let response = match request
            .request
            .ok_or_else(|| Error::InvalidArgument("missing universe request".to_owned()))?
        {
            universe_request::Request::CreateDatabase(r) => {
                let spec = r.spec.ok_or_else(|| {
                    Error::InvalidArgument("missing database specification".to_owned())
                })?;
                let desc = self.uv.create_database(spec).await?;
                universe_response::Response::CreateDatabase(CreateDatabaseResponse {
                    desc: Some(desc),
                })
            }
        };
        Ok(UniverseResponse {
            response: Some(response),
        })
    }

    async fn handle_database(&self, request: DatabaseRequest) -> Result<DatabaseResponse> {
        let db = self
            .uv
            .database(request.database_id)
            .await
            .ok_or_else(|| Error::NotFound(format!("database '{}'", request.database_id)))?;
        let response = match request
            .request
            .ok_or_else(|| Error::InvalidArgument("missing database request".to_owned()))?
        {
            database_request::Request::CreateCollection(r) => {
                let spec = r.spec.ok_or_else(|| {
                    Error::InvalidArgument("missing collection specification".to_owned())
                })?;
                let desc = db.create_collection(spec).await?;
                database_response::Response::CreateCollection(CreateCollectionResponse {
                    desc: Some(desc),
                })
            }
        };
        Ok(DatabaseResponse {
            response: Some(response),
        })
    }
}
