use std::{
    collections::{btree_map, BTreeMap},
    sync::Arc,
};

use super::{collection::Collection, Error, Result};
use crate::proto::*;

use tokio::sync::Mutex;

#[derive(Clone)]
pub struct Database {
    inner: Arc<Mutex<Inner>>,
}

struct Inner {
    desc: DatabaseDesc,
    collections: BTreeMap<u64, Collection>,
    collection_specs: BTreeMap<String, CollectionSpec>,
    next_collection_id: u64,
}

impl Database {
    pub fn new(id: u64, spec: DatabaseSpec) -> Self {
        let desc = DatabaseDesc {
            id,
            name: spec.name,
        };
        let inner = Inner {
            desc,
            collections: BTreeMap::new(),
            collection_specs: BTreeMap::new(),
            next_collection_id: 0,
        };
        Self {
            inner: Arc::new(Mutex::new(inner)),
        }
    }

    pub async fn desc(&self) -> DatabaseDesc {
        let inner = self.inner.lock().await;
        inner.desc.clone()
    }

    pub async fn create_collection(&self, spec: CollectionSpec) -> Result<CollectionDesc> {
        let mut inner = self.inner.lock().await;
        match inner.collection_specs.entry(spec.name.clone()) {
            btree_map::Entry::Vacant(ent) => {
                ent.insert(spec.clone());
                let id = inner.next_collection_id;
                inner.next_collection_id += 1;
                let co = Collection::new(id, spec);
                inner.collections.insert(id, co.clone());
                Ok(co.desc().await)
            }
            btree_map::Entry::Occupied(ent) => {
                Err(Error::AlreadyExists(format!("database '{}'", ent.key())))
            }
        }
    }
}
