use super::{Database, Error, Result};
use crate::proto::{monsta_client::MonstaClient, *};

use tonic::transport::Channel;

pub struct Universe {
    client: MonstaClient<Channel>,
}

impl Universe {
    pub async fn connect(addr: String) -> Result<Self> {
        let url = format!("http://{}", addr);
        let client = MonstaClient::connect(url).await.map_err(Error::unknown)?;
        Ok(Self { client })
    }

    pub async fn create_database(&self, spec: DatabaseSpec) -> Result<Database> {
        let req =
            universe_request::Request::CreateDatabase(CreateDatabaseRequest { spec: Some(spec) });
        let req = make_request(req);
        let mut client = self.client.clone();
        let res = client.call(req).await?;
        let res = res
            .into_inner()
            .universe
            .ok_or_else(|| Error::InvalidResponse("missing universe response".to_owned()))?;
        let desc = match res
            .response
            .ok_or_else(|| Error::InvalidResponse("missing universe response".to_owned()))?
        {
            universe_response::Response::CreateDatabase(r) => r
                .desc
                .ok_or_else(|| Error::InvalidResponse("missing database description".to_owned()))?,
        };
        Ok(Database::new(desc, self.client.clone()))
    }
}

fn make_request(request: universe_request::Request) -> Request {
    Request {
        universe: Some(UniverseRequest {
            request: Some(request),
        }),
        ..Default::default()
    }
}
