use thiserror::Error;
use tonic::{Code, Status};

#[derive(Error, Debug)]
pub enum Error {
    #[error("{0} is not found")]
    NotFound(String),
    #[error("{0} already exists")]
    AlreadyExists(String),
    #[error("invalid argument: {0}")]
    InvalidArgument(String),
    #[error("invalid response: {0}")]
    InvalidResponse(String),
    #[error(transparent)]
    Unknown(Box<dyn std::error::Error + Send + Sync>),
}

impl Error {
    pub fn unknown(err: impl std::error::Error + Send + Sync + 'static) -> Self {
        Self::Unknown(Box::new(err))
    }
}

impl From<Status> for Error {
    fn from(s: Status) -> Self {
        match s.code() {
            Code::NotFound => Error::NotFound(s.message().to_owned()),
            Code::AlreadyExists => Error::AlreadyExists(s.message().to_owned()),
            Code::InvalidArgument => Error::InvalidArgument(s.message().to_owned()),
            _ => Error::Unknown(Box::new(s)),
        }
    }
}

pub type Result<T> = std::result::Result<T, Error>;
