use super::{Collection, Error, Result};
use crate::proto::*;

use tonic::transport::Channel;

type MonstaClient = monsta_client::MonstaClient<Channel>;

pub struct Database {
    desc: DatabaseDesc,
    client: monsta_client::MonstaClient<Channel>,
}

impl Database {
    pub fn new(desc: DatabaseDesc, client: MonstaClient) -> Self {
        Self { desc, client }
    }

    pub fn desc(&self) -> &DatabaseDesc {
        &self.desc
    }

    pub async fn create_collection(&self, spec: CollectionSpec) -> Result<Collection> {
        let req = database_request::Request::CreateCollection(CreateCollectionRequest {
            spec: Some(spec),
        });
        let req = make_request(self.desc.id, req);
        let mut client = self.client.clone();
        let res = client.call(req).await?;
        let res = res
            .into_inner()
            .database
            .ok_or_else(|| Error::InvalidResponse("missing database response".to_owned()))?;
        let desc = match res
            .response
            .ok_or_else(|| Error::InvalidResponse("missing database response".to_owned()))?
        {
            database_response::Response::CreateCollection(r) => r.desc.ok_or_else(|| {
                Error::InvalidResponse("missing collection description".to_owned())
            })?,
        };
        Ok(Collection::new(desc, self.client.clone()))
    }
}

fn make_request(database_id: u64, request: database_request::Request) -> Request {
    Request {
        database: Some(DatabaseRequest {
            request: Some(request),
            database_id,
        }),
        ..Default::default()
    }
}
