use monsta_proto::*;

use crate::{client::Client, metadata, Database, Error, Result};

pub struct Universe {
    client: Client,
}

impl Universe {
    pub async fn connect(url: String) -> Result<Self> {
        let client = Client::connect(url).await?;
        Ok(Self { client })
    }

    pub async fn create_database(
        &self,
        spec: impl Into<metadata::DatabaseSpec>,
    ) -> Result<Database> {
        let req = DatabaseRequest {
            create_database: Some(CreateDatabaseRequest {
                spec: Some(spec.into().into()),
            }),
            ..Default::default()
        };
        let res = self.client.database_call(req).await?;
        let desc = res
            .create_database
            .and_then(|x| x.desc)
            .ok_or_else(|| Error::InvalidResponse)?;
        Ok(Database::new(self.client.clone(), desc.id))
    }

    pub async fn lookup_database(&self, name: impl Into<String>) -> Result<Database> {
        let desc = self.describe_database(name).await?;
        Ok(Database::new(self.client.clone(), desc.id))
    }

    pub async fn describe_database(&self, name: impl Into<String>) -> Result<DatabaseDesc> {
        let req = DatabaseRequest {
            describe_database: Some(DescribeDatabaseRequest {
                name: name.into(),
                ..Default::default()
            }),
            ..Default::default()
        };
        let res = self.client.database_call(req).await?;
        let desc = res
            .describe_database
            .and_then(|x| x.desc)
            .ok_or_else(|| Error::InvalidResponse)?;
        Ok(desc)
    }
}
