use monsta_proto::*;

use crate::{client::Client, metadata, Collection, Error, Result};

pub struct Database {
    client: Client,
    database_id: u64,
}

impl Database {
    pub fn new(client: Client, database_id: u64) -> Self {
        Self {
            client,
            database_id,
        }
    }

    pub async fn desc(&self) -> Result<DatabaseDesc> {
        let req = DatabaseRequest {
            describe_database: Some(DescribeDatabaseRequest {
                id: self.database_id,
                ..Default::default()
            }),
            ..Default::default()
        };
        let res = self.client.database_call(req).await?;
        let desc = res
            .describe_database
            .and_then(|x| x.desc)
            .ok_or_else(|| Error::InvalidResponse)?;
        Ok(desc)
    }

    pub async fn create_collection(
        &self,
        spec: impl Into<metadata::CollectionSpec>,
    ) -> Result<Collection> {
        let req = CollectionRequest {
            database_id: self.database_id,
            create_collection: Some(CreateCollectionRequest {
                spec: Some(spec.into().into()),
            }),
            ..Default::default()
        };
        let res = self.client.collection_call(req).await?;
        let desc = res
            .create_collection
            .and_then(|x| x.desc)
            .ok_or_else(|| Error::InvalidResponse)?;
        Ok(Collection::new(
            self.client.clone(),
            desc.database_id,
            desc.id,
        ))
    }

    pub async fn lookup_collection(&self, name: impl Into<String>) -> Result<Collection> {
        let desc = self.describe_collection(name).await?;
        Ok(Collection::new(
            self.client.clone(),
            desc.database_id,
            desc.id,
        ))
    }

    pub async fn describe_collection(&self, name: impl Into<String>) -> Result<CollectionDesc> {
        let req = CollectionRequest {
            database_id: self.database_id,
            describe_collection: Some(DescribeCollectionRequest {
                name: name.into(),
                ..Default::default()
            }),
            ..Default::default()
        };
        let res = self.client.collection_call(req).await?;
        let desc = res
            .describe_collection
            .and_then(|x| x.desc)
            .ok_or_else(|| Error::InvalidResponse)?;
        Ok(desc)
    }
}
