//! Contains options for ChangeStreams.
use serde::{Deserialize, Serialize};
use serde_with::skip_serializing_none;
use std::time::Duration;
use typed_builder::TypedBuilder;

use crate::{
    bson::Timestamp,
    change_stream::event::ResumeToken,
    collation::Collation,
    concern::ReadConcern,
    options::AggregateOptions,
    selection_criteria::SelectionCriteria,
};

/// These are the valid options that can be passed to the `watch` method for creating a
/// [`ChangeStream`](crate::change_stream::ChangeStream).
#[skip_serializing_none]
#[derive(Clone, Debug, Default, Deserialize, Serialize, TypedBuilder)]
#[serde(rename_all = "camelCase")]
#[non_exhaustive]
pub struct ChangeStreamOptions {
    #[rustfmt::skip]
    /// When set to [`FullDocumentType::UpdateLookup`], the
    /// [`ChangeStreamEvent::full_document`](crate::change_stream::event::ChangeStreamEvent::full_document)
    /// field  will be populated with a copy of the entire document that was
    /// updated from some time after the change occurred when an "update" event occurs. By
    /// default, the `full_document` field will be empty for updates.
    #[builder(default)]
    pub full_document: Option<FullDocumentType>,

    /// Specifies the logical starting point for the new change stream. Note that if a watched
    /// collection is dropped and recreated or newly renamed, `start_after` should be set instead.
    /// `resume_after` and `start_after` cannot be set simultaneously.
    ///
    /// For more information on resuming a change stream see the documentation [here](https://docs.mongodb.com/manual/changeStreams/#change-stream-resume-after)
    #[builder(default)]
    pub resume_after: Option<ResumeToken>,

    /// The change stream will only provide changes that occurred at or after the specified
    /// timestamp. Any command run against the server will return an operation time that can be
    /// used here.
    #[builder(default)]
    pub start_at_operation_time: Option<Timestamp>,

    /// Takes a resume token and starts a new change stream returning the first notification after
    /// the token. This will allow users to watch collections that have been dropped and
    /// recreated or newly renamed collections without missing any notifications.
    ///
    /// This feature is only available on MongoDB 4.2+.
    ///
    /// See the documentation [here](https://docs.mongodb.com/master/changeStreams/#change-stream-start-after) for more
    /// information.
    #[builder(default)]
    pub start_after: Option<ResumeToken>,

    /// If `true`, the change stream will monitor all changes for the given cluster.
    #[builder(default, setter(skip))]
    pub(crate) all_changes_for_cluster: Option<bool>,

    /// The maximum amount of time for the server to wait on new documents to satisfy a change
    /// stream query.
    #[builder(default)]
    #[serde(skip_serializing)]
    pub max_await_time: Option<Duration>,

    /// The number of documents to return per batch.
    #[builder(default)]
    #[serde(skip_serializing)]
    pub batch_size: Option<u32>,

    /// Specifies a collation.
    #[builder(default)]
    #[serde(skip_serializing)]
    pub collation: Option<Collation>,

    /// The read concern to use for the operation.
    ///
    /// If none is specified, the read concern defined on the object executing this operation will
    /// be used.
    #[builder(default)]
    #[serde(skip_serializing)]
    pub read_concern: Option<ReadConcern>,

    /// The criteria used to select a server for this operation.
    ///
    /// If none is specified, the selection criteria defined on the object executing this operation
    /// will be used.
    #[builder(default)]
    #[serde(skip_serializing)]
    pub selection_criteria: Option<SelectionCriteria>,
}

impl ChangeStreamOptions {
    pub(crate) fn aggregate_options(&self) -> AggregateOptions {
        AggregateOptions::builder()
            .batch_size(self.batch_size)
            .collation(self.collation.clone())
            .max_await_time(self.max_await_time)
            .read_concern(self.read_concern.clone())
            .selection_criteria(self.selection_criteria.clone())
            .build()
    }
}

#[rustfmt::skip]
/// Describes the modes for configuring the
/// [`ChangeStreamEvent::full_document`](crate::change_stream::event::ChangeStreamEvent::full_document)
/// field.
#[derive(Clone, Debug, Deserialize, Serialize)]
#[non_exhaustive]
pub enum FullDocumentType {
    #[rustfmt::skip]
    /// The
    /// [`ChangeStreamEvent::full_document`](crate::change_stream::event::ChangeStreamEvent::full_document)
    /// field will be populated with a copy of the entire document that was updated.
    #[serde(rename = "updateLookup")]
    UpdateLookup,

    /// User-defined other types for forward compatibility.
    Other(String),
}
