use std::{collections::HashMap, sync::Arc, time::Duration};

use serde::Deserialize;

use crate::{
    bson::{doc, DateTime},
    is_master::{IsMasterCommandResponse, IsMasterReply, LastWrite},
    options::ServerAddress,
    sdam::{
        description::topology::{test::f64_ms_as_duration, TopologyType},
        ServerDescription,
        ServerType,
        TopologyDescription,
    },
    selection_criteria::{SelectionCriteria, TagSet},
};

mod in_window;
mod logic;

#[derive(Debug, Deserialize)]
struct TestTopologyDescription {
    #[serde(rename = "type")]
    topology_type: TopologyType,
    servers: Vec<TestServerDescription>,
}

impl TestTopologyDescription {
    fn into_topology_description(
        self,
        heartbeat_frequency: Option<Duration>,
    ) -> TopologyDescription {
        let servers: HashMap<ServerAddress, ServerDescription> = self
            .servers
            .into_iter()
            .filter_map(|sd| {
                sd.into_server_description()
                    .map(|sd| (sd.address.clone(), sd))
            })
            .collect();

        TopologyDescription {
            single_seed: servers.len() == 1,
            topology_type: self.topology_type,
            set_name: None,
            max_set_version: None,
            max_election_id: None,
            compatibility_error: None,
            session_support_status: Default::default(),
            transaction_support_status: Default::default(),
            cluster_time: None,
            local_threshold: None,
            heartbeat_freq: heartbeat_frequency,
            servers,
        }
    }
}

#[derive(Debug, Deserialize)]
#[serde(rename_all = "camelCase", deny_unknown_fields)]
struct TestServerDescription {
    address: String,
    #[serde(rename = "avg_rtt_ms")]
    avg_rtt_ms: Option<f64>,
    #[serde(rename = "type")]
    server_type: TestServerType,
    tags: Option<TagSet>,
    last_update_time: Option<i32>,
    last_write: Option<LastWriteDate>,
    // We don't need to use this field, but it needs to be included during deserialization so that
    // we can use the deny_unknown_fields tag.
    _max_wire_version: Option<i32>,
}

impl TestServerDescription {
    fn into_server_description(self) -> Option<ServerDescription> {
        let server_type = match self.server_type.into_server_type() {
            Some(server_type) => server_type,
            None => return None,
        };

        let server_address = ServerAddress::parse(self.address).ok()?;
        let tags = self.tags;
        let last_write = self.last_write;
        let avg_rtt_ms = self.avg_rtt_ms;
        let reply = is_master_response_from_server_type(server_type).map(|mut command_response| {
            command_response.tags = tags;
            command_response.last_write = last_write.map(|last_write| LastWrite {
                last_write_date: DateTime::from_millis(last_write.last_write_date),
            });
            Ok(IsMasterReply {
                server_address: server_address.clone(),
                command_response,
                round_trip_time: avg_rtt_ms
                    .map(f64_ms_as_duration)
                    .unwrap_or_else(|| Duration::from_millis(1234)),
                cluster_time: None,
            })
        });

        let mut server_desc = ServerDescription::new(server_address, reply);
        server_desc.last_update_time = self
            .last_update_time
            .map(|i| DateTime::from_millis(i.into()));

        Some(server_desc)
    }
}

#[derive(Debug, Deserialize)]
#[serde(rename_all = "camelCase")]
struct LastWriteDate {
    last_write_date: i64,
}

#[derive(Clone, Copy, Debug, Deserialize)]
enum TestServerType {
    Standalone,
    Mongos,
    #[serde(rename = "RSPrimary")]
    RsPrimary,
    #[serde(rename = "RSSecondary")]
    RsSecondary,
    #[serde(rename = "RSArbiter")]
    RsArbiter,
    #[serde(rename = "RSOther")]
    RsOther,
    #[serde(rename = "RSGhost")]
    RsGhost,
    LoadBalancer,
    Unknown,
    PossiblePrimary,
}

impl TestServerType {
    fn into_server_type(self) -> Option<ServerType> {
        match self {
            TestServerType::Standalone => Some(ServerType::Standalone),
            TestServerType::Mongos => Some(ServerType::Mongos),
            TestServerType::RsPrimary => Some(ServerType::RsPrimary),
            TestServerType::RsSecondary => Some(ServerType::RsSecondary),
            TestServerType::RsArbiter => Some(ServerType::RsArbiter),
            TestServerType::RsOther => Some(ServerType::RsOther),
            TestServerType::RsGhost => Some(ServerType::RsGhost),
            TestServerType::LoadBalancer => Some(ServerType::LoadBalancer),
            TestServerType::Unknown => Some(ServerType::Unknown),
            TestServerType::PossiblePrimary => None,
        }
    }
}

fn is_master_response_from_server_type(server_type: ServerType) -> Option<IsMasterCommandResponse> {
    let mut response = IsMasterCommandResponse::default();

    match server_type {
        ServerType::Unknown => {
            return None;
        }
        ServerType::Mongos => {
            response.msg = Some("isdbgrid".into());
        }
        ServerType::RsPrimary => {
            response.set_name = Some("foo".into());
            response.is_writable_primary = Some(true);
        }
        ServerType::RsOther => {
            response.set_name = Some("foo".into());
            response.hidden = Some(true);
        }
        ServerType::RsSecondary => {
            response.set_name = Some("foo".into());
            response.secondary = Some(true);
        }
        ServerType::RsArbiter => {
            response.set_name = Some("foo".into());
            response.arbiter_only = Some(true);
        }
        ServerType::RsGhost => {
            response.is_replica_set = Some(true);
        }
        ServerType::Standalone | ServerType::LoadBalancer => {}
    };

    Some(response)
}

#[test]
fn predicate_omits_unavailable() {
    let criteria = SelectionCriteria::Predicate(Arc::new(|si| {
        !matches!(si.server_type(), ServerType::RsPrimary)
    }));

    let desc = TestTopologyDescription {
        topology_type: TopologyType::ReplicaSetWithPrimary,
        servers: vec![
            TestServerDescription {
                address: "localhost:27017".to_string(),
                avg_rtt_ms: Some(12.0),
                server_type: TestServerType::RsPrimary,
                tags: None,
                last_update_time: None,
                last_write: None,
                _max_wire_version: None,
            },
            TestServerDescription {
                address: "localhost:27018".to_string(),
                avg_rtt_ms: Some(12.0),
                server_type: TestServerType::Unknown,
                tags: None,
                last_update_time: None,
                last_write: None,
                _max_wire_version: None,
            },
            TestServerDescription {
                address: "localhost:27019".to_string(),
                avg_rtt_ms: Some(12.0),
                server_type: TestServerType::RsArbiter,
                tags: None,
                last_update_time: None,
                last_write: None,
                _max_wire_version: None,
            },
            TestServerDescription {
                address: "localhost:27020".to_string(),
                avg_rtt_ms: Some(12.0),
                server_type: TestServerType::RsGhost,
                tags: None,
                last_update_time: None,
                last_write: None,
                _max_wire_version: None,
            },
            TestServerDescription {
                address: "localhost:27021".to_string(),
                avg_rtt_ms: Some(12.0),
                server_type: TestServerType::RsOther,
                tags: None,
                last_update_time: None,
                last_write: None,
                _max_wire_version: None,
            },
        ],
    }
    .into_topology_description(None);
    pretty_assertions::assert_eq!(
        desc.suitable_servers_in_latency_window(&criteria).unwrap(),
        Vec::<&ServerDescription>::new()
    );
}
