use std::{sync::Arc, time::Duration};

use derivative::Derivative;
use serde::Deserialize;
use typed_builder::TypedBuilder;

use crate::{
    bson_util,
    client::{auth::Credential, options::ServerApi},
    event::cmap::CmapEventHandler,
    options::{ClientOptions, DriverInfo, StreamAddress, TlsOptions},
};

/// Contains the options for creating a connection pool. While these options are specified at the
/// client-level, `ConnectionPoolOptions` is exposed for the purpose of CMAP event handling.
#[derive(Clone, Default, Deserialize, TypedBuilder, Derivative)]
#[derivative(Debug, PartialEq)]
#[serde(rename_all = "camelCase")]
pub struct ConnectionPoolOptions {
    /// The application name specified by the user. This is sent to the server as part of the
    /// handshake that each connection makes when it's created.
    #[builder(default)]
    pub app_name: Option<String>,

    /// The connect timeout passed to each underlying TcpStream when attemtping to connect to the
    /// server.
    #[builder(default)]
    #[serde(skip)]
    pub connect_timeout: Option<Duration>,

    /// The credential to use for authenticating connections in this pool.
    #[builder(default)]
    #[serde(skip)]
    pub credential: Option<Credential>,

    /// Extra information to append to the driver version in the metadata of the handshake with the
    /// server. This should be used by libraries wrapping the driver, e.g. ODMs.
    #[builder(default)]
    #[serde(skip)]
    pub driver_info: Option<DriverInfo>,

    /// Processes all events generated by the pool.
    #[derivative(Debug = "ignore", PartialEq = "ignore")]
    #[builder(default)]
    #[serde(skip)]
    pub event_handler: Option<Arc<dyn CmapEventHandler>>,

    /// How often the background thread performs its maintenance (e.g. ensure minPoolSize).
    #[cfg(test)]
    #[builder(default)]
    #[serde(skip)]
    pub maintenance_frequency: Option<Duration>,

    /// Connections that have been ready for usage in the pool for longer than `max_idle_time` will
    /// not be used.
    ///
    /// The default is that connections will not be closed due to being idle.
    #[builder(default)]
    #[serde(rename = "maxIdleTimeMS")]
    #[serde(default)]
    #[serde(deserialize_with = "bson_util::deserialize_duration_from_u64_millis")]
    pub max_idle_time: Option<Duration>,

    /// The maximum number of connections that the pool can have at a given time. This includes
    /// connections which are currently checked out of the pool.
    ///
    /// The default is 100.
    #[builder(default)]
    pub max_pool_size: Option<u32>,

    /// The minimum number of connections that the pool can have at a given time. This includes
    /// connections which are currently checked out of the pool. If fewer than `min_pool_size`
    /// connections are in the pool, connections will be added to the pool in the background.
    ///
    /// The default is that no minimum is enforced
    #[builder(default)]
    pub min_pool_size: Option<u32>,

    /// Whether to start the pool as "ready" or not.
    /// For tests only.
    #[cfg(test)]
    #[builder(default)]
    pub ready: Option<bool>,

    /// The declared API version
    ///
    /// The default value is to have no declared API version
    #[builder(skip, default)]
    pub(crate) server_api: Option<ServerApi>,

    /// The options specifying how a TLS connection should be configured. If `tls_options` is
    /// `None`, then TLS will not be used for the connections.
    ///
    /// The default is not to use TLS for connections.
    #[builder(default)]
    #[serde(skip)]
    pub tls_options: Option<TlsOptions>,

    /// Rather than wait indefinitely for a connection to become available, instead return an error
    /// after the given duration.
    ///
    /// The default is to block indefinitely until a connection becomes available.
    #[builder(default)]
    #[serde(rename = "waitQueueTimeoutMS")]
    #[serde(default)]
    #[serde(deserialize_with = "bson_util::deserialize_duration_from_u64_millis")]
    pub wait_queue_timeout: Option<Duration>,
}

impl ConnectionPoolOptions {
    pub(crate) fn from_client_options(options: &ClientOptions) -> Self {
        Self::builder()
            .app_name(options.app_name.clone())
            .connect_timeout(options.connect_timeout)
            .credential(options.credential.clone())
            .driver_info(options.driver_info.clone())
            .event_handler(options.cmap_event_handler.clone())
            .max_idle_time(options.max_idle_time)
            .max_pool_size(options.max_pool_size)
            .min_pool_size(options.min_pool_size)
            .tls_options(options.tls_options())
            .wait_queue_timeout(options.wait_queue_timeout)
            .build()
    }
}

/// Options used for constructing a `Connection`.
#[derive(Derivative)]
#[derivative(Debug)]
#[derive(Clone)]
pub(crate) struct ConnectionOptions {
    pub(crate) connect_timeout: Option<Duration>,

    pub(crate) tls_options: Option<TlsOptions>,

    #[derivative(Debug = "ignore")]
    pub(crate) event_handler: Option<Arc<dyn CmapEventHandler>>,
}

impl From<ConnectionPoolOptions> for ConnectionOptions {
    fn from(pool_options: ConnectionPoolOptions) -> Self {
        Self {
            connect_timeout: pool_options.connect_timeout,
            tls_options: pool_options.tls_options,
            event_handler: pool_options.event_handler,
        }
    }
}

#[derive(Clone, Debug, TypedBuilder)]
pub(crate) struct StreamOptions {
    pub(crate) address: StreamAddress,

    #[builder(default)]
    pub(crate) connect_timeout: Option<Duration>,

    #[builder(default)]
    pub(crate) tls_options: Option<TlsOptions>,
}
