use std::env;
use std::fs;
use std::fs::File;
use std::io::{self, Write};
use std::path::PathBuf;

#[allow(unused_must_use)]
fn generate_files(files: Vec<PathBuf>) -> Result<(), String> {
    for file in files {
        match file.file_name() {
            None => Err("Unexpected file path is found".to_owned()),
            Some(file_name) => {
                let mut f = File::create(file_name).unwrap();
                let content = fs::read_to_string(file).unwrap();
                write!(f, "{}", content).unwrap();
                f.flush().unwrap();
                Ok(())
            },
        };
    }
    Ok(())
}

fn main() -> Result<(), io::Error> {
    let args: Vec<String> = env::args().collect();
    if args.len() > 1 {
        panic!("This tool expects no argument.")
    }

    let dir = fs::read_dir("./examples")?;
    let mut files: Vec<PathBuf> = Vec::new();
    for element in dir.into_iter() {
        files.push(element?.path());
    }

    files.sort();

    match generate_files(files) {
        Ok(()) => {
            println!("Generated Digdag workflow files!");
        }
        Err(e) => {
            println!("Error: {}", e);
        }
    }
    Ok(())
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    #[allow(unused_must_use)]
    fn test_generate_files() {
        let mut arr: Vec<PathBuf> = Vec::new();
        let tmpdir = env::temp_dir();
        let filepath = tmpdir.join("test.txt");
        fs::write(&filepath, "hello").ok();
        arr.push(filepath.to_path_buf());

        let _res = generate_files(arr);
        assert!(PathBuf::from("test.txt").exists());

        fs::remove_file("test.txt");
    }

    #[test]
    #[should_panic]
    fn test_empty_file() {
        let mut arr: Vec<PathBuf> = Vec::new();
        let tmpdir = env::temp_dir();
        let filepath = &tmpdir.join("test.txt");
        arr.push(filepath.to_path_buf());

        assert!(generate_files(arr).is_err());
    }
}
