use molecule_codegen::{ast, VERSION};
pub const PLUGIN_VERSION: &str = env!("CARGO_PKG_VERSION");

pub(crate) struct Generator;

use std::io;

mod generator;
use generator::Generator as _;
mod builder;
mod union;

impl Generator {
    pub fn generate<W: io::Write>(writer: &mut W, ast: &ast::Ast) -> io::Result<()> {
        writeln!(writer, "// Generated by Molecule {}", VERSION)?;
        writeln!(writer, "// Generated by Moleculec-Go {}", PLUGIN_VERSION)?;

        let import = format!(
            r#"
package {}
import (
    "bytes"
    "encoding/binary"
    "errors"
    "strconv"
    "strings"
       )"#,
            ast.namespace()
        );
        writeln!(writer, "{}", import)?;

        let major_imports = ast.imports();
        if major_imports.is_empty() {
            let primitive = String::from(
                r#"
type Number uint32
const HeaderSizeUint uint32 = 4
// Byte is the primitive type
type Byte [1]byte
func NewByte(b byte) Byte {
    return Byte([1]byte{b})
}
func ByteDefault() Byte {
    return Byte([1]byte{0})
}
func ByteFromSliceUnchecked(slice []byte) *Byte {
    b := new(Byte)
    b[0] = slice[0]
    return b
}
func (b *Byte) AsSlice() []byte {
    return b[:]
}
func ByteFromSlice(slice []byte, _compatible bool) (*Byte, error) {
    if len(slice) != 1 {
        return nil, errors.New("TotalSizeNotMatch")
    }
    b := new(Byte)
    b[0] = slice[0]
    return b, nil
}
func unpackNumber(b []byte) Number {
    bytesBuffer := bytes.NewBuffer(b)
    var x Number
    binary.Read(bytesBuffer, binary.LittleEndian, &x)
    return x
}
func packNumber(num Number) []byte {
    b := make([]byte, 4)
    binary.LittleEndian.PutUint32(b, uint32(num))
    return b
}
                    "#,
            );

            writeln!(writer, "{}", primitive)?;
        }

        for decl in ast.major_decls() {
            match decl.as_ref() {
                ast::TopDecl::Option_(ref i) => i.generate(writer)?,
                ast::TopDecl::Union(ref i) => i.generate(writer)?,
                ast::TopDecl::Array(ref i) => i.generate(writer)?,
                ast::TopDecl::Struct(ref i) => i.generate(writer)?,
                ast::TopDecl::FixVec(ref i) => i.generate(writer)?,
                ast::TopDecl::DynVec(ref i) => i.generate(writer)?,
                ast::TopDecl::Table(ref i) => i.generate(writer)?,
                ast::TopDecl::Primitive(_) => unreachable!(),
            };
        }
        Ok(())
    }
}
