//! Provides the error type for mobc-boltrs

use std::fmt::{Display, Formatter};

/// Error type for mobc-boltrs
///
/// # Examples
///
/// ```rust,no_run
/// # use mobc_boltrs::Error;
///
/// let e = Error::InvalidClientVersion { version: 143 };
/// ```
#[derive(Debug)]
pub enum Error {
    /// Wraps and returns an error generated by the Bolt client
    ClientError { source: bolt_client::error::Error },

    /// Returned when the client fails to initialize a session with the server
    ClientInitFailed { message: bolt_proto::Message },

    /// Returned when an address or hostname cannot be resolved
    InvalidAddress,

    /// Returned when the client requests an invalid version of the database
    InvalidClientVersion { version: u32 },

    /// Returned when the client metadata is incorrect, such as bad authentication credentials
    InvalidMetadata { metadata: String },

    /// Wraps and returns an error from the underlying I/O libraries
    IoError { source: std::io::Error },

    /// Wraps and returns a Bolt protocol error
    ProtocolError { source: bolt_proto::error::Error },
}

impl Display for Error {
    fn fmt(&self, f: &mut Formatter) -> std::fmt::Result {
        match self {
            Error::ClientError { source } => {
                write!(f, "Client Error. Source error: {}", source)
            }
            Error::ClientInitFailed { message } => {
                write!(f, "Client initialization failed. Received: {:?}", message)
            }
            Error::InvalidAddress => {
                write!(f, "Invalid host address")
            }
            Error::InvalidClientVersion { version } => {
                write!(f, "Invalid client version: {:#x}", version)
            }
            Error::InvalidMetadata { metadata } => {
                write!(f, "Invalid metadata: {}", metadata)
            }
            Error::IoError { source } => {
                write!(f, "IO Error. Source error: {}", source)
            }
            Error::ProtocolError { source } => {
                write!(f, "Protocol Error. Source error: {}", source)
            }
        }
    }
}

impl std::error::Error for Error {
    fn source(&self) -> Option<&(dyn std::error::Error + 'static)> {
        match self {
            Error::ClientError { source } => Some(source),
            Error::ClientInitFailed { message: _ } => None,
            Error::InvalidAddress => None,
            Error::InvalidClientVersion { version: _ } => None,
            Error::InvalidMetadata { metadata: _ } => None,
            Error::IoError { source } => Some(source),
            Error::ProtocolError { source } => Some(source),
        }
    }
}

impl From<bolt_client::error::Error> for Error {
    fn from(e: bolt_client::error::Error) -> Self {
        Error::ClientError { source: e }
    }
}

impl From<bolt_proto::error::Error> for Error {
    fn from(e: bolt_proto::error::Error) -> Self {
        Error::ProtocolError { source: e }
    }
}

impl From<std::io::Error> for Error {
    fn from(e: std::io::Error) -> Self {
        Error::IoError { source: e }
    }
}
