//! Private module for data structures similar to [`Arc`]

use std::hash::{Hash, Hasher};
use std::ops::Deref;
use std::sync::{Arc, Weak as WeakArc};

/// Same as [`Arc`], but equality is defined by-address rather than by equality
/// of content.
#[derive(Default, Debug)]
pub struct ArcByAddr<T>(Arc<T>);

impl<T> From<Arc<T>> for ArcByAddr<T> {
    fn from(arc: Arc<T>) -> Self {
        ArcByAddr(arc)
    }
}

impl<T> From<ArcByAddr<T>> for Arc<T> {
    fn from(arc_by_addr: ArcByAddr<T>) -> Self {
        arc_by_addr.0
    }
}

impl<T> ArcByAddr<T> {
    /// Create a new `ArcByAddr` with the given value
    pub fn new(value: T) -> Self {
        ArcByAddr::from(Arc::new(value))
    }
    /// Returns inner value if `ArcByAddr` has exactly one strong reference
    pub fn try_unwrap(this: Self) -> Result<T, Self> {
        Arc::try_unwrap(Arc::from(this)).map_err(ArcByAddr::from)
    }
}

impl<T> Deref for ArcByAddr<T> {
    type Target = Arc<T>;
    fn deref(&self) -> &Self::Target {
        &self.0
    }
}

impl<T> Clone for ArcByAddr<T> {
    fn clone(&self) -> Self {
        ArcByAddr(Arc::clone(&self))
    }
}

impl<T> PartialEq for ArcByAddr<T> {
    fn eq(&self, other: &Self) -> bool {
        Arc::ptr_eq(&self, &other)
    }
}

impl<T> Eq for ArcByAddr<T> {}

impl<T> Hash for ArcByAddr<T> {
    fn hash<H: Hasher>(&self, state: &mut H) {
        Arc::as_ptr(&self).hash(state);
    }
}

/// Collection of [`std::sync::Weak`]
#[derive(Debug)]
pub struct WeakVec<T> {
    vec: Vec<WeakArc<T>>,
    gc: usize,
}

impl<T> Default for WeakVec<T> {
    fn default() -> Self {
        WeakVec {
            vec: Default::default(),
            gc: Default::default(),
        }
    }
}

impl<T> WeakVec<T> {
    /// Iterate over elements in collection that have at least one strong
    /// reference (items will be provided as [`Arc`])
    pub fn iter(&self) -> impl Iterator<Item = Arc<T>> + '_ {
        self.vec.iter().filter_map(|x| WeakArc::upgrade(x))
    }
    /// Add new [`Weak`](WeakArc) to collection
    pub fn push(&mut self, element: WeakArc<T>) {
        self.vec.push(element);
        if self.vec.len() > self.gc {
            self.vec.retain(|weak| weak.strong_count() > 0);
            self.gc = self.vec.len().saturating_mul(2).max(8);
        }
    }
}
