//! Traits that are implemented for types that can be
//! [used as key](super::DbOptions::key_type) or
//! [value](super::DbOptions::value_type)
//!
//! Any type that implements [`Storable`] can be used as a key or value in a
//! [`Db`](super::Db). Implementation is `unsafe` and must be correct and not
//! change between (re-)opening environments.
//!
//! Types that have a fixed-length byte representation should additionally
//! implement [`StorableConstBytesLen`].

use super::helpers::IsType;
use super::{Owned, OwnedPointer, PointerIntoOwned};

// TODO: When stable, use `std::ffi` instead of `core::ffi`
// see: https://github.com/rust-lang/rust/commit/07ea143f96929ac7f0b7af0f025be48a472273e5
use core::ffi::{c_size_t, c_uint};
use std::cmp::Ordering;
use std::mem::{size_of, size_of_val};
use std::ops::Deref;
use std::ptr::{copy_nonoverlapping, read_unaligned};
use std::slice;
use std::str;

/// Types that can be stored
pub unsafe trait Storable: Ord + 'static {
    /// Does byte representation have fixed length?
    const CONST_BYTES_LEN: bool;
    /// Is type equivalent to [`c_uint`] or [`c_size_t`]?
    const OPTIMIZE_INT: bool = false;
    /// Is [`Ord`] consistent with lexicographical sorting of binary representation?
    const TRIVIAL_CMP: bool = false;
    /// Pointer to aligned version of Self
    type AlignedRef<'a>: Deref<Target = Self> + PointerIntoOwned;
    /// Pointer to byte representation
    type BytesRef<'a>: Deref<Target = [u8]>
    where
        Self: 'a;
    /// Converts to byte slice
    fn to_bytes(&self) -> Self::BytesRef<'_>;
    /// Length of byte representation
    fn bytes_len(&self) -> usize {
        self.to_bytes().len()
    }
    /// Converts from byte slice
    unsafe fn from_bytes_unchecked(bytes: &[u8]) -> Self::AlignedRef<'_>;
    /// Compares byte representation
    unsafe fn cmp_bytes_unchecked(a: &[u8], b: &[u8]) -> Ordering {
        Self::from_bytes_unchecked(a).cmp(&Self::from_bytes_unchecked(b))
    }
}

/// Types that can be stored with a fixed-length byte representation
pub unsafe trait StorableConstBytesLen: Storable {
    /// Length of byte representation as constant
    const BYTES_LEN: usize;
}

/// Implement [`Storable`] for variable-sized types that do not require
/// alignment and can be simply transmuted
macro_rules! impl_storable_transmute_varsize_trivial_cmp {
    ($elem:ty, $slice:ty, $owned:ty) => {
        unsafe impl Storable for $slice {
            const CONST_BYTES_LEN: bool = false;
            const TRIVIAL_CMP: bool = true;
            type AlignedRef<'a> = &'a Self;
            type BytesRef<'a> = &'a [u8];
            fn to_bytes(&self) -> Self::BytesRef<'_> {
                unsafe {
                    slice::from_raw_parts(self as *const Self as *const u8, size_of_val(self))
                }
            }
            unsafe fn from_bytes_unchecked(bytes: &[u8]) -> Self::AlignedRef<'_> {
                slice::from_raw_parts(
                    bytes as *const [u8] as *const $elem,
                    bytes.len() / size_of::<$elem>(),
                )
            }
        }
        unsafe impl Storable for $owned {
            const CONST_BYTES_LEN: bool = false;
            const TRIVIAL_CMP: bool = true;
            type AlignedRef<'a> = Owned<Self>;
            type BytesRef<'a> = &'a [u8];
            fn to_bytes(&self) -> Self::BytesRef<'_> {
                let slice: &$slice = self;
                unsafe {
                    slice::from_raw_parts(slice as *const $slice as *const u8, size_of_val(slice))
                }
            }
            unsafe fn from_bytes_unchecked(bytes: &[u8]) -> Self::AlignedRef<'_> {
                Owned(
                    slice::from_raw_parts(
                        bytes as *const [u8] as *const $elem,
                        bytes.len() / size_of::<$elem>(),
                    )
                    .to_owned(),
                )
            }
        }
    };
}

impl_storable_transmute_varsize_trivial_cmp!(bool, [bool], Vec<bool>);
impl_storable_transmute_varsize_trivial_cmp!(i8, [i8], Vec<i8>);
impl_storable_transmute_varsize_trivial_cmp!(u8, [u8], Vec<u8>);

unsafe impl Storable for str {
    const CONST_BYTES_LEN: bool = false;
    const TRIVIAL_CMP: bool = true;
    type AlignedRef<'a> = &'a Self;
    type BytesRef<'a> = &'a [u8];
    fn to_bytes(&self) -> Self::BytesRef<'_> {
        self.as_bytes()
    }
    unsafe fn from_bytes_unchecked(bytes: &[u8]) -> Self::AlignedRef<'_> {
        str::from_utf8_unchecked(bytes)
    }
}
unsafe impl Storable for String {
    const CONST_BYTES_LEN: bool = false;
    const TRIVIAL_CMP: bool = true;
    type AlignedRef<'a> = Owned<Self>;
    type BytesRef<'a> = &'a [u8];
    fn to_bytes(&self) -> Self::BytesRef<'_> {
        self.as_bytes()
    }
    unsafe fn from_bytes_unchecked(bytes: &[u8]) -> Self::AlignedRef<'_> {
        Owned(str::from_utf8_unchecked(bytes).to_owned())
    }
}
/// Implement [`Storable`] and [`StorableConstBytesLen`] for fixed-sized types
/// that do not require alignment and are stored like their memory
/// representation
macro_rules! impl_storable_fixed_size_always_aligned {
    ($type:ty, $optimize_int:expr) => {
        unsafe impl Storable for $type {
            const CONST_BYTES_LEN: bool = true;
            const OPTIMIZE_INT: bool = $optimize_int;
            type AlignedRef<'a> = &'a Self;
            type BytesRef<'a> = &'a [u8];
            fn to_bytes(&self) -> Self::BytesRef<'_> {
                unsafe {
                    slice::from_raw_parts(self as *const Self as *const u8, size_of::<Self>())
                }
            }
            fn bytes_len(&self) -> usize {
                Self::BYTES_LEN
            }
            unsafe fn from_bytes_unchecked(bytes: &[u8]) -> Self::AlignedRef<'_> {
                &*(bytes as *const _ as *const Self)
            }
        }
        unsafe impl StorableConstBytesLen for $type {
            const BYTES_LEN: usize = size_of::<Self>();
        }
    };
}

/// Implement [`Storable`] and [`StorableConstBytesLen`] for fixed-sized types
/// that need to be aligned but are otherwise stored like their memory
/// representation
macro_rules! impl_storable_fixed_size_force_align {
    ($type:ty, $optimize_int:expr) => {
        unsafe impl Storable for $type {
            const CONST_BYTES_LEN: bool = true;
            const OPTIMIZE_INT: bool = $optimize_int;
            type AlignedRef<'a> = Owned<Self>;
            type BytesRef<'a> = &'a [u8];
            fn to_bytes(&self) -> Self::BytesRef<'_> {
                unsafe {
                    slice::from_raw_parts(self as *const Self as *const u8, size_of::<Self>())
                }
            }
            fn bytes_len(&self) -> usize {
                Self::BYTES_LEN
            }
            unsafe fn from_bytes_unchecked(bytes: &[u8]) -> Self::AlignedRef<'_> {
                Owned(read_unaligned(bytes.as_ptr() as *const Self))
            }
        }
        unsafe impl StorableConstBytesLen for $type {
            const BYTES_LEN: usize = size_of::<Self>();
        }
    };
}

/// Implement [`Storable`] and [`StorableConstBytesLen`] for unsigned primitive
/// integers that do not require alignment
macro_rules! impl_storable_unsigned_always_aligned {
    ($type:ty) => {
        impl_storable_fixed_size_always_aligned!(
            $type,
            size_of::<Self>() == size_of::<c_uint>() || size_of::<Self>() == size_of::<c_size_t>()
        );
    };
}

/// Implement [`Storable`] and [`StorableConstBytesLen`] for signed primitive
/// integers that do not require alignment
macro_rules! impl_storable_signed_always_aligned {
    ($type:ty) => {
        impl_storable_fixed_size_always_aligned!($type, false);
    };
}

/// Implement [`Storable`] and [`StorableConstBytesLen`] for unsigned primitive
/// integers that need to be (re-)aligned
macro_rules! impl_storable_unsigned_force_align {
    ($type:ty) => {
        impl_storable_fixed_size_force_align!(
            $type,
            size_of::<Self>() == size_of::<c_uint>() || size_of::<Self>() == size_of::<c_size_t>()
        );
    };
}

/// Implement [`Storable`] and [`StorableConstBytesLen`] for signed primitive
/// integers that need to be (re-)aligned
macro_rules! impl_storable_signed_force_align {
    ($type:ty) => {
        impl_storable_fixed_size_force_align!($type, false);
    };
}

// NOTE: `bool` is guaranteed to have a size of 1 and thus is always aligned
impl_storable_unsigned_always_aligned!(bool);
impl_storable_unsigned_always_aligned!(u8);
impl_storable_unsigned_force_align!(u16);
impl_storable_unsigned_force_align!(u32);
impl_storable_unsigned_force_align!(u64);
impl_storable_unsigned_force_align!(u128);
impl_storable_unsigned_force_align!(usize);
impl_storable_signed_always_aligned!(i8);
impl_storable_signed_force_align!(i16);
impl_storable_signed_force_align!(i32);
impl_storable_signed_force_align!(i64);
impl_storable_signed_force_align!(i128);
impl_storable_signed_force_align!(isize);

/// Implement [`Storable`] for slices and [`Vec`]s of primitive integers that
/// need to be (re-)aligned
macro_rules! impl_storable_intslice_force_align {
    ($type:ty) => {
        unsafe impl Storable for [$type] {
            const CONST_BYTES_LEN: bool = false;
            type AlignedRef<'a> = OwnedPointer<Vec<$type>>;
            type BytesRef<'a> = &'a [u8];
            fn to_bytes(&self) -> Self::BytesRef<'_> {
                unsafe {
                    slice::from_raw_parts(self as *const [$type] as *const u8, self.bytes_len())
                }
            }
            fn bytes_len(&self) -> usize {
                self.len() * size_of::<$type>()
            }
            unsafe fn from_bytes_unchecked(bytes: &[u8]) -> Self::AlignedRef<'_> {
                let len = bytes.len() / size_of::<$type>();
                let mut vec: Vec<$type> = Vec::with_capacity(len);
                copy_nonoverlapping(bytes.as_ptr(), vec.as_mut_ptr() as *mut u8, bytes.len());
                vec.set_len(len);
                OwnedPointer(vec)
            }
        }
        unsafe impl Storable for Vec<$type> {
            const CONST_BYTES_LEN: bool = false;
            type AlignedRef<'a> = Owned<Vec<$type>>;
            type BytesRef<'a> = &'a [u8];
            fn to_bytes(&self) -> Self::BytesRef<'_> {
                unsafe {
                    slice::from_raw_parts(
                        self as &[$type] as *const [$type] as *const u8,
                        self.bytes_len(),
                    )
                }
            }
            fn bytes_len(&self) -> usize {
                self.len() * size_of::<$type>()
            }
            unsafe fn from_bytes_unchecked(bytes: &[u8]) -> Self::AlignedRef<'_> {
                let len = bytes.len() / size_of::<$type>();
                let mut vec: Vec<$type> = Vec::with_capacity(len);
                copy_nonoverlapping(bytes.as_ptr(), vec.as_mut_ptr() as *mut u8, bytes.len());
                vec.set_len(len);
                Owned(vec)
            }
        }
    };
}

impl_storable_intslice_force_align!(u16);
impl_storable_intslice_force_align!(u32);
impl_storable_intslice_force_align!(u64);
impl_storable_intslice_force_align!(u128);
impl_storable_intslice_force_align!(usize);
impl_storable_intslice_force_align!(i16);
impl_storable_intslice_force_align!(i32);
impl_storable_intslice_force_align!(i64);
impl_storable_intslice_force_align!(i128);
impl_storable_intslice_force_align!(isize);

/// Types that can be stored where the restored owned version is a certain type
pub trait StorableWithOwned<T>: Storable {
    /// Converts from byte slice into (owned) `Self`
    unsafe fn owned_from_bytes_unchecked(bytes: &[u8]) -> T;
}

impl<T, O> StorableWithOwned<O> for T
where
    T: ?Sized + Storable,
    for<'a> <<T as Storable>::AlignedRef<'a> as PointerIntoOwned>::Owned: IsType<O>,
{
    unsafe fn owned_from_bytes_unchecked(bytes: &[u8]) -> O {
        Self::from_bytes_unchecked(bytes).into_owned().identity()
    }
}

unsafe impl<T1, T2> Storable for (T1, T2)
where
    T1: StorableWithOwned<T1> + StorableConstBytesLen,
    T2: StorableWithOwned<T2>,
{
    const CONST_BYTES_LEN: bool = T2::CONST_BYTES_LEN;
    type AlignedRef<'a> = Owned<Self>;
    type BytesRef<'a> = Vec<u8>;
    fn to_bytes(&self) -> Self::BytesRef<'_> {
        let mut bytes = Vec::with_capacity(self.0.bytes_len() + self.1.bytes_len());
        bytes.extend_from_slice(&self.0.to_bytes());
        bytes.extend_from_slice(&self.1.to_bytes());
        bytes
    }
    unsafe fn from_bytes_unchecked(bytes: &[u8]) -> Self::AlignedRef<'_> {
        let v1: T1 = T1::owned_from_bytes_unchecked(&bytes[0..T1::BYTES_LEN]);
        let v2: T2 = T2::owned_from_bytes_unchecked(&bytes[T1::BYTES_LEN..]);
        Owned((v1, v2))
    }
}

unsafe impl<T1, T2> StorableConstBytesLen for (T1, T2)
where
    T1: StorableWithOwned<T1> + StorableConstBytesLen,
    T2: StorableWithOwned<T2> + StorableConstBytesLen,
{
    const BYTES_LEN: usize = T1::BYTES_LEN + T2::BYTES_LEN;
}

unsafe impl<T1, T2, T3> Storable for (T1, T2, T3)
where
    T1: StorableWithOwned<T1> + StorableConstBytesLen,
    T2: StorableWithOwned<T2> + StorableConstBytesLen,
    T3: StorableWithOwned<T3>,
{
    const CONST_BYTES_LEN: bool = T3::CONST_BYTES_LEN;
    type AlignedRef<'a> = Owned<Self>;
    type BytesRef<'a> = Vec<u8>;
    fn to_bytes(&self) -> Self::BytesRef<'_> {
        let mut bytes =
            Vec::with_capacity(self.0.bytes_len() + self.1.bytes_len() + self.2.bytes_len());
        bytes.extend_from_slice(&self.0.to_bytes());
        bytes.extend_from_slice(&self.1.to_bytes());
        bytes.extend_from_slice(&self.2.to_bytes());
        bytes
    }
    unsafe fn from_bytes_unchecked(bytes: &[u8]) -> Self::AlignedRef<'_> {
        let idx1 = T1::BYTES_LEN;
        let idx2 = idx1 + T2::BYTES_LEN;
        let v1: T1 = T1::owned_from_bytes_unchecked(&bytes[0..idx1]);
        let v2: T2 = T2::owned_from_bytes_unchecked(&bytes[idx1..idx2]);
        let v3: T3 = T3::owned_from_bytes_unchecked(&bytes[idx2..]);
        Owned((v1, v2, v3))
    }
}

unsafe impl<T1, T2, T3> StorableConstBytesLen for (T1, T2, T3)
where
    T1: StorableWithOwned<T1> + StorableConstBytesLen,
    T2: StorableWithOwned<T2> + StorableConstBytesLen,
    T3: StorableWithOwned<T3> + StorableConstBytesLen,
{
    const BYTES_LEN: usize = T1::BYTES_LEN + T2::BYTES_LEN + T3::BYTES_LEN;
}

unsafe impl<T1, T2, T3, T4> Storable for (T1, T2, T3, T4)
where
    T1: StorableWithOwned<T1> + StorableConstBytesLen,
    T2: StorableWithOwned<T2> + StorableConstBytesLen,
    T3: StorableWithOwned<T3> + StorableConstBytesLen,
    T4: StorableWithOwned<T4>,
{
    const CONST_BYTES_LEN: bool = T4::CONST_BYTES_LEN;
    type AlignedRef<'a> = Owned<Self>;
    type BytesRef<'a> = Vec<u8>;
    fn to_bytes(&self) -> Self::BytesRef<'_> {
        let mut bytes = Vec::with_capacity(
            self.0.bytes_len() + self.1.bytes_len() + self.2.bytes_len() + self.3.bytes_len(),
        );
        bytes.extend_from_slice(&self.0.to_bytes());
        bytes.extend_from_slice(&self.1.to_bytes());
        bytes.extend_from_slice(&self.2.to_bytes());
        bytes.extend_from_slice(&self.3.to_bytes());
        bytes
    }
    unsafe fn from_bytes_unchecked(bytes: &[u8]) -> Self::AlignedRef<'_> {
        let idx1 = T1::BYTES_LEN;
        let idx2 = idx1 + T2::BYTES_LEN;
        let idx3 = idx2 + T3::BYTES_LEN;
        let v1: T1 = T1::owned_from_bytes_unchecked(&bytes[0..idx1]);
        let v2: T2 = T2::owned_from_bytes_unchecked(&bytes[idx1..idx2]);
        let v3: T3 = T3::owned_from_bytes_unchecked(&bytes[idx2..idx3]);
        let v4: T4 = T4::owned_from_bytes_unchecked(&bytes[idx3..]);
        Owned((v1, v2, v3, v4))
    }
}

unsafe impl<T1, T2, T3, T4> StorableConstBytesLen for (T1, T2, T3, T4)
where
    T1: StorableWithOwned<T1> + StorableConstBytesLen,
    T2: StorableWithOwned<T2> + StorableConstBytesLen,
    T3: StorableWithOwned<T3> + StorableConstBytesLen,
    T4: StorableWithOwned<T4> + StorableConstBytesLen,
{
    const BYTES_LEN: usize = T1::BYTES_LEN + T2::BYTES_LEN + T3::BYTES_LEN + T4::BYTES_LEN;
}
