//! Smart pointers owning their targets

use std::ops::Deref;

/// Smart pointer to owned inner value
/// (see [`deref`](Owned::deref) implementation)
#[derive(PartialEq, Eq, PartialOrd, Ord, Hash, Debug)]
pub struct Owned<X>(pub X);

impl<X> Deref for Owned<X> {
    type Target = X;
    /// Returns a reference to the contained value
    fn deref(&self) -> &Self::Target {
        &self.0
    }
}

/// Smart pointer to value pointed-to by owned inner value
/// (see [`deref`](OwnedPointer::deref) implementation)
#[derive(PartialEq, Eq, PartialOrd, Ord, Hash, Debug)]
pub struct OwnedPointer<X>(pub X);

impl<X: Deref> Deref for OwnedPointer<X> {
    type Target = <X as Deref>::Target;
    /// Returns a reference to the value referenced by the contained value
    /// (double indirection)
    fn deref(&self) -> &Self::Target {
        &self.0
    }
}

/// Pointer types that can be converted into an owned type
pub trait PointerIntoOwned: Sized {
    /// The type the pointer can be converted into
    type Owned;
    /// Convert into owned type
    fn into_owned(self) -> Self::Owned;
}

impl<'a, T> PointerIntoOwned for &'a T
where
    T: ?Sized + ToOwned,
{
    type Owned = <T as ToOwned>::Owned;
    fn into_owned(self) -> Self::Owned {
        self.to_owned()
    }
}

impl<T> PointerIntoOwned for Owned<T> {
    type Owned = T;
    fn into_owned(self) -> Self::Owned {
        self.0
    }
}

impl<T> PointerIntoOwned for OwnedPointer<T> {
    type Owned = T;
    fn into_owned(self) -> Self::Owned {
        self.0
    }
}
