//! Key-value database using LMDB
//!
//! # Caveats / Safety
//!
//! Do not open the same environment twice in the same process (see
//! documentation of [`EnvBuilder`]). The file format is platform specific, so
//! endianness and word size must not change across use.
//!
//! When opening an already existing database, all options must be compatible
//! with the previously used options (see documentation of [`DbOptions`]).
//!
//! Stale readers might need to be cleared manually, see
//! [`Env::clear_stale_readers`].
//!
//! # Example
//!
//! ```
//! use mmtkvdb::{self as kv, Env as _, Txn as _};
//! use tempfile::tempdir;
//! fn doit() -> Result<(), std::io::Error> {
//!     let location = tempdir()?;
//!     let db1_opts = kv::DbOptions::new().name("db1");
//!     let db2_opts = kv::DbOptions::new().key_type::<str>().value_type::<i32>().name("db2");
//!     let env_builder = kv::EnvBuilder::new().dir(location.path()).max_dbs(2);
//!     {
//!         let mut env = unsafe { env_builder.open_rw()? };
//!         let db1 = unsafe { env.create_db(&db1_opts)? };
//!         let db2 = unsafe { env.create_db(&db2_opts)? };
//!         let mut txn = env.txn_rw()?;
//!         txn.put(&db1, "key1".as_bytes(), "value1".as_bytes())?;
//!         txn.put(&db2, "Prime", &17)?;
//!         txn.commit()?;
//!     }
//!     {
//!         let env = unsafe { env_builder.open_ro()? };
//!         let db1 = unsafe { env.open_db(&db1_opts)? };
//!         let db2 = unsafe { env.open_db(&db2_opts)? };
//!         let txn = env.txn_ro()?;
//!         assert_eq!(txn.get(&db1, "key1".as_bytes())?, Some("value1".as_bytes()));
//!         assert_eq!(txn.get_owned(&db2, "Prime")?, Some(17));
//!     }
//!     location.close()?;
//!     Ok(())
//! }
//! doit().expect("error during database access")
//! ```

#![feature(generic_associated_types)]
#![feature(hash_set_entry)]
#![feature(core_ffi_c, c_size_t)]
#![warn(missing_docs)]

mod helpers;
pub mod lmdb;
pub mod owning_pointer;
pub mod storable;
mod syncutils;

#[cfg(test)]
mod tests;

use helpers::*;
pub use owning_pointer::PointerIntoOwned;
use owning_pointer::*;
pub use storable::Storable;
use syncutils::*;

// TODO: When stable, use `std::ffi` instead of `core::ffi`
// see: https://github.com/rust-lang/rust/commit/07ea143f96929ac7f0b7af0f025be48a472273e5
use core::ffi::{c_int, c_uint};
use std::cell::UnsafeCell;
use std::collections::HashSet;
use std::ffi::{CStr, CString};
use std::hash::{Hash, Hasher};
use std::io;
use std::marker::PhantomData;
use std::mem::{forget, take, MaybeUninit};
use std::path::Path;
use std::ptr::{null, null_mut};
use std::slice;
use std::str;
use std::sync::{Arc, Mutex, Weak as WeakArc};

// Compile-time check enforcing `usize` is same size as `c_size_t`
const _: fn() = || {
    let _ = core::mem::transmute::<usize, core::ffi::c_size_t>;
};

/// Converts an error code from LMDB into a `Result`
fn check_err_code(code: c_int) -> Result<(), io::Error> {
    if code > 0 {
        Err(io::Error::from_raw_os_error(code))
    } else if code < 0 {
        Err(io::Error::new(
            io::ErrorKind::Other,
            unsafe { CStr::from_ptr(lmdb::mdb_strerror(code)) }
                .to_str()
                .unwrap_or("unknown error (LMDB error string is not valid UTF-8)"),
        ))
    } else {
        Ok(())
    }
}

/// Type alias for [`c_uint`] as used by LMDB to specify flags
type LmdbFlags = c_uint;

/// Macro checking if an `lmdb_flags` field has a certain flag set
macro_rules! flag_get {
    ($self:ident, $flag:ident) => {
        ($self.lmdb_flags & (lmdb::$flag as LmdbFlags) != 0)
    };
}

/// Macro setting or clearing a flag in the `lmdb_flags` field
macro_rules! flag_set {
    ($self:ident, $flag:ident, $value:expr) => {
        if $value {
            $self.lmdb_flags |= lmdb::$flag as LmdbFlags;
        } else {
            $self.lmdb_flags &= !(lmdb::$flag as LmdbFlags);
        }
    };
}

/// Internal struct with LMDB environment handle as raw pointer
///
/// On drop, the environment is closed. Contains an additional mutex to
/// synchronize opening/creating/closing databases.
///
/// The struct implements `Send` and `Sync`, thus when accessing
/// [`EnvBackend::inner`], proper care must be taken to syncronize access
/// according to the rules laid out in the LMDB API documentation.
#[derive(Debug)]
struct EnvBackend {
    inner: *mut lmdb::MDB_env,
    max_keysize: usize,
    db_mutex: Mutex<()>,
}

unsafe impl Send for EnvBackend {}
unsafe impl Sync for EnvBackend {}

impl PartialEq for EnvBackend {
    fn eq(&self, other: &Self) -> bool {
        self.inner == other.inner
    }
}

impl Eq for EnvBackend {}

impl Hash for EnvBackend {
    fn hash<H: Hasher>(&self, state: &mut H) {
        self.inner.hash(state);
    }
}

impl Drop for EnvBackend {
    fn drop(&mut self) {
        unsafe { lmdb::mdb_env_close(self.inner) }
    }
}

/// Read-only handle for accessing environment that stores key-value databases
///
/// An environment can be opened using [`EnvBuilder`].
///
/// Use [`Env::open_db`] to retrieve database handles
/// and [`Env::txn_ro`] to start a read-only transaction.
///
/// It's possible to clone the handle, in which case the environment will be
/// closed when the last handle is dropped.
#[derive(Clone, PartialEq, Eq, Hash, Debug)]
pub struct EnvRo {
    backend: Arc<EnvBackend>,
}

/// Read-write handle for accessing environment that stores key-value databases
///
/// An environment can be opened using [`EnvBuilder`].
///
/// Use [`Env::open_db`] or [`EnvRw::create_db`] to retrieve database handles
/// and [`EnvRw::txn_rw`] to start a read-write transaction.
///
/// It's possible to create a read-only handle by invoking [`Env::clone_ro`].
/// The environment is closed when the last handle is dropped.
#[derive(PartialEq, Eq, Hash, Debug)]
pub struct EnvRw {
    backend: Arc<EnvBackend>,
}

impl PartialEq<EnvRw> for EnvRo {
    /// Returns true if both handles refer to the same environment
    fn eq(&self, other: &EnvRw) -> bool {
        self.backend == other.backend
    }
}

impl PartialEq<EnvRo> for EnvRw {
    /// Returns true if both handles refer to the same environment
    fn eq(&self, other: &EnvRo) -> bool {
        self.backend == other.backend
    }
}

/// Internal struct with LMDB transaction handle as raw pointer
///
/// On drop, all associated cursors (which haven't been dropped and thus closed
/// yet) will be closed.
#[derive(Debug)]
struct TxnBackend<'a> {
    env_backend: &'a EnvBackend,
    inner: *mut lmdb::MDB_txn,
    cursors: WeakVec<CursorBackend>,
}

impl<'a> TxnBackend<'a> {
    fn close_cursors(&mut self) {
        for cursor in take(&mut self.cursors).iter() {
            // NOTE: mutex should never be poisoned
            *cursor.closed.lock().unwrap() = true;
            unsafe {
                lmdb::mdb_cursor_close(cursor.inner);
            }
        }
    }
}

impl<'a> Drop for TxnBackend<'a> {
    fn drop(&mut self) {
        self.close_cursors();
        unsafe {
            lmdb::mdb_txn_abort(self.inner);
        }
    }
}

/// Read-only transaction
///
/// A read-only transaction can be started with [`Env::txn_ro`].
///
/// The methods for read-only access to the database are accessible through the
/// [`Txn`] trait (which is implemented by `TxnRo` and [`TxnRw`]).
///
/// Read-only transactions do not need to be committed or aborted; their
/// handles can be simply dropped when not needed anymore.
///
/// Read-only transactions are [`Send`] and [`Sync`] but can't modify any data.
#[derive(Debug)]
pub struct TxnRo<'a> {
    backend: Mutex<TxnBackend<'a>>,
}

unsafe impl<'a> Send for TxnRo<'a> {}
unsafe impl<'a> Sync for TxnRo<'a> {}

/// Read-write transaction
///
/// A read-write transaction can be started with [`EnvRw::txn_rw`].
///
/// The methods for read-only access to the database are accessible through the
/// [`Txn`] trait (which is implemented by [`TxnRo`] and `TxnRw`).
/// Methods for write access, however, are available directly on the `TxnRw`
/// struct.
///
/// Read-write transactions are [aborted](TxnRw::abort) when dropped. To make
/// changes permanent, [`TxnRw::commit`] must be called.
///
/// handles can be simply dropped when not needed anymore.
///
/// Read-write transactions are neither [`Send`] nor [`Sync`].
#[derive(Debug)]
pub struct TxnRw<'a> {
    backend: UnsafeCell<TxnBackend<'a>>,
    used_dbs: HashSet<ArcByAddr<DbBackend>>,
}

/// Internal struct with LMDB database handle
///
/// `DbBackend` can be wrapped in an [`ArcByAddr`] to allow for equality checks
/// and hashing.
///
/// There is no strong reference to [`EnvBackend`] to avoid a lifetime argument
/// (particularly such that databases can persist beyond mutation of
/// [`EnvRw`]).
///
/// It is an error if a database handle is used on a different environment.
/// Method [`DbBackend::assert_env_backend`] can be used to panic in that case.
#[derive(Debug)]
struct DbBackend {
    env_backend: WeakArc<EnvBackend>,
    inner: lmdb::MDB_dbi,
}

impl DbBackend {
    /// Panic if database handle does not belong to environment
    fn assert_env_backend(&self, env_backend: &EnvBackend) {
        if WeakArc::upgrade(&self.env_backend).map_or(true, |x| &*x != env_backend) {
            panic!("database does not belong to environment");
        }
    }
}

impl Drop for DbBackend {
    fn drop(&mut self) {
        if let Some(env_backend) = WeakArc::upgrade(&self.env_backend) {
            // NOTE: Closing databases is optional. If mutex is poisoned, it's
            // not necessary to close the database (and no more databases can
            // be opened anyway).
            if let Ok(db_guard) = env_backend.db_mutex.lock() {
                unsafe {
                    lmdb::mdb_dbi_close(env_backend.inner, self.inner);
                }
                drop(db_guard);
            }
        }
    }
}

/// Database handle
///
/// These handles are created using [`DbOptions`] builder, which can be passed
/// by (shared) reference to [`Env::open_db`] or [`EnvRw::create_db`].
///
/// The type arguments `K`, `V`, and `C` reflect the type used for keys and
/// values, and whether duplicate keys are allowed (`C` being [`KeysUnique`] or
/// [`KeysDuplicate`]), respectively.
///
/// Operations on databases is done using the methods of the [`Txn`] trait or
/// [`TxnRw`] struct. The database handle must be passed to these methods by
/// (shared) reference.
///
/// Using a [`Db`] handle in a different environment (or in an environment that
/// has been closed and re-opened) will cause a panic.
#[derive(Debug)]
pub struct Db<K: ?Sized, V: ?Sized, C> {
    key: PhantomData<fn(&K) -> &K>,
    value: PhantomData<fn(&V) -> &V>,
    constraint: C,
    backend: ArcByAddr<DbBackend>,
}

impl<K: ?Sized, V: ?Sized, C> Clone for Db<K, V, C>
where
    C: Constraint,
{
    fn clone(&self) -> Self {
        Db {
            key: PhantomData,
            value: PhantomData,
            constraint: self.constraint,
            backend: self.backend.clone(),
        }
    }
}

impl<K: ?Sized, V: ?Sized, C> PartialEq for Db<K, V, C> {
    fn eq(&self, other: &Self) -> bool {
        self.backend == other.backend
    }
}

impl<K: ?Sized, V: ?Sized, C> Eq for Db<K, V, C> {}

/// Internal struct with LMDB cursor handle
///
/// There is no lifetime argument, but cursors will be closed when
/// [`TxnBackend`] gets dropped. In that case, this struct will be marked as
/// closed by setting [`CursorBackend::closed`] to true.
///
/// `CursorBackend` can be wrapped in an [`ArcByAddr`] to allow for equality
/// checks and hashing. (Equality can't be checked by comparing the inner raw
/// pointer because the address behind the raw pointer could be reused after
/// the cursor has been closed.)
///
/// Like database handles, cursor handles have no lifetime argument.
///
/// It is an error if a cursor handle is used on a different transaction.
/// Method [`CursorBackend::assert_txn_backend`] can be used to panic in that
/// case.
///
/// This struct implements `Send` and `Sync`, thus proper synchronization must
/// be ensured when using it (which is done by making operations work on
/// transaction handles).
#[derive(Debug)]
struct CursorBackend {
    inner: *mut lmdb::MDB_cursor,
    closed: Mutex<bool>,
    inner_txn: *mut lmdb::MDB_txn,
}

unsafe impl Send for CursorBackend {}
unsafe impl Sync for CursorBackend {}

impl CursorBackend {
    /// Panic if cursor handle does not belong to transaction
    fn assert_txn_backend(&self, txn_backend: &TxnBackend) {
        let closed = self.closed.lock().unwrap();
        if *closed || self.inner_txn != txn_backend.inner {
            // Avoid mutex poisoning by dropping mutex guard before panicking
            drop(closed);
            panic!("cursor does not belong to transaction");
        }
        drop(closed);
    }
}

impl Drop for CursorBackend {
    fn drop(&mut self) {
        // NOTE: mutex should never be poisoned
        if !*self.closed.get_mut().unwrap() {
            unsafe {
                lmdb::mdb_cursor_close(self.inner);
            }
        }
    }
}

#[derive(Debug)]
/// Cursor for reading from or writing to a particular database
///
/// A new cursor can be created using [`Txn::new_cursor`]. It is used by
/// invoking one of the cursor methods in [`Txn`] or [`TxnRw`].
pub struct Cursor<K: ?Sized, V: ?Sized, C> {
    db: Db<K, V, C>,
    backend: ArcByAddr<CursorBackend>,
}

impl<K: ?Sized, V: ?Sized, C> Clone for Cursor<K, V, C>
where
    C: Constraint,
{
    fn clone(&self) -> Self {
        Cursor {
            db: self.db.clone(),
            backend: self.backend.clone(),
        }
    }
}

impl<K: ?Sized, V: ?Sized, C> PartialEq for Cursor<K, V, C> {
    fn eq(&self, other: &Self) -> bool {
        self.backend == other.backend
    }
}

impl<K: ?Sized, V: ?Sized, C> Eq for Cursor<K, V, C> {}

/// Options for opening an environment
///
/// This struct follows the builder pattern. Start with [`EnvBuilder::new`] and
/// open the environment with [`EnvBuilder::open_ro`] (for read-only access) or
/// [`EnvBuilder::open_rw`] (for read-write access).
///
/// # Caveats / Safety
///
/// Do not open the same environment twice in the same process.
///
/// Since this cannot be easily ensured (and because the environment must be
/// free of corruption on the storage medium), [`open_ro`](EnvBuilder::open_ro)
/// and [`open_rw`](EnvBuilder::open_rw) are marked as `unsafe`.
/// Use [`Env::clone_ro`] to (re-)open an already open environment in read-only
/// mode.
///
/// The file format is platform specific, so endianness and word size must not
/// change across use.
///
/// # Example
///
/// ```
/// use mmtkvdb::{self as kv, Env as _, Txn as _};
/// use tempfile::tempdir;
/// let location = tempdir().unwrap();
/// let env_builder = kv::EnvBuilder::new().dir(location.path()).max_dbs(1);
/// let mut env_rw = unsafe { env_builder.open_rw() }.unwrap();
/// let env_ro = env_rw.clone_ro();
/// ```
#[derive(Clone, Debug)]
pub struct EnvBuilder<P> {
    path: P,
    lmdb_flags: LmdbFlags,
    max_size: usize,
    max_readers: usize,
    max_dbs: usize,
    unix_file_mode: u32,
}

/// Constraints on database (type argument `C` to [`DbOptions`] and [`Db`])
pub trait Constraint: Copy + 'static {
    /// Duplicate keys allowed?
    const DUPLICATE_KEYS: bool;
}

/// Type argument to [`DbOptions`] and [`Db`] indicating unique keys
#[derive(Clone, Copy)]
pub struct KeysUnique;

impl Constraint for KeysUnique {
    const DUPLICATE_KEYS: bool = false;
}

/// Type argument to [`DbOptions`] and [`Db`] indicating non-unique keys
#[derive(Clone, Copy)]
pub struct KeysDuplicate;

impl Constraint for KeysDuplicate {
    const DUPLICATE_KEYS: bool = true;
}

/// Options for opening a database
///
/// This struct follows the builder pattern. Start with [`DbOptions::new`] and
/// pass completed builder by (shared) reference to [`Env::open_db`] or
/// [`EnvRw::create_db`] to open or create a database.
///
/// # Caveats / Safety
///
/// When opening an already existing database, all options must be compatible
/// with the previously used options. It has be be ensured by the caller that
/// this is true. Thus [`Env::open_db`] and [`EnvRw::create_db`] are marked as
/// `unsafe`.
///
/// [`DbOptions::name`] has to be called last (after setting all other
/// options). This is due to some restrictions regarding `const fn`s.
/// A compiler error will be reported if the methods are invoked in the wrong
/// order.
///
/// # Type arguments
///
/// The type arguments `K`, `V`, and `C` reflect the type used for keys and
/// values, and whether duplicate keys are allowed (`C` being [`KeysUnique`] or
/// [`KeysDuplicate`]), respectively.
///
/// The fourth type parameter determines if a name has been set (or the unnamed
/// database has been selected), or if a name (or unnamed database) must be
/// selected yet.
///
/// # Example
///
/// ```
/// use mmtkvdb::{self as kv, Env as _, Txn as _};
/// use tempfile::tempdir;
/// let location = tempdir().unwrap();
/// let env_builder = kv::EnvBuilder::new().dir(location.path()).max_dbs(1);
/// let mut env_rw = unsafe { env_builder.open_rw() }.unwrap();
/// let db_opts = kv::DbOptions::new()
///     .key_type::<str>()
///     .value_type::<i32>()
///     .name("account_balance");
/// let db = unsafe { env_rw.create_db(&db_opts) }.unwrap();
/// ```
#[derive(Clone, Debug)]
pub struct DbOptions<K: ?Sized, V: ?Sized, C, N> {
    key: PhantomData<fn(&K) -> &K>,
    value: PhantomData<fn(&V) -> &V>,
    constraint: C,
    lmdb_flags: LmdbFlags,
    name: N,
}

/// Some default values used for [`EnvBuilder`]
pub mod env_builder_defaults {
    /// Default value for `max_size`
    pub const MAX_SIZE: usize = 1024 * 1024 * 1024;
    /// Default value for `max_readers`
    pub const MAX_READERS: usize = 126;
    /// Default value for `max_dbs`
    pub const MAX_DBS: usize = 0;
    /// Default value for `unix_file_mode`
    pub const UNIX_FILE_MODE: u32 = 0o600;
}

impl EnvBuilder<()> {
    /// Default options for environment
    pub const fn new() -> Self {
        Self {
            path: (),
            lmdb_flags: lmdb::MDB_NORDAHEAD as LmdbFlags | lmdb::MDB_NOTLS as LmdbFlags,
            max_size: env_builder_defaults::MAX_SIZE,
            max_readers: env_builder_defaults::MAX_READERS,
            max_dbs: env_builder_defaults::MAX_DBS,
            unix_file_mode: env_builder_defaults::UNIX_FILE_MODE,
        }
    }
}

impl<P> EnvBuilder<P> {
    /// Set directory location
    pub fn dir<'a>(mut self, path: &'a (impl AsRef<Path> + ?Sized)) -> EnvBuilder<&'a Path> {
        flag_set!(self, MDB_NOSUBDIR, false);
        EnvBuilder {
            path: path.as_ref(),
            lmdb_flags: self.lmdb_flags,
            max_size: self.max_size,
            max_readers: self.max_readers,
            max_dbs: self.max_dbs,
            unix_file_mode: self.unix_file_mode,
        }
    }
    /// Set file location
    pub fn file<'a>(mut self, path: &'a (impl AsRef<Path> + ?Sized)) -> EnvBuilder<&'a Path> {
        flag_set!(self, MDB_NOSUBDIR, true);
        EnvBuilder {
            path: path.as_ref(),
            lmdb_flags: self.lmdb_flags,
            max_size: self.max_size,
            max_readers: self.max_readers,
            max_dbs: self.max_dbs,
            unix_file_mode: self.unix_file_mode,
        }
    }
    /// Get writable memory map option
    pub const fn get_writemap(&self) -> bool {
        flag_get!(self, MDB_WRITEMAP)
    }
    /// Set or clear writable memory map option
    pub const fn writemap(mut self, flag: bool) -> Self {
        flag_set!(self, MDB_WRITEMAP, flag);
        self
    }
    /// Get no-meta-sync option
    pub const fn get_nometasync(&self) -> bool {
        flag_get!(self, MDB_NOMETASYNC)
    }
    /// Set or clear no-meta-sync option
    pub const fn nometasync(mut self, flag: bool) -> Self {
        flag_set!(self, MDB_NOMETASYNC, flag);
        self
    }
    /// Get maximum environment size
    pub const fn get_max_size(&self) -> usize {
        self.max_size
    }
    /// Set maximum environment size
    pub const fn max_size(mut self, max_size: usize) -> Self {
        self.max_size = max_size;
        self
    }
    /// Get maximum number of concurrent readers
    pub const fn get_max_readers(&self) -> usize {
        self.max_readers
    }
    /// Set maximum number of concurrent readers
    pub const fn max_readers(mut self, max_readers: usize) -> Self {
        self.max_readers = max_readers;
        self
    }
    /// Get maximum number of named databases
    pub const fn get_max_dbs(&self) -> usize {
        self.max_dbs
    }
    /// Set maximum number of named databases
    pub const fn max_dbs(mut self, max_dbs: usize) -> Self {
        self.max_dbs = max_dbs;
        self
    }
    /// Get file mode
    pub const fn get_unix_file_mode(&self) -> u32 {
        self.unix_file_mode
    }
    /// Set file mode
    pub const fn unix_file_mode(mut self, mode: u32) -> Self {
        self.unix_file_mode = mode;
        self
    }
}

impl<'a> EnvBuilder<&'a Path> {
    /// Open environment
    unsafe fn open_with_readonly_flag(&self, readonly: bool) -> Result<Arc<EnvBackend>, io::Error> {
        let path = path_to_cstring(self.path)?;
        let mut inner = MaybeUninit::<*mut lmdb::MDB_env>::uninit();
        check_err_code(lmdb::mdb_env_create(inner.as_mut_ptr()))?;
        let inner = inner.assume_init();
        let backend = EnvBackend {
            inner: inner,
            max_keysize: lmdb::mdb_env_get_maxkeysize(inner)
                .try_into()
                .unwrap_or(usize::MAX),
            db_mutex: Default::default(),
        };
        check_err_code(lmdb::mdb_env_set_mapsize(
            backend.inner,
            self.max_size
                .try_into()
                .map_err(|_| io::Error::new(io::ErrorKind::InvalidInput, "maximum size invalid"))?,
        ))?;
        check_err_code(lmdb::mdb_env_set_maxreaders(
            backend.inner,
            self.max_readers.try_into().map_err(|_| {
                io::Error::new(io::ErrorKind::InvalidInput, "maximum reader count invalid")
            })?,
        ))?;
        check_err_code(lmdb::mdb_env_set_maxdbs(
            backend.inner,
            self.max_dbs.try_into().map_err(|_| {
                io::Error::new(
                    io::ErrorKind::InvalidInput,
                    "maximum number of named databases invalid",
                )
            })?,
        ))?;
        check_err_code(lmdb::mdb_env_open(
            backend.inner,
            path.as_ptr(),
            if readonly {
                self.lmdb_flags | lmdb::MDB_RDONLY as LmdbFlags
            } else {
                self.lmdb_flags
            },
            self.unix_file_mode.try_into().map_err(|_| {
                io::Error::new(
                    io::ErrorKind::InvalidInput,
                    "file mode bits integer invalid",
                )
            })?,
        ))?;
        Ok(Arc::new(backend))
    }
    /// Open environment in read-only mode
    ///
    /// # Safety
    ///
    /// * Do not open the same environment twice in the same process.
    /// * The environment must be free of corruption on the storage medium.
    /// * The file format is platform specific, so endianness and word size
    ///   must not change across use.
    pub unsafe fn open_ro(&self) -> Result<EnvRo, io::Error> {
        Ok(EnvRo {
            backend: self.open_with_readonly_flag(true)?,
        })
    }
    /// Open environment in read-write mode
    ///
    /// # Safety
    ///
    /// * Do not open the same environment twice in the same process.
    /// * The environment must be free of corruption on the storage medium.
    /// * The file format is platform specific, so endianness and word size
    ///   must not change across use.
    pub unsafe fn open_rw(&self) -> Result<EnvRw, io::Error> {
        Ok(EnvRw {
            backend: self.open_with_readonly_flag(false)?,
        })
    }
}

impl DbOptions<[u8], [u8], KeysUnique, ()> {
    /// Default options for database
    pub const fn new() -> Self {
        Self {
            key: PhantomData,
            value: PhantomData,
            constraint: KeysUnique,
            lmdb_flags: 0,
            name: (),
        }
    }
}

impl<K: ?Sized, V: ?Sized, C> DbOptions<K, V, C, ()>
where
    C: Constraint,
{
    /// Clear `dupsort` option (also clears `reversedup` option)
    pub const fn keys_unique(mut self) -> DbOptions<K, V, KeysUnique, ()> {
        flag_set!(self, MDB_DUPSORT, false);
        flag_set!(self, MDB_REVERSEDUP, false);
        DbOptions {
            constraint: KeysUnique,
            key: PhantomData,
            value: PhantomData,
            lmdb_flags: self.lmdb_flags,
            name: self.name,
        }
    }
    /// Set `dupsort` option
    pub const fn keys_duplicate(mut self) -> DbOptions<K, V, KeysDuplicate, ()> {
        flag_set!(self, MDB_DUPSORT, true);
        DbOptions {
            constraint: KeysDuplicate,
            key: PhantomData,
            value: PhantomData,
            lmdb_flags: self.lmdb_flags,
            name: self.name,
        }
    }
    /// Set stored key type
    pub const fn key_type<T>(mut self) -> DbOptions<T, V, C, ()>
    where
        T: ?Sized + Storable,
    {
        flag_set!(self, MDB_INTEGERKEY, T::OPTIMIZE_INT);
        DbOptions {
            key: PhantomData,
            value: PhantomData,
            constraint: self.constraint,
            lmdb_flags: self.lmdb_flags,
            name: self.name,
        }
    }
    /// Set stored value type
    pub const fn value_type<T>(mut self) -> DbOptions<K, T, C, ()>
    where
        T: ?Sized + Storable,
    {
        flag_set!(self, MDB_DUPFIXED, T::CONST_BYTES_LEN);
        flag_set!(self, MDB_INTEGERDUP, T::OPTIMIZE_INT);
        DbOptions {
            value: PhantomData,
            key: PhantomData,
            constraint: self.constraint,
            lmdb_flags: self.lmdb_flags,
            name: self.name,
        }
    }
}

impl<K: ?Sized, V: ?Sized, C, N> DbOptions<K, V, C, N>
where
    C: Constraint,
{
    /// Get `reversekey` option
    pub const fn get_reversekey(&self) -> bool {
        flag_get!(self, MDB_REVERSEKEY)
    }
    /// Set or clear `reversekey` option
    pub const fn reversekey(mut self, flag: bool) -> Self {
        flag_set!(self, MDB_REVERSEKEY, flag);
        self
    }
    /// Return sanitized flags for LMDB
    /// (clear flags that are only used for for databases with duplicate keys
    /// if unique keys have been selected)
    const fn cleansed_lmdb_flags(&self) -> LmdbFlags {
        if flag_get!(self, MDB_DUPSORT) {
            self.lmdb_flags
        } else {
            self.lmdb_flags
                & !((lmdb::MDB_REVERSEDUP | lmdb::MDB_DUPFIXED | lmdb::MDB_INTEGERDUP) as LmdbFlags)
        }
    }
    /// Use unnamed database
    ///
    /// Should be used as last builder method because some other builder
    /// methods are not available anymore after calling this method.
    pub fn unnamed(self) -> DbOptions<K, V, C, Option<CString>> {
        DbOptions {
            value: PhantomData,
            key: PhantomData,
            constraint: self.constraint,
            lmdb_flags: self.lmdb_flags,
            name: None,
        }
    }
    /// Use named database (name must not contain a null byte)
    ///
    /// Should be used as last builder method because some other builder
    /// methods are not available anymore after calling this method.
    pub fn name<T: Into<Vec<u8>>>(self, name: T) -> DbOptions<K, V, C, Option<CString>> {
        let name = CString::new(name).expect("database name must not contain NULL byte");
        DbOptions {
            value: PhantomData,
            key: PhantomData,
            constraint: self.constraint,
            lmdb_flags: self.lmdb_flags,
            name: Some(name),
        }
    }
    /// Remove name information
    ///
    /// NOTE: This does not select the unnamed database. To select the unnamed
    /// database, use [`DbOptions::unnamed`].
    pub fn strip_name<T: Into<Vec<u8>>>(self) -> DbOptions<K, V, C, ()> {
        DbOptions {
            value: PhantomData,
            key: PhantomData,
            constraint: self.constraint,
            lmdb_flags: self.lmdb_flags,
            name: (),
        }
    }
}

impl<K: ?Sized, V: ?Sized, C: Constraint, N> DbOptions<K, V, C, N> {
    /// Returns true if duplicate keys are allowed (i.e. is `dupsort` option set?)
    pub fn has_duplicate_keys(&self) -> bool {
        C::DUPLICATE_KEYS
    }
}

impl<K: ?Sized, V: ?Sized, N> DbOptions<K, V, KeysDuplicate, N> {
    /// Get `reversedup` option
    pub const fn get_reversedup(&self) -> bool {
        flag_get!(self, MDB_REVERSEDUP)
    }
    /// Set or clear `reversedup` option
    pub const fn reversedup(mut self, flag: bool) -> Self {
        flag_set!(self, MDB_REVERSEDUP, flag);
        self
    }
}

/// Read-write or read-only handle for accessing environment that stores key-value databases
///
/// Use [`Env::clone_ro`] to create a new read-only handle (from either a
/// read-only or a read-write handle) for accessing the same environment.
/// The environment is closed when the last handle is dropped.
///
/// Use [`Env::open_db`] to retrieve database handles.
///
/// To modify data inside a database, a transaction is needed which can be
/// opened with either [`Env::txn_ro`] (read-only) or [`EnvRw::txn_rw`] (for
/// reading and writing).
pub trait Env {
    /// Start read-only transaction
    fn txn_ro(&self) -> Result<TxnRo<'_>, io::Error>;
    /// Get new read-only handle for environment
    fn clone_ro(&self) -> EnvRo;
    /// Get maximum size of keys and duplicate data
    fn max_keysize(&self) -> usize;
    /// Checks if key or duplicate data has valid size
    fn valid_keysize<K>(&self, key: &K) -> bool
    where
        K: ?Sized + Storable;
    /// Open databases in environment
    ///
    /// SAFETY: If a database exists already, it must have been created with
    /// compatible options.
    unsafe fn open_dbs<'a, K, V, C>(
        &self,
        options: impl IntoIterator<
            IntoIter = impl Iterator<Item = &'a DbOptions<K, V, C, Option<CString>>> + ExactSizeIterator,
        >,
    ) -> Result<Vec<Db<K, V, C>>, io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
        C: Constraint;
    /// Open database in environment
    ///
    /// SAFETY: If a database exists already, it must have been created with
    /// compatible options.
    unsafe fn open_db<K, V, C>(
        &self,
        options: &DbOptions<K, V, C, Option<CString>>,
    ) -> Result<Db<K, V, C>, io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
        C: Constraint,
    {
        Ok(self.open_dbs([options])?.pop().unwrap())
    }
    /// Clear stale readers
    ///
    /// Refer to LMDB's documentation when to clear stale readers
    fn clear_stale_readers(&self) -> Result<(), io::Error>;
}

/// Function passed to [`lmdb::mdb_set_compare`] to compare two keys or values
/// in the database
unsafe extern "C" fn compare_function<T>(a: *const lmdb::MDB_val, b: *const lmdb::MDB_val) -> c_int
where
    T: ?Sized + Storable,
{
    let a: &[u8] = slice::from_raw_parts((*a).mv_data as *const u8, (*a).mv_size);
    let b: &[u8] = slice::from_raw_parts((*b).mv_data as *const u8, (*b).mv_size);
    T::cmp_bytes_unchecked(a, b) as c_int
}

impl EnvBackend {
    /// Start read-only transaction
    fn txn_ro(&self) -> Result<TxnRo<'_>, io::Error> {
        unsafe {
            let mut inner_txn = MaybeUninit::<*mut lmdb::MDB_txn>::uninit();
            check_err_code(lmdb::mdb_txn_begin(
                self.inner,
                null_mut(),
                lmdb::MDB_RDONLY as LmdbFlags,
                inner_txn.as_mut_ptr(),
            ))?;
            let inner_txn = inner_txn.assume_init();
            Ok(TxnRo {
                backend: Mutex::new(TxnBackend {
                    env_backend: &self,
                    inner: inner_txn,
                    cursors: Default::default(),
                }),
            })
        }
    }
    /// Return true if given size (in bytes) is valid for a key (or value in a
    /// database with duplicate keys)
    fn size_valid_keysize(&self, size: usize) -> bool {
        (1..=self.max_keysize).contains(&size)
    }
    /// Return true if given slice has valid size/length as key (or value in a
    /// database with duplicate keys)
    fn valid_keysize(&self, key: &[u8]) -> bool {
        self.size_valid_keysize(key.len())
    }
    /// Return error if slice doesn't have valid size/length as key
    fn assert_valid_keysize(&self, key: &[u8]) -> Result<(), io::Error> {
        if self.valid_keysize(key) {
            Ok(())
        } else {
            Err(io::Error::new(
                io::ErrorKind::InvalidData,
                "invalid key size",
            ))
        }
    }
    /// Return error if slice doesn't have valid size/length as value in a
    /// database with duplicate keys
    fn assert_valid_valuesize(&self, value: &[u8]) -> Result<(), io::Error> {
        if self.valid_keysize(value) {
            Ok(())
        } else {
            Err(io::Error::new(
                io::ErrorKind::InvalidData,
                "invalid value size",
            ))
        }
    }
    /// Opens several databases at once (which must be of the same key and
    /// value type and key constraint)
    ///
    /// SAFETY: If a database exists already, it must have been created with
    /// compatible options.
    unsafe fn open_or_opt_create_dbs<'a, K, V, C>(
        self: &Arc<Self>,
        options: impl IntoIterator<
            IntoIter = impl Iterator<Item = &'a DbOptions<K, V, C, Option<CString>>> + ExactSizeIterator,
        >,
        create: bool,
    ) -> Result<Vec<Db<K, V, C>>, io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
        C: Constraint,
    {
        let options = options.into_iter();
        let mut dbs = Vec::with_capacity(options.len());
        let db_guard = self.db_mutex.lock().unwrap();
        let mut txn_inner = MaybeUninit::<*mut lmdb::MDB_txn>::uninit();
        check_err_code(lmdb::mdb_txn_begin(
            self.inner,
            null_mut(),
            if create {
                0 as LmdbFlags
            } else {
                lmdb::MDB_RDONLY as LmdbFlags
            },
            txn_inner.as_mut_ptr(),
        ))?;
        let txn_inner = txn_inner.assume_init();
        let txn = TxnBackend {
            env_backend: self,
            inner: txn_inner,
            cursors: Default::default(),
        };
        for option in options {
            let mut db_inner = MaybeUninit::<lmdb::MDB_dbi>::uninit();
            let mut flags = option.cleansed_lmdb_flags();
            if create {
                flags |= lmdb::MDB_CREATE as LmdbFlags;
            }
            check_err_code(lmdb::mdb_dbi_open(
                txn.inner,
                option.name.as_ref().map_or(null(), |x| x.as_ptr()),
                flags,
                db_inner.as_mut_ptr(),
            ))?;
            let db_inner = db_inner.assume_init();
            let db_backend = DbBackend {
                env_backend: Arc::downgrade(self),
                inner: db_inner,
            };
            if !K::TRIVIAL_CMP && !K::OPTIMIZE_INT {
                check_err_code(lmdb::mdb_set_compare(
                    txn.inner,
                    db_backend.inner,
                    Some(compare_function::<K>),
                ))?;
            }
            if !V::TRIVIAL_CMP && !V::OPTIMIZE_INT && C::DUPLICATE_KEYS {
                check_err_code(lmdb::mdb_set_dupsort(
                    txn.inner,
                    db_backend.inner,
                    Some(compare_function::<V>),
                ))?;
            }
            dbs.push(Db {
                key: PhantomData,
                value: PhantomData,
                constraint: option.constraint,
                backend: ArcByAddr::new(db_backend),
            });
        }
        txn.commit()?;
        drop(db_guard);
        Ok(dbs)
    }
    /// Call LMDB's `mdb_reader_check` function
    fn clear_stale_readers(&self) -> Result<(), io::Error> {
        unsafe {
            let mut dead = MaybeUninit::<c_int>::uninit();
            check_err_code(lmdb::mdb_reader_check(self.inner, dead.as_mut_ptr()))
        }
    }
}

impl Env for EnvRo {
    fn txn_ro(&self) -> Result<TxnRo<'_>, io::Error> {
        self.backend.txn_ro()
    }
    fn clone_ro(&self) -> EnvRo {
        EnvRo {
            backend: self.backend.clone(),
        }
    }
    fn max_keysize(&self) -> usize {
        self.backend.max_keysize
    }
    fn valid_keysize<K>(&self, key: &K) -> bool
    where
        K: ?Sized + Storable,
    {
        self.backend.size_valid_keysize(key.bytes_len())
    }
    unsafe fn open_dbs<'a, K, V, C>(
        &self,
        options: impl IntoIterator<
            IntoIter = impl Iterator<Item = &'a DbOptions<K, V, C, Option<CString>>> + ExactSizeIterator,
        >,
    ) -> Result<Vec<Db<K, V, C>>, io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
        C: Constraint,
    {
        self.backend.open_or_opt_create_dbs(options, false)
    }
    fn clear_stale_readers(&self) -> Result<(), io::Error> {
        self.backend.clear_stale_readers()
    }
}

impl Env for EnvRw {
    fn txn_ro(&self) -> Result<TxnRo<'_>, io::Error> {
        self.backend.txn_ro()
    }
    fn clone_ro(&self) -> EnvRo {
        EnvRo {
            backend: self.backend.clone(),
        }
    }
    fn max_keysize(&self) -> usize {
        self.backend.max_keysize
    }
    fn valid_keysize<K>(&self, key: &K) -> bool
    where
        K: ?Sized + Storable,
    {
        self.backend.size_valid_keysize(key.bytes_len())
    }
    unsafe fn open_dbs<'a, K, V, C>(
        &self,
        options: impl IntoIterator<
            IntoIter = impl Iterator<Item = &'a DbOptions<K, V, C, Option<CString>>> + ExactSizeIterator,
        >,
    ) -> Result<Vec<Db<K, V, C>>, io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
        C: Constraint,
    {
        self.backend.open_or_opt_create_dbs(options, false)
    }
    fn clear_stale_readers(&self) -> Result<(), io::Error> {
        self.backend.clear_stale_readers()
    }
}

impl EnvRw {
    /// Open databases in environment and create if non-existing
    ///
    /// SAFETY: If a database exists already, it must have been created with
    /// compatible options.
    pub unsafe fn create_dbs<'a, K, V, C>(
        &mut self,
        options: impl IntoIterator<
            IntoIter = impl Iterator<Item = &'a DbOptions<K, V, C, Option<CString>>> + ExactSizeIterator,
        >,
    ) -> Result<Vec<Db<K, V, C>>, io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
        C: Constraint,
    {
        self.backend.open_or_opt_create_dbs(options, true)
    }
    /// Open database in environment and create if non-existing
    ///
    /// SAFETY: If a database exists already, it must have been created with
    /// compatible options.
    pub unsafe fn create_db<K, V, C>(
        &mut self,
        options: &DbOptions<K, V, C, Option<CString>>,
    ) -> Result<Db<K, V, C>, io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
        C: Constraint,
    {
        Ok(self.create_dbs([options])?.pop().unwrap())
    }
    /// Delete database in environment
    pub fn drop_db<K, V, C>(&self, db: Db<K, V, C>) -> Result<(), io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
        C: Constraint,
    {
        let db_backend = ArcByAddr::try_unwrap(db.backend).map_err(|_| {
            io::Error::new(io::ErrorKind::Other, "database in use by current process")
        })?;
        db_backend.assert_env_backend(&self.backend);
        let db_guard = self.backend.db_mutex.lock().unwrap();
        unsafe {
            let mut txn_inner = MaybeUninit::<*mut lmdb::MDB_txn>::uninit();
            check_err_code(lmdb::mdb_txn_begin(
                self.backend.inner,
                null_mut(),
                0 as LmdbFlags,
                txn_inner.as_mut_ptr(),
            ))?;
            let txn_inner = txn_inner.assume_init();
            let txn = TxnBackend {
                env_backend: &self.backend,
                inner: txn_inner,
                cursors: Default::default(),
            };
            let db_inner = db_backend.inner;
            forget(db_backend);
            check_err_code(lmdb::mdb_drop(txn.inner, db_inner, 1))?;
            txn.commit()?;
        }
        drop(db_guard);
        Ok(())
    }
    /// Start read-write transaction
    pub fn txn_rw(&mut self) -> Result<TxnRw<'_>, io::Error> {
        unsafe {
            let mut txn_inner = MaybeUninit::<*mut lmdb::MDB_txn>::uninit();
            check_err_code(lmdb::mdb_txn_begin(
                self.backend.inner,
                null_mut(),
                0,
                txn_inner.as_mut_ptr(),
            ))?;
            let txn_inner = txn_inner.assume_init();
            Ok(TxnRw {
                backend: UnsafeCell::new(TxnBackend {
                    env_backend: &self.backend,
                    inner: txn_inner,
                    cursors: Default::default(),
                }),
                used_dbs: Default::default(),
            })
        }
    }
}

impl<'a> TxnBackend<'a> {
    /// Commit transaction
    fn commit(mut self) -> Result<(), io::Error> {
        unsafe {
            // Close cursors first
            self.close_cursors();
            // Do not run normal drop handler
            // (which would abort the transaction and close cursors)
            let inner = self.inner;
            forget(self);
            // Commit transaction
            check_err_code(lmdb::mdb_txn_commit(inner))?;
        }
        Ok(())
    }
    /// Get reference to value in database
    ///
    /// SAFETY: The value is only valid until end of a read-only transaction or
    /// until a read-write transaction ends or modifies data. The caller has to
    /// ensure that the lifetime `'b` is short enough.
    unsafe fn get_unsafe<'b, K, V, C>(
        &mut self,
        db: &Db<K, V, C>,
        key: &K,
    ) -> Result<Option<V::AlignedRef<'b>>, io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
        C: Constraint,
    {
        db.backend.assert_env_backend(&self.env_backend);
        let key: K::BytesRef<'_> = key.to_bytes();
        let lmdb_key = lmdb::MDB_val {
            mv_size: key.len(),
            mv_data: key.as_ptr() as *mut _,
        };
        let mut lmdb_data = MaybeUninit::<lmdb::MDB_val>::uninit();
        Ok(
            match lmdb::mdb_get(
                self.inner,
                db.backend.inner,
                &lmdb_key as *const _ as *mut lmdb::MDB_val,
                lmdb_data.as_mut_ptr(),
            ) {
                lmdb::MDB_NOTFOUND => None,
                status => {
                    check_err_code(status)?;
                    let lmdb_data = lmdb_data.assume_init();
                    Some(V::from_bytes_unchecked(slice::from_raw_parts(
                        lmdb_data.mv_data as *const u8,
                        lmdb_data.mv_size,
                    )))
                }
            },
        )
    }
    /// Create a new cursor
    fn new_cursor<K, V, C>(&mut self, db: &Db<K, V, C>) -> Result<Cursor<K, V, C>, io::Error>
    where
        K: ?Sized,
        V: ?Sized,
        C: Constraint,
    {
        db.backend.assert_env_backend(&self.env_backend);
        unsafe {
            let mut inner_cursor = MaybeUninit::<*mut lmdb::MDB_cursor>::uninit();
            check_err_code(lmdb::mdb_cursor_open(
                self.inner,
                db.backend.inner,
                inner_cursor.as_mut_ptr(),
            ))?;
            let inner_cursor = inner_cursor.assume_init();
            let cursor_backend = Arc::new(CursorBackend {
                inner: inner_cursor,
                closed: Default::default(),
                inner_txn: self.inner,
            });
            self.cursors.push(Arc::downgrade(&cursor_backend));
            Ok(Cursor {
                db: db.clone(),
                backend: ArcByAddr::from(cursor_backend),
            })
        }
    }
    /// Execute cursor operation
    ///
    /// SAFETY: Any returned keys or values are only valid until end of a
    /// read-only transaction or until a read-write transaction ends or
    /// modifies data. The caller has to ensure that the lifetime `'b` is short
    /// enough.
    ///
    /// The returned key or value may be retrieved by calling an `FnOnce`
    /// closure when needed.
    unsafe fn cursor_op_unsafe<'b, K, V, C>(
        &mut self,
        cursor: &Cursor<K, V, C>,
        key: Option<&K>,
        value: Option<&V>,
        op: lmdb::MDB_cursor_op,
    ) -> Result<
        Option<(
            impl FnOnce() -> K::AlignedRef<'b>,
            impl FnOnce() -> V::AlignedRef<'b>,
        )>,
        io::Error,
    >
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
        C: Constraint,
    {
        cursor.backend.assert_txn_backend(self);
        let mut lmdb_key = match key {
            Some(key) => {
                let key: K::BytesRef<'_> = key.to_bytes();
                lmdb::MDB_val {
                    mv_size: key.len(),
                    mv_data: key.as_ptr() as *mut _,
                }
            }
            None => lmdb::MDB_val {
                mv_size: 0,
                mv_data: null_mut(),
            },
        };
        let mut lmdb_data = match value {
            Some(value) => {
                let value: V::BytesRef<'_> = value.to_bytes();
                lmdb::MDB_val {
                    mv_size: value.len(),
                    mv_data: value.as_ptr() as *mut _,
                }
            }
            None => lmdb::MDB_val {
                mv_size: 0,
                mv_data: null_mut(),
            },
        };
        Ok(
            match lmdb::mdb_cursor_get(cursor.backend.inner, &mut lmdb_key, &mut lmdb_data, op) {
                lmdb::MDB_NOTFOUND => None,
                status => {
                    check_err_code(status)?;
                    Some((
                        move || {
                            K::from_bytes_unchecked(slice::from_raw_parts(
                                lmdb_key.mv_data as *const u8,
                                lmdb_key.mv_size,
                            ))
                        },
                        move || {
                            V::from_bytes_unchecked(slice::from_raw_parts(
                                lmdb_data.mv_data as *const u8,
                                lmdb_data.mv_size,
                            ))
                        },
                    ))
                }
            },
        )
    }
    /// Get number of values for current cursor position
    fn cursor_get_current_value_count<K, V, C>(
        &mut self,
        cursor: &Cursor<K, V, C>,
    ) -> Result<usize, io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
        C: Constraint,
    {
        cursor.backend.assert_txn_backend(self);
        unsafe {
            let mut count = MaybeUninit::<usize>::uninit();
            check_err_code(lmdb::mdb_cursor_count(
                cursor.backend.inner,
                count.as_mut_ptr(),
            ))?;
            Ok(count.assume_init())
        }
    }
}

/// Helper macro to create return type of cursor methods
macro_rules! cursor_return_type {
    (bool $(,)?) => (bool);
    (key $(,)?) => (Option<K::AlignedRef<'_>>);
    (value $(,)?) => (Option<V::AlignedRef<'_>>);
    ((key, value $(,)?)) => (Option<(K::AlignedRef<'_>, V::AlignedRef<'_>)>);
}

/// Create signature for cursor method
macro_rules! cursor_method_def {
    ($($meta:meta)*, $name:ident, (), $retval:tt $(,)?) => {
            $(#[$meta])*
            fn $name<K, V, C>(
                &self,
                cursor: &Cursor<K, V, C>,
            ) -> Result<cursor_return_type!($retval), io::Error>
            where
                K: ?Sized + Storable,
                V: ?Sized + Storable,
                C: Constraint;
    };
    ($($meta:meta)*, $name:ident, (key), $retval:tt $(,)?) => {
            $(#[$meta])*
            fn $name<K, V, C>(
                &self,
                cursor: &Cursor<K, V, C>,
                key: &K,
            ) -> Result<cursor_return_type!($retval), io::Error>
            where
                K: ?Sized + Storable,
                V: ?Sized + Storable,
                C: Constraint;
    };
    ($($meta:meta)*, $name:ident, (key, value), $retval:tt $(,)?) => {
            $(#[$meta])*
            fn $name<K, V, C>(
                &self,
                cursor: &Cursor<K, V, C>,
                key: &K,
                value: &V,
            ) -> Result<cursor_return_type!($retval), io::Error>
            where
                K: ?Sized + Storable,
                V: ?Sized + Storable,
                C: Constraint;
    };
}

/// Create signatures for several cursor methods
macro_rules! cursor_method_defs {
    {
        $(
            $(#[$meta:meta])*
            fn $name:ident $args:tt -> $retval:tt;
        )*
    } => {
        $(
            cursor_method_def!($($meta)*, $name, $args, $retval);
        )*
    };
}

/// Create signature for cursor method on databases with duplicate keys
macro_rules! cursor_method_dupkey_def {
    ($($meta:meta)*, $name:ident, (), $retval:tt $(,)?) => {
            $(#[$meta])*
            fn $name<K, V>(
                &self,
                cursor: &Cursor<K, V, KeysDuplicate>,
            ) -> Result<cursor_return_type!($retval), io::Error>
            where
                K: ?Sized + Storable,
                V: ?Sized + Storable;
    };
    ($($meta:meta)*, $name:ident, (key), $retval:tt $(,)?) => {
            $(#[$meta])*
            fn $name<K, V>(
                &self,
                cursor: &Cursor<K, V, KeysDuplicate>,
                key: &K,
            ) -> Result<cursor_return_type!($retval), io::Error>
            where
                K: ?Sized + Storable,
                V: ?Sized + Storable;
    };
    ($($meta:meta)*, $name:ident, (key, value), $retval:tt $(,)?) => {
            $(#[$meta])*
            fn $name<K, V>(
                &self,
                cursor: &Cursor<K, V, KeysDuplicate>,
                key: &K,
                value: &V,
            ) -> Result<cursor_return_type!($retval), io::Error>
            where
                K: ?Sized + Storable,
                V: ?Sized + Storable;
    };
}

/// Create signatures for several cursor methods on databases with duplicate
/// keys
macro_rules! cursor_method_dupkey_defs {
    {
        $(
            $(#[$meta:meta])*
            fn $name:ident $args:tt -> $retval:tt;
        )*
    } => {
        $(
            cursor_method_dupkey_def!($($meta)*, $name, $args, $retval);
        )*
    };
}

/// Read-write or read-only transaction
///
/// This trait provides the methods for read-access to databases.
/// The simplest interface is provided by [`Txn::get`], which returns a
/// [reference](Storable::AlignedRef) to the stored value for a given key
/// (or [`None`] if the key does not exist).
/// If an owned value is desired, use [`Txn::get_owned`] instead, which
/// automatically converts the reference (e.g. `&str`) into an owned value
/// (e.g. `String`) using [`PointerIntoOwned::into_owned`].
///
/// [`Txn::get`] and [`Txn::get_owned`] will only retrieve the first value in
/// case of [duplicate keys](DbOptions::keys_duplicate). For more sophisticated
/// access, use [`Txn::new_cursor`] to create a new [`Cursor`] which then can
/// be used with one of the cursor methods in the `Txn` trait (or the [`TxnRw`]
/// struct in case of write access, respectively).
pub trait Txn {
    /// Get reference to value in database
    fn get<K, V, C>(
        &self,
        db: &Db<K, V, C>,
        key: &K,
    ) -> Result<Option<V::AlignedRef<'_>>, io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
        C: Constraint;
    /// Get owned value from database
    fn get_owned<'a, K, V, C>(
        &'a self,
        db: &Db<K, V, C>,
        key: &K,
    ) -> Result<Option<<<V as Storable>::AlignedRef<'a> as PointerIntoOwned>::Owned>, io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
        C: Constraint,
    {
        Ok(self.get(db, key)?.map(|x| x.into_owned()))
    }
    /// Create a new cursor
    fn new_cursor<K, V, C>(&self, db: &Db<K, V, C>) -> Result<Cursor<K, V, C>, io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
        C: Constraint;
    /// Get number of values for current cursor position
    fn cursor_get_current_value_count<K, V>(
        &self,
        cursor: &Cursor<K, V, KeysDuplicate>,
    ) -> Result<usize, io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable;
    cursor_method_defs! {
        /// Set cursor to first entry in database
        fn cursor_set_first() -> bool;
        /// Set cursor to first entry in database and get pair
        fn cursor_set_first_get_pair() -> (key, value);
        /// Set cursor to last entry in database
        fn cursor_set_last() -> bool;
        /// Set cursor to last entry in database and get pair
        fn cursor_set_last_get_pair() -> (key, value);
        /// Get key at current cursor position
        fn cursor_get_current_key() -> key;
        /// Get value at current cursor position
        fn cursor_get_current_value() -> value;
        /// Get key-value pair at current cursor position
        fn cursor_get_current_pair() -> (key, value);
        /// Set cursor to key
        fn cursor_set_key(key) -> bool;
        /// Set cursor to key and get value
        fn cursor_set_key_get_value(key) -> value;
        /// Set cursor to key or next greater key if not existent and get pair
        fn cursor_search_key_get_pair(key) -> (key, value);
        /// Move cursor to next entry in database and get pair
        fn cursor_set_next_get_pair() -> (key, value);
        /// Move cursor to previous entry in database and get pair
        fn cursor_set_prev_get_pair() -> (key, value);
        /// Move cursor to first value of next key and get pair
        fn cursor_set_next_key_get_pair() -> (key, value);
        /// Move cursor to last value of previous key and get pair
        fn cursor_set_prev_key_get_pair() -> (key, value);
    }
    cursor_method_dupkey_defs! {
        /// Move cursor to first value of current key
        fn cursor_set_first_value() -> bool;
        /// Move cursor to first value of current key and get value
        fn cursor_set_first_value_get_value() -> value;
        /// Move cursor to last value of current key
        fn cursor_set_last_value() -> bool;
        /// Move cursor to last value of current key and get value
        fn cursor_set_last_value_get_value() -> value;
        /// Set cursor to key-value pair
        fn cursor_set_pair(key, value) -> bool;
        /// Set cursor to key and value or next greater value if not existent and get value
        fn cursor_set_key_search_value_get_value(key, value) -> value;
        /// Move cursor to next value of current key and get value
        fn cursor_set_next_value_get_value() -> value;
        /// Move cursor to next value of current key and get value
        fn cursor_set_prev_value_get_value() -> value;
    }
}

/// Helper macro to extract return value for cursor methods
macro_rules! cursor_return_value {
    (bool, $x:expr $(,)?) => {
        match $x {
            Some(_) => true,
            None => false,
        }
    };
    (key, $x:expr $(,)?) => {
        $x.map(|(k, _)| k())
    };
    (value, $x:expr $(,)?) => {
        $x.map(|(_, v)| v())
    };
    ((key, value), $x:expr $(,)?) => {
        $x.map(|(k, v)| (k(), v()))
    };
}

/// Create implementation for cursor method on [`TxnRo`]
macro_rules! cursor_method_impl_ro {
    ($name:ident, (), $retval:tt, $op:ident $(,)?) => {
        fn $name<K, V, C>(
            &self,
            cursor: &Cursor<K, V, C>,
        ) -> Result<cursor_return_type!($retval), io::Error>
        where
            K: ?Sized + Storable,
            V: ?Sized + Storable,
            C: Constraint,
        {
            let mut backend = self.backend.lock().unwrap();
            Ok(cursor_return_value!($retval, unsafe {
                backend.cursor_op_unsafe(cursor, None, None, lmdb::$op)?
            }))
        }
    };
    ($name:ident, (key), $retval:tt, $op:ident) => {
        fn $name<K, V, C>(
            &self,
            cursor: &Cursor<K, V, C>,
            key: &K,
        ) -> Result<cursor_return_type!($retval), io::Error>
        where
            K: ?Sized + Storable,
            V: ?Sized + Storable,
            C: Constraint,
        {
            let mut backend = self.backend.lock().unwrap();
            Ok(cursor_return_value!($retval, unsafe {
                backend.cursor_op_unsafe(cursor, Some(key), None, lmdb::$op)?
            }))
        }
    };
    ($name:ident, (key, value), $retval:tt, $op:ident) => {
        fn $name<K, V, C>(
            &self,
            cursor: &Cursor<K, V, C>,
            key: &K,
            value: &V,
        ) -> Result<cursor_return_type!($retval), io::Error>
        where
            K: ?Sized + Storable,
            V: ?Sized + Storable,
            C: Constraint,
        {
            let mut backend = self.backend.lock().unwrap();
            Ok(cursor_return_value!($retval, unsafe {
                backend.cursor_op_unsafe(cursor, Some(key), Some(value), lmdb::$op)?
            }))
        }
    };
}

/// Create implementation for cursor method on [`TxnRo`] for databases with
/// duplicate keys
macro_rules! cursor_method_dupkey_impl_ro {
    ($name:ident, (), $retval:tt, $op:ident $(,)?) => {
        fn $name<K, V>(
            &self,
            cursor: &Cursor<K, V, KeysDuplicate>,
        ) -> Result<cursor_return_type!($retval), io::Error>
        where
            K: ?Sized + Storable,
            V: ?Sized + Storable,
        {
            let mut backend = self.backend.lock().unwrap();
            Ok(cursor_return_value!($retval, unsafe {
                backend.cursor_op_unsafe(cursor, None, None, lmdb::$op)?
            }))
        }
    };
    ($name:ident, (key), $retval:tt, $op:ident) => {
        fn $name<K, V>(
            &self,
            cursor: &Cursor<K, V, KeysDuplicate>,
            key: &K,
        ) -> Result<cursor_return_type!($retval), io::Error>
        where
            K: ?Sized + Storable,
            V: ?Sized + Storable,
        {
            let mut backend = self.backend.lock().unwrap();
            Ok(cursor_return_value!($retval, unsafe {
                backend.cursor_op_unsafe(cursor, Some(key), None, lmdb::$op)?
            }))
        }
    };
    ($name:ident, (key, value), $retval:tt, $op:ident) => {
        fn $name<K, V>(
            &self,
            cursor: &Cursor<K, V, KeysDuplicate>,
            key: &K,
            value: &V,
        ) -> Result<cursor_return_type!($retval), io::Error>
        where
            K: ?Sized + Storable,
            V: ?Sized + Storable,
        {
            let mut backend = self.backend.lock().unwrap();
            Ok(cursor_return_value!($retval, unsafe {
                backend.cursor_op_unsafe(cursor, Some(key), Some(value), lmdb::$op)?
            }))
        }
    };
}

/// Create implementation for cursor method on [`TxnRw`]
macro_rules! cursor_method_impl_rw {
    ($name:ident, (), $retval:tt, $op:ident $(,)?) => {
        fn $name<K, V, C>(
            &self,
            cursor: &Cursor<K, V, C>,
        ) -> Result<cursor_return_type!($retval), io::Error>
        where
            K: ?Sized + Storable,
            V: ?Sized + Storable,
            C: Constraint,
        {
            Ok(cursor_return_value!($retval, unsafe {
                let backend = &mut *self.backend.get();
                backend.cursor_op_unsafe(cursor, None, None, lmdb::$op)?
            }))
        }
    };
    ($name:ident, (key), $retval:tt, $op:ident) => {
        fn $name<K, V, C>(
            &self,
            cursor: &Cursor<K, V, C>,
            key: &K,
        ) -> Result<cursor_return_type!($retval), io::Error>
        where
            K: ?Sized + Storable,
            V: ?Sized + Storable,
            C: Constraint,
        {
            Ok(cursor_return_value!($retval, unsafe {
                let backend = &mut *self.backend.get();
                backend.cursor_op_unsafe(cursor, Some(key), None, lmdb::$op)?
            }))
        }
    };
    ($name:ident, (key, value), $retval:tt, $op:ident) => {
        fn $name<K, V, C>(
            &self,
            cursor: &Cursor<K, V, C>,
            key: &K,
            value: &V,
        ) -> Result<cursor_return_type!($retval), io::Error>
        where
            K: ?Sized + Storable,
            V: ?Sized + Storable,
            C: Constraint,
        {
            Ok(cursor_return_value!($retval, unsafe {
                let backend = &mut *self.backend.get();
                backend.cursor_op_unsafe(cursor, Some(key), Some(value), lmdb::$op)?
            }))
        }
    };
}

/// Create implementation for cursor method on [`TxnRw`] for databases with
/// duplicate keys
macro_rules! cursor_method_dupkey_impl_rw {
    ($name:ident, (), $retval:tt, $op:ident $(,)?) => {
        fn $name<K, V>(
            &self,
            cursor: &Cursor<K, V, KeysDuplicate>,
        ) -> Result<cursor_return_type!($retval), io::Error>
        where
            K: ?Sized + Storable,
            V: ?Sized + Storable,
        {
            Ok(cursor_return_value!($retval, unsafe {
                let backend = &mut *self.backend.get();
                backend.cursor_op_unsafe(cursor, None, None, lmdb::$op)?
            }))
        }
    };
    ($name:ident, (key), $retval:tt, $op:ident) => {
        fn $name<K, V>(
            &self,
            cursor: &Cursor<K, V, KeysDuplicate>,
            key: &K,
        ) -> Result<cursor_return_type!($retval), io::Error>
        where
            K: ?Sized + Storable,
            V: ?Sized + Storable,
        {
            Ok(cursor_return_value!($retval, unsafe {
                let backend = &mut *self.backend.get();
                backend.cursor_op_unsafe(cursor, Some(key), None, lmdb::$op)?
            }))
        }
    };
    ($name:ident, (key, value), $retval:tt, $op:ident) => {
        fn $name<K, V>(
            &self,
            cursor: &Cursor<K, V, KeysDuplicate>,
            key: &K,
            value: &V,
        ) -> Result<cursor_return_type!($retval), io::Error>
        where
            K: ?Sized + Storable,
            V: ?Sized + Storable,
        {
            Ok(cursor_return_value!($retval, unsafe {
                let backend = &mut *self.backend.get();
                backend.cursor_op_unsafe(cursor, Some(key), Some(value), lmdb::$op)?
            }))
        }
    };
}

/// Create implementations for all cursor methods
macro_rules! cursor_method_impls {
    ($macro:ident, $dupkey_macro:ident) => {
        $macro!(cursor_set_first, (), bool, MDB_cursor_op_MDB_FIRST);
        $macro!(
            cursor_set_first_get_pair,
            (),
            (key, value),
            MDB_cursor_op_MDB_FIRST
        );
        $macro!(cursor_set_last, (), bool, MDB_cursor_op_MDB_LAST);
        $macro!(
            cursor_set_last_get_pair,
            (),
            (key, value),
            MDB_cursor_op_MDB_LAST
        );
        $macro!(
            cursor_get_current_key,
            (),
            key,
            MDB_cursor_op_MDB_GET_CURRENT
        );
        $macro!(
            cursor_get_current_value,
            (),
            value,
            MDB_cursor_op_MDB_GET_CURRENT
        );
        $macro!(
            cursor_get_current_pair,
            (),
            (key, value),
            MDB_cursor_op_MDB_GET_CURRENT
        );
        $macro!(cursor_set_key, (key), bool, MDB_cursor_op_MDB_SET);
        $macro!(
            cursor_set_key_get_value,
            (key),
            value,
            MDB_cursor_op_MDB_SET
        );
        $macro!(
            cursor_search_key_get_pair,
            (key),
            (key, value),
            MDB_cursor_op_MDB_SET_RANGE
        );
        $macro!(
            cursor_set_next_get_pair,
            (),
            (key, value),
            MDB_cursor_op_MDB_NEXT
        );
        $macro!(
            cursor_set_prev_get_pair,
            (),
            (key, value),
            MDB_cursor_op_MDB_PREV
        );
        $macro!(
            cursor_set_next_key_get_pair,
            (),
            (key, value),
            MDB_cursor_op_MDB_NEXT_NODUP
        );
        $macro!(
            cursor_set_prev_key_get_pair,
            (),
            (key, value),
            MDB_cursor_op_MDB_PREV_NODUP
        );
        $dupkey_macro!(
            cursor_set_first_value,
            (),
            bool,
            MDB_cursor_op_MDB_FIRST_DUP
        );
        $dupkey_macro!(
            cursor_set_first_value_get_value,
            (),
            value,
            MDB_cursor_op_MDB_FIRST_DUP
        );
        $dupkey_macro!(cursor_set_last_value, (), bool, MDB_cursor_op_MDB_LAST_DUP);
        $dupkey_macro!(
            cursor_set_last_value_get_value,
            (),
            value,
            MDB_cursor_op_MDB_LAST_DUP
        );
        $dupkey_macro!(
            cursor_set_pair,
            (key, value),
            bool,
            MDB_cursor_op_MDB_GET_BOTH
        );
        $dupkey_macro!(
            cursor_set_key_search_value_get_value,
            (key, value),
            value,
            MDB_cursor_op_MDB_GET_BOTH_RANGE
        );
        $dupkey_macro!(
            cursor_set_next_value_get_value,
            (),
            value,
            MDB_cursor_op_MDB_NEXT_DUP
        );
        $dupkey_macro!(
            cursor_set_prev_value_get_value,
            (),
            value,
            MDB_cursor_op_MDB_PREV_DUP
        );
    };
}

impl<'a> Txn for TxnRo<'a> {
    fn get<K, V, C>(
        &self,
        db: &Db<K, V, C>,
        key: &K,
    ) -> Result<Option<V::AlignedRef<'_>>, io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
        C: Constraint,
    {
        let mut backend = self.backend.lock().unwrap();
        unsafe { backend.get_unsafe::<K, V, C>(db, key) }
    }
    fn new_cursor<K, V, C>(&self, db: &Db<K, V, C>) -> Result<Cursor<K, V, C>, io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
        C: Constraint,
    {
        self.backend.lock().unwrap().new_cursor(db)
    }
    fn cursor_get_current_value_count<K, V>(
        &self,
        cursor: &Cursor<K, V, KeysDuplicate>,
    ) -> Result<usize, io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
    {
        self.backend
            .lock()
            .unwrap()
            .cursor_get_current_value_count(cursor)
    }
    cursor_method_impls!(cursor_method_impl_ro, cursor_method_dupkey_impl_ro);
}

impl<'a> Txn for TxnRw<'a> {
    fn get<K, V, C>(
        &self,
        db: &Db<K, V, C>,
        key: &K,
    ) -> Result<Option<V::AlignedRef<'_>>, io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
        C: Constraint,
    {
        unsafe {
            let backend = &mut *self.backend.get();
            backend.get_unsafe::<K, V, C>(db, key)
        }
    }
    fn new_cursor<K, V, C>(&self, db: &Db<K, V, C>) -> Result<Cursor<K, V, C>, io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
        C: Constraint,
    {
        let backend = unsafe { &mut *self.backend.get() };
        backend.new_cursor(db)
    }
    fn cursor_get_current_value_count<K, V>(
        &self,
        cursor: &Cursor<K, V, KeysDuplicate>,
    ) -> Result<usize, io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
    {
        let backend = unsafe { &mut *self.backend.get() };
        backend.cursor_get_current_value_count(cursor)
    }
    cursor_method_impls!(cursor_method_impl_rw, cursor_method_dupkey_impl_rw);
}

impl<'a> TxnRw<'a> {
    /// Commit transaction
    pub fn commit(self) -> Result<(), io::Error> {
        self.backend.into_inner().commit()
    }
    /// Abort transaction (same as [`drop`])
    pub fn abort(self) {}
    fn put_with_flags<K, V, C>(
        &mut self,
        db: &Db<K, V, C>,
        key: &K,
        value: &V,
        flags: LmdbFlags,
    ) -> Result<bool, io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
        C: Constraint,
    {
        let backend = self.backend.get_mut();
        db.backend.assert_env_backend(&backend.env_backend);
        let key: K::BytesRef<'_> = key.to_bytes();
        let value: V::BytesRef<'_> = value.to_bytes();
        backend.env_backend.assert_valid_keysize(&key)?;
        if C::DUPLICATE_KEYS {
            backend.env_backend.assert_valid_valuesize(&value)?;
        }
        self.used_dbs.get_or_insert_owned(&db.backend);
        let lmdb_key = lmdb::MDB_val {
            mv_size: key.len(),
            mv_data: key.as_ptr() as *mut _,
        };
        let mut lmdb_data = lmdb::MDB_val {
            mv_size: value.len(),
            mv_data: value.as_ptr() as *mut _,
        };
        Ok(
            match unsafe {
                lmdb::mdb_put(
                    backend.inner,
                    db.backend.inner,
                    &lmdb_key as *const _ as *mut lmdb::MDB_val,
                    &mut lmdb_data,
                    flags,
                )
            } {
                lmdb::MDB_KEYEXIST => false,
                status => {
                    check_err_code(status)?;
                    true
                }
            },
        )
    }
    /// Put value into database
    pub fn put<K, V, C>(&mut self, db: &Db<K, V, C>, key: &K, value: &V) -> Result<(), io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
        C: Constraint,
    {
        self.put_with_flags(db, key, value, 0)?;
        Ok(())
    }
    /// Put value into database unless key exists
    pub fn put_unless_key_exists<K, V, C>(
        &mut self,
        db: &Db<K, V, C>,
        key: &K,
        value: &V,
    ) -> Result<bool, io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
        C: Constraint,
    {
        self.put_with_flags(db, key, value, lmdb::MDB_NOOVERWRITE)
    }
    /// Put value into database unless key-value pair exists
    pub fn put_unless_pair_exists<K, V>(
        &mut self,
        db: &Db<K, V, KeysDuplicate>,
        key: &K,
        value: &V,
    ) -> Result<bool, io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
    {
        self.put_with_flags(db, key, value, lmdb::MDB_NODUPDATA)
    }
    /// Delete all values from database that match a given key
    pub fn delete_key<K, V, C>(&mut self, db: &Db<K, V, C>, key: &K) -> Result<bool, io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
        C: Constraint,
    {
        let backend = self.backend.get_mut();
        db.backend.assert_env_backend(&backend.env_backend);
        let key: K::BytesRef<'_> = key.to_bytes();
        self.used_dbs.get_or_insert_owned(&db.backend);
        let lmdb_key = lmdb::MDB_val {
            mv_size: key.len(),
            mv_data: key.as_ptr() as *mut _,
        };
        Ok(
            match unsafe {
                lmdb::mdb_del(
                    backend.inner,
                    db.backend.inner,
                    &lmdb_key as *const _ as *mut lmdb::MDB_val,
                    null_mut(),
                )
            } {
                lmdb::MDB_NOTFOUND => false,
                status => {
                    check_err_code(status)?;
                    true
                }
            },
        )
    }
    /// Delete key-value pair from database
    pub fn delete_pair<K, V>(
        &mut self,
        db: &Db<K, V, KeysDuplicate>,
        key: &K,
        value: &V,
    ) -> Result<bool, io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
    {
        let backend = self.backend.get_mut();
        db.backend.assert_env_backend(&backend.env_backend);
        let key: K::BytesRef<'_> = key.to_bytes();
        let value: V::BytesRef<'_> = value.to_bytes();
        self.used_dbs.get_or_insert_owned(&db.backend);
        let lmdb_key = lmdb::MDB_val {
            mv_size: key.len(),
            mv_data: key.as_ptr() as *mut _,
        };
        let lmdb_value = lmdb::MDB_val {
            mv_size: value.len(),
            mv_data: value.as_ptr() as *mut _,
        };
        Ok(
            match unsafe {
                lmdb::mdb_del(
                    backend.inner,
                    db.backend.inner,
                    &lmdb_key as *const _ as *mut lmdb::MDB_val,
                    &lmdb_value as *const _ as *mut lmdb::MDB_val,
                )
            } {
                lmdb::MDB_NOTFOUND => false,
                status => {
                    check_err_code(status)?;
                    true
                }
            },
        )
    }
    /// Delete key-value pair at current cursor position
    pub fn cursor_delete_current<K, V, C>(
        &mut self,
        cursor: &Cursor<K, V, C>,
    ) -> Result<(), io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
        C: Constraint,
    {
        unsafe {
            let backend = &mut *self.backend.get();
            cursor.backend.assert_txn_backend(backend);
            check_err_code(lmdb::mdb_cursor_del(cursor.backend.inner, 0))
        }
    }
    /// Delete all values with same key at current cursor position
    pub fn cursor_delete_current_key<K, V>(
        &mut self,
        cursor: &Cursor<K, V, KeysDuplicate>,
    ) -> Result<(), io::Error>
    where
        K: ?Sized + Storable,
        V: ?Sized + Storable,
    {
        unsafe {
            let backend = &mut *self.backend.get();
            cursor.backend.assert_txn_backend(backend);
            check_err_code(lmdb::mdb_cursor_del(
                cursor.backend.inner,
                lmdb::MDB_NODUPDATA as LmdbFlags,
            ))
        }
    }
}

impl<K, V, C> Cursor<K, V, C> {
    /// Associated database
    pub fn db(&self) -> &Db<K, V, C> {
        &self.db
    }
}

/// Version of underlying LMDB library
#[derive(Eq, PartialEq, Clone, Debug)]
pub struct LmdbVersion {
    /// Descriptive string
    pub string: &'static str,
    /// Major version number
    pub major: i32,
    /// Minor version number
    pub minor: i32,
    /// Patch version number
    pub patch: i32,
}

/// Retrieve version of underlying LMDB library
pub fn lmdb_version() -> LmdbVersion {
    let mut major: c_int = 0;
    let mut minor: c_int = 0;
    let mut patch: c_int = 0;
    let string: &CStr = unsafe {
        CStr::from_ptr(lmdb::mdb_version(
            &mut major as *mut c_int,
            &mut minor as *mut c_int,
            &mut patch as *mut c_int,
        ))
    };
    LmdbVersion {
        string: string.to_str().unwrap(),
        major: major.try_into().unwrap(),
        minor: minor.try_into().unwrap(),
        patch: patch.try_into().unwrap(),
    }
}
