//! Private module for miscellaneous helper functions

use std::ffi::{CString, NulError};
use std::path::Path;

/// Helper trait implemented for every ([`Sized`]) type argument
pub trait IsType<T> {
    /// Returns value with its type being same as the type argument
    fn identity(self) -> T;
}

impl<T> IsType<T> for T {
    fn identity(self) -> T {
        self
    }
}

#[cfg(unix)]
/// Fallible conversion from [`Path`] to [`CString`]
pub fn path_to_cstring(path: &Path) -> Result<CString, NulError> {
    use std::os::unix::ffi::OsStrExt;
    CString::new(path.as_os_str().as_bytes())
}

#[cfg(not(unix))]
/// Fallible conversion from [`Path`] to [`CString`]
pub fn path_to_cstring(path: &Path) -> Result<CString, NulError> {
    CString::new(path.as_os_str().to_string_lossy().into_owned().into_bytes())
}
