use super::*;
use tempfile::tempdir;

#[test]
fn test_lmdb_version() {
    lmdb_version();
}

#[test]
fn test_open_dir() {
    let td = tempdir().unwrap();
    unsafe { EnvBuilder::new().dir(td.path()).open_rw().unwrap() };
    td.close().unwrap();
}

#[test]
fn test_open_file() {
    let td = tempdir().unwrap();
    let mut path = td.path().to_path_buf();
    path.push("filename");
    unsafe {
        EnvBuilder::new().file(&path).open_rw().unwrap();
    }
    td.close().unwrap();
}

#[test]
fn test_create_and_drop_db() {
    let td = tempdir().unwrap();
    let mut env = unsafe {
        EnvBuilder::new()
            .dir(td.path())
            .max_dbs(1)
            .open_rw()
            .unwrap()
    };
    let db_options = DbOptions::new().name("Abc");
    let db = unsafe { env.create_db(&db_options).unwrap() };
    env.drop_db(db).unwrap();
    td.close().unwrap();
}

#[test]
fn test_put_and_get_str() {
    let db_opts = DbOptions::new()
        .key_type::<str>()
        .value_type::<str>()
        .unnamed();
    let td = tempdir().unwrap();
    let env_builder = EnvBuilder::new().dir(td.path());
    {
        let mut env = unsafe { env_builder.open_rw() }.unwrap();
        let db = unsafe { env.create_db(&db_opts).unwrap() };
        let mut txn = env.txn_rw().unwrap();
        txn.put(&db, "key1", "value1").unwrap();
        txn.commit().unwrap();
    }
    {
        let env = unsafe { env_builder.open_ro() }.unwrap();
        let db = unsafe { env.open_db(&db_opts).unwrap() };
        let txn = env.txn_ro().unwrap();
        assert_eq!(txn.get(&db, "key1").unwrap(), Some("value1"));
    }
    td.close().unwrap();
}

#[test]
fn test_put_and_get_integer() {
    let db_opts = DbOptions::new()
        .key_type::<str>()
        .value_type::<c_uint>()
        .unnamed();
    let td = tempdir().unwrap();
    let env_builder = EnvBuilder::new().dir(td.path());
    {
        let mut env = unsafe { env_builder.open_rw() }.unwrap();
        let db = unsafe { env.create_db(&db_opts).unwrap() };
        let mut txn = env.txn_rw().unwrap();
        txn.put(&db, "key1", &17).unwrap();
        txn.commit().unwrap();
    }
    {
        let env = unsafe { env_builder.open_ro() }.unwrap();
        let db = unsafe { env.open_db(&db_opts).unwrap() };
        let txn = env.txn_ro().unwrap();
        assert_eq!(txn.get_owned(&db, "key1").unwrap(), Some(17));
    }
    td.close().unwrap();
}

#[test]
fn test_put_and_get_constsize_pair() {
    let db_opts = DbOptions::new()
        .key_type::<str>()
        .value_type::<(i64, i32)>()
        .unnamed();
    let td = tempdir().unwrap();
    let env_builder = EnvBuilder::new().dir(td.path());
    {
        let mut env = unsafe { env_builder.open_rw() }.unwrap();
        let db = unsafe { env.create_db(&db_opts).unwrap() };
        let mut txn = env.txn_rw().unwrap();
        txn.put(&db, "key1", &(17, 19)).unwrap();
        txn.commit().unwrap();
    }
    {
        let env = unsafe { env_builder.open_ro() }.unwrap();
        let db = unsafe { env.open_db(&db_opts).unwrap() };
        let txn = env.txn_ro().unwrap();
        assert_eq!(txn.get_owned(&db, "key1").unwrap(), Some((17, 19)));
    }
    td.close().unwrap();
}

#[test]
fn test_put_and_get_varsize_pair() {
    let db_opts = DbOptions::new()
        .key_type::<str>()
        .value_type::<(i16, String)>()
        .unnamed();
    let td = tempdir().unwrap();
    let env_builder = EnvBuilder::new().dir(td.path());
    {
        let mut env = unsafe { env_builder.open_rw() }.unwrap();
        let db = unsafe { env.create_db(&db_opts).unwrap() };
        let mut txn = env.txn_rw().unwrap();
        txn.put(&db, "key1", &(3, "three".to_string())).unwrap();
        txn.commit().unwrap();
    }
    {
        let env = unsafe { env_builder.open_ro() }.unwrap();
        let db = unsafe { env.open_db(&db_opts).unwrap() };
        let txn = env.txn_ro().unwrap();
        assert_eq!(
            txn.get_owned(&db, "key1").unwrap(),
            Some((3, "three".to_string()))
        );
    }
    td.close().unwrap();
}

#[test]
fn test_put_and_get_boolslice() {
    let db_opts = DbOptions::new()
        .key_type::<[bool]>()
        .value_type::<[bool]>()
        .unnamed();
    let td = tempdir().unwrap();
    let env_builder = EnvBuilder::new().dir(td.path());
    {
        let mut env = unsafe { env_builder.open_rw() }.unwrap();
        let db = unsafe { env.create_db(&db_opts).unwrap() };
        let mut txn = env.txn_rw().unwrap();
        txn.put(&db, &[true, true, false], &[false, true, false, true])
            .unwrap();
        txn.commit().unwrap();
    }
    {
        let env = unsafe { env_builder.open_ro() }.unwrap();
        let db = unsafe { env.open_db(&db_opts).unwrap() };
        let txn = env.txn_ro().unwrap();
        assert_eq!(
            txn.get(&db, &[true, true, false]).unwrap(),
            Some(&*vec![false, true, false, true])
        );
        assert_eq!(txn.get(&db, &[true, true]).unwrap(), None);
    }
    td.close().unwrap();
}

#[test]
fn test_put_and_get_slice() {
    let db_opts = DbOptions::new()
        .key_type::<str>()
        .value_type::<[i32]>()
        .unnamed();
    let td = tempdir().unwrap();
    let env_builder = EnvBuilder::new().dir(td.path());
    {
        let mut env = unsafe { env_builder.open_rw() }.unwrap();
        let db = unsafe { env.create_db(&db_opts).unwrap() };
        let mut txn = env.txn_rw().unwrap();
        txn.put(&db, "key1", &[12, 14]).unwrap();
        txn.commit().unwrap();
    }
    {
        let env = unsafe { env_builder.open_ro() }.unwrap();
        let db = unsafe { env.open_db(&db_opts).unwrap() };
        let txn = env.txn_ro().unwrap();
        assert_eq!(txn.get(&db, "key1").unwrap().unwrap()[0], 12);
    }
    td.close().unwrap();
}

#[test]
fn test_cursor() {
    let db_opts = DbOptions::new()
        .key_type::<str>()
        .value_type::<str>()
        .unnamed();
    let td = tempdir().unwrap();
    let env_builder = EnvBuilder::new().dir(td.path());
    {
        let mut env = unsafe { env_builder.open_rw() }.unwrap();
        let db = unsafe { env.create_db(&db_opts).unwrap() };
        let mut txn = env.txn_rw().unwrap();
        txn.put(&db, "key1", "value1").unwrap();
        txn.put(&db, "key2", "value2").unwrap();
        txn.commit().unwrap();
    }
    {
        let env = unsafe { env_builder.open_ro() }.unwrap();
        let db = unsafe { env.open_db(&db_opts).unwrap() };
        let txn = env.txn_ro().unwrap();
        let cursor = txn.new_cursor(&db).unwrap();
        assert_eq!(
            txn.cursor_set_key_get_value(&cursor, "key1").unwrap(),
            Some("value1")
        );
        assert_eq!(
            txn.cursor_set_next_get_pair(&cursor).unwrap(),
            Some(("key2", "value2"))
        );
    }
    td.close().unwrap();
}

#[test]
fn test_cursor_dupkey() {
    let db_opts = DbOptions::new()
        .key_type::<str>()
        .value_type::<str>()
        .keys_duplicate()
        .unnamed();
    let td = tempdir().unwrap();
    let env_builder = EnvBuilder::new().dir(td.path());
    {
        let mut env = unsafe { env_builder.open_rw() }.unwrap();
        let db = unsafe { env.create_db(&db_opts).unwrap() };
        let mut txn = env.txn_rw().unwrap();
        txn.put(&db, "key1", "value1").unwrap();
        txn.put(&db, "key1", "value2").unwrap();
        txn.put(&db, "key2", "value").unwrap();
        txn.commit().unwrap();
    }
    {
        let env = unsafe { env_builder.open_ro() }.unwrap();
        let db = unsafe { env.open_db(&db_opts).unwrap() };
        let txn = env.txn_ro().unwrap();
        let cursor = txn.new_cursor(&db).unwrap();
        assert!(!txn.cursor_set_pair(&cursor, "key1", "value0").unwrap());
        assert!(txn.cursor_set_pair(&cursor, "key1", "value2").unwrap());
        assert_eq!(
            txn.cursor_get_current_pair(&cursor).unwrap(),
            Some(("key1", "value2"))
        );
        assert_eq!(
            txn.cursor_set_first_value_get_value(&cursor).unwrap(),
            Some("value1")
        );
        assert_eq!(
            txn.cursor_set_next_key_get_pair(&cursor).unwrap(),
            Some(("key2", "value"))
        );
        assert_eq!(
            txn.cursor_set_prev_key_get_pair(&cursor).unwrap(),
            Some(("key1", "value2"))
        );
        assert_eq!(txn.cursor_set_next_value_get_value(&cursor).unwrap(), None);
    }
    td.close().unwrap();
}

#[test]
fn test_cursor_delete() {
    let db_opts = DbOptions::new()
        .key_type::<str>()
        .value_type::<str>()
        .keys_duplicate()
        .unnamed();
    let td = tempdir().unwrap();
    let env_builder = EnvBuilder::new().dir(td.path());
    {
        let mut env = unsafe { env_builder.open_rw() }.unwrap();
        let db = unsafe { env.create_db(&db_opts).unwrap() };
        let mut txn = env.txn_rw().unwrap();
        txn.put(&db, "key1", "value1a").unwrap();
        txn.put(&db, "key1", "value1b").unwrap();
        txn.put(&db, "key2", "value2a").unwrap();
        txn.put(&db, "key2", "value2b").unwrap();
        txn.commit().unwrap();
        let mut txn = env.txn_rw().unwrap();
        let cursor = txn.new_cursor(&db).unwrap();
        assert!(txn.cursor_set_pair(&cursor, "key1", "value1b").unwrap());
        txn.cursor_delete_current_key(&cursor).unwrap();
        assert!(txn.cursor_set_pair(&cursor, "key2", "value2a").unwrap());
        txn.cursor_delete_current(&cursor).unwrap();
        assert!(!txn.cursor_set_pair(&cursor, "key1", "value1a").unwrap());
        assert!(!txn.cursor_set_pair(&cursor, "key1", "value1b").unwrap());
        assert!(!txn.cursor_set_pair(&cursor, "key2", "value2a").unwrap());
        assert!(txn.cursor_set_pair(&cursor, "key2", "value2b").unwrap());
    }
    td.close().unwrap();
}

#[test]
#[should_panic(expected = "database does not belong to environment")]
fn test_wrong_db() {
    let db_opts = DbOptions::new()
        .key_type::<str>()
        .value_type::<str>()
        .keys_duplicate()
        .name("mydb");
    let mut env1 = unsafe {
        EnvBuilder::new()
            .dir(tempdir().unwrap().path())
            .max_dbs(1)
            .open_rw()
            .unwrap()
    };
    let mut env2 = unsafe {
        EnvBuilder::new()
            .dir(tempdir().unwrap().path())
            .max_dbs(1)
            .open_rw()
            .unwrap()
    };
    let db = unsafe { env1.create_db(&db_opts).unwrap() };
    let mut txn = env2.txn_rw().unwrap();
    txn.put(&db, "key", "value").unwrap();
}

#[test]
#[should_panic(expected = "cursor does not belong to transaction")]
fn test_cursor_wrong_txn() {
    let db_opts = DbOptions::new()
        .key_type::<str>()
        .value_type::<str>()
        .keys_duplicate()
        .unnamed();
    let td = tempdir().unwrap();
    let env_builder = EnvBuilder::new().dir(td.path());
    let mut env = unsafe { env_builder.open_rw() }.unwrap();
    let db = unsafe { env.create_db(&db_opts).unwrap() };
    let txn1 = env.txn_rw().unwrap();
    let cursor = txn1.new_cursor(&db).unwrap();
    txn1.abort();
    let txn2 = env.txn_rw().unwrap();
    txn2.cursor_set_first(&cursor).unwrap();
}
