fn get_env(key: &str) -> Option<String> {
    println!("cargo:rerun-if-env-changed={}", key);
    match std::env::var(key) {
        Ok(val) => Some(val),
        Err(std::env::VarError::NotPresent) => None,
        Err(err) => panic!("could not get env \"{}\": {}", key, err),
    }
}

fn get_env_default<'a>(key: &'_ str, default: &'a str) -> std::borrow::Cow<'a, str> {
    match get_env(key) {
        Some(val) => std::borrow::Cow::Owned(val),
        None => std::borrow::Cow::Borrowed(default),
    }
}

fn main() {
    // get configuration from environment
    let lmdb_include = get_env("LMDB_INCLUDE");
    let lmdb_lib = get_env("LMDB_LIB");
    let lmdb_libname = get_env_default("LMDB_LIBNAME", "lmdb");

    // create automatic bindings
    {
        let mut builder = bindgen::Builder::default()
            .header_contents("lmdb-bindgen.h", "#include <lmdb.h>\n")
            .parse_callbacks(Box::new(bindgen::CargoCallbacks))
            .size_t_is_usize(true);
        if let Some(dir) = &lmdb_include {
            builder = builder.clang_arg(format!("-I{dir}"));
        }
        builder
            .generate()
            .expect("unable to generate bindings")
            .write_to_file(
                std::path::PathBuf::from(std::env::var("OUT_DIR").unwrap()).join("lmdb.rs"),
            )
            .expect("unable to write bindings");
    }

    // link with LMDB
    if let Some(dir) = lmdb_lib {
        println!("cargo:rustc-link-search=native={}", dir);
    }
    println!("cargo:rustc-link-lib={}", lmdb_libname);
}
