// Adapted from test dataprovided by Melexis, copyright notice reproduced below.

// @copyright (C) 2017 Melexis N.V.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#![allow(clippy::excessive_precision)]

use crate::{mlx90640, MelexisCamera};

/// Raw EEPROM dump
///
/// As given on the "Input data" sheet of the example data spreadsheet, on row 4.
pub const EEPROM_DATA: &[u8] = b"\
    \x00\xAE\x49\x9A\x00\x00\x20\x61\x00\x05\x03\x20\x03\xE0\x17\x10\
    \xA2\x24\x01\x85\x04\x99\x00\x00\x19\x01\x00\x00\x00\x00\xB5\x33\
    \x42\x10\xFF\xC2\x02\x02\x02\x02\xF2\x02\xF1\xF2\xD1\xE1\xAF\xC0\
    \xFF\x00\xF0\x02\xF1\x03\xE1\x03\xE1\xF5\xD1\xE4\xC1\xD5\x91\xC2\
    \x88\x95\x30\xD9\xED\xCB\x11\x0F\x33\x22\x22\x33\x00\x11\xCC\xEE\
    \xFF\xED\x11\x00\x22\x22\x33\x33\x22\x33\x00\x22\xDE\xF0\x9A\xCC\
    \x15\xCC\x2F\xA4\x25\x55\x9C\x78\x76\x66\x01\xC8\x3B\x38\x35\x34\
    \x24\x52\x04\x63\x13\xBB\x06\x23\xEC\x00\x97\x97\x97\x97\x2A\xFB\
    \x00\xAE\xFB\xE0\x1B\x70\xF3\xBE\x00\x0E\xF8\x6E\x1B\x7E\xF3\xCE\
    \xFF\xCE\xF4\x1E\x10\x2E\xEC\x0E\xFF\xDE\xEC\x3E\x13\x9E\xEF\x9E\
    \xFB\x9E\xF7\x7E\x13\xE0\xE7\xEE\xF7\xAE\xF7\x50\x0C\x30\xEB\xEE\
    \xF7\x30\xF0\x10\x0B\x50\xE4\x30\xF4\x20\xF3\x70\x07\xC0\xE4\x50\
    \x04\x70\xFB\xCE\xFF\x5C\x0F\x90\x07\xD0\xFC\x3E\xFF\x6C\x0F\x90\
    \x03\xA0\xFC\x0E\xF4\x0C\x0B\xF0\x03\xA0\xF4\x1E\xF7\x8C\x0B\x70\
    \xFF\x72\xFF\x6E\xF7\xDE\x07\xC0\xFF\xA2\x03\x30\xF4\x2E\x0B\xC0\
    \xFF\x22\xFC\x00\xF7\x5E\x04\x10\x00\x22\x03\x50\xF3\xA0\x08\x32\
    \x04\xDE\xFB\xF0\x1B\xCE\xF0\x0E\xFC\x5E\xFC\x80\x1B\xF0\xF0\x2E\
    \x00\x02\xF8\x1E\x14\x2E\xEC\x9E\x07\xDE\xF0\x9E\x17\xCE\xF3\xAE\
    \xFF\xC0\xFB\xB0\x10\x80\xEB\xFE\xFF\xE0\xFF\x90\x14\x60\xE4\xAE\
    \xFB\xC0\xF8\x40\x0F\xE0\xE8\x60\xF8\xC0\xF4\x00\x08\x42\xE4\xB0\
    \x08\x90\x03\xBE\xFF\x9C\x0F\xD0\x00\x20\x04\x50\xFF\xCC\x0F\xE0\
    \x07\xD0\x03\xFE\xFB\xEE\x0C\x60\x0B\x80\xF8\x6E\xFB\x8E\x13\x70\
    \x07\x82\x03\x8E\xF8\x5E\x0F\xC2\x07\xC2\x03\x7E\xF8\x4E\x08\x80\
    \x03\x92\x04\x20\xF7\xCE\x0C\x42\xFC\xB2\xFF\xE0\xF0\x20\x04\x90\
    \x10\x7E\x03\xD0\x1F\x90\xFB\xCE\x08\x9E\x00\x80\x18\x20\xF4\x0E\
    \x08\x00\xFC\x30\x14\x1E\xF0\x6E\x04\x00\xFF\xA0\x17\xCE\xF7\xB0\
    \x07\xD0\xFF\xB0\x18\x30\xF3\xFE\x00\x02\xFF\xE0\x14\xD0\xEC\xB0\
    \xFB\xE2\xFC\xB0\x13\xB0\xEC\xA0\xF8\xDE\xF4\x32\x08\x32\xE8\xD0\
    \x14\x20\xFF\x8E\xFF\x6E\x13\x80\x08\x40\x00\x5E\xFB\xEC\x0F\xB0\
    \x0B\xB2\xFF\xFE\xFB\xDE\x08\x20\x0B\xC0\x03\x60\xFB\x8C\x0F\x70\
    \x07\x94\x03\x6E\xFB\xFE\x0F\xA0\x0B\xC4\x03\x90\xF8\x9E\x0C\x72\
    \xFF\xB2\xFC\x70\xFB\x7E\x04\x70\xFC\xB0\xFF\xF0\xF3\xF0\x04\xA0\
    \x04\x9E\x03\xB0\x1F\x90\xF7\xD0\x04\x2E\x00\x70\x1F\x70\xFB\xBE\
    \x0F\x00\x03\xB0\x14\x2E\xF0\x1E\x07\xB0\xFF\xB0\x1B\x60\xF3\x7E\
    \xFB\xD0\xFF\x90\x14\x10\xF3\xC0\xFC\x00\x03\x70\x14\x82\xF0\x30\
    \xF8\x00\xFC\x50\x13\xC2\xF0\x50\x00\x70\xF8\x12\x0C\x02\xEC\x80\
    \x00\xD0\xFB\xFE\xFB\xCC\x08\x10\xFC\x60\xFC\xB0\xFB\xCE\x0F\xE0\
    \x0B\x40\xFF\xFE\xF0\x5C\x08\x40\x07\xD0\xFF\xD0\xF7\x9E\x0F\xB0\
    \xF8\x02\xFF\xD0\xF4\x4E\x0B\xF0\xFC\x32\x07\xA0\xF4\xBE\x0C\x60\
    \xF8\x22\x00\x80\xF0\x1E\x08\x92\x00\xB4\xF8\x50\xF0\x40\x04\xB2\
    \x08\x5E\x07\x82\x1F\x70\xFB\xEE\x00\x1E\x04\x20\x1F\x80\xFB\xB0\
    \x03\xB0\x03\x90\x17\xF0\xF0\x4E\x07\x70\xFF\xE0\x1B\x40\xF7\x6E\
    \xFF\xC0\xFF\xB0\x17\xE0\xEC\x1E\x03\xA0\x03\xA0\x10\xC0\xEC\x60\
    \xFB\xC2\xFC\x80\x0C\x00\xEC\x60\x00\x50\xF8\x00\x08\x02\xEC\x90\
    \x00\x80\xF7\xB0\xF7\xAE\x04\x10\xFC\x32\xFC\x50\xF7\xBE\x07\xF0\
    \xFF\xD2\xFB\xC0\xF0\x2E\x04\x60\x03\x82\xF4\x10\xF3\x6E\x0B\xA0\
    \xFB\xF2\xFB\xC0\xF0\x1C\x04\x40\xFF\xE2\xFB\xE0\xF0\xEE\x08\xA2\
    \xF8\x04\xFC\xB0\xEC\x3E\x04\xA2\x00\x82\xF8\x30\xE8\x30\x04\xB2\
    \x13\xF0\x03\x80\x1F\x40\xFB\xB0\x0F\x90\x04\x20\x17\xA0\xF7\xAE\
    \x0F\x40\xFF\xE2\x13\xAE\xF0\x3E\x0F\x12\xFF\x60\x0F\x50\xF3\x40\
    \x03\x62\xFF\x30\x17\x60\xEF\xD0\x07\x62\x03\x60\x10\x72\xEC\x50\
    \xF7\xB2\xF8\x52\x07\xB0\xE4\x80\xF8\x20\xF7\xC2\x03\xC2\xE4\x90\
    \x14\x22\x03\xAE\x03\x6E\x13\xC2\x13\xB2\x04\x40\xFF\xCE\x13\xD2\
    \x13\x62\x00\x02\xFB\xDE\x0C\x40\x17\x32\x03\x90\xFF\x8E\x17\x60\
    \x0B\x82\x07\x50\x03\x9E\x10\x00\x0F\x82\x0B\x80\xFC\xAE\x10\x80\
    \x0B\xD4\x04\x70\xFB\xCE\x0C\x92\x08\x32\x07\xE0\xF7\xFE\x0C\xA2\
    \x00\x10\x03\x80\x13\xD0\xF7\xA0\xFF\xBE\x00\x52\x13\x80\xF7\x70\
    \xFF\x70\xFF\xA0\x0F\xC0\xF3\xBE\x03\x40\xFF\x60\x0F\xC0\xF3\x70\
    \xFB\x30\xFB\x80\x0C\x10\xE4\x0E\xFB\xA0\xFB\xB0\x0C\x42\xE8\x60\
    \xFB\x92\xF4\xA2\x0B\x82\xE8\x50\xF8\x32\xFB\xA2\x00\x02\xE4\x70\
    \x00\x22\xF7\xA0\xEF\xFE\x0B\xC0\x03\xD2\xF8\x60\xF7\x9E\x0F\x92\
    \x03\x90\xFF\xB0\xF3\xFE\x0F\xC0\x07\x62\xFF\x70\xEF\xFE\x13\x80\
    \x03\x62\xFF\xB0\xF4\x2E\x08\x10\x07\xA2\x07\xC0\xF8\x7E\x0C\x82\
    \x0B\x94\x04\x90\xFB\x90\x10\x62\x08\x42\x07\xB0\xEC\x10\x0C\x82\
    \x08\x50\x13\xE2\x23\x60\x04\x20\x04\x60\x10\xB0\x1F\xB0\x03\xE0\
    \x0B\x80\x0B\xF0\x14\x30\xFC\x00\x0F\x90\x0B\xC2\x1B\xA0\xFF\xC0\
    \x07\xC2\x0B\x82\x1B\xF0\xF4\x4E\x0B\xB2\x0F\xD2\x14\xC2\xF8\xA0\
    \x07\x92\x08\x52\x13\xE2\xF8\x50\x00\xA0\x00\x32\x0C\x22\xF0\xD0\
    \xF4\x52\xEF\xE0\xEF\x7E\xFC\x32\xF0\x72\xF4\xC0\xEB\xCE\x03\xF0\
    \xFB\xA2\xF4\x00\xE4\x5E\x04\x10\xFF\xA2\xF7\xD0\xEB\xBE\x0B\xD0\
    \xFB\xC2\xFB\x80\xF0\x0E\x00\x50\x03\xD2\x03\xD0\xF0\xE0\x0C\xA0\
    \x03\x84\x04\x40\xF3\xEE\x0C\x52\x00\xA2\x00\x30\xEC\x20\x04\xC0\
    \x10\x22\x0F\xD2\x1F\x80\x03\xF0\x08\x30\x0C\x82\x17\xE0\xFF\xB0\
    \x04\x10\x04\x32\x08\x70\xF4\x8E\x0B\xD0\x07\xB2\x0F\x90\xFB\xB0\
    \xFF\xF0\x07\xA2\x14\x10\xF4\x10\x00\x22\x0B\xC2\x0C\xE0\xF8\x50\
    \xFF\xB2\x04\x90\x0B\xC0\xEC\xC0\xFC\x70\x00\x12\x04\x00\xF0\xB2\
    \x04\x02\xF7\xD0\xF3\x7E\x0B\xF2\x00\x22\xFC\x90\xEF\xFE\x0F\xC2\
    \xFC\x12\xF8\x4E\xE8\x7E\x04\x80\x07\xE2\xFF\xB0\xF7\xAE\x0F\xC0\
    \x00\x02\x07\xA0\xF8\x1E\x10\x02\x04\x22\x0F\xD0\xF8\xCE\x18\x42\
    \x07\xA4\x08\x80\xFB\xB0\x0C\xB0\x0C\x62\x0B\xF0\xFB\xF0\x10\xA0\
    \xF0\x30\x07\xD2\x0B\xE0\xF8\x00\xEC\xA0\x04\x82\x08\x30\xFB\xE0\
    \xF0\x40\xFC\x80\x08\x10\xF0\x30\xF4\x10\xF8\x30\x0B\xA0\xF7\xA0\
    \xF3\xD2\xFF\xF2\x08\x40\xEF\xF0\xF4\x00\x03\xB2\x08\x72\xF0\x30\
    \xEF\xB2\x00\x42\x03\xB2\xEC\x40\xFF\xE0\xFF\xE2\x00\x12\xF4\x20\
    \xF4\x22\xF7\xB0\xE7\xCE\x0B\xD2\xF0\x80\x00\x70\xEC\x2E\x0F\xE2\
    \xF8\x50\x00\x70\xF0\x0E\x0C\x42\x00\x20\x00\x30\xF7\xAE\x17\xB2\
    \x03\xD2\x04\x00\xF8\x4E\x17\xF0\x0B\xE2\x13\xA0\xFC\x4E\x18\x20\
    \x07\x92\x10\x20\xFB\x9E\x1C\x10\x1B\xC2\x13\xC0\xFB\xE0\x20\x02\
    \xF0\x40\x13\xA2\x0F\x80\xFC\x30\xF4\x6E\x0C\xC2\x17\xB2\x00\x10\
    \xFC\x10\x08\x72\x10\x00\xF8\xB0\x07\xBE\x0B\xE2\x13\xB0\xFF\xE0\
    \xF4\x10\x04\x50\x0C\x70\xF4\x20\x03\xC0\x0F\x82\x10\x60\xFF\xE0\
    \xFB\x70\x13\xD2\x0F\x90\xF8\x20\xFC\x40\x0F\xA2\x0B\xE2\xFC\x60\
    \xF0\x12\xFB\x80\xEB\x5E\x08\x02\xF4\x20\x00\x90\xF7\x8E\x13\xE2\
    \xFC\x02\x00\x60\xF4\x0E\x10\x90\x0F\x90\x0B\xD0\xFB\xAE\x1F\xD2\
    \x00\x02\x08\x20\xF8\x5E\x18\x00\x0F\x82\x1B\x60\xFC\x3E\x23\xC2\
    \x0B\x42\x1B\xA0\xFF\x7E\x27\xE0\x10\x12\x1B\x70\xFF\xC0\x20\x40\
    \xFC\x70\x1B\xA2\x0F\xA0\x0B\xA0\x00\x02\x14\x32\x0F\xE0\x00\x10\
    \xF8\x3E\x13\xE0\x08\x5E\x07\xE0\x00\x5E\x08\x42\x0F\xEE\x03\xD0\
    \xFC\x20\x0F\xE2\x14\x00\x07\x80\x0B\x90\x17\x72\x14\x10\x07\xB0\
    \xFB\x10\x17\xF2\x0B\x20\x03\xF0\xFC\x1E\x17\xB2\x07\xCE\x08\x30\
    \xE0\x50\xEF\x80\xD3\x8E\x03\x82\xEB\xE0\xF8\x10\xDF\xBE\x07\xD0\
    \xEC\x10\xFF\xC0\xE0\x1E\x0B\xB0\xF8\x20\xF8\x10\xEB\xBE\x0B\xA0\
    \xFB\xF0\x07\xA0\xF3\xEE\x1B\x50\x07\x52\x0F\x30\xF7\xEE\x1B\x80\
    \x02\xF2\x0F\xD0\xF7\x0E\x13\xC0\x0B\xE0\x13\x90\xF7\x9E\x1C\x00";

pub const K_V_DD: i16 = -3200;
pub const V_DD_25: i16 = -12544;
pub const K_V_PTAT: f32 = 0.002197;
pub const K_T_PTAT: f32 = 42.625000;
pub const V_PTAT_25: f32 = 12196f32;
pub const ALPHA_PTAT: f32 = 9.000000;
pub const GAIN_EE: f32 = 5580f32;
pub const K_V_CP: f32 = 0.375000;
pub const K_TA_CP: f32 = 0.004272;
pub const RESOLUTION_EE: u8 = 2;
pub const K_S_TA: f32 = -0.002441;
pub const ALPHA_CP: [f32; 2] = [0.0000000028812792152, 0.0000000029037892091];
pub const OFFSET_CP: [i16; 2] = [-69, -65];
pub const CORNER_TEMPERATURES: [i16; 4] = [-40, 0, 300, 500];
pub const K_S_TO: [f32; 4] = [-0.000200f32; 4];

pub const ALPHA_PIXELS: [f32; mlx90640::Mlx90640::NUM_PIXELS] = [
    0.0000000364125298802,
    0.0000000368781911675,
    0.0000000379259290639,
    0.0000000383915903512,
    0.0000000408363121096,
    0.0000000415348040406,
    0.0000000416512193624,
    0.0000000422332959715,
    0.0000000440959411208,
    0.0000000446780177299,
    0.0000000447944330517,
    0.0000000445616024081,
    0.0000000460750015918,
    0.0000000467734935228,
    0.0000000456093403045,
    0.0000000456093403045,
    0.0000000456093403045,
    0.0000000453765096609,
    0.0000000443287717644,
    0.0000000443287717644,
    0.0000000438631104771,
    0.0000000432810338680,
    0.0000000411855580751,
    0.0000000406034814660,
    0.0000000393229129259,
    0.0000000390900822822,
    0.0000000358304532710,
    0.0000000355976226274,
    0.0000000336185621563,
    0.0000000323379936162,
    0.0000000291947799269,
    0.0000000283798726741,
    0.0000000369946064893,
    0.0000000375766830985,
    0.0000000386244209949,
    0.0000000390900822822,
    0.0000000414183887187,
    0.0000000421168806497,
    0.0000000424661266152,
    0.0000000428153725807,
    0.0000000447944330517,
    0.0000000454929249827,
    0.0000000454929249827,
    0.0000000453765096609,
    0.0000000466570782010,
    0.0000000474719854537,
    0.0000000464242475573,
    0.0000000463078322355,
    0.0000000463078322355,
    0.0000000461914169136,
    0.0000000451436790172,
    0.0000000450272636954,
    0.0000000447944330517,
    0.0000000439795257989,
    0.0000000420004653279,
    0.0000000413019733969,
    0.0000000401378201786,
    0.0000000399049895350,
    0.0000000367617758457,
    0.0000000362961145584,
    0.0000000345498847309,
    0.0000000330364855472,
    0.0000000298932718579,
    0.0000000290783646051,
    0.0000000386244209949,
    0.0000000388572516385,
    0.0000000403706508223,
    0.0000000408363121096,
    0.0000000432810338680,
    0.0000000436302798335,
    0.0000000444451870862,
    0.0000000447944330517,
    0.0000000464242475573,
    0.0000000465406628791,
    0.0000000466570782010,
    0.0000000474719854537,
    0.0000000479376467410,
    0.0000000493346306030,
    0.0000000478212314192,
    0.0000000475884007756,
    0.0000000478212314192,
    0.0000000477048160974,
    0.0000000473555701319,
    0.0000000463078322355,
    0.0000000461914169136,
    0.0000000456093403045,
    0.0000000433974491898,
    0.0000000438631104771,
    0.0000000422332959715,
    0.0000000413019733969,
    0.0000000387408363167,
    0.0000000378095137421,
    0.0000000366453605238,
    0.0000000352483766619,
    0.0000000319887476508,
    0.0000000309410097543,
    0.0000000390900822822,
    0.0000000393229129259,
    0.0000000409527274314,
    0.0000000414183887187,
    0.0000000438631104771,
    0.0000000442123564426,
    0.0000000450272636954,
    0.0000000452600943390,
    0.0000000470063241664,
    0.0000000472391548101,
    0.0000000471227394883,
    0.0000000480540620629,
    0.0000000482868927065,
    0.0000000499167072121,
    0.0000000482868927065,
    0.0000000481704773847,
    0.0000000482868927065,
    0.0000000482868927065,
    0.0000000479376467410,
    0.0000000468899088446,
    0.0000000468899088446,
    0.0000000463078322355,
    0.0000000440959411208,
    0.0000000445616024081,
    0.0000000428153725807,
    0.0000000420004653279,
    0.0000000394393282477,
    0.0000000385080056731,
    0.0000000374602677766,
    0.0000000359468685929,
    0.0000000326872395817,
    0.0000000316395016853,
    0.0000000397885742132,
    0.0000000404870661441,
    0.0000000418840500060,
    0.0000000422332959715,
    0.0000000456093403045,
    0.0000000454929249827,
    0.0000000466570782010,
    0.0000000464242475573,
    0.0000000482868927065,
    0.0000000486361386720,
    0.0000000484033080284,
    0.0000000489853846375,
    0.0000000501495378558,
    0.0000000494510459248,
    0.0000000496838765685,
    0.0000000495674612466,
    0.0000000498002918903,
    0.0000000495674612466,
    0.0000000486361386720,
    0.0000000481704773847,
    0.0000000482868927065,
    0.0000000480540620629,
    0.0000000460750015918,
    0.0000000458421709482,
    0.0000000443287717644,
    0.0000000439795257989,
    0.0000000402542355005,
    0.0000000401378201786,
    0.0000000386244209949,
    0.0000000374602677766,
    0.0000000337349774782,
    0.0000000330364855472,
    0.0000000401378201786,
    0.0000000408363121096,
    0.0000000424661266152,
    0.0000000426989572588,
    0.0000000459585862700,
    0.0000000460750015918,
    0.0000000471227394883,
    0.0000000467734935228,
    0.0000000486361386720,
    0.0000000491017999593,
    0.0000000488689693157,
    0.0000000494510459248,
    0.0000000506151991431,
    0.0000000499167072121,
    0.0000000501495378558,
    0.0000000500331225339,
    0.0000000502659531776,
    0.0000000499167072121,
    0.0000000491017999593,
    0.0000000485197233502,
    0.0000000487525539938,
    0.0000000484033080284,
    0.0000000465406628791,
    0.0000000463078322355,
    0.0000000449108483735,
    0.0000000444451870862,
    0.0000000407198967878,
    0.0000000407198967878,
    0.0000000393229129259,
    0.0000000379259290639,
    0.0000000342006387655,
    0.0000000336185621563,
    0.0000000418840500060,
    0.0000000421168806497,
    0.0000000437466951553,
    0.0000000442123564426,
    0.0000000466570782010,
    0.0000000472391548101,
    0.0000000472391548101,
    0.0000000477048160974,
    0.0000000482868927065,
    0.0000000495674612466,
    0.0000000503823684994,
    0.0000000502659531776,
    0.0000000514301063959,
    0.0000000514301063959,
    0.0000000508480297867,
    0.0000000509644451085,
    0.0000000516629370395,
    0.0000000511972757522,
    0.0000000502659531776,
    0.0000000496838765685,
    0.0000000501495378558,
    0.0000000491017999593,
    0.0000000473555701319,
    0.0000000467734935228,
    0.0000000464242475573,
    0.0000000451436790172,
    0.0000000422332959715,
    0.0000000414183887187,
    0.0000000397885742132,
    0.0000000390900822822,
    0.0000000352483766619,
    0.0000000343170540873,
    0.0000000423497112934,
    0.0000000425825419370,
    0.0000000440959411208,
    0.0000000446780177299,
    0.0000000471227394883,
    0.0000000477048160974,
    0.0000000478212314192,
    0.0000000480540620629,
    0.0000000487525539938,
    0.0000000500331225339,
    0.0000000507316144649,
    0.0000000506151991431,
    0.0000000516629370395,
    0.0000000516629370395,
    0.0000000511972757522,
    0.0000000514301063959,
    0.0000000520121830050,
    0.0000000516629370395,
    0.0000000506151991431,
    0.0000000500331225339,
    0.0000000504987838212,
    0.0000000494510459248,
    0.0000000477048160974,
    0.0000000471227394883,
    0.0000000466570782010,
    0.0000000454929249827,
    0.0000000428153725807,
    0.0000000418840500060,
    0.0000000402542355005,
    0.0000000395557435695,
    0.0000000357140379492,
    0.0000000346663000528,
    0.0000000423497112934,
    0.0000000426989572588,
    0.0000000444451870862,
    0.0000000452600943390,
    0.0000000474719854537,
    0.0000000475884007756,
    0.0000000482868927065,
    0.0000000486361386720,
    0.0000000504987838212,
    0.0000000502659531776,
    0.0000000509644451085,
    0.0000000515465217177,
    0.0000000518957676832,
    0.0000000527106749360,
    0.0000000515465217177,
    0.0000000517793523613,
    0.0000000524778442923,
    0.0000000523614289705,
    0.0000000508480297867,
    0.0000000511972757522,
    0.0000000503823684994,
    0.0000000503823684994,
    0.0000000487525539938,
    0.0000000480540620629,
    0.0000000468899088446,
    0.0000000464242475573,
    0.0000000436302798335,
    0.0000000424661266152,
    0.0000000404870661441,
    0.0000000399049895350,
    0.0000000361796992365,
    0.0000000353647919837,
    0.0000000426989572588,
    0.0000000430482032243,
    0.0000000447944330517,
    0.0000000456093403045,
    0.0000000477048160974,
    0.0000000479376467410,
    0.0000000486361386720,
    0.0000000491017999593,
    0.0000000507316144649,
    0.0000000506151991431,
    0.0000000513136910740,
    0.0000000517793523613,
    0.0000000520121830050,
    0.0000000530599209014,
    0.0000000517793523613,
    0.0000000522450136486,
    0.0000000528270902578,
    0.0000000524778442923,
    0.0000000511972757522,
    0.0000000515465217177,
    0.0000000508480297867,
    0.0000000508480297867,
    0.0000000489853846375,
    0.0000000485197233502,
    0.0000000473555701319,
    0.0000000467734935228,
    0.0000000439795257989,
    0.0000000429317879025,
    0.0000000408363121096,
    0.0000000402542355005,
    0.0000000365289452020,
    0.0000000355976226274,
    0.0000000425825419370,
    0.0000000436302798335,
    0.0000000450272636954,
    0.0000000458421709482,
    0.0000000474719854537,
    0.0000000485197233502,
    0.0000000494510459248,
    0.0000000494510459248,
    0.0000000506151991431,
    0.0000000517793523613,
    0.0000000513136910740,
    0.0000000523614289705,
    0.0000000521285983268,
    0.0000000527106749360,
    0.0000000525942596141,
    0.0000000524778442923,
    0.0000000527106749360,
    0.0000000523614289705,
    0.0000000508480297867,
    0.0000000516629370395,
    0.0000000508480297867,
    0.0000000508480297867,
    0.0000000491017999593,
    0.0000000488689693157,
    0.0000000477048160974,
    0.0000000470063241664,
    0.0000000439795257989,
    0.0000000436302798335,
    0.0000000410691427533,
    0.0000000403706508223,
    0.0000000366453605238,
    0.0000000362961145584,
    0.0000000429317879025,
    0.0000000438631104771,
    0.0000000452600943390,
    0.0000000459585862700,
    0.0000000477048160974,
    0.0000000487525539938,
    0.0000000496838765685,
    0.0000000498002918903,
    0.0000000508480297867,
    0.0000000520121830050,
    0.0000000516629370395,
    0.0000000524778442923,
    0.0000000523614289705,
    0.0000000530599209014,
    0.0000000529435055796,
    0.0000000527106749360,
    0.0000000529435055796,
    0.0000000525942596141,
    0.0000000511972757522,
    0.0000000520121830050,
    0.0000000510808604304,
    0.0000000510808604304,
    0.0000000494510459248,
    0.0000000492182152811,
    0.0000000479376467410,
    0.0000000472391548101,
    0.0000000440959411208,
    0.0000000437466951553,
    0.0000000411855580751,
    0.0000000406034814660,
    0.0000000369946064893,
    0.0000000364125298802,
    0.0000000428153725807,
    0.0000000436302798335,
    0.0000000460750015918,
    0.0000000457257556263,
    0.0000000477048160974,
    0.0000000488689693157,
    0.0000000492182152811,
    0.0000000491017999593,
    0.0000000509644451085,
    0.0000000513136910740,
    0.0000000515465217177,
    0.0000000514301063959,
    0.0000000524778442923,
    0.0000000527106749360,
    0.0000000534091668669,
    0.0000000528270902578,
    0.0000000523614289705,
    0.0000000529435055796,
    0.0000000521285983268,
    0.0000000520121830050,
    0.0000000513136910740,
    0.0000000514301063959,
    0.0000000487525539938,
    0.0000000489853846375,
    0.0000000474719854537,
    0.0000000475884007756,
    0.0000000436302798335,
    0.0000000432810338680,
    0.0000000411855580751,
    0.0000000401378201786,
    0.0000000371110218111,
    0.0000000360632839147,
    0.0000000429317879025,
    0.0000000438631104771,
    0.0000000463078322355,
    0.0000000459585862700,
    0.0000000479376467410,
    0.0000000489853846375,
    0.0000000493346306030,
    0.0000000493346306030,
    0.0000000511972757522,
    0.0000000514301063959,
    0.0000000518957676832,
    0.0000000515465217177,
    0.0000000527106749360,
    0.0000000528270902578,
    0.0000000537584128324,
    0.0000000529435055796,
    0.0000000527106749360,
    0.0000000532927515451,
    0.0000000522450136486,
    0.0000000521285983268,
    0.0000000513136910740,
    0.0000000515465217177,
    0.0000000491017999593,
    0.0000000492182152811,
    0.0000000474719854537,
    0.0000000474719854537,
    0.0000000437466951553,
    0.0000000433974491898,
    0.0000000413019733969,
    0.0000000402542355005,
    0.0000000372274371330,
    0.0000000361796992365,
    0.0000000423497112934,
    0.0000000433974491898,
    0.0000000443287717644,
    0.0000000457257556263,
    0.0000000480540620629,
    0.0000000486361386720,
    0.0000000486361386720,
    0.0000000489853846375,
    0.0000000501495378558,
    0.0000000509644451085,
    0.0000000514301063959,
    0.0000000510808604304,
    0.0000000521285983268,
    0.0000000524778442923,
    0.0000000522450136486,
    0.0000000524778442923,
    0.0000000524778442923,
    0.0000000520121830050,
    0.0000000509644451085,
    0.0000000515465217177,
    0.0000000504987838212,
    0.0000000507316144649,
    0.0000000487525539938,
    0.0000000485197233502,
    0.0000000465406628791,
    0.0000000460750015918,
    0.0000000433974491898,
    0.0000000423497112934,
    0.0000000410691427533,
    0.0000000402542355005,
    0.0000000364125298802,
    0.0000000358304532710,
    0.0000000423497112934,
    0.0000000433974491898,
    0.0000000444451870862,
    0.0000000458421709482,
    0.0000000481704773847,
    0.0000000487525539938,
    0.0000000487525539938,
    0.0000000491017999593,
    0.0000000503823684994,
    0.0000000510808604304,
    0.0000000516629370395,
    0.0000000511972757522,
    0.0000000522450136486,
    0.0000000525942596141,
    0.0000000523614289705,
    0.0000000525942596141,
    0.0000000524778442923,
    0.0000000520121830050,
    0.0000000510808604304,
    0.0000000516629370395,
    0.0000000507316144649,
    0.0000000507316144649,
    0.0000000489853846375,
    0.0000000485197233502,
    0.0000000464242475573,
    0.0000000459585862700,
    0.0000000433974491898,
    0.0000000423497112934,
    0.0000000410691427533,
    0.0000000402542355005,
    0.0000000364125298802,
    0.0000000357140379492,
    0.0000000410691427533,
    0.0000000423497112934,
    0.0000000436302798335,
    0.0000000444451870862,
    0.0000000467734935228,
    0.0000000473555701319,
    0.0000000480540620629,
    0.0000000477048160974,
    0.0000000502659531776,
    0.0000000504987838212,
    0.0000000509644451085,
    0.0000000510808604304,
    0.0000000516629370395,
    0.0000000514301063959,
    0.0000000511972757522,
    0.0000000514301063959,
    0.0000000518957676832,
    0.0000000513136910740,
    0.0000000502659531776,
    0.0000000502659531776,
    0.0000000503823684994,
    0.0000000496838765685,
    0.0000000480540620629,
    0.0000000470063241664,
    0.0000000458421709482,
    0.0000000456093403045,
    0.0000000422332959715,
    0.0000000422332959715,
    0.0000000397885742132,
    0.0000000390900822822,
    0.0000000352483766619,
    0.0000000346663000528,
    0.0000000408363121096,
    0.0000000423497112934,
    0.0000000435138645116,
    0.0000000444451870862,
    0.0000000466570782010,
    0.0000000474719854537,
    0.0000000481704773847,
    0.0000000478212314192,
    0.0000000502659531776,
    0.0000000506151991431,
    0.0000000509644451085,
    0.0000000510808604304,
    0.0000000517793523613,
    0.0000000514301063959,
    0.0000000513136910740,
    0.0000000515465217177,
    0.0000000520121830050,
    0.0000000513136910740,
    0.0000000502659531776,
    0.0000000501495378558,
    0.0000000503823684994,
    0.0000000498002918903,
    0.0000000478212314192,
    0.0000000468899088446,
    0.0000000457257556263,
    0.0000000454929249827,
    0.0000000421168806497,
    0.0000000421168806497,
    0.0000000396721588913,
    0.0000000388572516385,
    0.0000000351319613401,
    0.0000000345498847309,
    0.0000000402542355005,
    0.0000000414183887187,
    0.0000000433974491898,
    0.0000000436302798335,
    0.0000000466570782010,
    0.0000000464242475573,
    0.0000000477048160974,
    0.0000000471227394883,
    0.0000000496838765685,
    0.0000000501495378558,
    0.0000000493346306030,
    0.0000000495674612466,
    0.0000000511972757522,
    0.0000000514301063959,
    0.0000000503823684994,
    0.0000000503823684994,
    0.0000000507316144649,
    0.0000000509644451085,
    0.0000000496838765685,
    0.0000000491017999593,
    0.0000000492182152811,
    0.0000000486361386720,
    0.0000000463078322355,
    0.0000000458421709482,
    0.0000000449108483735,
    0.0000000440959411208,
    0.0000000411855580751,
    0.0000000403706508223,
    0.0000000378095137421,
    0.0000000378095137421,
    0.0000000344334694091,
    0.0000000326872395817,
    0.0000000401378201786,
    0.0000000411855580751,
    0.0000000431646185461,
    0.0000000432810338680,
    0.0000000464242475573,
    0.0000000463078322355,
    0.0000000475884007756,
    0.0000000471227394883,
    0.0000000498002918903,
    0.0000000500331225339,
    0.0000000492182152811,
    0.0000000496838765685,
    0.0000000513136910740,
    0.0000000514301063959,
    0.0000000503823684994,
    0.0000000504987838212,
    0.0000000507316144649,
    0.0000000510808604304,
    0.0000000496838765685,
    0.0000000491017999593,
    0.0000000489853846375,
    0.0000000485197233502,
    0.0000000459585862700,
    0.0000000457257556263,
    0.0000000446780177299,
    0.0000000438631104771,
    0.0000000409527274314,
    0.0000000400214048568,
    0.0000000375766830985,
    0.0000000375766830985,
    0.0000000340842234436,
    0.0000000324544089381,
    0.0000000385080056731,
    0.0000000392064976040,
    0.0000000408363121096,
    0.0000000421168806497,
    0.0000000443287717644,
    0.0000000450272636954,
    0.0000000449108483735,
    0.0000000456093403045,
    0.0000000474719854537,
    0.0000000481704773847,
    0.0000000473555701319,
    0.0000000486361386720,
    0.0000000486361386720,
    0.0000000489853846375,
    0.0000000486361386720,
    0.0000000489853846375,
    0.0000000493346306030,
    0.0000000498002918903,
    0.0000000481704773847,
    0.0000000475884007756,
    0.0000000468899088446,
    0.0000000464242475573,
    0.0000000443287717644,
    0.0000000433974491898,
    0.0000000425825419370,
    0.0000000414183887187,
    0.0000000390900822822,
    0.0000000382751750294,
    0.0000000366453605238,
    0.0000000354812073056,
    0.0000000322215782944,
    0.0000000312902557198,
    0.0000000381587597076,
    0.0000000389736669604,
    0.0000000404870661441,
    0.0000000417676346842,
    0.0000000438631104771,
    0.0000000446780177299,
    0.0000000445616024081,
    0.0000000452600943390,
    0.0000000473555701319,
    0.0000000480540620629,
    0.0000000473555701319,
    0.0000000484033080284,
    0.0000000484033080284,
    0.0000000488689693157,
    0.0000000485197233502,
    0.0000000488689693157,
    0.0000000492182152811,
    0.0000000494510459248,
    0.0000000479376467410,
    0.0000000473555701319,
    0.0000000464242475573,
    0.0000000461914169136,
    0.0000000439795257989,
    0.0000000431646185461,
    0.0000000422332959715,
    0.0000000410691427533,
    0.0000000388572516385,
    0.0000000378095137421,
    0.0000000362961145584,
    0.0000000351319613401,
    0.0000000319887476508,
    0.0000000310574250761,
    0.0000000369946064893,
    0.0000000373438524548,
    0.0000000392064976040,
    0.0000000392064976040,
    0.0000000417676346842,
    0.0000000421168806497,
    0.0000000433974491898,
    0.0000000437466951553,
    0.0000000458421709482,
    0.0000000452600943390,
    0.0000000460750015918,
    0.0000000452600943390,
    0.0000000479376467410,
    0.0000000478212314192,
    0.0000000471227394883,
    0.0000000470063241664,
    0.0000000475884007756,
    0.0000000471227394883,
    0.0000000454929249827,
    0.0000000445616024081,
    0.0000000446780177299,
    0.0000000444451870862,
    0.0000000418840500060,
    0.0000000411855580751,
    0.0000000400214048568,
    0.0000000397885742132,
    0.0000000364125298802,
    0.0000000360632839147,
    0.0000000344334694091,
    0.0000000337349774782,
    0.0000000301261025015,
    0.0000000290783646051,
    0.0000000367617758457,
    0.0000000371110218111,
    0.0000000389736669604,
    0.0000000389736669604,
    0.0000000415348040406,
    0.0000000418840500060,
    0.0000000430482032243,
    0.0000000432810338680,
    0.0000000456093403045,
    0.0000000450272636954,
    0.0000000456093403045,
    0.0000000449108483735,
    0.0000000475884007756,
    0.0000000474719854537,
    0.0000000467734935228,
    0.0000000466570782010,
    0.0000000472391548101,
    0.0000000466570782010,
    0.0000000452600943390,
    0.0000000442123564426,
    0.0000000442123564426,
    0.0000000439795257989,
    0.0000000415348040406,
    0.0000000408363121096,
    0.0000000397885742132,
    0.0000000395557435695,
    0.0000000361796992365,
    0.0000000357140379492,
    0.0000000340842234436,
    0.0000000335021468345,
    0.0000000297768565360,
    0.0000000287291186396,
];

pub const OFFSET_REFERENCE_PIXELS: [i16; mlx90640::Mlx90640::NUM_PIXELS] = [
    -54, -56, -50, -60, -50, -56, -48, -60, -49, -57, -48, -61, -49, -59, -48, -63, -46, -59, -48,
    -65, -49, -61, -49, -66, -47, -64, -50, -69, -53, -66, -51, -75, -61, -64, -65, -61, -57, -63,
    -63, -61, -56, -63, -63, -62, -56, -65, -63, -64, -53, -65, -63, -65, -55, -66, -63, -66, -53,
    -69, -63, -69, -58, -70, -64, -74, -53, -56, -50, -60, -51, -55, -48, -60, -48, -56, -47, -61,
    -47, -58, -47, -62, -45, -58, -48, -64, -47, -59, -47, -67, -46, -62, -49, -68, -52, -65, -50,
    -75, -60, -62, -65, -61, -58, -61, -63, -61, -55, -62, -62, -61, -54, -64, -62, -62, -51, -64,
    -62, -63, -53, -66, -62, -66, -52, -67, -63, -67, -59, -71, -64, -75, -50, -54, -49, -58, -48,
    -54, -48, -59, -46, -55, -47, -60, -47, -55, -47, -61, -43, -57, -46, -62, -46, -59, -47, -65,
    -46, -61, -48, -67, -52, -65, -50, -74, -57, -63, -65, -60, -56, -62, -64, -61, -54, -63, -62,
    -62, -54, -62, -62, -63, -51, -64, -62, -63, -52, -66, -62, -65, -53, -69, -62, -69, -59, -71,
    -64, -75, -53, -54, -49, -59, -49, -54, -47, -58, -45, -54, -47, -60, -47, -55, -46, -62, -46,
    -57, -47, -62, -47, -58, -47, -64, -46, -61, -48, -66, -50, -64, -49, -73, -62, -64, -66, -62,
    -59, -63, -64, -61, -54, -63, -64, -62, -55, -63, -63, -63, -54, -65, -63, -64, -55, -65, -63,
    -65, -54, -68, -64, -68, -58, -72, -64, -75, -52, -53, -49, -58, -50, -53, -47, -58, -48, -54,
    -47, -60, -47, -55, -46, -61, -45, -57, -47, -63, -46, -58, -48, -65, -46, -61, -49, -67, -50,
    -64, -50, -73, -62, -65, -67, -63, -59, -63, -65, -63, -57, -64, -64, -63, -56, -65, -64, -64,
    -54, -66, -64, -65, -55, -68, -64, -66, -54, -69, -65, -69, -58, -72, -66, -75, -50, -54, -49,
    -58, -47, -53, -49, -59, -45, -55, -48, -60, -45, -55, -49, -62, -44, -57, -47, -63, -45, -58,
    -48, -65, -47, -62, -51, -69, -52, -65, -52, -75, -61, -66, -68, -64, -58, -65, -67, -64, -56,
    -66, -66, -65, -55, -66, -65, -65, -54, -67, -64, -66, -55, -68, -65, -68, -54, -71, -66, -71,
    -60, -73, -67, -77, -54, -54, -52, -59, -51, -54, -50, -59, -49, -55, -49, -60, -48, -55, -49,
    -62, -46, -58, -49, -65, -48, -60, -49, -66, -46, -63, -50, -68, -52, -64, -52, -75, -66, -69,
    -73, -66, -62, -68, -69, -65, -60, -67, -68, -65, -59, -67, -69, -66, -56, -69, -67, -68, -57,
    -69, -66, -69, -54, -71, -66, -70, -60, -73, -69, -77, -56, -54, -52, -59, -53, -54, -51, -60,
    -50, -56, -51, -61, -49, -56, -50, -63, -47, -58, -50, -65, -48, -59, -51, -66, -47, -62, -52,
    -68, -54, -66, -53, -76, -69, -71, -73, -69, -66, -69, -72, -68, -62, -69, -71, -67, -61, -69,
    -70, -68, -58, -70, -68, -70, -58, -70, -68, -69, -56, -71, -68, -71, -62, -74, -69, -79, -54,
    -55, -53, -60, -52, -55, -53, -61, -51, -57, -54, -63, -50, -57, -53, -64, -49, -59, -51, -65,
    -50, -60, -53, -66, -49, -63, -54, -71, -55, -66, -55, -76, -69, -73, -76, -70, -66, -71, -75,
    -69, -65, -72, -74, -71, -63, -71, -71, -71, -60, -71, -70, -70, -61, -71, -70, -70, -59, -74,
    -70, -75, -63, -76, -70, -80, -62, -57, -58, -62, -59, -57, -56, -62, -56, -59, -54, -64, -55,
    -60, -54, -65, -52, -61, -54, -67, -53, -62, -54, -68, -53, -64, -56, -71, -55, -67, -56, -75,
    -77, -77, -83, -74, -74, -74, -79, -73, -70, -74, -76, -73, -68, -74, -75, -73, -64, -75, -74,
    -73, -64, -74, -73, -74, -63, -76, -74, -75, -64, -78, -74, -80, -66, -58, -61, -65, -61, -59,
    -57, -64, -57, -60, -56, -66, -55, -60, -56, -67, -55, -63, -57, -69, -54, -63, -56, -69, -54,
    -64, -57, -72, -59, -67, -58, -77, -82, -80, -86, -78, -77, -78, -81, -76, -73, -78, -79, -76,
    -69, -76, -78, -75, -68, -78, -78, -76, -67, -76, -77, -76, -66, -78, -77, -77, -70, -80, -77,
    -84, -67, -60, -65, -66, -62, -61, -63, -68, -62, -62, -62, -67, -60, -64, -61, -70, -57, -65,
    -59, -69, -56, -65, -59, -71, -58, -67, -62, -74, -63, -69, -63, -78, -94, -91, -100, -88, -88,
    -88, -95, -87, -85, -87, -92, -86, -82, -88, -90, -88, -78, -87, -88, -84, -77, -87, -87, -86,
    -76, -89, -87, -90, -80, -90, -87, -93,
];

pub const K_TA_PIXELS: [f32; mlx90640::Mlx90640::NUM_PIXELS] = [
    0.00671387, 0.00646973, 0.00720215, 0.00598145, 0.00671387, 0.00598145, 0.00671387, 0.00598145,
    0.00671387, 0.00598145, 0.00671387, 0.00598145, 0.00671387, 0.00598145, 0.00671387, 0.00598145,
    0.00671387, 0.00598145, 0.00720215, 0.00598145, 0.00671387, 0.00646973, 0.00720215, 0.00598145,
    0.00720215, 0.00646973, 0.00720215, 0.00646973, 0.00720215, 0.00646973, 0.00720215, 0.00646973,
    0.00683594, 0.00585938, 0.00585938, 0.00634766, 0.00683594, 0.00585938, 0.00585938, 0.00634766,
    0.00683594, 0.00585938, 0.00585938, 0.00634766, 0.00683594, 0.00585938, 0.00585938, 0.00634766,
    0.00732422, 0.00585938, 0.00634766, 0.00634766, 0.00732422, 0.00634766, 0.00634766, 0.00634766,
    0.00732422, 0.00634766, 0.00634766, 0.00634766, 0.00732422, 0.00634766, 0.00683594, 0.00683594,
    0.00671387, 0.00646973, 0.00671387, 0.00598145, 0.00671387, 0.00646973, 0.00720215, 0.00598145,
    0.00769043, 0.00598145, 0.00671387, 0.00598145, 0.00671387, 0.00598145, 0.00671387, 0.00598145,
    0.00720215, 0.00646973, 0.00720215, 0.00598145, 0.00720215, 0.00646973, 0.00720215, 0.00598145,
    0.00720215, 0.00646973, 0.00720215, 0.00646973, 0.00720215, 0.00646973, 0.00769043, 0.00646973,
    0.00683594, 0.00585938, 0.00585938, 0.00634766, 0.00683594, 0.00634766, 0.00585938, 0.00634766,
    0.00683594, 0.00585938, 0.00634766, 0.00634766, 0.00683594, 0.00585938, 0.00634766, 0.00634766,
    0.00732422, 0.00585938, 0.00634766, 0.00683594, 0.00732422, 0.00585938, 0.00634766, 0.00634766,
    0.00732422, 0.00634766, 0.00634766, 0.00683594, 0.00732422, 0.00634766, 0.00683594, 0.00634766,
    0.00671387, 0.00646973, 0.00720215, 0.00598145, 0.00671387, 0.00646973, 0.00720215, 0.00598145,
    0.00720215, 0.00646973, 0.00671387, 0.00598145, 0.00720215, 0.00646973, 0.00671387, 0.00646973,
    0.00720215, 0.00646973, 0.00720215, 0.00598145, 0.00769043, 0.00646973, 0.00720215, 0.00646973,
    0.00769043, 0.00646973, 0.00720215, 0.00646973, 0.00671387, 0.00695801, 0.00769043, 0.00646973,
    0.00683594, 0.00585938, 0.00634766, 0.00634766, 0.00683594, 0.00585938, 0.00585938, 0.00634766,
    0.00732422, 0.00585938, 0.00634766, 0.00634766, 0.00683594, 0.00634766, 0.00585938, 0.00634766,
    0.00781250, 0.00585938, 0.00634766, 0.00634766, 0.00781250, 0.00634766, 0.00634766, 0.00683594,
    0.00732422, 0.00634766, 0.00634766, 0.00634766, 0.00683594, 0.00634766, 0.00683594, 0.00634766,
    0.00671387, 0.00646973, 0.00720215, 0.00646973, 0.00671387, 0.00646973, 0.00720215, 0.00598145,
    0.00720215, 0.00646973, 0.00671387, 0.00598145, 0.00720215, 0.00646973, 0.00720215, 0.00598145,
    0.00720215, 0.00646973, 0.00720215, 0.00646973, 0.00720215, 0.00646973, 0.00769043, 0.00646973,
    0.00720215, 0.00646973, 0.00769043, 0.00646973, 0.00720215, 0.00695801, 0.00769043, 0.00646973,
    0.00683594, 0.00585938, 0.00585938, 0.00634766, 0.00683594, 0.00634766, 0.00634766, 0.00634766,
    0.00683594, 0.00585938, 0.00585938, 0.00634766, 0.00683594, 0.00634766, 0.00634766, 0.00634766,
    0.00732422, 0.00634766, 0.00634766, 0.00634766, 0.00732422, 0.00634766, 0.00634766, 0.00634766,
    0.00732422, 0.00634766, 0.00634766, 0.00683594, 0.00781250, 0.00634766, 0.00683594, 0.00683594,
    0.00671387, 0.00695801, 0.00720215, 0.00598145, 0.00671387, 0.00646973, 0.00720215, 0.00646973,
    0.00720215, 0.00646973, 0.00720215, 0.00598145, 0.00720215, 0.00646973, 0.00720215, 0.00598145,
    0.00720215, 0.00646973, 0.00720215, 0.00598145, 0.00720215, 0.00646973, 0.00720215, 0.00646973,
    0.00769043, 0.00646973, 0.00720215, 0.00646973, 0.00720215, 0.00646973, 0.00769043, 0.00646973,
    0.00683594, 0.00634766, 0.00634766, 0.00634766, 0.00732422, 0.00634766, 0.00634766, 0.00634766,
    0.00732422, 0.00634766, 0.00634766, 0.00634766, 0.00732422, 0.00634766, 0.00634766, 0.00634766,
    0.00732422, 0.00634766, 0.00585938, 0.00634766, 0.00732422, 0.00634766, 0.00634766, 0.00683594,
    0.00781250, 0.00634766, 0.00634766, 0.00683594, 0.00732422, 0.00634766, 0.00683594, 0.00683594,
    0.00720215, 0.00646973, 0.00720215, 0.00646973, 0.00720215, 0.00646973, 0.00720215, 0.00598145,
    0.00720215, 0.00695801, 0.00671387, 0.00598145, 0.00769043, 0.00646973, 0.00720215, 0.00646973,
    0.00769043, 0.00646973, 0.00720215, 0.00646973, 0.00769043, 0.00646973, 0.00769043, 0.00646973,
    0.00769043, 0.00695801, 0.00720215, 0.00646973, 0.00720215, 0.00695801, 0.00769043, 0.00646973,
    0.00732422, 0.00585938, 0.00634766, 0.00683594, 0.00732422, 0.00634766, 0.00634766, 0.00683594,
    0.00732422, 0.00683594, 0.00634766, 0.00634766, 0.00732422, 0.00634766, 0.00634766, 0.00634766,
    0.00732422, 0.00634766, 0.00634766, 0.00634766, 0.00732422, 0.00634766, 0.00634766, 0.00634766,
    0.00781250, 0.00634766, 0.00634766, 0.00683594, 0.00732422, 0.00634766, 0.00634766, 0.00683594,
    0.00720215, 0.00646973, 0.00720215, 0.00646973, 0.00671387, 0.00695801, 0.00720215, 0.00646973,
    0.00720215, 0.00646973, 0.00720215, 0.00598145, 0.00720215, 0.00646973, 0.00720215, 0.00646973,
    0.00720215, 0.00646973, 0.00720215, 0.00598145, 0.00720215, 0.00646973, 0.00769043, 0.00646973,
    0.00769043, 0.00695801, 0.00769043, 0.00646973, 0.00769043, 0.00695801, 0.00769043, 0.00646973,
    0.00732422, 0.00634766, 0.00634766, 0.00634766, 0.00732422, 0.00634766, 0.00634766, 0.00683594,
    0.00683594, 0.00634766, 0.00634766, 0.00634766, 0.00732422, 0.00634766, 0.00634766, 0.00634766,
    0.00732422, 0.00634766, 0.00634766, 0.00634766, 0.00732422, 0.00634766, 0.00634766, 0.00683594,
    0.00781250, 0.00634766, 0.00683594, 0.00683594, 0.00732422, 0.00634766, 0.00683594, 0.00683594,
    0.00720215, 0.00695801, 0.00720215, 0.00646973, 0.00720215, 0.00646973, 0.00720215, 0.00646973,
    0.00720215, 0.00646973, 0.00720215, 0.00646973, 0.00720215, 0.00695801, 0.00720215, 0.00646973,
    0.00769043, 0.00695801, 0.00720215, 0.00598145, 0.00769043, 0.00695801, 0.00769043, 0.00646973,
    0.00769043, 0.00695801, 0.00769043, 0.00646973, 0.00720215, 0.00695801, 0.00769043, 0.00646973,
    0.00732422, 0.00634766, 0.00634766, 0.00683594, 0.00732422, 0.00634766, 0.00634766, 0.00634766,
    0.00732422, 0.00634766, 0.00634766, 0.00634766, 0.00732422, 0.00634766, 0.00634766, 0.00634766,
    0.00732422, 0.00634766, 0.00634766, 0.00634766, 0.00732422, 0.00634766, 0.00683594, 0.00634766,
    0.00781250, 0.00634766, 0.00634766, 0.00683594, 0.00732422, 0.00634766, 0.00683594, 0.00634766,
    0.00769043, 0.00695801, 0.00720215, 0.00646973, 0.00720215, 0.00695801, 0.00720215, 0.00646973,
    0.00720215, 0.00695801, 0.00720215, 0.00598145, 0.00720215, 0.00695801, 0.00720215, 0.00646973,
    0.00720215, 0.00695801, 0.00720215, 0.00646973, 0.00769043, 0.00695801, 0.00720215, 0.00646973,
    0.00769043, 0.00646973, 0.00720215, 0.00646973, 0.00720215, 0.00695801, 0.00720215, 0.00695801,
    0.00732422, 0.00634766, 0.00634766, 0.00683594, 0.00732422, 0.00634766, 0.00634766, 0.00683594,
    0.00732422, 0.00585938, 0.00634766, 0.00634766, 0.00732422, 0.00634766, 0.00634766, 0.00634766,
    0.00732422, 0.00634766, 0.00634766, 0.00683594, 0.00732422, 0.00634766, 0.00634766, 0.00683594,
    0.00781250, 0.00634766, 0.00683594, 0.00634766, 0.00732422, 0.00634766, 0.00683594, 0.00634766,
    0.00720215, 0.00695801, 0.00720215, 0.00646973, 0.00720215, 0.00695801, 0.00720215, 0.00646973,
    0.00720215, 0.00646973, 0.00720215, 0.00646973, 0.00720215, 0.00646973, 0.00720215, 0.00646973,
    0.00769043, 0.00695801, 0.00720215, 0.00646973, 0.00720215, 0.00695801, 0.00769043, 0.00646973,
    0.00769043, 0.00695801, 0.00769043, 0.00646973, 0.00720215, 0.00695801, 0.00769043, 0.00646973,
    0.00732422, 0.00634766, 0.00634766, 0.00683594, 0.00683594, 0.00634766, 0.00634766, 0.00683594,
    0.00683594, 0.00634766, 0.00634766, 0.00683594, 0.00683594, 0.00634766, 0.00634766, 0.00683594,
    0.00732422, 0.00634766, 0.00634766, 0.00634766, 0.00732422, 0.00634766, 0.00634766, 0.00634766,
    0.00732422, 0.00634766, 0.00634766, 0.00634766, 0.00732422, 0.00634766, 0.00683594, 0.00683594,
    0.00720215, 0.00695801, 0.00720215, 0.00646973, 0.00671387, 0.00695801, 0.00769043, 0.00646973,
    0.00720215, 0.00695801, 0.00720215, 0.00646973, 0.00671387, 0.00695801, 0.00720215, 0.00646973,
    0.00720215, 0.00646973, 0.00720215, 0.00646973, 0.00720215, 0.00695801, 0.00720215, 0.00646973,
    0.00720215, 0.00695801, 0.00720215, 0.00646973, 0.00720215, 0.00695801, 0.00769043, 0.00646973,
    0.00732422, 0.00634766, 0.00634766, 0.00683594, 0.00683594, 0.00634766, 0.00634766, 0.00683594,
    0.00732422, 0.00634766, 0.00634766, 0.00634766, 0.00683594, 0.00634766, 0.00634766, 0.00683594,
    0.00732422, 0.00634766, 0.00634766, 0.00634766, 0.00732422, 0.00634766, 0.00634766, 0.00683594,
    0.00732422, 0.00634766, 0.00634766, 0.00634766, 0.00732422, 0.00634766, 0.00683594, 0.00634766,
    0.00720215, 0.00695801, 0.00720215, 0.00646973, 0.00769043, 0.00695801, 0.00720215, 0.00646973,
    0.00671387, 0.00646973, 0.00671387, 0.00646973, 0.00671387, 0.00695801, 0.00671387, 0.00646973,
    0.00720215, 0.00695801, 0.00720215, 0.00646973, 0.00720215, 0.00695801, 0.00720215, 0.00646973,
    0.00720215, 0.00695801, 0.00720215, 0.00646973, 0.00671387, 0.00695801, 0.00671387, 0.00646973,
    0.00683594, 0.00634766, 0.00634766, 0.00683594, 0.00683594, 0.00634766, 0.00634766, 0.00634766,
    0.00683594, 0.00634766, 0.00634766, 0.00634766, 0.00683594, 0.00634766, 0.00634766, 0.00634766,
    0.00683594, 0.00634766, 0.00634766, 0.00634766, 0.00732422, 0.00634766, 0.00634766, 0.00634766,
    0.00732422, 0.00634766, 0.00634766, 0.00634766, 0.00683594, 0.00634766, 0.00634766, 0.00634766,
];

pub const K_V_PIXELS: [f32; mlx90640::Mlx90640::NUM_PIXELS] = [
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000, 0.43750000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
    0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000, 0.37500000,
];

pub const FRAME_0_DATA: &[u8] = b"\
    \xFF\xB3\xFF\xAC\xFF\xB4\xFF\xAA\xFF\xB3\xFF\xAC\xFF\xB6\xFF\xA9\
    \xFF\xB2\xFF\xA8\xFF\xB4\xFF\xA6\xFF\xB1\xFF\xA5\xFF\xB4\xFF\xA2\
    \xFF\xB4\xFF\xA5\xFF\xB4\xFF\xA4\xFF\xB6\xFF\xA7\xFF\xB5\xFF\xA4\
    \xFF\xBA\xFF\xA6\xFF\xB8\xFF\xA5\xFF\xB6\xFF\xAA\xFF\xBD\xFF\xA4\
    \xFF\xA9\xFF\xA8\xFF\xA6\xFF\xA8\xFF\xA9\xFF\xA6\xFF\xA6\xFF\xA5\
    \xFF\xAA\xFF\xA2\xFF\xA5\xFF\xA2\xFF\xA9\xFF\x9F\xFF\xA5\xFF\xA1\
    \xFF\xAD\xFF\xA0\xFF\xA6\xFF\xA2\xFF\xAB\xFF\xA3\xFF\xA8\xFF\xA4\
    \xFF\xB2\xFF\xA2\xFF\xAB\xFF\xA3\xFF\xB0\xFF\xA4\xFF\xAF\xFF\xA3\
    \xFF\xB2\xFF\xAC\xFF\xB4\xFF\xAB\xFF\xB1\xFF\xAC\xFF\xB1\xFF\xA8\
    \xFF\xB1\xFF\xA9\xFF\xB3\xFF\xA5\xFF\xB2\xFF\xA5\xFF\xB1\xFF\xA3\
    \xFF\xB5\xFF\xA5\xFF\xB1\xFF\xA3\xFF\xB6\xFF\xA7\xFF\xB5\xFF\xA2\
    \xFF\xB8\xFF\xA7\xFF\xB7\xFF\xA3\xFF\xB6\xFF\xA8\xFF\xB9\xFF\xA2\
    \xFF\xA7\xFF\xA7\xFF\xA4\xFF\xA7\xFF\xA9\xFF\xA5\xFF\xA4\xFF\xA6\
    \xFF\xA8\xFF\xA3\xFF\xA4\xFF\xA2\xFF\xAC\xFF\x9F\xFF\xA2\xFF\xA3\
    \xFF\xAD\xFF\xA0\xFF\xA4\xFF\xA2\xFF\xAC\xFF\xA2\xFF\xA8\xFF\xA1\
    \xFF\xB1\xFF\xA3\xFF\xA8\xFF\xA3\xFF\xAD\xFF\xA1\xFF\xAC\xFF\xA1\
    \xFF\xB3\xFF\xAD\xFF\xB5\xFF\xA9\xFF\xB2\xFF\xAB\xFF\xB2\xFF\xA8\
    \xFF\xB4\xFF\xA9\xFF\xB1\xFF\xA4\xFF\xB1\xFF\xA6\xFF\xB3\xFF\xA2\
    \xFF\xB5\xFF\xA7\xFF\xB2\xFF\xA3\xFF\xB2\xFF\xA5\xFF\xB4\xFF\xA2\
    \xFF\xB6\xFF\xA5\xFF\xB8\xFF\xA5\xFF\xB4\xFF\xA7\xFF\xB9\xFF\xA0\
    \xFF\xAA\xFF\xA5\xFF\xA4\xFF\xA6\xFF\xAA\xFF\xA4\xFF\xA2\xFF\xA4\
    \xFF\xA9\xFF\xA1\xFF\xA2\xFF\xA3\xFF\xAA\xFF\xA1\xFF\xA3\xFF\xA0\
    \xFF\xAD\xFF\x9F\xFF\xA4\xFF\xA3\xFF\xAB\xFF\xA0\xFF\xA3\xFF\xA1\
    \xFF\xAD\xFF\x9E\xFF\xA9\xFF\xA1\xFF\xAB\xFF\xA0\xFF\xAA\xFF\x9D\
    \xFF\xB0\xFF\xAD\xFF\xB2\xFF\xAA\xFF\xB2\xFF\xAB\xFF\xB3\xFF\xA9\
    \xFF\xB8\xFF\xAA\xFF\xB6\xFF\xA4\xFF\xB2\xFF\xA6\xFF\xB1\xFF\xA4\
    \xFF\xB2\xFF\xA4\xFF\xB2\xFF\xA4\xFF\xB2\xFF\xA7\xFF\xB5\xFF\xA4\
    \xFF\xB2\xFF\xA5\xFF\xB4\xFF\xA2\xFF\xB4\xFF\xA6\xFF\xB9\xFF\xA2\
    \xFF\xA5\xFF\xA4\xFF\xA1\xFF\xA4\xFF\xA6\xFF\xA0\xFF\xA1\xFF\xA3\
    \xFF\xA8\xFF\xA7\xFF\xA1\xFF\xAA\xFF\xA6\xFF\xA1\xFF\xA1\xFF\xA1\
    \xFF\xA8\xFF\x9D\xFF\xA2\xFF\x9F\xFF\xA8\xFF\x9F\xFF\xA2\xFF\x9F\
    \xFF\xAB\xFF\x9E\xFF\xA4\xFF\xA0\xFF\xAB\xFF\x9F\xFF\xA8\xFF\x9B\
    \xFF\xAF\xFF\xAE\xFF\xB3\xFF\xA9\xFF\xAF\xFF\xAB\xFF\xB4\xFF\xA8\
    \xFF\xBB\xFF\xAA\xFF\xC6\xFF\xA6\xFF\xC0\xFF\xA8\xFF\xB6\xFF\xA2\
    \xFF\xB0\xFF\xA5\xFF\xB1\xFF\xA2\xFF\xB3\xFF\xA6\xFF\xB2\xFF\xA2\
    \xFF\xB4\xFF\xA2\xFF\xB4\xFF\xA2\xFF\xB4\xFF\xA6\xFF\xB7\xFF\xA1\
    \xFF\xA3\xFF\xA2\xFF\x9F\xFF\xA1\xFF\xA5\xFF\xA2\xFF\xA2\xFF\xA4\
    \xFF\xAA\xFF\xB2\xFF\xA4\xFF\xB4\xFF\xA8\xFF\xAD\xFF\xA1\xFF\xA6\
    \xFF\xA8\xFF\x9D\xFF\xA2\xFF\x9D\xFF\xAB\xFF\x9D\xFF\xA3\xFF\x9F\
    \xFF\xAD\xFF\x9C\xFF\xA3\xFF\x9F\xFF\xAB\xFF\x9D\xFF\xA6\xFF\x9C\
    \xFF\xB3\xFF\xAD\xFF\xB3\xFF\xA9\xFF\xB4\xFF\xAC\xFF\xB5\xFF\xA9\
    \xFF\xC8\xFF\xAE\xFF\xC8\xFF\xAB\xFF\xC9\xFF\xAE\xFF\xC2\xFF\xA6\
    \xFF\xBD\xFF\xA9\xFF\xB5\xFF\xA8\xFF\xB2\xFF\xAC\xFF\xB0\xFF\xA4\
    \xFF\xB3\xFF\xA4\xFF\xB2\xFF\xA1\xFF\xB2\xFF\xA4\xFF\xB4\xFF\x9C\
    \xFF\xA4\xFF\xA1\xFF\x9F\xFF\xA2\xFF\xA7\xFF\xA1\xFF\xA2\xFF\xAA\
    \xFF\xAD\xFF\xB7\xFF\xA7\xFF\xB8\xFF\xAD\xFF\xB5\xFF\xA3\xFF\xB1\
    \xFF\xAF\xFF\xA8\xFF\xAA\xFF\xA0\xFF\xB2\xFF\x9B\xFF\xA5\xFF\x9D\
    \xFF\xAC\xFF\x9A\xFF\xA4\xFF\x9D\xFF\xAB\xFF\x9C\xFF\xA5\xFF\x9A\
    \xFF\xAE\xFF\xAD\xFF\xAD\xFF\xA8\xFF\xB1\xFF\xAD\xFF\xBB\xFF\xAE\
    \xFF\xCB\xFF\xB2\xFF\xCE\xFF\xAE\xFF\xCB\xFF\xB0\xFF\xC5\xFF\xAB\
    \xFF\xC6\xFF\xB2\xFF\xBD\xFF\xB0\xFF\xB2\xFF\xB1\xFF\xB0\xFF\xA8\
    \xFF\xB3\xFF\xAA\xFF\xB1\xFF\xA4\xFF\xB1\xFF\xA5\xFF\xB4\xFF\x9D\
    \xFF\x9F\xFF\x9E\xFF\x98\xFF\x9F\xFF\xA3\xFF\xA0\xFF\xA2\xFF\xB3\
    \xFF\xAC\xFF\xBB\xFF\xA7\xFF\xBF\xFF\xAE\xFF\xB7\xFF\xA5\xFF\xB3\
    \xFF\xB4\xFF\xB0\xFF\xAE\xFF\xA6\xFF\xB6\xFF\x9B\xFF\xAB\xFF\x99\
    \xFF\xB4\xFF\x9A\xFF\xA7\xFF\x9D\xFF\xAC\xFF\x9D\xFF\xA5\xFF\x9A\
    \xFF\xAC\xFF\xAD\xFF\xB0\xFF\xA8\xFF\xB1\xFF\xAD\xFF\xC3\xFF\xAF\
    \xFF\xCA\xFF\xB3\xFF\xCD\xFF\xAD\xFF\xCA\xFF\xB0\xFF\xC8\xFF\xAD\
    \xFF\xC6\xFF\xB8\xFF\xB9\xFF\xB1\xFF\xB2\xFF\xB5\xFF\xAF\xFF\xAD\
    \xFF\xB3\xFF\xAF\xFF\xB0\xFF\xA9\xFF\xB0\xFF\xA6\xFF\xB4\xFF\x9D\
    \xFF\x9E\xFF\x9C\xFF\x9C\xFF\x9F\xFF\xA1\xFF\xA4\xFF\xA0\xFF\xB3\
    \xFF\xAB\xFF\xB5\xFF\xA4\xFF\xB9\xFF\xAC\xFF\xB3\xFF\xA7\xFF\xB2\
    \xFF\xB5\xFF\xAF\xFF\xB1\xFF\x9D\xFF\xB7\xFF\x9B\xFF\xAD\xFF\x9A\
    \xFF\xB6\xFF\x9A\xFF\xA8\xFF\x9C\xFF\xAA\xFF\x9B\xFF\xA5\xFF\x9A\
    \xFF\xAE\xFF\xAC\xFF\xB0\xFF\xAB\xFF\xB6\xFF\xAE\xFF\xC0\xFF\xAD\
    \xFF\xC3\xFF\xB0\xFF\xC2\xFF\xAB\xFF\xC4\xFF\xB0\xFF\xC2\xFF\xB0\
    \xFF\xC4\xFF\xB8\xFF\xB2\xFF\xB3\xFF\xAE\xFF\xB4\xFF\xAE\xFF\xAF\
    \xFF\xB0\xFF\xAF\xFF\xAF\xFF\xA5\xFF\xB2\xFF\xA4\xFF\xB3\xFF\x9D\
    \xFF\x9E\xFF\x9A\xFF\x9A\xFF\x9F\xFF\xA1\xFF\xA4\xFF\x9D\xFF\xAF\
    \xFF\xA7\xFF\xAC\xFF\x9F\xFF\xAD\xFF\xAA\xFF\xB0\xFF\xA7\xFF\xAF\
    \xFF\xB5\xFF\xA6\xFF\xAF\xFF\x9A\xFF\xB5\xFF\x98\xFF\xAA\xFF\x9B\
    \xFF\xB3\xFF\x99\xFF\xA5\xFF\x9A\xFF\xAA\xFF\x9F\xFF\xA5\xFF\x9A\
    \xFF\xA7\xFF\xAC\xFF\xAA\xFF\xA8\xFF\xAA\xFF\xAD\xFF\xB0\xFF\xAB\
    \xFF\xB9\xFF\xAD\xFF\xBF\xFF\xAB\xFF\xBD\xFF\xAF\xFF\xC0\xFF\xB0\
    \xFF\xBA\xFF\xB4\xFF\xAE\xFF\xAF\xFF\xAC\xFF\xB0\xFF\xAC\xFF\xAC\
    \xFF\xB0\xFF\xAE\xFF\xB0\xFF\xA6\xFF\xBB\xFF\xA5\xFF\xBC\xFF\xA0\
    \xFF\x96\xFF\x96\xFF\x92\xFF\x99\xFF\x99\xFF\x98\xFF\x97\xFF\x9E\
    \xFF\xA0\xFF\xA0\xFF\x9E\xFF\xA7\xFF\xA5\xFF\xA8\xFF\xA3\xFF\xA9\
    \xFF\xAE\xFF\x9A\xFF\xA5\xFF\x97\xFF\xAE\xFF\x95\xFF\xA6\xFF\x99\
    \xFF\xAF\xFF\x98\xFF\xA2\xFF\xA0\xFF\xAB\xFF\xA9\xFF\xA3\xFF\xA6\
    \xFF\xA4\xFF\xAE\xFF\xA7\xFF\xA5\xFF\xA7\xFF\xA7\xFF\xA9\xFF\xA6\
    \xFF\xAC\xFF\xA6\xFF\xB0\xFF\xA3\xFF\xB7\xFF\xAD\xFF\xB7\xFF\xA9\
    \xFF\xAF\xFF\xAB\xFF\xA8\xFF\xA8\xFF\xAC\xFF\xAD\xFF\xAB\xFF\xAA\
    \xFF\xAF\xFF\xAE\xFF\xB6\xFF\xA7\xFF\xBC\xFF\xAB\xFF\xC4\xFF\xA4\
    \xFF\x93\xFF\x95\xFF\x90\xFF\x94\xFF\x94\xFF\x93\xFF\x92\xFF\x96\
    \xFF\x99\xFF\x93\xFF\x96\xFF\x97\xFF\xA0\xFF\x9E\xFF\x9B\xFF\x9E\
    \xFF\xA4\xFF\x93\xFF\x9D\xFF\x94\xFF\xA9\xFF\x96\xFF\x9F\xFF\x96\
    \xFF\xAB\xFF\x97\xFF\xA1\xFF\xA5\xFF\xA9\xFF\xAA\xFF\xA4\xFF\xA6\
    \xFF\xA4\xFF\xAC\xFF\xA4\xFF\xA6\xFF\xA6\xFF\xA7\xFF\xA6\xFF\xA1\
    \xFF\xA5\xFF\xA6\xFF\xA4\xFF\xA3\xFF\xA7\xFF\xA4\xFF\xA7\xFF\x9F\
    \xFF\xAB\xFF\xA3\xFF\xA9\xFF\xA3\xFF\xAA\xFF\xA7\xFF\xAB\xFF\xA6\
    \xFF\xAE\xFF\xAA\xFF\xB7\xFF\xA6\xFF\xBB\xFF\xAA\xFF\xBB\xFF\xA3\
    \xFF\x87\xFF\x8A\xFF\x84\xFF\x8C\xFF\x8A\xFF\x8B\xFF\x86\xFF\x8B\
    \xFF\x8B\xFF\x89\xFF\x85\xFF\x8B\xFF\x8F\xFF\x89\xFF\x8A\xFF\x8B\
    \xFF\x91\xFF\x8A\xFF\x8C\xFF\x8D\xFF\x9A\xFF\x8B\xFF\x95\xFF\x8E\
    \xFF\x9E\xFF\x93\xFF\x98\xFF\x9D\xFF\x9E\xFF\x9D\xFF\x9A\xFF\x99\
    \x4D\xFA\x1A\x56\x7F\xFF\x1A\x56\x7F\xFF\x1A\x55\x7F\xFF\x1A\x55\
    \xFF\xB9\xCE\x07\x15\x84\xD6\x53\xFF\xF9\x00\x09\x00\x00\xFF\xFD\
    \x19\x76\x03\xFD\x02\x97\x7F\xFF\x19\x76\x03\xFD\x02\x97\x7F\xFF\
    \x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\
    \x06\x95\x7F\xFF\x1A\x56\x7F\xFF\x1A\x56\x7F\xFF\x1A\x55\x7F\xFF\
    \xFF\xBD\xF5\x7A\xCE\xF2\xD8\xE0\x00\x09\xFF\xFD\xFF\xFC\x00\x00\
    \x00\xED\x00\x46\x2A\xD6\x00\x35\x00\xEE\x00\x46\x2A\xD6\x00\x35\
    \x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01";

pub const FRAME_0_STATUS_REGISTER: &[u8] = b"\x00\x00";

pub const FRAME_1_DATA: &[u8] = b"\
    \xFF\xB3\xFF\xAD\xFF\xB4\xFF\xAA\xFF\xB3\xFF\xAD\xFF\xB6\xFF\xA8\
    \xFF\xB2\xFF\xA9\xFF\xB4\xFF\xA4\xFF\xB1\xFF\xA6\xFF\xB4\xFF\xA1\
    \xFF\xB4\xFF\xA8\xFF\xB4\xFF\xA1\xFF\xB6\xFF\xA8\xFF\xB5\xFF\xA2\
    \xFF\xBA\xFF\xA7\xFF\xB8\xFF\xA3\xFF\xB6\xFF\xAA\xFF\xBD\xFF\xA3\
    \xFF\xAC\xFF\xA8\xFF\xA5\xFF\xA8\xFF\xAC\xFF\xA6\xFF\xA5\xFF\xA5\
    \xFF\xAA\xFF\xA2\xFF\xA2\xFF\xA2\xFF\xAB\xFF\x9F\xFF\xA0\xFF\xA1\
    \xFF\xAB\xFF\xA0\xFF\xA4\xFF\xA2\xFF\xAB\xFF\xA3\xFF\xA6\xFF\xA4\
    \xFF\xB1\xFF\xA2\xFF\xA9\xFF\xA3\xFF\xB0\xFF\xA4\xFF\xAD\xFF\xA3\
    \xFF\xB2\xFF\xAF\xFF\xB4\xFF\xA8\xFF\xB1\xFF\xAE\xFF\xB1\xFF\xA5\
    \xFF\xB1\xFF\xAA\xFF\xB3\xFF\xA3\xFF\xB2\xFF\xA7\xFF\xB1\xFF\xA3\
    \xFF\xB5\xFF\xA6\xFF\xB1\xFF\xA0\xFF\xB6\xFF\xA7\xFF\xB5\xFF\xA0\
    \xFF\xB8\xFF\xA7\xFF\xB7\xFF\xA2\xFF\xB6\xFF\xA8\xFF\xB9\xFF\x9F\
    \xFF\xAA\xFF\xA7\xFF\xA4\xFF\xA7\xFF\xA9\xFF\xA5\xFF\xA3\xFF\xA6\
    \xFF\xA9\xFF\xA3\xFF\xA3\xFF\xA2\xFF\xAA\xFF\x9F\xFF\xA2\xFF\xA3\
    \xFF\xAD\xFF\xA0\xFF\xA2\xFF\xA2\xFF\xAD\xFF\xA2\xFF\xA6\xFF\xA1\
    \xFF\xB1\xFF\xA3\xFF\xA7\xFF\xA3\xFF\xAD\xFF\xA1\xFF\xA9\xFF\xA1\
    \xFF\xB3\xFF\xAF\xFF\xB5\xFF\xAA\xFF\xB2\xFF\xAD\xFF\xB2\xFF\xA6\
    \xFF\xB4\xFF\xAB\xFF\xB1\xFF\xA4\xFF\xB1\xFF\xA8\xFF\xB3\xFF\xA0\
    \xFF\xB5\xFF\xA6\xFF\xB2\xFF\xA3\xFF\xB2\xFF\xA8\xFF\xB4\xFF\xA1\
    \xFF\xB6\xFF\xA6\xFF\xB8\xFF\xA2\xFF\xB4\xFF\xA7\xFF\xB9\xFF\x9C\
    \xFF\xAB\xFF\xA5\xFF\xA3\xFF\xA6\xFF\xAA\xFF\xA4\xFF\xA4\xFF\xA4\
    \xFF\xB0\xFF\xA1\xFF\xA4\xFF\xA3\xFF\xA9\xFF\xA1\xFF\xA0\xFF\xA0\
    \xFF\xAE\xFF\x9F\xFF\xA2\xFF\xA3\xFF\xAD\xFF\xA0\xFF\xA4\xFF\xA1\
    \xFF\xAE\xFF\x9E\xFF\xA8\xFF\xA1\xFF\xAC\xFF\xA0\xFF\xA8\xFF\x9D\
    \xFF\xB0\xFF\xAE\xFF\xB2\xFF\xA8\xFF\xB2\xFF\xAC\xFF\xB3\xFF\xA9\
    \xFF\xB8\xFF\xB8\xFF\xB6\xFF\xAB\xFF\xB2\xFF\xAA\xFF\xB1\xFF\xA1\
    \xFF\xB2\xFF\xA5\xFF\xB2\xFF\xA0\xFF\xB2\xFF\xA5\xFF\xB5\xFF\xA1\
    \xFF\xB2\xFF\xA6\xFF\xB4\xFF\xA3\xFF\xB4\xFF\xA5\xFF\xB9\xFF\x9B\
    \xFF\xA6\xFF\xA4\xFF\xA2\xFF\xA4\xFF\xA7\xFF\xA0\xFF\xA2\xFF\xA3\
    \xFF\xBB\xFF\xA7\xFF\xAF\xFF\xAA\xFF\xAF\xFF\xA1\xFF\xA1\xFF\xA1\
    \xFF\xA9\xFF\x9D\xFF\xA0\xFF\x9F\xFF\xA8\xFF\x9F\xFF\xA2\xFF\x9F\
    \xFF\xAC\xFF\x9E\xFF\xA4\xFF\xA0\xFF\xAC\xFF\x9F\xFF\xA6\xFF\x9B\
    \xFF\xAF\xFF\xAF\xFF\xB3\xFF\xA9\xFF\xAF\xFF\xAF\xFF\xB4\xFF\xB0\
    \xFF\xBB\xFF\xC0\xFF\xC6\xFF\xB7\xFF\xC0\xFF\xB7\xFF\xB6\xFF\xA7\
    \xFF\xB0\xFF\xA6\xFF\xB1\xFF\xA0\xFF\xB3\xFF\xA4\xFF\xB2\xFF\xA0\
    \xFF\xB4\xFF\xA4\xFF\xB4\xFF\xA0\xFF\xB4\xFF\xA5\xFF\xB7\xFF\x9A\
    \xFF\xA6\xFF\xA2\xFF\x9F\xFF\xA1\xFF\xA8\xFF\xA2\xFF\xAA\xFF\xA4\
    \xFF\xBE\xFF\xB2\xFF\xB6\xFF\xB4\xFF\xBE\xFF\xAD\xFF\xAC\xFF\xA6\
    \xFF\xAC\xFF\x9D\xFF\xA0\xFF\x9D\xFF\xA9\xFF\x9D\xFF\xA0\xFF\x9F\
    \xFF\xAA\xFF\x9C\xFF\xA1\xFF\x9F\xFF\xAC\xFF\x9D\xFF\xA2\xFF\x9C\
    \xFF\xB3\xFF\xAF\xFF\xB3\xFF\xAA\xFF\xB4\xFF\xB1\xFF\xB5\xFF\xBB\
    \xFF\xC8\xFF\xC4\xFF\xC8\xFF\xBC\xFF\xC9\xFF\xBF\xFF\xC2\xFF\xB5\
    \xFF\xBD\xFF\xB0\xFF\xB5\xFF\xA0\xFF\xB2\xFF\xA6\xFF\xB0\xFF\x9E\
    \xFF\xB3\xFF\xA3\xFF\xB2\xFF\x9E\xFF\xB2\xFF\xA4\xFF\xB4\xFF\x9B\
    \xFF\xA8\xFF\xA1\xFF\x9D\xFF\xA2\xFF\xAC\xFF\xA1\xFF\xB1\xFF\xAA\
    \xFF\xC7\xFF\xB7\xFF\xB9\xFF\xB8\xFF\xC0\xFF\xB5\xFF\xB2\xFF\xB1\
    \xFF\xBB\xFF\xA8\xFF\xA5\xFF\xA0\xFF\xA7\xFF\x9B\xFF\x9C\xFF\x9D\
    \xFF\xA9\xFF\x9A\xFF\xA1\xFF\x9D\xFF\xA9\xFF\x9C\xFF\xA5\xFF\x9A\
    \xFF\xAE\xFF\xAD\xFF\xAD\xFF\xA7\xFF\xB1\xFF\xB8\xFF\xBB\xFF\xC0\
    \xFF\xCB\xFF\xCA\xFF\xCE\xFF\xC1\xFF\xCB\xFF\xBF\xFF\xC5\xFF\xB6\
    \xFF\xC6\xFF\xB6\xFF\xBD\xFF\xA0\xFF\xB2\xFF\xA4\xFF\xB0\xFF\x9E\
    \xFF\xB3\xFF\xA1\xFF\xB1\xFF\x9F\xFF\xB1\xFF\xA4\xFF\xB4\xFF\x9C\
    \xFF\xA3\xFF\x9E\xFF\x9A\xFF\x9F\xFF\xAE\xFF\xA0\xFF\xB1\xFF\xB3\
    \xFF\xC1\xFF\xBB\xFF\xB9\xFF\xBF\xFF\xBF\xFF\xB7\xFF\xB1\xFF\xB3\
    \xFF\xB8\xFF\xB0\xFF\xA0\xFF\xA6\xFF\xA7\xFF\x9B\xFF\x9E\xFF\x99\
    \xFF\xAB\xFF\x9A\xFF\xA0\xFF\x9D\xFF\xAA\xFF\x9D\xFF\xA2\xFF\x9A\
    \xFF\xAC\xFF\xAF\xFF\xB0\xFF\xA9\xFF\xB1\xFF\xBD\xFF\xC3\xFF\xBA\
    \xFF\xCA\xFF\xC2\xFF\xCD\xFF\xBC\xFF\xCA\xFF\xBE\xFF\xC8\xFF\xB7\
    \xFF\xC6\xFF\xB1\xFF\xB9\xFF\xA0\xFF\xB2\xFF\xA5\xFF\xAF\xFF\x9D\
    \xFF\xB3\xFF\xA4\xFF\xB0\xFF\xA0\xFF\xB0\xFF\xA3\xFF\xB4\xFF\x9B\
    \xFF\x9E\xFF\x9C\xFF\x9C\xFF\x9F\xFF\xAE\xFF\xA4\xFF\xAE\xFF\xB3\
    \xFF\xB8\xFF\xB5\xFF\xAF\xFF\xB9\xFF\xBA\xFF\xB3\xFF\xAF\xFF\xB2\
    \xFF\xB4\xFF\xAF\xFF\x9D\xFF\x9D\xFF\xA5\xFF\x9B\xFF\x9D\xFF\x9A\
    \xFF\xA9\xFF\x9A\xFF\xA1\xFF\x9C\xFF\xA8\xFF\x9B\xFF\xA3\xFF\x9A\
    \xFF\xAE\xFF\xAF\xFF\xB0\xFF\xA8\xFF\xB6\xFF\xB5\xFF\xC0\xFF\xB5\
    \xFF\xC3\xFF\xBC\xFF\xC2\xFF\xB6\xFF\xC4\xFF\xBD\xFF\xC2\xFF\xB0\
    \xFF\xC4\xFF\xA9\xFF\xB2\xFF\x9E\xFF\xAE\xFF\xA3\xFF\xAE\xFF\x9E\
    \xFF\xB0\xFF\xA3\xFF\xAF\xFF\xA1\xFF\xB2\xFF\xA6\xFF\xB3\xFF\x9E\
    \xFF\x9E\xFF\x9A\xFF\x98\xFF\x9F\xFF\xA4\xFF\xA4\xFF\xA2\xFF\xAF\
    \xFF\xB0\xFF\xAC\xFF\xAA\xFF\xAD\xFF\xB6\xFF\xB0\xFF\xAA\xFF\xAF\
    \xFF\xAB\xFF\xA6\xFF\x99\xFF\x9A\xFF\xA3\xFF\x98\xFF\x99\xFF\x9B\
    \xFF\xA7\xFF\x99\xFF\xA2\xFF\x9A\xFF\xAE\xFF\x9F\xFF\xA3\xFF\x9A\
    \xFF\xA7\xFF\xAC\xFF\xAA\xFF\xA5\xFF\xAA\xFF\xAA\xFF\xB0\xFF\xA8\
    \xFF\xB9\xFF\xB1\xFF\xBF\xFF\xB2\xFF\xBD\xFF\xB8\xFF\xC0\xFF\xA7\
    \xFF\xBA\xFF\xA4\xFF\xAE\xFF\x9D\xFF\xAC\xFF\xA3\xFF\xAC\xFF\x9D\
    \xFF\xB0\xFF\xA4\xFF\xB0\xFF\xAA\xFF\xBB\xFF\xB3\xFF\xBC\xFF\xA3\
    \xFF\x99\xFF\x96\xFF\x8F\xFF\x99\xFF\x97\xFF\x98\xFF\x93\xFF\x9E\
    \xFF\x9E\xFF\xA0\xFF\x9E\xFF\xA7\xFF\xAC\xFF\xA8\xFF\x9F\xFF\xA9\
    \xFF\xA1\xFF\x9A\xFF\x94\xFF\x97\xFF\xA0\xFF\x95\xFF\x98\xFF\x99\
    \xFF\xA6\xFF\x98\xFF\xA5\xFF\xA0\xFF\xB8\xFF\xA9\xFF\xAD\xFF\xA6\
    \xFF\xA4\xFF\xAC\xFF\xA7\xFF\xA3\xFF\xA7\xFF\xA7\xFF\xA9\xFF\xA3\
    \xFF\xAC\xFF\xA6\xFF\xB0\xFF\xA4\xFF\xB7\xFF\xAD\xFF\xB7\xFF\xA1\
    \xFF\xAF\xFF\x9F\xFF\xA8\xFF\x9C\xFF\xAC\xFF\xA3\xFF\xAB\xFF\x9E\
    \xFF\xAF\xFF\xAA\xFF\xB6\xFF\xAE\xFF\xBC\xFF\xBA\xFF\xC4\xFF\xA9\
    \xFF\x95\xFF\x95\xFF\x8E\xFF\x94\xFF\x96\xFF\x93\xFF\x91\xFF\x96\
    \xFF\x99\xFF\x93\xFF\x94\xFF\x97\xFF\x9E\xFF\x9E\xFF\x94\xFF\x9E\
    \xFF\x9D\xFF\x93\xFF\x91\xFF\x94\xFF\x9F\xFF\x96\xFF\x95\xFF\x96\
    \xFF\xA8\xFF\x97\xFF\xA6\xFF\xA5\xFF\xB6\xFF\xAA\xFF\xAB\xFF\xA6\
    \xFF\xA4\xFF\xAA\xFF\xA4\xFF\xA2\xFF\xA6\xFF\xA7\xFF\xA6\xFF\x9E\
    \xFF\xA5\xFF\xA4\xFF\xA4\xFF\x9F\xFF\xA7\xFF\xA1\xFF\xA7\xFF\x9B\
    \xFF\xAB\xFF\xA0\xFF\xA9\xFF\x9D\xFF\xAA\xFF\xA1\xFF\xAB\xFF\x9F\
    \xFF\xAE\xFF\xAE\xFF\xB7\xFF\xAD\xFF\xBB\xFF\xB5\xFF\xBB\xFF\xA3\
    \xFF\x87\xFF\x8A\xFF\x80\xFF\x8C\xFF\x8B\xFF\x8B\xFF\x83\xFF\x8B\
    \xFF\x8B\xFF\x89\xFF\x86\xFF\x8B\xFF\x8E\xFF\x89\xFF\x86\xFF\x8B\
    \xFF\x93\xFF\x8A\xFF\x87\xFF\x8D\xFF\x95\xFF\x8B\xFF\x8D\xFF\x8E\
    \xFF\xA6\xFF\x93\xFF\xA0\xFF\x9D\xFF\xA9\xFF\x9D\xFF\x9C\xFF\x99\
    \x4D\xFA\x1A\x58\x7F\xFF\x1A\x58\x7F\xFF\x1A\x57\x7F\xFF\x1A\x57\
    \xFF\xB9\xCE\x35\x15\x84\xD6\x4E\xFF\xF9\x00\x0B\x00\x00\xFF\xFE\
    \x19\x76\x03\xFD\x02\x97\x7F\xFF\x19\x76\x03\xFD\x02\x97\x7F\xFF\
    \x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\
    \x06\x96\x7F\xFF\x1A\x58\x7F\xFF\x1A\x58\x7F\xFF\x1A\x57\x7F\xFF\
    \xFF\xBF\xF5\x7A\xCF\x18\xD8\xE0\x00\x0A\xFF\xFD\xFF\xFE\x00\x00\
    \x00\xF1\x00\x46\x2A\xC3\x00\x35\x00\xF1\x00\x46\x2A\xC3\x00\x35\
    \x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01";

pub const FRAME_1_STATUS_REGISTER: &[u8] = b"\x00\x01";

// This is given separately in the spreadsheet for each frame of data, but it's the same value for
// both frames.
pub const CONTROL_REGISTER: &[u8] = b"\x19\x01";

// This covers the temperatures after both frames have been processed
pub const TEMPERATURES: [f32; mlx90640::Mlx90640::NUM_PIXELS] = [
    28.711, 27.792, 28.178, 28.268, 28.313, 28.450, 28.631, 28.353, 28.295, 28.221, 28.586, 28.024,
    28.336, 28.298, 28.683, 27.972, 28.248, 28.537, 28.573, 28.238, 29.127, 28.777, 28.629, 28.172,
    29.070, 28.718, 28.894, 28.509, 28.883, 29.076, 29.687, 29.163, 28.942, 28.618, 28.332, 28.134,
    28.541, 28.517, 28.394, 27.982, 28.318, 28.092, 28.151, 27.914, 28.753, 28.156, 27.861, 28.261,
    28.118, 28.198, 28.582, 28.528, 28.359, 28.837, 28.618, 28.730, 28.610, 28.814, 28.605, 28.558,
    28.881, 28.605, 29.019, 29.005, 28.513, 28.605, 28.482, 28.154, 28.428, 28.766, 27.946, 28.048,
    28.248, 28.450, 28.387, 28.199, 28.340, 28.594, 28.128, 28.421, 28.544, 28.256, 28.321, 28.073,
    28.986, 28.410, 28.445, 28.408, 28.728, 28.530, 28.778, 28.326, 29.035, 28.677, 28.639, 28.351,
    28.483, 28.116, 28.430, 28.237, 28.430, 28.173, 28.300, 28.519, 28.179, 28.303, 28.405, 28.049,
    28.329, 28.246, 28.347, 28.466, 28.346, 28.247, 28.304, 28.378, 28.597, 28.835, 28.656, 28.480,
    28.723, 28.828, 28.500, 28.428, 28.796, 28.528, 28.297, 29.030, 28.163, 28.340, 28.719, 28.336,
    28.268, 28.561, 28.439, 28.254, 28.607, 28.734, 28.199, 28.376, 28.440, 28.248, 28.730, 27.926,
    28.361, 28.278, 28.256, 28.490, 28.253, 28.898, 28.563, 28.491, 28.593, 28.432, 28.962, 28.417,
    28.746, 28.817, 28.917, 27.554, 28.126, 28.112, 28.473, 27.955, 28.466, 28.365, 28.968, 28.291,
    29.580, 28.335, 28.803, 28.610, 28.409, 28.281, 28.132, 28.303, 28.803, 28.246, 28.443, 28.710,
    28.649, 28.723, 28.537, 28.532, 28.564, 28.493, 28.667, 28.686, 28.746, 28.565, 28.278, 28.197,
    28.489, 28.339, 28.295, 28.444, 28.610, 28.566, 28.497, 28.802, 29.183, 31.090, 29.365, 29.834,
    28.763, 28.843, 28.320, 28.427, 28.594, 28.279, 28.641, 28.140, 28.629, 28.230, 28.950, 28.392,
    27.980, 28.579, 28.349, 28.584, 28.453, 28.300, 28.852, 27.232, 28.490, 28.364, 28.630, 28.244,
    28.633, 28.023, 28.703, 28.247, 31.642, 29.572, 31.352, 30.034, 29.809, 28.667, 28.683, 28.642,
    28.605, 28.316, 28.433, 28.311, 28.448, 28.444, 28.483, 28.168, 28.580, 28.403, 28.510, 28.438,
    28.743, 28.800, 27.995, 27.913, 28.091, 28.460, 28.598, 28.515, 28.313, 28.996, 28.813, 30.302,
    30.533, 32.592, 32.362, 32.093, 31.328, 31.273, 29.314, 29.422, 28.126, 28.591, 28.517, 28.460,
    28.644, 28.192, 28.679, 28.554, 28.485, 28.324, 28.714, 28.284, 28.549, 28.354, 28.739, 27.168,
    28.536, 28.275, 28.361, 27.955, 28.948, 28.448, 30.536, 28.965, 32.884, 31.879, 32.682, 32.094,
    32.728, 31.286, 30.920, 29.814, 29.226, 28.593, 28.639, 28.303, 28.674, 28.808, 28.445, 28.596,
    28.308, 28.363, 28.237, 28.585, 28.762, 28.420, 27.634, 28.341, 28.595, 28.757, 28.668, 28.844,
    28.724, 29.482, 29.526, 32.560, 32.332, 33.558, 32.878, 32.994, 32.587, 32.664, 32.080, 32.187,
    30.319, 30.384, 29.259, 28.561, 28.347, 28.620, 28.371, 28.259, 28.591, 28.458, 28.779, 28.458,
    28.650, 28.496, 28.577, 28.159, 28.835, 28.330, 28.219, 28.480, 29.530, 28.753, 32.294, 30.412,
    34.299, 33.225, 33.583, 33.175, 32.902, 32.848, 32.213, 31.967, 31.870, 30.751, 29.610, 29.090,
    28.328, 28.461, 27.930, 28.636, 28.181, 28.437, 28.476, 28.707, 28.600, 28.479, 28.681, 28.493,
    28.450, 28.323, 28.205, 28.425, 28.945, 31.092, 30.841, 33.522, 33.628, 34.564, 34.155, 33.856,
    33.454, 32.664, 32.621, 32.372, 32.219, 31.654, 31.181, 28.926, 28.945, 28.697, 28.535, 28.474,
    28.354, 28.337, 28.338, 28.403, 28.487, 28.216, 28.645, 28.384, 28.889, 28.420, 28.804, 28.235,
    30.758, 29.189, 32.667, 32.258, 33.935, 34.054, 33.947, 34.386, 33.459, 33.366, 32.776, 32.502,
    31.703, 32.534, 29.363, 30.553, 28.741, 28.703, 28.474, 28.123, 28.522, 28.465, 28.276, 28.435,
    28.844, 28.666, 28.557, 28.458, 28.412, 28.779, 28.618, 28.839, 29.437, 31.981, 32.515, 32.602,
    33.633, 33.339, 34.348, 33.200, 33.460, 32.708, 33.297, 32.716, 32.441, 30.769, 30.583, 28.879,
    28.901, 28.662, 28.751, 28.224, 28.455, 28.549, 28.551, 28.501, 28.674, 28.491, 28.816, 28.358,
    28.403, 28.379, 29.006, 28.929, 31.579, 30.135, 32.674, 32.781, 32.740, 33.364, 32.741, 33.703,
    32.951, 33.035, 32.577, 32.681, 31.359, 32.508, 28.893, 29.271, 28.502, 28.813, 28.726, 28.176,
    28.417, 28.281, 28.839, 28.308, 28.837, 28.441, 28.693, 28.851, 28.266, 28.880, 28.759, 28.696,
    30.103, 30.616, 32.329, 31.805, 32.560, 32.485, 32.963, 32.454, 32.587, 32.690, 32.781, 31.630,
    32.408, 29.459, 29.431, 28.422, 28.536, 28.365, 28.849, 28.236, 28.187, 28.445, 28.597, 29.407,
    29.227, 29.049, 28.880, 29.072, 28.191, 28.259, 28.706, 28.992, 29.507, 30.442, 30.965, 32.244,
    31.855, 32.250, 32.388, 32.312, 32.607, 32.854, 31.842, 32.680, 30.104, 31.065, 28.447, 28.621,
    28.679, 28.344, 28.156, 28.447, 28.570, 28.667, 29.436, 28.698, 30.319, 29.718, 28.776, 28.963,
    28.429, 28.555, 28.570, 28.395, 29.163, 28.748, 29.828, 29.420, 31.671, 30.793, 32.379, 31.911,
    32.265, 32.313, 32.592, 30.116, 31.175, 28.890, 29.218, 28.496, 28.587, 28.648, 28.508, 28.310,
    28.957, 28.748, 29.226, 31.269, 31.199, 32.328, 31.547, 29.811, 28.888, 28.137, 28.285, 28.463,
    28.483, 28.557, 28.797, 29.704, 29.399, 30.460, 30.472, 31.603, 31.694, 31.990, 30.547, 32.004,
    28.930, 29.626, 28.220, 28.473, 28.552, 28.220, 28.351, 28.694, 29.062, 28.703, 30.867, 29.831,
    32.820, 32.548, 32.478, 32.176, 28.473, 28.493, 28.264, 28.442, 28.662, 28.390, 28.372, 28.661,
    29.225, 28.774, 29.797, 29.597, 30.999, 30.204, 31.244, 29.254, 29.575, 28.162, 28.510, 28.532,
    28.530, 28.600, 28.449, 28.428, 28.642, 29.765, 30.602, 32.329, 32.418, 34.008, 34.239, 32.016,
    28.925, 28.322, 28.395, 28.078, 28.614, 28.183, 28.452, 28.517, 28.848, 28.593, 28.970, 29.005,
    29.060, 30.394, 28.891, 30.257, 28.808, 28.746, 28.235, 28.295, 28.682, 28.561, 28.342, 28.250,
    29.916, 28.593, 31.676, 31.333, 33.850, 33.254, 32.702, 33.224, 28.516, 28.235, 28.317, 28.028,
    28.465, 28.464, 28.845, 28.247, 28.642, 28.392, 28.464, 28.436, 28.858, 28.573, 28.979, 28.471,
    29.039, 28.500, 28.832, 28.364, 28.280, 28.374, 28.837, 28.830, 29.056, 31.258, 31.946, 32.501,
    33.101, 33.231, 32.988, 30.287, 28.260, 28.161, 28.310, 28.257, 28.417, 28.346, 28.377, 28.301,
    28.276, 28.101, 28.675, 28.288, 28.494, 28.615, 28.389, 28.926, 28.616, 28.512, 27.969, 28.178,
    28.515, 28.392, 28.549, 28.412, 31.642, 30.166, 32.618, 32.614, 33.084, 32.530, 31.086, 32.028,
];
