use std::error::Error as StdError;
use std::time::Instant;

use mlx9064x::mlx90640::Mlx90640;
use mlx9064x::test::{i2c_mock::example_mlx90640_at_address, mlx90640_example_data};
use mlx9064x::{MelexisCamera, Mlx90640Driver};

const NUM_ITERATIONS: i32 = 100_000;

fn main() -> Result<(), AnyError> {
    let address: u8 = 0x30;
    let mut mock_bus = example_mlx90640_at_address(address);
    mock_bus.set_data_available(true);
    // Clone the bus, as it's using Rc internally so we can swap the data out as needed.
    let mut camera = Mlx90640Driver::new(mock_bus.clone(), address).unwrap();
    let mut temperatures = [0f32; Mlx90640::NUM_PIXELS];
    let start_time = Instant::now();
    let mut frame_zero: bool = true;
    for _ in 0..NUM_ITERATIONS {
        camera.generate_image_if_ready(&mut temperatures).unwrap();
        frame_zero = !frame_zero;
        if frame_zero {
            mock_bus.update_frame(
                &mlx90640_example_data::FRAME_0_DATA[..],
                &mlx90640_example_data::FRAME_0_STATUS_REGISTER[..],
            )
        } else {
            mock_bus.update_frame(
                &mlx90640_example_data::FRAME_1_DATA[..],
                &mlx90640_example_data::FRAME_1_STATUS_REGISTER[..],
            )
        }
        mock_bus.set_data_available(true);
    }
    let elapsed = start_time.elapsed();
    println!(
        "{} iterations took {} seconds",
        NUM_ITERATIONS,
        elapsed.as_secs_f64()
    );
    Ok(())
}

// It's anyhow::Error, but less functional and less tested.
#[derive(Debug)]
enum AnyError {
    Wrapped(Box<dyn StdError + 'static>),
    String(String),
}

impl std::fmt::Display for AnyError {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match self {
            AnyError::Wrapped(err) => write!(f, "{}", err),
            AnyError::String(s) => write!(f, "{}", s),
        }
    }
}

impl<E> From<E> for AnyError
where
    E: StdError + 'static,
{
    fn from(err: E) -> Self {
        AnyError::Wrapped(Box::new(err))
    }
}
