use std::{
    convert::TryFrom,
    env,
    fs::File,
    io,
    io::Write,
    path::PathBuf,
    process::{Command, Stdio},
};
extern crate tinytemplate;
use mit_commit::{CommitMessage, Trailer};
use mit_commit_message_lints::{
    external::{Git2, Vcs},
    mit::{get_commit_coauthor_configuration, Author},
    relates::{entities::RelateTo, vcs::get_relate_to_configuration},
};
use serde::Serialize;
use tinytemplate::TinyTemplate;

use crate::{
    cli::app,
    errors::MitPrepareCommitMessageError,
    MitPrepareCommitMessageError::MissingCommitFilePath,
};

mod cli;
mod errors;

#[derive(Serialize)]
struct Context {
    value: String,
}

fn main() -> Result<(), errors::MitPrepareCommitMessageError> {
    let matches = app().get_matches();

    let commit_message_path = matches
        .value_of("commit-message-path")
        .map(PathBuf::from)
        .ok_or(MissingCommitFilePath)?;
    let current_dir =
        env::current_dir().map_err(|err| MitPrepareCommitMessageError::new_cwd_io(&err))?;

    let mut git_config = Git2::try_from(current_dir)?;

    if let Some(authors) = get_commit_coauthor_configuration(&mut git_config)? {
        append_coauthors_to_commit_message(commit_message_path.clone(), &authors)?;
    }

    let relates_to_template = matches
        .value_of("relates-to-template")
        .map(String::from)
        .or(get_relates_to_template(&mut git_config)?);

    if let Some(exec) = matches.value_of("relates-to-exec") {
        append_relate_to_trailer_to_commit_message(
            commit_message_path,
            &get_relates_to_from_exec(exec)?,
            relates_to_template,
        )?;
    } else if let Some(relates_to) = get_relate_to_configuration(&mut git_config)? {
        append_relate_to_trailer_to_commit_message(
            commit_message_path,
            &relates_to,
            relates_to_template,
        )?;
    }

    Ok(())
}

fn get_relates_to_template(vcs: &mut Git2) -> Result<Option<String>, MitPrepareCommitMessageError> {
    Ok(vcs.get_str("mit.relate.template")?.map(String::from))
}

fn append_coauthors_to_commit_message(
    commit_message_path: PathBuf,
    authors: &[Author],
) -> Result<(), MitPrepareCommitMessageError> {
    let path = String::from(commit_message_path.to_string_lossy());
    let mut commit_message = CommitMessage::try_from(commit_message_path.clone())?;

    let trailers = authors
        .iter()
        .map(|x| Trailer::new("Co-authored-by", &format!("{} <{}>", x.name(), x.email())))
        .collect::<Vec<_>>();

    for trailer in trailers {
        if !commit_message
            .get_trailers()
            .iter()
            .any(|existing_trailer| &trailer == existing_trailer)
        {
            commit_message = commit_message.add_trailer(trailer);
        }
    }

    File::create(commit_message_path)
        .and_then(|mut file| file.write_all(String::from(commit_message).as_bytes()))
        .map_err(|err| MitPrepareCommitMessageError::new_io(path, &err))
}

fn append_relate_to_trailer_to_commit_message(
    commit_message_path: PathBuf,
    relates: &RelateTo,
    template: Option<String>,
) -> Result<(), MitPrepareCommitMessageError> {
    let path = String::from(commit_message_path.to_string_lossy());
    let commit_message = CommitMessage::try_from(commit_message_path.clone())?;

    let mut tt = TinyTemplate::new();
    let defaulted_template = template.unwrap_or_else(|| "{ value }".to_string());
    tt.add_template("template", &defaulted_template)?;
    let value = tt.render(
        "template",
        &Context {
            value: relates.to(),
        },
    )?;
    let trailer = Trailer::new("Relates-to", &value);
    add_trailer_if_not_existing(commit_message_path, &commit_message, &trailer)
        .map_err(|err| MitPrepareCommitMessageError::new_io(path, &err))?;

    Ok(())
}

fn add_trailer_if_not_existing(
    commit_message_path: PathBuf,
    commit_message: &CommitMessage,
    trailer: &Trailer,
) -> Result<(), io::Error> {
    if commit_message
        .get_trailers()
        .iter()
        .any(|existing_trailer| trailer == existing_trailer)
    {
        Ok(())
    } else {
        File::create(commit_message_path).and_then(|mut file| {
            file.write_all(String::from(commit_message.add_trailer(trailer.clone())).as_bytes())
        })
    }
}

fn get_relates_to_from_exec(command: &str) -> Result<RelateTo, MitPrepareCommitMessageError> {
    let commandline = shell_words::split(command)?;
    Command::new(commandline.first().unwrap_or(&String::from("")))
        .stderr(Stdio::inherit())
        .args(commandline.iter().skip(1).collect::<Vec<_>>())
        .output()
        .map_err(|error| MitPrepareCommitMessageError::new_exec(command.into(), &error))
        .and_then(|x| {
            Ok(RelateTo::new(
                &String::from_utf8(x.stdout).map_err(MitPrepareCommitMessageError::from)?,
            ))
        })
}
