use std::{error, fmt::Display};

use miette::{Diagnostic, LabeledSpan, Result, Severity, SourceCode};
use mit_lint::Problem;
use thiserror::Error;

#[derive(Error, Debug, Diagnostic)]
pub(crate) enum MitCommitMsgError {
    #[error("expected file path name")]
    #[diagnostic()]
    CommitPathMissing,
    #[error("{0}")]
    #[diagnostic()]
    Clipboard(#[source] Box<dyn error::Error + Sync + Send>),
}

#[derive(Error, Debug)]
#[error("multiple lint problems")]
pub(crate) struct AggregateProblem(Vec<Problem>);

impl AggregateProblem {
    pub(crate) fn to(problems: Vec<Problem>) -> Result<()> {
        if problems.len() == 1 {
            return Err(problems.first().cloned().unwrap().into());
        } else if problems.is_empty() {
            Ok(())
        } else {
            Err(AggregateProblem(problems).into())
        }
    }
}

impl Diagnostic for AggregateProblem {
    fn code<'a>(&'a self) -> Option<Box<dyn Display + 'a>> {
        let collection = self
            .0
            .iter()
            .filter_map(|x| (x as &dyn Diagnostic).code().map(|x| x.to_string()))
            .collect::<Vec<String>>();

        if collection.is_empty() {
            None
        } else {
            Some(Box::new(collection.join(" ")))
        }
    }

    fn severity(&self) -> Option<Severity> {
        let collection = self
            .0
            .iter()
            .filter_map(|x| x.severity())
            .collect::<Vec<Severity>>();

        if collection.is_empty() {
            None
        } else if collection.iter().any(|x| x == &Severity::Error) {
            Some(Severity::Error)
        } else if collection.iter().any(|x| x == &Severity::Warning) {
            Some(Severity::Warning)
        } else {
            Some(Severity::Advice)
        }
    }

    fn help<'a>(&'a self) -> Option<Box<dyn Display + 'a>> {
        let collection = self
            .0
            .iter()
            .filter_map(|x| (x as &dyn Diagnostic).help().map(|x| x.to_string()))
            .collect::<Vec<String>>();

        if collection.is_empty() {
            None
        } else {
            Some(Box::new(collection.join("\n\n---\n\n")))
        }
    }

    fn url<'a>(&'a self) -> Option<Box<dyn Display + 'a>> {
        let collection = self
            .0
            .iter()
            .filter_map(|x| (x as &dyn Diagnostic).url().map(|x| x.to_string()))
            .collect::<Vec<String>>();

        if collection.is_empty() {
            None
        } else {
            Some(Box::new(collection.join(" ")))
        }
    }

    fn source_code(&self) -> Option<&dyn SourceCode> {
        self.0.first().and_then(|x| x.source_code())
    }

    fn labels(&self) -> Option<Box<dyn Iterator<Item = LabeledSpan> + '_>> {
        let collection = self
            .0
            .iter()
            .filter_map(|x| x.labels())
            .flatten()
            .collect::<Vec<LabeledSpan>>();

        if collection.is_empty() {
            return None;
        }

        Some(Box::new(collection.into_iter()))
    }

    fn related<'a>(&'a self) -> Option<Box<dyn Iterator<Item = &'a dyn Diagnostic> + 'a>> {
        let collection = self
            .0
            .iter()
            .filter_map(|x| x.related())
            .flatten()
            .collect::<Vec<&dyn Diagnostic>>();

        if collection.is_empty() {
            return None;
        }

        Some(Box::new(collection.into_iter()))
    }
}
