use crate::common::File;

use clap::ArgMatches;
use minipac::handler::Action;
use minipac::handler::Handler;
use minipac::hostname::Hostname;
use minipac::server::Server;
use std::time::Duration;

enum SentState {
	Nothing,
	Meta,
	Data,
}

struct ServerHandler {
	filename: String,
	data: Vec<u8>,
	sent: SentState,
}

impl ServerHandler {
	fn new(filename: String, data: Vec<u8>) -> Self {
		Self {
			filename,
			data,
			sent: SentState::Nothing,
		}
	}
}

impl Handler for ServerHandler {
	fn recv(&mut self, _: &[u8]) -> Action {
		Action::Nothing
	}

	fn process(&mut self) -> Action {
		match &self.sent {
			SentState::Nothing => {
				self.sent = SentState::Meta;

				Action::SendData(
					bincode::serialize(&File::Metadata(
						self.filename.clone(),
						self.data.len() as u32,
					))
					.unwrap(),
				)
			}

			SentState::Meta => {
				self.sent = SentState::Data;

				Action::SendData(bincode::serialize(&File::Data(self.data.clone())).unwrap())
			}

			SentState::Data => Action::Nothing,
		}
	}
}

pub fn serve(matches: &ArgMatches) -> Result<(), String> {
	// Safe to unwrap because it's required
	let hostname = matches.value_of("hostname").unwrap();
	let hostname = Hostname::new(hostname).ok_or_else(|| "Invalid hostname".to_string())?;

	// Safe to unwrap because it's required
	let filename = matches.value_of("file").unwrap();
	let file = std::fs::read(filename).map_err(|e| format!("Could not read file: {}", e))?;

	let baud_rate = matches.value_of("baudrate").unwrap_or("1200");
	let baud_rate = baud_rate
		.parse()
		.map_err(|_| "Invalid baud rate".to_string())?;

	let addr = matches.value_of("kiss").unwrap_or("localhost:8001");
	let mut server = Server::new(
		addr,
		hostname,
		Duration::from_secs(3),
		1000,
		3,
		baud_rate,
		|_| ServerHandler::new(filename.to_string(), file.clone()),
	)
	.map_err(|e| format!("{}", e))?;

	println!("Serving {}. Press Ctrl+C to quit", filename);

	server.serve_forever().map_err(|e| format!("{}", e))?;

	Ok(())
}
