use crate::common::File;

use clap::ArgMatches;
use minipac::client::Client;
use minipac::error::Error;
use minipac::handler::Action;
use minipac::handler::Handler;
use minipac::hostname::Hostname;
use std::io::prelude::*;
use std::time::Duration;

#[derive(Default)]
struct ClientHandler {
	file: Option<std::fs::File>,
}

impl Handler for ClientHandler {
	fn recv(&mut self, data: &[u8]) -> Action {
		let f: Result<File, _> = bincode::deserialize(data);
		match f {
			Ok(f) => match (f, self.file.as_mut()) {
				(File::Metadata(name, len), _) => {
					println!("Receiving {} - {} bytes", name, len);

					let file = match std::fs::File::create(&name) {
						Ok(x) => x,
						Err(e) => {
							eprintln!("Could not open {}: {}", name, e);
							return Action::Disconnect;
						}
					};

					self.file = Some(file);

					Action::Nothing
				}

				(File::Data(data), Some(file)) => {
					match file.write_all(&data) {
						Ok(_) => {
							println!("File saved.");
						}
						Err(e) => {
							eprintln!("Error saving file: {}", e);
						}
					}

					Action::Disconnect
				}

				(_, _) => Action::Nothing,
			},

			Err(_) => {
				eprintln!("Received malformed data. Ignoring.");
				Action::Nothing
			}
		}
	}

	fn process(&mut self) -> Action {
		Action::Nothing
	}
}

pub fn recv(matches: &ArgMatches) -> Result<(), String> {
	// Safe to unwrap because it's required
	let hostname = matches.value_of("hostname").unwrap();
	let hostname = Hostname::new(hostname).ok_or_else(|| "Invalid hostname".to_string())?;

	// Safe to unwrap because it's required
	let serv_hostname = matches.value_of("server").unwrap();
	let serv_hostname =
		Hostname::new(serv_hostname).ok_or_else(|| "Invalid server hostname".to_string())?;

	let baud_rate = matches.value_of("baudrate").unwrap_or("1200");
	let baud_rate = baud_rate
		.parse()
		.map_err(|_| "Invalid baud rate".to_string())?;

	let addr = matches.value_of("kiss").unwrap_or("localhost:8001");
	let mut client = Client::new(
		addr,
		hostname,
		serv_hostname,
		Duration::from_secs(3),
		1000,
		3,
		baud_rate,
	)
	.map_err(|e| format!("{}", e))?;

	let mut handler = ClientHandler::default();
	match client.connect(&mut handler) {
		Ok(()) => {}
		Err(Error::Disconnected(_)) => {
			println!("Disconnected from {}", serv_hostname)
		}

		Err(e) => {
			return Err(format!("{}", e));
		}
	}

	Ok(())
}
