//! Varios X11 keycode to event mappings
//! for keyboard, mouse, basically any long table of keycode->enum belongs here
//!
//! All the data entries in this file came from sokol_app.h

use super::X11Display;
use crate::event::{KeyCode, KeyMods, MouseButton};

impl X11Display {
    pub unsafe fn translate_key(&mut self, scancode: i32) -> KeyCode {
        let mut dummy: libc::c_int = 0;
        let keysyms = (self.libx11.XGetKeyboardMapping)(
            self.display,
            scancode as _,
            1 as libc::c_int,
            &mut dummy,
        );
        assert!(!keysyms.is_null());

        let keysym = *keysyms.offset(0 as libc::c_int as isize);
        (self.libx11.XFree)(keysyms as *mut libc::c_void);
        match keysym {
            65307 => return KeyCode::Escape,
            65289 => return KeyCode::Tab,
            65505 => return KeyCode::LeftShift,
            65506 => return KeyCode::RightShift,
            65507 => return KeyCode::LeftControl,
            65508 => return KeyCode::RightControl,
            65511 | 65513 => return KeyCode::LeftAlt,
            65406 | 65027 | 65512 | 65514 => return KeyCode::RightAlt,
            65515 => return KeyCode::LeftSuper,
            65516 => return KeyCode::RightSuper,
            65383 => return KeyCode::Menu,
            65407 => return KeyCode::NumLock,
            65509 => return KeyCode::CapsLock,
            65377 => return KeyCode::PrintScreen,
            65300 => return KeyCode::ScrollLock,
            65299 => return KeyCode::Pause,
            65535 => return KeyCode::Delete,
            65288 => return KeyCode::Backspace,
            65293 => return KeyCode::Enter,
            65360 => return KeyCode::Home,
            65367 => return KeyCode::End,
            65365 => return KeyCode::PageUp,
            65366 => return KeyCode::PageDown,
            65379 => return KeyCode::Insert,
            65361 => return KeyCode::Left,
            65363 => return KeyCode::Right,
            65364 => return KeyCode::Down,
            65362 => return KeyCode::Up,
            65470 => return KeyCode::F1,
            65471 => return KeyCode::F2,
            65472 => return KeyCode::F3,
            65473 => return KeyCode::F4,
            65474 => return KeyCode::F5,
            65475 => return KeyCode::F6,
            65476 => return KeyCode::F7,
            65477 => return KeyCode::F8,
            65478 => return KeyCode::F9,
            65479 => return KeyCode::F10,
            65480 => return KeyCode::F11,
            65481 => return KeyCode::F12,
            65482 => return KeyCode::F13,
            65483 => return KeyCode::F14,
            65484 => return KeyCode::F15,
            65485 => return KeyCode::F16,
            65486 => return KeyCode::F17,
            65487 => return KeyCode::F18,
            65488 => return KeyCode::F19,
            65489 => return KeyCode::F20,
            65490 => return KeyCode::F21,
            65491 => return KeyCode::F22,
            65492 => return KeyCode::F23,
            65493 => return KeyCode::F24,
            65494 => return KeyCode::F25,
            65455 => return KeyCode::KpDivide,
            65450 => return KeyCode::KpMultiply,
            65453 => return KeyCode::KpSubtract,
            65451 => return KeyCode::KpAdd,
            65438 => return KeyCode::Kp0,
            65436 => return KeyCode::Kp1,
            65433 => return KeyCode::Kp2,
            65435 => return KeyCode::Kp3,
            65430 => return KeyCode::Kp4,
            65437 => return KeyCode::Kp5,
            65432 => return KeyCode::Kp6,
            65429 => return KeyCode::Kp7,
            65431 => return KeyCode::Kp8,
            65434 => return KeyCode::Kp9,
            65439 => return KeyCode::KpDecimal,
            65469 => return KeyCode::KpEqual,
            65421 => return KeyCode::KpEnter,
            97 => return KeyCode::A,
            98 => return KeyCode::B,
            99 => return KeyCode::C,
            100 => return KeyCode::D,
            101 => return KeyCode::E,
            102 => return KeyCode::F,
            103 => return KeyCode::G,
            104 => return KeyCode::H,
            105 => return KeyCode::I,
            106 => return KeyCode::J,
            107 => return KeyCode::K,
            108 => return KeyCode::L,
            109 => return KeyCode::M,
            110 => return KeyCode::N,
            111 => return KeyCode::O,
            112 => return KeyCode::P,
            113 => return KeyCode::Q,
            114 => return KeyCode::R,
            115 => return KeyCode::S,
            116 => return KeyCode::T,
            117 => return KeyCode::U,
            118 => return KeyCode::V,
            119 => return KeyCode::W,
            120 => return KeyCode::X,
            121 => return KeyCode::Y,
            122 => return KeyCode::Z,
            49 => return KeyCode::Key1,
            50 => return KeyCode::Key2,
            51 => return KeyCode::Key3,
            52 => return KeyCode::Key4,
            53 => return KeyCode::Key5,
            54 => return KeyCode::Key6,
            55 => return KeyCode::Key7,
            56 => return KeyCode::Key8,
            57 => return KeyCode::Key9,
            48 => return KeyCode::Key0,
            32 => return KeyCode::Space,
            45 => return KeyCode::Minus,
            61 => return KeyCode::Equal,
            91 => return KeyCode::LeftBracket,
            93 => return KeyCode::RightBracket,
            92 => return KeyCode::Backslash,
            59 => return KeyCode::Semicolon,
            39 => return KeyCode::Apostrophe,
            96 => return KeyCode::GraveAccent,
            44 => return KeyCode::Comma,
            46 => return KeyCode::Period,
            47 => return KeyCode::Slash,
            60 => return KeyCode::World1,
            _ => return KeyCode::Unknown,
        };
    }

    pub unsafe fn translate_mod(&self, x11_mods: i32) -> KeyMods {
        let mut mods = KeyMods::default();
        if x11_mods & super::libx11::ShiftMask != 0 {
            mods.shift = true;
        }
        if x11_mods & super::libx11::ControlMask != 0 {
            mods.ctrl = true;
        }
        if x11_mods & super::libx11::Mod1Mask != 0 {
            mods.alt = true;
        }
        if x11_mods & super::libx11::Mod4Mask != 0 {
            mods.logo = true;
        }
        return mods;
    }

    pub unsafe fn translate_mouse_button(&self, button: i32) -> MouseButton {
        match button {
            1 => return MouseButton::Left,
            2 => return MouseButton::Middle,
            3 => return MouseButton::Right,
            _ => return MouseButton::Unknown,
        };
    }
}

#[derive(Copy, Clone)]
#[repr(C)]
pub struct CodePair {
    pub keysym: u16,
    pub ucs: u16,
}
impl CodePair {
    const fn new(keysym: u16, ucs: u16) -> CodePair {
        CodePair { keysym, ucs }
    }
}

const KEYSYMTAB: [CodePair; 828] = [
    CodePair::new(0x01a1, 0x0104),
    CodePair::new(0x01a2, 0x02d8),
    CodePair::new(0x01a3, 0x0141),
    CodePair::new(0x01a5, 0x013d),
    CodePair::new(0x01a6, 0x015a),
    CodePair::new(0x01a9, 0x0160),
    CodePair::new(0x01aa, 0x015e),
    CodePair::new(0x01ab, 0x0164),
    CodePair::new(0x01ac, 0x0179),
    CodePair::new(0x01ae, 0x017d),
    CodePair::new(0x01af, 0x017b),
    CodePair::new(0x01b1, 0x0105),
    CodePair::new(0x01b2, 0x02db),
    CodePair::new(0x01b3, 0x0142),
    CodePair::new(0x01b5, 0x013e),
    CodePair::new(0x01b6, 0x015b),
    CodePair::new(0x01b7, 0x02c7),
    CodePair::new(0x01b9, 0x0161),
    CodePair::new(0x01ba, 0x015f),
    CodePair::new(0x01bb, 0x0165),
    CodePair::new(0x01bc, 0x017a),
    CodePair::new(0x01bd, 0x02dd),
    CodePair::new(0x01be, 0x017e),
    CodePair::new(0x01bf, 0x017c),
    CodePair::new(0x01c0, 0x0154),
    CodePair::new(0x01c3, 0x0102),
    CodePair::new(0x01c5, 0x0139),
    CodePair::new(0x01c6, 0x0106),
    CodePair::new(0x01c8, 0x010c),
    CodePair::new(0x01ca, 0x0118),
    CodePair::new(0x01cc, 0x011a),
    CodePair::new(0x01cf, 0x010e),
    CodePair::new(0x01d0, 0x0110),
    CodePair::new(0x01d1, 0x0143),
    CodePair::new(0x01d2, 0x0147),
    CodePair::new(0x01d5, 0x0150),
    CodePair::new(0x01d8, 0x0158),
    CodePair::new(0x01d9, 0x016e),
    CodePair::new(0x01db, 0x0170),
    CodePair::new(0x01de, 0x0162),
    CodePair::new(0x01e0, 0x0155),
    CodePair::new(0x01e3, 0x0103),
    CodePair::new(0x01e5, 0x013a),
    CodePair::new(0x01e6, 0x0107),
    CodePair::new(0x01e8, 0x010d),
    CodePair::new(0x01ea, 0x0119),
    CodePair::new(0x01ec, 0x011b),
    CodePair::new(0x01ef, 0x010f),
    CodePair::new(0x01f0, 0x0111),
    CodePair::new(0x01f1, 0x0144),
    CodePair::new(0x01f2, 0x0148),
    CodePair::new(0x01f5, 0x0151),
    CodePair::new(0x01f8, 0x0159),
    CodePair::new(0x01f9, 0x016f),
    CodePair::new(0x01fb, 0x0171),
    CodePair::new(0x01fe, 0x0163),
    CodePair::new(0x01ff, 0x02d9),
    CodePair::new(0x02a1, 0x0126),
    CodePair::new(0x02a6, 0x0124),
    CodePair::new(0x02a9, 0x0130),
    CodePair::new(0x02ab, 0x011e),
    CodePair::new(0x02ac, 0x0134),
    CodePair::new(0x02b1, 0x0127),
    CodePair::new(0x02b6, 0x0125),
    CodePair::new(0x02b9, 0x0131),
    CodePair::new(0x02bb, 0x011f),
    CodePair::new(0x02bc, 0x0135),
    CodePair::new(0x02c5, 0x010a),
    CodePair::new(0x02c6, 0x0108),
    CodePair::new(0x02d5, 0x0120),
    CodePair::new(0x02d8, 0x011c),
    CodePair::new(0x02dd, 0x016c),
    CodePair::new(0x02de, 0x015c),
    CodePair::new(0x02e5, 0x010b),
    CodePair::new(0x02e6, 0x0109),
    CodePair::new(0x02f5, 0x0121),
    CodePair::new(0x02f8, 0x011d),
    CodePair::new(0x02fd, 0x016d),
    CodePair::new(0x02fe, 0x015d),
    CodePair::new(0x03a2, 0x0138),
    CodePair::new(0x03a3, 0x0156),
    CodePair::new(0x03a5, 0x0128),
    CodePair::new(0x03a6, 0x013b),
    CodePair::new(0x03aa, 0x0112),
    CodePair::new(0x03ab, 0x0122),
    CodePair::new(0x03ac, 0x0166),
    CodePair::new(0x03b3, 0x0157),
    CodePair::new(0x03b5, 0x0129),
    CodePair::new(0x03b6, 0x013c),
    CodePair::new(0x03ba, 0x0113),
    CodePair::new(0x03bb, 0x0123),
    CodePair::new(0x03bc, 0x0167),
    CodePair::new(0x03bd, 0x014a),
    CodePair::new(0x03bf, 0x014b),
    CodePair::new(0x03c0, 0x0100),
    CodePair::new(0x03c7, 0x012e),
    CodePair::new(0x03cc, 0x0116),
    CodePair::new(0x03cf, 0x012a),
    CodePair::new(0x03d1, 0x0145),
    CodePair::new(0x03d2, 0x014c),
    CodePair::new(0x03d3, 0x0136),
    CodePair::new(0x03d9, 0x0172),
    CodePair::new(0x03dd, 0x0168),
    CodePair::new(0x03de, 0x016a),
    CodePair::new(0x03e0, 0x0101),
    CodePair::new(0x03e7, 0x012f),
    CodePair::new(0x03ec, 0x0117),
    CodePair::new(0x03ef, 0x012b),
    CodePair::new(0x03f1, 0x0146),
    CodePair::new(0x03f2, 0x014d),
    CodePair::new(0x03f3, 0x0137),
    CodePair::new(0x03f9, 0x0173),
    CodePair::new(0x03fd, 0x0169),
    CodePair::new(0x03fe, 0x016b),
    CodePair::new(0x047e, 0x203e),
    CodePair::new(0x04a1, 0x3002),
    CodePair::new(0x04a2, 0x300c),
    CodePair::new(0x04a3, 0x300d),
    CodePair::new(0x04a4, 0x3001),
    CodePair::new(0x04a5, 0x30fb),
    CodePair::new(0x04a6, 0x30f2),
    CodePair::new(0x04a7, 0x30a1),
    CodePair::new(0x04a8, 0x30a3),
    CodePair::new(0x04a9, 0x30a5),
    CodePair::new(0x04aa, 0x30a7),
    CodePair::new(0x04ab, 0x30a9),
    CodePair::new(0x04ac, 0x30e3),
    CodePair::new(0x04ad, 0x30e5),
    CodePair::new(0x04ae, 0x30e7),
    CodePair::new(0x04af, 0x30c3),
    CodePair::new(0x04b0, 0x30fc),
    CodePair::new(0x04b1, 0x30a2),
    CodePair::new(0x04b2, 0x30a4),
    CodePair::new(0x04b3, 0x30a6),
    CodePair::new(0x04b4, 0x30a8),
    CodePair::new(0x04b5, 0x30aa),
    CodePair::new(0x04b6, 0x30ab),
    CodePair::new(0x04b7, 0x30ad),
    CodePair::new(0x04b8, 0x30af),
    CodePair::new(0x04b9, 0x30b1),
    CodePair::new(0x04ba, 0x30b3),
    CodePair::new(0x04bb, 0x30b5),
    CodePair::new(0x04bc, 0x30b7),
    CodePair::new(0x04bd, 0x30b9),
    CodePair::new(0x04be, 0x30bb),
    CodePair::new(0x04bf, 0x30bd),
    CodePair::new(0x04c0, 0x30bf),
    CodePair::new(0x04c1, 0x30c1),
    CodePair::new(0x04c2, 0x30c4),
    CodePair::new(0x04c3, 0x30c6),
    CodePair::new(0x04c4, 0x30c8),
    CodePair::new(0x04c5, 0x30ca),
    CodePair::new(0x04c6, 0x30cb),
    CodePair::new(0x04c7, 0x30cc),
    CodePair::new(0x04c8, 0x30cd),
    CodePair::new(0x04c9, 0x30ce),
    CodePair::new(0x04ca, 0x30cf),
    CodePair::new(0x04cb, 0x30d2),
    CodePair::new(0x04cc, 0x30d5),
    CodePair::new(0x04cd, 0x30d8),
    CodePair::new(0x04ce, 0x30db),
    CodePair::new(0x04cf, 0x30de),
    CodePair::new(0x04d0, 0x30df),
    CodePair::new(0x04d1, 0x30e0),
    CodePair::new(0x04d2, 0x30e1),
    CodePair::new(0x04d3, 0x30e2),
    CodePair::new(0x04d4, 0x30e4),
    CodePair::new(0x04d5, 0x30e6),
    CodePair::new(0x04d6, 0x30e8),
    CodePair::new(0x04d7, 0x30e9),
    CodePair::new(0x04d8, 0x30ea),
    CodePair::new(0x04d9, 0x30eb),
    CodePair::new(0x04da, 0x30ec),
    CodePair::new(0x04db, 0x30ed),
    CodePair::new(0x04dc, 0x30ef),
    CodePair::new(0x04dd, 0x30f3),
    CodePair::new(0x04de, 0x309b),
    CodePair::new(0x04df, 0x309c),
    CodePair::new(0x05ac, 0x060c),
    CodePair::new(0x05bb, 0x061b),
    CodePair::new(0x05bf, 0x061f),
    CodePair::new(0x05c1, 0x0621),
    CodePair::new(0x05c2, 0x0622),
    CodePair::new(0x05c3, 0x0623),
    CodePair::new(0x05c4, 0x0624),
    CodePair::new(0x05c5, 0x0625),
    CodePair::new(0x05c6, 0x0626),
    CodePair::new(0x05c7, 0x0627),
    CodePair::new(0x05c8, 0x0628),
    CodePair::new(0x05c9, 0x0629),
    CodePair::new(0x05ca, 0x062a),
    CodePair::new(0x05cb, 0x062b),
    CodePair::new(0x05cc, 0x062c),
    CodePair::new(0x05cd, 0x062d),
    CodePair::new(0x05ce, 0x062e),
    CodePair::new(0x05cf, 0x062f),
    CodePair::new(0x05d0, 0x0630),
    CodePair::new(0x05d1, 0x0631),
    CodePair::new(0x05d2, 0x0632),
    CodePair::new(0x05d3, 0x0633),
    CodePair::new(0x05d4, 0x0634),
    CodePair::new(0x05d5, 0x0635),
    CodePair::new(0x05d6, 0x0636),
    CodePair::new(0x05d7, 0x0637),
    CodePair::new(0x05d8, 0x0638),
    CodePair::new(0x05d9, 0x0639),
    CodePair::new(0x05da, 0x063a),
    CodePair::new(0x05e0, 0x0640),
    CodePair::new(0x05e1, 0x0641),
    CodePair::new(0x05e2, 0x0642),
    CodePair::new(0x05e3, 0x0643),
    CodePair::new(0x05e4, 0x0644),
    CodePair::new(0x05e5, 0x0645),
    CodePair::new(0x05e6, 0x0646),
    CodePair::new(0x05e7, 0x0647),
    CodePair::new(0x05e8, 0x0648),
    CodePair::new(0x05e9, 0x0649),
    CodePair::new(0x05ea, 0x064a),
    CodePair::new(0x05eb, 0x064b),
    CodePair::new(0x05ec, 0x064c),
    CodePair::new(0x05ed, 0x064d),
    CodePair::new(0x05ee, 0x064e),
    CodePair::new(0x05ef, 0x064f),
    CodePair::new(0x05f0, 0x0650),
    CodePair::new(0x05f1, 0x0651),
    CodePair::new(0x05f2, 0x0652),
    CodePair::new(0x06a1, 0x0452),
    CodePair::new(0x06a2, 0x0453),
    CodePair::new(0x06a3, 0x0451),
    CodePair::new(0x06a4, 0x0454),
    CodePair::new(0x06a5, 0x0455),
    CodePair::new(0x06a6, 0x0456),
    CodePair::new(0x06a7, 0x0457),
    CodePair::new(0x06a8, 0x0458),
    CodePair::new(0x06a9, 0x0459),
    CodePair::new(0x06aa, 0x045a),
    CodePair::new(0x06ab, 0x045b),
    CodePair::new(0x06ac, 0x045c),
    CodePair::new(0x06ae, 0x045e),
    CodePair::new(0x06af, 0x045f),
    CodePair::new(0x06b0, 0x2116),
    CodePair::new(0x06b1, 0x0402),
    CodePair::new(0x06b2, 0x0403),
    CodePair::new(0x06b3, 0x0401),
    CodePair::new(0x06b4, 0x0404),
    CodePair::new(0x06b5, 0x0405),
    CodePair::new(0x06b6, 0x0406),
    CodePair::new(0x06b7, 0x0407),
    CodePair::new(0x06b8, 0x0408),
    CodePair::new(0x06b9, 0x0409),
    CodePair::new(0x06ba, 0x040a),
    CodePair::new(0x06bb, 0x040b),
    CodePair::new(0x06bc, 0x040c),
    CodePair::new(0x06be, 0x040e),
    CodePair::new(0x06bf, 0x040f),
    CodePair::new(0x06c0, 0x044e),
    CodePair::new(0x06c1, 0x0430),
    CodePair::new(0x06c2, 0x0431),
    CodePair::new(0x06c3, 0x0446),
    CodePair::new(0x06c4, 0x0434),
    CodePair::new(0x06c5, 0x0435),
    CodePair::new(0x06c6, 0x0444),
    CodePair::new(0x06c7, 0x0433),
    CodePair::new(0x06c8, 0x0445),
    CodePair::new(0x06c9, 0x0438),
    CodePair::new(0x06ca, 0x0439),
    CodePair::new(0x06cb, 0x043a),
    CodePair::new(0x06cc, 0x043b),
    CodePair::new(0x06cd, 0x043c),
    CodePair::new(0x06ce, 0x043d),
    CodePair::new(0x06cf, 0x043e),
    CodePair::new(0x06d0, 0x043f),
    CodePair::new(0x06d1, 0x044f),
    CodePair::new(0x06d2, 0x0440),
    CodePair::new(0x06d3, 0x0441),
    CodePair::new(0x06d4, 0x0442),
    CodePair::new(0x06d5, 0x0443),
    CodePair::new(0x06d6, 0x0436),
    CodePair::new(0x06d7, 0x0432),
    CodePair::new(0x06d8, 0x044c),
    CodePair::new(0x06d9, 0x044b),
    CodePair::new(0x06da, 0x0437),
    CodePair::new(0x06db, 0x0448),
    CodePair::new(0x06dc, 0x044d),
    CodePair::new(0x06dd, 0x0449),
    CodePair::new(0x06de, 0x0447),
    CodePair::new(0x06df, 0x044a),
    CodePair::new(0x06e0, 0x042e),
    CodePair::new(0x06e1, 0x0410),
    CodePair::new(0x06e2, 0x0411),
    CodePair::new(0x06e3, 0x0426),
    CodePair::new(0x06e4, 0x0414),
    CodePair::new(0x06e5, 0x0415),
    CodePair::new(0x06e6, 0x0424),
    CodePair::new(0x06e7, 0x0413),
    CodePair::new(0x06e8, 0x0425),
    CodePair::new(0x06e9, 0x0418),
    CodePair::new(0x06ea, 0x0419),
    CodePair::new(0x06eb, 0x041a),
    CodePair::new(0x06ec, 0x041b),
    CodePair::new(0x06ed, 0x041c),
    CodePair::new(0x06ee, 0x041d),
    CodePair::new(0x06ef, 0x041e),
    CodePair::new(0x06f0, 0x041f),
    CodePair::new(0x06f1, 0x042f),
    CodePair::new(0x06f2, 0x0420),
    CodePair::new(0x06f3, 0x0421),
    CodePair::new(0x06f4, 0x0422),
    CodePair::new(0x06f5, 0x0423),
    CodePair::new(0x06f6, 0x0416),
    CodePair::new(0x06f7, 0x0412),
    CodePair::new(0x06f8, 0x042c),
    CodePair::new(0x06f9, 0x042b),
    CodePair::new(0x06fa, 0x0417),
    CodePair::new(0x06fb, 0x0428),
    CodePair::new(0x06fc, 0x042d),
    CodePair::new(0x06fd, 0x0429),
    CodePair::new(0x06fe, 0x0427),
    CodePair::new(0x06ff, 0x042a),
    CodePair::new(0x07a1, 0x0386),
    CodePair::new(0x07a2, 0x0388),
    CodePair::new(0x07a3, 0x0389),
    CodePair::new(0x07a4, 0x038a),
    CodePair::new(0x07a5, 0x03aa),
    CodePair::new(0x07a7, 0x038c),
    CodePair::new(0x07a8, 0x038e),
    CodePair::new(0x07a9, 0x03ab),
    CodePair::new(0x07ab, 0x038f),
    CodePair::new(0x07ae, 0x0385),
    CodePair::new(0x07af, 0x2015),
    CodePair::new(0x07b1, 0x03ac),
    CodePair::new(0x07b2, 0x03ad),
    CodePair::new(0x07b3, 0x03ae),
    CodePair::new(0x07b4, 0x03af),
    CodePair::new(0x07b5, 0x03ca),
    CodePair::new(0x07b6, 0x0390),
    CodePair::new(0x07b7, 0x03cc),
    CodePair::new(0x07b8, 0x03cd),
    CodePair::new(0x07b9, 0x03cb),
    CodePair::new(0x07ba, 0x03b0),
    CodePair::new(0x07bb, 0x03ce),
    CodePair::new(0x07c1, 0x0391),
    CodePair::new(0x07c2, 0x0392),
    CodePair::new(0x07c3, 0x0393),
    CodePair::new(0x07c4, 0x0394),
    CodePair::new(0x07c5, 0x0395),
    CodePair::new(0x07c6, 0x0396),
    CodePair::new(0x07c7, 0x0397),
    CodePair::new(0x07c8, 0x0398),
    CodePair::new(0x07c9, 0x0399),
    CodePair::new(0x07ca, 0x039a),
    CodePair::new(0x07cb, 0x039b),
    CodePair::new(0x07cc, 0x039c),
    CodePair::new(0x07cd, 0x039d),
    CodePair::new(0x07ce, 0x039e),
    CodePair::new(0x07cf, 0x039f),
    CodePair::new(0x07d0, 0x03a0),
    CodePair::new(0x07d1, 0x03a1),
    CodePair::new(0x07d2, 0x03a3),
    CodePair::new(0x07d4, 0x03a4),
    CodePair::new(0x07d5, 0x03a5),
    CodePair::new(0x07d6, 0x03a6),
    CodePair::new(0x07d7, 0x03a7),
    CodePair::new(0x07d8, 0x03a8),
    CodePair::new(0x07d9, 0x03a9),
    CodePair::new(0x07e1, 0x03b1),
    CodePair::new(0x07e2, 0x03b2),
    CodePair::new(0x07e3, 0x03b3),
    CodePair::new(0x07e4, 0x03b4),
    CodePair::new(0x07e5, 0x03b5),
    CodePair::new(0x07e6, 0x03b6),
    CodePair::new(0x07e7, 0x03b7),
    CodePair::new(0x07e8, 0x03b8),
    CodePair::new(0x07e9, 0x03b9),
    CodePair::new(0x07ea, 0x03ba),
    CodePair::new(0x07eb, 0x03bb),
    CodePair::new(0x07ec, 0x03bc),
    CodePair::new(0x07ed, 0x03bd),
    CodePair::new(0x07ee, 0x03be),
    CodePair::new(0x07ef, 0x03bf),
    CodePair::new(0x07f0, 0x03c0),
    CodePair::new(0x07f1, 0x03c1),
    CodePair::new(0x07f2, 0x03c3),
    CodePair::new(0x07f3, 0x03c2),
    CodePair::new(0x07f4, 0x03c4),
    CodePair::new(0x07f5, 0x03c5),
    CodePair::new(0x07f6, 0x03c6),
    CodePair::new(0x07f7, 0x03c7),
    CodePair::new(0x07f8, 0x03c8),
    CodePair::new(0x07f9, 0x03c9),
    CodePair::new(0x08a1, 0x23b7),
    CodePair::new(0x08a2, 0x250c),
    CodePair::new(0x08a3, 0x2500),
    CodePair::new(0x08a4, 0x2320),
    CodePair::new(0x08a5, 0x2321),
    CodePair::new(0x08a6, 0x2502),
    CodePair::new(0x08a7, 0x23a1),
    CodePair::new(0x08a8, 0x23a3),
    CodePair::new(0x08a9, 0x23a4),
    CodePair::new(0x08aa, 0x23a6),
    CodePair::new(0x08ab, 0x239b),
    CodePair::new(0x08ac, 0x239d),
    CodePair::new(0x08ad, 0x239e),
    CodePair::new(0x08ae, 0x23a0),
    CodePair::new(0x08af, 0x23a8),
    CodePair::new(0x08b0, 0x23ac),
    CodePair::new(0x08bc, 0x2264),
    CodePair::new(0x08bd, 0x2260),
    CodePair::new(0x08be, 0x2265),
    CodePair::new(0x08bf, 0x222b),
    CodePair::new(0x08c0, 0x2234),
    CodePair::new(0x08c1, 0x221d),
    CodePair::new(0x08c2, 0x221e),
    CodePair::new(0x08c5, 0x2207),
    CodePair::new(0x08c8, 0x223c),
    CodePair::new(0x08c9, 0x2243),
    CodePair::new(0x08cd, 0x21d4),
    CodePair::new(0x08ce, 0x21d2),
    CodePair::new(0x08cf, 0x2261),
    CodePair::new(0x08d6, 0x221a),
    CodePair::new(0x08da, 0x2282),
    CodePair::new(0x08db, 0x2283),
    CodePair::new(0x08dc, 0x2229),
    CodePair::new(0x08dd, 0x222a),
    CodePair::new(0x08de, 0x2227),
    CodePair::new(0x08df, 0x2228),
    CodePair::new(0x08ef, 0x2202),
    CodePair::new(0x08f6, 0x0192),
    CodePair::new(0x08fb, 0x2190),
    CodePair::new(0x08fc, 0x2191),
    CodePair::new(0x08fd, 0x2192),
    CodePair::new(0x08fe, 0x2193),
    CodePair::new(0x09e0, 0x25c6),
    CodePair::new(0x09e1, 0x2592),
    CodePair::new(0x09e2, 0x2409),
    CodePair::new(0x09e3, 0x240c),
    CodePair::new(0x09e4, 0x240d),
    CodePair::new(0x09e5, 0x240a),
    CodePair::new(0x09e8, 0x2424),
    CodePair::new(0x09e9, 0x240b),
    CodePair::new(0x09ea, 0x2518),
    CodePair::new(0x09eb, 0x2510),
    CodePair::new(0x09ec, 0x250c),
    CodePair::new(0x09ed, 0x2514),
    CodePair::new(0x09ee, 0x253c),
    CodePair::new(0x09ef, 0x23ba),
    CodePair::new(0x09f0, 0x23bb),
    CodePair::new(0x09f1, 0x2500),
    CodePair::new(0x09f2, 0x23bc),
    CodePair::new(0x09f3, 0x23bd),
    CodePair::new(0x09f4, 0x251c),
    CodePair::new(0x09f5, 0x2524),
    CodePair::new(0x09f6, 0x2534),
    CodePair::new(0x09f7, 0x252c),
    CodePair::new(0x09f8, 0x2502),
    CodePair::new(0x0aa1, 0x2003),
    CodePair::new(0x0aa2, 0x2002),
    CodePair::new(0x0aa3, 0x2004),
    CodePair::new(0x0aa4, 0x2005),
    CodePair::new(0x0aa5, 0x2007),
    CodePair::new(0x0aa6, 0x2008),
    CodePair::new(0x0aa7, 0x2009),
    CodePair::new(0x0aa8, 0x200a),
    CodePair::new(0x0aa9, 0x2014),
    CodePair::new(0x0aaa, 0x2013),
    CodePair::new(0x0aae, 0x2026),
    CodePair::new(0x0aaf, 0x2025),
    CodePair::new(0x0ab0, 0x2153),
    CodePair::new(0x0ab1, 0x2154),
    CodePair::new(0x0ab2, 0x2155),
    CodePair::new(0x0ab3, 0x2156),
    CodePair::new(0x0ab4, 0x2157),
    CodePair::new(0x0ab5, 0x2158),
    CodePair::new(0x0ab6, 0x2159),
    CodePair::new(0x0ab7, 0x215a),
    CodePair::new(0x0ab8, 0x2105),
    CodePair::new(0x0abb, 0x2012),
    CodePair::new(0x0abc, 0x2329),
    CodePair::new(0x0abe, 0x232a),
    CodePair::new(0x0ac3, 0x215b),
    CodePair::new(0x0ac4, 0x215c),
    CodePair::new(0x0ac5, 0x215d),
    CodePair::new(0x0ac6, 0x215e),
    CodePair::new(0x0ac9, 0x2122),
    CodePair::new(0x0aca, 0x2613),
    CodePair::new(0x0acc, 0x25c1),
    CodePair::new(0x0acd, 0x25b7),
    CodePair::new(0x0ace, 0x25cb),
    CodePair::new(0x0acf, 0x25af),
    CodePair::new(0x0ad0, 0x2018),
    CodePair::new(0x0ad1, 0x2019),
    CodePair::new(0x0ad2, 0x201c),
    CodePair::new(0x0ad3, 0x201d),
    CodePair::new(0x0ad4, 0x211e),
    CodePair::new(0x0ad6, 0x2032),
    CodePair::new(0x0ad7, 0x2033),
    CodePair::new(0x0ad9, 0x271d),
    CodePair::new(0x0adb, 0x25ac),
    CodePair::new(0x0adc, 0x25c0),
    CodePair::new(0x0add, 0x25b6),
    CodePair::new(0x0ade, 0x25cf),
    CodePair::new(0x0adf, 0x25ae),
    CodePair::new(0x0ae0, 0x25e6),
    CodePair::new(0x0ae1, 0x25ab),
    CodePair::new(0x0ae2, 0x25ad),
    CodePair::new(0x0ae3, 0x25b3),
    CodePair::new(0x0ae4, 0x25bd),
    CodePair::new(0x0ae5, 0x2606),
    CodePair::new(0x0ae6, 0x2022),
    CodePair::new(0x0ae7, 0x25aa),
    CodePair::new(0x0ae8, 0x25b2),
    CodePair::new(0x0ae9, 0x25bc),
    CodePair::new(0x0aea, 0x261c),
    CodePair::new(0x0aeb, 0x261e),
    CodePair::new(0x0aec, 0x2663),
    CodePair::new(0x0aed, 0x2666),
    CodePair::new(0x0aee, 0x2665),
    CodePair::new(0x0af0, 0x2720),
    CodePair::new(0x0af1, 0x2020),
    CodePair::new(0x0af2, 0x2021),
    CodePair::new(0x0af3, 0x2713),
    CodePair::new(0x0af4, 0x2717),
    CodePair::new(0x0af5, 0x266f),
    CodePair::new(0x0af6, 0x266d),
    CodePair::new(0x0af7, 0x2642),
    CodePair::new(0x0af8, 0x2640),
    CodePair::new(0x0af9, 0x260e),
    CodePair::new(0x0afa, 0x2315),
    CodePair::new(0x0afb, 0x2117),
    CodePair::new(0x0afc, 0x2038),
    CodePair::new(0x0afd, 0x201a),
    CodePair::new(0x0afe, 0x201e),
    CodePair::new(0x0ba3, 0x003c),
    CodePair::new(0x0ba6, 0x003e),
    CodePair::new(0x0ba8, 0x2228),
    CodePair::new(0x0ba9, 0x2227),
    CodePair::new(0x0bc0, 0x00af),
    CodePair::new(0x0bc2, 0x22a5),
    CodePair::new(0x0bc3, 0x2229),
    CodePair::new(0x0bc4, 0x230a),
    CodePair::new(0x0bc6, 0x005f),
    CodePair::new(0x0bca, 0x2218),
    CodePair::new(0x0bcc, 0x2395),
    CodePair::new(0x0bce, 0x22a4),
    CodePair::new(0x0bcf, 0x25cb),
    CodePair::new(0x0bd3, 0x2308),
    CodePair::new(0x0bd6, 0x222a),
    CodePair::new(0x0bd8, 0x2283),
    CodePair::new(0x0bda, 0x2282),
    CodePair::new(0x0bdc, 0x22a2),
    CodePair::new(0x0bfc, 0x22a3),
    CodePair::new(0x0cdf, 0x2017),
    CodePair::new(0x0ce0, 0x05d0),
    CodePair::new(0x0ce1, 0x05d1),
    CodePair::new(0x0ce2, 0x05d2),
    CodePair::new(0x0ce3, 0x05d3),
    CodePair::new(0x0ce4, 0x05d4),
    CodePair::new(0x0ce5, 0x05d5),
    CodePair::new(0x0ce6, 0x05d6),
    CodePair::new(0x0ce7, 0x05d7),
    CodePair::new(0x0ce8, 0x05d8),
    CodePair::new(0x0ce9, 0x05d9),
    CodePair::new(0x0cea, 0x05da),
    CodePair::new(0x0ceb, 0x05db),
    CodePair::new(0x0cec, 0x05dc),
    CodePair::new(0x0ced, 0x05dd),
    CodePair::new(0x0cee, 0x05de),
    CodePair::new(0x0cef, 0x05df),
    CodePair::new(0x0cf0, 0x05e0),
    CodePair::new(0x0cf1, 0x05e1),
    CodePair::new(0x0cf2, 0x05e2),
    CodePair::new(0x0cf3, 0x05e3),
    CodePair::new(0x0cf4, 0x05e4),
    CodePair::new(0x0cf5, 0x05e5),
    CodePair::new(0x0cf6, 0x05e6),
    CodePair::new(0x0cf7, 0x05e7),
    CodePair::new(0x0cf8, 0x05e8),
    CodePair::new(0x0cf9, 0x05e9),
    CodePair::new(0x0cfa, 0x05ea),
    CodePair::new(0x0da1, 0x0e01),
    CodePair::new(0x0da2, 0x0e02),
    CodePair::new(0x0da3, 0x0e03),
    CodePair::new(0x0da4, 0x0e04),
    CodePair::new(0x0da5, 0x0e05),
    CodePair::new(0x0da6, 0x0e06),
    CodePair::new(0x0da7, 0x0e07),
    CodePair::new(0x0da8, 0x0e08),
    CodePair::new(0x0da9, 0x0e09),
    CodePair::new(0x0daa, 0x0e0a),
    CodePair::new(0x0dab, 0x0e0b),
    CodePair::new(0x0dac, 0x0e0c),
    CodePair::new(0x0dad, 0x0e0d),
    CodePair::new(0x0dae, 0x0e0e),
    CodePair::new(0x0daf, 0x0e0f),
    CodePair::new(0x0db0, 0x0e10),
    CodePair::new(0x0db1, 0x0e11),
    CodePair::new(0x0db2, 0x0e12),
    CodePair::new(0x0db3, 0x0e13),
    CodePair::new(0x0db4, 0x0e14),
    CodePair::new(0x0db5, 0x0e15),
    CodePair::new(0x0db6, 0x0e16),
    CodePair::new(0x0db7, 0x0e17),
    CodePair::new(0x0db8, 0x0e18),
    CodePair::new(0x0db9, 0x0e19),
    CodePair::new(0x0dba, 0x0e1a),
    CodePair::new(0x0dbb, 0x0e1b),
    CodePair::new(0x0dbc, 0x0e1c),
    CodePair::new(0x0dbd, 0x0e1d),
    CodePair::new(0x0dbe, 0x0e1e),
    CodePair::new(0x0dbf, 0x0e1f),
    CodePair::new(0x0dc0, 0x0e20),
    CodePair::new(0x0dc1, 0x0e21),
    CodePair::new(0x0dc2, 0x0e22),
    CodePair::new(0x0dc3, 0x0e23),
    CodePair::new(0x0dc4, 0x0e24),
    CodePair::new(0x0dc5, 0x0e25),
    CodePair::new(0x0dc6, 0x0e26),
    CodePair::new(0x0dc7, 0x0e27),
    CodePair::new(0x0dc8, 0x0e28),
    CodePair::new(0x0dc9, 0x0e29),
    CodePair::new(0x0dca, 0x0e2a),
    CodePair::new(0x0dcb, 0x0e2b),
    CodePair::new(0x0dcc, 0x0e2c),
    CodePair::new(0x0dcd, 0x0e2d),
    CodePair::new(0x0dce, 0x0e2e),
    CodePair::new(0x0dcf, 0x0e2f),
    CodePair::new(0x0dd0, 0x0e30),
    CodePair::new(0x0dd1, 0x0e31),
    CodePair::new(0x0dd2, 0x0e32),
    CodePair::new(0x0dd3, 0x0e33),
    CodePair::new(0x0dd4, 0x0e34),
    CodePair::new(0x0dd5, 0x0e35),
    CodePair::new(0x0dd6, 0x0e36),
    CodePair::new(0x0dd7, 0x0e37),
    CodePair::new(0x0dd8, 0x0e38),
    CodePair::new(0x0dd9, 0x0e39),
    CodePair::new(0x0dda, 0x0e3a),
    CodePair::new(0x0ddf, 0x0e3f),
    CodePair::new(0x0de0, 0x0e40),
    CodePair::new(0x0de1, 0x0e41),
    CodePair::new(0x0de2, 0x0e42),
    CodePair::new(0x0de3, 0x0e43),
    CodePair::new(0x0de4, 0x0e44),
    CodePair::new(0x0de5, 0x0e45),
    CodePair::new(0x0de6, 0x0e46),
    CodePair::new(0x0de7, 0x0e47),
    CodePair::new(0x0de8, 0x0e48),
    CodePair::new(0x0de9, 0x0e49),
    CodePair::new(0x0dea, 0x0e4a),
    CodePair::new(0x0deb, 0x0e4b),
    CodePair::new(0x0dec, 0x0e4c),
    CodePair::new(0x0ded, 0x0e4d),
    CodePair::new(0x0df0, 0x0e50),
    CodePair::new(0x0df1, 0x0e51),
    CodePair::new(0x0df2, 0x0e52),
    CodePair::new(0x0df3, 0x0e53),
    CodePair::new(0x0df4, 0x0e54),
    CodePair::new(0x0df5, 0x0e55),
    CodePair::new(0x0df6, 0x0e56),
    CodePair::new(0x0df7, 0x0e57),
    CodePair::new(0x0df8, 0x0e58),
    CodePair::new(0x0df9, 0x0e59),
    CodePair::new(0x0ea1, 0x3131),
    CodePair::new(0x0ea2, 0x3132),
    CodePair::new(0x0ea3, 0x3133),
    CodePair::new(0x0ea4, 0x3134),
    CodePair::new(0x0ea5, 0x3135),
    CodePair::new(0x0ea6, 0x3136),
    CodePair::new(0x0ea7, 0x3137),
    CodePair::new(0x0ea8, 0x3138),
    CodePair::new(0x0ea9, 0x3139),
    CodePair::new(0x0eaa, 0x313a),
    CodePair::new(0x0eab, 0x313b),
    CodePair::new(0x0eac, 0x313c),
    CodePair::new(0x0ead, 0x313d),
    CodePair::new(0x0eae, 0x313e),
    CodePair::new(0x0eaf, 0x313f),
    CodePair::new(0x0eb0, 0x3140),
    CodePair::new(0x0eb1, 0x3141),
    CodePair::new(0x0eb2, 0x3142),
    CodePair::new(0x0eb3, 0x3143),
    CodePair::new(0x0eb4, 0x3144),
    CodePair::new(0x0eb5, 0x3145),
    CodePair::new(0x0eb6, 0x3146),
    CodePair::new(0x0eb7, 0x3147),
    CodePair::new(0x0eb8, 0x3148),
    CodePair::new(0x0eb9, 0x3149),
    CodePair::new(0x0eba, 0x314a),
    CodePair::new(0x0ebb, 0x314b),
    CodePair::new(0x0ebc, 0x314c),
    CodePair::new(0x0ebd, 0x314d),
    CodePair::new(0x0ebe, 0x314e),
    CodePair::new(0x0ebf, 0x314f),
    CodePair::new(0x0ec0, 0x3150),
    CodePair::new(0x0ec1, 0x3151),
    CodePair::new(0x0ec2, 0x3152),
    CodePair::new(0x0ec3, 0x3153),
    CodePair::new(0x0ec4, 0x3154),
    CodePair::new(0x0ec5, 0x3155),
    CodePair::new(0x0ec6, 0x3156),
    CodePair::new(0x0ec7, 0x3157),
    CodePair::new(0x0ec8, 0x3158),
    CodePair::new(0x0ec9, 0x3159),
    CodePair::new(0x0eca, 0x315a),
    CodePair::new(0x0ecb, 0x315b),
    CodePair::new(0x0ecc, 0x315c),
    CodePair::new(0x0ecd, 0x315d),
    CodePair::new(0x0ece, 0x315e),
    CodePair::new(0x0ecf, 0x315f),
    CodePair::new(0x0ed0, 0x3160),
    CodePair::new(0x0ed1, 0x3161),
    CodePair::new(0x0ed2, 0x3162),
    CodePair::new(0x0ed3, 0x3163),
    CodePair::new(0x0ed4, 0x11a8),
    CodePair::new(0x0ed5, 0x11a9),
    CodePair::new(0x0ed6, 0x11aa),
    CodePair::new(0x0ed7, 0x11ab),
    CodePair::new(0x0ed8, 0x11ac),
    CodePair::new(0x0ed9, 0x11ad),
    CodePair::new(0x0eda, 0x11ae),
    CodePair::new(0x0edb, 0x11af),
    CodePair::new(0x0edc, 0x11b0),
    CodePair::new(0x0edd, 0x11b1),
    CodePair::new(0x0ede, 0x11b2),
    CodePair::new(0x0edf, 0x11b3),
    CodePair::new(0x0ee0, 0x11b4),
    CodePair::new(0x0ee1, 0x11b5),
    CodePair::new(0x0ee2, 0x11b6),
    CodePair::new(0x0ee3, 0x11b7),
    CodePair::new(0x0ee4, 0x11b8),
    CodePair::new(0x0ee5, 0x11b9),
    CodePair::new(0x0ee6, 0x11ba),
    CodePair::new(0x0ee7, 0x11bb),
    CodePair::new(0x0ee8, 0x11bc),
    CodePair::new(0x0ee9, 0x11bd),
    CodePair::new(0x0eea, 0x11be),
    CodePair::new(0x0eeb, 0x11bf),
    CodePair::new(0x0eec, 0x11c0),
    CodePair::new(0x0eed, 0x11c1),
    CodePair::new(0x0eee, 0x11c2),
    CodePair::new(0x0eef, 0x316d),
    CodePair::new(0x0ef0, 0x3171),
    CodePair::new(0x0ef1, 0x3178),
    CodePair::new(0x0ef2, 0x317f),
    CodePair::new(0x0ef3, 0x3181),
    CodePair::new(0x0ef4, 0x3184),
    CodePair::new(0x0ef5, 0x3186),
    CodePair::new(0x0ef6, 0x318d),
    CodePair::new(0x0ef7, 0x318e),
    CodePair::new(0x0ef8, 0x11eb),
    CodePair::new(0x0ef9, 0x11f0),
    CodePair::new(0x0efa, 0x11f9),
    CodePair::new(0x0eff, 0x20a9),
    CodePair::new(0x13a4, 0x20ac),
    CodePair::new(0x13bc, 0x0152),
    CodePair::new(0x13bd, 0x0153),
    CodePair::new(0x13be, 0x0178),
    CodePair::new(0x20ac, 0x20ac),
    CodePair::new(0xfe50, '`' as u16),
    CodePair::new(0xfe51, 0x00b4),
    CodePair::new(0xfe52, '^' as u16),
    CodePair::new(0xfe53, '~' as u16),
    CodePair::new(0xfe54, 0x00af),
    CodePair::new(0xfe55, 0x02d8),
    CodePair::new(0xfe56, 0x02d9),
    CodePair::new(0xfe57, 0x00a8),
    CodePair::new(0xfe58, 0x02da),
    CodePair::new(0xfe59, 0x02dd),
    CodePair::new(0xfe5a, 0x02c7),
    CodePair::new(0xfe5b, 0x00b8),
    CodePair::new(0xfe5c, 0x02db),
    CodePair::new(0xfe5d, 0x037a),
    CodePair::new(0xfe5e, 0x309b),
    CodePair::new(0xfe5f, 0x309c),
    CodePair::new(0xfe63, '/' as u16),
    CodePair::new(0xfe64, 0x02bc),
    CodePair::new(0xfe65, 0x02bd),
    CodePair::new(0xfe66, 0x02f5),
    CodePair::new(0xfe67, 0x02f3),
    CodePair::new(0xfe68, 0x02cd),
    CodePair::new(0xfe69, 0xa788),
    CodePair::new(0xfe6a, 0x02f7),
    CodePair::new(0xfe6e, ',' as u16),
    CodePair::new(0xfe6f, 0x00a4),
    CodePair::new(0xfe80, 'a' as u16), // XK_dead_a
    CodePair::new(0xfe81, 'A' as u16), // XK_dead_A
    CodePair::new(0xfe82, 'e' as u16), // XK_dead_e
    CodePair::new(0xfe83, 'E' as u16), // XK_dead_E
    CodePair::new(0xfe84, 'i' as u16), // XK_dead_i
    CodePair::new(0xfe85, 'I' as u16), // XK_dead_I
    CodePair::new(0xfe86, 'o' as u16), // XK_dead_o
    CodePair::new(0xfe87, 'O' as u16), // XK_dead_O
    CodePair::new(0xfe88, 'u' as u16), // XK_dead_u
    CodePair::new(0xfe89, 'U' as u16), // XK_dead_U
    CodePair::new(0xfe8a, 0x0259),
    CodePair::new(0xfe8b, 0x018f),
    CodePair::new(0xfe8c, 0x00b5),
    CodePair::new(0xfe90, '_' as u16),
    CodePair::new(0xfe91, 0x02c8),
    CodePair::new(0xfe92, 0x02cc),
    CodePair::new(0xff80 /*XKB_KEY_KP_Space*/, ' ' as u16),
    CodePair::new(0xff95 /*XKB_KEY_KP_7*/, 0x0037),
    CodePair::new(0xff96 /*XKB_KEY_KP_4*/, 0x0034),
    CodePair::new(0xff97 /*XKB_KEY_KP_8*/, 0x0038),
    CodePair::new(0xff98 /*XKB_KEY_KP_6*/, 0x0036),
    CodePair::new(0xff99 /*XKB_KEY_KP_2*/, 0x0032),
    CodePair::new(0xff9a /*XKB_KEY_KP_9*/, 0x0039),
    CodePair::new(0xff9b /*XKB_KEY_KP_3*/, 0x0033),
    CodePair::new(0xff9c /*XKB_KEY_KP_1*/, 0x0031),
    CodePair::new(0xff9d /*XKB_KEY_KP_5*/, 0x0035),
    CodePair::new(0xff9e /*XKB_KEY_KP_0*/, 0x0030),
    CodePair::new(0xffaa /*XKB_KEY_KP_Multiply*/, '*' as u16),
    CodePair::new(0xffab /*XKB_KEY_KP_Add*/, '+' as u16),
    CodePair::new(0xffac /*XKB_KEY_KP_Separator*/, ',' as u16),
    CodePair::new(0xffad /*XKB_KEY_KP_Subtract*/, '-' as u16),
    CodePair::new(0xffae /*XKB_KEY_KP_Decimal*/, '.' as u16),
    CodePair::new(0xffaf /*XKB_KEY_KP_Divide*/, '/' as u16),
    CodePair::new(0xffb0 /*XKB_KEY_KP_0*/, 0x0030),
    CodePair::new(0xffb1 /*XKB_KEY_KP_1*/, 0x0031),
    CodePair::new(0xffb2 /*XKB_KEY_KP_2*/, 0x0032),
    CodePair::new(0xffb3 /*XKB_KEY_KP_3*/, 0x0033),
    CodePair::new(0xffb4 /*XKB_KEY_KP_4*/, 0x0034),
    CodePair::new(0xffb5 /*XKB_KEY_KP_5*/, 0x0035),
    CodePair::new(0xffb6 /*XKB_KEY_KP_6*/, 0x0036),
    CodePair::new(0xffb7 /*XKB_KEY_KP_7*/, 0x0037),
    CodePair::new(0xffb8 /*XKB_KEY_KP_8*/, 0x0038),
    CodePair::new(0xffb9 /*XKB_KEY_KP_9*/, 0x0039),
    CodePair::new(0xffbd /*XKB_KEY_KP_Equal*/, '=' as u16),
];

impl X11Display {
    pub unsafe extern "C" fn keysym_to_unicode(&self, keysym: super::libx11::KeySym) -> i32 {
        let mut min = 0 as libc::c_int;
        let mut max = (::std::mem::size_of::<[CodePair; 828]>() as libc::c_ulong)
            .wrapping_div(::std::mem::size_of::<CodePair>() as libc::c_ulong)
            .wrapping_sub(1 as libc::c_int as libc::c_ulong) as libc::c_int;
        let mut mid;
        if keysym >= 0x20 as libc::c_int as libc::c_ulong
            && keysym <= 0x7e as libc::c_int as libc::c_ulong
            || keysym >= 0xa0 as libc::c_int as libc::c_ulong
                && keysym <= 0xff as libc::c_int as libc::c_ulong
        {
            return keysym as i32;
        }
        if keysym & 0xff000000 as libc::c_uint as libc::c_ulong
            == 0x1000000 as libc::c_int as libc::c_ulong
        {
            return (keysym & 0xffffff as libc::c_int as libc::c_ulong) as i32;
        }
        while max >= min {
            mid = (min + max) / 2 as libc::c_int;
            if (KEYSYMTAB[mid as usize].keysym as libc::c_ulong) < keysym {
                min = mid + 1 as libc::c_int
            } else if KEYSYMTAB[mid as usize].keysym as libc::c_ulong > keysym {
                max = mid - 1 as libc::c_int
            } else {
                return KEYSYMTAB[mid as usize].ucs as i32;
            }
        }
        return -(1 as libc::c_int);
    }
}
