use std::collections::hash_map::DefaultHasher;
use std::hash::Hash;
use std::hash::Hasher;
use std::thread;
use std::time::Duration;

use minipac::client::Client;
use minipac::error::{ConnectError, Error};
use minipac::hostname::Hostname;
use minipac::server::Server;

fn calculate_hash<T: Hash>(t: &T) -> u64 {
	let mut s = DefaultHasher::new();
	t.hash(&mut s);
	s.finish()
}

#[test]
fn can_ping_server_when_disconnected() {
	let _ = Server::new(
		"localhost:8001",
		Hostname::new("N0CALL-4").unwrap(),
		Duration::from_secs(1),
		1000,
		3,
		1200,
	)
	.unwrap();

	let client = Client::new(
		"localhost:8001",
		Hostname::new("DEFGH-213").unwrap(),
		Duration::from_secs(3),
		1000,
		1,
		1200,
	)
	.unwrap();

	assert!(client.ping(Hostname::new("N0CALL-4").unwrap()).unwrap());
}

#[test]
fn can_ping_server_when_connected() {
	let _server = Server::new(
		"localhost:8002",
		Hostname::new("N0CALL-3").unwrap(),
		Duration::from_secs(1),
		1000,
		3,
		1200,
	)
	.unwrap();

	let client = Client::new(
		"localhost:8002",
		Hostname::new("DEFGH-212").unwrap(),
		Duration::from_secs(3),
		1000,
		1,
		1200,
	)
	.unwrap();

	let _stream = client.connect(Hostname::new("N0CALL-3").unwrap()).unwrap();

	assert!(client.ping(Hostname::new("N0CALL-3").unwrap()).unwrap());
}

#[test]
fn can_ping_server_when_connected_then_disconnected() {
	let _server = Server::new(
		"localhost:8000",
		Hostname::new("N0CALL-3").unwrap(),
		Duration::from_secs(1),
		1000,
		3,
		1200,
	)
	.unwrap();

	let client = Client::new(
		"localhost:8000",
		Hostname::new("DEFGH-212").unwrap(),
		Duration::from_secs(3),
		1000,
		1,
		1200,
	)
	.unwrap();

	{
		// connect, then immediately disconnect when the stream is dropped
		let _stream = client.connect(Hostname::new("N0CALL-3").unwrap()).unwrap();
	}

	{
		// do it again for good measure
		let _stream = client.connect(Hostname::new("N0CALL-3").unwrap()).unwrap();
	}

	assert!(client.ping(Hostname::new("N0CALL-3").unwrap()).unwrap());
}

#[test]
fn can_ping_client() {
	let client = Client::new(
		"localhost:8009",
		Hostname::new("DEFGH-212").unwrap(),
		Duration::from_secs(3),
		1000,
		1,
		1200,
	)
	.unwrap();

	let server = Server::new(
		"localhost:8009",
		Hostname::new("N0CALL-3").unwrap(),
		Duration::from_secs(1),
		1000,
		3,
		1200,
	)
	.unwrap();

	assert!(server.ping(Hostname::new("DEFGH-212").unwrap()).unwrap());

	// connect then ping again

	let _stream = client.connect(Hostname::new("N0CALL-3").unwrap()).unwrap();

	assert!(server.ping(Hostname::new("DEFGH-212").unwrap()).unwrap());
}

#[test]
fn cant_ping_nonexistant_host() {
	let _server = Server::new(
		"localhost:8003",
		Hostname::new("N0CALL-3").unwrap(),
		Duration::from_secs(1),
		1000,
		3,
		1200,
	)
	.unwrap();

	let client = Client::new(
		"localhost:8003",
		Hostname::new("DEFGH-212").unwrap(),
		Duration::from_secs(1),
		1000,
		1,
		1200,
	)
	.unwrap();

	assert!(!client.ping(Hostname::new("N0CALL-4").unwrap(),).unwrap());

	let _server = Server::new(
		"localhost:8004",
		Hostname::new("N0CALL-3").unwrap(),
		Duration::from_secs(1),
		1000,
		3,
		1200,
	)
	.unwrap();

	let client = Client::new(
		"localhost:8004",
		Hostname::new("DEFGH-212").unwrap(),
		Duration::from_secs(1),
		1000,
		1,
		1200,
	)
	.unwrap();

	assert!(!client.ping(Hostname::new("N0CAL3-3").unwrap(),).unwrap());
}

#[test]
fn can_establish_connection_and_transfer_data_single_packet() {
	let server = Server::new(
		"localhost:8005",
		Hostname::new("N0CALL-3").unwrap(),
		Duration::from_secs(1),
		1000,
		3,
		1200,
	)
	.unwrap();

	// echo server
	thread::spawn(move || {
		let (host, mut stream) = server.accept().unwrap();
		assert_eq!(host, Hostname::new("DEFGH-212").unwrap());

		let r = stream.read().unwrap();
		assert_eq!(vec![5, 34, 54], r);
		stream.write(r).unwrap();
	});

	let client = Client::new(
		"localhost:8005",
		Hostname::new("DEFGH-212").unwrap(),
		Duration::from_secs(1),
		1000,
		1,
		1200,
	)
	.unwrap();

	let mut stream = client.connect(Hostname::new("N0CALL-3").unwrap()).unwrap();

	stream.write(vec![5, 34, 54]).unwrap();
	assert_eq!(vec![5, 34, 54], stream.read().unwrap());

	// Try to read again
	let res = stream.read();
	assert!(matches!(res, Err(Error::Disconnected)));
}

#[test]
fn can_establish_connection_and_transfer_data_many_packets() {
	let server = Server::new(
		"localhost:8006",
		Hostname::new("N0CALL-3").unwrap(),
		Duration::from_secs(1),
		1000,
		3,
		1200,
	)
	.unwrap();

	let data: Vec<u8> = (0..(100_000)).map(|_| rand::random()).collect();

	// echo server
	{
		let data = data.clone();
		thread::spawn(move || {
			let (host, mut stream) = server.accept().unwrap();
			assert_eq!(host, Hostname::new("DEFGH-212").unwrap());

			let r = stream.read().unwrap();
			assert_eq!(data, r);
			stream.write(r).unwrap();
		});
	}

	let client = Client::new(
		"localhost:8006",
		Hostname::new("DEFGH-212").unwrap(),
		Duration::from_secs(1),
		1000,
		1,
		1200,
	)
	.unwrap();

	let mut stream = client.connect(Hostname::new("N0CALL-3").unwrap()).unwrap();

	stream.write(data.clone()).unwrap();
	assert_eq!(data, stream.read().unwrap());

	// Try to read again
	let res = stream.read();
	assert!(matches!(res, Err(Error::Disconnected)));
}

// Warning - very slow(>60 seconds)
#[test]
fn can_establish_connection_and_transfer_data_max_packets() {
	let server = Server::new(
		"localhost:8007",
		Hostname::new("N0CALL-3").unwrap(),
		Duration::from_secs(1),
		1000,
		3,
		1200,
	)
	.unwrap();

	// Can't be random because
	// it needs to compress to a predictable size
	// so that it uses exactly 65536 packets
	let data: Vec<u8> = (0..(978 * 65536 - 1500))
		.map(|x| (calculate_hash(&x)) as u8)
		.collect();

	// echo server
	{
		let data = data.clone();
		thread::spawn(move || {
			let (host, mut stream) = server.accept().unwrap();
			assert_eq!(host, Hostname::new("DEFGH-212").unwrap());

			let r = stream.read().unwrap();
			assert_eq!(data, r);
			stream.write(r).unwrap();
		});
	}

	let client = Client::new(
		"localhost:8007",
		Hostname::new("DEFGH-212").unwrap(),
		Duration::from_secs(1),
		1000,
		1,
		1200,
	)
	.unwrap();

	let mut stream = client.connect(Hostname::new("N0CALL-3").unwrap()).unwrap();

	stream.write(data.clone()).unwrap();
	assert_eq!(data, stream.read().unwrap());

	// Try to read again
	let res = stream.read();
	assert!(matches!(res, Err(Error::Disconnected)));
}

#[test]
fn multiple_actions_in_quick_succession() {
	let server = Server::new(
		"localhost:8008",
		Hostname::new("N0CALL-3").unwrap(),
		Duration::from_secs(1),
		1000,
		3,
		1200,
	)
	.unwrap();

	let data: Vec<u8> = (0..(100_000)).map(|_| rand::random()).collect();

	// echo server
	{
		let data = data.clone();
		thread::spawn(move || {
			let (host, mut stream) = server.accept().unwrap();
			assert_eq!(host, Hostname::new("DEFGH-212").unwrap());

			let r = stream.read().unwrap();
			assert_eq!(data, r);
			stream.write(r).unwrap();

			let r = stream.read().unwrap();
			assert_eq!(data, r);
			stream.write(r).unwrap();
		});
	}

	let client = Client::new(
		"localhost:8008",
		Hostname::new("DEFGH-212").unwrap(),
		Duration::from_secs(1),
		1000,
		1,
		1200,
	)
	.unwrap();

	let mut stream = client.connect(Hostname::new("N0CALL-3").unwrap()).unwrap();

	stream.write(data.clone()).unwrap();
	stream.write(data.clone()).unwrap();
	assert_eq!(data, stream.read().unwrap());
	assert_eq!(data, stream.read().unwrap());

	// Try to read again
	let res = stream.read();
	assert!(matches!(res, Err(Error::Disconnected)));
}

#[test]
fn two_connections_one_client() {
	let data1: Vec<u8> = (0..(1_000)).map(|_| rand::random()).collect();
	let data2: Vec<u8> = (0..(1_000)).map(|_| rand::random()).collect();

	let server1 = Server::new(
		"localhost:8010",
		Hostname::new("N0CALL-3").unwrap(),
		Duration::from_secs(1),
		1000,
		3,
		1200,
	)
	.unwrap();

	// echo server1
	{
		let data1 = data1.clone();
		thread::spawn(move || {
			let (host, mut stream) = server1.accept().unwrap();
			assert_eq!(host, Hostname::new("DEFGH-212").unwrap());

			let r = stream.read().unwrap();
			assert_eq!(data1, r);
			stream.write(r).unwrap();
		});
	}

	let server2 = Server::new(
		"localhost:8010",
		Hostname::new("N0CALL-4").unwrap(),
		Duration::from_secs(1),
		1000,
		3,
		1200,
	)
	.unwrap();

	// echo server2
	{
		let data2 = data2.clone();
		thread::spawn(move || {
			let (host, mut stream) = server2.accept().unwrap();
			assert_eq!(host, Hostname::new("DEFGH-212").unwrap());

			let r = stream.read().unwrap();
			assert_eq!(data2, r);
			stream.write(r).unwrap();
		});
	}

	let client = Client::new(
		"localhost:8010",
		Hostname::new("DEFGH-212").unwrap(),
		Duration::from_secs(1),
		1000,
		1,
		1200,
	)
	.unwrap();

	let mut stream1 = client.connect(Hostname::new("N0CALL-3").unwrap()).unwrap();
	let mut stream2 = client.connect(Hostname::new("N0CALL-4").unwrap()).unwrap();

	stream1.write(data1.clone()).unwrap();

	// sleep to prevent a race condition w/ the relay server
	// kind of janky, but necessary since the relay server doesn't understand KISS
	// and the servers will talk over each other otherwise
	thread::sleep(Duration::from_millis(1000));

	stream2.write(data2.clone()).unwrap();
	assert_eq!(data1, stream1.read().unwrap());
	assert_eq!(data2, stream2.read().unwrap());

	// Try to read again
	let res = stream1.read();
	assert!(matches!(res, Err(Error::Disconnected)));

	let res = stream2.read();
	assert!(matches!(res, Err(Error::Disconnected)));
}

#[test]
fn test_cant_connect_to_non_existant_server() {
	let _server = Server::new(
		"localhost:8011",
		Hostname::new("N0CALL-3").unwrap(),
		Duration::from_secs(1),
		1000,
		3,
		1200,
	)
	.unwrap();

	let client = Client::new(
		"localhost:8011",
		Hostname::new("DEFGH-212").unwrap(),
		Duration::from_secs(1),
		1000,
		1,
		1200,
	)
	.unwrap();

	assert!(matches!(
		client.connect(Hostname::new("N0CALL-4").unwrap()),
		Err(ConnectError::ConnectionFailure),
	));
}

#[test]
fn test_disconnection_kills_client_stream() {
	let server = Server::new(
		"localhost:8012",
		Hostname::new("N0CALL-3").unwrap(),
		Duration::from_secs(1),
		1000,
		3,
		1200,
	)
	.unwrap();

	let client = Client::new(
		"localhost:8012",
		Hostname::new("DEFGH-212").unwrap(),
		Duration::from_secs(1),
		1000,
		1,
		1200,
	)
	.unwrap();

	let mut cli_stream = client.connect(Hostname::new("N0CALL-3").unwrap()).unwrap();
	let (_, serv_stream) = server.accept().unwrap();

	thread::spawn(move || {
		// wait 1 second then disconnect
		thread::sleep(Duration::from_secs(1));
		drop(serv_stream)
	});

	assert!(matches!(cli_stream.read(), Err(Error::Disconnected)));
}

#[test]
fn test_disconnection_kills_server_stream() {
	let server = Server::new(
		"localhost:8013",
		Hostname::new("N0CALL-3").unwrap(),
		Duration::from_secs(1),
		1000,
		3,
		1200,
	)
	.unwrap();

	let client = Client::new(
		"localhost:8013",
		Hostname::new("DEFGH-212").unwrap(),
		Duration::from_secs(1),
		1000,
		1,
		1200,
	)
	.unwrap();

	let cli_stream = client.connect(Hostname::new("N0CALL-3").unwrap()).unwrap();
	let (_, mut serv_stream) = server.accept().unwrap();

	thread::spawn(move || {
		// wait 1 second then disconnect
		thread::sleep(Duration::from_secs(1));
		drop(cli_stream)
	});

	assert!(matches!(serv_stream.read(), Err(Error::Disconnected)));
}
