use crate::error;

use std::sync::{
	mpsc,
	mpsc::{Receiver, Sender, TryRecvError},
};

pub struct ReaderStream {
	receiver: Receiver<Vec<u8>>,
}

impl ReaderStream {
	pub fn read(&mut self) -> Result<Vec<u8>, error::Error> {
		self.receiver.recv().map_err(|_| error::Error::Disconnected)
	}

	/// Like read, but non-blocking
	pub fn try_read(&mut self) -> Result<Vec<u8>, TryRecvError> {
		self.receiver.try_recv()
	}
}

pub struct WriterStream {
	sender: Sender<Vec<u8>>,
}

impl WriterStream {
	pub fn write(&mut self, data: Vec<u8>) -> Result<(), error::Error> {
		self.sender
			.send(data)
			.map_err(|_| error::Error::Disconnected)
	}
}

pub struct Stream {
	reader: ReaderStream,
	writer: WriterStream,
}

impl Stream {
	pub fn new_entangled() -> (Self, Self) {
		let (s1, r1) = mpsc::channel();
		let (s2, r2) = mpsc::channel();

		(
			Self {
				writer: WriterStream { sender: s1 },
				reader: ReaderStream { receiver: r2 },
			},
			Self {
				writer: WriterStream { sender: s2 },
				reader: ReaderStream { receiver: r1 },
			},
		)
	}

	pub fn split(self) -> (ReaderStream, WriterStream) {
		(self.reader, self.writer)
	}

	pub fn read(&mut self) -> Result<Vec<u8>, error::Error> {
		self.reader.read()
	}

	/// Like read, but non-blocking
	pub fn try_recv(&mut self) -> Result<Vec<u8>, TryRecvError> {
		self.reader.try_read()
	}

	pub fn write(&mut self, data: Vec<u8>) -> Result<(), error::Error> {
		self.writer.write(data)
	}
}
