use crate::error::{BrokenPipeError, Error, MapBrokenPipe};
use crate::hostname::Hostname;
use crate::packet::try_read_packet;
use crate::packet::{Packet, PacketData};
use crate::state::ConnectionState;
use crate::stream::Stream;

use kiss_tnc::tnc::Tnc;
use std::collections::HashMap;
use std::net::ToSocketAddrs;
use std::sync::{
	mpsc,
	mpsc::{Receiver, Sender, TryRecvError},
};
use std::thread;
use std::time::{Duration, Instant};

pub struct Server {
	stream_receiver: Receiver<(Hostname, Stream)>,

	// sends channels which packets should be sent through
	// only used for pings right now
	channel_sender: mpsc::Sender<mpsc::Sender<Packet>>,

	// packets to be sent over the air
	// also only for pings right now
	packet_sender: mpsc::Sender<Vec<u8>>,

	hostname: Hostname,
	timeout: Duration,
}

impl Server {
	// TODO builder
	pub fn new<A: ToSocketAddrs>(
		addr: A,
		hostname: Hostname,
		timeout: Duration,
		max_packet_size: usize,
		max_retries: usize,
		baud_rate: usize,
	) -> Result<Self, std::io::Error> {
		let mut tnc = Tnc::connect(addr)?;

		let (stream_sender, stream_receiver) = mpsc::channel();
		let (packet_sender, packet_receiver) = mpsc::channel();
		let (channel_sender, channel_receiver) = mpsc::channel::<mpsc::Sender<Packet>>();
		let (received_packet_tx, received_packet_rx) = mpsc::channel();

		// take incoming frames and stuff them in a channel
		{
			let mut tnc = tnc.try_clone()?;

			thread::spawn(move || {
				let mut channel = None;
				while let Ok((_, data)) = tnc.read_frame() {
					if let Ok(ch) = channel_receiver.try_recv() {
						channel = Some(ch);
					}

					if let Some(packet) = Packet::from_bytes(&data) {
						if let Some(ch) = &channel {
							if ch.send(packet.clone()).is_err() {
								channel = None;
							}
						}

						if received_packet_tx.send(packet).is_err() {
							// Channel closed. End this thread
							break;
						};
					}
				}
			});
		}

		// processes incoming frames
		{
			let packet_sender = packet_sender.clone();
			thread::spawn(move || {
				let mut reader = ServerReader::new(
					hostname,
					stream_sender,
					packet_sender,
					timeout,
					max_packet_size,
					max_retries,
					baud_rate,
				);

				loop {
					match received_packet_rx.try_recv() {
						Ok(packet) => {
							if packet.to == hostname {
								if let Err(BrokenPipeError) = reader.process_packet(packet) {
									// channel died
									// kill our thread
									break;
								}
							}
						}

						Err(TryRecvError::Empty) => {
							thread::sleep(Duration::from_millis(10));
						}

						Err(TryRecvError::Disconnected) => {
							// channel died
							// kill our thread
							break;
						}
					}

					let mut disconnected_hosts = vec![];
					for (host, state) in &mut reader.connections {
						match state.process() {
							Ok(()) => {}

							Err(Error::Disconnected) => {
								if let Err(BrokenPipeError) = state.disconnect(hostname, *host) {
									// channel died
									// kill our thread
									break;
								}

								disconnected_hosts.push(*host);
							}

							Err(Error::BrokenPipe) => {
								// channel died
								// kill our thread
								break;
							}
						}
					}

					for host in disconnected_hosts {
						reader.connections.remove(&host);
					}
				}
			});
		}

		// processes outgoing frames
		thread::spawn(move || loop {
			while let Ok(data) = packet_receiver.recv() {
				if tnc.send_frame(&data).is_err() {
					break;
				}
			}
		});

		Ok(Self {
			stream_receiver,
			channel_sender,
			packet_sender,
			hostname,
			timeout,
		})
	}

	/// Accepts a connection from a client.
	/// Returns an error if the channel is disconnected (i.e. something went wrong with the TNC connection)
	pub fn accept(&self) -> Result<(Hostname, Stream), BrokenPipeError> {
		self.stream_receiver.recv().map_broken_pipe()
	}

	/// Pings a host
	pub fn ping(&self, to: Hostname) -> Result<bool, BrokenPipeError> {
		let id = rand::random();
		let next = (id % 255) + 1;

		let (packet_sender, packet_receiver) = mpsc::channel();
		self.channel_sender.send(packet_sender).map_broken_pipe()?;

		let mut buf = vec![];
		Packet::new(self.hostname, to, id, PacketData::Ping).to_bytes(&mut buf);
		self.packet_sender.send(buf).map_broken_pipe()?;

		let max_time = Instant::now() + self.timeout;
		while Instant::now() < max_time {
			if let Some(packet) = try_read_packet(&packet_receiver)? {
				if packet == Packet::new(to, self.hostname, next, PacketData::Pong) {
					// Ping success
					return Ok(true);
				}
			}

			thread::sleep(Duration::from_millis(10));
		}

		// No response
		Ok(false)
	}
}

struct ServerReader {
	hostname: Hostname,
	stream_sender: Sender<(Hostname, Stream)>,
	packet_sender: Sender<Vec<u8>>,
	connections: HashMap<Hostname, ConnectionState>,

	// Needed to pass the settings to the ConnectionState
	timeout: Duration,
	max_packet_size: usize,
	max_retries: usize,
	baud_rate: usize,
}

impl ServerReader {
	fn new(
		hostname: Hostname,
		stream_sender: Sender<(Hostname, Stream)>,
		packet_sender: Sender<Vec<u8>>,
		timeout: Duration,
		max_packet_size: usize,
		max_retries: usize,
		baud_rate: usize,
	) -> Self {
		Self {
			hostname,
			stream_sender,
			packet_sender,
			connections: HashMap::new(),

			timeout,
			max_packet_size,
			max_retries,
			baud_rate,
		}
	}

	fn process_packet(&mut self, packet: Packet) -> Result<(), BrokenPipeError> {
		match &packet.data {
			PacketData::Connect => self.connection_from(packet.from)?,
			PacketData::Ping => self.pong(packet.from, packet.seq_num)?,
			_ => {}
		}

		let from = packet.from;
		if let Some(conn) = self.connections.get_mut(&from) {
			match conn.incoming_packet(packet) {
				Ok(()) => {}
				Err(Error::Disconnected) => {
					self.connections.remove(&from);
				}
				Err(Error::BrokenPipe) => return Err(BrokenPipeError),
			}
		}

		Ok(())
	}

	fn pong(&mut self, to: Hostname, id: u8) -> Result<(), BrokenPipeError> {
		let mut buf = vec![];
		Packet::new(self.hostname, to, (id % 255) + 1, PacketData::Pong).to_bytes(&mut buf);
		self.packet_sender.send(buf).map_broken_pipe()?;

		Ok(())
	}

	fn connection_from(&mut self, from: Hostname) -> Result<(), BrokenPipeError> {
		let (s1, s2) = Stream::new_entangled();

		let mut state = ConnectionState::new(
			self.timeout,
			self.max_packet_size - crate::packet::PACKET_OVERHEAD_BYTES,
			self.max_retries,
			self.baud_rate,
			s1,
			self.packet_sender.clone(),
			self.hostname,
			from,
		);

		// Send ACK
		let mut buf = vec![];
		Packet::new(self.hostname, from, state.seq_inc(), PacketData::ConnectAck)
			.to_bytes(&mut buf);
		self.packet_sender.send(buf).map_broken_pipe()?;

		// If we're already connected, reset the connection

		self.connections.insert(from, state);
		self.stream_sender.send((from, s2)).map_broken_pipe()?;

		Ok(())
	}
}
