use crate::error::BrokenPipeError;
use crate::hostname::Hostname;

use crc::{Crc, CRC_16_IBM_SDLC};
use std::sync::{mpsc, mpsc::TryRecvError};

// 8 bytes for hostname
// 8 bytes for hostname
// 1 byte for packet type
// 1 byte for seq num
// Up to 2 bytes for packet metadata (frame id or # of frames)
// 2 bytes for checksum
pub(crate) const PACKET_OVERHEAD_BYTES: usize = 8 + 8 + 1 + 1 + 2 + 2;

const X25: Crc<u16> = Crc::<u16>::new(&CRC_16_IBM_SDLC);

#[derive(Debug, PartialEq, Clone)]
pub(crate) struct Packet {
	pub from: Hostname,
	pub to: Hostname,
	pub seq_num: u8,
	pub data: PacketData,
}

impl Packet {
	pub fn new(from: Hostname, to: Hostname, seq_num: u8, data: PacketData) -> Self {
		Self {
			from,
			to,
			data,
			seq_num,
		}
	}

	pub fn to_bytes(&self, buf: &mut Vec<u8>) {
		let start_idx = buf.len();

		self.from.to_bytes(buf);
		self.to.to_bytes(buf);
		buf.push(self.seq_num);
		self.data.to_bytes(buf);

		// Add checksum to tail of buffer
		let checksum = X25.checksum(&buf[start_idx..]);
		buf.extend(u16_to_u8s(checksum));
	}

	pub fn from_bytes(buf: &[u8]) -> Option<Self> {
		if buf.len() < 19 {
			return None;
		}

		let checksum_actual = u8s_to_u16(&buf[(buf.len() - 2)..]);
		let buf = &buf[0..(buf.len() - 2)];
		let checksum_expected = X25.checksum(buf);

		if checksum_actual != checksum_expected {
			return None;
		}

		let from = Hostname::from_bytes(&buf[0..8])?;
		let to = Hostname::from_bytes(&buf[8..16])?;
		let seq_num = buf[16];
		let data = PacketData::from_bytes(&buf[17..])?;

		Some(Self {
			from,
			to,
			seq_num,
			data,
		})
	}
}

const PING: u8 = 0;
const PONG: u8 = 1;
const CONNECT: u8 = 2;
const CONNECT_ACK: u8 = 3;
const DISCONNECT: u8 = 4;
const DISCONNECT_ACK: u8 = 5;
const DATA_START: u8 = 6;
const DATA_START_ACK: u8 = 7;
const DATA_MID: u8 = 8;
const DATA_RESEND: u8 = 9;

#[derive(PartialEq, Eq, Debug, Clone)]
pub(crate) enum PacketData {
	Ping,
	Pong,

	Connect,
	ConnectAck,

	Disconnect,
	DisconnectAck,

	// Highest frame ID, data
	DataStart(u16, Vec<u8>),
	DataStartAck,
	// Frame ID, data
	DataMid(u16, Vec<u8>),
	// Frame IDs to resend
	// DataResend w/ empty vec = DataAck
	DataResend(Vec<u16>),
}

impl PacketData {
	pub fn to_bytes(&self, buf: &mut Vec<u8>) {
		match self {
			Self::Ping => {
				buf.push(PING);
			}

			Self::Pong => {
				buf.push(PONG);
			}

			Self::Connect => buf.push(CONNECT),

			Self::ConnectAck => buf.push(CONNECT_ACK),

			Self::Disconnect => buf.push(DISCONNECT),
			Self::DisconnectAck => buf.push(DISCONNECT_ACK),

			Self::DataStart(num_frames, data) => {
				buf.push(DATA_START);
				buf.extend(u16_to_u8s(*num_frames)); // Two bytes for total # of frames
				buf.extend(data);
			}

			Self::DataStartAck => buf.push(DATA_START_ACK),

			Self::DataMid(frame_id, data) => {
				buf.push(DATA_MID);
				buf.extend(u16_to_u8s(*frame_id)); // Two bytes for current frame id
				buf.extend(data);
			}

			Self::DataResend(frame_ids) => {
				buf.push(DATA_RESEND);
				buf.extend(
					frame_ids
						.iter()
						.map(|frame_id| u16_to_u8s(*frame_id))
						.flatten()
						.collect::<Vec<_>>(),
				);
			}
		}
	}

	pub fn from_bytes(buf: &[u8]) -> Option<Self> {
		let id = buf[0];
		let buf = &buf[1..];

		match id {
			PING => {
				if buf.is_empty() {
					Some(Self::Ping)
				} else {
					None
				}
			}

			PONG => {
				if buf.is_empty() {
					Some(Self::Pong)
				} else {
					None
				}
			}

			CONNECT => {
				if buf.is_empty() {
					Some(Self::Connect)
				} else {
					None
				}
			}

			CONNECT_ACK => {
				if buf.is_empty() {
					Some(Self::ConnectAck)
				} else {
					None
				}
			}

			DISCONNECT => {
				if buf.is_empty() {
					Some(Self::Disconnect)
				} else {
					None
				}
			}

			DISCONNECT_ACK => {
				if buf.is_empty() {
					Some(Self::DisconnectAck)
				} else {
					None
				}
			}

			DATA_START => {
				if buf.len() >= 2 {
					// Num additional frames - not including this one
					let num_frames = &buf[0..2];

					let data = buf[2..].to_owned();

					Some(Self::DataStart(u8s_to_u16(num_frames), data))
				} else {
					None
				}
			}

			DATA_START_ACK => {
				if buf.is_empty() {
					Some(Self::DataStartAck)
				} else {
					None
				}
			}

			DATA_MID => {
				if buf.len() >= 2 {
					let frame_id = &buf[0..2];
					let data = buf[2..].to_owned();

					Some(Self::DataMid(u8s_to_u16(frame_id), data))
				} else {
					None
				}
			}

			DATA_RESEND => {
				if buf.len() % 2 == 0 {
					let frame_ids = buf.chunks(2).map(u8s_to_u16).collect();

					Some(Self::DataResend(frame_ids))
				} else {
					None
				}
			}

			_ => None,
		}
	}
}

fn u16_to_u8s(i: u16) -> [u8; 2] {
	let lower = (i & 255) as u8;
	let upper = ((i >> 8) & 255) as u8;

	[lower, upper]
}

fn u8s_to_u16(i: &[u8]) -> u16 {
	i[0] as u16 + ((i[1] as u16) << 8)
}

pub(crate) fn try_read_packet(
	packet_receiver: &mpsc::Receiver<Packet>,
) -> Result<Option<Packet>, BrokenPipeError> {
	match packet_receiver.try_recv() {
		Ok(packet) => Ok(Some(packet)),
		Err(TryRecvError::Empty) => Ok(None),
		Err(TryRecvError::Disconnected) => Err(BrokenPipeError),
	}
}

pub(crate) fn is_valid_seq_num(cur: u8, new: u8) -> bool {
	if new > cur {
		true
	} else {
		(255 - cur + new) < 100
	}
}

#[cfg(test)]
mod tests {
	use super::*;

	#[test]
	fn encode_and_decode_ping() {
		let packet = Packet::new(
			Hostname::new("N0CALL-0").unwrap(),
			Hostname::new("ABCDEFG-243").unwrap(),
			37,
			PacketData::Ping,
		);

		let mut buf = vec![];
		packet.to_bytes(&mut buf);

		let new_packet = Packet::from_bytes(&buf).unwrap();
		assert_eq!(packet, new_packet);
	}

	#[test]
	fn encode_and_decode_data_start() {
		let packet = Packet::new(
			Hostname::new("N0CALL-0").unwrap(),
			Hostname::new("ABCDEFG-243").unwrap(),
			1,
			PacketData::DataStart(
				5,
				vec![
					1, 2, 5, 7, 8, 43, 54, 7, 46, 56, 25, 6, 4, 7, 98, 34, 7, 54, 85, 35,
				],
			),
		);

		let mut buf = vec![];
		packet.to_bytes(&mut buf);

		let new_packet = Packet::from_bytes(&buf).unwrap();
		assert_eq!(packet, new_packet);
	}

	#[test]
	fn encode_and_decode_data_start_ack() {
		let packet = Packet::new(
			Hostname::new("N0CALL-0").unwrap(),
			Hostname::new("ABCDEFG-243").unwrap(),
			2,
			PacketData::DataStartAck,
		);

		let mut buf = vec![];
		packet.to_bytes(&mut buf);

		let new_packet = Packet::from_bytes(&buf).unwrap();
		assert_eq!(packet, new_packet);
	}

	#[test]
	fn test_crc_detects_error_properly_all_bitflips() {
		let packet = Packet::new(
			Hostname::new("N0CALL-0").unwrap(),
			Hostname::new("ABCDEFG-243").unwrap(),
			37,
			PacketData::Ping,
		);

		let mut buf = vec![];
		packet.to_bytes(&mut buf);

		for i in 0..buf.len() {
			for j in 0..8 {
				// Flip single bit (jth position of ith byte)
				buf[i] ^= 1 << j;

				// Ensure the packet was not decoded

				let new_packet = Packet::from_bytes(&buf);
				assert!(new_packet.is_none());

				// Flip it back and confirm it decodes
				buf[i] ^= 1 << j;

				let new_packet = Packet::from_bytes(&buf).unwrap();
				assert_eq!(packet, new_packet);
			}
		}
	}

	#[test]
	fn seq_num_invalid_if_eq() {
		for i in 0..=255 {
			assert!(!is_valid_seq_num(i, i));
		}
	}

	#[test]
	fn seq_num_valid_if_greater() {
		for i in 0..=254 {
			for j in (i + 1)..=255 {
				assert!(is_valid_seq_num(i, j), "{} -> {} was not valid", i, j);
			}
		}
	}

	#[test]
	fn seq_num_valid_if_less_but_within_100() {
		for i in 156i32..=255 {
			let max = (i + 99) % 255;

			assert!(max < i);

			for j in 0..=max {
				assert!(is_valid_seq_num(i as u8, j as u8));
			}

			for j in (max + 1)..=i {
				assert!(!is_valid_seq_num(i as u8, j as u8));
			}
		}
	}
}
