use thiserror::Error;

#[derive(Debug, Error)]
pub enum Error {
	#[error("Disconnected from node")]
	Disconnected,

	#[error("Broken pipe while communicating with the TNC")]
	BrokenPipe,
}

impl From<BrokenPipeError> for Error {
	fn from(_: BrokenPipeError) -> Self {
		Self::BrokenPipe
	}
}

#[derive(Debug, Error)]
pub enum ConnectError {
	#[error("Could not connect to node")]
	ConnectionFailure,

	#[error("Already connected from this host to that host")]
	DuplicateConnection,

	#[error("Broken pipe while communicating with the TNC")]
	BrokenPipe,

	#[error("Error while communicating with the TNC: {0}")]
	IoError(std::io::Error),
}

impl From<BrokenPipeError> for ConnectError {
	fn from(_: BrokenPipeError) -> Self {
		Self::BrokenPipe
	}
}

impl From<std::io::Error> for ConnectError {
	fn from(e: std::io::Error) -> Self {
		Self::IoError(e)
	}
}

#[derive(Debug, Error)]
#[error("Broken pipe while communicating with the TNC")]
pub struct BrokenPipeError;

pub(crate) trait MapBrokenPipe<T> {
	fn map_broken_pipe(self) -> Result<T, BrokenPipeError>;
	fn map_disconnected(self) -> Result<T, Error>;
}

impl<T, Q> MapBrokenPipe<T> for Result<T, Q> {
	fn map_broken_pipe(self) -> Result<T, BrokenPipeError> {
		match self {
			Ok(x) => Ok(x),
			Err(_) => Err(BrokenPipeError),
		}
	}

	fn map_disconnected(self) -> Result<T, Error> {
		match self {
			Ok(x) => Ok(x),
			Err(_) => Err(Error::Disconnected),
		}
	}
}
