use crate::error::{BrokenPipeError, ConnectError, Error, MapBrokenPipe};
use crate::hostname::Hostname;
use crate::packet::try_read_packet;
use crate::packet::{Packet, PacketData};
use crate::state::ConnectionState;
use crate::stream::Stream;

use kiss_tnc::tnc::Tnc;
use std::net::ToSocketAddrs;
use std::sync::{mpsc, mpsc::TryRecvError};
use std::thread;
use std::time::{Duration, Instant};

pub struct Client {
	us: Hostname,
	timeout: Duration,
	max_packet_size: usize,
	max_retries: usize,
	baud_rate: usize,

	// sends packets to the Tnc (mpsc)
	packet_sender: mpsc::Sender<Vec<u8>>,
	// sends channels which packets should be sent through
	channel_sender: mpsc::Sender<mpsc::Sender<Packet>>,
}

impl Client {
	/// Create a new client
	/// Used for initiating connections and sending pings
	pub fn new<A: ToSocketAddrs>(
		addr: A,
		us: Hostname,
		timeout: Duration,
		max_packet_size: usize,
		max_retries: usize,
		baud_rate: usize,
	) -> Result<Client, ConnectError> {
		let mut tnc = Tnc::connect(addr)?;

		let (packet_sender, packet_receiver) = mpsc::channel::<Vec<_>>();
		let (channel_sender, channel_receiver) = mpsc::channel();

		// take incoming frames and stuff them in a channel
		{
			let mut tnc = tnc.try_clone()?;
			let packet_sender = packet_sender.clone();

			thread::spawn(move || {
				let mut channels: Vec<mpsc::Sender<Packet>> = vec![];
				while let Ok((_, data)) = tnc.read_frame() {
					loop {
						match channel_receiver.try_recv() {
							Ok(ch) => channels.push(ch),
							Err(TryRecvError::Disconnected) => {
								// end thread
								return;
							}
							Err(TryRecvError::Empty) => {
								// break loop
								break;
							}
						}
					}

					if let Some(packet) = Packet::from_bytes(&data) {
						for i in (0..channels.len()).rev() {
							if channels[i].send(packet.clone()).is_err() {
								channels.remove(i);
							}
						}

						// handle pings
						if packet.to == us && packet.data == PacketData::Ping {
							// send pong
							let mut buf = vec![];
							Packet::new(
								us,
								packet.from,
								(packet.seq_num % 255) + 1,
								PacketData::Pong,
							)
							.to_bytes(&mut buf);
							if packet_sender.send(buf).is_err() {
								break;
							}
						}
					}
				}
			});
		}

		// process outgoing frames
		{
			thread::spawn(move || {
				while let Ok(data) = packet_receiver.recv() {
					if tnc.send_frame(&data).is_err() {
						break;
					}
				}
			});
		}

		Ok(Self {
			us,
			timeout,
			max_packet_size,
			max_retries,
			baud_rate,
			packet_sender,
			channel_sender,
		})
	}

	/// Connects to the specified server.
	/// Returns a stream which can be used to send/receive data.
	pub fn connect(&self, to: Hostname) -> Result<Stream, ConnectError> {
		let (s1, s2) = Stream::new_entangled();
		let (packet_sender, packet_receiver) = mpsc::channel();
		self.channel_sender.send(packet_sender).map_broken_pipe()?;

		let mut client = ClientReader {
			state: ConnectionState::new(
				self.timeout,
				self.max_packet_size - crate::packet::PACKET_OVERHEAD_BYTES,
				self.max_retries,
				self.baud_rate,
				s1,
				self.packet_sender.clone(),
				self.us,
				to,
			),
			packet_receiver,
		};

		client.actually_connect()?;

		// process incoming frames
		{
			thread::spawn(move || {
				let _ = client.process_forever();
			});
		}

		Ok(s2)
	}

	/// Pings a host
	pub fn ping(&self, to: Hostname) -> Result<bool, BrokenPipeError> {
		let id = rand::random();
		let next = (id % 255) + 1;

		let (packet_sender, packet_receiver) = mpsc::channel();
		self.channel_sender.send(packet_sender).map_broken_pipe()?;

		let mut buf = vec![];
		Packet::new(self.us, to, id, PacketData::Ping).to_bytes(&mut buf);
		self.packet_sender.send(buf).map_broken_pipe()?;

		let max_time = Instant::now() + self.timeout;
		while Instant::now() < max_time {
			if let Some(packet) = try_read_packet(&packet_receiver)? {
				if packet == Packet::new(to, self.us, next, PacketData::Pong) {
					// Ping success
					return Ok(true);
				}
			}

			thread::sleep(Duration::from_millis(10));
		}

		// No response
		Ok(false)
	}
}

pub struct ClientReader {
	state: ConnectionState,
	packet_receiver: mpsc::Receiver<Packet>,
}

impl ClientReader {
	fn process_forever(&mut self) -> Result<(), Error> {
		loop {
			match self.packet_receiver.try_recv() {
				Ok(packet) => {
					if packet.to == self.state.us && packet.from == self.state.them {
						self.process_packet(packet)?;
					}
				}

				Err(TryRecvError::Empty) => {
					thread::sleep(Duration::from_millis(10));
				}

				Err(TryRecvError::Disconnected) => {
					// channel died
					// kill our thread
					return Err(Error::BrokenPipe);
				}
			}

			self.state.process()?;
		}
	}

	fn process_packet(&mut self, packet: Packet) -> Result<(), Error> {
		self.state.incoming_packet(packet)
	}

	fn actually_connect(&mut self) -> Result<(), ConnectError> {
		let mut connected = false;
		for _ in 0..self.state.max_retries {
			match self.try_establish_connection() {
				Ok(()) => {
					connected = true;
					break;
				}

				Err(ConnectError::ConnectionFailure) => {} // try again
				Err(e) => {
					return Err(e);
				}
			}
		}

		if !connected {
			return Err(ConnectError::ConnectionFailure);
		}

		Ok(())
	}

	fn try_establish_connection(&mut self) -> Result<(), ConnectError> {
		// Send connect packet
		let mut buf = vec![];
		Packet::new(
			self.state.us,
			self.state.them,
			self.state.seq_inc(),
			PacketData::Connect,
		)
		.to_bytes(&mut buf);
		self.state.packet_sender.send(buf).map_broken_pipe()?;

		let max_time = Instant::now() + self.state.timeout;
		while Instant::now() < max_time {
			if let Some(packet) = try_read_packet(&self.packet_receiver)? {
				if packet.from == self.state.them
					&& packet.to == self.state.us
					&& packet.data == PacketData::ConnectAck
				{
					// Connection established
					return Ok(());
				}
			}
		}

		Err(ConnectError::ConnectionFailure)
	}
}
