use std::collections::hash_map::DefaultHasher;
use std::collections::VecDeque;
use std::hash::Hash;
use std::hash::Hasher;
use std::io::{Read, Write};
use std::sync::mpsc;
use std::sync::mpsc::{Receiver, Sender};
use std::thread;
use std::time::Duration;

use minipac::client::Client;
use minipac::error::Error;
use minipac::handler::{Action, Handler};
use minipac::hostname::Hostname;
use minipac::server::Server;

struct EntangledStream {
	tx: Sender<u8>,
	rx: Receiver<u8>,
	recv_buffer: VecDeque<u8>,
}

impl EntangledStream {
	fn new_pair() -> (Self, Self) {
		let (tx1, rx1) = mpsc::channel();
		let (tx2, rx2) = mpsc::channel();

		(
			Self {
				tx: tx1,
				rx: rx2,
				recv_buffer: VecDeque::new(),
			},
			Self {
				tx: tx2,
				rx: rx1,
				recv_buffer: VecDeque::new(),
			},
		)
	}
}

impl Read for EntangledStream {
	fn read(&mut self, buf: &mut [u8]) -> Result<usize, std::io::Error> {
		for _ in 0..10 {
			if self.recv_buffer.len() < buf.len() {
				thread::sleep(Duration::from_millis(10));
			}

			while let Ok(b) = self.rx.try_recv() {
				self.recv_buffer.push_back(b);
			}
		}

		if self.recv_buffer.len() >= buf.len() {
			for b in &mut buf.iter_mut() {
				*b = self.recv_buffer.pop_front().unwrap();
			}

			Ok(buf.len())
		} else {
			Err(std::io::Error::new(std::io::ErrorKind::TimedOut, "timeout"))
		}
	}
}

impl Write for EntangledStream {
	fn write(&mut self, buf: &[u8]) -> Result<usize, std::io::Error> {
		for &b in buf {
			self.tx.send(b).unwrap();
		}

		Ok(buf.len())
	}
	fn flush(&mut self) -> Result<(), std::io::Error> {
		Ok(())
	}
}

struct LoggerHandler {}

impl LoggerHandler {
	fn new() -> Self {
		Self {}
	}
}

impl Handler for LoggerHandler {
	fn recv(&mut self, data: &[u8]) -> Action {
		Action::SendData(data.to_owned())
	}
	fn process(&mut self) -> Action {
		Action::Nothing
	}
}

fn calculate_hash<T: Hash>(t: &T) -> u64 {
	let mut s = DefaultHasher::new();
	t.hash(&mut s);
	s.finish()
}

#[test]
fn can_ping() {
	let (s1, s2) = EntangledStream::new_pair();

	let mut server = Server::new_with_stream(
		s1,
		Hostname::new("N0CALL-3").unwrap(),
		Duration::from_secs(1),
		1000,
		3,
		1200,
		|_| LoggerHandler::new(),
	);

	thread::spawn(move || {
		server.serve_forever().unwrap();
	});

	let mut client = Client::new_with_stream(
		s2,
		Hostname::new("DEFGH-212").unwrap(),
		Hostname::new("N0CALL-3").unwrap(),
		Duration::from_secs(1),
		1000,
		1,
		1200,
	);

	assert!(client.ping().unwrap());
}

#[test]
fn cant_ping_nonexistant_host() {
	let (s1, s2) = EntangledStream::new_pair();

	let mut server = Server::new_with_stream(
		s1,
		Hostname::new("N0CALL-3").unwrap(),
		Duration::from_secs(1),
		1000,
		3,
		1200,
		|_| LoggerHandler::new(),
	);

	thread::spawn(move || {
		server.serve_forever().unwrap();
	});

	let mut client = Client::new_with_stream(
		s2,
		Hostname::new("DEFGH-212").unwrap(),
		Hostname::new("N0CALL-4").unwrap(),
		Duration::from_secs(1),
		1000,
		1,
		1200,
	);

	assert!(!client.ping().unwrap());

	let (s1, s2) = EntangledStream::new_pair();

	let mut server = Server::new_with_stream(
		s1,
		Hostname::new("N0CALL-3").unwrap(),
		Duration::from_secs(1),
		1000,
		3,
		1200,
		|_| LoggerHandler::new(),
	);

	thread::spawn(move || {
		server.serve_forever().unwrap();
	});

	let mut client = Client::new_with_stream(
		s2,
		Hostname::new("DEFGH-212").unwrap(),
		Hostname::new("N0CAL3-3").unwrap(),
		Duration::from_secs(1),
		1000,
		1,
		1200,
	);

	assert!(!client.ping().unwrap());
}

#[test]
fn can_establish_connection_and_transfer_data_single_packet() {
	let (s1, s2) = EntangledStream::new_pair();

	let mut server = Server::new_with_stream(
		s1,
		Hostname::new("N0CALL-3").unwrap(),
		Duration::from_secs(1),
		1000,
		3,
		1200,
		|_| LoggerHandler::new(),
	);

	thread::spawn(move || {
		server.serve_forever().unwrap();
	});

	let mut client = Client::new_with_stream(
		s2,
		Hostname::new("DEFGH-212").unwrap(),
		Hostname::new("N0CALL-3").unwrap(),
		Duration::from_secs(1),
		1000,
		1,
		1200,
	);

	struct ClientHandler(bool, Vec<u8>, bool);
	impl Handler for ClientHandler {
		fn recv(&mut self, data: &[u8]) -> Action {
			if data == self.1 {
				self.2 = true;

				Action::Disconnect
			} else {
				Action::Nothing
			}
		}

		fn process(&mut self) -> Action {
			if !self.0 {
				self.0 = true;
				Action::SendData(self.1.clone())
			} else {
				Action::Nothing
			}
		}
	}

	let mut handler = ClientHandler(false, vec![], false);
	let res = client.connect(&mut handler);
	assert!(matches!(
		res,
		Err(Error::Disconnected(callsign)) if callsign == Hostname::new("N0CALL-3").unwrap()
	));

	assert!(handler.0);
	assert!(handler.2);
}

#[test]
fn can_establish_connection_and_transfer_data_many_packets() {
	let (s1, s2) = EntangledStream::new_pair();

	let mut server = Server::new_with_stream(
		s1,
		Hostname::new("N0CALL-3").unwrap(),
		Duration::from_secs(1),
		1000,
		3,
		1200,
		|_| LoggerHandler::new(),
	);

	thread::spawn(move || {
		server.serve_forever().unwrap();
	});

	let mut client = Client::new_with_stream(
		s2,
		Hostname::new("DEFGH-212").unwrap(),
		Hostname::new("N0CALL-3").unwrap(),
		Duration::from_secs(1),
		1000,
		1,
		1200,
	);

	struct ClientHandler(bool, Vec<u8>, bool);
	impl Handler for ClientHandler {
		fn recv(&mut self, data: &[u8]) -> Action {
			if data == self.1 {
				self.2 = true;

				Action::Disconnect
			} else {
				Action::Nothing
			}
		}

		fn process(&mut self) -> Action {
			if !self.0 {
				self.0 = true;
				Action::SendData(self.1.clone())
			} else {
				Action::Nothing
			}
		}
	}

	let data: Vec<u8> = (0..(100_000)).map(|_| rand::random()).collect();
	let mut handler = ClientHandler(false, data, false);
	let res = client.connect(&mut handler);
	assert!(matches!(
		res,
		Err(Error::Disconnected(callsign)) if callsign == Hostname::new("N0CALL-3").unwrap()
	));

	assert!(handler.0);
	assert!(handler.2);
}

// Warning - very slow(>60 seconds)
#[test]
fn can_establish_connection_and_transfer_data_max_packets() {
	let (s1, s2) = EntangledStream::new_pair();

	let mut server = Server::new_with_stream(
		s1,
		Hostname::new("N0CALL-3").unwrap(),
		Duration::from_secs(1),
		1000,
		3,
		1200,
		|_| LoggerHandler::new(),
	);

	thread::spawn(move || {
		server.serve_forever().unwrap();
	});

	let mut client = Client::new_with_stream(
		s2,
		Hostname::new("DEFGH-212").unwrap(),
		Hostname::new("N0CALL-3").unwrap(),
		Duration::from_secs(1),
		1000,
		1,
		1200,
	);

	struct ClientHandler(bool, Vec<u8>, bool);
	impl Handler for ClientHandler {
		fn recv(&mut self, data: &[u8]) -> Action {
			if data == self.1 {
				self.2 = true;

				Action::Disconnect
			} else {
				Action::Nothing
			}
		}

		fn process(&mut self) -> Action {
			if !self.0 {
				self.0 = true;
				Action::SendData(self.1.clone())
			} else {
				Action::Nothing
			}
		}
	}

	// Can't be random because
	// it needs to compress to a predictable size
	// so that it uses exactly 65536 packets
	let data: Vec<u8> = (0..(979 * 65536 - 3000))
		.map(|x| (calculate_hash(&x)) as u8)
		.collect();
	let mut handler = ClientHandler(false, data, false);
	let res = client.connect(&mut handler);
	assert!(matches!(
		res,
		Err(Error::Disconnected(callsign)) if callsign == Hostname::new("N0CALL-3").unwrap()
	));

	assert!(handler.0);
	assert!(handler.2);
}

#[test]
fn multiple_actions_in_quick_succession() {
	let (s1, s2) = EntangledStream::new_pair();

	let mut server = Server::new_with_stream(
		s1,
		Hostname::new("N0CALL-3").unwrap(),
		Duration::from_secs(1),
		1000,
		3,
		1200,
		|_| LoggerHandler::new(),
	);

	thread::spawn(move || {
		server.serve_forever().unwrap();
	});

	let mut client = Client::new_with_stream(
		s2,
		Hostname::new("DEFGH-212").unwrap(),
		Hostname::new("N0CALL-3").unwrap(),
		Duration::from_secs(1),
		1000,
		1,
		1200,
	);

	struct ClientHandler(i32, Vec<u8>, i32);
	impl Handler for ClientHandler {
		fn recv(&mut self, data: &[u8]) -> Action {
			if data == self.1 {
				self.2 += 1;
			}

			if self.2 == 2 {
				Action::Disconnect
			} else {
				Action::Nothing
			}
		}

		fn process(&mut self) -> Action {
			match self.0 {
				0 => {
					self.0 += 1;
					Action::SendData(self.1.clone())
				}
				1 => {
					self.0 += 1;
					Action::SendData(self.1.clone())
				}
				_ => Action::Nothing,
			}
		}
	}

	let data: Vec<u8> = (0..(100_000)).map(|_| rand::random()).collect();
	let mut handler = ClientHandler(0, data, 0);
	let res = client.connect(&mut handler);
	assert!(matches!(
		res,
		Err(Error::Disconnected(callsign)) if callsign == Hostname::new("N0CALL-3").unwrap()
	));

	assert_eq!(2, handler.0);
	assert_eq!(2, handler.2);
}
