use std::fmt::Display;

const CALLSIGN_LEN: usize = 7;

/// A hostname looks like N0CALL-0, where the final number can be anything from 0 to 255.
/// Default is 0 if not specified(e.g. N0CALL)
/// A valid callsign is between 3 and 7 characters long.
#[derive(Copy, Clone, Debug, PartialEq, Eq, Hash)]
pub struct Hostname {
	callsign: [u8; CALLSIGN_LEN],
	index: u8,
}

impl Hostname {
	pub fn new(from: &str) -> Option<Self> {
		let from = from.to_uppercase();

		let mut splitter = from.splitn(2, '-');
		let call = splitter.next()?;
		let index = splitter
			.next()
			.map(|s| s.parse::<u8>())
			.unwrap_or(Ok(0))
			.ok()?;

		let call = call.as_bytes();
		let len = call.len();
		if len > CALLSIGN_LEN || len == 0 {
			return None;
		}

		let mut callsign = [0; CALLSIGN_LEN];
		callsign[..len].clone_from_slice(&call[..len]);

		Some(Self { callsign, index })
	}

	pub fn to_bytes(&self, buf: &mut Vec<u8>) {
		buf.extend(self.callsign);
		buf.push(self.index);
	}

	pub fn from_bytes(buf: &[u8]) -> Option<Self> {
		if buf.len() != 8 {
			return None;
		}

		let mut callsign = [0; 7];
		callsign.copy_from_slice(&buf[0..7]);
		let index = buf[7];

		Some(Self { callsign, index })
	}
}

impl Display for Hostname {
	fn fmt(&self, ft: &mut std::fmt::Formatter<'_>) -> Result<(), std::fmt::Error> {
		let callsign: String = self.callsign.iter().map(|x| *x as char).collect();
		write!(ft, "{}-{}", callsign, self.index)
	}
}

#[cfg(test)]
mod tests {
	use super::*;

	fn str_to_arr(x: &str) -> [u8; 7] {
		let x = x.as_bytes();
		let mut ret = [0; 7];

		ret[..7].clone_from_slice(&x[..7]);

		ret
	}

	#[test]
	fn valid_hostname_no_idx() {
		let host = Hostname::new("N0CALL");

		assert_eq!(
			Some(Hostname {
				callsign: str_to_arr("N0CALL\0"),
				index: 0
			}),
			host
		);
	}

	#[test]
	fn valid_hostname_idx() {
		let host = Hostname::new("N0CALL-7");

		assert_eq!(
			Some(Hostname {
				callsign: str_to_arr("N0CALL\0"),
				index: 7
			}),
			host
		);
	}

	#[test]
	fn callsign_missing() {
		let host = Hostname::new("-7");

		assert_eq!(None, host);
	}

	#[test]
	fn empty_string() {
		let host = Hostname::new("-7");

		assert_eq!(None, host);
	}

	#[test]
	fn negative_idx() {
		let host = Hostname::new("N0CALL--1");

		assert_eq!(None, host);
	}

	#[test]
	fn idx_max() {
		let host = Hostname::new("N0CALL-255");

		assert_eq!(
			Some(Hostname {
				callsign: str_to_arr("N0CALL\0"),
				index: 255
			}),
			host
		);
	}

	#[test]
	fn idx_min() {
		let host = Hostname::new("N0CALL-0");

		assert_eq!(
			Some(Hostname {
				callsign: str_to_arr("N0CALL\0"),
				index: 0
			}),
			host
		);
	}

	#[test]
	fn idx_too_big() {
		let host = Hostname::new("N0CALL-256");

		assert_eq!(None, host);
	}

	#[test]
	fn seven_char_call() {
		let host = Hostname::new("N0CALL1-7");

		assert_eq!(
			Some(Hostname {
				callsign: str_to_arr("N0CALL1\0"),
				index: 7
			}),
			host
		);
	}

	#[test]
	fn eight_char_call() {
		let host = Hostname::new("N0CALL12-7");

		assert_eq!(None, host);
	}

	#[test]
	fn invalid_idx() {
		let host = Hostname::new("N0CALL-");

		assert_eq!(None, host);

		let host = Hostname::new("N0CALL-H");

		assert_eq!(None, host);
	}

	#[test]
	fn capitalize_callsigns() {
		let host = Hostname::new("n0cAlL");
		let host2 = Hostname::new("N0CALL");

		assert_eq!(host, host2);
	}
}
