use std::{hash::Hash, collections::HashMap};

pub struct Cacher<T,K,V>
    where 
        T: Fn(K) -> V,
        K: Hash + Eq,
        V: Copy,
{
    calculation: T,
    value: HashMap<K,V>,
}

impl<T,K,V> Cacher<T,K,V> 
    where 
        T: Fn(K) -> V,
        K: Hash + Eq + Copy,
        V: Copy
{
    pub fn new(calculation: T) -> Cacher<T,K,V> {
        let value: HashMap<K,V> = HashMap::new();
        Cacher {
            calculation,
            value,
        }
    }

    pub fn value(&mut self, arg: K) -> V {
        // let tmp = arg.clone();
        match self.value.get(&arg) {
            Some(v) => *v,
            None => {
                let v = (self.calculation)(arg);
                self.value.insert(arg, v);
                v
            },
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn call_with_different_values() {
        let mut c = Cacher::new(|a| a);
        let mut c2 = Cacher::new(|a: &str| a);

        let v1 = c.value(1);
        let s = "hello";
        let v2 = c2.value(&s[..]);
        let v3 = c.value(2);

        assert_eq!(v1, 1);
        assert_eq!(v2, "hello");
        assert_eq!(v3, 2);
    }
}

