use std::error::Error;
use std::{fs, env};

pub struct Config {
    pub query: String,
    pub filename: String,
    pub case_sensitive: bool,
}

impl Config {
    pub fn new(mut args: env::Args) -> Result<Config, &'static str> {
        args.next(); // skip program name

        let query = match args.next() {
            Some(value) => value,
            None => return Err("Didn't get a query string")
        };

        let filename = match args.next() {
            Some(value) => value,
            None => return Err("Didn't get a filename string")
        };

        let case_sensitive_as_arg = args.next().is_none();
        let case_sensitive_as_env = env::var("CASE_INSENSITIVE").is_err();

        let case_sensitive = case_sensitive_as_arg && case_sensitive_as_env;

        Ok(Config { query, filename, case_sensitive })
    }
}

pub fn run(config: Config) -> Result<(), Box<dyn Error>> {
    let content = fs::read_to_string(config.filename)?;

    let search_result = if config.case_sensitive {
        search(&config.query, &content)
    } else {
        search_case_insensitive(&config.query, &content)
    };

    for line in search_result {
        println!("{}", line)
    }

    Ok(())
}

fn search<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
    contents
        .lines()
        .filter(|line| line.contains(query))
        .collect()
}

fn search_case_insensitive<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
    let query = query.to_lowercase();
    contents
        .lines()
        .filter(|line| line.to_lowercase().contains(&query))
        .collect()
}

#[cfg(test)]
mod test {
    use super::*;
    use std::env::{Args, ArgsOs};

    #[test]
    fn new_config_ok() {
        // let args = vec!["test".to_string(), "poem.txt".to_string(), "other argument".to_string()];
        //
        // let config_result = Config::new(&args);
        //
        // assert!(config_result.is_ok())
    }

    #[test]
    fn new_config_err() {
        // let args = vec!["test".to_string()];
        //
        // let config_result = Config::new(&args);
        //
        // assert!(config_result.is_err());
    }

    #[test]
    fn run_ok() {
        let config = Config {
            query: "test".to_string(),
            filename: "poem.txt".to_string(),
            case_sensitive: false,
        };

        let run_result = run(config);

        assert!(run_result.is_ok());
    }

    #[test]
    fn run_err() {
        let config = Config {
            query: "test".to_string(),
            filename: "unknown.txt".to_string(),
            case_sensitive: false,
        };

        let run_result = run(config);

        assert!(run_result.is_err());
    }

    #[test]
    fn case_sensitive() {
        let query = "duct";
        let contents = "\
Rust:
safe, fast, productive.
Pick three.
Duct tape.";

        assert_eq!(vec!["safe, fast, productive."], search(query, contents))
    }

    #[test]
    fn case_insensitive() {
        let query = "rUsT";
        let contents = "\
Rust:
safe, fast, productive.
Pick three.
Trust me.";

        assert_eq!(vec!["Rust:", "Trust me."], search_case_insensitive(query, contents))
    }
}
