//! # minigrep
//!
//! `minigrep` is a _very_ basic reimplementation of `grep`.

use std::env;
use std::error::Error;
use std::fs;

pub fn run(config: Config) -> Result<(), Box<dyn Error>> {
    let contents = fs::read_to_string(config.filename)?;
    let results;

    if config.case_sensitive {
        results = search(&config.query, &contents);
    } else {
        results = search_case_insensitive(&config.query, & contents);
    }
    for line in results {
        println!("{}", line);
    }

    Ok(())
}

pub struct Config {
    pub query: String,
    pub filename: String,
    pub case_sensitive: bool,
}

impl Config {
    pub fn new(mut args: env::Args) -> Result<Config, &'static str> {
        args.next(); // skip binary name

        let query = match args.next() {
            Some(arg) => arg,
            None => return Err("No query string"),
        };

        let filename = match args.next(){
            Some(arg) => arg,
            None => return Err("No filename"),
        };

        let case_sensitive = env::var("CASE_INSENSITIVE").is_err();

        Ok(Config { query, filename, case_sensitive })
    }
}
/// Searches for lines containing a query string in a multiline string
///
/// # Examples
///
/// ```
/// let poem = "\
/// Mary had 
/// a little lamb,
/// It's fleece was 
/// white as snow.";
///
/// assert_eq!(
///     vec!["It's fleece was ", "white as snow."],
///     minigrep_oldironhorse::search("as", poem));
/// ```
pub fn search<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
    contents.lines().filter(|line| line.contains(query)).collect()
}

fn search_case_insensitive<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
    let query = query.to_lowercase();
    contents.lines().filter(|line| line.to_lowercase().contains(&query)).collect()
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn one_result() {
        let query = "duct";
        let contents = "
Rust:
safe, fast, productive.
Pick three.
Duct tape.";
        assert_eq!(vec!["safe, fast, productive."], search(query, contents));
    }

    #[test]
    fn case_insensitive() {
        let query = "rUsT";
        let contents = "
Rust:
safe, fast, productive.
Pick three.
Trust me.";
        assert_eq!(vec!["Rust:", "Trust me."], search_case_insensitive(query, contents));
    }
}
