//! # minigrep crate
//!
//! 这是文档注释
//! `minigrep` is a collection of utilities to make performing certain
//! calculations more convenient

use std::{env, fs};
use std::error::Error;

use art::mix;
use art::PrimaryColor;

mod art;
mod closures_learning;
mod iterator_learning;

pub fn run(config: Config) -> Result<(), Box<dyn Error>> {
    let contents = fs::read_to_string(config.file_name)?;
    let results = if config.case_sensitive {
        search(&config.query, &contents)
    } else {
        search_case_insensitive(&config.query, &contents)
    };

    for line in results { println!("{}", line); }

    Ok(())
}

pub struct Config {
    pub query: String,
    pub file_name: String,
    pub case_sensitive: bool,
}

impl Config {
    pub fn new(mut args: env::Args) -> Result<Config, &'static str> {
        if args.len() < 3 {
            return Err("not enough arguments");
        }

        args.next();
        let query = match args.next() {
            Some(arg) => arg,
            None => return Err("Didn't get a query string"),
        };

        let file_name = match args.next() {
            Some(arg) => arg,
            None => return Err("Didn't get a file name"),
        };

        let bash_case_sensitive = env::var("CASE_INSENSITIVE").is_err();

        let case_sensitive = match args.next() {
            Some(arg) => { if arg.to_uppercase() == "INSENSITIVE" { false } else { true } }
            None => bash_case_sensitive,
        };


        Ok(Config { query, file_name, case_sensitive })
    }
}

pub fn search<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
    contents.lines()
        .filter(|line| line.contains(query))
        .collect()
}

pub fn search_case_insensitive<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
    let query = query.to_uppercase();
    // 迭代器是 Rust 的零成本抽象 不会引入运行时开销
    contents.lines()
        .filter(|line| line.to_uppercase().contains(&query))
        .collect()
}

/// 这是方法注释 cargo test可以跑示例
/// Adds one to the number given
///
/// # Examples
/// ```
/// let arg = 5;
/// let answer = minigrep_learning::add_one(arg);
///
/// assert_eq!(6, answer);
/// ```
pub fn add_one(x: i32) -> i32 {
    x + 1
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn one_result() {
        let query = "duct";
        let contents = "Rust:\r\n\
        safe, fast, productive.\r\n\
        Pick three.";

        assert_eq!(vec!["safe, fast, productive."], search(query, contents))
    }

    #[test]
    fn case_sensitive() {
        let query = "duct";
        let contents = "Rust:\r\n\
        safe, fast, productive.\r\n\
        Pick three.\r\n\
        Duct tape.";
        assert_eq!(vec!["safe, fast, productive."], search(query, contents));
    }

    #[test]
    fn case_insensitive() {
        let query = "rUsT";
        let contents = "Rust:\r\n\
        safe, fast, productive.\r\n\
        Pick three.\r\n\
        Trust me.";

        assert_eq!(vec!["Rust:", "Trust me."], search_case_insensitive(query, contents));
    }

    #[test]
    fn crate_doc() {
        let red = PrimaryColor::Red;
        let yellow = PrimaryColor::Yellow;
        mix(red, yellow);
    }
}
