use std::borrow::Borrow;
use std::cmp::Reverse;
use std::collections::{BinaryHeap, HashMap};
use std::ops::Not;

#[allow(unused)]
fn main() {
    let v = vec![1, 2, 3];
    // Rust 的迭代器是惰性的 在使用之前都不会有效果
    let v_iter = v.iter();

    for val in v_iter {
        println!("Got: {}", val);
    }
}

#[derive(PartialEq, Debug)]
struct Shoe {
    size: u32,
    style: String,
}

struct Counter {
    count: u32,
}

impl Counter {
    fn new() -> Counter {
        Counter { count: 0 }
    }
}

impl Iterator for Counter {
    type Item = u32;

    fn next(&mut self) -> Option<Self::Item> {
        self.count += 1;
        if self.count < 6 {
            return Some(self.count);
        }
        None
    }
}

fn shoes_in_my_size(shoes: Vec<Shoe>, shoe_size: u32) -> Vec<Shoe> {
    shoes.into_iter()
        .filter(|x| x.size == shoe_size)
        .collect()
}

#[cfg(test)]
mod iterator_test {
    use crate::iterator_learning::{Counter, Shoe, shoes_in_my_size};

    #[test]
    fn iterator_demonstration() {
        let v1 = vec![1, 2, 3];
        let mut v1_iter = v1.iter();

        assert_eq!(v1_iter.next(), Some(&1));
        assert_eq!(v1_iter.next(), Some(&2));
        assert_eq!(v1_iter.next(), Some(&3));
        assert_eq!(v1_iter.next(), None);
    }

    #[test]
    fn iterator_sum() {
        let v1 = vec![1, 2, 3];
        let mut v1_iter = v1.iter();
        let total: i32 = v1_iter.sum();
        assert_eq!(total, 6);
        // 调用 sum 之后不再允许使用迭代器 因为调用 sum 时 已经获取了所有权
        // assert_eq!(v1_iter.next(), Some(&1));
    }

    #[test]
    fn iterator_adapter() {
        let v1: Vec<i32> = vec![1, 2, 3];
        let v2: Vec<_> = v1.iter().map(|x| x + 1).collect();
        assert_eq!(vec![2, 3, 4], v2);
    }

    #[test]
    fn filters_by_size() {
        let shoes = vec![Shoe { size: 10, style: String::from("sneaker") },
                         Shoe { size: 13, style: String::from("sandal") },
                         Shoe { size: 10, style: String::from("boot") }, ];

        let in_my_size = shoes_in_my_size(shoes, 10);

        assert_eq!(in_my_size, vec![
            Shoe { size: 10, style: String::from("sneaker") },
            Shoe { size: 10, style: String::from("boot") },
        ]);
    }

    #[test]
    fn calling_next_directly() {
        let mut counter = Counter::new();
        assert_eq!(counter.next(), Some(1));
        assert_eq!(counter.next(), Some(2));
        assert_eq!(counter.next(), Some(3));
        assert_eq!(counter.next(), Some(4));
        assert_eq!(counter.next(), Some(5));
        assert_eq!(counter.next(), None);
    }

    #[test]
    fn using_other_iterator_trait_methods() {
        // 1,2,3,4,5
        // 1,2,3,4
        // 任意一个迭代器返回 None 时 就停止遍历 所以序列1中的最后一个元素不会匹配
        let sum: u32 = Counter::new()
            .zip(Counter::new().skip(1))
            .map(|(a, b)| a * b)
            .filter(|x| x % 3 == 0)
            .sum();

        assert_eq!(sum, 18);
    }
}
