use std::collections::HashMap;
use std::hash::Hash;
use std::thread;
use std::time::Duration;

fn simulated_expensive_calculation(intensity: u32) -> u32 {
    println!("calculating slowly...");
    thread::sleep(Duration::from_secs(2));
    intensity
}

fn add_one_v1(x: u32) -> u32 {
    x + 1
}

fn generate_workout(intensity: u32, random_number: u32) {
    let mut expensive_closure = Cacher::new(|num| {
        println!("calculating slowly...");
        thread::sleep(Duration::from_secs(2));
        num
    });

    if intensity < 25 {
        println!("Today, do {} pushups!",
                 expensive_closure.value(intensity));
        println!("Next, do {} situps!",
                 expensive_closure.value(intensity));
    } else {
        if random_number == 3 {
            println!("Take a break today! Remember to stay hydrated!");
        } else {
            println!("Today, run for {} minutes!",
                     expensive_closure.value(intensity));
        }
    }
}

struct Cacher<T, K, V>
    where T: Fn(K) -> V, K: Eq + Hash + Copy, V: Copy {
    calculation: T,
    cache: HashMap<K, V>,
}

impl<T, K, V> Cacher<T, K, V>
    where T: Fn(K) -> V, K: Eq + Hash + Copy, V: Copy {
    fn new(calculation: T) -> Cacher<T, K, V> {
        Cacher { calculation, cache: HashMap::new() }
    }

    fn value(&mut self, arg: K) -> V {
        let value = self.cache.get(&arg);
        match value {
            Some(v) => *v,
            None => {
                let v = (self.calculation)(arg);
                self.cache.insert(arg, v);
                v
            }
        }
    }
}

#[cfg(test)]
mod closures_test {
    use std::num::ParseIntError;

    use super::*;

    #[test]
    fn test_simulated_expensive_calculation() {
        let simulated_user_specified_value = 12;
        let simulated_random_number = 8;

        generate_workout(simulated_user_specified_value, simulated_random_number);
    }

    #[test]
    fn test_closures() {
        // 完整闭包定义
        let add_one_v2 = |x: u32| { x + 1 };
        // 只有一行的闭包 可以省略{}
        let add_one_v3 = |x| x + 1;
        add_one_v3(4);

        let example_closure = |x| x;
        // 闭包的参数 可以进行类型推断 但是如果使用不同类型调用会异常
        let s = example_closure(String::from("hello"));
        // let n = example_closure(5);
    }

    #[test]
    fn call_with_different_values() {
        let mut c = Cacher::new(|a| a);
        // 闭包做值缓存后 后续就不会再调用了
        let v1 = c.value(1);
        let v2 = c.value(2);

        assert_eq!(v2, 2);

        let mut d = Cacher::new(|a: &str| {
            let mut result: usize = 0;
            match a.to_string().parse() {
                Ok(b) => result = b,
                _ => {}
            }
            result
        });
        let v3 = d.value("321");
        assert_eq!(v3, 321);
    }

    #[test]
    fn equal_to_x() {
        let x = 4;
        // 闭包可以捕获环境 访问其被定义的作用域中的变量 函数则不可以
        let equal_to_x = |z| z == x;
        let y = 4;
        assert!(equal_to_x(y));
    }

    #[test]
    fn equal_to_x_move() {
        let x = vec![1, 2, 3];
        // move关键字 可以使闭包或者其使用环境值的所有权
        let equal_to_x = move |z| z == x;
        // 因为 x 已被移动到了闭包 闭包获取了 x 的所有权 所以这里不再能使用
        // println!("can't use x hare: {:?}", x);

        let y = vec![1, 2, 3];
        assert!(equal_to_x(y));
    }
}