use std::error::Error;
use std::fs;
use std::env;

/// 执行逻辑.
/// 
/// 这里是run函数的doc文档说明.
pub fn run(config: Config) -> Result<(), Box<dyn Error>> {
  let contents = fs::read_to_string(config.filename)?;
  
  let results = if config.case_sensitive {
    search(&config.query, &contents)
  } else {
    search_case_insensitive(&config.query, &contents)
  };

  for line in results {
    println!("{}", line);
  }
  Ok(())
}

/// 配置结构体
/// 
/// ```
/// pub struct Config {
///   pub query: String,
///   pub filename: String,
///   pub case_sensitive: bool,
/// }
/// ```
pub struct Config {
  pub query: String,
  pub filename: String,
  pub case_sensitive: bool,
}

// 构造函数
impl Config {
  pub fn new(mut args: std::env::Args) -> Result<Config, &'static str> {
    args.next();

    let query = match args.next() {
      Some(arg) => arg,
      None => return Err("Didn't get a query string"),
    };

    let filename = match args.next() {
      Some(arg) => arg,
      None => return Err("Didn't get a file name"),
    };

    let case_sensitive = env::var("CASE_INSENSITIVE").is_err();

    Ok(Config{query, filename, case_sensitive})
  }
}

#[cfg(test)]
mod tests {
  use super::*;

  #[test]
  fn one_result() {
    let query = "duct";
    let contents = "\
Rust:
safe, fast, productive.
Pick three.";

    assert_eq!(
      vec!["safe, fast, productive."],
      search(query, contents)
    );
  }

  #[test]
  fn case_insensitive() {
    let query = "rUst";
    let contents = "\
Rust:
safe, fast, productive.
Trust me.";

    assert_eq!(
      vec!["Rust:", "Trust me."],
      search_case_insensitive(query, contents)
    );
  }
}

pub fn search<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
  // let mut results = Vec::new();

  // for line in contents.lines() {
  //   if (line.contains(query)) {
  //       results.push(line);
  //   }
  // }
  // results
  contents.lines()
    .filter(|line| line.contains(query))
    .collect()
}

pub fn search_case_insensitive<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
  let query = query.to_lowercase();
  contents.lines()
    .filter(|line| line.to_lowercase().contains(&query))
    .collect()

  // let mut results = Vec::new();

  // for line in contents.lines() {
  //   if line.to_lowercase().contains(&query) {
  //     results.push(line);
  //   }
  // }
  // results
}

