//! # Minigrep
//! A toy command line tool

use std::error::Error;
use std::{env, fs};

pub struct Config {
    pub query: String,
    pub filename: String,
    pub case_sensitive: bool,
}

impl Config {
    pub fn new(mut args: env::Args) -> Result<Config, &'static str> {
        // skip the program name
        args.next();
        // parse query
        let query = match args.next() {
            Some(str) => str,
            None => return Err("Didn't get a query string"),
        };
        let filename = match args.next() {
            Some(str) => str,
            None => return Err("Didn't get a query filename"),
        };
        let case_sensitive = env::var("CASE_SENSITIVE").is_err();
        Ok(Config {
            query,
            filename,
            case_sensitive,
        })
    }
}

pub fn run(config: Config) -> Result<(), Box<dyn Error>> {
    // read file content
    let contents = fs::read_to_string(config.filename)?;
    let results = if config.case_sensitive {
        search(&config.query, &contents)
    } else {
        search_case_sensitive(&config.query, &contents)
    };

    for line in results {
        println!("{}", line);
    }
    Ok(())
}

/// Search given pattern in lines **with case**, return all the matched lines
///
/// # Examples
///
/// ```
/// let query = "foo";
/// let contents = "\
/// Foo
/// foo
/// bar";
/// assert_eq!(minigrep::search(query, contents), vec!["Foo", "foo"])
/// ```
pub fn search_case_sensitive<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
    contents
        .lines()
        .filter(|line| line.contains(query))
        .collect()
}

/// Search given pattern in lines **without case**, return all the matched lines
///
/// # Examples
///
/// ```
/// let query = "foo";
/// let contents = "\
/// Foo
/// foo
/// bar";
/// assert_eq!(minigrep::search_case_sensitive(query, contents), vec!["foo"])
/// ```
pub fn search<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
    contents
        .lines()
        .filter(|line| line.to_lowercase().contains(&query.to_lowercase()))
        .collect()
}

#[cfg(test)]
mod test {
    use super::*;

    #[test]
    fn one_result() {
        let query = "duct";
        let contents = "\
Rust:
safe, fast, productive.
Pick three.
Duct tape.";
        assert_eq!(
            vec!["safe, fast, productive.", "Duct tape."],
            search(query, contents)
        );
    }

    #[test]
    fn case_sensitive() {
        let query = "Rust";
        let contents = "\
Rust:
safe, fast, productive.
Pick three.
Trust me.";
        assert_eq!(vec!["Rust:"], search_case_sensitive(query, contents));
    }
}
