use std::fs;
use std::error::Error;
use std::env;

pub fn run(config: Config) -> Result<(), Box<dyn Error>> {
    let contents = fs::read_to_string(config.filename)?;

    let results = if config.case_sensitive {
        search(&config.query, &contents)
    } else {
        search_case_insensitive(&config.query, &contents)
    };

    for line in results {
        println!("{}", line);
    }

    Ok(())
}

pub struct Config {
    query: String,
    filename: String,
    case_sensitive: bool,
}

impl Config {
    /// Using Iterator<Item = String> as a parameter because we cannot manually create a env::Args, but we know env::Args will yield a list of strings.
    /// # example
    /// ```should_panic
    /// let config = minigrep::Config::new(std::env::args()).unwrap_or_else(|err| {
    ///     eprintln!("Problem parsing arguments: {}", err);
    ///     std::process::exit(1);
    /// });
    /// ```
    pub fn new(mut args: impl Iterator<Item = String>) -> Result<Config, &'static str> {
        args.next();

        let query = match args.next() {
            Some(arg) => arg,
            None => return Err("Didn't get a query string")
        };

        let filename = match args.next() {
            Some(arg) => arg,
            None => return Err("Didn't get a filename")
        };

        let case_sensitive = env::var("CASE_INSENSITIVE").is_err();

        Ok(Config {query, filename, case_sensitive})
    }
}

pub fn search<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
    contents
        .lines()
        .filter(|line| line.contains(query))
        .collect()
}

pub fn search_case_insensitive<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
    let query = query.to_lowercase();

    contents
        .lines()
        .filter(|line| line.to_lowercase().contains(&query))
        .collect()
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_config_new() {
        // iterator of an array returns references. so use vec instead
        if let Err(e) = Config::new(vec!["".to_string(), String::from("test")].into_iter()) {        
            assert_eq!(e, "Didn't get a filename");
        }
        if let Ok(config) = Config::new(vec!["".to_string(), String::from("test_query"), String::from("test_name")].into_iter()) {
            assert_eq!(config.query, "test_query");
            assert_eq!(config.filename, "test_name");
        }
    }

    fn contents() -> &'static str {
        "\
Rust:
safe, fast, productive.
Pick three.
Trust me."
    }

    #[test]
    fn test_search() {
        assert_eq!(vec!["safe, fast, productive."], search("duct", contents()));
        assert_eq!(vec!["Trust me."], search("rust", contents()))
    }

    #[test]
    fn incentive_search() {
        let query = "rUsT";
        assert_eq!(vec!["Rust:", "Trust me."], search_case_insensitive(query, contents()));
    }
}
