
use lazy_static::lazy_static;
use std::collections::HashMap;
use crate::common::spec::tag::ns::Namespace;

pub struct AttributeMinification {
    pub boolean: bool,
    pub case_insensitive: bool,
    pub collapse: bool,
    pub default_value: Option<&'static [u8]>,
    pub redundant_if_empty: bool,
    pub trim: bool,
}

pub enum AttrMapEntry {
    AllNamespaceElements(AttributeMinification),
    SpecificNamespaceElements(HashMap<&'static [u8], AttributeMinification>),
}

pub struct ByNamespace {
    // Make pub so this struct can be statically created in gen/attrs.rs.
    pub html: Option<AttrMapEntry>,
    pub svg: Option<AttrMapEntry>,
}

impl ByNamespace {
    fn get(&self, ns: Namespace) -> Option<&AttrMapEntry> {
        match ns {
            Namespace::Html => self.html.as_ref(),
            Namespace::Svg => self.svg.as_ref(),
        }
    }
}

pub struct AttrMap(HashMap<&'static [u8], ByNamespace>);

impl AttrMap {
    pub const fn new(map: HashMap<&'static [u8], ByNamespace>) -> AttrMap {
        AttrMap(map)
    }

    pub fn get(&self, ns: Namespace, tag: &[u8], attr: &[u8]) -> Option<&AttributeMinification> {
        self.0.get(attr).and_then(|namespaces| namespaces.get(ns)).and_then(|entry| match entry {
            AttrMapEntry::AllNamespaceElements(min) => Some(min),
            AttrMapEntry::SpecificNamespaceElements(map) => map.get(tag),
        })
    }
}


lazy_static! {
  pub static ref ATTRS: AttrMap = {
    let mut m = HashMap::<&'static [u8], ByNamespace>::new();
    m.insert(b"alt", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
    });

    m.insert(b"accesskey", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"class", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: true, 
    default_value: None,
    redundant_if_empty: true, 
    trim: true, 
}
)),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"contenteditable", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"contextmenu", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"dir", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"draggable", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"hidden", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"id", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"lang", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"placeholder", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"slot", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"spellcheck", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"style", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: true, 
}
)),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: true, 
}
)),
    });

    m.insert(b"tabindex", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"title", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"translate", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"radiogroup", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"role", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
    });

    m.insert(b"about", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"datatype", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"inlist", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"prefix", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"property", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"resource", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"typeof", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"vocab", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"autocapitalize", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"autocorrect", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"autosave", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"color", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"itemprop", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"itemscope", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"itemtype", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"itemid", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"itemref", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"results", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"security", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"unselectable", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"inputmode", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"is", ByNamespace {
      html: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"download", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"a", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"area", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"href", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"a", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"area", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"base", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"link", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"hreflang", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"a", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"area", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"link", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"media", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"a", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"area", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"link", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"meta", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"source", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"style", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: Some(b"all"),
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"ping", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"a", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"rel", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"a", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"area", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"link", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"target", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"a", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: Some(b"_self"),
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"area", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"base", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"form", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: Some(b"_self"),
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"type", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"a", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"button", 
AttributeMinification {
    boolean: false,
    case_insensitive: true, 
    collapse: false, 
    default_value: Some(b"submit"),
    redundant_if_empty: false, 
    trim: true, 
}
);
        m.insert(b"embed", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"input", 
AttributeMinification {
    boolean: false,
    case_insensitive: true, 
    collapse: false, 
    default_value: Some(b"text"),
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"link", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: Some(b"text/css"),
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"menu", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"object", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"ol", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"script", 
AttributeMinification {
    boolean: false,
    case_insensitive: true, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: true, 
}
);
        m.insert(b"source", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"style", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: Some(b"text/css"),
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"referrerpolicy", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"a", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"area", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"iframe", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: Some(b"no-referrer-when-downgrade"),
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"img", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: Some(b"no-referrer-when-downgrade"),
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"link", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"script", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"coords", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"area", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"shape", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"area", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: Some(b"rect"),
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"cite", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"blockquote", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"del", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"ins", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"quote", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"autofocus", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"button", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"input", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"keygen", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"select", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"textarea", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"disabled", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"button", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"fieldset", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"input", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"keygen", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"optgroup", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"option", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"select", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"textarea", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"form", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"button", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"fieldset", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"input", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"keygen", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"label", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"meter", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"object", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"output", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"select", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"textarea", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"formaction", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"button", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"input", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"formenctype", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"button", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: Some(b"application/x-www-form-urlencoded"),
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"input", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: Some(b"application/x-www-form-urlencoded"),
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"formmethod", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"button", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"input", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"formnovalidate", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"button", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"input", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"formtarget", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"button", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"input", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"name", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"button", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"fieldset", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"form", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"iframe", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"input", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"keygen", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"map", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"meta", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"object", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"output", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"param", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"select", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"slot", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"textarea", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"value", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"button", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"data", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"input", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"li", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"meter", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"option", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"param", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"progress", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"select", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"textarea", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"height", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"canvas", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"embed", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"iframe", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: Some(b"150"),
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"img", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"input", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"object", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"td", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"video", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"width", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"canvas", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"col", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"embed", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"iframe", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: Some(b"300"),
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"img", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"input", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"object", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"table", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"td", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"video", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"span", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"col", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: Some(b"1"),
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"colgroup", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: Some(b"1"),
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"datetime", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"del", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"ins", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"time", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"open", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"details", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"dialog", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"ontoggle", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"details", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"src", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"embed", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"iframe", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"img", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"input", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"media", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"script", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"source", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"track", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"acceptcharset", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"form", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"action", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"form", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"autocomplete", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"form", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"input", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"select", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"textarea", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"enctype", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"form", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: Some(b"application/x-www-form-urlencoded"),
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"method", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"form", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: Some(b"get"),
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"novalidate", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"form", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"manifest", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"html", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"allow", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"iframe", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"allowfullscreen", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"iframe", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"allowtransparency", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"iframe", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"frameborder", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"iframe", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: Some(b"1"),
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"loading", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"iframe", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: Some(b"eager"),
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"img", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: Some(b"eager"),
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"marginheight", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"iframe", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"marginwidth", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"iframe", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"sandbox", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"iframe", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"scrolling", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"iframe", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"seamless", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"iframe", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"srcdoc", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"iframe", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"crossorigin", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"img", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"input", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"link", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"media", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"script", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
    });

    m.insert(b"decoding", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"img", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: Some(b"auto"),
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"sizes", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"img", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"link", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"source", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"srcset", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"img", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"source", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"usemap", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"img", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"object", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"accept", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"input", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"capture", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"input", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"checked", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"input", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"enterkeyhint", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"input", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"list", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"input", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"max", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"input", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"meter", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"progress", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"maxlength", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"input", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"textarea", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"min", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"input", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"meter", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"minlength", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"input", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"textarea", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"multiple", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"input", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"select", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"pattern", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"input", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"readonly", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"input", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"textarea", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"required", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"input", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"select", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"textarea", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"size", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"input", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"select", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"step", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"input", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"onchange", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"input", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"select", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"textarea", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"challenge", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"keygen", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"keytype", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"keygen", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"keyparams", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"keygen", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"htmlfor", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"label", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"output", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"as", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"link", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"integrity", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"link", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"script", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"charset", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"link", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"meta", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"script", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"autoplay", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"media", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"controls", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"media", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"controlslist", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"media", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"loop", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"media", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"mediagroup", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"media", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"muted", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"media", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"playsinline", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"media", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"video", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"preload", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"media", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"content", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"meta", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"httpequiv", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"meta", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"high", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"meter", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"low", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"meter", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"optimum", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"meter", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"classid", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"object", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"data", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"object", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"wmode", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"object", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"reversed", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"ol", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"start", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"ol", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"label", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"optgroup", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"option", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"track", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"selected", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"option", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"async", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"script", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"defer", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"script", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"nomodule", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"script", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"nonce", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"script", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"style", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"scoped", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"style", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"cellpadding", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"table", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"cellspacing", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"table", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"summary", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"table", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"align", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"td", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        m.insert(b"th", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"colspan", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"td", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"th", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"headers", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"td", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"th", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"rowspan", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"td", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"th", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"scope", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"td", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"th", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"abbr", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"td", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        m.insert(b"th", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"valign", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"td", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"cols", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"textarea", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"dirname", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"textarea", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"rows", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"textarea", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"wrap", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"textarea", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"default", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"track", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"kind", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"track", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"srclang", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"track", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"poster", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"video", 
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"disablepictureinpicture", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"video", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"disableremoteplayback", ByNamespace {
      html: Some({
        let mut m = HashMap::<&'static [u8], AttributeMinification>::new();
        m.insert(b"video", 
AttributeMinification {
    boolean: true,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
);
        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
    });

    m.insert(b"accentheight", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"accumulate", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
    });

    m.insert(b"additive", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
    });

    m.insert(b"alignmentbaseline", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
    });

    m.insert(b"allowreorder", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
    });

    m.insert(b"alphabetic", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"amplitude", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"arabicform", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
    });

    m.insert(b"ascent", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"attributename", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"attributetype", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"autoreverse", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
    });

    m.insert(b"azimuth", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"basefrequency", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"baselineshift", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"baseprofile", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"bbox", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"begin", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"bias", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"by", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"calcmode", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"capheight", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"clip", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"clippath", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"clippathunits", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"cliprule", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"colorinterpolation", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"colorinterpolationfilters", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
    });

    m.insert(b"colorprofile", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"colorrendering", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"contentscripttype", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"contentstyletype", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"cursor", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"cx", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"cy", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"d", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: true, 
    default_value: None,
    redundant_if_empty: true, 
    trim: true, 
}
)),
    });

    m.insert(b"decelerate", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"descent", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"diffuseconstant", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"direction", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"display", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"divisor", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"dominantbaseline", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"dur", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"dx", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"dy", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"edgemode", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"elevation", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"enablebackground", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"end", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"exponent", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"externalresourcesrequired", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
    });

    m.insert(b"fill", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"fillopacity", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"fillrule", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
    });

    m.insert(b"filter", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"filterres", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"filterunits", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"floodcolor", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"floodopacity", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"focusable", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
    });

    m.insert(b"fontfamily", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"fontsize", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"fontsizeadjust", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"fontstretch", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"fontstyle", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"fontvariant", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"fontweight", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"format", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"from", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"fx", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"fy", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"g1", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"g2", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"glyphname", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"glyphorientationhorizontal", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"glyphorientationvertical", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"glyphref", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"gradienttransform", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"gradientunits", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"hanging", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"horizadvx", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"horizoriginx", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"ideographic", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"imagerendering", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"in2", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"in", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"intercept", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"k1", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"k2", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"k3", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"k4", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"k", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"kernelmatrix", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"kernelunitlength", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"kerning", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"keypoints", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"keysplines", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"keytimes", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"lengthadjust", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"letterspacing", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"lightingcolor", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"limitingconeangle", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"local", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"markerend", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"markerheight", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"markermid", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"markerstart", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"markerunits", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"markerwidth", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"mask", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"maskcontentunits", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"maskunits", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"mathematical", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"mode", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"numoctaves", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"offset", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"opacity", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"operator", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"order", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"orient", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"orientation", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"origin", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"overflow", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"overlineposition", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"overlinethickness", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"paintorder", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"panose1", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"path", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"pathlength", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"patterncontentunits", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"patterntransform", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"patternunits", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"pointerevents", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"points", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"pointsatx", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"pointsaty", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"pointsatz", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"preservealpha", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
    });

    m.insert(b"preserveaspectratio", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"primitiveunits", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"r", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"radius", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"refx", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"refy", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"renderingintent", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"repeatcount", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"repeatdur", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"requiredextensions", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"requiredfeatures", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"restart", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"result", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"rotate", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"rx", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"ry", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"scale", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"seed", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"shaperendering", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"slope", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"spacing", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"specularconstant", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"specularexponent", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"speed", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"spreadmethod", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"startoffset", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"stddeviation", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"stemh", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"stemv", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"stitchtiles", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"stopcolor", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"stopopacity", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"strikethroughposition", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"strikethroughthickness", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"string", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"stroke", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"strokedasharray", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"strokedashoffset", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"strokelinecap", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
    });

    m.insert(b"strokelinejoin", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: false, 
    trim: false, 
}
)),
    });

    m.insert(b"strokemiterlimit", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"strokeopacity", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"strokewidth", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"surfacescale", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"systemlanguage", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"tablevalues", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"targetx", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"targety", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"textanchor", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"textdecoration", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"textlength", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"textrendering", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"to", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"transform", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"u1", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"u2", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"underlineposition", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"underlinethickness", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"unicode", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"unicodebidi", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"unicoderange", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"unitsperem", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"valphabetic", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"values", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"vectoreffect", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"version", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"vertadvy", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"vertoriginx", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"vertoriginy", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"vhanging", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"videographic", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"viewbox", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"viewtarget", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"visibility", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"vmathematical", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"widths", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"wordspacing", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"writingmode", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"x1", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"x2", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"x", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"xchannelselector", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"xheight", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"xlinkactuate", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"xlinkarcrole", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"xlinkhref", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"xlinkrole", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"xlinkshow", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"xlinktitle", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"xlinktype", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"xmlbase", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"xmllang", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"xmlns", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"xmlnsxlink", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"xmlspace", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"y1", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"y2", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"y", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"ychannelselector", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"z", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });

    m.insert(b"zoomandpan", ByNamespace {
      html: Some({
        let  m = HashMap::<&'static [u8], AttributeMinification>::new();

        AttrMapEntry::SpecificNamespaceElements(m)
      }),
      svg: Some(AttrMapEntry::AllNamespaceElements(
AttributeMinification {
    boolean: false,
    case_insensitive: false, 
    collapse: false, 
    default_value: None,
    redundant_if_empty: true, 
    trim: false, 
}
)),
    });


    AttrMap::new(m)
  };
}