//! Rust log-compatible pretty/ndjson logger, based on femme
//!
//! ## Examples
//! ```
//! minifemme::start(minifemme::LevelFilter::Info, minifemme::LogMode::Pretty);
//! log::warn!("Unauthorized access attempt on /login");
//! log::info!("Listening on port 8080");
//! ```

pub use log::LevelFilter;

#[cfg(not(target_arch = "wasm32"))]
mod ndjson;

#[cfg(not(target_arch = "wasm32"))]
mod pretty;

#[cfg(target_arch = "wasm32")]
mod wasm;

/// Selects between ndjson-based logging or pretty-printed
/// logging.
pub enum LogMode {
    /// Log as NdJson
    NdJson,
    /// Log in a nice, pleasing format
    Pretty,
}

/// Starts logging with a log level and log mode.
/// All messages under the specified log level will statically be filtered out.
/// The logger will use the mode specified. This can't be changed later.
/// (Except if the program is running in a WASM environment.)
///
/// ```
/// minifemme::start(minifemme::LevelFilter::Info, minifemme::LogMode::Pretty);
/// log::warn!("Unauthorized access attempt on /login");
/// log::info!("Listening on port 8080");
/// ```
pub fn start(level: LevelFilter, mode: LogMode) {
    #[cfg(target_arch = "wasm32")]
    wasm::start(level);

    #[cfg(not(target_arch = "wasm32"))]
    match mode {
        LogMode::NdJson => ndjson::start(level),
        LogMode::Pretty => pretty::start(level),
    }
}
