//! Contains a type to read a world's map data
use futures::stream::TryStreamExt;
#[cfg(feature = "redis")]
use redis::{aio::MultiplexedConnection as RedisConn, AsyncCommands};
use sqlx::prelude::*;
use sqlx::sqlite::{SqliteConnectOptions, SqlitePool};
use std::path::Path;
use url::Host;

#[cfg(feature = "smartstring")]
use smartstring::alias::String;

use crate::map_block::{MapBlock, MapBlockError, Node, NodeIter};
use crate::positions::{get_block_as_integer, get_integer_as_block, Position};

/// An error in the underlying database or in the map block binary format
#[derive(thiserror::Error, Debug)]
pub enum MapDataError {
    #[cfg(any(feature = "sqlite", feature = "postgres"))]
    #[error("Database error: {0}")]
    /// sqlx based error. This covers Sqlite and Postgres errors.
    SqlError(#[from] sqlx::Error),
    #[cfg(feature = "redis")]
    #[error("Database error: {0}")]
    /// Redis connection error
    RedisError(#[from] redis::RedisError),
    #[error("MapBlockError: {0}")]
    /// Error while reading a map block
    MapBlockError(#[from] MapBlockError),
}

/// A handle to the world data
/// 
/// Can be used to query MapBlocks and nodes.
pub enum MapData {
    #[cfg(any(feature = "sqlite", feature = "postgres"))]
    /// This variant covers the SQLite and PostgreSQL database backends
    Sqlite(SqlitePool),
    #[cfg(feature = "redis")]
    /// This variant supports Redis as database backend
    Redis {
        /// The connection to the Redis instance
        connection: RedisConn,
        /// The hash in which the world's data is stored in
        hash: String,
    },
}

impl MapData {
    #[cfg(feature = "sqlite")]
    /// Connects to the "map.sqlite" database.
    ///
    /// ```
    /// use minetestworld::MapData;
    /// use async_std::task;
    ///
    /// let meta = task::block_on(async {
    ///     MapData::from_sqlite_file("TestWorld/map.sqlite").await.unwrap();
    /// });
    /// ```
    pub async fn from_sqlite_file<P: AsRef<Path>>(filename: P) -> Result<MapData, MapDataError> {
        Ok(MapData::Sqlite(
            SqlitePool::connect_with(
                SqliteConnectOptions::new()
                    .immutable(true)
                    .filename(filename),
            )
            .await?,
        ))
    }

    #[cfg(feature = "redis")]
    /// Connects to a Redis server given the connection parameters
    pub async fn from_redis_connection_params(
        host: Host,
        port: Option<u16>,
        hash: String,
    ) -> Result<MapData, MapDataError> {
        Ok(MapData::Redis {
            connection: redis::Client::open(format!(
                "redis://{host}{}/",
                port.map(|p| format!(":{p}")).unwrap_or_default()
            ))?
            .get_multiplexed_async_std_connection()
            .await?,
            hash,
        })
    }

    /// Returns the positions of all mapblocks
    ///
    /// Note that the unit of the coordinates will be
    /// [MAPBLOCK_LENGTH][`crate::map_block::MAPBLOCK_LENGTH`].
    pub async fn all_mapblock_positions(&self) -> Result<Vec<Position>, MapDataError> {
        match self {
            #[cfg(feature = "sqlite")]
            MapData::Sqlite(pool) => {
                let mut result = vec![];
                let mut rows = sqlx::query("SELECT pos FROM blocks")
                    .bind("pos")
                    .fetch(pool);
                while let Some(row) = rows.try_next().await? {
                    let pos_index = row.try_get("pos")?;
                    result.push(get_integer_as_block(pos_index));
                }
                Ok(result)
            }
            #[cfg(feature = "redis")]
            MapData::Redis { connection, hash } => {
                let mut v: Vec<i64> = connection.clone().hkeys(hash.to_string()).await?;
                Ok(v.drain(..).map(get_integer_as_block).collect())
            }
        }
    }

    /// Queries the backend for the data of a single mapblock
    pub async fn get_block_data(&self, pos: Position) -> Result<Vec<u8>, MapDataError> {
        let pos = get_block_as_integer(pos);
        match self {
            #[cfg(feature = "sqlite")]
            MapData::Sqlite(pool) => Ok(sqlx::query("SELECT data FROM blocks WHERE pos = ?")
                .bind(pos)
                .fetch_one(pool)
                .await?
                .try_get("data")?),
            #[cfg(feature = "redis")]
            MapData::Redis { connection, hash } => {
                Ok(connection.clone().hget(hash.to_string(), pos).await?)
            }
        }
    }

    /// Queries the backend for a specific map block
    /// 
    /// `pos` is a map block position.
    pub async fn get_mapblock(&self, pos: Position) -> Result<MapBlock, MapDataError> {
        Ok(MapBlock::from_data(
            self.get_block_data(pos).await?.as_slice(),
        )?)
    }

    /// Enumerate all nodes from the mapblock at `pos`
    /// 
    /// Returns all nodes along with their relative position within the map block
    pub async fn iter_mapblock_nodes(
        &self,
        mapblock_pos: Position,
    ) -> Result<impl Iterator<Item = (Position, Node)>, MapDataError> {
        let data = self.get_block_data(mapblock_pos).await?;
        let mapblock = MapBlock::from_data(data.as_slice())?;
        Ok(NodeIter::new(mapblock, mapblock_pos))
    }
}
