pub use self::types::*;
pub use self::enumerations::*;
pub use self::functions::*;

use std::os::raw::c_void;


#[derive(Copy, Clone)]
struct FnPtr {
    ptr: *const c_void,
    is_loaded: bool
}

#[allow(dead_code)]
impl FnPtr {
    fn new(ptr: *const c_void) -> FnPtr {
        if !ptr.is_null() {
            FnPtr { ptr, is_loaded: true }
        } else {
            FnPtr { ptr: FnPtr::not_initialized as *const c_void, is_loaded: false }
        }
    }

    fn set_ptr(&mut self, ptr: *const c_void) {
        *self = Self::new(ptr);
    }

    fn aliased(&mut self, other: &FnPtr) {
        if !self.is_loaded && other.is_loaded {
            *self = *other;
        }
    }

    #[inline(never)]
    fn not_initialized() -> ! { panic!("gl: function not initialized") }
}

unsafe impl Sync for FnPtr {}
unsafe impl Send for FnPtr {}

pub mod types {
#![allow(dead_code, non_snake_case, non_camel_case_types)]

use std::os::raw;

pub type GLvoid = raw::c_void;

pub type GLbyte = raw::c_char;
pub type GLubyte = raw::c_uchar;
pub type GLchar = raw::c_char;
pub type GLboolean = raw::c_uchar;

pub type GLshort = raw::c_short;
pub type GLushort = raw::c_ushort;

pub type GLint = raw::c_int;
pub type GLuint = raw::c_uint;
pub type GLint64 = i64;
pub type GLuint64 = u64;

pub type GLintptr = isize;
pub type GLsizeiptr = isize;
pub type GLintptrARB = isize;
pub type GLsizeiptrARB = isize;
pub type GLint64EXT = i64;
pub type GLuint64EXT = u64;

pub type GLsizei = GLint;
pub type GLclampx = raw::c_int;
pub type GLfixed = GLint;
pub type GLhalf = raw::c_ushort;
pub type GLhalfNV = raw::c_ushort;
pub type GLhalfARB = raw::c_ushort;

pub type GLenum = raw::c_uint;
pub type GLbitfield = raw::c_uint;

pub type GLfloat = raw::c_float;
pub type GLdouble = raw::c_double;
pub type GLclampf = raw::c_float;
pub type GLclampd = raw::c_double;

pub type GLcharARB = raw::c_char;

#[cfg(target_os = "macos")]
pub type GLhandleARB = *const raw::c_void;
#[cfg(not(target_os = "macos"))]
pub type GLhandleARB = raw::c_uint;

pub enum __GLsync {}

pub type GLsync = *const __GLsync;

pub enum _cl_context {}

pub enum _cl_event {}

pub type GLvdpauSurfaceNV = GLintptr;
pub type GLeglClientBufferEXT = *const raw::c_void;
pub type GLeglImageOES = *const raw::c_void;


pub type GLDEBUGPROC = extern "system" fn (
    source: GLenum,
    type_: GLenum,
    id: GLuint,
    severity: GLenum,
    length: GLsizei,
    message: *const GLchar,
    userParam: *mut raw::c_void,
);
pub type GLDEBUGPROCARB = extern "system" fn (
    source: GLenum,
    type_: GLenum,
    id: GLuint,
    severity: GLenum,
    length: GLsizei,
    message: *const GLchar,
    userParam: *mut raw::c_void,
);
pub type GLDEBUGPROCKHR = extern "system" fn (
    source: GLenum,
    type_: GLenum,
    id: GLuint,
    severity: GLenum,
    length: GLsizei,
    message: *const GLchar,
    userParam: *mut GLvoid,
);
pub type GLDEBUGPROCAMD = extern "system" fn (
    id: GLuint,
    category: GLenum,
    severity: GLenum,
    length: GLsizei,
    message: *const GLchar,
    userParam: *mut GLvoid,
);
pub type GLVULKANPROCNV = extern "system" fn ();
}

pub mod enumerations {
    #![allow(dead_code, non_upper_case_globals, unused_imports)]

    use std::os::raw::*;
    use super::types::*;

    pub const _1PASS_EXT: c_uint = 0x80A1;
    pub const _1PASS_SGIS: c_uint = 0x80A1;
    pub const _2PASS_0_EXT: c_uint = 0x80A2;
    pub const _2PASS_0_SGIS: c_uint = 0x80A2;
    pub const _2PASS_1_EXT: c_uint = 0x80A3;
    pub const _2PASS_1_SGIS: c_uint = 0x80A3;
    pub const _2X_BIT_ATI: c_uint = 0x00000001;
    pub const _422_AVERAGE_EXT: c_uint = 0x80CE;
    pub const _422_EXT: c_uint = 0x80CC;
    pub const _422_REV_AVERAGE_EXT: c_uint = 0x80CF;
    pub const _422_REV_EXT: c_uint = 0x80CD;
    pub const _4PASS_0_EXT: c_uint = 0x80A4;
    pub const _4PASS_0_SGIS: c_uint = 0x80A4;
    pub const _4PASS_1_EXT: c_uint = 0x80A5;
    pub const _4PASS_1_SGIS: c_uint = 0x80A5;
    pub const _4PASS_2_EXT: c_uint = 0x80A6;
    pub const _4PASS_2_SGIS: c_uint = 0x80A6;
    pub const _4PASS_3_EXT: c_uint = 0x80A7;
    pub const _4PASS_3_SGIS: c_uint = 0x80A7;
    pub const _4X_BIT_ATI: c_uint = 0x00000002;
    pub const _8X_BIT_ATI: c_uint = 0x00000004;
    pub const ABGR_EXT: c_uint = 0x8000;
    pub const ACCUM_ADJACENT_PAIRS_NV: c_uint = 0x90AD;
    pub const ACTIVE_ATOMIC_COUNTER_BUFFERS: c_uint = 0x92D9;
    pub const ACTIVE_ATTRIBUTES: c_uint = 0x8B89;
    pub const ACTIVE_ATTRIBUTE_MAX_LENGTH: c_uint = 0x8B8A;
    pub const ACTIVE_PROGRAM: c_uint = 0x8259;
    pub const ACTIVE_PROGRAM_EXT: c_uint = 0x8B8D;
    pub const ACTIVE_RESOURCES: c_uint = 0x92F5;
    pub const ACTIVE_STENCIL_FACE_EXT: c_uint = 0x8911;
    pub const ACTIVE_SUBROUTINES: c_uint = 0x8DE5;
    pub const ACTIVE_SUBROUTINE_MAX_LENGTH: c_uint = 0x8E48;
    pub const ACTIVE_SUBROUTINE_UNIFORMS: c_uint = 0x8DE6;
    pub const ACTIVE_SUBROUTINE_UNIFORM_LOCATIONS: c_uint = 0x8E47;
    pub const ACTIVE_SUBROUTINE_UNIFORM_MAX_LENGTH: c_uint = 0x8E49;
    pub const ACTIVE_TEXTURE: c_uint = 0x84E0;
    pub const ACTIVE_TEXTURE_ARB: c_uint = 0x84E0;
    pub const ACTIVE_UNIFORMS: c_uint = 0x8B86;
    pub const ACTIVE_UNIFORM_BLOCKS: c_uint = 0x8A36;
    pub const ACTIVE_UNIFORM_BLOCK_MAX_NAME_LENGTH: c_uint = 0x8A35;
    pub const ACTIVE_UNIFORM_MAX_LENGTH: c_uint = 0x8B87;
    pub const ACTIVE_VARIABLES: c_uint = 0x9305;
    pub const ACTIVE_VARYINGS_NV: c_uint = 0x8C81;
    pub const ACTIVE_VARYING_MAX_LENGTH_NV: c_uint = 0x8C82;
    pub const ACTIVE_VERTEX_UNITS_ARB: c_uint = 0x86A5;
    pub const ADD_ATI: c_uint = 0x8963;
    pub const ADD_SIGNED_ARB: c_uint = 0x8574;
    pub const ADD_SIGNED_EXT: c_uint = 0x8574;
    pub const ADJACENT_PAIRS_NV: c_uint = 0x90AE;
    pub const AFFINE_2D_NV: c_uint = 0x9092;
    pub const AFFINE_3D_NV: c_uint = 0x9094;
    pub const ALIASED_LINE_WIDTH_RANGE: c_uint = 0x846E;
    pub const ALLOW_DRAW_FRG_HINT_PGI: c_uint = 0x1A210;
    pub const ALLOW_DRAW_MEM_HINT_PGI: c_uint = 0x1A211;
    pub const ALLOW_DRAW_OBJ_HINT_PGI: c_uint = 0x1A20E;
    pub const ALLOW_DRAW_WIN_HINT_PGI: c_uint = 0x1A20F;
    pub const ALL_BARRIER_BITS: c_uint = 0xFFFFFFFF;
    pub const ALL_BARRIER_BITS_EXT: c_uint = 0xFFFFFFFF;
    pub const ALL_COMPLETED_NV: c_uint = 0x84F2;
    pub const ALL_PIXELS_AMD: c_uint = 0xFFFFFFFF;
    pub const ALL_SHADER_BITS: c_uint = 0xFFFFFFFF;
    pub const ALL_STATIC_DATA_IBM: c_uint = 103060;
    pub const ALPHA: c_uint = 0x1906;
    pub const ALPHA12_EXT: c_uint = 0x803D;
    pub const ALPHA16F_ARB: c_uint = 0x881C;
    pub const ALPHA16F_EXT: c_uint = 0x881C;
    pub const ALPHA16I_EXT: c_uint = 0x8D8A;
    pub const ALPHA16UI_EXT: c_uint = 0x8D78;
    pub const ALPHA16_EXT: c_uint = 0x803E;
    pub const ALPHA16_SNORM: c_uint = 0x9018;
    pub const ALPHA32F_ARB: c_uint = 0x8816;
    pub const ALPHA32F_EXT: c_uint = 0x8816;
    pub const ALPHA32I_EXT: c_uint = 0x8D84;
    pub const ALPHA32UI_EXT: c_uint = 0x8D72;
    pub const ALPHA4_EXT: c_uint = 0x803B;
    pub const ALPHA8I_EXT: c_uint = 0x8D90;
    pub const ALPHA8UI_EXT: c_uint = 0x8D7E;
    pub const ALPHA8_EXT: c_uint = 0x803C;
    pub const ALPHA8_SNORM: c_uint = 0x9014;
    pub const ALPHA_FLOAT16_APPLE: c_uint = 0x881C;
    pub const ALPHA_FLOAT16_ATI: c_uint = 0x881C;
    pub const ALPHA_FLOAT32_APPLE: c_uint = 0x8816;
    pub const ALPHA_FLOAT32_ATI: c_uint = 0x8816;
    pub const ALPHA_INTEGER_EXT: c_uint = 0x8D97;
    pub const ALPHA_MAX_CLAMP_INGR: c_uint = 0x8567;
    pub const ALPHA_MAX_SGIX: c_uint = 0x8321;
    pub const ALPHA_MIN_CLAMP_INGR: c_uint = 0x8563;
    pub const ALPHA_MIN_SGIX: c_uint = 0x8320;
    pub const ALPHA_REF_COMMAND_NV: c_uint = 0x000F;
    pub const ALPHA_SNORM: c_uint = 0x9010;
    pub const ALPHA_TO_COVERAGE_DITHER_DEFAULT_NV: c_uint = 0x934D;
    pub const ALPHA_TO_COVERAGE_DITHER_DISABLE_NV: c_uint = 0x934F;
    pub const ALPHA_TO_COVERAGE_DITHER_ENABLE_NV: c_uint = 0x934E;
    pub const ALPHA_TO_COVERAGE_DITHER_MODE_NV: c_uint = 0x92BF;
    pub const ALREADY_SIGNALED: c_uint = 0x911A;
    pub const ALWAYS: c_uint = 0x0207;
    pub const ALWAYS_FAST_HINT_PGI: c_uint = 0x1A20C;
    pub const ALWAYS_SOFT_HINT_PGI: c_uint = 0x1A20D;
    pub const AND: c_uint = 0x1501;
    pub const AND_INVERTED: c_uint = 0x1504;
    pub const AND_REVERSE: c_uint = 0x1502;
    pub const ANY_SAMPLES_PASSED: c_uint = 0x8C2F;
    pub const ANY_SAMPLES_PASSED_CONSERVATIVE: c_uint = 0x8D6A;
    pub const ARC_TO_NV: c_uint = 0xFE;
    pub const ARRAY_BUFFER: c_uint = 0x8892;
    pub const ARRAY_BUFFER_ARB: c_uint = 0x8892;
    pub const ARRAY_BUFFER_BINDING: c_uint = 0x8894;
    pub const ARRAY_BUFFER_BINDING_ARB: c_uint = 0x8894;
    pub const ARRAY_ELEMENT_LOCK_COUNT_EXT: c_uint = 0x81A9;
    pub const ARRAY_ELEMENT_LOCK_FIRST_EXT: c_uint = 0x81A8;
    pub const ARRAY_OBJECT_BUFFER_ATI: c_uint = 0x8766;
    pub const ARRAY_OBJECT_OFFSET_ATI: c_uint = 0x8767;
    pub const ARRAY_SIZE: c_uint = 0x92FB;
    pub const ARRAY_STRIDE: c_uint = 0x92FE;
    pub const ASYNC_DRAW_PIXELS_SGIX: c_uint = 0x835D;
    pub const ASYNC_HISTOGRAM_SGIX: c_uint = 0x832C;
    pub const ASYNC_MARKER_SGIX: c_uint = 0x8329;
    pub const ASYNC_READ_PIXELS_SGIX: c_uint = 0x835E;
    pub const ASYNC_TEX_IMAGE_SGIX: c_uint = 0x835C;
    pub const ATOMIC_COUNTER_BARRIER_BIT: c_uint = 0x00001000;
    pub const ATOMIC_COUNTER_BARRIER_BIT_EXT: c_uint = 0x00001000;
    pub const ATOMIC_COUNTER_BUFFER: c_uint = 0x92C0;
    pub const ATOMIC_COUNTER_BUFFER_ACTIVE_ATOMIC_COUNTERS: c_uint = 0x92C5;
    pub const ATOMIC_COUNTER_BUFFER_ACTIVE_ATOMIC_COUNTER_INDICES: c_uint = 0x92C6;
    pub const ATOMIC_COUNTER_BUFFER_BINDING: c_uint = 0x92C1;
    pub const ATOMIC_COUNTER_BUFFER_DATA_SIZE: c_uint = 0x92C4;
    pub const ATOMIC_COUNTER_BUFFER_INDEX: c_uint = 0x9301;
    pub const ATOMIC_COUNTER_BUFFER_REFERENCED_BY_COMPUTE_SHADER: c_uint = 0x90ED;
    pub const ATOMIC_COUNTER_BUFFER_REFERENCED_BY_FRAGMENT_SHADER: c_uint = 0x92CB;
    pub const ATOMIC_COUNTER_BUFFER_REFERENCED_BY_GEOMETRY_SHADER: c_uint = 0x92CA;
    pub const ATOMIC_COUNTER_BUFFER_REFERENCED_BY_MESH_SHADER_NV: c_uint = 0x959E;
    pub const ATOMIC_COUNTER_BUFFER_REFERENCED_BY_TASK_SHADER_NV: c_uint = 0x959F;
    pub const ATOMIC_COUNTER_BUFFER_REFERENCED_BY_TESS_CONTROL_SHADER: c_uint = 0x92C8;
    pub const ATOMIC_COUNTER_BUFFER_REFERENCED_BY_TESS_EVALUATION_SHADER: c_uint = 0x92C9;
    pub const ATOMIC_COUNTER_BUFFER_REFERENCED_BY_VERTEX_SHADER: c_uint = 0x92C7;
    pub const ATOMIC_COUNTER_BUFFER_SIZE: c_uint = 0x92C3;
    pub const ATOMIC_COUNTER_BUFFER_START: c_uint = 0x92C2;
    pub const ATTACHED_MEMORY_OBJECT_NV: c_uint = 0x95A4;
    pub const ATTACHED_MEMORY_OFFSET_NV: c_uint = 0x95A5;
    pub const ATTACHED_SHADERS: c_uint = 0x8B85;
    pub const ATTENUATION_EXT: c_uint = 0x834D;
    pub const ATTRIBUTE_ADDRESS_COMMAND_NV: c_uint = 0x0009;
    pub const ATTRIB_ARRAY_POINTER_NV: c_uint = 0x8645;
    pub const ATTRIB_ARRAY_SIZE_NV: c_uint = 0x8623;
    pub const ATTRIB_ARRAY_STRIDE_NV: c_uint = 0x8624;
    pub const ATTRIB_ARRAY_TYPE_NV: c_uint = 0x8625;
    pub const AUTO_GENERATE_MIPMAP: c_uint = 0x8295;
    pub const AUX_DEPTH_STENCIL_APPLE: c_uint = 0x8A14;
    pub const AVERAGE_EXT: c_uint = 0x8335;
    pub const AVERAGE_HP: c_uint = 0x8160;
    pub const BACK: c_uint = 0x0405;
    pub const BACK_LEFT: c_uint = 0x0402;
    pub const BACK_NORMALS_HINT_PGI: c_uint = 0x1A223;
    pub const BACK_PRIMARY_COLOR_NV: c_uint = 0x8C77;
    pub const BACK_RIGHT: c_uint = 0x0403;
    pub const BACK_SECONDARY_COLOR_NV: c_uint = 0x8C78;
    pub const BEVEL_NV: c_uint = 0x90A6;
    pub const BGR: c_uint = 0x80E0;
    pub const BGRA: c_uint = 0x80E1;
    pub const BGRA8_EXT: c_uint = 0x93A1;
    pub const BGRA_EXT: c_uint = 0x80E1;
    pub const BGRA_INTEGER: c_uint = 0x8D9B;
    pub const BGRA_INTEGER_EXT: c_uint = 0x8D9B;
    pub const BGR_EXT: c_uint = 0x80E0;
    pub const BGR_INTEGER: c_uint = 0x8D9A;
    pub const BGR_INTEGER_EXT: c_uint = 0x8D9A;
    pub const BIAS_BIT_ATI: c_uint = 0x00000008;
    pub const BIAS_BY_NEGATIVE_ONE_HALF_NV: c_uint = 0x8541;
    pub const BINORMAL_ARRAY_EXT: c_uint = 0x843A;
    pub const BINORMAL_ARRAY_POINTER_EXT: c_uint = 0x8443;
    pub const BINORMAL_ARRAY_STRIDE_EXT: c_uint = 0x8441;
    pub const BINORMAL_ARRAY_TYPE_EXT: c_uint = 0x8440;
    pub const BLACKHOLE_RENDER_INTEL: c_uint = 0x83FC;
    pub const BLEND: c_uint = 0x0BE2;
    pub const BLEND_ADVANCED_COHERENT_KHR: c_uint = 0x9285;
    pub const BLEND_ADVANCED_COHERENT_NV: c_uint = 0x9285;
    pub const BLEND_COLOR: c_uint = 0x8005;
    pub const BLEND_COLOR_COMMAND_NV: c_uint = 0x000B;
    pub const BLEND_COLOR_EXT: c_uint = 0x8005;
    pub const BLEND_DST: c_uint = 0x0BE0;
    pub const BLEND_DST_ALPHA: c_uint = 0x80CA;
    pub const BLEND_DST_ALPHA_EXT: c_uint = 0x80CA;
    pub const BLEND_DST_RGB: c_uint = 0x80C8;
    pub const BLEND_DST_RGB_EXT: c_uint = 0x80C8;
    pub const BLEND_EQUATION: c_uint = 0x8009;
    pub const BLEND_EQUATION_ALPHA: c_uint = 0x883D;
    pub const BLEND_EQUATION_ALPHA_EXT: c_uint = 0x883D;
    pub const BLEND_EQUATION_EXT: c_uint = 0x8009;
    pub const BLEND_EQUATION_RGB: c_uint = 0x8009;
    pub const BLEND_EQUATION_RGB_EXT: c_uint = 0x8009;
    pub const BLEND_OVERLAP_NV: c_uint = 0x9281;
    pub const BLEND_PREMULTIPLIED_SRC_NV: c_uint = 0x9280;
    pub const BLEND_SRC: c_uint = 0x0BE1;
    pub const BLEND_SRC_ALPHA: c_uint = 0x80CB;
    pub const BLEND_SRC_ALPHA_EXT: c_uint = 0x80CB;
    pub const BLEND_SRC_RGB: c_uint = 0x80C9;
    pub const BLEND_SRC_RGB_EXT: c_uint = 0x80C9;
    pub const BLOCK_INDEX: c_uint = 0x92FD;
    pub const BLUE: c_uint = 0x1905;
    pub const BLUE_BIT_ATI: c_uint = 0x00000004;
    pub const BLUE_INTEGER: c_uint = 0x8D96;
    pub const BLUE_INTEGER_EXT: c_uint = 0x8D96;
    pub const BLUE_MAX_CLAMP_INGR: c_uint = 0x8566;
    pub const BLUE_MIN_CLAMP_INGR: c_uint = 0x8562;
    pub const BLUE_NV: c_uint = 0x1905;
    pub const BOLD_BIT_NV: c_uint = 0x01;
    pub const BOOL: c_uint = 0x8B56;
    pub const BOOL_ARB: c_uint = 0x8B56;
    pub const BOOL_VEC2: c_uint = 0x8B57;
    pub const BOOL_VEC2_ARB: c_uint = 0x8B57;
    pub const BOOL_VEC3: c_uint = 0x8B58;
    pub const BOOL_VEC3_ARB: c_uint = 0x8B58;
    pub const BOOL_VEC4: c_uint = 0x8B59;
    pub const BOOL_VEC4_ARB: c_uint = 0x8B59;
    pub const BOUNDING_BOX_NV: c_uint = 0x908D;
    pub const BOUNDING_BOX_OF_BOUNDING_BOXES_NV: c_uint = 0x909C;
    pub const BUFFER: c_uint = 0x82E0;
    pub const BUFFER_ACCESS: c_uint = 0x88BB;
    pub const BUFFER_ACCESS_ARB: c_uint = 0x88BB;
    pub const BUFFER_ACCESS_FLAGS: c_uint = 0x911F;
    pub const BUFFER_BINDING: c_uint = 0x9302;
    pub const BUFFER_DATA_SIZE: c_uint = 0x9303;
    pub const BUFFER_FLUSHING_UNMAP_APPLE: c_uint = 0x8A13;
    pub const BUFFER_GPU_ADDRESS_NV: c_uint = 0x8F1D;
    pub const BUFFER_IMMUTABLE_STORAGE: c_uint = 0x821F;
    pub const BUFFER_MAPPED: c_uint = 0x88BC;
    pub const BUFFER_MAPPED_ARB: c_uint = 0x88BC;
    pub const BUFFER_MAP_LENGTH: c_uint = 0x9120;
    pub const BUFFER_MAP_OFFSET: c_uint = 0x9121;
    pub const BUFFER_MAP_POINTER: c_uint = 0x88BD;
    pub const BUFFER_MAP_POINTER_ARB: c_uint = 0x88BD;
    pub const BUFFER_OBJECT_APPLE: c_uint = 0x85B3;
    pub const BUFFER_OBJECT_EXT: c_uint = 0x9151;
    pub const BUFFER_SERIALIZED_MODIFY_APPLE: c_uint = 0x8A12;
    pub const BUFFER_SIZE: c_uint = 0x8764;
    pub const BUFFER_SIZE_ARB: c_uint = 0x8764;
    pub const BUFFER_STORAGE_FLAGS: c_uint = 0x8220;
    pub const BUFFER_UPDATE_BARRIER_BIT: c_uint = 0x00000200;
    pub const BUFFER_UPDATE_BARRIER_BIT_EXT: c_uint = 0x00000200;
    pub const BUFFER_USAGE: c_uint = 0x8765;
    pub const BUFFER_USAGE_ARB: c_uint = 0x8765;
    pub const BUFFER_VARIABLE: c_uint = 0x92E5;
    pub const BUMP_ENVMAP_ATI: c_uint = 0x877B;
    pub const BUMP_NUM_TEX_UNITS_ATI: c_uint = 0x8777;
    pub const BUMP_ROT_MATRIX_ATI: c_uint = 0x8775;
    pub const BUMP_ROT_MATRIX_SIZE_ATI: c_uint = 0x8776;
    pub const BUMP_TARGET_ATI: c_uint = 0x877C;
    pub const BUMP_TEX_UNITS_ATI: c_uint = 0x8778;
    pub const BYTE: c_uint = 0x1400;
    pub const CALLIGRAPHIC_FRAGMENT_SGIX: c_uint = 0x8183;
    pub const CAVEAT_SUPPORT: c_uint = 0x82B8;
    pub const CCW: c_uint = 0x0901;
    pub const CIRCULAR_CCW_ARC_TO_NV: c_uint = 0xF8;
    pub const CIRCULAR_CW_ARC_TO_NV: c_uint = 0xFA;
    pub const CIRCULAR_TANGENT_ARC_TO_NV: c_uint = 0xFC;
    pub const CLAMP_FRAGMENT_COLOR_ARB: c_uint = 0x891B;
    pub const CLAMP_READ_COLOR: c_uint = 0x891C;
    pub const CLAMP_READ_COLOR_ARB: c_uint = 0x891C;
    pub const CLAMP_TO_BORDER: c_uint = 0x812D;
    pub const CLAMP_TO_BORDER_ARB: c_uint = 0x812D;
    pub const CLAMP_TO_BORDER_SGIS: c_uint = 0x812D;
    pub const CLAMP_TO_EDGE: c_uint = 0x812F;
    pub const CLAMP_TO_EDGE_SGIS: c_uint = 0x812F;
    pub const CLAMP_VERTEX_COLOR_ARB: c_uint = 0x891A;
    pub const CLEAR: c_uint = 0x1500;
    pub const CLEAR_BUFFER: c_uint = 0x82B4;
    pub const CLEAR_TEXTURE: c_uint = 0x9365;
    pub const CLIENT_ACTIVE_TEXTURE_ARB: c_uint = 0x84E1;
    pub const CLIENT_MAPPED_BUFFER_BARRIER_BIT: c_uint = 0x00004000;
    pub const CLIENT_STORAGE_BIT: c_uint = 0x0200;
    pub const CLIPPING_INPUT_PRIMITIVES: c_uint = 0x82F6;
    pub const CLIPPING_INPUT_PRIMITIVES_ARB: c_uint = 0x82F6;
    pub const CLIPPING_OUTPUT_PRIMITIVES: c_uint = 0x82F7;
    pub const CLIPPING_OUTPUT_PRIMITIVES_ARB: c_uint = 0x82F7;
    pub const CLIP_DEPTH_MODE: c_uint = 0x935D;
    pub const CLIP_DISTANCE0: c_uint = 0x3000;
    pub const CLIP_DISTANCE1: c_uint = 0x3001;
    pub const CLIP_DISTANCE2: c_uint = 0x3002;
    pub const CLIP_DISTANCE3: c_uint = 0x3003;
    pub const CLIP_DISTANCE4: c_uint = 0x3004;
    pub const CLIP_DISTANCE5: c_uint = 0x3005;
    pub const CLIP_DISTANCE6: c_uint = 0x3006;
    pub const CLIP_DISTANCE7: c_uint = 0x3007;
    pub const CLIP_DISTANCE_NV: c_uint = 0x8C7A;
    pub const CLIP_FAR_HINT_PGI: c_uint = 0x1A221;
    pub const CLIP_NEAR_HINT_PGI: c_uint = 0x1A220;
    pub const CLIP_ORIGIN: c_uint = 0x935C;
    pub const CLIP_VOLUME_CLIPPING_HINT_EXT: c_uint = 0x80F0;
    pub const CLOSE_PATH_NV: c_uint = 0x00;
    pub const CMYKA_EXT: c_uint = 0x800D;
    pub const CMYK_EXT: c_uint = 0x800C;
    pub const CND0_ATI: c_uint = 0x896B;
    pub const CND_ATI: c_uint = 0x896A;
    pub const COLOR: c_uint = 0x1800;
    pub const COLOR3_BIT_PGI: c_uint = 0x00010000;
    pub const COLOR4_BIT_PGI: c_uint = 0x00020000;
    pub const COLORBURN_KHR: c_uint = 0x929A;
    pub const COLORBURN_NV: c_uint = 0x929A;
    pub const COLORDODGE_KHR: c_uint = 0x9299;
    pub const COLORDODGE_NV: c_uint = 0x9299;
    pub const COLOR_ALPHA_PAIRING_ATI: c_uint = 0x8975;
    pub const COLOR_ARRAY_ADDRESS_NV: c_uint = 0x8F23;
    pub const COLOR_ARRAY_BUFFER_BINDING_ARB: c_uint = 0x8898;
    pub const COLOR_ARRAY_COUNT_EXT: c_uint = 0x8084;
    pub const COLOR_ARRAY_EXT: c_uint = 0x8076;
    pub const COLOR_ARRAY_LENGTH_NV: c_uint = 0x8F2D;
    pub const COLOR_ARRAY_LIST_IBM: c_uint = 103072;
    pub const COLOR_ARRAY_LIST_STRIDE_IBM: c_uint = 103082;
    pub const COLOR_ARRAY_PARALLEL_POINTERS_INTEL: c_uint = 0x83F7;
    pub const COLOR_ARRAY_POINTER_EXT: c_uint = 0x8090;
    pub const COLOR_ARRAY_SIZE_EXT: c_uint = 0x8081;
    pub const COLOR_ARRAY_STRIDE_EXT: c_uint = 0x8083;
    pub const COLOR_ARRAY_TYPE_EXT: c_uint = 0x8082;
    pub const COLOR_ATTACHMENT0: c_uint = 0x8CE0;
    pub const COLOR_ATTACHMENT0_EXT: c_uint = 0x8CE0;
    pub const COLOR_ATTACHMENT1: c_uint = 0x8CE1;
    pub const COLOR_ATTACHMENT10: c_uint = 0x8CEA;
    pub const COLOR_ATTACHMENT10_EXT: c_uint = 0x8CEA;
    pub const COLOR_ATTACHMENT11: c_uint = 0x8CEB;
    pub const COLOR_ATTACHMENT11_EXT: c_uint = 0x8CEB;
    pub const COLOR_ATTACHMENT12: c_uint = 0x8CEC;
    pub const COLOR_ATTACHMENT12_EXT: c_uint = 0x8CEC;
    pub const COLOR_ATTACHMENT13: c_uint = 0x8CED;
    pub const COLOR_ATTACHMENT13_EXT: c_uint = 0x8CED;
    pub const COLOR_ATTACHMENT14: c_uint = 0x8CEE;
    pub const COLOR_ATTACHMENT14_EXT: c_uint = 0x8CEE;
    pub const COLOR_ATTACHMENT15: c_uint = 0x8CEF;
    pub const COLOR_ATTACHMENT15_EXT: c_uint = 0x8CEF;
    pub const COLOR_ATTACHMENT16: c_uint = 0x8CF0;
    pub const COLOR_ATTACHMENT17: c_uint = 0x8CF1;
    pub const COLOR_ATTACHMENT18: c_uint = 0x8CF2;
    pub const COLOR_ATTACHMENT19: c_uint = 0x8CF3;
    pub const COLOR_ATTACHMENT1_EXT: c_uint = 0x8CE1;
    pub const COLOR_ATTACHMENT2: c_uint = 0x8CE2;
    pub const COLOR_ATTACHMENT20: c_uint = 0x8CF4;
    pub const COLOR_ATTACHMENT21: c_uint = 0x8CF5;
    pub const COLOR_ATTACHMENT22: c_uint = 0x8CF6;
    pub const COLOR_ATTACHMENT23: c_uint = 0x8CF7;
    pub const COLOR_ATTACHMENT24: c_uint = 0x8CF8;
    pub const COLOR_ATTACHMENT25: c_uint = 0x8CF9;
    pub const COLOR_ATTACHMENT26: c_uint = 0x8CFA;
    pub const COLOR_ATTACHMENT27: c_uint = 0x8CFB;
    pub const COLOR_ATTACHMENT28: c_uint = 0x8CFC;
    pub const COLOR_ATTACHMENT29: c_uint = 0x8CFD;
    pub const COLOR_ATTACHMENT2_EXT: c_uint = 0x8CE2;
    pub const COLOR_ATTACHMENT3: c_uint = 0x8CE3;
    pub const COLOR_ATTACHMENT30: c_uint = 0x8CFE;
    pub const COLOR_ATTACHMENT31: c_uint = 0x8CFF;
    pub const COLOR_ATTACHMENT3_EXT: c_uint = 0x8CE3;
    pub const COLOR_ATTACHMENT4: c_uint = 0x8CE4;
    pub const COLOR_ATTACHMENT4_EXT: c_uint = 0x8CE4;
    pub const COLOR_ATTACHMENT5: c_uint = 0x8CE5;
    pub const COLOR_ATTACHMENT5_EXT: c_uint = 0x8CE5;
    pub const COLOR_ATTACHMENT6: c_uint = 0x8CE6;
    pub const COLOR_ATTACHMENT6_EXT: c_uint = 0x8CE6;
    pub const COLOR_ATTACHMENT7: c_uint = 0x8CE7;
    pub const COLOR_ATTACHMENT7_EXT: c_uint = 0x8CE7;
    pub const COLOR_ATTACHMENT8: c_uint = 0x8CE8;
    pub const COLOR_ATTACHMENT8_EXT: c_uint = 0x8CE8;
    pub const COLOR_ATTACHMENT9: c_uint = 0x8CE9;
    pub const COLOR_ATTACHMENT9_EXT: c_uint = 0x8CE9;
    pub const COLOR_BUFFER_BIT: c_uint = 0x00004000;
    pub const COLOR_CLEAR_UNCLAMPED_VALUE_ATI: c_uint = 0x8835;
    pub const COLOR_CLEAR_VALUE: c_uint = 0x0C22;
    pub const COLOR_COMPONENTS: c_uint = 0x8283;
    pub const COLOR_ENCODING: c_uint = 0x8296;
    pub const COLOR_FLOAT_APPLE: c_uint = 0x8A0F;
    pub const COLOR_INDEX12_EXT: c_uint = 0x80E6;
    pub const COLOR_INDEX16_EXT: c_uint = 0x80E7;
    pub const COLOR_INDEX1_EXT: c_uint = 0x80E2;
    pub const COLOR_INDEX2_EXT: c_uint = 0x80E3;
    pub const COLOR_INDEX4_EXT: c_uint = 0x80E4;
    pub const COLOR_INDEX8_EXT: c_uint = 0x80E5;
    pub const COLOR_LOGIC_OP: c_uint = 0x0BF2;
    pub const COLOR_MATRIX_SGI: c_uint = 0x80B1;
    pub const COLOR_MATRIX_STACK_DEPTH_SGI: c_uint = 0x80B2;
    pub const COLOR_RENDERABLE: c_uint = 0x8286;
    pub const COLOR_SAMPLES_NV: c_uint = 0x8E20;
    pub const COLOR_SUM_ARB: c_uint = 0x8458;
    pub const COLOR_SUM_CLAMP_NV: c_uint = 0x854F;
    pub const COLOR_SUM_EXT: c_uint = 0x8458;
    pub const COLOR_TABLE_ALPHA_SIZE_SGI: c_uint = 0x80DD;
    pub const COLOR_TABLE_BIAS_SGI: c_uint = 0x80D7;
    pub const COLOR_TABLE_BLUE_SIZE_SGI: c_uint = 0x80DC;
    pub const COLOR_TABLE_FORMAT_SGI: c_uint = 0x80D8;
    pub const COLOR_TABLE_GREEN_SIZE_SGI: c_uint = 0x80DB;
    pub const COLOR_TABLE_INTENSITY_SIZE_SGI: c_uint = 0x80DF;
    pub const COLOR_TABLE_LUMINANCE_SIZE_SGI: c_uint = 0x80DE;
    pub const COLOR_TABLE_RED_SIZE_SGI: c_uint = 0x80DA;
    pub const COLOR_TABLE_SCALE_SGI: c_uint = 0x80D6;
    pub const COLOR_TABLE_SGI: c_uint = 0x80D0;
    pub const COLOR_TABLE_WIDTH_SGI: c_uint = 0x80D9;
    pub const COLOR_WRITEMASK: c_uint = 0x0C23;
    pub const COMBINE4_NV: c_uint = 0x8503;
    pub const COMBINER0_NV: c_uint = 0x8550;
    pub const COMBINER1_NV: c_uint = 0x8551;
    pub const COMBINER2_NV: c_uint = 0x8552;
    pub const COMBINER3_NV: c_uint = 0x8553;
    pub const COMBINER4_NV: c_uint = 0x8554;
    pub const COMBINER5_NV: c_uint = 0x8555;
    pub const COMBINER6_NV: c_uint = 0x8556;
    pub const COMBINER7_NV: c_uint = 0x8557;
    pub const COMBINER_AB_DOT_PRODUCT_NV: c_uint = 0x8545;
    pub const COMBINER_AB_OUTPUT_NV: c_uint = 0x854A;
    pub const COMBINER_BIAS_NV: c_uint = 0x8549;
    pub const COMBINER_CD_DOT_PRODUCT_NV: c_uint = 0x8546;
    pub const COMBINER_CD_OUTPUT_NV: c_uint = 0x854B;
    pub const COMBINER_COMPONENT_USAGE_NV: c_uint = 0x8544;
    pub const COMBINER_INPUT_NV: c_uint = 0x8542;
    pub const COMBINER_MAPPING_NV: c_uint = 0x8543;
    pub const COMBINER_MUX_SUM_NV: c_uint = 0x8547;
    pub const COMBINER_SCALE_NV: c_uint = 0x8548;
    pub const COMBINER_SUM_OUTPUT_NV: c_uint = 0x854C;
    pub const COMBINE_ALPHA_ARB: c_uint = 0x8572;
    pub const COMBINE_ALPHA_EXT: c_uint = 0x8572;
    pub const COMBINE_ARB: c_uint = 0x8570;
    pub const COMBINE_EXT: c_uint = 0x8570;
    pub const COMBINE_RGB_ARB: c_uint = 0x8571;
    pub const COMBINE_RGB_EXT: c_uint = 0x8571;
    pub const COMMAND_BARRIER_BIT: c_uint = 0x00000040;
    pub const COMMAND_BARRIER_BIT_EXT: c_uint = 0x00000040;
    pub const COMPARE_REF_DEPTH_TO_TEXTURE_EXT: c_uint = 0x884E;
    pub const COMPARE_REF_TO_TEXTURE: c_uint = 0x884E;
    pub const COMPARE_R_TO_TEXTURE_ARB: c_uint = 0x884E;
    pub const COMPATIBLE_SUBROUTINES: c_uint = 0x8E4B;
    pub const COMPILE_STATUS: c_uint = 0x8B81;
    pub const COMPLETION_STATUS_ARB: c_uint = 0x91B1;
    pub const COMPLETION_STATUS_KHR: c_uint = 0x91B1;
    pub const COMPRESSED_ALPHA_ARB: c_uint = 0x84E9;
    pub const COMPRESSED_INTENSITY_ARB: c_uint = 0x84EC;
    pub const COMPRESSED_LUMINANCE_ALPHA_ARB: c_uint = 0x84EB;
    pub const COMPRESSED_LUMINANCE_ALPHA_LATC2_EXT: c_uint = 0x8C72;
    pub const COMPRESSED_LUMINANCE_ARB: c_uint = 0x84EA;
    pub const COMPRESSED_LUMINANCE_LATC1_EXT: c_uint = 0x8C70;
    pub const COMPRESSED_R11_EAC: c_uint = 0x9270;
    pub const COMPRESSED_RED: c_uint = 0x8225;
    pub const COMPRESSED_RED_GREEN_RGTC2_EXT: c_uint = 0x8DBD;
    pub const COMPRESSED_RED_RGTC1: c_uint = 0x8DBB;
    pub const COMPRESSED_RED_RGTC1_EXT: c_uint = 0x8DBB;
    pub const COMPRESSED_RG: c_uint = 0x8226;
    pub const COMPRESSED_RG11_EAC: c_uint = 0x9272;
    pub const COMPRESSED_RGB: c_uint = 0x84ED;
    pub const COMPRESSED_RGB8_ETC2: c_uint = 0x9274;
    pub const COMPRESSED_RGB8_PUNCHTHROUGH_ALPHA1_ETC2: c_uint = 0x9276;
    pub const COMPRESSED_RGBA: c_uint = 0x84EE;
    pub const COMPRESSED_RGBA8_ETC2_EAC: c_uint = 0x9278;
    pub const COMPRESSED_RGBA_ARB: c_uint = 0x84EE;
    pub const COMPRESSED_RGBA_ASTC_10x10_KHR: c_uint = 0x93BB;
    pub const COMPRESSED_RGBA_ASTC_10x5_KHR: c_uint = 0x93B8;
    pub const COMPRESSED_RGBA_ASTC_10x6_KHR: c_uint = 0x93B9;
    pub const COMPRESSED_RGBA_ASTC_10x8_KHR: c_uint = 0x93BA;
    pub const COMPRESSED_RGBA_ASTC_12x10_KHR: c_uint = 0x93BC;
    pub const COMPRESSED_RGBA_ASTC_12x12_KHR: c_uint = 0x93BD;
    pub const COMPRESSED_RGBA_ASTC_4x4_KHR: c_uint = 0x93B0;
    pub const COMPRESSED_RGBA_ASTC_5x4_KHR: c_uint = 0x93B1;
    pub const COMPRESSED_RGBA_ASTC_5x5_KHR: c_uint = 0x93B2;
    pub const COMPRESSED_RGBA_ASTC_6x5_KHR: c_uint = 0x93B3;
    pub const COMPRESSED_RGBA_ASTC_6x6_KHR: c_uint = 0x93B4;
    pub const COMPRESSED_RGBA_ASTC_8x5_KHR: c_uint = 0x93B5;
    pub const COMPRESSED_RGBA_ASTC_8x6_KHR: c_uint = 0x93B6;
    pub const COMPRESSED_RGBA_ASTC_8x8_KHR: c_uint = 0x93B7;
    pub const COMPRESSED_RGBA_BPTC_UNORM: c_uint = 0x8E8C;
    pub const COMPRESSED_RGBA_BPTC_UNORM_ARB: c_uint = 0x8E8C;
    pub const COMPRESSED_RGBA_FXT1_3DFX: c_uint = 0x86B1;
    pub const COMPRESSED_RGBA_S3TC_DXT1_EXT: c_uint = 0x83F1;
    pub const COMPRESSED_RGBA_S3TC_DXT3_EXT: c_uint = 0x83F2;
    pub const COMPRESSED_RGBA_S3TC_DXT5_EXT: c_uint = 0x83F3;
    pub const COMPRESSED_RGB_ARB: c_uint = 0x84ED;
    pub const COMPRESSED_RGB_BPTC_SIGNED_FLOAT: c_uint = 0x8E8E;
    pub const COMPRESSED_RGB_BPTC_SIGNED_FLOAT_ARB: c_uint = 0x8E8E;
    pub const COMPRESSED_RGB_BPTC_UNSIGNED_FLOAT: c_uint = 0x8E8F;
    pub const COMPRESSED_RGB_BPTC_UNSIGNED_FLOAT_ARB: c_uint = 0x8E8F;
    pub const COMPRESSED_RGB_FXT1_3DFX: c_uint = 0x86B0;
    pub const COMPRESSED_RGB_S3TC_DXT1_EXT: c_uint = 0x83F0;
    pub const COMPRESSED_RG_RGTC2: c_uint = 0x8DBD;
    pub const COMPRESSED_SIGNED_LUMINANCE_ALPHA_LATC2_EXT: c_uint = 0x8C73;
    pub const COMPRESSED_SIGNED_LUMINANCE_LATC1_EXT: c_uint = 0x8C71;
    pub const COMPRESSED_SIGNED_R11_EAC: c_uint = 0x9271;
    pub const COMPRESSED_SIGNED_RED_GREEN_RGTC2_EXT: c_uint = 0x8DBE;
    pub const COMPRESSED_SIGNED_RED_RGTC1: c_uint = 0x8DBC;
    pub const COMPRESSED_SIGNED_RED_RGTC1_EXT: c_uint = 0x8DBC;
    pub const COMPRESSED_SIGNED_RG11_EAC: c_uint = 0x9273;
    pub const COMPRESSED_SIGNED_RG_RGTC2: c_uint = 0x8DBE;
    pub const COMPRESSED_SLUMINANCE_ALPHA_EXT: c_uint = 0x8C4B;
    pub const COMPRESSED_SLUMINANCE_EXT: c_uint = 0x8C4A;
    pub const COMPRESSED_SRGB: c_uint = 0x8C48;
    pub const COMPRESSED_SRGB8_ALPHA8_ASTC_10x10_KHR: c_uint = 0x93DB;
    pub const COMPRESSED_SRGB8_ALPHA8_ASTC_10x5_KHR: c_uint = 0x93D8;
    pub const COMPRESSED_SRGB8_ALPHA8_ASTC_10x6_KHR: c_uint = 0x93D9;
    pub const COMPRESSED_SRGB8_ALPHA8_ASTC_10x8_KHR: c_uint = 0x93DA;
    pub const COMPRESSED_SRGB8_ALPHA8_ASTC_12x10_KHR: c_uint = 0x93DC;
    pub const COMPRESSED_SRGB8_ALPHA8_ASTC_12x12_KHR: c_uint = 0x93DD;
    pub const COMPRESSED_SRGB8_ALPHA8_ASTC_4x4_KHR: c_uint = 0x93D0;
    pub const COMPRESSED_SRGB8_ALPHA8_ASTC_5x4_KHR: c_uint = 0x93D1;
    pub const COMPRESSED_SRGB8_ALPHA8_ASTC_5x5_KHR: c_uint = 0x93D2;
    pub const COMPRESSED_SRGB8_ALPHA8_ASTC_6x5_KHR: c_uint = 0x93D3;
    pub const COMPRESSED_SRGB8_ALPHA8_ASTC_6x6_KHR: c_uint = 0x93D4;
    pub const COMPRESSED_SRGB8_ALPHA8_ASTC_8x5_KHR: c_uint = 0x93D5;
    pub const COMPRESSED_SRGB8_ALPHA8_ASTC_8x6_KHR: c_uint = 0x93D6;
    pub const COMPRESSED_SRGB8_ALPHA8_ASTC_8x8_KHR: c_uint = 0x93D7;
    pub const COMPRESSED_SRGB8_ALPHA8_ETC2_EAC: c_uint = 0x9279;
    pub const COMPRESSED_SRGB8_ETC2: c_uint = 0x9275;
    pub const COMPRESSED_SRGB8_PUNCHTHROUGH_ALPHA1_ETC2: c_uint = 0x9277;
    pub const COMPRESSED_SRGB_ALPHA: c_uint = 0x8C49;
    pub const COMPRESSED_SRGB_ALPHA_BPTC_UNORM: c_uint = 0x8E8D;
    pub const COMPRESSED_SRGB_ALPHA_BPTC_UNORM_ARB: c_uint = 0x8E8D;
    pub const COMPRESSED_SRGB_ALPHA_EXT: c_uint = 0x8C49;
    pub const COMPRESSED_SRGB_ALPHA_S3TC_DXT1_EXT: c_uint = 0x8C4D;
    pub const COMPRESSED_SRGB_ALPHA_S3TC_DXT3_EXT: c_uint = 0x8C4E;
    pub const COMPRESSED_SRGB_ALPHA_S3TC_DXT5_EXT: c_uint = 0x8C4F;
    pub const COMPRESSED_SRGB_EXT: c_uint = 0x8C48;
    pub const COMPRESSED_SRGB_S3TC_DXT1_EXT: c_uint = 0x8C4C;
    pub const COMPRESSED_TEXTURE_FORMATS: c_uint = 0x86A3;
    pub const COMPRESSED_TEXTURE_FORMATS_ARB: c_uint = 0x86A3;
    pub const COMPUTE_PROGRAM_NV: c_uint = 0x90FB;
    pub const COMPUTE_PROGRAM_PARAMETER_BUFFER_NV: c_uint = 0x90FC;
    pub const COMPUTE_SHADER: c_uint = 0x91B9;
    pub const COMPUTE_SHADER_BIT: c_uint = 0x00000020;
    pub const COMPUTE_SHADER_INVOCATIONS: c_uint = 0x82F5;
    pub const COMPUTE_SHADER_INVOCATIONS_ARB: c_uint = 0x82F5;
    pub const COMPUTE_SUBROUTINE: c_uint = 0x92ED;
    pub const COMPUTE_SUBROUTINE_UNIFORM: c_uint = 0x92F3;
    pub const COMPUTE_TEXTURE: c_uint = 0x82A0;
    pub const COMPUTE_WORK_GROUP_SIZE: c_uint = 0x8267;
    pub const COMP_BIT_ATI: c_uint = 0x00000002;
    pub const CONDITION_SATISFIED: c_uint = 0x911C;
    pub const CONFORMANT_NV: c_uint = 0x9374;
    pub const CONIC_CURVE_TO_NV: c_uint = 0x1A;
    pub const CONJOINT_NV: c_uint = 0x9284;
    pub const CONSERVATIVE_RASTERIZATION_INTEL: c_uint = 0x83FE;
    pub const CONSERVATIVE_RASTERIZATION_NV: c_uint = 0x9346;
    pub const CONSERVATIVE_RASTER_DILATE_GRANULARITY_NV: c_uint = 0x937B;
    pub const CONSERVATIVE_RASTER_DILATE_NV: c_uint = 0x9379;
    pub const CONSERVATIVE_RASTER_DILATE_RANGE_NV: c_uint = 0x937A;
    pub const CONSERVATIVE_RASTER_MODE_NV: c_uint = 0x954D;
    pub const CONSERVATIVE_RASTER_MODE_POST_SNAP_NV: c_uint = 0x954E;
    pub const CONSERVATIVE_RASTER_MODE_PRE_SNAP_NV: c_uint = 0x9550;
    pub const CONSERVATIVE_RASTER_MODE_PRE_SNAP_TRIANGLES_NV: c_uint = 0x954F;
    pub const CONSERVE_MEMORY_HINT_PGI: c_uint = 0x1A1FD;
    pub const CONSTANT_ALPHA: c_uint = 0x8003;
    pub const CONSTANT_ALPHA_EXT: c_uint = 0x8003;
    pub const CONSTANT_ARB: c_uint = 0x8576;
    pub const CONSTANT_BORDER_HP: c_uint = 0x8151;
    pub const CONSTANT_COLOR: c_uint = 0x8001;
    pub const CONSTANT_COLOR0_NV: c_uint = 0x852A;
    pub const CONSTANT_COLOR1_NV: c_uint = 0x852B;
    pub const CONSTANT_COLOR_EXT: c_uint = 0x8001;
    pub const CONSTANT_EXT: c_uint = 0x8576;
    pub const CONST_EYE_NV: c_uint = 0x86E5;
    pub const CONTEXT_COMPATIBILITY_PROFILE_BIT: c_uint = 0x00000002;
    pub const CONTEXT_CORE_PROFILE_BIT: c_uint = 0x00000001;
    pub const CONTEXT_FLAGS: c_uint = 0x821E;
    pub const CONTEXT_FLAG_DEBUG_BIT: c_uint = 0x00000002;
    pub const CONTEXT_FLAG_FORWARD_COMPATIBLE_BIT: c_uint = 0x00000001;
    pub const CONTEXT_FLAG_NO_ERROR_BIT: c_uint = 0x00000008;
    pub const CONTEXT_FLAG_NO_ERROR_BIT_KHR: c_uint = 0x00000008;
    pub const CONTEXT_FLAG_ROBUST_ACCESS_BIT: c_uint = 0x00000004;
    pub const CONTEXT_FLAG_ROBUST_ACCESS_BIT_ARB: c_uint = 0x00000004;
    pub const CONTEXT_LOST: c_uint = 0x0507;
    pub const CONTEXT_PROFILE_MASK: c_uint = 0x9126;
    pub const CONTEXT_RELEASE_BEHAVIOR: c_uint = 0x82FB;
    pub const CONTEXT_RELEASE_BEHAVIOR_FLUSH: c_uint = 0x82FC;
    pub const CONTEXT_ROBUST_ACCESS: c_uint = 0x90F3;
    pub const CONTINUOUS_AMD: c_uint = 0x9007;
    pub const CONTRAST_NV: c_uint = 0x92A1;
    pub const CONVEX_HULL_NV: c_uint = 0x908B;
    pub const CONVOLUTION_1D_EXT: c_uint = 0x8010;
    pub const CONVOLUTION_2D_EXT: c_uint = 0x8011;
    pub const CONVOLUTION_BORDER_COLOR_HP: c_uint = 0x8154;
    pub const CONVOLUTION_BORDER_MODE_EXT: c_uint = 0x8013;
    pub const CONVOLUTION_FILTER_BIAS_EXT: c_uint = 0x8015;
    pub const CONVOLUTION_FILTER_SCALE_EXT: c_uint = 0x8014;
    pub const CONVOLUTION_FORMAT_EXT: c_uint = 0x8017;
    pub const CONVOLUTION_HEIGHT_EXT: c_uint = 0x8019;
    pub const CONVOLUTION_HINT_SGIX: c_uint = 0x8316;
    pub const CONVOLUTION_WIDTH_EXT: c_uint = 0x8018;
    pub const CON_0_ATI: c_uint = 0x8941;
    pub const CON_10_ATI: c_uint = 0x894B;
    pub const CON_11_ATI: c_uint = 0x894C;
    pub const CON_12_ATI: c_uint = 0x894D;
    pub const CON_13_ATI: c_uint = 0x894E;
    pub const CON_14_ATI: c_uint = 0x894F;
    pub const CON_15_ATI: c_uint = 0x8950;
    pub const CON_16_ATI: c_uint = 0x8951;
    pub const CON_17_ATI: c_uint = 0x8952;
    pub const CON_18_ATI: c_uint = 0x8953;
    pub const CON_19_ATI: c_uint = 0x8954;
    pub const CON_1_ATI: c_uint = 0x8942;
    pub const CON_20_ATI: c_uint = 0x8955;
    pub const CON_21_ATI: c_uint = 0x8956;
    pub const CON_22_ATI: c_uint = 0x8957;
    pub const CON_23_ATI: c_uint = 0x8958;
    pub const CON_24_ATI: c_uint = 0x8959;
    pub const CON_25_ATI: c_uint = 0x895A;
    pub const CON_26_ATI: c_uint = 0x895B;
    pub const CON_27_ATI: c_uint = 0x895C;
    pub const CON_28_ATI: c_uint = 0x895D;
    pub const CON_29_ATI: c_uint = 0x895E;
    pub const CON_2_ATI: c_uint = 0x8943;
    pub const CON_30_ATI: c_uint = 0x895F;
    pub const CON_31_ATI: c_uint = 0x8960;
    pub const CON_3_ATI: c_uint = 0x8944;
    pub const CON_4_ATI: c_uint = 0x8945;
    pub const CON_5_ATI: c_uint = 0x8946;
    pub const CON_6_ATI: c_uint = 0x8947;
    pub const CON_7_ATI: c_uint = 0x8948;
    pub const CON_8_ATI: c_uint = 0x8949;
    pub const CON_9_ATI: c_uint = 0x894A;
    pub const COORD_REPLACE_ARB: c_uint = 0x8862;
    pub const COORD_REPLACE_NV: c_uint = 0x8862;
    pub const COPY: c_uint = 0x1503;
    pub const COPY_INVERTED: c_uint = 0x150C;
    pub const COPY_READ_BUFFER: c_uint = 0x8F36;
    pub const COPY_READ_BUFFER_BINDING: c_uint = 0x8F36;
    pub const COPY_WRITE_BUFFER: c_uint = 0x8F37;
    pub const COPY_WRITE_BUFFER_BINDING: c_uint = 0x8F37;
    pub const COUNTER_RANGE_AMD: c_uint = 0x8BC1;
    pub const COUNTER_TYPE_AMD: c_uint = 0x8BC0;
    pub const COUNT_DOWN_NV: c_uint = 0x9089;
    pub const COUNT_UP_NV: c_uint = 0x9088;
    pub const COVERAGE_MODULATION_NV: c_uint = 0x9332;
    pub const COVERAGE_MODULATION_TABLE_NV: c_uint = 0x9331;
    pub const COVERAGE_MODULATION_TABLE_SIZE_NV: c_uint = 0x9333;
    pub const CUBIC_CURVE_TO_NV: c_uint = 0x0C;
    pub const CUBIC_EXT: c_uint = 0x8334;
    pub const CUBIC_HP: c_uint = 0x815F;
    pub const CULL_FACE: c_uint = 0x0B44;
    pub const CULL_FACE_MODE: c_uint = 0x0B45;
    pub const CULL_FRAGMENT_NV: c_uint = 0x86E7;
    pub const CULL_MODES_NV: c_uint = 0x86E0;
    pub const CULL_VERTEX_EXT: c_uint = 0x81AA;
    pub const CULL_VERTEX_EYE_POSITION_EXT: c_uint = 0x81AB;
    pub const CULL_VERTEX_IBM: c_uint = 103050;
    pub const CULL_VERTEX_OBJECT_POSITION_EXT: c_uint = 0x81AC;
    pub const CURRENT_ATTRIB_NV: c_uint = 0x8626;
    pub const CURRENT_BINORMAL_EXT: c_uint = 0x843C;
    pub const CURRENT_FOG_COORDINATE_EXT: c_uint = 0x8453;
    pub const CURRENT_MATRIX_ARB: c_uint = 0x8641;
    pub const CURRENT_MATRIX_INDEX_ARB: c_uint = 0x8845;
    pub const CURRENT_MATRIX_NV: c_uint = 0x8641;
    pub const CURRENT_MATRIX_STACK_DEPTH_ARB: c_uint = 0x8640;
    pub const CURRENT_MATRIX_STACK_DEPTH_NV: c_uint = 0x8640;
    pub const CURRENT_OCCLUSION_QUERY_ID_NV: c_uint = 0x8865;
    pub const CURRENT_PALETTE_MATRIX_ARB: c_uint = 0x8843;
    pub const CURRENT_PROGRAM: c_uint = 0x8B8D;
    pub const CURRENT_QUERY: c_uint = 0x8865;
    pub const CURRENT_QUERY_ARB: c_uint = 0x8865;
    pub const CURRENT_RASTER_NORMAL_SGIX: c_uint = 0x8406;
    pub const CURRENT_SECONDARY_COLOR_EXT: c_uint = 0x8459;
    pub const CURRENT_TANGENT_EXT: c_uint = 0x843B;
    pub const CURRENT_TIME_NV: c_uint = 0x8E28;
    pub const CURRENT_VERTEX_ATTRIB: c_uint = 0x8626;
    pub const CURRENT_VERTEX_ATTRIB_ARB: c_uint = 0x8626;
    pub const CURRENT_VERTEX_EXT: c_uint = 0x87E2;
    pub const CURRENT_VERTEX_WEIGHT_EXT: c_uint = 0x850B;
    pub const CURRENT_WEIGHT_ARB: c_uint = 0x86A8;
    pub const CW: c_uint = 0x0900;
    pub const D3D12_FENCE_VALUE_EXT: c_uint = 0x9595;
    pub const DARKEN_KHR: c_uint = 0x9297;
    pub const DARKEN_NV: c_uint = 0x9297;
    pub const DATA_BUFFER_AMD: c_uint = 0x9151;
    pub const DEBUG_CALLBACK_FUNCTION: c_uint = 0x8244;
    pub const DEBUG_CALLBACK_FUNCTION_ARB: c_uint = 0x8244;
    pub const DEBUG_CALLBACK_USER_PARAM: c_uint = 0x8245;
    pub const DEBUG_CALLBACK_USER_PARAM_ARB: c_uint = 0x8245;
    pub const DEBUG_CATEGORY_API_ERROR_AMD: c_uint = 0x9149;
    pub const DEBUG_CATEGORY_APPLICATION_AMD: c_uint = 0x914F;
    pub const DEBUG_CATEGORY_DEPRECATION_AMD: c_uint = 0x914B;
    pub const DEBUG_CATEGORY_OTHER_AMD: c_uint = 0x9150;
    pub const DEBUG_CATEGORY_PERFORMANCE_AMD: c_uint = 0x914D;
    pub const DEBUG_CATEGORY_SHADER_COMPILER_AMD: c_uint = 0x914E;
    pub const DEBUG_CATEGORY_UNDEFINED_BEHAVIOR_AMD: c_uint = 0x914C;
    pub const DEBUG_CATEGORY_WINDOW_SYSTEM_AMD: c_uint = 0x914A;
    pub const DEBUG_GROUP_STACK_DEPTH: c_uint = 0x826D;
    pub const DEBUG_LOGGED_MESSAGES: c_uint = 0x9145;
    pub const DEBUG_LOGGED_MESSAGES_AMD: c_uint = 0x9145;
    pub const DEBUG_LOGGED_MESSAGES_ARB: c_uint = 0x9145;
    pub const DEBUG_NEXT_LOGGED_MESSAGE_LENGTH: c_uint = 0x8243;
    pub const DEBUG_NEXT_LOGGED_MESSAGE_LENGTH_ARB: c_uint = 0x8243;
    pub const DEBUG_OUTPUT: c_uint = 0x92E0;
    pub const DEBUG_OUTPUT_SYNCHRONOUS: c_uint = 0x8242;
    pub const DEBUG_OUTPUT_SYNCHRONOUS_ARB: c_uint = 0x8242;
    pub const DEBUG_SEVERITY_HIGH: c_uint = 0x9146;
    pub const DEBUG_SEVERITY_HIGH_AMD: c_uint = 0x9146;
    pub const DEBUG_SEVERITY_HIGH_ARB: c_uint = 0x9146;
    pub const DEBUG_SEVERITY_LOW: c_uint = 0x9148;
    pub const DEBUG_SEVERITY_LOW_AMD: c_uint = 0x9148;
    pub const DEBUG_SEVERITY_LOW_ARB: c_uint = 0x9148;
    pub const DEBUG_SEVERITY_MEDIUM: c_uint = 0x9147;
    pub const DEBUG_SEVERITY_MEDIUM_AMD: c_uint = 0x9147;
    pub const DEBUG_SEVERITY_MEDIUM_ARB: c_uint = 0x9147;
    pub const DEBUG_SEVERITY_NOTIFICATION: c_uint = 0x826B;
    pub const DEBUG_SOURCE_API: c_uint = 0x8246;
    pub const DEBUG_SOURCE_API_ARB: c_uint = 0x8246;
    pub const DEBUG_SOURCE_APPLICATION: c_uint = 0x824A;
    pub const DEBUG_SOURCE_APPLICATION_ARB: c_uint = 0x824A;
    pub const DEBUG_SOURCE_OTHER: c_uint = 0x824B;
    pub const DEBUG_SOURCE_OTHER_ARB: c_uint = 0x824B;
    pub const DEBUG_SOURCE_SHADER_COMPILER: c_uint = 0x8248;
    pub const DEBUG_SOURCE_SHADER_COMPILER_ARB: c_uint = 0x8248;
    pub const DEBUG_SOURCE_THIRD_PARTY: c_uint = 0x8249;
    pub const DEBUG_SOURCE_THIRD_PARTY_ARB: c_uint = 0x8249;
    pub const DEBUG_SOURCE_WINDOW_SYSTEM: c_uint = 0x8247;
    pub const DEBUG_SOURCE_WINDOW_SYSTEM_ARB: c_uint = 0x8247;
    pub const DEBUG_TYPE_DEPRECATED_BEHAVIOR: c_uint = 0x824D;
    pub const DEBUG_TYPE_DEPRECATED_BEHAVIOR_ARB: c_uint = 0x824D;
    pub const DEBUG_TYPE_ERROR: c_uint = 0x824C;
    pub const DEBUG_TYPE_ERROR_ARB: c_uint = 0x824C;
    pub const DEBUG_TYPE_MARKER: c_uint = 0x8268;
    pub const DEBUG_TYPE_OTHER: c_uint = 0x8251;
    pub const DEBUG_TYPE_OTHER_ARB: c_uint = 0x8251;
    pub const DEBUG_TYPE_PERFORMANCE: c_uint = 0x8250;
    pub const DEBUG_TYPE_PERFORMANCE_ARB: c_uint = 0x8250;
    pub const DEBUG_TYPE_POP_GROUP: c_uint = 0x826A;
    pub const DEBUG_TYPE_PORTABILITY: c_uint = 0x824F;
    pub const DEBUG_TYPE_PORTABILITY_ARB: c_uint = 0x824F;
    pub const DEBUG_TYPE_PUSH_GROUP: c_uint = 0x8269;
    pub const DEBUG_TYPE_UNDEFINED_BEHAVIOR: c_uint = 0x824E;
    pub const DEBUG_TYPE_UNDEFINED_BEHAVIOR_ARB: c_uint = 0x824E;
    pub const DECODE_EXT: c_uint = 0x8A49;
    pub const DECR: c_uint = 0x1E03;
    pub const DECR_WRAP: c_uint = 0x8508;
    pub const DECR_WRAP_EXT: c_uint = 0x8508;
    pub const DEDICATED_MEMORY_OBJECT_EXT: c_uint = 0x9581;
    pub const DEFORMATIONS_MASK_SGIX: c_uint = 0x8196;
    pub const DELETE_STATUS: c_uint = 0x8B80;
    pub const DEPENDENT_AR_TEXTURE_2D_NV: c_uint = 0x86E9;
    pub const DEPENDENT_GB_TEXTURE_2D_NV: c_uint = 0x86EA;
    pub const DEPENDENT_HILO_TEXTURE_2D_NV: c_uint = 0x8858;
    pub const DEPENDENT_RGB_TEXTURE_3D_NV: c_uint = 0x8859;
    pub const DEPENDENT_RGB_TEXTURE_CUBE_MAP_NV: c_uint = 0x885A;
    pub const DEPTH: c_uint = 0x1801;
    pub const DEPTH24_STENCIL8: c_uint = 0x88F0;
    pub const DEPTH24_STENCIL8_EXT: c_uint = 0x88F0;
    pub const DEPTH32F_STENCIL8: c_uint = 0x8CAD;
    pub const DEPTH32F_STENCIL8_NV: c_uint = 0x8DAC;
    pub const DEPTH_ATTACHMENT: c_uint = 0x8D00;
    pub const DEPTH_ATTACHMENT_EXT: c_uint = 0x8D00;
    pub const DEPTH_BOUNDS_EXT: c_uint = 0x8891;
    pub const DEPTH_BOUNDS_TEST_EXT: c_uint = 0x8890;
    pub const DEPTH_BUFFER_BIT: c_uint = 0x00000100;
    pub const DEPTH_BUFFER_FLOAT_MODE_NV: c_uint = 0x8DAF;
    pub const DEPTH_CLAMP: c_uint = 0x864F;
    pub const DEPTH_CLAMP_FAR_AMD: c_uint = 0x901F;
    pub const DEPTH_CLAMP_NEAR_AMD: c_uint = 0x901E;
    pub const DEPTH_CLAMP_NV: c_uint = 0x864F;
    pub const DEPTH_CLEAR_VALUE: c_uint = 0x0B73;
    pub const DEPTH_COMPONENT: c_uint = 0x1902;
    pub const DEPTH_COMPONENT16: c_uint = 0x81A5;
    pub const DEPTH_COMPONENT16_ARB: c_uint = 0x81A5;
    pub const DEPTH_COMPONENT16_SGIX: c_uint = 0x81A5;
    pub const DEPTH_COMPONENT24: c_uint = 0x81A6;
    pub const DEPTH_COMPONENT24_ARB: c_uint = 0x81A6;
    pub const DEPTH_COMPONENT24_SGIX: c_uint = 0x81A6;
    pub const DEPTH_COMPONENT32: c_uint = 0x81A7;
    pub const DEPTH_COMPONENT32F: c_uint = 0x8CAC;
    pub const DEPTH_COMPONENT32F_NV: c_uint = 0x8DAB;
    pub const DEPTH_COMPONENT32_ARB: c_uint = 0x81A7;
    pub const DEPTH_COMPONENT32_SGIX: c_uint = 0x81A7;
    pub const DEPTH_COMPONENTS: c_uint = 0x8284;
    pub const DEPTH_FUNC: c_uint = 0x0B74;
    pub const DEPTH_RANGE: c_uint = 0x0B70;
    pub const DEPTH_RENDERABLE: c_uint = 0x8287;
    pub const DEPTH_SAMPLES_NV: c_uint = 0x932D;
    pub const DEPTH_STENCIL: c_uint = 0x84F9;
    pub const DEPTH_STENCIL_ATTACHMENT: c_uint = 0x821A;
    pub const DEPTH_STENCIL_EXT: c_uint = 0x84F9;
    pub const DEPTH_STENCIL_NV: c_uint = 0x84F9;
    pub const DEPTH_STENCIL_TEXTURE_MODE: c_uint = 0x90EA;
    pub const DEPTH_STENCIL_TO_BGRA_NV: c_uint = 0x886F;
    pub const DEPTH_STENCIL_TO_RGBA_NV: c_uint = 0x886E;
    pub const DEPTH_TEST: c_uint = 0x0B71;
    pub const DEPTH_TEXTURE_MODE_ARB: c_uint = 0x884B;
    pub const DEPTH_WRITEMASK: c_uint = 0x0B72;
    pub const DETACHED_BUFFERS_NV: c_uint = 0x95AB;
    pub const DETACHED_MEMORY_INCARNATION_NV: c_uint = 0x95A9;
    pub const DETACHED_TEXTURES_NV: c_uint = 0x95AA;
    pub const DETAIL_TEXTURE_2D_BINDING_SGIS: c_uint = 0x8096;
    pub const DETAIL_TEXTURE_2D_SGIS: c_uint = 0x8095;
    pub const DETAIL_TEXTURE_FUNC_POINTS_SGIS: c_uint = 0x809C;
    pub const DETAIL_TEXTURE_LEVEL_SGIS: c_uint = 0x809A;
    pub const DETAIL_TEXTURE_MODE_SGIS: c_uint = 0x809B;
    pub const DEVICE_LUID_EXT: c_uint = 0x9599;
    pub const DEVICE_NODE_MASK_EXT: c_uint = 0x959A;
    pub const DEVICE_UUID_EXT: c_uint = 0x9597;
    pub const DIFFERENCE_KHR: c_uint = 0x929E;
    pub const DIFFERENCE_NV: c_uint = 0x929E;
    pub const DISCARD_ATI: c_uint = 0x8763;
    pub const DISCARD_NV: c_uint = 0x8530;
    pub const DISCRETE_AMD: c_uint = 0x9006;
    pub const DISJOINT_NV: c_uint = 0x9283;
    pub const DISPATCH_INDIRECT_BUFFER: c_uint = 0x90EE;
    pub const DISPATCH_INDIRECT_BUFFER_BINDING: c_uint = 0x90EF;
    pub const DISTANCE_ATTENUATION_EXT: c_uint = 0x8129;
    pub const DISTANCE_ATTENUATION_SGIS: c_uint = 0x8129;
    pub const DITHER: c_uint = 0x0BD0;
    pub const DONT_CARE: c_uint = 0x1100;
    pub const DOT2_ADD_ATI: c_uint = 0x896C;
    pub const DOT3_ATI: c_uint = 0x8966;
    pub const DOT3_RGBA_ARB: c_uint = 0x86AF;
    pub const DOT3_RGBA_EXT: c_uint = 0x8741;
    pub const DOT3_RGB_ARB: c_uint = 0x86AE;
    pub const DOT3_RGB_EXT: c_uint = 0x8740;
    pub const DOT4_ATI: c_uint = 0x8967;
    pub const DOT_PRODUCT_AFFINE_DEPTH_REPLACE_NV: c_uint = 0x885D;
    pub const DOT_PRODUCT_CONST_EYE_REFLECT_CUBE_MAP_NV: c_uint = 0x86F3;
    pub const DOT_PRODUCT_DEPTH_REPLACE_NV: c_uint = 0x86ED;
    pub const DOT_PRODUCT_DIFFUSE_CUBE_MAP_NV: c_uint = 0x86F1;
    pub const DOT_PRODUCT_NV: c_uint = 0x86EC;
    pub const DOT_PRODUCT_PASS_THROUGH_NV: c_uint = 0x885B;
    pub const DOT_PRODUCT_REFLECT_CUBE_MAP_NV: c_uint = 0x86F2;
    pub const DOT_PRODUCT_TEXTURE_1D_NV: c_uint = 0x885C;
    pub const DOT_PRODUCT_TEXTURE_2D_NV: c_uint = 0x86EE;
    pub const DOT_PRODUCT_TEXTURE_3D_NV: c_uint = 0x86EF;
    pub const DOT_PRODUCT_TEXTURE_CUBE_MAP_NV: c_uint = 0x86F0;
    pub const DOT_PRODUCT_TEXTURE_RECTANGLE_NV: c_uint = 0x864E;
    pub const DOUBLE: c_uint = 0x140A;
    pub const DOUBLEBUFFER: c_uint = 0x0C32;
    pub const DOUBLE_MAT2: c_uint = 0x8F46;
    pub const DOUBLE_MAT2_EXT: c_uint = 0x8F46;
    pub const DOUBLE_MAT2x3: c_uint = 0x8F49;
    pub const DOUBLE_MAT2x3_EXT: c_uint = 0x8F49;
    pub const DOUBLE_MAT2x4: c_uint = 0x8F4A;
    pub const DOUBLE_MAT2x4_EXT: c_uint = 0x8F4A;
    pub const DOUBLE_MAT3: c_uint = 0x8F47;
    pub const DOUBLE_MAT3_EXT: c_uint = 0x8F47;
    pub const DOUBLE_MAT3x2: c_uint = 0x8F4B;
    pub const DOUBLE_MAT3x2_EXT: c_uint = 0x8F4B;
    pub const DOUBLE_MAT3x4: c_uint = 0x8F4C;
    pub const DOUBLE_MAT3x4_EXT: c_uint = 0x8F4C;
    pub const DOUBLE_MAT4: c_uint = 0x8F48;
    pub const DOUBLE_MAT4_EXT: c_uint = 0x8F48;
    pub const DOUBLE_MAT4x2: c_uint = 0x8F4D;
    pub const DOUBLE_MAT4x2_EXT: c_uint = 0x8F4D;
    pub const DOUBLE_MAT4x3: c_uint = 0x8F4E;
    pub const DOUBLE_MAT4x3_EXT: c_uint = 0x8F4E;
    pub const DOUBLE_VEC2: c_uint = 0x8FFC;
    pub const DOUBLE_VEC2_EXT: c_uint = 0x8FFC;
    pub const DOUBLE_VEC3: c_uint = 0x8FFD;
    pub const DOUBLE_VEC3_EXT: c_uint = 0x8FFD;
    pub const DOUBLE_VEC4: c_uint = 0x8FFE;
    pub const DOUBLE_VEC4_EXT: c_uint = 0x8FFE;
    pub const DRAW_ARRAYS_COMMAND_NV: c_uint = 0x0003;
    pub const DRAW_ARRAYS_INSTANCED_COMMAND_NV: c_uint = 0x0007;
    pub const DRAW_ARRAYS_STRIP_COMMAND_NV: c_uint = 0x0005;
    pub const DRAW_BUFFER: c_uint = 0x0C01;
    pub const DRAW_BUFFER0: c_uint = 0x8825;
    pub const DRAW_BUFFER0_ARB: c_uint = 0x8825;
    pub const DRAW_BUFFER0_ATI: c_uint = 0x8825;
    pub const DRAW_BUFFER1: c_uint = 0x8826;
    pub const DRAW_BUFFER10: c_uint = 0x882F;
    pub const DRAW_BUFFER10_ARB: c_uint = 0x882F;
    pub const DRAW_BUFFER10_ATI: c_uint = 0x882F;
    pub const DRAW_BUFFER11: c_uint = 0x8830;
    pub const DRAW_BUFFER11_ARB: c_uint = 0x8830;
    pub const DRAW_BUFFER11_ATI: c_uint = 0x8830;
    pub const DRAW_BUFFER12: c_uint = 0x8831;
    pub const DRAW_BUFFER12_ARB: c_uint = 0x8831;
    pub const DRAW_BUFFER12_ATI: c_uint = 0x8831;
    pub const DRAW_BUFFER13: c_uint = 0x8832;
    pub const DRAW_BUFFER13_ARB: c_uint = 0x8832;
    pub const DRAW_BUFFER13_ATI: c_uint = 0x8832;
    pub const DRAW_BUFFER14: c_uint = 0x8833;
    pub const DRAW_BUFFER14_ARB: c_uint = 0x8833;
    pub const DRAW_BUFFER14_ATI: c_uint = 0x8833;
    pub const DRAW_BUFFER15: c_uint = 0x8834;
    pub const DRAW_BUFFER15_ARB: c_uint = 0x8834;
    pub const DRAW_BUFFER15_ATI: c_uint = 0x8834;
    pub const DRAW_BUFFER1_ARB: c_uint = 0x8826;
    pub const DRAW_BUFFER1_ATI: c_uint = 0x8826;
    pub const DRAW_BUFFER2: c_uint = 0x8827;
    pub const DRAW_BUFFER2_ARB: c_uint = 0x8827;
    pub const DRAW_BUFFER2_ATI: c_uint = 0x8827;
    pub const DRAW_BUFFER3: c_uint = 0x8828;
    pub const DRAW_BUFFER3_ARB: c_uint = 0x8828;
    pub const DRAW_BUFFER3_ATI: c_uint = 0x8828;
    pub const DRAW_BUFFER4: c_uint = 0x8829;
    pub const DRAW_BUFFER4_ARB: c_uint = 0x8829;
    pub const DRAW_BUFFER4_ATI: c_uint = 0x8829;
    pub const DRAW_BUFFER5: c_uint = 0x882A;
    pub const DRAW_BUFFER5_ARB: c_uint = 0x882A;
    pub const DRAW_BUFFER5_ATI: c_uint = 0x882A;
    pub const DRAW_BUFFER6: c_uint = 0x882B;
    pub const DRAW_BUFFER6_ARB: c_uint = 0x882B;
    pub const DRAW_BUFFER6_ATI: c_uint = 0x882B;
    pub const DRAW_BUFFER7: c_uint = 0x882C;
    pub const DRAW_BUFFER7_ARB: c_uint = 0x882C;
    pub const DRAW_BUFFER7_ATI: c_uint = 0x882C;
    pub const DRAW_BUFFER8: c_uint = 0x882D;
    pub const DRAW_BUFFER8_ARB: c_uint = 0x882D;
    pub const DRAW_BUFFER8_ATI: c_uint = 0x882D;
    pub const DRAW_BUFFER9: c_uint = 0x882E;
    pub const DRAW_BUFFER9_ARB: c_uint = 0x882E;
    pub const DRAW_BUFFER9_ATI: c_uint = 0x882E;
    pub const DRAW_ELEMENTS_COMMAND_NV: c_uint = 0x0002;
    pub const DRAW_ELEMENTS_INSTANCED_COMMAND_NV: c_uint = 0x0006;
    pub const DRAW_ELEMENTS_STRIP_COMMAND_NV: c_uint = 0x0004;
    pub const DRAW_FRAMEBUFFER: c_uint = 0x8CA9;
    pub const DRAW_FRAMEBUFFER_BINDING: c_uint = 0x8CA6;
    pub const DRAW_FRAMEBUFFER_BINDING_EXT: c_uint = 0x8CA6;
    pub const DRAW_FRAMEBUFFER_EXT: c_uint = 0x8CA9;
    pub const DRAW_INDIRECT_ADDRESS_NV: c_uint = 0x8F41;
    pub const DRAW_INDIRECT_BUFFER: c_uint = 0x8F3F;
    pub const DRAW_INDIRECT_BUFFER_BINDING: c_uint = 0x8F43;
    pub const DRAW_INDIRECT_LENGTH_NV: c_uint = 0x8F42;
    pub const DRAW_INDIRECT_UNIFIED_NV: c_uint = 0x8F40;
    pub const DRAW_PIXELS_APPLE: c_uint = 0x8A0A;
    pub const DRIVER_UUID_EXT: c_uint = 0x9598;
    pub const DSDT8_MAG8_INTENSITY8_NV: c_uint = 0x870B;
    pub const DSDT8_MAG8_NV: c_uint = 0x870A;
    pub const DSDT8_NV: c_uint = 0x8709;
    pub const DSDT_MAG_INTENSITY_NV: c_uint = 0x86DC;
    pub const DSDT_MAG_NV: c_uint = 0x86F6;
    pub const DSDT_MAG_VIB_NV: c_uint = 0x86F7;
    pub const DSDT_NV: c_uint = 0x86F5;
    pub const DST_ALPHA: c_uint = 0x0304;
    pub const DST_ATOP_NV: c_uint = 0x928F;
    pub const DST_COLOR: c_uint = 0x0306;
    pub const DST_IN_NV: c_uint = 0x928B;
    pub const DST_NV: c_uint = 0x9287;
    pub const DST_OUT_NV: c_uint = 0x928D;
    pub const DST_OVER_NV: c_uint = 0x9289;
    pub const DS_BIAS_NV: c_uint = 0x8716;
    pub const DS_SCALE_NV: c_uint = 0x8710;
    pub const DT_BIAS_NV: c_uint = 0x8717;
    pub const DT_SCALE_NV: c_uint = 0x8711;
    pub const DU8DV8_ATI: c_uint = 0x877A;
    pub const DUAL_ALPHA12_SGIS: c_uint = 0x8112;
    pub const DUAL_ALPHA16_SGIS: c_uint = 0x8113;
    pub const DUAL_ALPHA4_SGIS: c_uint = 0x8110;
    pub const DUAL_ALPHA8_SGIS: c_uint = 0x8111;
    pub const DUAL_INTENSITY12_SGIS: c_uint = 0x811A;
    pub const DUAL_INTENSITY16_SGIS: c_uint = 0x811B;
    pub const DUAL_INTENSITY4_SGIS: c_uint = 0x8118;
    pub const DUAL_INTENSITY8_SGIS: c_uint = 0x8119;
    pub const DUAL_LUMINANCE12_SGIS: c_uint = 0x8116;
    pub const DUAL_LUMINANCE16_SGIS: c_uint = 0x8117;
    pub const DUAL_LUMINANCE4_SGIS: c_uint = 0x8114;
    pub const DUAL_LUMINANCE8_SGIS: c_uint = 0x8115;
    pub const DUAL_LUMINANCE_ALPHA4_SGIS: c_uint = 0x811C;
    pub const DUAL_LUMINANCE_ALPHA8_SGIS: c_uint = 0x811D;
    pub const DUAL_TEXTURE_SELECT_SGIS: c_uint = 0x8124;
    pub const DUDV_ATI: c_uint = 0x8779;
    pub const DUP_FIRST_CUBIC_CURVE_TO_NV: c_uint = 0xF2;
    pub const DUP_LAST_CUBIC_CURVE_TO_NV: c_uint = 0xF4;
    pub const DYNAMIC_ATI: c_uint = 0x8761;
    pub const DYNAMIC_COPY: c_uint = 0x88EA;
    pub const DYNAMIC_COPY_ARB: c_uint = 0x88EA;
    pub const DYNAMIC_DRAW: c_uint = 0x88E8;
    pub const DYNAMIC_DRAW_ARB: c_uint = 0x88E8;
    pub const DYNAMIC_READ: c_uint = 0x88E9;
    pub const DYNAMIC_READ_ARB: c_uint = 0x88E9;
    pub const DYNAMIC_STORAGE_BIT: c_uint = 0x0100;
    pub const EDGEFLAG_BIT_PGI: c_uint = 0x00040000;
    pub const EDGE_FLAG_ARRAY_ADDRESS_NV: c_uint = 0x8F26;
    pub const EDGE_FLAG_ARRAY_BUFFER_BINDING_ARB: c_uint = 0x889B;
    pub const EDGE_FLAG_ARRAY_COUNT_EXT: c_uint = 0x808D;
    pub const EDGE_FLAG_ARRAY_EXT: c_uint = 0x8079;
    pub const EDGE_FLAG_ARRAY_LENGTH_NV: c_uint = 0x8F30;
    pub const EDGE_FLAG_ARRAY_LIST_IBM: c_uint = 103075;
    pub const EDGE_FLAG_ARRAY_LIST_STRIDE_IBM: c_uint = 103085;
    pub const EDGE_FLAG_ARRAY_POINTER_EXT: c_uint = 0x8093;
    pub const EDGE_FLAG_ARRAY_STRIDE_EXT: c_uint = 0x808C;
    pub const EFFECTIVE_RASTER_SAMPLES_EXT: c_uint = 0x932C;
    pub const EIGHTH_BIT_ATI: c_uint = 0x00000020;
    pub const ELEMENT_ADDRESS_COMMAND_NV: c_uint = 0x0008;
    pub const ELEMENT_ARRAY_ADDRESS_NV: c_uint = 0x8F29;
    pub const ELEMENT_ARRAY_APPLE: c_uint = 0x8A0C;
    pub const ELEMENT_ARRAY_ATI: c_uint = 0x8768;
    pub const ELEMENT_ARRAY_BARRIER_BIT: c_uint = 0x00000002;
    pub const ELEMENT_ARRAY_BARRIER_BIT_EXT: c_uint = 0x00000002;
    pub const ELEMENT_ARRAY_BUFFER: c_uint = 0x8893;
    pub const ELEMENT_ARRAY_BUFFER_ARB: c_uint = 0x8893;
    pub const ELEMENT_ARRAY_BUFFER_BINDING: c_uint = 0x8895;
    pub const ELEMENT_ARRAY_BUFFER_BINDING_ARB: c_uint = 0x8895;
    pub const ELEMENT_ARRAY_LENGTH_NV: c_uint = 0x8F33;
    pub const ELEMENT_ARRAY_POINTER_APPLE: c_uint = 0x8A0E;
    pub const ELEMENT_ARRAY_POINTER_ATI: c_uint = 0x876A;
    pub const ELEMENT_ARRAY_TYPE_APPLE: c_uint = 0x8A0D;
    pub const ELEMENT_ARRAY_TYPE_ATI: c_uint = 0x8769;
    pub const ELEMENT_ARRAY_UNIFIED_NV: c_uint = 0x8F1F;
    pub const EMBOSS_CONSTANT_NV: c_uint = 0x855E;
    pub const EMBOSS_LIGHT_NV: c_uint = 0x855D;
    pub const EMBOSS_MAP_NV: c_uint = 0x855F;
    pub const EQUAL: c_uint = 0x0202;
    pub const EQUIV: c_uint = 0x1509;
    pub const EVAL_2D_NV: c_uint = 0x86C0;
    pub const EVAL_FRACTIONAL_TESSELLATION_NV: c_uint = 0x86C5;
    pub const EVAL_TRIANGULAR_2D_NV: c_uint = 0x86C1;
    pub const EVAL_VERTEX_ATTRIB0_NV: c_uint = 0x86C6;
    pub const EVAL_VERTEX_ATTRIB10_NV: c_uint = 0x86D0;
    pub const EVAL_VERTEX_ATTRIB11_NV: c_uint = 0x86D1;
    pub const EVAL_VERTEX_ATTRIB12_NV: c_uint = 0x86D2;
    pub const EVAL_VERTEX_ATTRIB13_NV: c_uint = 0x86D3;
    pub const EVAL_VERTEX_ATTRIB14_NV: c_uint = 0x86D4;
    pub const EVAL_VERTEX_ATTRIB15_NV: c_uint = 0x86D5;
    pub const EVAL_VERTEX_ATTRIB1_NV: c_uint = 0x86C7;
    pub const EVAL_VERTEX_ATTRIB2_NV: c_uint = 0x86C8;
    pub const EVAL_VERTEX_ATTRIB3_NV: c_uint = 0x86C9;
    pub const EVAL_VERTEX_ATTRIB4_NV: c_uint = 0x86CA;
    pub const EVAL_VERTEX_ATTRIB5_NV: c_uint = 0x86CB;
    pub const EVAL_VERTEX_ATTRIB6_NV: c_uint = 0x86CC;
    pub const EVAL_VERTEX_ATTRIB7_NV: c_uint = 0x86CD;
    pub const EVAL_VERTEX_ATTRIB8_NV: c_uint = 0x86CE;
    pub const EVAL_VERTEX_ATTRIB9_NV: c_uint = 0x86CF;
    pub const EXCLUSION_KHR: c_uint = 0x92A0;
    pub const EXCLUSION_NV: c_uint = 0x92A0;
    pub const EXCLUSIVE_EXT: c_uint = 0x8F11;
    pub const EXPAND_NEGATE_NV: c_uint = 0x8539;
    pub const EXPAND_NORMAL_NV: c_uint = 0x8538;
    pub const EXTENSIONS: c_uint = 0x1F03;
    pub const EXTERNAL_VIRTUAL_MEMORY_BUFFER_AMD: c_uint = 0x9160;
    pub const EYE_DISTANCE_TO_LINE_SGIS: c_uint = 0x81F2;
    pub const EYE_DISTANCE_TO_POINT_SGIS: c_uint = 0x81F0;
    pub const EYE_LINE_SGIS: c_uint = 0x81F6;
    pub const EYE_PLANE: c_uint = 0x2502;
    pub const EYE_PLANE_ABSOLUTE_NV: c_uint = 0x855C;
    pub const EYE_POINT_SGIS: c_uint = 0x81F4;
    pub const EYE_RADIAL_NV: c_uint = 0x855B;
    pub const E_TIMES_F_NV: c_uint = 0x8531;
    pub const FACTOR_MAX_AMD: c_uint = 0x901D;
    pub const FACTOR_MIN_AMD: c_uint = 0x901C;
    pub const FAILURE_NV: c_uint = 0x9030;
    pub const FALSE: c_uchar = 0;
    pub const FASTEST: c_uint = 0x1101;
    pub const FENCE_APPLE: c_uint = 0x8A0B;
    pub const FENCE_CONDITION_NV: c_uint = 0x84F4;
    pub const FENCE_STATUS_NV: c_uint = 0x84F3;
    pub const FIELDS_NV: c_uint = 0x8E27;
    pub const FIELD_LOWER_NV: c_uint = 0x9023;
    pub const FIELD_UPPER_NV: c_uint = 0x9022;
    pub const FILE_NAME_NV: c_uint = 0x9074;
    pub const FILL: c_uint = 0x1B02;
    pub const FILL_RECTANGLE_NV: c_uint = 0x933C;
    pub const FILTER: c_uint = 0x829A;
    pub const FILTER4_SGIS: c_uint = 0x8146;
    pub const FIRST_TO_REST_NV: c_uint = 0x90AF;
    pub const FIRST_VERTEX_CONVENTION: c_uint = 0x8E4D;
    pub const FIRST_VERTEX_CONVENTION_EXT: c_uint = 0x8E4D;
    pub const FIXED: c_uint = 0x140C;
    pub const FIXED_OES: c_uint = 0x140C;
    pub const FIXED_ONLY: c_uint = 0x891D;
    pub const FIXED_ONLY_ARB: c_uint = 0x891D;
    pub const FLOAT: c_uint = 0x1406;
    pub const FLOAT16_MAT2_AMD: c_uint = 0x91C5;
    pub const FLOAT16_MAT2x3_AMD: c_uint = 0x91C8;
    pub const FLOAT16_MAT2x4_AMD: c_uint = 0x91C9;
    pub const FLOAT16_MAT3_AMD: c_uint = 0x91C6;
    pub const FLOAT16_MAT3x2_AMD: c_uint = 0x91CA;
    pub const FLOAT16_MAT3x4_AMD: c_uint = 0x91CB;
    pub const FLOAT16_MAT4_AMD: c_uint = 0x91C7;
    pub const FLOAT16_MAT4x2_AMD: c_uint = 0x91CC;
    pub const FLOAT16_MAT4x3_AMD: c_uint = 0x91CD;
    pub const FLOAT16_NV: c_uint = 0x8FF8;
    pub const FLOAT16_VEC2_NV: c_uint = 0x8FF9;
    pub const FLOAT16_VEC3_NV: c_uint = 0x8FFA;
    pub const FLOAT16_VEC4_NV: c_uint = 0x8FFB;
    pub const FLOAT_32_UNSIGNED_INT_24_8_REV: c_uint = 0x8DAD;
    pub const FLOAT_32_UNSIGNED_INT_24_8_REV_NV: c_uint = 0x8DAD;
    pub const FLOAT_CLEAR_COLOR_VALUE_NV: c_uint = 0x888D;
    pub const FLOAT_MAT2: c_uint = 0x8B5A;
    pub const FLOAT_MAT2_ARB: c_uint = 0x8B5A;
    pub const FLOAT_MAT2x3: c_uint = 0x8B65;
    pub const FLOAT_MAT2x4: c_uint = 0x8B66;
    pub const FLOAT_MAT3: c_uint = 0x8B5B;
    pub const FLOAT_MAT3_ARB: c_uint = 0x8B5B;
    pub const FLOAT_MAT3x2: c_uint = 0x8B67;
    pub const FLOAT_MAT3x4: c_uint = 0x8B68;
    pub const FLOAT_MAT4: c_uint = 0x8B5C;
    pub const FLOAT_MAT4_ARB: c_uint = 0x8B5C;
    pub const FLOAT_MAT4x2: c_uint = 0x8B69;
    pub const FLOAT_MAT4x3: c_uint = 0x8B6A;
    pub const FLOAT_R16_NV: c_uint = 0x8884;
    pub const FLOAT_R32_NV: c_uint = 0x8885;
    pub const FLOAT_RG16_NV: c_uint = 0x8886;
    pub const FLOAT_RG32_NV: c_uint = 0x8887;
    pub const FLOAT_RGB16_NV: c_uint = 0x8888;
    pub const FLOAT_RGB32_NV: c_uint = 0x8889;
    pub const FLOAT_RGBA16_NV: c_uint = 0x888A;
    pub const FLOAT_RGBA32_NV: c_uint = 0x888B;
    pub const FLOAT_RGBA_MODE_NV: c_uint = 0x888E;
    pub const FLOAT_RGBA_NV: c_uint = 0x8883;
    pub const FLOAT_RGB_NV: c_uint = 0x8882;
    pub const FLOAT_RG_NV: c_uint = 0x8881;
    pub const FLOAT_R_NV: c_uint = 0x8880;
    pub const FLOAT_VEC2: c_uint = 0x8B50;
    pub const FLOAT_VEC2_ARB: c_uint = 0x8B50;
    pub const FLOAT_VEC3: c_uint = 0x8B51;
    pub const FLOAT_VEC3_ARB: c_uint = 0x8B51;
    pub const FLOAT_VEC4: c_uint = 0x8B52;
    pub const FLOAT_VEC4_ARB: c_uint = 0x8B52;
    pub const FOG: c_uint = 0x0B60;
    pub const FOG_COORDINATE_ARRAY_BUFFER_BINDING_ARB: c_uint = 0x889D;
    pub const FOG_COORDINATE_ARRAY_EXT: c_uint = 0x8457;
    pub const FOG_COORDINATE_ARRAY_LIST_IBM: c_uint = 103076;
    pub const FOG_COORDINATE_ARRAY_LIST_STRIDE_IBM: c_uint = 103086;
    pub const FOG_COORDINATE_ARRAY_POINTER_EXT: c_uint = 0x8456;
    pub const FOG_COORDINATE_ARRAY_STRIDE_EXT: c_uint = 0x8455;
    pub const FOG_COORDINATE_ARRAY_TYPE_EXT: c_uint = 0x8454;
    pub const FOG_COORDINATE_EXT: c_uint = 0x8451;
    pub const FOG_COORDINATE_SOURCE_EXT: c_uint = 0x8450;
    pub const FOG_COORD_ARRAY_ADDRESS_NV: c_uint = 0x8F28;
    pub const FOG_COORD_ARRAY_LENGTH_NV: c_uint = 0x8F32;
    pub const FOG_DISTANCE_MODE_NV: c_uint = 0x855A;
    pub const FOG_FUNC_POINTS_SGIS: c_uint = 0x812B;
    pub const FOG_FUNC_SGIS: c_uint = 0x812A;
    pub const FOG_OFFSET_SGIX: c_uint = 0x8198;
    pub const FOG_OFFSET_VALUE_SGIX: c_uint = 0x8199;
    pub const FOG_SPECULAR_TEXTURE_WIN: c_uint = 0x80EC;
    pub const FONT_ASCENDER_BIT_NV: c_uint = 0x00200000;
    pub const FONT_DESCENDER_BIT_NV: c_uint = 0x00400000;
    pub const FONT_GLYPHS_AVAILABLE_NV: c_uint = 0x9368;
    pub const FONT_HAS_KERNING_BIT_NV: c_uint = 0x10000000;
    pub const FONT_HEIGHT_BIT_NV: c_uint = 0x00800000;
    pub const FONT_MAX_ADVANCE_HEIGHT_BIT_NV: c_uint = 0x02000000;
    pub const FONT_MAX_ADVANCE_WIDTH_BIT_NV: c_uint = 0x01000000;
    pub const FONT_NUM_GLYPH_INDICES_BIT_NV: c_uint = 0x20000000;
    pub const FONT_TARGET_UNAVAILABLE_NV: c_uint = 0x9369;
    pub const FONT_UNAVAILABLE_NV: c_uint = 0x936A;
    pub const FONT_UNDERLINE_POSITION_BIT_NV: c_uint = 0x04000000;
    pub const FONT_UNDERLINE_THICKNESS_BIT_NV: c_uint = 0x08000000;
    pub const FONT_UNINTELLIGIBLE_NV: c_uint = 0x936B;
    pub const FONT_UNITS_PER_EM_BIT_NV: c_uint = 0x00100000;
    pub const FONT_X_MAX_BOUNDS_BIT_NV: c_uint = 0x00040000;
    pub const FONT_X_MIN_BOUNDS_BIT_NV: c_uint = 0x00010000;
    pub const FONT_Y_MAX_BOUNDS_BIT_NV: c_uint = 0x00080000;
    pub const FONT_Y_MIN_BOUNDS_BIT_NV: c_uint = 0x00020000;
    pub const FORCE_BLUE_TO_ONE_NV: c_uint = 0x8860;
    pub const FORMAT_SUBSAMPLE_244_244_OML: c_uint = 0x8983;
    pub const FORMAT_SUBSAMPLE_24_24_OML: c_uint = 0x8982;
    pub const FRACTIONAL_EVEN: c_uint = 0x8E7C;
    pub const FRACTIONAL_ODD: c_uint = 0x8E7B;
    pub const FRAGMENT_COLOR_EXT: c_uint = 0x834C;
    pub const FRAGMENT_COLOR_MATERIAL_FACE_SGIX: c_uint = 0x8402;
    pub const FRAGMENT_COLOR_MATERIAL_PARAMETER_SGIX: c_uint = 0x8403;
    pub const FRAGMENT_COLOR_MATERIAL_SGIX: c_uint = 0x8401;
    pub const FRAGMENT_COVERAGE_COLOR_NV: c_uint = 0x92DE;
    pub const FRAGMENT_COVERAGE_TO_COLOR_NV: c_uint = 0x92DD;
    pub const FRAGMENT_DEPTH_EXT: c_uint = 0x8452;
    pub const FRAGMENT_INPUT_NV: c_uint = 0x936D;
    pub const FRAGMENT_INTERPOLATION_OFFSET_BITS: c_uint = 0x8E5D;
    pub const FRAGMENT_LIGHT0_SGIX: c_uint = 0x840C;
    pub const FRAGMENT_LIGHT1_SGIX: c_uint = 0x840D;
    pub const FRAGMENT_LIGHT2_SGIX: c_uint = 0x840E;
    pub const FRAGMENT_LIGHT3_SGIX: c_uint = 0x840F;
    pub const FRAGMENT_LIGHT4_SGIX: c_uint = 0x8410;
    pub const FRAGMENT_LIGHT5_SGIX: c_uint = 0x8411;
    pub const FRAGMENT_LIGHT6_SGIX: c_uint = 0x8412;
    pub const FRAGMENT_LIGHT7_SGIX: c_uint = 0x8413;
    pub const FRAGMENT_LIGHTING_SGIX: c_uint = 0x8400;
    pub const FRAGMENT_LIGHT_MODEL_AMBIENT_SGIX: c_uint = 0x840A;
    pub const FRAGMENT_LIGHT_MODEL_LOCAL_VIEWER_SGIX: c_uint = 0x8408;
    pub const FRAGMENT_LIGHT_MODEL_NORMAL_INTERPOLATION_SGIX: c_uint = 0x840B;
    pub const FRAGMENT_LIGHT_MODEL_TWO_SIDE_SGIX: c_uint = 0x8409;
    pub const FRAGMENT_MATERIAL_EXT: c_uint = 0x8349;
    pub const FRAGMENT_NORMAL_EXT: c_uint = 0x834A;
    pub const FRAGMENT_PROGRAM_ARB: c_uint = 0x8804;
    pub const FRAGMENT_PROGRAM_BINDING_NV: c_uint = 0x8873;
    pub const FRAGMENT_PROGRAM_INTERPOLATION_OFFSET_BITS_NV: c_uint = 0x8E5D;
    pub const FRAGMENT_PROGRAM_NV: c_uint = 0x8870;
    pub const FRAGMENT_PROGRAM_PARAMETER_BUFFER_NV: c_uint = 0x8DA4;
    pub const FRAGMENT_SHADER: c_uint = 0x8B30;
    pub const FRAGMENT_SHADER_ARB: c_uint = 0x8B30;
    pub const FRAGMENT_SHADER_ATI: c_uint = 0x8920;
    pub const FRAGMENT_SHADER_BIT: c_uint = 0x00000002;
    pub const FRAGMENT_SHADER_DERIVATIVE_HINT: c_uint = 0x8B8B;
    pub const FRAGMENT_SHADER_DERIVATIVE_HINT_ARB: c_uint = 0x8B8B;
    pub const FRAGMENT_SHADER_DISCARDS_SAMPLES_EXT: c_uint = 0x8A52;
    pub const FRAGMENT_SHADER_INVOCATIONS: c_uint = 0x82F4;
    pub const FRAGMENT_SHADER_INVOCATIONS_ARB: c_uint = 0x82F4;
    pub const FRAGMENT_SUBROUTINE: c_uint = 0x92EC;
    pub const FRAGMENT_SUBROUTINE_UNIFORM: c_uint = 0x92F2;
    pub const FRAGMENT_TEXTURE: c_uint = 0x829F;
    pub const FRAMEBUFFER: c_uint = 0x8D40;
    pub const FRAMEBUFFER_ATTACHMENT_ALPHA_SIZE: c_uint = 0x8215;
    pub const FRAMEBUFFER_ATTACHMENT_BLUE_SIZE: c_uint = 0x8214;
    pub const FRAMEBUFFER_ATTACHMENT_COLOR_ENCODING: c_uint = 0x8210;
    pub const FRAMEBUFFER_ATTACHMENT_COMPONENT_TYPE: c_uint = 0x8211;
    pub const FRAMEBUFFER_ATTACHMENT_DEPTH_SIZE: c_uint = 0x8216;
    pub const FRAMEBUFFER_ATTACHMENT_GREEN_SIZE: c_uint = 0x8213;
    pub const FRAMEBUFFER_ATTACHMENT_LAYERED: c_uint = 0x8DA7;
    pub const FRAMEBUFFER_ATTACHMENT_LAYERED_ARB: c_uint = 0x8DA7;
    pub const FRAMEBUFFER_ATTACHMENT_LAYERED_EXT: c_uint = 0x8DA7;
    pub const FRAMEBUFFER_ATTACHMENT_OBJECT_NAME: c_uint = 0x8CD1;
    pub const FRAMEBUFFER_ATTACHMENT_OBJECT_NAME_EXT: c_uint = 0x8CD1;
    pub const FRAMEBUFFER_ATTACHMENT_OBJECT_TYPE: c_uint = 0x8CD0;
    pub const FRAMEBUFFER_ATTACHMENT_OBJECT_TYPE_EXT: c_uint = 0x8CD0;
    pub const FRAMEBUFFER_ATTACHMENT_RED_SIZE: c_uint = 0x8212;
    pub const FRAMEBUFFER_ATTACHMENT_STENCIL_SIZE: c_uint = 0x8217;
    pub const FRAMEBUFFER_ATTACHMENT_TEXTURE_3D_ZOFFSET_EXT: c_uint = 0x8CD4;
    pub const FRAMEBUFFER_ATTACHMENT_TEXTURE_BASE_VIEW_INDEX_OVR: c_uint = 0x9632;
    pub const FRAMEBUFFER_ATTACHMENT_TEXTURE_CUBE_MAP_FACE: c_uint = 0x8CD3;
    pub const FRAMEBUFFER_ATTACHMENT_TEXTURE_CUBE_MAP_FACE_EXT: c_uint = 0x8CD3;
    pub const FRAMEBUFFER_ATTACHMENT_TEXTURE_LAYER: c_uint = 0x8CD4;
    pub const FRAMEBUFFER_ATTACHMENT_TEXTURE_LAYER_EXT: c_uint = 0x8CD4;
    pub const FRAMEBUFFER_ATTACHMENT_TEXTURE_LEVEL: c_uint = 0x8CD2;
    pub const FRAMEBUFFER_ATTACHMENT_TEXTURE_LEVEL_EXT: c_uint = 0x8CD2;
    pub const FRAMEBUFFER_ATTACHMENT_TEXTURE_NUM_VIEWS_OVR: c_uint = 0x9630;
    pub const FRAMEBUFFER_BARRIER_BIT: c_uint = 0x00000400;
    pub const FRAMEBUFFER_BARRIER_BIT_EXT: c_uint = 0x00000400;
    pub const FRAMEBUFFER_BINDING: c_uint = 0x8CA6;
    pub const FRAMEBUFFER_BINDING_EXT: c_uint = 0x8CA6;
    pub const FRAMEBUFFER_BLEND: c_uint = 0x828B;
    pub const FRAMEBUFFER_COMPLETE: c_uint = 0x8CD5;
    pub const FRAMEBUFFER_COMPLETE_EXT: c_uint = 0x8CD5;
    pub const FRAMEBUFFER_DEFAULT: c_uint = 0x8218;
    pub const FRAMEBUFFER_DEFAULT_FIXED_SAMPLE_LOCATIONS: c_uint = 0x9314;
    pub const FRAMEBUFFER_DEFAULT_HEIGHT: c_uint = 0x9311;
    pub const FRAMEBUFFER_DEFAULT_LAYERS: c_uint = 0x9312;
    pub const FRAMEBUFFER_DEFAULT_SAMPLES: c_uint = 0x9313;
    pub const FRAMEBUFFER_DEFAULT_WIDTH: c_uint = 0x9310;
    pub const FRAMEBUFFER_EXT: c_uint = 0x8D40;
    pub const FRAMEBUFFER_FLIP_X_MESA: c_uint = 0x8BBC;
    pub const FRAMEBUFFER_FLIP_Y_MESA: c_uint = 0x8BBB;
    pub const FRAMEBUFFER_INCOMPLETE_ATTACHMENT: c_uint = 0x8CD6;
    pub const FRAMEBUFFER_INCOMPLETE_ATTACHMENT_EXT: c_uint = 0x8CD6;
    pub const FRAMEBUFFER_INCOMPLETE_DIMENSIONS_EXT: c_uint = 0x8CD9;
    pub const FRAMEBUFFER_INCOMPLETE_DRAW_BUFFER: c_uint = 0x8CDB;
    pub const FRAMEBUFFER_INCOMPLETE_DRAW_BUFFER_EXT: c_uint = 0x8CDB;
    pub const FRAMEBUFFER_INCOMPLETE_FORMATS_EXT: c_uint = 0x8CDA;
    pub const FRAMEBUFFER_INCOMPLETE_LAYER_COUNT_ARB: c_uint = 0x8DA9;
    pub const FRAMEBUFFER_INCOMPLETE_LAYER_COUNT_EXT: c_uint = 0x8DA9;
    pub const FRAMEBUFFER_INCOMPLETE_LAYER_TARGETS: c_uint = 0x8DA8;
    pub const FRAMEBUFFER_INCOMPLETE_LAYER_TARGETS_ARB: c_uint = 0x8DA8;
    pub const FRAMEBUFFER_INCOMPLETE_LAYER_TARGETS_EXT: c_uint = 0x8DA8;
    pub const FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT: c_uint = 0x8CD7;
    pub const FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT_EXT: c_uint = 0x8CD7;
    pub const FRAMEBUFFER_INCOMPLETE_MULTISAMPLE: c_uint = 0x8D56;
    pub const FRAMEBUFFER_INCOMPLETE_MULTISAMPLE_EXT: c_uint = 0x8D56;
    pub const FRAMEBUFFER_INCOMPLETE_READ_BUFFER: c_uint = 0x8CDC;
    pub const FRAMEBUFFER_INCOMPLETE_READ_BUFFER_EXT: c_uint = 0x8CDC;
    pub const FRAMEBUFFER_INCOMPLETE_VIEW_TARGETS_OVR: c_uint = 0x9633;
    pub const FRAMEBUFFER_PROGRAMMABLE_SAMPLE_LOCATIONS_ARB: c_uint = 0x9342;
    pub const FRAMEBUFFER_PROGRAMMABLE_SAMPLE_LOCATIONS_NV: c_uint = 0x9342;
    pub const FRAMEBUFFER_RENDERABLE: c_uint = 0x8289;
    pub const FRAMEBUFFER_RENDERABLE_LAYERED: c_uint = 0x828A;
    pub const FRAMEBUFFER_SAMPLE_LOCATION_PIXEL_GRID_ARB: c_uint = 0x9343;
    pub const FRAMEBUFFER_SAMPLE_LOCATION_PIXEL_GRID_NV: c_uint = 0x9343;
    pub const FRAMEBUFFER_SRGB: c_uint = 0x8DB9;
    pub const FRAMEBUFFER_SRGB_CAPABLE_EXT: c_uint = 0x8DBA;
    pub const FRAMEBUFFER_SRGB_EXT: c_uint = 0x8DB9;
    pub const FRAMEBUFFER_SWAP_XY_MESA: c_uint = 0x8BBD;
    pub const FRAMEBUFFER_UNDEFINED: c_uint = 0x8219;
    pub const FRAMEBUFFER_UNSUPPORTED: c_uint = 0x8CDD;
    pub const FRAMEBUFFER_UNSUPPORTED_EXT: c_uint = 0x8CDD;
    pub const FRAMEZOOM_FACTOR_SGIX: c_uint = 0x818C;
    pub const FRAMEZOOM_SGIX: c_uint = 0x818B;
    pub const FRAME_NV: c_uint = 0x8E26;
    pub const FRONT: c_uint = 0x0404;
    pub const FRONT_AND_BACK: c_uint = 0x0408;
    pub const FRONT_FACE: c_uint = 0x0B46;
    pub const FRONT_FACE_COMMAND_NV: c_uint = 0x0012;
    pub const FRONT_LEFT: c_uint = 0x0400;
    pub const FRONT_RIGHT: c_uint = 0x0401;
    pub const FULL_RANGE_EXT: c_uint = 0x87E1;
    pub const FULL_STIPPLE_HINT_PGI: c_uint = 0x1A219;
    pub const FULL_SUPPORT: c_uint = 0x82B7;
    pub const FUNC_ADD: c_uint = 0x8006;
    pub const FUNC_ADD_EXT: c_uint = 0x8006;
    pub const FUNC_REVERSE_SUBTRACT: c_uint = 0x800B;
    pub const FUNC_REVERSE_SUBTRACT_EXT: c_uint = 0x800B;
    pub const FUNC_SUBTRACT: c_uint = 0x800A;
    pub const FUNC_SUBTRACT_EXT: c_uint = 0x800A;
    pub const GENERATE_MIPMAP_HINT_SGIS: c_uint = 0x8192;
    pub const GENERATE_MIPMAP_SGIS: c_uint = 0x8191;
    pub const GENERIC_ATTRIB_NV: c_uint = 0x8C7D;
    pub const GEOMETRY_DEFORMATION_BIT_SGIX: c_uint = 0x00000002;
    pub const GEOMETRY_DEFORMATION_SGIX: c_uint = 0x8194;
    pub const GEOMETRY_INPUT_TYPE: c_uint = 0x8917;
    pub const GEOMETRY_INPUT_TYPE_ARB: c_uint = 0x8DDB;
    pub const GEOMETRY_INPUT_TYPE_EXT: c_uint = 0x8DDB;
    pub const GEOMETRY_OUTPUT_TYPE: c_uint = 0x8918;
    pub const GEOMETRY_OUTPUT_TYPE_ARB: c_uint = 0x8DDC;
    pub const GEOMETRY_OUTPUT_TYPE_EXT: c_uint = 0x8DDC;
    pub const GEOMETRY_PROGRAM_NV: c_uint = 0x8C26;
    pub const GEOMETRY_PROGRAM_PARAMETER_BUFFER_NV: c_uint = 0x8DA3;
    pub const GEOMETRY_SHADER: c_uint = 0x8DD9;
    pub const GEOMETRY_SHADER_ARB: c_uint = 0x8DD9;
    pub const GEOMETRY_SHADER_BIT: c_uint = 0x00000004;
    pub const GEOMETRY_SHADER_EXT: c_uint = 0x8DD9;
    pub const GEOMETRY_SHADER_INVOCATIONS: c_uint = 0x887F;
    pub const GEOMETRY_SHADER_PRIMITIVES_EMITTED: c_uint = 0x82F3;
    pub const GEOMETRY_SHADER_PRIMITIVES_EMITTED_ARB: c_uint = 0x82F3;
    pub const GEOMETRY_SUBROUTINE: c_uint = 0x92EB;
    pub const GEOMETRY_SUBROUTINE_UNIFORM: c_uint = 0x92F1;
    pub const GEOMETRY_TEXTURE: c_uint = 0x829E;
    pub const GEOMETRY_VERTICES_OUT: c_uint = 0x8916;
    pub const GEOMETRY_VERTICES_OUT_ARB: c_uint = 0x8DDA;
    pub const GEOMETRY_VERTICES_OUT_EXT: c_uint = 0x8DDA;
    pub const GEQUAL: c_uint = 0x0206;
    pub const GET_TEXTURE_IMAGE_FORMAT: c_uint = 0x8291;
    pub const GET_TEXTURE_IMAGE_TYPE: c_uint = 0x8292;
    pub const GLOBAL_ALPHA_FACTOR_SUN: c_uint = 0x81DA;
    pub const GLOBAL_ALPHA_SUN: c_uint = 0x81D9;
    pub const GLYPH_HAS_KERNING_BIT_NV: c_uint = 0x100;
    pub const GLYPH_HEIGHT_BIT_NV: c_uint = 0x02;
    pub const GLYPH_HORIZONTAL_BEARING_ADVANCE_BIT_NV: c_uint = 0x10;
    pub const GLYPH_HORIZONTAL_BEARING_X_BIT_NV: c_uint = 0x04;
    pub const GLYPH_HORIZONTAL_BEARING_Y_BIT_NV: c_uint = 0x08;
    pub const GLYPH_VERTICAL_BEARING_ADVANCE_BIT_NV: c_uint = 0x80;
    pub const GLYPH_VERTICAL_BEARING_X_BIT_NV: c_uint = 0x20;
    pub const GLYPH_VERTICAL_BEARING_Y_BIT_NV: c_uint = 0x40;
    pub const GLYPH_WIDTH_BIT_NV: c_uint = 0x01;
    pub const GPU_ADDRESS_NV: c_uint = 0x8F34;
    pub const GPU_MEMORY_INFO_CURRENT_AVAILABLE_VIDMEM_NVX: c_uint = 0x9049;
    pub const GPU_MEMORY_INFO_DEDICATED_VIDMEM_NVX: c_uint = 0x9047;
    pub const GPU_MEMORY_INFO_EVICTED_MEMORY_NVX: c_uint = 0x904B;
    pub const GPU_MEMORY_INFO_EVICTION_COUNT_NVX: c_uint = 0x904A;
    pub const GPU_MEMORY_INFO_TOTAL_AVAILABLE_MEMORY_NVX: c_uint = 0x9048;
    pub const GREATER: c_uint = 0x0204;
    pub const GREEN: c_uint = 0x1904;
    pub const GREEN_BIT_ATI: c_uint = 0x00000002;
    pub const GREEN_INTEGER: c_uint = 0x8D95;
    pub const GREEN_INTEGER_EXT: c_uint = 0x8D95;
    pub const GREEN_MAX_CLAMP_INGR: c_uint = 0x8565;
    pub const GREEN_MIN_CLAMP_INGR: c_uint = 0x8561;
    pub const GREEN_NV: c_uint = 0x1904;
    pub const GUILTY_CONTEXT_RESET: c_uint = 0x8253;
    pub const GUILTY_CONTEXT_RESET_ARB: c_uint = 0x8253;
    pub const HALF_APPLE: c_uint = 0x140B;
    pub const HALF_BIAS_NEGATE_NV: c_uint = 0x853B;
    pub const HALF_BIAS_NORMAL_NV: c_uint = 0x853A;
    pub const HALF_BIT_ATI: c_uint = 0x00000008;
    pub const HALF_FLOAT: c_uint = 0x140B;
    pub const HALF_FLOAT_ARB: c_uint = 0x140B;
    pub const HALF_FLOAT_NV: c_uint = 0x140B;
    pub const HANDLE_TYPE_D3D11_IMAGE_EXT: c_uint = 0x958B;
    pub const HANDLE_TYPE_D3D11_IMAGE_KMT_EXT: c_uint = 0x958C;
    pub const HANDLE_TYPE_D3D12_FENCE_EXT: c_uint = 0x9594;
    pub const HANDLE_TYPE_D3D12_RESOURCE_EXT: c_uint = 0x958A;
    pub const HANDLE_TYPE_D3D12_TILEPOOL_EXT: c_uint = 0x9589;
    pub const HANDLE_TYPE_OPAQUE_FD_EXT: c_uint = 0x9586;
    pub const HANDLE_TYPE_OPAQUE_WIN32_EXT: c_uint = 0x9587;
    pub const HANDLE_TYPE_OPAQUE_WIN32_KMT_EXT: c_uint = 0x9588;
    pub const HARDLIGHT_KHR: c_uint = 0x929B;
    pub const HARDLIGHT_NV: c_uint = 0x929B;
    pub const HARDMIX_NV: c_uint = 0x92A9;
    pub const HIGH_FLOAT: c_uint = 0x8DF2;
    pub const HIGH_INT: c_uint = 0x8DF5;
    pub const HILO16_NV: c_uint = 0x86F8;
    pub const HILO8_NV: c_uint = 0x885E;
    pub const HILO_NV: c_uint = 0x86F4;
    pub const HISTOGRAM_ALPHA_SIZE_EXT: c_uint = 0x802B;
    pub const HISTOGRAM_BLUE_SIZE_EXT: c_uint = 0x802A;
    pub const HISTOGRAM_EXT: c_uint = 0x8024;
    pub const HISTOGRAM_FORMAT_EXT: c_uint = 0x8027;
    pub const HISTOGRAM_GREEN_SIZE_EXT: c_uint = 0x8029;
    pub const HISTOGRAM_LUMINANCE_SIZE_EXT: c_uint = 0x802C;
    pub const HISTOGRAM_RED_SIZE_EXT: c_uint = 0x8028;
    pub const HISTOGRAM_SINK_EXT: c_uint = 0x802D;
    pub const HISTOGRAM_WIDTH_EXT: c_uint = 0x8026;
    pub const HI_BIAS_NV: c_uint = 0x8714;
    pub const HI_SCALE_NV: c_uint = 0x870E;
    pub const HORIZONTAL_LINE_TO_NV: c_uint = 0x06;
    pub const HSL_COLOR_KHR: c_uint = 0x92AF;
    pub const HSL_COLOR_NV: c_uint = 0x92AF;
    pub const HSL_HUE_KHR: c_uint = 0x92AD;
    pub const HSL_HUE_NV: c_uint = 0x92AD;
    pub const HSL_LUMINOSITY_KHR: c_uint = 0x92B0;
    pub const HSL_LUMINOSITY_NV: c_uint = 0x92B0;
    pub const HSL_SATURATION_KHR: c_uint = 0x92AE;
    pub const HSL_SATURATION_NV: c_uint = 0x92AE;
    pub const IDENTITY_NV: c_uint = 0x862A;
    pub const IGNORE_BORDER_HP: c_uint = 0x8150;
    pub const IMAGE_1D: c_uint = 0x904C;
    pub const IMAGE_1D_ARRAY: c_uint = 0x9052;
    pub const IMAGE_1D_ARRAY_EXT: c_uint = 0x9052;
    pub const IMAGE_1D_EXT: c_uint = 0x904C;
    pub const IMAGE_2D: c_uint = 0x904D;
    pub const IMAGE_2D_ARRAY: c_uint = 0x9053;
    pub const IMAGE_2D_ARRAY_EXT: c_uint = 0x9053;
    pub const IMAGE_2D_EXT: c_uint = 0x904D;
    pub const IMAGE_2D_MULTISAMPLE: c_uint = 0x9055;
    pub const IMAGE_2D_MULTISAMPLE_ARRAY: c_uint = 0x9056;
    pub const IMAGE_2D_MULTISAMPLE_ARRAY_EXT: c_uint = 0x9056;
    pub const IMAGE_2D_MULTISAMPLE_EXT: c_uint = 0x9055;
    pub const IMAGE_2D_RECT: c_uint = 0x904F;
    pub const IMAGE_2D_RECT_EXT: c_uint = 0x904F;
    pub const IMAGE_3D: c_uint = 0x904E;
    pub const IMAGE_3D_EXT: c_uint = 0x904E;
    pub const IMAGE_BINDING_ACCESS: c_uint = 0x8F3E;
    pub const IMAGE_BINDING_ACCESS_EXT: c_uint = 0x8F3E;
    pub const IMAGE_BINDING_FORMAT: c_uint = 0x906E;
    pub const IMAGE_BINDING_FORMAT_EXT: c_uint = 0x906E;
    pub const IMAGE_BINDING_LAYER: c_uint = 0x8F3D;
    pub const IMAGE_BINDING_LAYERED: c_uint = 0x8F3C;
    pub const IMAGE_BINDING_LAYERED_EXT: c_uint = 0x8F3C;
    pub const IMAGE_BINDING_LAYER_EXT: c_uint = 0x8F3D;
    pub const IMAGE_BINDING_LEVEL: c_uint = 0x8F3B;
    pub const IMAGE_BINDING_LEVEL_EXT: c_uint = 0x8F3B;
    pub const IMAGE_BINDING_NAME: c_uint = 0x8F3A;
    pub const IMAGE_BINDING_NAME_EXT: c_uint = 0x8F3A;
    pub const IMAGE_BUFFER: c_uint = 0x9051;
    pub const IMAGE_BUFFER_EXT: c_uint = 0x9051;
    pub const IMAGE_CLASS_10_10_10_2: c_uint = 0x82C3;
    pub const IMAGE_CLASS_11_11_10: c_uint = 0x82C2;
    pub const IMAGE_CLASS_1_X_16: c_uint = 0x82BE;
    pub const IMAGE_CLASS_1_X_32: c_uint = 0x82BB;
    pub const IMAGE_CLASS_1_X_8: c_uint = 0x82C1;
    pub const IMAGE_CLASS_2_X_16: c_uint = 0x82BD;
    pub const IMAGE_CLASS_2_X_32: c_uint = 0x82BA;
    pub const IMAGE_CLASS_2_X_8: c_uint = 0x82C0;
    pub const IMAGE_CLASS_4_X_16: c_uint = 0x82BC;
    pub const IMAGE_CLASS_4_X_32: c_uint = 0x82B9;
    pub const IMAGE_CLASS_4_X_8: c_uint = 0x82BF;
    pub const IMAGE_COMPATIBILITY_CLASS: c_uint = 0x82A8;
    pub const IMAGE_CUBE: c_uint = 0x9050;
    pub const IMAGE_CUBE_EXT: c_uint = 0x9050;
    pub const IMAGE_CUBE_MAP_ARRAY: c_uint = 0x9054;
    pub const IMAGE_CUBE_MAP_ARRAY_EXT: c_uint = 0x9054;
    pub const IMAGE_CUBIC_WEIGHT_HP: c_uint = 0x815E;
    pub const IMAGE_FORMAT_COMPATIBILITY_BY_CLASS: c_uint = 0x90C9;
    pub const IMAGE_FORMAT_COMPATIBILITY_BY_SIZE: c_uint = 0x90C8;
    pub const IMAGE_FORMAT_COMPATIBILITY_TYPE: c_uint = 0x90C7;
    pub const IMAGE_MAG_FILTER_HP: c_uint = 0x815C;
    pub const IMAGE_MIN_FILTER_HP: c_uint = 0x815D;
    pub const IMAGE_PIXEL_FORMAT: c_uint = 0x82A9;
    pub const IMAGE_PIXEL_TYPE: c_uint = 0x82AA;
    pub const IMAGE_ROTATE_ANGLE_HP: c_uint = 0x8159;
    pub const IMAGE_ROTATE_ORIGIN_X_HP: c_uint = 0x815A;
    pub const IMAGE_ROTATE_ORIGIN_Y_HP: c_uint = 0x815B;
    pub const IMAGE_SCALE_X_HP: c_uint = 0x8155;
    pub const IMAGE_SCALE_Y_HP: c_uint = 0x8156;
    pub const IMAGE_TEXEL_SIZE: c_uint = 0x82A7;
    pub const IMAGE_TRANSFORM_2D_HP: c_uint = 0x8161;
    pub const IMAGE_TRANSLATE_X_HP: c_uint = 0x8157;
    pub const IMAGE_TRANSLATE_Y_HP: c_uint = 0x8158;
    pub const IMPLEMENTATION_COLOR_READ_FORMAT: c_uint = 0x8B9B;
    pub const IMPLEMENTATION_COLOR_READ_FORMAT_OES: c_uint = 0x8B9B;
    pub const IMPLEMENTATION_COLOR_READ_TYPE: c_uint = 0x8B9A;
    pub const IMPLEMENTATION_COLOR_READ_TYPE_OES: c_uint = 0x8B9A;
    pub const INCLUSIVE_EXT: c_uint = 0x8F10;
    pub const INCR: c_uint = 0x1E02;
    pub const INCR_WRAP: c_uint = 0x8507;
    pub const INCR_WRAP_EXT: c_uint = 0x8507;
    pub const INDEX_ARRAY_ADDRESS_NV: c_uint = 0x8F24;
    pub const INDEX_ARRAY_BUFFER_BINDING_ARB: c_uint = 0x8899;
    pub const INDEX_ARRAY_COUNT_EXT: c_uint = 0x8087;
    pub const INDEX_ARRAY_EXT: c_uint = 0x8077;
    pub const INDEX_ARRAY_LENGTH_NV: c_uint = 0x8F2E;
    pub const INDEX_ARRAY_LIST_IBM: c_uint = 103073;
    pub const INDEX_ARRAY_LIST_STRIDE_IBM: c_uint = 103083;
    pub const INDEX_ARRAY_POINTER_EXT: c_uint = 0x8091;
    pub const INDEX_ARRAY_STRIDE_EXT: c_uint = 0x8086;
    pub const INDEX_ARRAY_TYPE_EXT: c_uint = 0x8085;
    pub const INDEX_BIT_PGI: c_uint = 0x00080000;
    pub const INDEX_MATERIAL_EXT: c_uint = 0x81B8;
    pub const INDEX_MATERIAL_FACE_EXT: c_uint = 0x81BA;
    pub const INDEX_MATERIAL_PARAMETER_EXT: c_uint = 0x81B9;
    pub const INDEX_TEST_EXT: c_uint = 0x81B5;
    pub const INDEX_TEST_FUNC_EXT: c_uint = 0x81B6;
    pub const INDEX_TEST_REF_EXT: c_uint = 0x81B7;
    pub const INFO_LOG_LENGTH: c_uint = 0x8B84;
    pub const INNOCENT_CONTEXT_RESET: c_uint = 0x8254;
    pub const INNOCENT_CONTEXT_RESET_ARB: c_uint = 0x8254;
    pub const INSTRUMENT_BUFFER_POINTER_SGIX: c_uint = 0x8180;
    pub const INSTRUMENT_MEASUREMENTS_SGIX: c_uint = 0x8181;
    pub const INT: c_uint = 0x1404;
    pub const INT16_NV: c_uint = 0x8FE4;
    pub const INT16_VEC2_NV: c_uint = 0x8FE5;
    pub const INT16_VEC3_NV: c_uint = 0x8FE6;
    pub const INT16_VEC4_NV: c_uint = 0x8FE7;
    pub const INT64_ARB: c_uint = 0x140E;
    pub const INT64_NV: c_uint = 0x140E;
    pub const INT64_VEC2_ARB: c_uint = 0x8FE9;
    pub const INT64_VEC2_NV: c_uint = 0x8FE9;
    pub const INT64_VEC3_ARB: c_uint = 0x8FEA;
    pub const INT64_VEC3_NV: c_uint = 0x8FEA;
    pub const INT64_VEC4_ARB: c_uint = 0x8FEB;
    pub const INT64_VEC4_NV: c_uint = 0x8FEB;
    pub const INT8_NV: c_uint = 0x8FE0;
    pub const INT8_VEC2_NV: c_uint = 0x8FE1;
    pub const INT8_VEC3_NV: c_uint = 0x8FE2;
    pub const INT8_VEC4_NV: c_uint = 0x8FE3;
    pub const INTENSITY12_EXT: c_uint = 0x804C;
    pub const INTENSITY16F_ARB: c_uint = 0x881D;
    pub const INTENSITY16I_EXT: c_uint = 0x8D8B;
    pub const INTENSITY16UI_EXT: c_uint = 0x8D79;
    pub const INTENSITY16_EXT: c_uint = 0x804D;
    pub const INTENSITY16_SNORM: c_uint = 0x901B;
    pub const INTENSITY32F_ARB: c_uint = 0x8817;
    pub const INTENSITY32I_EXT: c_uint = 0x8D85;
    pub const INTENSITY32UI_EXT: c_uint = 0x8D73;
    pub const INTENSITY4_EXT: c_uint = 0x804A;
    pub const INTENSITY8I_EXT: c_uint = 0x8D91;
    pub const INTENSITY8UI_EXT: c_uint = 0x8D7F;
    pub const INTENSITY8_EXT: c_uint = 0x804B;
    pub const INTENSITY8_SNORM: c_uint = 0x9017;
    pub const INTENSITY_EXT: c_uint = 0x8049;
    pub const INTENSITY_FLOAT16_APPLE: c_uint = 0x881D;
    pub const INTENSITY_FLOAT16_ATI: c_uint = 0x881D;
    pub const INTENSITY_FLOAT32_APPLE: c_uint = 0x8817;
    pub const INTENSITY_FLOAT32_ATI: c_uint = 0x8817;
    pub const INTENSITY_SNORM: c_uint = 0x9013;
    pub const INTERLACE_OML: c_uint = 0x8980;
    pub const INTERLACE_READ_INGR: c_uint = 0x8568;
    pub const INTERLACE_READ_OML: c_uint = 0x8981;
    pub const INTERLACE_SGIX: c_uint = 0x8094;
    pub const INTERLEAVED_ATTRIBS: c_uint = 0x8C8C;
    pub const INTERLEAVED_ATTRIBS_EXT: c_uint = 0x8C8C;
    pub const INTERLEAVED_ATTRIBS_NV: c_uint = 0x8C8C;
    pub const INTERNALFORMAT_ALPHA_SIZE: c_uint = 0x8274;
    pub const INTERNALFORMAT_ALPHA_TYPE: c_uint = 0x827B;
    pub const INTERNALFORMAT_BLUE_SIZE: c_uint = 0x8273;
    pub const INTERNALFORMAT_BLUE_TYPE: c_uint = 0x827A;
    pub const INTERNALFORMAT_DEPTH_SIZE: c_uint = 0x8275;
    pub const INTERNALFORMAT_DEPTH_TYPE: c_uint = 0x827C;
    pub const INTERNALFORMAT_GREEN_SIZE: c_uint = 0x8272;
    pub const INTERNALFORMAT_GREEN_TYPE: c_uint = 0x8279;
    pub const INTERNALFORMAT_PREFERRED: c_uint = 0x8270;
    pub const INTERNALFORMAT_RED_SIZE: c_uint = 0x8271;
    pub const INTERNALFORMAT_RED_TYPE: c_uint = 0x8278;
    pub const INTERNALFORMAT_SHARED_SIZE: c_uint = 0x8277;
    pub const INTERNALFORMAT_STENCIL_SIZE: c_uint = 0x8276;
    pub const INTERNALFORMAT_STENCIL_TYPE: c_uint = 0x827D;
    pub const INTERNALFORMAT_SUPPORTED: c_uint = 0x826F;
    pub const INTERPOLATE_ARB: c_uint = 0x8575;
    pub const INTERPOLATE_EXT: c_uint = 0x8575;
    pub const INT_2_10_10_10_REV: c_uint = 0x8D9F;
    pub const INT_IMAGE_1D: c_uint = 0x9057;
    pub const INT_IMAGE_1D_ARRAY: c_uint = 0x905D;
    pub const INT_IMAGE_1D_ARRAY_EXT: c_uint = 0x905D;
    pub const INT_IMAGE_1D_EXT: c_uint = 0x9057;
    pub const INT_IMAGE_2D: c_uint = 0x9058;
    pub const INT_IMAGE_2D_ARRAY: c_uint = 0x905E;
    pub const INT_IMAGE_2D_ARRAY_EXT: c_uint = 0x905E;
    pub const INT_IMAGE_2D_EXT: c_uint = 0x9058;
    pub const INT_IMAGE_2D_MULTISAMPLE: c_uint = 0x9060;
    pub const INT_IMAGE_2D_MULTISAMPLE_ARRAY: c_uint = 0x9061;
    pub const INT_IMAGE_2D_MULTISAMPLE_ARRAY_EXT: c_uint = 0x9061;
    pub const INT_IMAGE_2D_MULTISAMPLE_EXT: c_uint = 0x9060;
    pub const INT_IMAGE_2D_RECT: c_uint = 0x905A;
    pub const INT_IMAGE_2D_RECT_EXT: c_uint = 0x905A;
    pub const INT_IMAGE_3D: c_uint = 0x9059;
    pub const INT_IMAGE_3D_EXT: c_uint = 0x9059;
    pub const INT_IMAGE_BUFFER: c_uint = 0x905C;
    pub const INT_IMAGE_BUFFER_EXT: c_uint = 0x905C;
    pub const INT_IMAGE_CUBE: c_uint = 0x905B;
    pub const INT_IMAGE_CUBE_EXT: c_uint = 0x905B;
    pub const INT_IMAGE_CUBE_MAP_ARRAY: c_uint = 0x905F;
    pub const INT_IMAGE_CUBE_MAP_ARRAY_EXT: c_uint = 0x905F;
    pub const INT_SAMPLER_1D: c_uint = 0x8DC9;
    pub const INT_SAMPLER_1D_ARRAY: c_uint = 0x8DCE;
    pub const INT_SAMPLER_1D_ARRAY_EXT: c_uint = 0x8DCE;
    pub const INT_SAMPLER_1D_EXT: c_uint = 0x8DC9;
    pub const INT_SAMPLER_2D: c_uint = 0x8DCA;
    pub const INT_SAMPLER_2D_ARRAY: c_uint = 0x8DCF;
    pub const INT_SAMPLER_2D_ARRAY_EXT: c_uint = 0x8DCF;
    pub const INT_SAMPLER_2D_EXT: c_uint = 0x8DCA;
    pub const INT_SAMPLER_2D_MULTISAMPLE: c_uint = 0x9109;
    pub const INT_SAMPLER_2D_MULTISAMPLE_ARRAY: c_uint = 0x910C;
    pub const INT_SAMPLER_2D_RECT: c_uint = 0x8DCD;
    pub const INT_SAMPLER_2D_RECT_EXT: c_uint = 0x8DCD;
    pub const INT_SAMPLER_3D: c_uint = 0x8DCB;
    pub const INT_SAMPLER_3D_EXT: c_uint = 0x8DCB;
    pub const INT_SAMPLER_BUFFER: c_uint = 0x8DD0;
    pub const INT_SAMPLER_BUFFER_AMD: c_uint = 0x9002;
    pub const INT_SAMPLER_BUFFER_EXT: c_uint = 0x8DD0;
    pub const INT_SAMPLER_CUBE: c_uint = 0x8DCC;
    pub const INT_SAMPLER_CUBE_EXT: c_uint = 0x8DCC;
    pub const INT_SAMPLER_CUBE_MAP_ARRAY: c_uint = 0x900E;
    pub const INT_SAMPLER_CUBE_MAP_ARRAY_ARB: c_uint = 0x900E;
    pub const INT_SAMPLER_RENDERBUFFER_NV: c_uint = 0x8E57;
    pub const INT_VEC2: c_uint = 0x8B53;
    pub const INT_VEC2_ARB: c_uint = 0x8B53;
    pub const INT_VEC3: c_uint = 0x8B54;
    pub const INT_VEC3_ARB: c_uint = 0x8B54;
    pub const INT_VEC4: c_uint = 0x8B55;
    pub const INT_VEC4_ARB: c_uint = 0x8B55;
    pub const INVALID_ENUM: c_uint = 0x0500;
    pub const INVALID_FRAMEBUFFER_OPERATION: c_uint = 0x0506;
    pub const INVALID_FRAMEBUFFER_OPERATION_EXT: c_uint = 0x0506;
    pub const INVALID_INDEX: c_uint = 0xFFFFFFFF;
    pub const INVALID_OPERATION: c_uint = 0x0502;
    pub const INVALID_VALUE: c_uint = 0x0501;
    pub const INVARIANT_DATATYPE_EXT: c_uint = 0x87EB;
    pub const INVARIANT_EXT: c_uint = 0x87C2;
    pub const INVARIANT_VALUE_EXT: c_uint = 0x87EA;
    pub const INVERSE_NV: c_uint = 0x862B;
    pub const INVERSE_TRANSPOSE_NV: c_uint = 0x862D;
    pub const INVERT: c_uint = 0x150A;
    pub const INVERTED_SCREEN_W_REND: c_uint = 0x8491;
    pub const INVERT_OVG_NV: c_uint = 0x92B4;
    pub const INVERT_RGB_NV: c_uint = 0x92A3;
    pub const IR_INSTRUMENT1_SGIX: c_uint = 0x817F;
    pub const ISOLINES: c_uint = 0x8E7A;
    pub const IS_PER_PATCH: c_uint = 0x92E7;
    pub const IS_ROW_MAJOR: c_uint = 0x9300;
    pub const ITALIC_BIT_NV: c_uint = 0x02;
    pub const IUI_N3F_V2F_EXT: c_uint = 0x81AF;
    pub const IUI_N3F_V3F_EXT: c_uint = 0x81B0;
    pub const IUI_V2F_EXT: c_uint = 0x81AD;
    pub const IUI_V3F_EXT: c_uint = 0x81AE;
    pub const KEEP: c_uint = 0x1E00;
    pub const LARGE_CCW_ARC_TO_NV: c_uint = 0x16;
    pub const LARGE_CW_ARC_TO_NV: c_uint = 0x18;
    pub const LAST_VERTEX_CONVENTION: c_uint = 0x8E4E;
    pub const LAST_VERTEX_CONVENTION_EXT: c_uint = 0x8E4E;
    pub const LAST_VIDEO_CAPTURE_STATUS_NV: c_uint = 0x9027;
    pub const LAYER_NV: c_uint = 0x8DAA;
    pub const LAYER_PROVOKING_VERTEX: c_uint = 0x825E;
    pub const LAYOUT_COLOR_ATTACHMENT_EXT: c_uint = 0x958E;
    pub const LAYOUT_DEFAULT_INTEL: c_uint = 0;
    pub const LAYOUT_DEPTH_ATTACHMENT_STENCIL_READ_ONLY_EXT: c_uint = 0x9531;
    pub const LAYOUT_DEPTH_READ_ONLY_STENCIL_ATTACHMENT_EXT: c_uint = 0x9530;
    pub const LAYOUT_DEPTH_STENCIL_ATTACHMENT_EXT: c_uint = 0x958F;
    pub const LAYOUT_DEPTH_STENCIL_READ_ONLY_EXT: c_uint = 0x9590;
    pub const LAYOUT_GENERAL_EXT: c_uint = 0x958D;
    pub const LAYOUT_LINEAR_CPU_CACHED_INTEL: c_uint = 2;
    pub const LAYOUT_LINEAR_INTEL: c_uint = 1;
    pub const LAYOUT_SHADER_READ_ONLY_EXT: c_uint = 0x9591;
    pub const LAYOUT_TRANSFER_DST_EXT: c_uint = 0x9593;
    pub const LAYOUT_TRANSFER_SRC_EXT: c_uint = 0x9592;
    pub const LEFT: c_uint = 0x0406;
    pub const LEQUAL: c_uint = 0x0203;
    pub const LERP_ATI: c_uint = 0x8969;
    pub const LESS: c_uint = 0x0201;
    pub const LGPU_SEPARATE_STORAGE_BIT_NVX: c_uint = 0x0800;
    pub const LIGHTEN_KHR: c_uint = 0x9298;
    pub const LIGHTEN_NV: c_uint = 0x9298;
    pub const LIGHT_ENV_MODE_SGIX: c_uint = 0x8407;
    pub const LIGHT_MODEL_COLOR_CONTROL_EXT: c_uint = 0x81F8;
    pub const LIGHT_MODEL_SPECULAR_VECTOR_APPLE: c_uint = 0x85B0;
    pub const LINE: c_uint = 0x1B01;
    pub const LINEAR: c_uint = 0x2601;
    pub const LINEARBURN_NV: c_uint = 0x92A5;
    pub const LINEARDODGE_NV: c_uint = 0x92A4;
    pub const LINEARLIGHT_NV: c_uint = 0x92A7;
    pub const LINEAR_CLIPMAP_LINEAR_SGIX: c_uint = 0x8170;
    pub const LINEAR_CLIPMAP_NEAREST_SGIX: c_uint = 0x844F;
    pub const LINEAR_DETAIL_ALPHA_SGIS: c_uint = 0x8098;
    pub const LINEAR_DETAIL_COLOR_SGIS: c_uint = 0x8099;
    pub const LINEAR_DETAIL_SGIS: c_uint = 0x8097;
    pub const LINEAR_MIPMAP_LINEAR: c_uint = 0x2703;
    pub const LINEAR_MIPMAP_NEAREST: c_uint = 0x2701;
    pub const LINEAR_SHARPEN_ALPHA_SGIS: c_uint = 0x80AE;
    pub const LINEAR_SHARPEN_COLOR_SGIS: c_uint = 0x80AF;
    pub const LINEAR_SHARPEN_SGIS: c_uint = 0x80AD;
    pub const LINEAR_TILING_EXT: c_uint = 0x9585;
    pub const LINES: c_uint = 0x0001;
    pub const LINES_ADJACENCY: c_uint = 0x000A;
    pub const LINES_ADJACENCY_ARB: c_uint = 0x000A;
    pub const LINES_ADJACENCY_EXT: c_uint = 0x000A;
    pub const LINE_LOOP: c_uint = 0x0002;
    pub const LINE_SMOOTH: c_uint = 0x0B20;
    pub const LINE_SMOOTH_HINT: c_uint = 0x0C52;
    pub const LINE_STRIP: c_uint = 0x0003;
    pub const LINE_STRIP_ADJACENCY: c_uint = 0x000B;
    pub const LINE_STRIP_ADJACENCY_ARB: c_uint = 0x000B;
    pub const LINE_STRIP_ADJACENCY_EXT: c_uint = 0x000B;
    pub const LINE_TO_NV: c_uint = 0x04;
    pub const LINE_WIDTH: c_uint = 0x0B21;
    pub const LINE_WIDTH_COMMAND_NV: c_uint = 0x000D;
    pub const LINE_WIDTH_GRANULARITY: c_uint = 0x0B23;
    pub const LINE_WIDTH_RANGE: c_uint = 0x0B22;
    pub const LINK_STATUS: c_uint = 0x8B82;
    pub const LIST_PRIORITY_SGIX: c_uint = 0x8182;
    pub const LOCAL_CONSTANT_DATATYPE_EXT: c_uint = 0x87ED;
    pub const LOCAL_CONSTANT_EXT: c_uint = 0x87C3;
    pub const LOCAL_CONSTANT_VALUE_EXT: c_uint = 0x87EC;
    pub const LOCAL_EXT: c_uint = 0x87C4;
    pub const LOCATION: c_uint = 0x930E;
    pub const LOCATION_COMPONENT: c_uint = 0x934A;
    pub const LOCATION_INDEX: c_uint = 0x930F;
    pub const LOGIC_OP_MODE: c_uint = 0x0BF0;
    pub const LOSE_CONTEXT_ON_RESET: c_uint = 0x8252;
    pub const LOSE_CONTEXT_ON_RESET_ARB: c_uint = 0x8252;
    pub const LOWER_LEFT: c_uint = 0x8CA1;
    pub const LOW_FLOAT: c_uint = 0x8DF0;
    pub const LOW_INT: c_uint = 0x8DF3;
    pub const LO_BIAS_NV: c_uint = 0x8715;
    pub const LO_SCALE_NV: c_uint = 0x870F;
    pub const LUID_SIZE_EXT: c_uint = 8;
    pub const LUMINANCE12_ALPHA12_EXT: c_uint = 0x8047;
    pub const LUMINANCE12_ALPHA4_EXT: c_uint = 0x8046;
    pub const LUMINANCE12_EXT: c_uint = 0x8041;
    pub const LUMINANCE16F_ARB: c_uint = 0x881E;
    pub const LUMINANCE16F_EXT: c_uint = 0x881E;
    pub const LUMINANCE16I_EXT: c_uint = 0x8D8C;
    pub const LUMINANCE16UI_EXT: c_uint = 0x8D7A;
    pub const LUMINANCE16_ALPHA16_EXT: c_uint = 0x8048;
    pub const LUMINANCE16_ALPHA16_SNORM: c_uint = 0x901A;
    pub const LUMINANCE16_EXT: c_uint = 0x8042;
    pub const LUMINANCE16_SNORM: c_uint = 0x9019;
    pub const LUMINANCE32F_ARB: c_uint = 0x8818;
    pub const LUMINANCE32F_EXT: c_uint = 0x8818;
    pub const LUMINANCE32I_EXT: c_uint = 0x8D86;
    pub const LUMINANCE32UI_EXT: c_uint = 0x8D74;
    pub const LUMINANCE4_ALPHA4_EXT: c_uint = 0x8043;
    pub const LUMINANCE4_EXT: c_uint = 0x803F;
    pub const LUMINANCE6_ALPHA2_EXT: c_uint = 0x8044;
    pub const LUMINANCE8I_EXT: c_uint = 0x8D92;
    pub const LUMINANCE8UI_EXT: c_uint = 0x8D80;
    pub const LUMINANCE8_ALPHA8_EXT: c_uint = 0x8045;
    pub const LUMINANCE8_ALPHA8_SNORM: c_uint = 0x9016;
    pub const LUMINANCE8_EXT: c_uint = 0x8040;
    pub const LUMINANCE8_SNORM: c_uint = 0x9015;
    pub const LUMINANCE_ALPHA16F_ARB: c_uint = 0x881F;
    pub const LUMINANCE_ALPHA16F_EXT: c_uint = 0x881F;
    pub const LUMINANCE_ALPHA16I_EXT: c_uint = 0x8D8D;
    pub const LUMINANCE_ALPHA16UI_EXT: c_uint = 0x8D7B;
    pub const LUMINANCE_ALPHA32F_ARB: c_uint = 0x8819;
    pub const LUMINANCE_ALPHA32F_EXT: c_uint = 0x8819;
    pub const LUMINANCE_ALPHA32I_EXT: c_uint = 0x8D87;
    pub const LUMINANCE_ALPHA32UI_EXT: c_uint = 0x8D75;
    pub const LUMINANCE_ALPHA8I_EXT: c_uint = 0x8D93;
    pub const LUMINANCE_ALPHA8UI_EXT: c_uint = 0x8D81;
    pub const LUMINANCE_ALPHA_FLOAT16_APPLE: c_uint = 0x881F;
    pub const LUMINANCE_ALPHA_FLOAT16_ATI: c_uint = 0x881F;
    pub const LUMINANCE_ALPHA_FLOAT32_APPLE: c_uint = 0x8819;
    pub const LUMINANCE_ALPHA_FLOAT32_ATI: c_uint = 0x8819;
    pub const LUMINANCE_ALPHA_INTEGER_EXT: c_uint = 0x8D9D;
    pub const LUMINANCE_ALPHA_SNORM: c_uint = 0x9012;
    pub const LUMINANCE_FLOAT16_APPLE: c_uint = 0x881E;
    pub const LUMINANCE_FLOAT16_ATI: c_uint = 0x881E;
    pub const LUMINANCE_FLOAT32_APPLE: c_uint = 0x8818;
    pub const LUMINANCE_FLOAT32_ATI: c_uint = 0x8818;
    pub const LUMINANCE_INTEGER_EXT: c_uint = 0x8D9C;
    pub const LUMINANCE_SNORM: c_uint = 0x9011;
    pub const MAD_ATI: c_uint = 0x8968;
    pub const MAGNITUDE_BIAS_NV: c_uint = 0x8718;
    pub const MAGNITUDE_SCALE_NV: c_uint = 0x8712;
    pub const MAJOR_VERSION: c_uint = 0x821B;
    pub const MANUAL_GENERATE_MIPMAP: c_uint = 0x8294;
    pub const MAP1_BINORMAL_EXT: c_uint = 0x8446;
    pub const MAP1_TANGENT_EXT: c_uint = 0x8444;
    pub const MAP1_VERTEX_ATTRIB0_4_NV: c_uint = 0x8660;
    pub const MAP1_VERTEX_ATTRIB10_4_NV: c_uint = 0x866A;
    pub const MAP1_VERTEX_ATTRIB11_4_NV: c_uint = 0x866B;
    pub const MAP1_VERTEX_ATTRIB12_4_NV: c_uint = 0x866C;
    pub const MAP1_VERTEX_ATTRIB13_4_NV: c_uint = 0x866D;
    pub const MAP1_VERTEX_ATTRIB14_4_NV: c_uint = 0x866E;
    pub const MAP1_VERTEX_ATTRIB15_4_NV: c_uint = 0x866F;
    pub const MAP1_VERTEX_ATTRIB1_4_NV: c_uint = 0x8661;
    pub const MAP1_VERTEX_ATTRIB2_4_NV: c_uint = 0x8662;
    pub const MAP1_VERTEX_ATTRIB3_4_NV: c_uint = 0x8663;
    pub const MAP1_VERTEX_ATTRIB4_4_NV: c_uint = 0x8664;
    pub const MAP1_VERTEX_ATTRIB5_4_NV: c_uint = 0x8665;
    pub const MAP1_VERTEX_ATTRIB6_4_NV: c_uint = 0x8666;
    pub const MAP1_VERTEX_ATTRIB7_4_NV: c_uint = 0x8667;
    pub const MAP1_VERTEX_ATTRIB8_4_NV: c_uint = 0x8668;
    pub const MAP1_VERTEX_ATTRIB9_4_NV: c_uint = 0x8669;
    pub const MAP2_BINORMAL_EXT: c_uint = 0x8447;
    pub const MAP2_TANGENT_EXT: c_uint = 0x8445;
    pub const MAP2_VERTEX_ATTRIB0_4_NV: c_uint = 0x8670;
    pub const MAP2_VERTEX_ATTRIB10_4_NV: c_uint = 0x867A;
    pub const MAP2_VERTEX_ATTRIB11_4_NV: c_uint = 0x867B;
    pub const MAP2_VERTEX_ATTRIB12_4_NV: c_uint = 0x867C;
    pub const MAP2_VERTEX_ATTRIB13_4_NV: c_uint = 0x867D;
    pub const MAP2_VERTEX_ATTRIB14_4_NV: c_uint = 0x867E;
    pub const MAP2_VERTEX_ATTRIB15_4_NV: c_uint = 0x867F;
    pub const MAP2_VERTEX_ATTRIB1_4_NV: c_uint = 0x8671;
    pub const MAP2_VERTEX_ATTRIB2_4_NV: c_uint = 0x8672;
    pub const MAP2_VERTEX_ATTRIB3_4_NV: c_uint = 0x8673;
    pub const MAP2_VERTEX_ATTRIB4_4_NV: c_uint = 0x8674;
    pub const MAP2_VERTEX_ATTRIB5_4_NV: c_uint = 0x8675;
    pub const MAP2_VERTEX_ATTRIB6_4_NV: c_uint = 0x8676;
    pub const MAP2_VERTEX_ATTRIB7_4_NV: c_uint = 0x8677;
    pub const MAP2_VERTEX_ATTRIB8_4_NV: c_uint = 0x8678;
    pub const MAP2_VERTEX_ATTRIB9_4_NV: c_uint = 0x8679;
    pub const MAP_ATTRIB_U_ORDER_NV: c_uint = 0x86C3;
    pub const MAP_ATTRIB_V_ORDER_NV: c_uint = 0x86C4;
    pub const MAP_COHERENT_BIT: c_uint = 0x0080;
    pub const MAP_FLUSH_EXPLICIT_BIT: c_uint = 0x0010;
    pub const MAP_INVALIDATE_BUFFER_BIT: c_uint = 0x0008;
    pub const MAP_INVALIDATE_RANGE_BIT: c_uint = 0x0004;
    pub const MAP_PERSISTENT_BIT: c_uint = 0x0040;
    pub const MAP_READ_BIT: c_uint = 0x0001;
    pub const MAP_TESSELLATION_NV: c_uint = 0x86C2;
    pub const MAP_UNSYNCHRONIZED_BIT: c_uint = 0x0020;
    pub const MAP_WRITE_BIT: c_uint = 0x0002;
    pub const MATERIAL_SIDE_HINT_PGI: c_uint = 0x1A22C;
    pub const MATRIX0_ARB: c_uint = 0x88C0;
    pub const MATRIX0_NV: c_uint = 0x8630;
    pub const MATRIX10_ARB: c_uint = 0x88CA;
    pub const MATRIX11_ARB: c_uint = 0x88CB;
    pub const MATRIX12_ARB: c_uint = 0x88CC;
    pub const MATRIX13_ARB: c_uint = 0x88CD;
    pub const MATRIX14_ARB: c_uint = 0x88CE;
    pub const MATRIX15_ARB: c_uint = 0x88CF;
    pub const MATRIX16_ARB: c_uint = 0x88D0;
    pub const MATRIX17_ARB: c_uint = 0x88D1;
    pub const MATRIX18_ARB: c_uint = 0x88D2;
    pub const MATRIX19_ARB: c_uint = 0x88D3;
    pub const MATRIX1_ARB: c_uint = 0x88C1;
    pub const MATRIX1_NV: c_uint = 0x8631;
    pub const MATRIX20_ARB: c_uint = 0x88D4;
    pub const MATRIX21_ARB: c_uint = 0x88D5;
    pub const MATRIX22_ARB: c_uint = 0x88D6;
    pub const MATRIX23_ARB: c_uint = 0x88D7;
    pub const MATRIX24_ARB: c_uint = 0x88D8;
    pub const MATRIX25_ARB: c_uint = 0x88D9;
    pub const MATRIX26_ARB: c_uint = 0x88DA;
    pub const MATRIX27_ARB: c_uint = 0x88DB;
    pub const MATRIX28_ARB: c_uint = 0x88DC;
    pub const MATRIX29_ARB: c_uint = 0x88DD;
    pub const MATRIX2_ARB: c_uint = 0x88C2;
    pub const MATRIX2_NV: c_uint = 0x8632;
    pub const MATRIX30_ARB: c_uint = 0x88DE;
    pub const MATRIX31_ARB: c_uint = 0x88DF;
    pub const MATRIX3_ARB: c_uint = 0x88C3;
    pub const MATRIX3_NV: c_uint = 0x8633;
    pub const MATRIX4_ARB: c_uint = 0x88C4;
    pub const MATRIX4_NV: c_uint = 0x8634;
    pub const MATRIX5_ARB: c_uint = 0x88C5;
    pub const MATRIX5_NV: c_uint = 0x8635;
    pub const MATRIX6_ARB: c_uint = 0x88C6;
    pub const MATRIX6_NV: c_uint = 0x8636;
    pub const MATRIX7_ARB: c_uint = 0x88C7;
    pub const MATRIX7_NV: c_uint = 0x8637;
    pub const MATRIX8_ARB: c_uint = 0x88C8;
    pub const MATRIX9_ARB: c_uint = 0x88C9;
    pub const MATRIX_EXT: c_uint = 0x87C0;
    pub const MATRIX_INDEX_ARRAY_ARB: c_uint = 0x8844;
    pub const MATRIX_INDEX_ARRAY_POINTER_ARB: c_uint = 0x8849;
    pub const MATRIX_INDEX_ARRAY_SIZE_ARB: c_uint = 0x8846;
    pub const MATRIX_INDEX_ARRAY_STRIDE_ARB: c_uint = 0x8848;
    pub const MATRIX_INDEX_ARRAY_TYPE_ARB: c_uint = 0x8847;
    pub const MATRIX_PALETTE_ARB: c_uint = 0x8840;
    pub const MATRIX_STRIDE: c_uint = 0x92FF;
    pub const MAT_AMBIENT_AND_DIFFUSE_BIT_PGI: c_uint = 0x00200000;
    pub const MAT_AMBIENT_BIT_PGI: c_uint = 0x00100000;
    pub const MAT_COLOR_INDEXES_BIT_PGI: c_uint = 0x01000000;
    pub const MAT_DIFFUSE_BIT_PGI: c_uint = 0x00400000;
    pub const MAT_EMISSION_BIT_PGI: c_uint = 0x00800000;
    pub const MAT_SHININESS_BIT_PGI: c_uint = 0x02000000;
    pub const MAT_SPECULAR_BIT_PGI: c_uint = 0x04000000;
    pub const MAX: c_uint = 0x8008;
    pub const MAX_3D_TEXTURE_SIZE: c_uint = 0x8073;
    pub const MAX_3D_TEXTURE_SIZE_EXT: c_uint = 0x8073;
    pub const MAX_4D_TEXTURE_SIZE_SGIS: c_uint = 0x8138;
    pub const MAX_ACTIVE_LIGHTS_SGIX: c_uint = 0x8405;
    pub const MAX_ARRAY_TEXTURE_LAYERS: c_uint = 0x88FF;
    pub const MAX_ARRAY_TEXTURE_LAYERS_EXT: c_uint = 0x88FF;
    pub const MAX_ASYNC_DRAW_PIXELS_SGIX: c_uint = 0x8360;
    pub const MAX_ASYNC_HISTOGRAM_SGIX: c_uint = 0x832D;
    pub const MAX_ASYNC_READ_PIXELS_SGIX: c_uint = 0x8361;
    pub const MAX_ASYNC_TEX_IMAGE_SGIX: c_uint = 0x835F;
    pub const MAX_ATOMIC_COUNTER_BUFFER_BINDINGS: c_uint = 0x92DC;
    pub const MAX_ATOMIC_COUNTER_BUFFER_SIZE: c_uint = 0x92D8;
    pub const MAX_BINDABLE_UNIFORM_SIZE_EXT: c_uint = 0x8DED;
    pub const MAX_CLIPMAP_DEPTH_SGIX: c_uint = 0x8177;
    pub const MAX_CLIPMAP_VIRTUAL_DEPTH_SGIX: c_uint = 0x8178;
    pub const MAX_CLIP_DISTANCES: c_uint = 0x0D32;
    pub const MAX_COARSE_FRAGMENT_SAMPLES_NV: c_uint = 0x955F;
    pub const MAX_COLOR_ATTACHMENTS: c_uint = 0x8CDF;
    pub const MAX_COLOR_ATTACHMENTS_EXT: c_uint = 0x8CDF;
    pub const MAX_COLOR_FRAMEBUFFER_SAMPLES_AMD: c_uint = 0x91B3;
    pub const MAX_COLOR_FRAMEBUFFER_STORAGE_SAMPLES_AMD: c_uint = 0x91B4;
    pub const MAX_COLOR_MATRIX_STACK_DEPTH_SGI: c_uint = 0x80B3;
    pub const MAX_COLOR_TEXTURE_SAMPLES: c_uint = 0x910E;
    pub const MAX_COMBINED_ATOMIC_COUNTERS: c_uint = 0x92D7;
    pub const MAX_COMBINED_ATOMIC_COUNTER_BUFFERS: c_uint = 0x92D1;
    pub const MAX_COMBINED_CLIP_AND_CULL_DISTANCES: c_uint = 0x82FA;
    pub const MAX_COMBINED_COMPUTE_UNIFORM_COMPONENTS: c_uint = 0x8266;
    pub const MAX_COMBINED_DIMENSIONS: c_uint = 0x8282;
    pub const MAX_COMBINED_FRAGMENT_UNIFORM_COMPONENTS: c_uint = 0x8A33;
    pub const MAX_COMBINED_GEOMETRY_UNIFORM_COMPONENTS: c_uint = 0x8A32;
    pub const MAX_COMBINED_IMAGE_UNIFORMS: c_uint = 0x90CF;
    pub const MAX_COMBINED_IMAGE_UNITS_AND_FRAGMENT_OUTPUTS: c_uint = 0x8F39;
    pub const MAX_COMBINED_IMAGE_UNITS_AND_FRAGMENT_OUTPUTS_EXT: c_uint = 0x8F39;
    pub const MAX_COMBINED_MESH_UNIFORM_COMPONENTS_NV: c_uint = 0x8E67;
    pub const MAX_COMBINED_SHADER_OUTPUT_RESOURCES: c_uint = 0x8F39;
    pub const MAX_COMBINED_SHADER_STORAGE_BLOCKS: c_uint = 0x90DC;
    pub const MAX_COMBINED_TASK_UNIFORM_COMPONENTS_NV: c_uint = 0x8E6F;
    pub const MAX_COMBINED_TESS_CONTROL_UNIFORM_COMPONENTS: c_uint = 0x8E1E;
    pub const MAX_COMBINED_TESS_EVALUATION_UNIFORM_COMPONENTS: c_uint = 0x8E1F;
    pub const MAX_COMBINED_TEXTURE_IMAGE_UNITS: c_uint = 0x8B4D;
    pub const MAX_COMBINED_TEXTURE_IMAGE_UNITS_ARB: c_uint = 0x8B4D;
    pub const MAX_COMBINED_UNIFORM_BLOCKS: c_uint = 0x8A2E;
    pub const MAX_COMBINED_VERTEX_UNIFORM_COMPONENTS: c_uint = 0x8A31;
    pub const MAX_COMPUTE_ATOMIC_COUNTERS: c_uint = 0x8265;
    pub const MAX_COMPUTE_ATOMIC_COUNTER_BUFFERS: c_uint = 0x8264;
    pub const MAX_COMPUTE_FIXED_GROUP_INVOCATIONS_ARB: c_uint = 0x90EB;
    pub const MAX_COMPUTE_FIXED_GROUP_SIZE_ARB: c_uint = 0x91BF;
    pub const MAX_COMPUTE_IMAGE_UNIFORMS: c_uint = 0x91BD;
    pub const MAX_COMPUTE_SHADER_STORAGE_BLOCKS: c_uint = 0x90DB;
    pub const MAX_COMPUTE_SHARED_MEMORY_SIZE: c_uint = 0x8262;
    pub const MAX_COMPUTE_TEXTURE_IMAGE_UNITS: c_uint = 0x91BC;
    pub const MAX_COMPUTE_UNIFORM_BLOCKS: c_uint = 0x91BB;
    pub const MAX_COMPUTE_UNIFORM_COMPONENTS: c_uint = 0x8263;
    pub const MAX_COMPUTE_VARIABLE_GROUP_INVOCATIONS_ARB: c_uint = 0x9344;
    pub const MAX_COMPUTE_VARIABLE_GROUP_SIZE_ARB: c_uint = 0x9345;
    pub const MAX_COMPUTE_WORK_GROUP_COUNT: c_uint = 0x91BE;
    pub const MAX_COMPUTE_WORK_GROUP_INVOCATIONS: c_uint = 0x90EB;
    pub const MAX_COMPUTE_WORK_GROUP_SIZE: c_uint = 0x91BF;
    pub const MAX_CONVOLUTION_HEIGHT_EXT: c_uint = 0x801B;
    pub const MAX_CONVOLUTION_WIDTH_EXT: c_uint = 0x801A;
    pub const MAX_CUBE_MAP_TEXTURE_SIZE: c_uint = 0x851C;
    pub const MAX_CUBE_MAP_TEXTURE_SIZE_ARB: c_uint = 0x851C;
    pub const MAX_CUBE_MAP_TEXTURE_SIZE_EXT: c_uint = 0x851C;
    pub const MAX_CULL_DISTANCES: c_uint = 0x82F9;
    pub const MAX_DEBUG_GROUP_STACK_DEPTH: c_uint = 0x826C;
    pub const MAX_DEBUG_LOGGED_MESSAGES: c_uint = 0x9144;
    pub const MAX_DEBUG_LOGGED_MESSAGES_AMD: c_uint = 0x9144;
    pub const MAX_DEBUG_LOGGED_MESSAGES_ARB: c_uint = 0x9144;
    pub const MAX_DEBUG_MESSAGE_LENGTH: c_uint = 0x9143;
    pub const MAX_DEBUG_MESSAGE_LENGTH_AMD: c_uint = 0x9143;
    pub const MAX_DEBUG_MESSAGE_LENGTH_ARB: c_uint = 0x9143;
    pub const MAX_DEEP_3D_TEXTURE_DEPTH_NV: c_uint = 0x90D1;
    pub const MAX_DEEP_3D_TEXTURE_WIDTH_HEIGHT_NV: c_uint = 0x90D0;
    pub const MAX_DEFORMATION_ORDER_SGIX: c_uint = 0x8197;
    pub const MAX_DEPTH: c_uint = 0x8280;
    pub const MAX_DEPTH_STENCIL_FRAMEBUFFER_SAMPLES_AMD: c_uint = 0x91B5;
    pub const MAX_DEPTH_TEXTURE_SAMPLES: c_uint = 0x910F;
    pub const MAX_DETACHED_BUFFERS_NV: c_uint = 0x95AD;
    pub const MAX_DETACHED_TEXTURES_NV: c_uint = 0x95AC;
    pub const MAX_DRAW_BUFFERS: c_uint = 0x8824;
    pub const MAX_DRAW_BUFFERS_ARB: c_uint = 0x8824;
    pub const MAX_DRAW_BUFFERS_ATI: c_uint = 0x8824;
    pub const MAX_DRAW_MESH_TASKS_COUNT_NV: c_uint = 0x953D;
    pub const MAX_DUAL_SOURCE_DRAW_BUFFERS: c_uint = 0x88FC;
    pub const MAX_ELEMENTS_INDICES: c_uint = 0x80E9;
    pub const MAX_ELEMENTS_INDICES_EXT: c_uint = 0x80E9;
    pub const MAX_ELEMENTS_VERTICES: c_uint = 0x80E8;
    pub const MAX_ELEMENTS_VERTICES_EXT: c_uint = 0x80E8;
    pub const MAX_ELEMENT_INDEX: c_uint = 0x8D6B;
    pub const MAX_EXT: c_uint = 0x8008;
    pub const MAX_FOG_FUNC_POINTS_SGIS: c_uint = 0x812C;
    pub const MAX_FRAGMENT_ATOMIC_COUNTERS: c_uint = 0x92D6;
    pub const MAX_FRAGMENT_ATOMIC_COUNTER_BUFFERS: c_uint = 0x92D0;
    pub const MAX_FRAGMENT_BINDABLE_UNIFORMS_EXT: c_uint = 0x8DE3;
    pub const MAX_FRAGMENT_IMAGE_UNIFORMS: c_uint = 0x90CE;
    pub const MAX_FRAGMENT_INPUT_COMPONENTS: c_uint = 0x9125;
    pub const MAX_FRAGMENT_INTERPOLATION_OFFSET: c_uint = 0x8E5C;
    pub const MAX_FRAGMENT_INTERPOLATION_OFFSET_NV: c_uint = 0x8E5C;
    pub const MAX_FRAGMENT_LIGHTS_SGIX: c_uint = 0x8404;
    pub const MAX_FRAGMENT_PROGRAM_LOCAL_PARAMETERS_NV: c_uint = 0x8868;
    pub const MAX_FRAGMENT_SHADER_STORAGE_BLOCKS: c_uint = 0x90DA;
    pub const MAX_FRAGMENT_UNIFORM_BLOCKS: c_uint = 0x8A2D;
    pub const MAX_FRAGMENT_UNIFORM_COMPONENTS: c_uint = 0x8B49;
    pub const MAX_FRAGMENT_UNIFORM_COMPONENTS_ARB: c_uint = 0x8B49;
    pub const MAX_FRAGMENT_UNIFORM_VECTORS: c_uint = 0x8DFD;
    pub const MAX_FRAMEBUFFER_HEIGHT: c_uint = 0x9316;
    pub const MAX_FRAMEBUFFER_LAYERS: c_uint = 0x9317;
    pub const MAX_FRAMEBUFFER_SAMPLES: c_uint = 0x9318;
    pub const MAX_FRAMEBUFFER_WIDTH: c_uint = 0x9315;
    pub const MAX_FRAMEZOOM_FACTOR_SGIX: c_uint = 0x818D;
    pub const MAX_GENERAL_COMBINERS_NV: c_uint = 0x854D;
    pub const MAX_GEOMETRY_ATOMIC_COUNTERS: c_uint = 0x92D5;
    pub const MAX_GEOMETRY_ATOMIC_COUNTER_BUFFERS: c_uint = 0x92CF;
    pub const MAX_GEOMETRY_BINDABLE_UNIFORMS_EXT: c_uint = 0x8DE4;
    pub const MAX_GEOMETRY_IMAGE_UNIFORMS: c_uint = 0x90CD;
    pub const MAX_GEOMETRY_INPUT_COMPONENTS: c_uint = 0x9123;
    pub const MAX_GEOMETRY_OUTPUT_COMPONENTS: c_uint = 0x9124;
    pub const MAX_GEOMETRY_OUTPUT_VERTICES: c_uint = 0x8DE0;
    pub const MAX_GEOMETRY_OUTPUT_VERTICES_ARB: c_uint = 0x8DE0;
    pub const MAX_GEOMETRY_OUTPUT_VERTICES_EXT: c_uint = 0x8DE0;
    pub const MAX_GEOMETRY_PROGRAM_INVOCATIONS_NV: c_uint = 0x8E5A;
    pub const MAX_GEOMETRY_SHADER_INVOCATIONS: c_uint = 0x8E5A;
    pub const MAX_GEOMETRY_SHADER_STORAGE_BLOCKS: c_uint = 0x90D7;
    pub const MAX_GEOMETRY_TEXTURE_IMAGE_UNITS: c_uint = 0x8C29;
    pub const MAX_GEOMETRY_TEXTURE_IMAGE_UNITS_ARB: c_uint = 0x8C29;
    pub const MAX_GEOMETRY_TEXTURE_IMAGE_UNITS_EXT: c_uint = 0x8C29;
    pub const MAX_GEOMETRY_TOTAL_OUTPUT_COMPONENTS: c_uint = 0x8DE1;
    pub const MAX_GEOMETRY_TOTAL_OUTPUT_COMPONENTS_ARB: c_uint = 0x8DE1;
    pub const MAX_GEOMETRY_TOTAL_OUTPUT_COMPONENTS_EXT: c_uint = 0x8DE1;
    pub const MAX_GEOMETRY_UNIFORM_BLOCKS: c_uint = 0x8A2C;
    pub const MAX_GEOMETRY_UNIFORM_COMPONENTS: c_uint = 0x8DDF;
    pub const MAX_GEOMETRY_UNIFORM_COMPONENTS_ARB: c_uint = 0x8DDF;
    pub const MAX_GEOMETRY_UNIFORM_COMPONENTS_EXT: c_uint = 0x8DDF;
    pub const MAX_GEOMETRY_VARYING_COMPONENTS_ARB: c_uint = 0x8DDD;
    pub const MAX_GEOMETRY_VARYING_COMPONENTS_EXT: c_uint = 0x8DDD;
    pub const MAX_HEIGHT: c_uint = 0x827F;
    pub const MAX_IMAGE_SAMPLES: c_uint = 0x906D;
    pub const MAX_IMAGE_SAMPLES_EXT: c_uint = 0x906D;
    pub const MAX_IMAGE_UNITS: c_uint = 0x8F38;
    pub const MAX_IMAGE_UNITS_EXT: c_uint = 0x8F38;
    pub const MAX_INTEGER_SAMPLES: c_uint = 0x9110;
    pub const MAX_LABEL_LENGTH: c_uint = 0x82E8;
    pub const MAX_LAYERS: c_uint = 0x8281;
    pub const MAX_LGPU_GPUS_NVX: c_uint = 0x92BA;
    pub const MAX_MAP_TESSELLATION_NV: c_uint = 0x86D6;
    pub const MAX_MATRIX_PALETTE_STACK_DEPTH_ARB: c_uint = 0x8841;
    pub const MAX_MESH_ATOMIC_COUNTERS_NV: c_uint = 0x8E65;
    pub const MAX_MESH_ATOMIC_COUNTER_BUFFERS_NV: c_uint = 0x8E64;
    pub const MAX_MESH_IMAGE_UNIFORMS_NV: c_uint = 0x8E62;
    pub const MAX_MESH_OUTPUT_PRIMITIVES_NV: c_uint = 0x9539;
    pub const MAX_MESH_OUTPUT_VERTICES_NV: c_uint = 0x9538;
    pub const MAX_MESH_SHADER_STORAGE_BLOCKS_NV: c_uint = 0x8E66;
    pub const MAX_MESH_TEXTURE_IMAGE_UNITS_NV: c_uint = 0x8E61;
    pub const MAX_MESH_TOTAL_MEMORY_SIZE_NV: c_uint = 0x9536;
    pub const MAX_MESH_UNIFORM_BLOCKS_NV: c_uint = 0x8E60;
    pub const MAX_MESH_UNIFORM_COMPONENTS_NV: c_uint = 0x8E63;
    pub const MAX_MESH_VIEWS_NV: c_uint = 0x9557;
    pub const MAX_MESH_WORK_GROUP_INVOCATIONS_NV: c_uint = 0x95A2;
    pub const MAX_MESH_WORK_GROUP_SIZE_NV: c_uint = 0x953B;
    pub const MAX_MULTISAMPLE_COVERAGE_MODES_NV: c_uint = 0x8E11;
    pub const MAX_NAME_LENGTH: c_uint = 0x92F6;
    pub const MAX_NUM_ACTIVE_VARIABLES: c_uint = 0x92F7;
    pub const MAX_NUM_COMPATIBLE_SUBROUTINES: c_uint = 0x92F8;
    pub const MAX_OPTIMIZED_VERTEX_SHADER_INSTRUCTIONS_EXT: c_uint = 0x87CA;
    pub const MAX_OPTIMIZED_VERTEX_SHADER_INVARIANTS_EXT: c_uint = 0x87CD;
    pub const MAX_OPTIMIZED_VERTEX_SHADER_LOCALS_EXT: c_uint = 0x87CE;
    pub const MAX_OPTIMIZED_VERTEX_SHADER_LOCAL_CONSTANTS_EXT: c_uint = 0x87CC;
    pub const MAX_OPTIMIZED_VERTEX_SHADER_VARIANTS_EXT: c_uint = 0x87CB;
    pub const MAX_PALETTE_MATRICES_ARB: c_uint = 0x8842;
    pub const MAX_PATCH_VERTICES: c_uint = 0x8E7D;
    pub const MAX_PIXEL_TRANSFORM_2D_STACK_DEPTH_EXT: c_uint = 0x8337;
    pub const MAX_PN_TRIANGLES_TESSELATION_LEVEL_ATI: c_uint = 0x87F1;
    pub const MAX_PROGRAM_ADDRESS_REGISTERS_ARB: c_uint = 0x88B1;
    pub const MAX_PROGRAM_ALU_INSTRUCTIONS_ARB: c_uint = 0x880B;
    pub const MAX_PROGRAM_ATTRIBS_ARB: c_uint = 0x88AD;
    pub const MAX_PROGRAM_ATTRIB_COMPONENTS_NV: c_uint = 0x8908;
    pub const MAX_PROGRAM_CALL_DEPTH_NV: c_uint = 0x88F5;
    pub const MAX_PROGRAM_ENV_PARAMETERS_ARB: c_uint = 0x88B5;
    pub const MAX_PROGRAM_EXEC_INSTRUCTIONS_NV: c_uint = 0x88F4;
    pub const MAX_PROGRAM_GENERIC_ATTRIBS_NV: c_uint = 0x8DA5;
    pub const MAX_PROGRAM_GENERIC_RESULTS_NV: c_uint = 0x8DA6;
    pub const MAX_PROGRAM_IF_DEPTH_NV: c_uint = 0x88F6;
    pub const MAX_PROGRAM_INSTRUCTIONS_ARB: c_uint = 0x88A1;
    pub const MAX_PROGRAM_LOCAL_PARAMETERS_ARB: c_uint = 0x88B4;
    pub const MAX_PROGRAM_LOOP_COUNT_NV: c_uint = 0x88F8;
    pub const MAX_PROGRAM_LOOP_DEPTH_NV: c_uint = 0x88F7;
    pub const MAX_PROGRAM_MATRICES_ARB: c_uint = 0x862F;
    pub const MAX_PROGRAM_MATRIX_STACK_DEPTH_ARB: c_uint = 0x862E;
    pub const MAX_PROGRAM_NATIVE_ADDRESS_REGISTERS_ARB: c_uint = 0x88B3;
    pub const MAX_PROGRAM_NATIVE_ALU_INSTRUCTIONS_ARB: c_uint = 0x880E;
    pub const MAX_PROGRAM_NATIVE_ATTRIBS_ARB: c_uint = 0x88AF;
    pub const MAX_PROGRAM_NATIVE_INSTRUCTIONS_ARB: c_uint = 0x88A3;
    pub const MAX_PROGRAM_NATIVE_PARAMETERS_ARB: c_uint = 0x88AB;
    pub const MAX_PROGRAM_NATIVE_TEMPORARIES_ARB: c_uint = 0x88A7;
    pub const MAX_PROGRAM_NATIVE_TEX_INDIRECTIONS_ARB: c_uint = 0x8810;
    pub const MAX_PROGRAM_NATIVE_TEX_INSTRUCTIONS_ARB: c_uint = 0x880F;
    pub const MAX_PROGRAM_OUTPUT_VERTICES_NV: c_uint = 0x8C27;
    pub const MAX_PROGRAM_PARAMETERS_ARB: c_uint = 0x88A9;
    pub const MAX_PROGRAM_PARAMETER_BUFFER_BINDINGS_NV: c_uint = 0x8DA0;
    pub const MAX_PROGRAM_PARAMETER_BUFFER_SIZE_NV: c_uint = 0x8DA1;
    pub const MAX_PROGRAM_PATCH_ATTRIBS_NV: c_uint = 0x86D8;
    pub const MAX_PROGRAM_RESULT_COMPONENTS_NV: c_uint = 0x8909;
    pub const MAX_PROGRAM_SUBROUTINE_NUM_NV: c_uint = 0x8F45;
    pub const MAX_PROGRAM_SUBROUTINE_PARAMETERS_NV: c_uint = 0x8F44;
    pub const MAX_PROGRAM_TEMPORARIES_ARB: c_uint = 0x88A5;
    pub const MAX_PROGRAM_TEXEL_OFFSET: c_uint = 0x8905;
    pub const MAX_PROGRAM_TEXEL_OFFSET_EXT: c_uint = 0x8905;
    pub const MAX_PROGRAM_TEXEL_OFFSET_NV: c_uint = 0x8905;
    pub const MAX_PROGRAM_TEXTURE_GATHER_COMPONENTS_ARB: c_uint = 0x8F9F;
    pub const MAX_PROGRAM_TEXTURE_GATHER_OFFSET: c_uint = 0x8E5F;
    pub const MAX_PROGRAM_TEXTURE_GATHER_OFFSET_ARB: c_uint = 0x8E5F;
    pub const MAX_PROGRAM_TEXTURE_GATHER_OFFSET_NV: c_uint = 0x8E5F;
    pub const MAX_PROGRAM_TEX_INDIRECTIONS_ARB: c_uint = 0x880D;
    pub const MAX_PROGRAM_TEX_INSTRUCTIONS_ARB: c_uint = 0x880C;
    pub const MAX_PROGRAM_TOTAL_OUTPUT_COMPONENTS_NV: c_uint = 0x8C28;
    pub const MAX_RASTER_SAMPLES_EXT: c_uint = 0x9329;
    pub const MAX_RATIONAL_EVAL_ORDER_NV: c_uint = 0x86D7;
    pub const MAX_RECTANGLE_TEXTURE_SIZE: c_uint = 0x84F8;
    pub const MAX_RECTANGLE_TEXTURE_SIZE_ARB: c_uint = 0x84F8;
    pub const MAX_RECTANGLE_TEXTURE_SIZE_NV: c_uint = 0x84F8;
    pub const MAX_RENDERBUFFER_SIZE: c_uint = 0x84E8;
    pub const MAX_RENDERBUFFER_SIZE_EXT: c_uint = 0x84E8;
    pub const MAX_SAMPLES: c_uint = 0x8D57;
    pub const MAX_SAMPLES_EXT: c_uint = 0x8D57;
    pub const MAX_SAMPLE_MASK_WORDS: c_uint = 0x8E59;
    pub const MAX_SAMPLE_MASK_WORDS_NV: c_uint = 0x8E59;
    pub const MAX_SERVER_WAIT_TIMEOUT: c_uint = 0x9111;
    pub const MAX_SHADER_BUFFER_ADDRESS_NV: c_uint = 0x8F35;
    pub const MAX_SHADER_COMPILER_THREADS_ARB: c_uint = 0x91B0;
    pub const MAX_SHADER_COMPILER_THREADS_KHR: c_uint = 0x91B0;
    pub const MAX_SHADER_STORAGE_BLOCK_SIZE: c_uint = 0x90DE;
    pub const MAX_SHADER_STORAGE_BUFFER_BINDINGS: c_uint = 0x90DD;
    pub const MAX_SHININESS_NV: c_uint = 0x8504;
    pub const MAX_SPARSE_3D_TEXTURE_SIZE_AMD: c_uint = 0x9199;
    pub const MAX_SPARSE_3D_TEXTURE_SIZE_ARB: c_uint = 0x9199;
    pub const MAX_SPARSE_ARRAY_TEXTURE_LAYERS: c_uint = 0x919A;
    pub const MAX_SPARSE_ARRAY_TEXTURE_LAYERS_ARB: c_uint = 0x919A;
    pub const MAX_SPARSE_TEXTURE_SIZE_AMD: c_uint = 0x9198;
    pub const MAX_SPARSE_TEXTURE_SIZE_ARB: c_uint = 0x9198;
    pub const MAX_SPOT_EXPONENT_NV: c_uint = 0x8505;
    pub const MAX_SUBPIXEL_PRECISION_BIAS_BITS_NV: c_uint = 0x9349;
    pub const MAX_SUBROUTINES: c_uint = 0x8DE7;
    pub const MAX_SUBROUTINE_UNIFORM_LOCATIONS: c_uint = 0x8DE8;
    pub const MAX_TASK_ATOMIC_COUNTERS_NV: c_uint = 0x8E6D;
    pub const MAX_TASK_ATOMIC_COUNTER_BUFFERS_NV: c_uint = 0x8E6C;
    pub const MAX_TASK_IMAGE_UNIFORMS_NV: c_uint = 0x8E6A;
    pub const MAX_TASK_OUTPUT_COUNT_NV: c_uint = 0x953A;
    pub const MAX_TASK_SHADER_STORAGE_BLOCKS_NV: c_uint = 0x8E6E;
    pub const MAX_TASK_TEXTURE_IMAGE_UNITS_NV: c_uint = 0x8E69;
    pub const MAX_TASK_TOTAL_MEMORY_SIZE_NV: c_uint = 0x9537;
    pub const MAX_TASK_UNIFORM_BLOCKS_NV: c_uint = 0x8E68;
    pub const MAX_TASK_UNIFORM_COMPONENTS_NV: c_uint = 0x8E6B;
    pub const MAX_TASK_WORK_GROUP_INVOCATIONS_NV: c_uint = 0x95A3;
    pub const MAX_TASK_WORK_GROUP_SIZE_NV: c_uint = 0x953C;
    pub const MAX_TESS_CONTROL_ATOMIC_COUNTERS: c_uint = 0x92D3;
    pub const MAX_TESS_CONTROL_ATOMIC_COUNTER_BUFFERS: c_uint = 0x92CD;
    pub const MAX_TESS_CONTROL_IMAGE_UNIFORMS: c_uint = 0x90CB;
    pub const MAX_TESS_CONTROL_INPUT_COMPONENTS: c_uint = 0x886C;
    pub const MAX_TESS_CONTROL_OUTPUT_COMPONENTS: c_uint = 0x8E83;
    pub const MAX_TESS_CONTROL_SHADER_STORAGE_BLOCKS: c_uint = 0x90D8;
    pub const MAX_TESS_CONTROL_TEXTURE_IMAGE_UNITS: c_uint = 0x8E81;
    pub const MAX_TESS_CONTROL_TOTAL_OUTPUT_COMPONENTS: c_uint = 0x8E85;
    pub const MAX_TESS_CONTROL_UNIFORM_BLOCKS: c_uint = 0x8E89;
    pub const MAX_TESS_CONTROL_UNIFORM_COMPONENTS: c_uint = 0x8E7F;
    pub const MAX_TESS_EVALUATION_ATOMIC_COUNTERS: c_uint = 0x92D4;
    pub const MAX_TESS_EVALUATION_ATOMIC_COUNTER_BUFFERS: c_uint = 0x92CE;
    pub const MAX_TESS_EVALUATION_IMAGE_UNIFORMS: c_uint = 0x90CC;
    pub const MAX_TESS_EVALUATION_INPUT_COMPONENTS: c_uint = 0x886D;
    pub const MAX_TESS_EVALUATION_OUTPUT_COMPONENTS: c_uint = 0x8E86;
    pub const MAX_TESS_EVALUATION_SHADER_STORAGE_BLOCKS: c_uint = 0x90D9;
    pub const MAX_TESS_EVALUATION_TEXTURE_IMAGE_UNITS: c_uint = 0x8E82;
    pub const MAX_TESS_EVALUATION_UNIFORM_BLOCKS: c_uint = 0x8E8A;
    pub const MAX_TESS_EVALUATION_UNIFORM_COMPONENTS: c_uint = 0x8E80;
    pub const MAX_TESS_GEN_LEVEL: c_uint = 0x8E7E;
    pub const MAX_TESS_PATCH_COMPONENTS: c_uint = 0x8E84;
    pub const MAX_TEXTURE_BUFFER_SIZE: c_uint = 0x8C2B;
    pub const MAX_TEXTURE_BUFFER_SIZE_ARB: c_uint = 0x8C2B;
    pub const MAX_TEXTURE_BUFFER_SIZE_EXT: c_uint = 0x8C2B;
    pub const MAX_TEXTURE_COORDS_ARB: c_uint = 0x8871;
    pub const MAX_TEXTURE_COORDS_NV: c_uint = 0x8871;
    pub const MAX_TEXTURE_IMAGE_UNITS: c_uint = 0x8872;
    pub const MAX_TEXTURE_IMAGE_UNITS_ARB: c_uint = 0x8872;
    pub const MAX_TEXTURE_IMAGE_UNITS_NV: c_uint = 0x8872;
    pub const MAX_TEXTURE_LOD_BIAS: c_uint = 0x84FD;
    pub const MAX_TEXTURE_LOD_BIAS_EXT: c_uint = 0x84FD;
    pub const MAX_TEXTURE_MAX_ANISOTROPY: c_uint = 0x84FF;
    pub const MAX_TEXTURE_MAX_ANISOTROPY_EXT: c_uint = 0x84FF;
    pub const MAX_TEXTURE_SIZE: c_uint = 0x0D33;
    pub const MAX_TEXTURE_UNITS_ARB: c_uint = 0x84E2;
    pub const MAX_TIMELINE_SEMAPHORE_VALUE_DIFFERENCE_NV: c_uint = 0x95B6;
    pub const MAX_TRACK_MATRICES_NV: c_uint = 0x862F;
    pub const MAX_TRACK_MATRIX_STACK_DEPTH_NV: c_uint = 0x862E;
    pub const MAX_TRANSFORM_FEEDBACK_BUFFERS: c_uint = 0x8E70;
    pub const MAX_TRANSFORM_FEEDBACK_INTERLEAVED_COMPONENTS: c_uint = 0x8C8A;
    pub const MAX_TRANSFORM_FEEDBACK_INTERLEAVED_COMPONENTS_EXT: c_uint = 0x8C8A;
    pub const MAX_TRANSFORM_FEEDBACK_INTERLEAVED_COMPONENTS_NV: c_uint = 0x8C8A;
    pub const MAX_TRANSFORM_FEEDBACK_SEPARATE_ATTRIBS: c_uint = 0x8C8B;
    pub const MAX_TRANSFORM_FEEDBACK_SEPARATE_ATTRIBS_EXT: c_uint = 0x8C8B;
    pub const MAX_TRANSFORM_FEEDBACK_SEPARATE_ATTRIBS_NV: c_uint = 0x8C8B;
    pub const MAX_TRANSFORM_FEEDBACK_SEPARATE_COMPONENTS: c_uint = 0x8C80;
    pub const MAX_TRANSFORM_FEEDBACK_SEPARATE_COMPONENTS_EXT: c_uint = 0x8C80;
    pub const MAX_TRANSFORM_FEEDBACK_SEPARATE_COMPONENTS_NV: c_uint = 0x8C80;
    pub const MAX_UNIFORM_BLOCK_SIZE: c_uint = 0x8A30;
    pub const MAX_UNIFORM_BUFFER_BINDINGS: c_uint = 0x8A2F;
    pub const MAX_UNIFORM_LOCATIONS: c_uint = 0x826E;
    pub const MAX_VARYING_COMPONENTS: c_uint = 0x8B4B;
    pub const MAX_VARYING_COMPONENTS_EXT: c_uint = 0x8B4B;
    pub const MAX_VARYING_FLOATS: c_uint = 0x8B4B;
    pub const MAX_VARYING_FLOATS_ARB: c_uint = 0x8B4B;
    pub const MAX_VARYING_VECTORS: c_uint = 0x8DFC;
    pub const MAX_VERTEX_ARRAY_RANGE_ELEMENT_NV: c_uint = 0x8520;
    pub const MAX_VERTEX_ATOMIC_COUNTERS: c_uint = 0x92D2;
    pub const MAX_VERTEX_ATOMIC_COUNTER_BUFFERS: c_uint = 0x92CC;
    pub const MAX_VERTEX_ATTRIBS: c_uint = 0x8869;
    pub const MAX_VERTEX_ATTRIBS_ARB: c_uint = 0x8869;
    pub const MAX_VERTEX_ATTRIB_BINDINGS: c_uint = 0x82DA;
    pub const MAX_VERTEX_ATTRIB_RELATIVE_OFFSET: c_uint = 0x82D9;
    pub const MAX_VERTEX_ATTRIB_STRIDE: c_uint = 0x82E5;
    pub const MAX_VERTEX_BINDABLE_UNIFORMS_EXT: c_uint = 0x8DE2;
    pub const MAX_VERTEX_HINT_PGI: c_uint = 0x1A22D;
    pub const MAX_VERTEX_IMAGE_UNIFORMS: c_uint = 0x90CA;
    pub const MAX_VERTEX_OUTPUT_COMPONENTS: c_uint = 0x9122;
    pub const MAX_VERTEX_SHADER_INSTRUCTIONS_EXT: c_uint = 0x87C5;
    pub const MAX_VERTEX_SHADER_INVARIANTS_EXT: c_uint = 0x87C7;
    pub const MAX_VERTEX_SHADER_LOCALS_EXT: c_uint = 0x87C9;
    pub const MAX_VERTEX_SHADER_LOCAL_CONSTANTS_EXT: c_uint = 0x87C8;
    pub const MAX_VERTEX_SHADER_STORAGE_BLOCKS: c_uint = 0x90D6;
    pub const MAX_VERTEX_SHADER_VARIANTS_EXT: c_uint = 0x87C6;
    pub const MAX_VERTEX_STREAMS: c_uint = 0x8E71;
    pub const MAX_VERTEX_STREAMS_ATI: c_uint = 0x876B;
    pub const MAX_VERTEX_TEXTURE_IMAGE_UNITS: c_uint = 0x8B4C;
    pub const MAX_VERTEX_TEXTURE_IMAGE_UNITS_ARB: c_uint = 0x8B4C;
    pub const MAX_VERTEX_UNIFORM_BLOCKS: c_uint = 0x8A2B;
    pub const MAX_VERTEX_UNIFORM_COMPONENTS: c_uint = 0x8B4A;
    pub const MAX_VERTEX_UNIFORM_COMPONENTS_ARB: c_uint = 0x8B4A;
    pub const MAX_VERTEX_UNIFORM_VECTORS: c_uint = 0x8DFB;
    pub const MAX_VERTEX_UNITS_ARB: c_uint = 0x86A4;
    pub const MAX_VERTEX_VARYING_COMPONENTS_ARB: c_uint = 0x8DDE;
    pub const MAX_VERTEX_VARYING_COMPONENTS_EXT: c_uint = 0x8DDE;
    pub const MAX_VIEWPORTS: c_uint = 0x825B;
    pub const MAX_VIEWPORT_DIMS: c_uint = 0x0D3A;
    pub const MAX_VIEWS_OVR: c_uint = 0x9631;
    pub const MAX_WIDTH: c_uint = 0x827E;
    pub const MAX_WINDOW_RECTANGLES_EXT: c_uint = 0x8F14;
    pub const MEDIUM_FLOAT: c_uint = 0x8DF1;
    pub const MEDIUM_INT: c_uint = 0x8DF4;
    pub const MEMORY_ATTACHABLE_ALIGNMENT_NV: c_uint = 0x95A6;
    pub const MEMORY_ATTACHABLE_NV: c_uint = 0x95A8;
    pub const MEMORY_ATTACHABLE_SIZE_NV: c_uint = 0x95A7;
    pub const MESH_OUTPUT_PER_PRIMITIVE_GRANULARITY_NV: c_uint = 0x9543;
    pub const MESH_OUTPUT_PER_VERTEX_GRANULARITY_NV: c_uint = 0x92DF;
    pub const MESH_OUTPUT_TYPE_NV: c_uint = 0x957B;
    pub const MESH_PRIMITIVES_OUT_NV: c_uint = 0x957A;
    pub const MESH_SHADER_BIT_NV: c_uint = 0x00000040;
    pub const MESH_SHADER_NV: c_uint = 0x9559;
    pub const MESH_SUBROUTINE_NV: c_uint = 0x957C;
    pub const MESH_SUBROUTINE_UNIFORM_NV: c_uint = 0x957E;
    pub const MESH_VERTICES_OUT_NV: c_uint = 0x9579;
    pub const MESH_WORK_GROUP_SIZE_NV: c_uint = 0x953E;
    pub const MIN: c_uint = 0x8007;
    pub const MINMAX_EXT: c_uint = 0x802E;
    pub const MINMAX_FORMAT_EXT: c_uint = 0x802F;
    pub const MINMAX_SINK_EXT: c_uint = 0x8030;
    pub const MINOR_VERSION: c_uint = 0x821C;
    pub const MINUS_CLAMPED_NV: c_uint = 0x92B3;
    pub const MINUS_NV: c_uint = 0x929F;
    pub const MIN_EXT: c_uint = 0x8007;
    pub const MIN_FRAGMENT_INTERPOLATION_OFFSET: c_uint = 0x8E5B;
    pub const MIN_FRAGMENT_INTERPOLATION_OFFSET_NV: c_uint = 0x8E5B;
    pub const MIN_LOD_WARNING_AMD: c_uint = 0x919C;
    pub const MIN_MAP_BUFFER_ALIGNMENT: c_uint = 0x90BC;
    pub const MIN_PROGRAM_TEXEL_OFFSET: c_uint = 0x8904;
    pub const MIN_PROGRAM_TEXEL_OFFSET_EXT: c_uint = 0x8904;
    pub const MIN_PROGRAM_TEXEL_OFFSET_NV: c_uint = 0x8904;
    pub const MIN_PROGRAM_TEXTURE_GATHER_OFFSET: c_uint = 0x8E5E;
    pub const MIN_PROGRAM_TEXTURE_GATHER_OFFSET_ARB: c_uint = 0x8E5E;
    pub const MIN_PROGRAM_TEXTURE_GATHER_OFFSET_NV: c_uint = 0x8E5E;
    pub const MIN_SAMPLE_SHADING_VALUE: c_uint = 0x8C37;
    pub const MIN_SAMPLE_SHADING_VALUE_ARB: c_uint = 0x8C37;
    pub const MIN_SPARSE_LEVEL_AMD: c_uint = 0x919B;
    pub const MIPMAP: c_uint = 0x8293;
    pub const MIRRORED_REPEAT: c_uint = 0x8370;
    pub const MIRRORED_REPEAT_ARB: c_uint = 0x8370;
    pub const MIRRORED_REPEAT_IBM: c_uint = 0x8370;
    pub const MIRROR_CLAMP_ATI: c_uint = 0x8742;
    pub const MIRROR_CLAMP_EXT: c_uint = 0x8742;
    pub const MIRROR_CLAMP_TO_BORDER_EXT: c_uint = 0x8912;
    pub const MIRROR_CLAMP_TO_EDGE: c_uint = 0x8743;
    pub const MIRROR_CLAMP_TO_EDGE_ATI: c_uint = 0x8743;
    pub const MIRROR_CLAMP_TO_EDGE_EXT: c_uint = 0x8743;
    pub const MITER_REVERT_NV: c_uint = 0x90A7;
    pub const MITER_TRUNCATE_NV: c_uint = 0x90A8;
    pub const MIXED_DEPTH_SAMPLES_SUPPORTED_NV: c_uint = 0x932F;
    pub const MIXED_STENCIL_SAMPLES_SUPPORTED_NV: c_uint = 0x9330;
    pub const MODELVIEW0_ARB: c_uint = 0x1700;
    pub const MODELVIEW0_EXT: c_uint = 0x1700;
    pub const MODELVIEW0_MATRIX_EXT: c_uint = 0x0BA6;
    pub const MODELVIEW0_STACK_DEPTH_EXT: c_uint = 0x0BA3;
    pub const MODELVIEW10_ARB: c_uint = 0x872A;
    pub const MODELVIEW11_ARB: c_uint = 0x872B;
    pub const MODELVIEW12_ARB: c_uint = 0x872C;
    pub const MODELVIEW13_ARB: c_uint = 0x872D;
    pub const MODELVIEW14_ARB: c_uint = 0x872E;
    pub const MODELVIEW15_ARB: c_uint = 0x872F;
    pub const MODELVIEW16_ARB: c_uint = 0x8730;
    pub const MODELVIEW17_ARB: c_uint = 0x8731;
    pub const MODELVIEW18_ARB: c_uint = 0x8732;
    pub const MODELVIEW19_ARB: c_uint = 0x8733;
    pub const MODELVIEW1_ARB: c_uint = 0x850A;
    pub const MODELVIEW1_EXT: c_uint = 0x850A;
    pub const MODELVIEW1_MATRIX_EXT: c_uint = 0x8506;
    pub const MODELVIEW1_STACK_DEPTH_EXT: c_uint = 0x8502;
    pub const MODELVIEW20_ARB: c_uint = 0x8734;
    pub const MODELVIEW21_ARB: c_uint = 0x8735;
    pub const MODELVIEW22_ARB: c_uint = 0x8736;
    pub const MODELVIEW23_ARB: c_uint = 0x8737;
    pub const MODELVIEW24_ARB: c_uint = 0x8738;
    pub const MODELVIEW25_ARB: c_uint = 0x8739;
    pub const MODELVIEW26_ARB: c_uint = 0x873A;
    pub const MODELVIEW27_ARB: c_uint = 0x873B;
    pub const MODELVIEW28_ARB: c_uint = 0x873C;
    pub const MODELVIEW29_ARB: c_uint = 0x873D;
    pub const MODELVIEW2_ARB: c_uint = 0x8722;
    pub const MODELVIEW30_ARB: c_uint = 0x873E;
    pub const MODELVIEW31_ARB: c_uint = 0x873F;
    pub const MODELVIEW3_ARB: c_uint = 0x8723;
    pub const MODELVIEW4_ARB: c_uint = 0x8724;
    pub const MODELVIEW5_ARB: c_uint = 0x8725;
    pub const MODELVIEW6_ARB: c_uint = 0x8726;
    pub const MODELVIEW7_ARB: c_uint = 0x8727;
    pub const MODELVIEW8_ARB: c_uint = 0x8728;
    pub const MODELVIEW9_ARB: c_uint = 0x8729;
    pub const MODELVIEW_PROJECTION_NV: c_uint = 0x8629;
    pub const MODULATE_ADD_ATI: c_uint = 0x8744;
    pub const MODULATE_SIGNED_ADD_ATI: c_uint = 0x8745;
    pub const MODULATE_SUBTRACT_ATI: c_uint = 0x8746;
    pub const MOVE_TO_CONTINUES_NV: c_uint = 0x90B6;
    pub const MOVE_TO_NV: c_uint = 0x02;
    pub const MOVE_TO_RESETS_NV: c_uint = 0x90B5;
    pub const MOV_ATI: c_uint = 0x8961;
    pub const MULTICAST_GPUS_NV: c_uint = 0x92BA;
    pub const MULTICAST_PROGRAMMABLE_SAMPLE_LOCATION_NV: c_uint = 0x9549;
    pub const MULTIPLY_KHR: c_uint = 0x9294;
    pub const MULTIPLY_NV: c_uint = 0x9294;
    pub const MULTISAMPLE: c_uint = 0x809D;
    pub const MULTISAMPLES_NV: c_uint = 0x9371;
    pub const MULTISAMPLE_3DFX: c_uint = 0x86B2;
    pub const MULTISAMPLE_ARB: c_uint = 0x809D;
    pub const MULTISAMPLE_BIT_3DFX: c_uint = 0x20000000;
    pub const MULTISAMPLE_BIT_ARB: c_uint = 0x20000000;
    pub const MULTISAMPLE_BIT_EXT: c_uint = 0x20000000;
    pub const MULTISAMPLE_COVERAGE_MODES_NV: c_uint = 0x8E12;
    pub const MULTISAMPLE_EXT: c_uint = 0x809D;
    pub const MULTISAMPLE_FILTER_HINT_NV: c_uint = 0x8534;
    pub const MULTISAMPLE_LINE_WIDTH_GRANULARITY_ARB: c_uint = 0x9382;
    pub const MULTISAMPLE_LINE_WIDTH_RANGE_ARB: c_uint = 0x9381;
    pub const MULTISAMPLE_RASTERIZATION_ALLOWED_EXT: c_uint = 0x932B;
    pub const MULTISAMPLE_SGIS: c_uint = 0x809D;
    pub const MUL_ATI: c_uint = 0x8964;
    pub const MVP_MATRIX_EXT: c_uint = 0x87E3;
    pub const NAMED_STRING_LENGTH_ARB: c_uint = 0x8DE9;
    pub const NAMED_STRING_TYPE_ARB: c_uint = 0x8DEA;
    pub const NAME_LENGTH: c_uint = 0x92F9;
    pub const NAND: c_uint = 0x150E;
    pub const NATIVE_GRAPHICS_BEGIN_HINT_PGI: c_uint = 0x1A203;
    pub const NATIVE_GRAPHICS_END_HINT_PGI: c_uint = 0x1A204;
    pub const NATIVE_GRAPHICS_HANDLE_PGI: c_uint = 0x1A202;
    pub const NEAREST: c_uint = 0x2600;
    pub const NEAREST_CLIPMAP_LINEAR_SGIX: c_uint = 0x844E;
    pub const NEAREST_CLIPMAP_NEAREST_SGIX: c_uint = 0x844D;
    pub const NEAREST_MIPMAP_LINEAR: c_uint = 0x2702;
    pub const NEAREST_MIPMAP_NEAREST: c_uint = 0x2700;
    pub const NEGATE_BIT_ATI: c_uint = 0x00000004;
    pub const NEGATIVE_ONE_EXT: c_uint = 0x87DF;
    pub const NEGATIVE_ONE_TO_ONE: c_uint = 0x935E;
    pub const NEGATIVE_W_EXT: c_uint = 0x87DC;
    pub const NEGATIVE_X_EXT: c_uint = 0x87D9;
    pub const NEGATIVE_Y_EXT: c_uint = 0x87DA;
    pub const NEGATIVE_Z_EXT: c_uint = 0x87DB;
    pub const NEVER: c_uint = 0x0200;
    pub const NEXT_BUFFER_NV: c_int = -2;
    pub const NEXT_VIDEO_CAPTURE_BUFFER_STATUS_NV: c_uint = 0x9025;
    pub const NICEST: c_uint = 0x1102;
    pub const NONE: c_uint = 0;
    pub const NOOP: c_uint = 0x1505;
    pub const NOP_COMMAND_NV: c_uint = 0x0001;
    pub const NOR: c_uint = 0x1508;
    pub const NORMALIZED_RANGE_EXT: c_uint = 0x87E0;
    pub const NORMAL_ARRAY_ADDRESS_NV: c_uint = 0x8F22;
    pub const NORMAL_ARRAY_BUFFER_BINDING_ARB: c_uint = 0x8897;
    pub const NORMAL_ARRAY_COUNT_EXT: c_uint = 0x8080;
    pub const NORMAL_ARRAY_EXT: c_uint = 0x8075;
    pub const NORMAL_ARRAY_LENGTH_NV: c_uint = 0x8F2C;
    pub const NORMAL_ARRAY_LIST_IBM: c_uint = 103071;
    pub const NORMAL_ARRAY_LIST_STRIDE_IBM: c_uint = 103081;
    pub const NORMAL_ARRAY_PARALLEL_POINTERS_INTEL: c_uint = 0x83F6;
    pub const NORMAL_ARRAY_POINTER_EXT: c_uint = 0x808F;
    pub const NORMAL_ARRAY_STRIDE_EXT: c_uint = 0x807F;
    pub const NORMAL_ARRAY_TYPE_EXT: c_uint = 0x807E;
    pub const NORMAL_BIT_PGI: c_uint = 0x08000000;
    pub const NORMAL_MAP_ARB: c_uint = 0x8511;
    pub const NORMAL_MAP_EXT: c_uint = 0x8511;
    pub const NORMAL_MAP_NV: c_uint = 0x8511;
    pub const NOTEQUAL: c_uint = 0x0205;
    pub const NO_ERROR: c_uint = 0;
    pub const NO_RESET_NOTIFICATION: c_uint = 0x8261;
    pub const NO_RESET_NOTIFICATION_ARB: c_uint = 0x8261;
    pub const NUM_ACTIVE_VARIABLES: c_uint = 0x9304;
    pub const NUM_COMPATIBLE_SUBROUTINES: c_uint = 0x8E4A;
    pub const NUM_COMPRESSED_TEXTURE_FORMATS: c_uint = 0x86A2;
    pub const NUM_COMPRESSED_TEXTURE_FORMATS_ARB: c_uint = 0x86A2;
    pub const NUM_DEVICE_UUIDS_EXT: c_uint = 0x9596;
    pub const NUM_EXTENSIONS: c_uint = 0x821D;
    pub const NUM_FILL_STREAMS_NV: c_uint = 0x8E29;
    pub const NUM_FRAGMENT_CONSTANTS_ATI: c_uint = 0x896F;
    pub const NUM_FRAGMENT_REGISTERS_ATI: c_uint = 0x896E;
    pub const NUM_GENERAL_COMBINERS_NV: c_uint = 0x854E;
    pub const NUM_INPUT_INTERPOLATOR_COMPONENTS_ATI: c_uint = 0x8973;
    pub const NUM_INSTRUCTIONS_PER_PASS_ATI: c_uint = 0x8971;
    pub const NUM_INSTRUCTIONS_TOTAL_ATI: c_uint = 0x8972;
    pub const NUM_LOOPBACK_COMPONENTS_ATI: c_uint = 0x8974;
    pub const NUM_PASSES_ATI: c_uint = 0x8970;
    pub const NUM_PROGRAM_BINARY_FORMATS: c_uint = 0x87FE;
    pub const NUM_SAMPLE_COUNTS: c_uint = 0x9380;
    pub const NUM_SHADER_BINARY_FORMATS: c_uint = 0x8DF9;
    pub const NUM_SHADING_LANGUAGE_VERSIONS: c_uint = 0x82E9;
    pub const NUM_SPARSE_LEVELS_ARB: c_uint = 0x91AA;
    pub const NUM_SPIR_V_EXTENSIONS: c_uint = 0x9554;
    pub const NUM_SUPPORTED_MULTISAMPLE_MODES_AMD: c_uint = 0x91B6;
    pub const NUM_TILING_TYPES_EXT: c_uint = 0x9582;
    pub const NUM_VIDEO_CAPTURE_STREAMS_NV: c_uint = 0x9024;
    pub const NUM_VIRTUAL_PAGE_SIZES_ARB: c_uint = 0x91A8;
    pub const NUM_WINDOW_RECTANGLES_EXT: c_uint = 0x8F15;
    pub const OBJECT_ACTIVE_ATTRIBUTES_ARB: c_uint = 0x8B89;
    pub const OBJECT_ACTIVE_ATTRIBUTE_MAX_LENGTH_ARB: c_uint = 0x8B8A;
    pub const OBJECT_ACTIVE_UNIFORMS_ARB: c_uint = 0x8B86;
    pub const OBJECT_ACTIVE_UNIFORM_MAX_LENGTH_ARB: c_uint = 0x8B87;
    pub const OBJECT_ATTACHED_OBJECTS_ARB: c_uint = 0x8B85;
    pub const OBJECT_BUFFER_SIZE_ATI: c_uint = 0x8764;
    pub const OBJECT_BUFFER_USAGE_ATI: c_uint = 0x8765;
    pub const OBJECT_COMPILE_STATUS_ARB: c_uint = 0x8B81;
    pub const OBJECT_DELETE_STATUS_ARB: c_uint = 0x8B80;
    pub const OBJECT_DISTANCE_TO_LINE_SGIS: c_uint = 0x81F3;
    pub const OBJECT_DISTANCE_TO_POINT_SGIS: c_uint = 0x81F1;
    pub const OBJECT_INFO_LOG_LENGTH_ARB: c_uint = 0x8B84;
    pub const OBJECT_LINE_SGIS: c_uint = 0x81F7;
    pub const OBJECT_LINK_STATUS_ARB: c_uint = 0x8B82;
    pub const OBJECT_POINT_SGIS: c_uint = 0x81F5;
    pub const OBJECT_SHADER_SOURCE_LENGTH_ARB: c_uint = 0x8B88;
    pub const OBJECT_SUBTYPE_ARB: c_uint = 0x8B4F;
    pub const OBJECT_TYPE: c_uint = 0x9112;
    pub const OBJECT_TYPE_ARB: c_uint = 0x8B4E;
    pub const OBJECT_VALIDATE_STATUS_ARB: c_uint = 0x8B83;
    pub const OCCLUSION_QUERY_EVENT_MASK_AMD: c_uint = 0x874F;
    pub const OCCLUSION_TEST_HP: c_uint = 0x8165;
    pub const OCCLUSION_TEST_RESULT_HP: c_uint = 0x8166;
    pub const OFFSET: c_uint = 0x92FC;
    pub const OFFSET_HILO_PROJECTIVE_TEXTURE_2D_NV: c_uint = 0x8856;
    pub const OFFSET_HILO_PROJECTIVE_TEXTURE_RECTANGLE_NV: c_uint = 0x8857;
    pub const OFFSET_HILO_TEXTURE_2D_NV: c_uint = 0x8854;
    pub const OFFSET_HILO_TEXTURE_RECTANGLE_NV: c_uint = 0x8855;
    pub const OFFSET_PROJECTIVE_TEXTURE_2D_NV: c_uint = 0x8850;
    pub const OFFSET_PROJECTIVE_TEXTURE_2D_SCALE_NV: c_uint = 0x8851;
    pub const OFFSET_PROJECTIVE_TEXTURE_RECTANGLE_NV: c_uint = 0x8852;
    pub const OFFSET_PROJECTIVE_TEXTURE_RECTANGLE_SCALE_NV: c_uint = 0x8853;
    pub const OFFSET_TEXTURE_2D_BIAS_NV: c_uint = 0x86E3;
    pub const OFFSET_TEXTURE_2D_MATRIX_NV: c_uint = 0x86E1;
    pub const OFFSET_TEXTURE_2D_NV: c_uint = 0x86E8;
    pub const OFFSET_TEXTURE_2D_SCALE_NV: c_uint = 0x86E2;
    pub const OFFSET_TEXTURE_BIAS_NV: c_uint = 0x86E3;
    pub const OFFSET_TEXTURE_MATRIX_NV: c_uint = 0x86E1;
    pub const OFFSET_TEXTURE_RECTANGLE_NV: c_uint = 0x864C;
    pub const OFFSET_TEXTURE_RECTANGLE_SCALE_NV: c_uint = 0x864D;
    pub const OFFSET_TEXTURE_SCALE_NV: c_uint = 0x86E2;
    pub const ONE: c_uint = 1;
    pub const ONE_EXT: c_uint = 0x87DE;
    pub const ONE_MINUS_CONSTANT_ALPHA: c_uint = 0x8004;
    pub const ONE_MINUS_CONSTANT_ALPHA_EXT: c_uint = 0x8004;
    pub const ONE_MINUS_CONSTANT_COLOR: c_uint = 0x8002;
    pub const ONE_MINUS_CONSTANT_COLOR_EXT: c_uint = 0x8002;
    pub const ONE_MINUS_DST_ALPHA: c_uint = 0x0305;
    pub const ONE_MINUS_DST_COLOR: c_uint = 0x0307;
    pub const ONE_MINUS_SRC1_ALPHA: c_uint = 0x88FB;
    pub const ONE_MINUS_SRC1_COLOR: c_uint = 0x88FA;
    pub const ONE_MINUS_SRC_ALPHA: c_uint = 0x0303;
    pub const ONE_MINUS_SRC_COLOR: c_uint = 0x0301;
    pub const OPERAND0_ALPHA_ARB: c_uint = 0x8598;
    pub const OPERAND0_ALPHA_EXT: c_uint = 0x8598;
    pub const OPERAND0_RGB_ARB: c_uint = 0x8590;
    pub const OPERAND0_RGB_EXT: c_uint = 0x8590;
    pub const OPERAND1_ALPHA_ARB: c_uint = 0x8599;
    pub const OPERAND1_ALPHA_EXT: c_uint = 0x8599;
    pub const OPERAND1_RGB_ARB: c_uint = 0x8591;
    pub const OPERAND1_RGB_EXT: c_uint = 0x8591;
    pub const OPERAND2_ALPHA_ARB: c_uint = 0x859A;
    pub const OPERAND2_ALPHA_EXT: c_uint = 0x859A;
    pub const OPERAND2_RGB_ARB: c_uint = 0x8592;
    pub const OPERAND2_RGB_EXT: c_uint = 0x8592;
    pub const OPERAND3_ALPHA_NV: c_uint = 0x859B;
    pub const OPERAND3_RGB_NV: c_uint = 0x8593;
    pub const OPTIMAL_TILING_EXT: c_uint = 0x9584;
    pub const OP_ADD_EXT: c_uint = 0x8787;
    pub const OP_CLAMP_EXT: c_uint = 0x878E;
    pub const OP_CROSS_PRODUCT_EXT: c_uint = 0x8797;
    pub const OP_DOT3_EXT: c_uint = 0x8784;
    pub const OP_DOT4_EXT: c_uint = 0x8785;
    pub const OP_EXP_BASE_2_EXT: c_uint = 0x8791;
    pub const OP_FLOOR_EXT: c_uint = 0x878F;
    pub const OP_FRAC_EXT: c_uint = 0x8789;
    pub const OP_INDEX_EXT: c_uint = 0x8782;
    pub const OP_LOG_BASE_2_EXT: c_uint = 0x8792;
    pub const OP_MADD_EXT: c_uint = 0x8788;
    pub const OP_MAX_EXT: c_uint = 0x878A;
    pub const OP_MIN_EXT: c_uint = 0x878B;
    pub const OP_MOV_EXT: c_uint = 0x8799;
    pub const OP_MULTIPLY_MATRIX_EXT: c_uint = 0x8798;
    pub const OP_MUL_EXT: c_uint = 0x8786;
    pub const OP_NEGATE_EXT: c_uint = 0x8783;
    pub const OP_POWER_EXT: c_uint = 0x8793;
    pub const OP_RECIP_EXT: c_uint = 0x8794;
    pub const OP_RECIP_SQRT_EXT: c_uint = 0x8795;
    pub const OP_ROUND_EXT: c_uint = 0x8790;
    pub const OP_SET_GE_EXT: c_uint = 0x878C;
    pub const OP_SET_LT_EXT: c_uint = 0x878D;
    pub const OP_SUB_EXT: c_uint = 0x8796;
    pub const OR: c_uint = 0x1507;
    pub const OR_INVERTED: c_uint = 0x150D;
    pub const OR_REVERSE: c_uint = 0x150B;
    pub const OUTPUT_COLOR0_EXT: c_uint = 0x879B;
    pub const OUTPUT_COLOR1_EXT: c_uint = 0x879C;
    pub const OUTPUT_FOG_EXT: c_uint = 0x87BD;
    pub const OUTPUT_TEXTURE_COORD0_EXT: c_uint = 0x879D;
    pub const OUTPUT_TEXTURE_COORD10_EXT: c_uint = 0x87A7;
    pub const OUTPUT_TEXTURE_COORD11_EXT: c_uint = 0x87A8;
    pub const OUTPUT_TEXTURE_COORD12_EXT: c_uint = 0x87A9;
    pub const OUTPUT_TEXTURE_COORD13_EXT: c_uint = 0x87AA;
    pub const OUTPUT_TEXTURE_COORD14_EXT: c_uint = 0x87AB;
    pub const OUTPUT_TEXTURE_COORD15_EXT: c_uint = 0x87AC;
    pub const OUTPUT_TEXTURE_COORD16_EXT: c_uint = 0x87AD;
    pub const OUTPUT_TEXTURE_COORD17_EXT: c_uint = 0x87AE;
    pub const OUTPUT_TEXTURE_COORD18_EXT: c_uint = 0x87AF;
    pub const OUTPUT_TEXTURE_COORD19_EXT: c_uint = 0x87B0;
    pub const OUTPUT_TEXTURE_COORD1_EXT: c_uint = 0x879E;
    pub const OUTPUT_TEXTURE_COORD20_EXT: c_uint = 0x87B1;
    pub const OUTPUT_TEXTURE_COORD21_EXT: c_uint = 0x87B2;
    pub const OUTPUT_TEXTURE_COORD22_EXT: c_uint = 0x87B3;
    pub const OUTPUT_TEXTURE_COORD23_EXT: c_uint = 0x87B4;
    pub const OUTPUT_TEXTURE_COORD24_EXT: c_uint = 0x87B5;
    pub const OUTPUT_TEXTURE_COORD25_EXT: c_uint = 0x87B6;
    pub const OUTPUT_TEXTURE_COORD26_EXT: c_uint = 0x87B7;
    pub const OUTPUT_TEXTURE_COORD27_EXT: c_uint = 0x87B8;
    pub const OUTPUT_TEXTURE_COORD28_EXT: c_uint = 0x87B9;
    pub const OUTPUT_TEXTURE_COORD29_EXT: c_uint = 0x87BA;
    pub const OUTPUT_TEXTURE_COORD2_EXT: c_uint = 0x879F;
    pub const OUTPUT_TEXTURE_COORD30_EXT: c_uint = 0x87BB;
    pub const OUTPUT_TEXTURE_COORD31_EXT: c_uint = 0x87BC;
    pub const OUTPUT_TEXTURE_COORD3_EXT: c_uint = 0x87A0;
    pub const OUTPUT_TEXTURE_COORD4_EXT: c_uint = 0x87A1;
    pub const OUTPUT_TEXTURE_COORD5_EXT: c_uint = 0x87A2;
    pub const OUTPUT_TEXTURE_COORD6_EXT: c_uint = 0x87A3;
    pub const OUTPUT_TEXTURE_COORD7_EXT: c_uint = 0x87A4;
    pub const OUTPUT_TEXTURE_COORD8_EXT: c_uint = 0x87A5;
    pub const OUTPUT_TEXTURE_COORD9_EXT: c_uint = 0x87A6;
    pub const OUTPUT_VERTEX_EXT: c_uint = 0x879A;
    pub const OUT_OF_MEMORY: c_uint = 0x0505;
    pub const OVERLAY_KHR: c_uint = 0x9296;
    pub const OVERLAY_NV: c_uint = 0x9296;
    pub const PACK_ALIGNMENT: c_uint = 0x0D05;
    pub const PACK_CMYK_HINT_EXT: c_uint = 0x800E;
    pub const PACK_COMPRESSED_BLOCK_DEPTH: c_uint = 0x912D;
    pub const PACK_COMPRESSED_BLOCK_HEIGHT: c_uint = 0x912C;
    pub const PACK_COMPRESSED_BLOCK_SIZE: c_uint = 0x912E;
    pub const PACK_COMPRESSED_BLOCK_WIDTH: c_uint = 0x912B;
    pub const PACK_IMAGE_DEPTH_SGIS: c_uint = 0x8131;
    pub const PACK_IMAGE_HEIGHT: c_uint = 0x806C;
    pub const PACK_IMAGE_HEIGHT_EXT: c_uint = 0x806C;
    pub const PACK_INVERT_MESA: c_uint = 0x8758;
    pub const PACK_LSB_FIRST: c_uint = 0x0D01;
    pub const PACK_RESAMPLE_OML: c_uint = 0x8984;
    pub const PACK_RESAMPLE_SGIX: c_uint = 0x842E;
    pub const PACK_ROW_BYTES_APPLE: c_uint = 0x8A15;
    pub const PACK_ROW_LENGTH: c_uint = 0x0D02;
    pub const PACK_SKIP_IMAGES: c_uint = 0x806B;
    pub const PACK_SKIP_IMAGES_EXT: c_uint = 0x806B;
    pub const PACK_SKIP_PIXELS: c_uint = 0x0D04;
    pub const PACK_SKIP_ROWS: c_uint = 0x0D03;
    pub const PACK_SKIP_VOLUMES_SGIS: c_uint = 0x8130;
    pub const PACK_SUBSAMPLE_RATE_SGIX: c_uint = 0x85A0;
    pub const PACK_SWAP_BYTES: c_uint = 0x0D00;
    pub const PALETTE4_R5_G6_B5_OES: c_uint = 0x8B92;
    pub const PALETTE4_RGB5_A1_OES: c_uint = 0x8B94;
    pub const PALETTE4_RGB8_OES: c_uint = 0x8B90;
    pub const PALETTE4_RGBA4_OES: c_uint = 0x8B93;
    pub const PALETTE4_RGBA8_OES: c_uint = 0x8B91;
    pub const PALETTE8_R5_G6_B5_OES: c_uint = 0x8B97;
    pub const PALETTE8_RGB5_A1_OES: c_uint = 0x8B99;
    pub const PALETTE8_RGB8_OES: c_uint = 0x8B95;
    pub const PALETTE8_RGBA4_OES: c_uint = 0x8B98;
    pub const PALETTE8_RGBA8_OES: c_uint = 0x8B96;
    pub const PARALLEL_ARRAYS_INTEL: c_uint = 0x83F4;
    pub const PARAMETER_BUFFER: c_uint = 0x80EE;
    pub const PARAMETER_BUFFER_ARB: c_uint = 0x80EE;
    pub const PARAMETER_BUFFER_BINDING: c_uint = 0x80EF;
    pub const PARAMETER_BUFFER_BINDING_ARB: c_uint = 0x80EF;
    pub const PARTIAL_SUCCESS_NV: c_uint = 0x902E;
    pub const PASS_THROUGH_NV: c_uint = 0x86E6;
    pub const PATCHES: c_uint = 0x000E;
    pub const PATCH_DEFAULT_INNER_LEVEL: c_uint = 0x8E73;
    pub const PATCH_DEFAULT_OUTER_LEVEL: c_uint = 0x8E74;
    pub const PATCH_VERTICES: c_uint = 0x8E72;
    pub const PATH_CLIENT_LENGTH_NV: c_uint = 0x907F;
    pub const PATH_COMMAND_COUNT_NV: c_uint = 0x909D;
    pub const PATH_COMPUTED_LENGTH_NV: c_uint = 0x90A0;
    pub const PATH_COORD_COUNT_NV: c_uint = 0x909E;
    pub const PATH_COVER_DEPTH_FUNC_NV: c_uint = 0x90BF;
    pub const PATH_DASH_ARRAY_COUNT_NV: c_uint = 0x909F;
    pub const PATH_DASH_CAPS_NV: c_uint = 0x907B;
    pub const PATH_DASH_OFFSET_NV: c_uint = 0x907E;
    pub const PATH_DASH_OFFSET_RESET_NV: c_uint = 0x90B4;
    pub const PATH_END_CAPS_NV: c_uint = 0x9076;
    pub const PATH_ERROR_POSITION_NV: c_uint = 0x90AB;
    pub const PATH_FILL_BOUNDING_BOX_NV: c_uint = 0x90A1;
    pub const PATH_FILL_COVER_MODE_NV: c_uint = 0x9082;
    pub const PATH_FILL_MASK_NV: c_uint = 0x9081;
    pub const PATH_FILL_MODE_NV: c_uint = 0x9080;
    pub const PATH_FORMAT_PS_NV: c_uint = 0x9071;
    pub const PATH_FORMAT_SVG_NV: c_uint = 0x9070;
    pub const PATH_GEN_COEFF_NV: c_uint = 0x90B1;
    pub const PATH_GEN_COMPONENTS_NV: c_uint = 0x90B3;
    pub const PATH_GEN_MODE_NV: c_uint = 0x90B0;
    pub const PATH_INITIAL_DASH_CAP_NV: c_uint = 0x907C;
    pub const PATH_INITIAL_END_CAP_NV: c_uint = 0x9077;
    pub const PATH_JOIN_STYLE_NV: c_uint = 0x9079;
    pub const PATH_MAX_MODELVIEW_STACK_DEPTH_NV: c_uint = 0x0D36;
    pub const PATH_MAX_PROJECTION_STACK_DEPTH_NV: c_uint = 0x0D38;
    pub const PATH_MITER_LIMIT_NV: c_uint = 0x907A;
    pub const PATH_MODELVIEW_MATRIX_NV: c_uint = 0x0BA6;
    pub const PATH_MODELVIEW_NV: c_uint = 0x1700;
    pub const PATH_MODELVIEW_STACK_DEPTH_NV: c_uint = 0x0BA3;
    pub const PATH_OBJECT_BOUNDING_BOX_NV: c_uint = 0x908A;
    pub const PATH_PROJECTION_MATRIX_NV: c_uint = 0x0BA7;
    pub const PATH_PROJECTION_NV: c_uint = 0x1701;
    pub const PATH_PROJECTION_STACK_DEPTH_NV: c_uint = 0x0BA4;
    pub const PATH_STENCIL_DEPTH_OFFSET_FACTOR_NV: c_uint = 0x90BD;
    pub const PATH_STENCIL_DEPTH_OFFSET_UNITS_NV: c_uint = 0x90BE;
    pub const PATH_STENCIL_FUNC_NV: c_uint = 0x90B7;
    pub const PATH_STENCIL_REF_NV: c_uint = 0x90B8;
    pub const PATH_STENCIL_VALUE_MASK_NV: c_uint = 0x90B9;
    pub const PATH_STROKE_BOUNDING_BOX_NV: c_uint = 0x90A2;
    pub const PATH_STROKE_COVER_MODE_NV: c_uint = 0x9083;
    pub const PATH_STROKE_MASK_NV: c_uint = 0x9084;
    pub const PATH_STROKE_WIDTH_NV: c_uint = 0x9075;
    pub const PATH_TERMINAL_DASH_CAP_NV: c_uint = 0x907D;
    pub const PATH_TERMINAL_END_CAP_NV: c_uint = 0x9078;
    pub const PATH_TRANSPOSE_MODELVIEW_MATRIX_NV: c_uint = 0x84E3;
    pub const PATH_TRANSPOSE_PROJECTION_MATRIX_NV: c_uint = 0x84E4;
    pub const PERCENTAGE_AMD: c_uint = 0x8BC3;
    pub const PERFMON_RESULT_AMD: c_uint = 0x8BC6;
    pub const PERFMON_RESULT_AVAILABLE_AMD: c_uint = 0x8BC4;
    pub const PERFMON_RESULT_SIZE_AMD: c_uint = 0x8BC5;
    pub const PERFORMANCE_MONITOR_AMD: c_uint = 0x9152;
    pub const PERFQUERY_COUNTER_DATA_BOOL32_INTEL: c_uint = 0x94FC;
    pub const PERFQUERY_COUNTER_DATA_DOUBLE_INTEL: c_uint = 0x94FB;
    pub const PERFQUERY_COUNTER_DATA_FLOAT_INTEL: c_uint = 0x94FA;
    pub const PERFQUERY_COUNTER_DATA_UINT32_INTEL: c_uint = 0x94F8;
    pub const PERFQUERY_COUNTER_DATA_UINT64_INTEL: c_uint = 0x94F9;
    pub const PERFQUERY_COUNTER_DESC_LENGTH_MAX_INTEL: c_uint = 0x94FF;
    pub const PERFQUERY_COUNTER_DURATION_NORM_INTEL: c_uint = 0x94F1;
    pub const PERFQUERY_COUNTER_DURATION_RAW_INTEL: c_uint = 0x94F2;
    pub const PERFQUERY_COUNTER_EVENT_INTEL: c_uint = 0x94F0;
    pub const PERFQUERY_COUNTER_NAME_LENGTH_MAX_INTEL: c_uint = 0x94FE;
    pub const PERFQUERY_COUNTER_RAW_INTEL: c_uint = 0x94F4;
    pub const PERFQUERY_COUNTER_THROUGHPUT_INTEL: c_uint = 0x94F3;
    pub const PERFQUERY_COUNTER_TIMESTAMP_INTEL: c_uint = 0x94F5;
    pub const PERFQUERY_DONOT_FLUSH_INTEL: c_uint = 0x83F9;
    pub const PERFQUERY_FLUSH_INTEL: c_uint = 0x83FA;
    pub const PERFQUERY_GLOBAL_CONTEXT_INTEL: c_uint = 0x00000001;
    pub const PERFQUERY_GPA_EXTENDED_COUNTERS_INTEL: c_uint = 0x9500;
    pub const PERFQUERY_QUERY_NAME_LENGTH_MAX_INTEL: c_uint = 0x94FD;
    pub const PERFQUERY_SINGLE_CONTEXT_INTEL: c_uint = 0x00000000;
    pub const PERFQUERY_WAIT_INTEL: c_uint = 0x83FB;
    pub const PERTURB_EXT: c_uint = 0x85AE;
    pub const PER_GPU_STORAGE_BIT_NV: c_uint = 0x0800;
    pub const PER_GPU_STORAGE_NV: c_uint = 0x9548;
    pub const PER_STAGE_CONSTANTS_NV: c_uint = 0x8535;
    pub const PHONG_HINT_WIN: c_uint = 0x80EB;
    pub const PHONG_WIN: c_uint = 0x80EA;
    pub const PINLIGHT_NV: c_uint = 0x92A8;
    pub const PIXELS_PER_SAMPLE_PATTERN_X_AMD: c_uint = 0x91AE;
    pub const PIXELS_PER_SAMPLE_PATTERN_Y_AMD: c_uint = 0x91AF;
    pub const PIXEL_BUFFER_BARRIER_BIT: c_uint = 0x00000080;
    pub const PIXEL_BUFFER_BARRIER_BIT_EXT: c_uint = 0x00000080;
    pub const PIXEL_COUNTER_BITS_NV: c_uint = 0x8864;
    pub const PIXEL_COUNT_AVAILABLE_NV: c_uint = 0x8867;
    pub const PIXEL_COUNT_NV: c_uint = 0x8866;
    pub const PIXEL_CUBIC_WEIGHT_EXT: c_uint = 0x8333;
    pub const PIXEL_FRAGMENT_ALPHA_SOURCE_SGIS: c_uint = 0x8355;
    pub const PIXEL_FRAGMENT_RGB_SOURCE_SGIS: c_uint = 0x8354;
    pub const PIXEL_GROUP_COLOR_SGIS: c_uint = 0x8356;
    pub const PIXEL_MAG_FILTER_EXT: c_uint = 0x8331;
    pub const PIXEL_MIN_FILTER_EXT: c_uint = 0x8332;
    pub const PIXEL_PACK_BUFFER: c_uint = 0x88EB;
    pub const PIXEL_PACK_BUFFER_ARB: c_uint = 0x88EB;
    pub const PIXEL_PACK_BUFFER_BINDING: c_uint = 0x88ED;
    pub const PIXEL_PACK_BUFFER_BINDING_ARB: c_uint = 0x88ED;
    pub const PIXEL_PACK_BUFFER_BINDING_EXT: c_uint = 0x88ED;
    pub const PIXEL_PACK_BUFFER_EXT: c_uint = 0x88EB;
    pub const PIXEL_SUBSAMPLE_2424_SGIX: c_uint = 0x85A3;
    pub const PIXEL_SUBSAMPLE_4242_SGIX: c_uint = 0x85A4;
    pub const PIXEL_SUBSAMPLE_4444_SGIX: c_uint = 0x85A2;
    pub const PIXEL_TEXTURE_SGIS: c_uint = 0x8353;
    pub const PIXEL_TEX_GEN_MODE_SGIX: c_uint = 0x832B;
    pub const PIXEL_TEX_GEN_SGIX: c_uint = 0x8139;
    pub const PIXEL_TILE_BEST_ALIGNMENT_SGIX: c_uint = 0x813E;
    pub const PIXEL_TILE_CACHE_INCREMENT_SGIX: c_uint = 0x813F;
    pub const PIXEL_TILE_CACHE_SIZE_SGIX: c_uint = 0x8145;
    pub const PIXEL_TILE_GRID_DEPTH_SGIX: c_uint = 0x8144;
    pub const PIXEL_TILE_GRID_HEIGHT_SGIX: c_uint = 0x8143;
    pub const PIXEL_TILE_GRID_WIDTH_SGIX: c_uint = 0x8142;
    pub const PIXEL_TILE_HEIGHT_SGIX: c_uint = 0x8141;
    pub const PIXEL_TILE_WIDTH_SGIX: c_uint = 0x8140;
    pub const PIXEL_TRANSFORM_2D_EXT: c_uint = 0x8330;
    pub const PIXEL_TRANSFORM_2D_MATRIX_EXT: c_uint = 0x8338;
    pub const PIXEL_TRANSFORM_2D_STACK_DEPTH_EXT: c_uint = 0x8336;
    pub const PIXEL_UNPACK_BUFFER: c_uint = 0x88EC;
    pub const PIXEL_UNPACK_BUFFER_ARB: c_uint = 0x88EC;
    pub const PIXEL_UNPACK_BUFFER_BINDING: c_uint = 0x88EF;
    pub const PIXEL_UNPACK_BUFFER_BINDING_ARB: c_uint = 0x88EF;
    pub const PIXEL_UNPACK_BUFFER_BINDING_EXT: c_uint = 0x88EF;
    pub const PIXEL_UNPACK_BUFFER_EXT: c_uint = 0x88EC;
    pub const PLUS_CLAMPED_ALPHA_NV: c_uint = 0x92B2;
    pub const PLUS_CLAMPED_NV: c_uint = 0x92B1;
    pub const PLUS_DARKER_NV: c_uint = 0x9292;
    pub const PLUS_NV: c_uint = 0x9291;
    pub const PN_TRIANGLES_ATI: c_uint = 0x87F0;
    pub const PN_TRIANGLES_NORMAL_MODE_ATI: c_uint = 0x87F3;
    pub const PN_TRIANGLES_NORMAL_MODE_LINEAR_ATI: c_uint = 0x87F7;
    pub const PN_TRIANGLES_NORMAL_MODE_QUADRATIC_ATI: c_uint = 0x87F8;
    pub const PN_TRIANGLES_POINT_MODE_ATI: c_uint = 0x87F2;
    pub const PN_TRIANGLES_POINT_MODE_CUBIC_ATI: c_uint = 0x87F6;
    pub const PN_TRIANGLES_POINT_MODE_LINEAR_ATI: c_uint = 0x87F5;
    pub const PN_TRIANGLES_TESSELATION_LEVEL_ATI: c_uint = 0x87F4;
    pub const POINT: c_uint = 0x1B00;
    pub const POINTS: c_uint = 0x0000;
    pub const POINT_DISTANCE_ATTENUATION_ARB: c_uint = 0x8129;
    pub const POINT_FADE_THRESHOLD_SIZE: c_uint = 0x8128;
    pub const POINT_FADE_THRESHOLD_SIZE_ARB: c_uint = 0x8128;
    pub const POINT_FADE_THRESHOLD_SIZE_EXT: c_uint = 0x8128;
    pub const POINT_FADE_THRESHOLD_SIZE_SGIS: c_uint = 0x8128;
    pub const POINT_SIZE: c_uint = 0x0B11;
    pub const POINT_SIZE_GRANULARITY: c_uint = 0x0B13;
    pub const POINT_SIZE_MAX_ARB: c_uint = 0x8127;
    pub const POINT_SIZE_MAX_EXT: c_uint = 0x8127;
    pub const POINT_SIZE_MAX_SGIS: c_uint = 0x8127;
    pub const POINT_SIZE_MIN_ARB: c_uint = 0x8126;
    pub const POINT_SIZE_MIN_EXT: c_uint = 0x8126;
    pub const POINT_SIZE_MIN_SGIS: c_uint = 0x8126;
    pub const POINT_SIZE_RANGE: c_uint = 0x0B12;
    pub const POINT_SPRITE_ARB: c_uint = 0x8861;
    pub const POINT_SPRITE_COORD_ORIGIN: c_uint = 0x8CA0;
    pub const POINT_SPRITE_NV: c_uint = 0x8861;
    pub const POINT_SPRITE_R_MODE_NV: c_uint = 0x8863;
    pub const POLYGON_MODE: c_uint = 0x0B40;
    pub const POLYGON_OFFSET_BIAS_EXT: c_uint = 0x8039;
    pub const POLYGON_OFFSET_CLAMP: c_uint = 0x8E1B;
    pub const POLYGON_OFFSET_CLAMP_EXT: c_uint = 0x8E1B;
    pub const POLYGON_OFFSET_COMMAND_NV: c_uint = 0x000E;
    pub const POLYGON_OFFSET_EXT: c_uint = 0x8037;
    pub const POLYGON_OFFSET_FACTOR: c_uint = 0x8038;
    pub const POLYGON_OFFSET_FACTOR_EXT: c_uint = 0x8038;
    pub const POLYGON_OFFSET_FILL: c_uint = 0x8037;
    pub const POLYGON_OFFSET_LINE: c_uint = 0x2A02;
    pub const POLYGON_OFFSET_POINT: c_uint = 0x2A01;
    pub const POLYGON_OFFSET_UNITS: c_uint = 0x2A00;
    pub const POLYGON_SMOOTH: c_uint = 0x0B41;
    pub const POLYGON_SMOOTH_HINT: c_uint = 0x0C53;
    pub const POST_COLOR_MATRIX_ALPHA_BIAS_SGI: c_uint = 0x80BB;
    pub const POST_COLOR_MATRIX_ALPHA_SCALE_SGI: c_uint = 0x80B7;
    pub const POST_COLOR_MATRIX_BLUE_BIAS_SGI: c_uint = 0x80BA;
    pub const POST_COLOR_MATRIX_BLUE_SCALE_SGI: c_uint = 0x80B6;
    pub const POST_COLOR_MATRIX_COLOR_TABLE_SGI: c_uint = 0x80D2;
    pub const POST_COLOR_MATRIX_GREEN_BIAS_SGI: c_uint = 0x80B9;
    pub const POST_COLOR_MATRIX_GREEN_SCALE_SGI: c_uint = 0x80B5;
    pub const POST_COLOR_MATRIX_RED_BIAS_SGI: c_uint = 0x80B8;
    pub const POST_COLOR_MATRIX_RED_SCALE_SGI: c_uint = 0x80B4;
    pub const POST_CONVOLUTION_ALPHA_BIAS_EXT: c_uint = 0x8023;
    pub const POST_CONVOLUTION_ALPHA_SCALE_EXT: c_uint = 0x801F;
    pub const POST_CONVOLUTION_BLUE_BIAS_EXT: c_uint = 0x8022;
    pub const POST_CONVOLUTION_BLUE_SCALE_EXT: c_uint = 0x801E;
    pub const POST_CONVOLUTION_COLOR_TABLE_SGI: c_uint = 0x80D1;
    pub const POST_CONVOLUTION_GREEN_BIAS_EXT: c_uint = 0x8021;
    pub const POST_CONVOLUTION_GREEN_SCALE_EXT: c_uint = 0x801D;
    pub const POST_CONVOLUTION_RED_BIAS_EXT: c_uint = 0x8020;
    pub const POST_CONVOLUTION_RED_SCALE_EXT: c_uint = 0x801C;
    pub const POST_IMAGE_TRANSFORM_COLOR_TABLE_HP: c_uint = 0x8162;
    pub const POST_TEXTURE_FILTER_BIAS_RANGE_SGIX: c_uint = 0x817B;
    pub const POST_TEXTURE_FILTER_BIAS_SGIX: c_uint = 0x8179;
    pub const POST_TEXTURE_FILTER_SCALE_RANGE_SGIX: c_uint = 0x817C;
    pub const POST_TEXTURE_FILTER_SCALE_SGIX: c_uint = 0x817A;
    pub const PREFER_DOUBLEBUFFER_HINT_PGI: c_uint = 0x1A1F8;
    pub const PRESENT_DURATION_NV: c_uint = 0x8E2B;
    pub const PRESENT_TIME_NV: c_uint = 0x8E2A;
    pub const PRESERVE_ATI: c_uint = 0x8762;
    pub const PREVIOUS_ARB: c_uint = 0x8578;
    pub const PREVIOUS_EXT: c_uint = 0x8578;
    pub const PREVIOUS_TEXTURE_INPUT_NV: c_uint = 0x86E4;
    pub const PRIMARY_COLOR_ARB: c_uint = 0x8577;
    pub const PRIMARY_COLOR_EXT: c_uint = 0x8577;
    pub const PRIMARY_COLOR_NV: c_uint = 0x852C;
    pub const PRIMITIVES_GENERATED: c_uint = 0x8C87;
    pub const PRIMITIVES_GENERATED_EXT: c_uint = 0x8C87;
    pub const PRIMITIVES_GENERATED_NV: c_uint = 0x8C87;
    pub const PRIMITIVES_SUBMITTED: c_uint = 0x82EF;
    pub const PRIMITIVES_SUBMITTED_ARB: c_uint = 0x82EF;
    pub const PRIMITIVE_BOUNDING_BOX_ARB: c_uint = 0x92BE;
    pub const PRIMITIVE_ID_NV: c_uint = 0x8C7C;
    pub const PRIMITIVE_RESTART: c_uint = 0x8F9D;
    pub const PRIMITIVE_RESTART_FIXED_INDEX: c_uint = 0x8D69;
    pub const PRIMITIVE_RESTART_FOR_PATCHES_SUPPORTED: c_uint = 0x8221;
    pub const PRIMITIVE_RESTART_INDEX: c_uint = 0x8F9E;
    pub const PRIMITIVE_RESTART_INDEX_NV: c_uint = 0x8559;
    pub const PRIMITIVE_RESTART_NV: c_uint = 0x8558;
    pub const PROGRAM: c_uint = 0x82E2;
    pub const PROGRAMMABLE_SAMPLE_LOCATION_ARB: c_uint = 0x9341;
    pub const PROGRAMMABLE_SAMPLE_LOCATION_NV: c_uint = 0x9341;
    pub const PROGRAMMABLE_SAMPLE_LOCATION_TABLE_SIZE_ARB: c_uint = 0x9340;
    pub const PROGRAMMABLE_SAMPLE_LOCATION_TABLE_SIZE_NV: c_uint = 0x9340;
    pub const PROGRAM_ADDRESS_REGISTERS_ARB: c_uint = 0x88B0;
    pub const PROGRAM_ALU_INSTRUCTIONS_ARB: c_uint = 0x8805;
    pub const PROGRAM_ATTRIBS_ARB: c_uint = 0x88AC;
    pub const PROGRAM_ATTRIB_COMPONENTS_NV: c_uint = 0x8906;
    pub const PROGRAM_BINARY_FORMATS: c_uint = 0x87FF;
    pub const PROGRAM_BINARY_FORMAT_MESA: c_uint = 0x875F;
    pub const PROGRAM_BINARY_LENGTH: c_uint = 0x8741;
    pub const PROGRAM_BINARY_RETRIEVABLE_HINT: c_uint = 0x8257;
    pub const PROGRAM_BINDING_ARB: c_uint = 0x8677;
    pub const PROGRAM_ERROR_POSITION_ARB: c_uint = 0x864B;
    pub const PROGRAM_ERROR_POSITION_NV: c_uint = 0x864B;
    pub const PROGRAM_ERROR_STRING_ARB: c_uint = 0x8874;
    pub const PROGRAM_ERROR_STRING_NV: c_uint = 0x8874;
    pub const PROGRAM_FORMAT_ARB: c_uint = 0x8876;
    pub const PROGRAM_FORMAT_ASCII_ARB: c_uint = 0x8875;
    pub const PROGRAM_INPUT: c_uint = 0x92E3;
    pub const PROGRAM_INSTRUCTIONS_ARB: c_uint = 0x88A0;
    pub const PROGRAM_LENGTH_ARB: c_uint = 0x8627;
    pub const PROGRAM_LENGTH_NV: c_uint = 0x8627;
    pub const PROGRAM_MATRIX_EXT: c_uint = 0x8E2D;
    pub const PROGRAM_MATRIX_STACK_DEPTH_EXT: c_uint = 0x8E2F;
    pub const PROGRAM_NATIVE_ADDRESS_REGISTERS_ARB: c_uint = 0x88B2;
    pub const PROGRAM_NATIVE_ALU_INSTRUCTIONS_ARB: c_uint = 0x8808;
    pub const PROGRAM_NATIVE_ATTRIBS_ARB: c_uint = 0x88AE;
    pub const PROGRAM_NATIVE_INSTRUCTIONS_ARB: c_uint = 0x88A2;
    pub const PROGRAM_NATIVE_PARAMETERS_ARB: c_uint = 0x88AA;
    pub const PROGRAM_NATIVE_TEMPORARIES_ARB: c_uint = 0x88A6;
    pub const PROGRAM_NATIVE_TEX_INDIRECTIONS_ARB: c_uint = 0x880A;
    pub const PROGRAM_NATIVE_TEX_INSTRUCTIONS_ARB: c_uint = 0x8809;
    pub const PROGRAM_OBJECT_ARB: c_uint = 0x8B40;
    pub const PROGRAM_OBJECT_EXT: c_uint = 0x8B40;
    pub const PROGRAM_OUTPUT: c_uint = 0x92E4;
    pub const PROGRAM_PARAMETERS_ARB: c_uint = 0x88A8;
    pub const PROGRAM_PARAMETER_NV: c_uint = 0x8644;
    pub const PROGRAM_PIPELINE: c_uint = 0x82E4;
    pub const PROGRAM_PIPELINE_BINDING: c_uint = 0x825A;
    pub const PROGRAM_PIPELINE_OBJECT_EXT: c_uint = 0x8A4F;
    pub const PROGRAM_POINT_SIZE: c_uint = 0x8642;
    pub const PROGRAM_POINT_SIZE_ARB: c_uint = 0x8642;
    pub const PROGRAM_POINT_SIZE_EXT: c_uint = 0x8642;
    pub const PROGRAM_RESIDENT_NV: c_uint = 0x8647;
    pub const PROGRAM_RESULT_COMPONENTS_NV: c_uint = 0x8907;
    pub const PROGRAM_SEPARABLE: c_uint = 0x8258;
    pub const PROGRAM_STRING_ARB: c_uint = 0x8628;
    pub const PROGRAM_STRING_NV: c_uint = 0x8628;
    pub const PROGRAM_TARGET_NV: c_uint = 0x8646;
    pub const PROGRAM_TEMPORARIES_ARB: c_uint = 0x88A4;
    pub const PROGRAM_TEX_INDIRECTIONS_ARB: c_uint = 0x8807;
    pub const PROGRAM_TEX_INSTRUCTIONS_ARB: c_uint = 0x8806;
    pub const PROGRAM_UNDER_NATIVE_LIMITS_ARB: c_uint = 0x88B6;
    pub const PROTECTED_MEMORY_OBJECT_EXT: c_uint = 0x959B;
    pub const PROVOKING_VERTEX: c_uint = 0x8E4F;
    pub const PROVOKING_VERTEX_EXT: c_uint = 0x8E4F;
    pub const PROXY_COLOR_TABLE_SGI: c_uint = 0x80D3;
    pub const PROXY_HISTOGRAM_EXT: c_uint = 0x8025;
    pub const PROXY_POST_COLOR_MATRIX_COLOR_TABLE_SGI: c_uint = 0x80D5;
    pub const PROXY_POST_CONVOLUTION_COLOR_TABLE_SGI: c_uint = 0x80D4;
    pub const PROXY_POST_IMAGE_TRANSFORM_COLOR_TABLE_HP: c_uint = 0x8163;
    pub const PROXY_TEXTURE_1D: c_uint = 0x8063;
    pub const PROXY_TEXTURE_1D_ARRAY: c_uint = 0x8C19;
    pub const PROXY_TEXTURE_1D_ARRAY_EXT: c_uint = 0x8C19;
    pub const PROXY_TEXTURE_1D_EXT: c_uint = 0x8063;
    pub const PROXY_TEXTURE_1D_STACK_MESAX: c_uint = 0x875B;
    pub const PROXY_TEXTURE_2D: c_uint = 0x8064;
    pub const PROXY_TEXTURE_2D_ARRAY: c_uint = 0x8C1B;
    pub const PROXY_TEXTURE_2D_ARRAY_EXT: c_uint = 0x8C1B;
    pub const PROXY_TEXTURE_2D_EXT: c_uint = 0x8064;
    pub const PROXY_TEXTURE_2D_MULTISAMPLE: c_uint = 0x9101;
    pub const PROXY_TEXTURE_2D_MULTISAMPLE_ARRAY: c_uint = 0x9103;
    pub const PROXY_TEXTURE_2D_STACK_MESAX: c_uint = 0x875C;
    pub const PROXY_TEXTURE_3D: c_uint = 0x8070;
    pub const PROXY_TEXTURE_3D_EXT: c_uint = 0x8070;
    pub const PROXY_TEXTURE_4D_SGIS: c_uint = 0x8135;
    pub const PROXY_TEXTURE_COLOR_TABLE_SGI: c_uint = 0x80BD;
    pub const PROXY_TEXTURE_CUBE_MAP: c_uint = 0x851B;
    pub const PROXY_TEXTURE_CUBE_MAP_ARB: c_uint = 0x851B;
    pub const PROXY_TEXTURE_CUBE_MAP_ARRAY: c_uint = 0x900B;
    pub const PROXY_TEXTURE_CUBE_MAP_ARRAY_ARB: c_uint = 0x900B;
    pub const PROXY_TEXTURE_CUBE_MAP_EXT: c_uint = 0x851B;
    pub const PROXY_TEXTURE_RECTANGLE: c_uint = 0x84F7;
    pub const PROXY_TEXTURE_RECTANGLE_ARB: c_uint = 0x84F7;
    pub const PROXY_TEXTURE_RECTANGLE_NV: c_uint = 0x84F7;
    pub const PURGEABLE_APPLE: c_uint = 0x8A1D;
    pub const PURGED_CONTEXT_RESET_NV: c_uint = 0x92BB;
    pub const QUADRATIC_CURVE_TO_NV: c_uint = 0x0A;
    pub const QUADS: c_uint = 0x0007;
    pub const QUADS_FOLLOW_PROVOKING_VERTEX_CONVENTION: c_uint = 0x8E4C;
    pub const QUADS_FOLLOW_PROVOKING_VERTEX_CONVENTION_EXT: c_uint = 0x8E4C;
    pub const QUAD_ALPHA4_SGIS: c_uint = 0x811E;
    pub const QUAD_ALPHA8_SGIS: c_uint = 0x811F;
    pub const QUAD_INTENSITY4_SGIS: c_uint = 0x8122;
    pub const QUAD_INTENSITY8_SGIS: c_uint = 0x8123;
    pub const QUAD_LUMINANCE4_SGIS: c_uint = 0x8120;
    pub const QUAD_LUMINANCE8_SGIS: c_uint = 0x8121;
    pub const QUAD_MESH_SUN: c_uint = 0x8614;
    pub const QUAD_TEXTURE_SELECT_SGIS: c_uint = 0x8125;
    pub const QUARTER_BIT_ATI: c_uint = 0x00000010;
    pub const QUERY: c_uint = 0x82E3;
    pub const QUERY_ALL_EVENT_BITS_AMD: c_uint = 0xFFFFFFFF;
    pub const QUERY_BUFFER: c_uint = 0x9192;
    pub const QUERY_BUFFER_AMD: c_uint = 0x9192;
    pub const QUERY_BUFFER_BARRIER_BIT: c_uint = 0x00008000;
    pub const QUERY_BUFFER_BINDING: c_uint = 0x9193;
    pub const QUERY_BUFFER_BINDING_AMD: c_uint = 0x9193;
    pub const QUERY_BY_REGION_NO_WAIT: c_uint = 0x8E16;
    pub const QUERY_BY_REGION_NO_WAIT_INVERTED: c_uint = 0x8E1A;
    pub const QUERY_BY_REGION_NO_WAIT_NV: c_uint = 0x8E16;
    pub const QUERY_BY_REGION_WAIT: c_uint = 0x8E15;
    pub const QUERY_BY_REGION_WAIT_INVERTED: c_uint = 0x8E19;
    pub const QUERY_BY_REGION_WAIT_NV: c_uint = 0x8E15;
    pub const QUERY_COUNTER_BITS: c_uint = 0x8864;
    pub const QUERY_COUNTER_BITS_ARB: c_uint = 0x8864;
    pub const QUERY_DEPTH_BOUNDS_FAIL_EVENT_BIT_AMD: c_uint = 0x00000008;
    pub const QUERY_DEPTH_FAIL_EVENT_BIT_AMD: c_uint = 0x00000002;
    pub const QUERY_DEPTH_PASS_EVENT_BIT_AMD: c_uint = 0x00000001;
    pub const QUERY_NO_WAIT: c_uint = 0x8E14;
    pub const QUERY_NO_WAIT_INVERTED: c_uint = 0x8E18;
    pub const QUERY_NO_WAIT_NV: c_uint = 0x8E14;
    pub const QUERY_OBJECT_AMD: c_uint = 0x9153;
    pub const QUERY_OBJECT_EXT: c_uint = 0x9153;
    pub const QUERY_RESOURCE_BUFFEROBJECT_NV: c_uint = 0x9547;
    pub const QUERY_RESOURCE_MEMTYPE_VIDMEM_NV: c_uint = 0x9542;
    pub const QUERY_RESOURCE_RENDERBUFFER_NV: c_uint = 0x9546;
    pub const QUERY_RESOURCE_SYS_RESERVED_NV: c_uint = 0x9544;
    pub const QUERY_RESOURCE_TEXTURE_NV: c_uint = 0x9545;
    pub const QUERY_RESOURCE_TYPE_VIDMEM_ALLOC_NV: c_uint = 0x9540;
    pub const QUERY_RESULT: c_uint = 0x8866;
    pub const QUERY_RESULT_ARB: c_uint = 0x8866;
    pub const QUERY_RESULT_AVAILABLE: c_uint = 0x8867;
    pub const QUERY_RESULT_AVAILABLE_ARB: c_uint = 0x8867;
    pub const QUERY_RESULT_NO_WAIT: c_uint = 0x9194;
    pub const QUERY_RESULT_NO_WAIT_AMD: c_uint = 0x9194;
    pub const QUERY_STENCIL_FAIL_EVENT_BIT_AMD: c_uint = 0x00000004;
    pub const QUERY_TARGET: c_uint = 0x82EA;
    pub const QUERY_WAIT: c_uint = 0x8E13;
    pub const QUERY_WAIT_INVERTED: c_uint = 0x8E17;
    pub const QUERY_WAIT_NV: c_uint = 0x8E13;
    pub const R11F_G11F_B10F: c_uint = 0x8C3A;
    pub const R11F_G11F_B10F_EXT: c_uint = 0x8C3A;
    pub const R16: c_uint = 0x822A;
    pub const R16F: c_uint = 0x822D;
    pub const R16F_EXT: c_uint = 0x822D;
    pub const R16I: c_uint = 0x8233;
    pub const R16UI: c_uint = 0x8234;
    pub const R16_SNORM: c_uint = 0x8F98;
    pub const R1UI_C3F_V3F_SUN: c_uint = 0x85C6;
    pub const R1UI_C4F_N3F_V3F_SUN: c_uint = 0x85C8;
    pub const R1UI_C4UB_V3F_SUN: c_uint = 0x85C5;
    pub const R1UI_N3F_V3F_SUN: c_uint = 0x85C7;
    pub const R1UI_T2F_C4F_N3F_V3F_SUN: c_uint = 0x85CB;
    pub const R1UI_T2F_N3F_V3F_SUN: c_uint = 0x85CA;
    pub const R1UI_T2F_V3F_SUN: c_uint = 0x85C9;
    pub const R1UI_V3F_SUN: c_uint = 0x85C4;
    pub const R32F: c_uint = 0x822E;
    pub const R32F_EXT: c_uint = 0x822E;
    pub const R32I: c_uint = 0x8235;
    pub const R32UI: c_uint = 0x8236;
    pub const R3_G3_B2: c_uint = 0x2A10;
    pub const R8: c_uint = 0x8229;
    pub const R8I: c_uint = 0x8231;
    pub const R8UI: c_uint = 0x8232;
    pub const R8_EXT: c_uint = 0x8229;
    pub const R8_SNORM: c_uint = 0x8F94;
    pub const RASTERIZER_DISCARD: c_uint = 0x8C89;
    pub const RASTERIZER_DISCARD_EXT: c_uint = 0x8C89;
    pub const RASTERIZER_DISCARD_NV: c_uint = 0x8C89;
    pub const RASTER_FIXED_SAMPLE_LOCATIONS_EXT: c_uint = 0x932A;
    pub const RASTER_MULTISAMPLE_EXT: c_uint = 0x9327;
    pub const RASTER_POSITION_UNCLIPPED_IBM: c_uint = 0x19262;
    pub const RASTER_SAMPLES_EXT: c_uint = 0x9328;
    pub const READ_BUFFER: c_uint = 0x0C02;
    pub const READ_FRAMEBUFFER: c_uint = 0x8CA8;
    pub const READ_FRAMEBUFFER_BINDING: c_uint = 0x8CAA;
    pub const READ_FRAMEBUFFER_BINDING_EXT: c_uint = 0x8CAA;
    pub const READ_FRAMEBUFFER_EXT: c_uint = 0x8CA8;
    pub const READ_ONLY: c_uint = 0x88B8;
    pub const READ_ONLY_ARB: c_uint = 0x88B8;
    pub const READ_PIXELS: c_uint = 0x828C;
    pub const READ_PIXELS_FORMAT: c_uint = 0x828D;
    pub const READ_PIXELS_TYPE: c_uint = 0x828E;
    pub const READ_PIXEL_DATA_RANGE_LENGTH_NV: c_uint = 0x887B;
    pub const READ_PIXEL_DATA_RANGE_NV: c_uint = 0x8879;
    pub const READ_PIXEL_DATA_RANGE_POINTER_NV: c_uint = 0x887D;
    pub const READ_WRITE: c_uint = 0x88BA;
    pub const READ_WRITE_ARB: c_uint = 0x88BA;
    pub const RECLAIM_MEMORY_HINT_PGI: c_uint = 0x1A1FE;
    pub const RECT_NV: c_uint = 0xF6;
    pub const RED: c_uint = 0x1903;
    pub const REDUCE_EXT: c_uint = 0x8016;
    pub const RED_BIT_ATI: c_uint = 0x00000001;
    pub const RED_INTEGER: c_uint = 0x8D94;
    pub const RED_INTEGER_EXT: c_uint = 0x8D94;
    pub const RED_MAX_CLAMP_INGR: c_uint = 0x8564;
    pub const RED_MIN_CLAMP_INGR: c_uint = 0x8560;
    pub const RED_NV: c_uint = 0x1903;
    pub const RED_SNORM: c_uint = 0x8F90;
    pub const REFERENCED_BY_COMPUTE_SHADER: c_uint = 0x930B;
    pub const REFERENCED_BY_FRAGMENT_SHADER: c_uint = 0x930A;
    pub const REFERENCED_BY_GEOMETRY_SHADER: c_uint = 0x9309;
    pub const REFERENCED_BY_MESH_SHADER_NV: c_uint = 0x95A0;
    pub const REFERENCED_BY_TASK_SHADER_NV: c_uint = 0x95A1;
    pub const REFERENCED_BY_TESS_CONTROL_SHADER: c_uint = 0x9307;
    pub const REFERENCED_BY_TESS_EVALUATION_SHADER: c_uint = 0x9308;
    pub const REFERENCED_BY_VERTEX_SHADER: c_uint = 0x9306;
    pub const REFERENCE_PLANE_EQUATION_SGIX: c_uint = 0x817E;
    pub const REFERENCE_PLANE_SGIX: c_uint = 0x817D;
    pub const REFLECTION_MAP_ARB: c_uint = 0x8512;
    pub const REFLECTION_MAP_EXT: c_uint = 0x8512;
    pub const REFLECTION_MAP_NV: c_uint = 0x8512;
    pub const REGISTER_COMBINERS_NV: c_uint = 0x8522;
    pub const REG_0_ATI: c_uint = 0x8921;
    pub const REG_10_ATI: c_uint = 0x892B;
    pub const REG_11_ATI: c_uint = 0x892C;
    pub const REG_12_ATI: c_uint = 0x892D;
    pub const REG_13_ATI: c_uint = 0x892E;
    pub const REG_14_ATI: c_uint = 0x892F;
    pub const REG_15_ATI: c_uint = 0x8930;
    pub const REG_16_ATI: c_uint = 0x8931;
    pub const REG_17_ATI: c_uint = 0x8932;
    pub const REG_18_ATI: c_uint = 0x8933;
    pub const REG_19_ATI: c_uint = 0x8934;
    pub const REG_1_ATI: c_uint = 0x8922;
    pub const REG_20_ATI: c_uint = 0x8935;
    pub const REG_21_ATI: c_uint = 0x8936;
    pub const REG_22_ATI: c_uint = 0x8937;
    pub const REG_23_ATI: c_uint = 0x8938;
    pub const REG_24_ATI: c_uint = 0x8939;
    pub const REG_25_ATI: c_uint = 0x893A;
    pub const REG_26_ATI: c_uint = 0x893B;
    pub const REG_27_ATI: c_uint = 0x893C;
    pub const REG_28_ATI: c_uint = 0x893D;
    pub const REG_29_ATI: c_uint = 0x893E;
    pub const REG_2_ATI: c_uint = 0x8923;
    pub const REG_30_ATI: c_uint = 0x893F;
    pub const REG_31_ATI: c_uint = 0x8940;
    pub const REG_3_ATI: c_uint = 0x8924;
    pub const REG_4_ATI: c_uint = 0x8925;
    pub const REG_5_ATI: c_uint = 0x8926;
    pub const REG_6_ATI: c_uint = 0x8927;
    pub const REG_7_ATI: c_uint = 0x8928;
    pub const REG_8_ATI: c_uint = 0x8929;
    pub const REG_9_ATI: c_uint = 0x892A;
    pub const RELATIVE_ARC_TO_NV: c_uint = 0xFF;
    pub const RELATIVE_CONIC_CURVE_TO_NV: c_uint = 0x1B;
    pub const RELATIVE_CUBIC_CURVE_TO_NV: c_uint = 0x0D;
    pub const RELATIVE_HORIZONTAL_LINE_TO_NV: c_uint = 0x07;
    pub const RELATIVE_LARGE_CCW_ARC_TO_NV: c_uint = 0x17;
    pub const RELATIVE_LARGE_CW_ARC_TO_NV: c_uint = 0x19;
    pub const RELATIVE_LINE_TO_NV: c_uint = 0x05;
    pub const RELATIVE_MOVE_TO_NV: c_uint = 0x03;
    pub const RELATIVE_QUADRATIC_CURVE_TO_NV: c_uint = 0x0B;
    pub const RELATIVE_RECT_NV: c_uint = 0xF7;
    pub const RELATIVE_ROUNDED_RECT2_NV: c_uint = 0xEB;
    pub const RELATIVE_ROUNDED_RECT4_NV: c_uint = 0xED;
    pub const RELATIVE_ROUNDED_RECT8_NV: c_uint = 0xEF;
    pub const RELATIVE_ROUNDED_RECT_NV: c_uint = 0xE9;
    pub const RELATIVE_SMALL_CCW_ARC_TO_NV: c_uint = 0x13;
    pub const RELATIVE_SMALL_CW_ARC_TO_NV: c_uint = 0x15;
    pub const RELATIVE_SMOOTH_CUBIC_CURVE_TO_NV: c_uint = 0x11;
    pub const RELATIVE_SMOOTH_QUADRATIC_CURVE_TO_NV: c_uint = 0x0F;
    pub const RELATIVE_VERTICAL_LINE_TO_NV: c_uint = 0x09;
    pub const RELEASED_APPLE: c_uint = 0x8A19;
    pub const RENDERBUFFER: c_uint = 0x8D41;
    pub const RENDERBUFFER_ALPHA_SIZE: c_uint = 0x8D53;
    pub const RENDERBUFFER_ALPHA_SIZE_EXT: c_uint = 0x8D53;
    pub const RENDERBUFFER_BINDING: c_uint = 0x8CA7;
    pub const RENDERBUFFER_BINDING_EXT: c_uint = 0x8CA7;
    pub const RENDERBUFFER_BLUE_SIZE: c_uint = 0x8D52;
    pub const RENDERBUFFER_BLUE_SIZE_EXT: c_uint = 0x8D52;
    pub const RENDERBUFFER_COLOR_SAMPLES_NV: c_uint = 0x8E10;
    pub const RENDERBUFFER_COVERAGE_SAMPLES_NV: c_uint = 0x8CAB;
    pub const RENDERBUFFER_DEPTH_SIZE: c_uint = 0x8D54;
    pub const RENDERBUFFER_DEPTH_SIZE_EXT: c_uint = 0x8D54;
    pub const RENDERBUFFER_EXT: c_uint = 0x8D41;
    pub const RENDERBUFFER_FREE_MEMORY_ATI: c_uint = 0x87FD;
    pub const RENDERBUFFER_GREEN_SIZE: c_uint = 0x8D51;
    pub const RENDERBUFFER_GREEN_SIZE_EXT: c_uint = 0x8D51;
    pub const RENDERBUFFER_HEIGHT: c_uint = 0x8D43;
    pub const RENDERBUFFER_HEIGHT_EXT: c_uint = 0x8D43;
    pub const RENDERBUFFER_INTERNAL_FORMAT: c_uint = 0x8D44;
    pub const RENDERBUFFER_INTERNAL_FORMAT_EXT: c_uint = 0x8D44;
    pub const RENDERBUFFER_RED_SIZE: c_uint = 0x8D50;
    pub const RENDERBUFFER_RED_SIZE_EXT: c_uint = 0x8D50;
    pub const RENDERBUFFER_SAMPLES: c_uint = 0x8CAB;
    pub const RENDERBUFFER_SAMPLES_EXT: c_uint = 0x8CAB;
    pub const RENDERBUFFER_STENCIL_SIZE: c_uint = 0x8D55;
    pub const RENDERBUFFER_STENCIL_SIZE_EXT: c_uint = 0x8D55;
    pub const RENDERBUFFER_STORAGE_SAMPLES_AMD: c_uint = 0x91B2;
    pub const RENDERBUFFER_WIDTH: c_uint = 0x8D42;
    pub const RENDERBUFFER_WIDTH_EXT: c_uint = 0x8D42;
    pub const RENDERER: c_uint = 0x1F01;
    pub const RENDER_GPU_MASK_NV: c_uint = 0x9558;
    pub const REPEAT: c_uint = 0x2901;
    pub const REPLACE: c_uint = 0x1E01;
    pub const REPLACEMENT_CODE_ARRAY_POINTER_SUN: c_uint = 0x85C3;
    pub const REPLACEMENT_CODE_ARRAY_STRIDE_SUN: c_uint = 0x85C2;
    pub const REPLACEMENT_CODE_ARRAY_SUN: c_uint = 0x85C0;
    pub const REPLACEMENT_CODE_ARRAY_TYPE_SUN: c_uint = 0x85C1;
    pub const REPLACEMENT_CODE_SUN: c_uint = 0x81D8;
    pub const REPLACE_EXT: c_uint = 0x8062;
    pub const REPLACE_MIDDLE_SUN: c_uint = 0x0002;
    pub const REPLACE_OLDEST_SUN: c_uint = 0x0003;
    pub const REPLACE_VALUE_AMD: c_uint = 0x874B;
    pub const REPLICATE_BORDER_HP: c_uint = 0x8153;
    pub const REPRESENTATIVE_FRAGMENT_TEST_NV: c_uint = 0x937F;
    pub const RESAMPLE_AVERAGE_OML: c_uint = 0x8988;
    pub const RESAMPLE_DECIMATE_OML: c_uint = 0x8989;
    pub const RESAMPLE_DECIMATE_SGIX: c_uint = 0x8430;
    pub const RESAMPLE_REPLICATE_OML: c_uint = 0x8986;
    pub const RESAMPLE_REPLICATE_SGIX: c_uint = 0x8433;
    pub const RESAMPLE_ZERO_FILL_OML: c_uint = 0x8987;
    pub const RESAMPLE_ZERO_FILL_SGIX: c_uint = 0x8434;
    pub const RESCALE_NORMAL_EXT: c_uint = 0x803A;
    pub const RESET_NOTIFICATION_STRATEGY: c_uint = 0x8256;
    pub const RESET_NOTIFICATION_STRATEGY_ARB: c_uint = 0x8256;
    pub const RESTART_PATH_NV: c_uint = 0xF0;
    pub const RESTART_SUN: c_uint = 0x0001;
    pub const RETAINED_APPLE: c_uint = 0x8A1B;
    pub const RG: c_uint = 0x8227;
    pub const RG16: c_uint = 0x822C;
    pub const RG16F: c_uint = 0x822F;
    pub const RG16F_EXT: c_uint = 0x822F;
    pub const RG16I: c_uint = 0x8239;
    pub const RG16UI: c_uint = 0x823A;
    pub const RG16_SNORM: c_uint = 0x8F99;
    pub const RG32F: c_uint = 0x8230;
    pub const RG32F_EXT: c_uint = 0x8230;
    pub const RG32I: c_uint = 0x823B;
    pub const RG32UI: c_uint = 0x823C;
    pub const RG8: c_uint = 0x822B;
    pub const RG8I: c_uint = 0x8237;
    pub const RG8UI: c_uint = 0x8238;
    pub const RG8_EXT: c_uint = 0x822B;
    pub const RG8_SNORM: c_uint = 0x8F95;
    pub const RGB: c_uint = 0x1907;
    pub const RGB10: c_uint = 0x8052;
    pub const RGB10_A2: c_uint = 0x8059;
    pub const RGB10_A2UI: c_uint = 0x906F;
    pub const RGB10_A2_EXT: c_uint = 0x8059;
    pub const RGB10_EXT: c_uint = 0x8052;
    pub const RGB12: c_uint = 0x8053;
    pub const RGB12_EXT: c_uint = 0x8053;
    pub const RGB16: c_uint = 0x8054;
    pub const RGB16F: c_uint = 0x881B;
    pub const RGB16F_ARB: c_uint = 0x881B;
    pub const RGB16F_EXT: c_uint = 0x881B;
    pub const RGB16I: c_uint = 0x8D89;
    pub const RGB16I_EXT: c_uint = 0x8D89;
    pub const RGB16UI: c_uint = 0x8D77;
    pub const RGB16UI_EXT: c_uint = 0x8D77;
    pub const RGB16_EXT: c_uint = 0x8054;
    pub const RGB16_SNORM: c_uint = 0x8F9A;
    pub const RGB2_EXT: c_uint = 0x804E;
    pub const RGB32F: c_uint = 0x8815;
    pub const RGB32F_ARB: c_uint = 0x8815;
    pub const RGB32F_EXT: c_uint = 0x8815;
    pub const RGB32I: c_uint = 0x8D83;
    pub const RGB32I_EXT: c_uint = 0x8D83;
    pub const RGB32UI: c_uint = 0x8D71;
    pub const RGB32UI_EXT: c_uint = 0x8D71;
    pub const RGB4: c_uint = 0x804F;
    pub const RGB4_EXT: c_uint = 0x804F;
    pub const RGB4_S3TC: c_uint = 0x83A1;
    pub const RGB5: c_uint = 0x8050;
    pub const RGB565: c_uint = 0x8D62;
    pub const RGB5_A1: c_uint = 0x8057;
    pub const RGB5_A1_EXT: c_uint = 0x8057;
    pub const RGB5_EXT: c_uint = 0x8050;
    pub const RGB8: c_uint = 0x8051;
    pub const RGB8I: c_uint = 0x8D8F;
    pub const RGB8I_EXT: c_uint = 0x8D8F;
    pub const RGB8UI: c_uint = 0x8D7D;
    pub const RGB8UI_EXT: c_uint = 0x8D7D;
    pub const RGB8_EXT: c_uint = 0x8051;
    pub const RGB8_SNORM: c_uint = 0x8F96;
    pub const RGB9_E5: c_uint = 0x8C3D;
    pub const RGB9_E5_EXT: c_uint = 0x8C3D;
    pub const RGBA: c_uint = 0x1908;
    pub const RGBA12: c_uint = 0x805A;
    pub const RGBA12_EXT: c_uint = 0x805A;
    pub const RGBA16: c_uint = 0x805B;
    pub const RGBA16F: c_uint = 0x881A;
    pub const RGBA16F_ARB: c_uint = 0x881A;
    pub const RGBA16F_EXT: c_uint = 0x881A;
    pub const RGBA16I: c_uint = 0x8D88;
    pub const RGBA16I_EXT: c_uint = 0x8D88;
    pub const RGBA16UI: c_uint = 0x8D76;
    pub const RGBA16UI_EXT: c_uint = 0x8D76;
    pub const RGBA16_EXT: c_uint = 0x805B;
    pub const RGBA16_SNORM: c_uint = 0x8F9B;
    pub const RGBA2: c_uint = 0x8055;
    pub const RGBA2_EXT: c_uint = 0x8055;
    pub const RGBA32F: c_uint = 0x8814;
    pub const RGBA32F_ARB: c_uint = 0x8814;
    pub const RGBA32F_EXT: c_uint = 0x8814;
    pub const RGBA32I: c_uint = 0x8D82;
    pub const RGBA32I_EXT: c_uint = 0x8D82;
    pub const RGBA32UI: c_uint = 0x8D70;
    pub const RGBA32UI_EXT: c_uint = 0x8D70;
    pub const RGBA4: c_uint = 0x8056;
    pub const RGBA4_DXT5_S3TC: c_uint = 0x83A5;
    pub const RGBA4_EXT: c_uint = 0x8056;
    pub const RGBA4_S3TC: c_uint = 0x83A3;
    pub const RGBA8: c_uint = 0x8058;
    pub const RGBA8I: c_uint = 0x8D8E;
    pub const RGBA8I_EXT: c_uint = 0x8D8E;
    pub const RGBA8UI: c_uint = 0x8D7C;
    pub const RGBA8UI_EXT: c_uint = 0x8D7C;
    pub const RGBA8_EXT: c_uint = 0x8058;
    pub const RGBA8_SNORM: c_uint = 0x8F97;
    pub const RGBA_DXT5_S3TC: c_uint = 0x83A4;
    pub const RGBA_FLOAT16_APPLE: c_uint = 0x881A;
    pub const RGBA_FLOAT16_ATI: c_uint = 0x881A;
    pub const RGBA_FLOAT32_APPLE: c_uint = 0x8814;
    pub const RGBA_FLOAT32_ATI: c_uint = 0x8814;
    pub const RGBA_FLOAT_MODE_ARB: c_uint = 0x8820;
    pub const RGBA_FLOAT_MODE_ATI: c_uint = 0x8820;
    pub const RGBA_INTEGER: c_uint = 0x8D99;
    pub const RGBA_INTEGER_EXT: c_uint = 0x8D99;
    pub const RGBA_INTEGER_MODE_EXT: c_uint = 0x8D9E;
    pub const RGBA_S3TC: c_uint = 0x83A2;
    pub const RGBA_SIGNED_COMPONENTS_EXT: c_uint = 0x8C3C;
    pub const RGBA_SNORM: c_uint = 0x8F93;
    pub const RGBA_UNSIGNED_DOT_PRODUCT_MAPPING_NV: c_uint = 0x86D9;
    pub const RGB_422_APPLE: c_uint = 0x8A1F;
    pub const RGB_FLOAT16_APPLE: c_uint = 0x881B;
    pub const RGB_FLOAT16_ATI: c_uint = 0x881B;
    pub const RGB_FLOAT32_APPLE: c_uint = 0x8815;
    pub const RGB_FLOAT32_ATI: c_uint = 0x8815;
    pub const RGB_INTEGER: c_uint = 0x8D98;
    pub const RGB_INTEGER_EXT: c_uint = 0x8D98;
    pub const RGB_RAW_422_APPLE: c_uint = 0x8A51;
    pub const RGB_S3TC: c_uint = 0x83A0;
    pub const RGB_SCALE_ARB: c_uint = 0x8573;
    pub const RGB_SCALE_EXT: c_uint = 0x8573;
    pub const RGB_SNORM: c_uint = 0x8F92;
    pub const RG_INTEGER: c_uint = 0x8228;
    pub const RG_SNORM: c_uint = 0x8F91;
    pub const RIGHT: c_uint = 0x0407;
    pub const ROUNDED_RECT2_NV: c_uint = 0xEA;
    pub const ROUNDED_RECT4_NV: c_uint = 0xEC;
    pub const ROUNDED_RECT8_NV: c_uint = 0xEE;
    pub const ROUNDED_RECT_NV: c_uint = 0xE8;
    pub const ROUND_NV: c_uint = 0x90A4;
    pub const SAMPLER: c_uint = 0x82E6;
    pub const SAMPLER_1D: c_uint = 0x8B5D;
    pub const SAMPLER_1D_ARB: c_uint = 0x8B5D;
    pub const SAMPLER_1D_ARRAY: c_uint = 0x8DC0;
    pub const SAMPLER_1D_ARRAY_EXT: c_uint = 0x8DC0;
    pub const SAMPLER_1D_ARRAY_SHADOW: c_uint = 0x8DC3;
    pub const SAMPLER_1D_ARRAY_SHADOW_EXT: c_uint = 0x8DC3;
    pub const SAMPLER_1D_SHADOW: c_uint = 0x8B61;
    pub const SAMPLER_1D_SHADOW_ARB: c_uint = 0x8B61;
    pub const SAMPLER_2D: c_uint = 0x8B5E;
    pub const SAMPLER_2D_ARB: c_uint = 0x8B5E;
    pub const SAMPLER_2D_ARRAY: c_uint = 0x8DC1;
    pub const SAMPLER_2D_ARRAY_EXT: c_uint = 0x8DC1;
    pub const SAMPLER_2D_ARRAY_SHADOW: c_uint = 0x8DC4;
    pub const SAMPLER_2D_ARRAY_SHADOW_EXT: c_uint = 0x8DC4;
    pub const SAMPLER_2D_MULTISAMPLE: c_uint = 0x9108;
    pub const SAMPLER_2D_MULTISAMPLE_ARRAY: c_uint = 0x910B;
    pub const SAMPLER_2D_RECT: c_uint = 0x8B63;
    pub const SAMPLER_2D_RECT_ARB: c_uint = 0x8B63;
    pub const SAMPLER_2D_RECT_SHADOW: c_uint = 0x8B64;
    pub const SAMPLER_2D_RECT_SHADOW_ARB: c_uint = 0x8B64;
    pub const SAMPLER_2D_SHADOW: c_uint = 0x8B62;
    pub const SAMPLER_2D_SHADOW_ARB: c_uint = 0x8B62;
    pub const SAMPLER_3D: c_uint = 0x8B5F;
    pub const SAMPLER_3D_ARB: c_uint = 0x8B5F;
    pub const SAMPLER_BINDING: c_uint = 0x8919;
    pub const SAMPLER_BUFFER: c_uint = 0x8DC2;
    pub const SAMPLER_BUFFER_AMD: c_uint = 0x9001;
    pub const SAMPLER_BUFFER_EXT: c_uint = 0x8DC2;
    pub const SAMPLER_CUBE: c_uint = 0x8B60;
    pub const SAMPLER_CUBE_ARB: c_uint = 0x8B60;
    pub const SAMPLER_CUBE_MAP_ARRAY: c_uint = 0x900C;
    pub const SAMPLER_CUBE_MAP_ARRAY_ARB: c_uint = 0x900C;
    pub const SAMPLER_CUBE_MAP_ARRAY_SHADOW: c_uint = 0x900D;
    pub const SAMPLER_CUBE_MAP_ARRAY_SHADOW_ARB: c_uint = 0x900D;
    pub const SAMPLER_CUBE_SHADOW: c_uint = 0x8DC5;
    pub const SAMPLER_CUBE_SHADOW_EXT: c_uint = 0x8DC5;
    pub const SAMPLER_OBJECT_AMD: c_uint = 0x9155;
    pub const SAMPLER_RENDERBUFFER_NV: c_uint = 0x8E56;
    pub const SAMPLES: c_uint = 0x80A9;
    pub const SAMPLES_3DFX: c_uint = 0x86B4;
    pub const SAMPLES_ARB: c_uint = 0x80A9;
    pub const SAMPLES_EXT: c_uint = 0x80A9;
    pub const SAMPLES_PASSED: c_uint = 0x8914;
    pub const SAMPLES_PASSED_ARB: c_uint = 0x8914;
    pub const SAMPLES_SGIS: c_uint = 0x80A9;
    pub const SAMPLE_ALPHA_TO_COVERAGE: c_uint = 0x809E;
    pub const SAMPLE_ALPHA_TO_COVERAGE_ARB: c_uint = 0x809E;
    pub const SAMPLE_ALPHA_TO_MASK_EXT: c_uint = 0x809E;
    pub const SAMPLE_ALPHA_TO_MASK_SGIS: c_uint = 0x809E;
    pub const SAMPLE_ALPHA_TO_ONE: c_uint = 0x809F;
    pub const SAMPLE_ALPHA_TO_ONE_ARB: c_uint = 0x809F;
    pub const SAMPLE_ALPHA_TO_ONE_EXT: c_uint = 0x809F;
    pub const SAMPLE_ALPHA_TO_ONE_SGIS: c_uint = 0x809F;
    pub const SAMPLE_BUFFERS: c_uint = 0x80A8;
    pub const SAMPLE_BUFFERS_3DFX: c_uint = 0x86B3;
    pub const SAMPLE_BUFFERS_ARB: c_uint = 0x80A8;
    pub const SAMPLE_BUFFERS_EXT: c_uint = 0x80A8;
    pub const SAMPLE_BUFFERS_SGIS: c_uint = 0x80A8;
    pub const SAMPLE_COVERAGE: c_uint = 0x80A0;
    pub const SAMPLE_COVERAGE_ARB: c_uint = 0x80A0;
    pub const SAMPLE_COVERAGE_INVERT: c_uint = 0x80AB;
    pub const SAMPLE_COVERAGE_INVERT_ARB: c_uint = 0x80AB;
    pub const SAMPLE_COVERAGE_VALUE: c_uint = 0x80AA;
    pub const SAMPLE_COVERAGE_VALUE_ARB: c_uint = 0x80AA;
    pub const SAMPLE_LOCATION_ARB: c_uint = 0x8E50;
    pub const SAMPLE_LOCATION_NV: c_uint = 0x8E50;
    pub const SAMPLE_LOCATION_PIXEL_GRID_HEIGHT_ARB: c_uint = 0x933F;
    pub const SAMPLE_LOCATION_PIXEL_GRID_HEIGHT_NV: c_uint = 0x933F;
    pub const SAMPLE_LOCATION_PIXEL_GRID_WIDTH_ARB: c_uint = 0x933E;
    pub const SAMPLE_LOCATION_PIXEL_GRID_WIDTH_NV: c_uint = 0x933E;
    pub const SAMPLE_LOCATION_SUBPIXEL_BITS_ARB: c_uint = 0x933D;
    pub const SAMPLE_LOCATION_SUBPIXEL_BITS_NV: c_uint = 0x933D;
    pub const SAMPLE_MASK: c_uint = 0x8E51;
    pub const SAMPLE_MASK_EXT: c_uint = 0x80A0;
    pub const SAMPLE_MASK_INVERT_EXT: c_uint = 0x80AB;
    pub const SAMPLE_MASK_INVERT_SGIS: c_uint = 0x80AB;
    pub const SAMPLE_MASK_NV: c_uint = 0x8E51;
    pub const SAMPLE_MASK_SGIS: c_uint = 0x80A0;
    pub const SAMPLE_MASK_VALUE: c_uint = 0x8E52;
    pub const SAMPLE_MASK_VALUE_EXT: c_uint = 0x80AA;
    pub const SAMPLE_MASK_VALUE_NV: c_uint = 0x8E52;
    pub const SAMPLE_MASK_VALUE_SGIS: c_uint = 0x80AA;
    pub const SAMPLE_PATTERN_EXT: c_uint = 0x80AC;
    pub const SAMPLE_PATTERN_SGIS: c_uint = 0x80AC;
    pub const SAMPLE_POSITION: c_uint = 0x8E50;
    pub const SAMPLE_POSITION_NV: c_uint = 0x8E50;
    pub const SAMPLE_SHADING: c_uint = 0x8C36;
    pub const SAMPLE_SHADING_ARB: c_uint = 0x8C36;
    pub const SATURATE_BIT_ATI: c_uint = 0x00000040;
    pub const SCALAR_EXT: c_uint = 0x87BE;
    pub const SCALEBIAS_HINT_SGIX: c_uint = 0x8322;
    pub const SCALED_RESOLVE_FASTEST_EXT: c_uint = 0x90BA;
    pub const SCALED_RESOLVE_NICEST_EXT: c_uint = 0x90BB;
    pub const SCALE_BY_FOUR_NV: c_uint = 0x853F;
    pub const SCALE_BY_ONE_HALF_NV: c_uint = 0x8540;
    pub const SCALE_BY_TWO_NV: c_uint = 0x853E;
    pub const SCISSOR_BOX: c_uint = 0x0C10;
    pub const SCISSOR_BOX_EXCLUSIVE_NV: c_uint = 0x9556;
    pub const SCISSOR_COMMAND_NV: c_uint = 0x0011;
    pub const SCISSOR_TEST: c_uint = 0x0C11;
    pub const SCISSOR_TEST_EXCLUSIVE_NV: c_uint = 0x9555;
    pub const SCREEN_COORDINATES_REND: c_uint = 0x8490;
    pub const SCREEN_KHR: c_uint = 0x9295;
    pub const SCREEN_NV: c_uint = 0x9295;
    pub const SECONDARY_COLOR_ARRAY_ADDRESS_NV: c_uint = 0x8F27;
    pub const SECONDARY_COLOR_ARRAY_BUFFER_BINDING_ARB: c_uint = 0x889C;
    pub const SECONDARY_COLOR_ARRAY_EXT: c_uint = 0x845E;
    pub const SECONDARY_COLOR_ARRAY_LENGTH_NV: c_uint = 0x8F31;
    pub const SECONDARY_COLOR_ARRAY_LIST_IBM: c_uint = 103077;
    pub const SECONDARY_COLOR_ARRAY_LIST_STRIDE_IBM: c_uint = 103087;
    pub const SECONDARY_COLOR_ARRAY_POINTER_EXT: c_uint = 0x845D;
    pub const SECONDARY_COLOR_ARRAY_SIZE_EXT: c_uint = 0x845A;
    pub const SECONDARY_COLOR_ARRAY_STRIDE_EXT: c_uint = 0x845C;
    pub const SECONDARY_COLOR_ARRAY_TYPE_EXT: c_uint = 0x845B;
    pub const SECONDARY_COLOR_NV: c_uint = 0x852D;
    pub const SECONDARY_INTERPOLATOR_ATI: c_uint = 0x896D;
    pub const SEMAPHORE_TYPE_BINARY_NV: c_uint = 0x95B4;
    pub const SEMAPHORE_TYPE_NV: c_uint = 0x95B3;
    pub const SEMAPHORE_TYPE_TIMELINE_NV: c_uint = 0x95B5;
    pub const SEPARABLE_2D_EXT: c_uint = 0x8012;
    pub const SEPARATE_ATTRIBS: c_uint = 0x8C8D;
    pub const SEPARATE_ATTRIBS_EXT: c_uint = 0x8C8D;
    pub const SEPARATE_ATTRIBS_NV: c_uint = 0x8C8D;
    pub const SEPARATE_SPECULAR_COLOR_EXT: c_uint = 0x81FA;
    pub const SET: c_uint = 0x150F;
    pub const SET_AMD: c_uint = 0x874A;
    pub const SHADER: c_uint = 0x82E1;
    pub const SHADER_BINARY_FORMATS: c_uint = 0x8DF8;
    pub const SHADER_BINARY_FORMAT_SPIR_V: c_uint = 0x9551;
    pub const SHADER_BINARY_FORMAT_SPIR_V_ARB: c_uint = 0x9551;
    pub const SHADER_COMPILER: c_uint = 0x8DFA;
    pub const SHADER_CONSISTENT_NV: c_uint = 0x86DD;
    pub const SHADER_GLOBAL_ACCESS_BARRIER_BIT_NV: c_uint = 0x00000010;
    pub const SHADER_IMAGE_ACCESS_BARRIER_BIT: c_uint = 0x00000020;
    pub const SHADER_IMAGE_ACCESS_BARRIER_BIT_EXT: c_uint = 0x00000020;
    pub const SHADER_IMAGE_ATOMIC: c_uint = 0x82A6;
    pub const SHADER_IMAGE_LOAD: c_uint = 0x82A4;
    pub const SHADER_IMAGE_STORE: c_uint = 0x82A5;
    pub const SHADER_INCLUDE_ARB: c_uint = 0x8DAE;
    pub const SHADER_OBJECT_ARB: c_uint = 0x8B48;
    pub const SHADER_OBJECT_EXT: c_uint = 0x8B48;
    pub const SHADER_OPERATION_NV: c_uint = 0x86DF;
    pub const SHADER_SOURCE_LENGTH: c_uint = 0x8B88;
    pub const SHADER_STORAGE_BARRIER_BIT: c_uint = 0x00002000;
    pub const SHADER_STORAGE_BLOCK: c_uint = 0x92E6;
    pub const SHADER_STORAGE_BUFFER: c_uint = 0x90D2;
    pub const SHADER_STORAGE_BUFFER_BINDING: c_uint = 0x90D3;
    pub const SHADER_STORAGE_BUFFER_OFFSET_ALIGNMENT: c_uint = 0x90DF;
    pub const SHADER_STORAGE_BUFFER_SIZE: c_uint = 0x90D5;
    pub const SHADER_STORAGE_BUFFER_START: c_uint = 0x90D4;
    pub const SHADER_TYPE: c_uint = 0x8B4F;
    pub const SHADING_LANGUAGE_VERSION: c_uint = 0x8B8C;
    pub const SHADING_LANGUAGE_VERSION_ARB: c_uint = 0x8B8C;
    pub const SHADING_RATE_16_INVOCATIONS_PER_PIXEL_NV: c_uint = 0x956F;
    pub const SHADING_RATE_1_INVOCATION_PER_1X2_PIXELS_NV: c_uint = 0x9566;
    pub const SHADING_RATE_1_INVOCATION_PER_2X1_PIXELS_NV: c_uint = 0x9567;
    pub const SHADING_RATE_1_INVOCATION_PER_2X2_PIXELS_NV: c_uint = 0x9568;
    pub const SHADING_RATE_1_INVOCATION_PER_2X4_PIXELS_NV: c_uint = 0x9569;
    pub const SHADING_RATE_1_INVOCATION_PER_4X2_PIXELS_NV: c_uint = 0x956A;
    pub const SHADING_RATE_1_INVOCATION_PER_4X4_PIXELS_NV: c_uint = 0x956B;
    pub const SHADING_RATE_1_INVOCATION_PER_PIXEL_NV: c_uint = 0x9565;
    pub const SHADING_RATE_2_INVOCATIONS_PER_PIXEL_NV: c_uint = 0x956C;
    pub const SHADING_RATE_4_INVOCATIONS_PER_PIXEL_NV: c_uint = 0x956D;
    pub const SHADING_RATE_8_INVOCATIONS_PER_PIXEL_NV: c_uint = 0x956E;
    pub const SHADING_RATE_IMAGE_BINDING_NV: c_uint = 0x955B;
    pub const SHADING_RATE_IMAGE_NV: c_uint = 0x9563;
    pub const SHADING_RATE_IMAGE_PALETTE_COUNT_NV: c_uint = 0x95B2;
    pub const SHADING_RATE_IMAGE_PALETTE_SIZE_NV: c_uint = 0x955E;
    pub const SHADING_RATE_IMAGE_PER_PRIMITIVE_NV: c_uint = 0x95B1;
    pub const SHADING_RATE_IMAGE_TEXEL_HEIGHT_NV: c_uint = 0x955D;
    pub const SHADING_RATE_IMAGE_TEXEL_WIDTH_NV: c_uint = 0x955C;
    pub const SHADING_RATE_NO_INVOCATIONS_NV: c_uint = 0x9564;
    pub const SHADING_RATE_SAMPLE_ORDER_DEFAULT_NV: c_uint = 0x95AE;
    pub const SHADING_RATE_SAMPLE_ORDER_PIXEL_MAJOR_NV: c_uint = 0x95AF;
    pub const SHADING_RATE_SAMPLE_ORDER_SAMPLE_MAJOR_NV: c_uint = 0x95B0;
    pub const SHADOW_AMBIENT_SGIX: c_uint = 0x80BF;
    pub const SHADOW_ATTENUATION_EXT: c_uint = 0x834E;
    pub const SHARED_EDGE_NV: c_uint = 0xC0;
    pub const SHARED_TEXTURE_PALETTE_EXT: c_uint = 0x81FB;
    pub const SHARPEN_TEXTURE_FUNC_POINTS_SGIS: c_uint = 0x80B0;
    pub const SHORT: c_uint = 0x1402;
    pub const SIGNALED: c_uint = 0x9119;
    pub const SIGNED_ALPHA8_NV: c_uint = 0x8706;
    pub const SIGNED_ALPHA_NV: c_uint = 0x8705;
    pub const SIGNED_HILO16_NV: c_uint = 0x86FA;
    pub const SIGNED_HILO8_NV: c_uint = 0x885F;
    pub const SIGNED_HILO_NV: c_uint = 0x86F9;
    pub const SIGNED_IDENTITY_NV: c_uint = 0x853C;
    pub const SIGNED_INTENSITY8_NV: c_uint = 0x8708;
    pub const SIGNED_INTENSITY_NV: c_uint = 0x8707;
    pub const SIGNED_LUMINANCE8_ALPHA8_NV: c_uint = 0x8704;
    pub const SIGNED_LUMINANCE8_NV: c_uint = 0x8702;
    pub const SIGNED_LUMINANCE_ALPHA_NV: c_uint = 0x8703;
    pub const SIGNED_LUMINANCE_NV: c_uint = 0x8701;
    pub const SIGNED_NEGATE_NV: c_uint = 0x853D;
    pub const SIGNED_NORMALIZED: c_uint = 0x8F9C;
    pub const SIGNED_RGB8_NV: c_uint = 0x86FF;
    pub const SIGNED_RGB8_UNSIGNED_ALPHA8_NV: c_uint = 0x870D;
    pub const SIGNED_RGBA8_NV: c_uint = 0x86FC;
    pub const SIGNED_RGBA_NV: c_uint = 0x86FB;
    pub const SIGNED_RGB_NV: c_uint = 0x86FE;
    pub const SIGNED_RGB_UNSIGNED_ALPHA_NV: c_uint = 0x870C;
    pub const SIMULTANEOUS_TEXTURE_AND_DEPTH_TEST: c_uint = 0x82AC;
    pub const SIMULTANEOUS_TEXTURE_AND_DEPTH_WRITE: c_uint = 0x82AE;
    pub const SIMULTANEOUS_TEXTURE_AND_STENCIL_TEST: c_uint = 0x82AD;
    pub const SIMULTANEOUS_TEXTURE_AND_STENCIL_WRITE: c_uint = 0x82AF;
    pub const SINGLE_COLOR_EXT: c_uint = 0x81F9;
    pub const SKIP_COMPONENTS1_NV: c_int = -6;
    pub const SKIP_COMPONENTS2_NV: c_int = -5;
    pub const SKIP_COMPONENTS3_NV: c_int = -4;
    pub const SKIP_COMPONENTS4_NV: c_int = -3;
    pub const SKIP_DECODE_EXT: c_uint = 0x8A4A;
    pub const SKIP_MISSING_GLYPH_NV: c_uint = 0x90A9;
    pub const SLICE_ACCUM_SUN: c_uint = 0x85CC;
    pub const SLUMINANCE8_ALPHA8_EXT: c_uint = 0x8C45;
    pub const SLUMINANCE8_EXT: c_uint = 0x8C47;
    pub const SLUMINANCE_ALPHA_EXT: c_uint = 0x8C44;
    pub const SLUMINANCE_EXT: c_uint = 0x8C46;
    pub const SMALL_CCW_ARC_TO_NV: c_uint = 0x12;
    pub const SMALL_CW_ARC_TO_NV: c_uint = 0x14;
    pub const SMOOTH_CUBIC_CURVE_TO_NV: c_uint = 0x10;
    pub const SMOOTH_LINE_WIDTH_GRANULARITY: c_uint = 0x0B23;
    pub const SMOOTH_LINE_WIDTH_RANGE: c_uint = 0x0B22;
    pub const SMOOTH_POINT_SIZE_GRANULARITY: c_uint = 0x0B13;
    pub const SMOOTH_POINT_SIZE_RANGE: c_uint = 0x0B12;
    pub const SMOOTH_QUADRATIC_CURVE_TO_NV: c_uint = 0x0E;
    pub const SM_COUNT_NV: c_uint = 0x933B;
    pub const SOFTLIGHT_KHR: c_uint = 0x929C;
    pub const SOFTLIGHT_NV: c_uint = 0x929C;
    pub const SOURCE0_ALPHA_ARB: c_uint = 0x8588;
    pub const SOURCE0_ALPHA_EXT: c_uint = 0x8588;
    pub const SOURCE0_RGB_ARB: c_uint = 0x8580;
    pub const SOURCE0_RGB_EXT: c_uint = 0x8580;
    pub const SOURCE1_ALPHA: c_uint = 0x8589;
    pub const SOURCE1_ALPHA_ARB: c_uint = 0x8589;
    pub const SOURCE1_ALPHA_EXT: c_uint = 0x8589;
    pub const SOURCE1_RGB_ARB: c_uint = 0x8581;
    pub const SOURCE1_RGB_EXT: c_uint = 0x8581;
    pub const SOURCE2_ALPHA_ARB: c_uint = 0x858A;
    pub const SOURCE2_ALPHA_EXT: c_uint = 0x858A;
    pub const SOURCE2_RGB_ARB: c_uint = 0x8582;
    pub const SOURCE2_RGB_EXT: c_uint = 0x8582;
    pub const SOURCE3_ALPHA_NV: c_uint = 0x858B;
    pub const SOURCE3_RGB_NV: c_uint = 0x8583;
    pub const SPARE0_NV: c_uint = 0x852E;
    pub const SPARE0_PLUS_SECONDARY_COLOR_NV: c_uint = 0x8532;
    pub const SPARE1_NV: c_uint = 0x852F;
    pub const SPARSE_BUFFER_PAGE_SIZE_ARB: c_uint = 0x82F8;
    pub const SPARSE_STORAGE_BIT_ARB: c_uint = 0x0400;
    pub const SPARSE_TEXTURE_FULL_ARRAY_CUBE_MIPMAPS_ARB: c_uint = 0x91A9;
    pub const SPIR_V_BINARY: c_uint = 0x9552;
    pub const SPIR_V_BINARY_ARB: c_uint = 0x9552;
    pub const SPIR_V_EXTENSIONS: c_uint = 0x9553;
    pub const SPRITE_AXIAL_SGIX: c_uint = 0x814C;
    pub const SPRITE_AXIS_SGIX: c_uint = 0x814A;
    pub const SPRITE_EYE_ALIGNED_SGIX: c_uint = 0x814E;
    pub const SPRITE_MODE_SGIX: c_uint = 0x8149;
    pub const SPRITE_OBJECT_ALIGNED_SGIX: c_uint = 0x814D;
    pub const SPRITE_SGIX: c_uint = 0x8148;
    pub const SPRITE_TRANSLATION_SGIX: c_uint = 0x814B;
    pub const SQUARE_NV: c_uint = 0x90A3;
    pub const SR8_EXT: c_uint = 0x8FBD;
    pub const SRC1_ALPHA: c_uint = 0x8589;
    pub const SRC1_COLOR: c_uint = 0x88F9;
    pub const SRC_ALPHA: c_uint = 0x0302;
    pub const SRC_ALPHA_SATURATE: c_uint = 0x0308;
    pub const SRC_ATOP_NV: c_uint = 0x928E;
    pub const SRC_COLOR: c_uint = 0x0300;
    pub const SRC_IN_NV: c_uint = 0x928A;
    pub const SRC_NV: c_uint = 0x9286;
    pub const SRC_OUT_NV: c_uint = 0x928C;
    pub const SRC_OVER_NV: c_uint = 0x9288;
    pub const SRG8_EXT: c_uint = 0x8FBE;
    pub const SRGB: c_uint = 0x8C40;
    pub const SRGB8: c_uint = 0x8C41;
    pub const SRGB8_ALPHA8: c_uint = 0x8C43;
    pub const SRGB8_ALPHA8_EXT: c_uint = 0x8C43;
    pub const SRGB8_EXT: c_uint = 0x8C41;
    pub const SRGB_ALPHA: c_uint = 0x8C42;
    pub const SRGB_ALPHA_EXT: c_uint = 0x8C42;
    pub const SRGB_DECODE_ARB: c_uint = 0x8299;
    pub const SRGB_EXT: c_uint = 0x8C40;
    pub const SRGB_READ: c_uint = 0x8297;
    pub const SRGB_WRITE: c_uint = 0x8298;
    pub const STACK_OVERFLOW: c_uint = 0x0503;
    pub const STACK_UNDERFLOW: c_uint = 0x0504;
    pub const STANDARD_FONT_FORMAT_NV: c_uint = 0x936C;
    pub const STANDARD_FONT_NAME_NV: c_uint = 0x9072;
    pub const STATIC_ATI: c_uint = 0x8760;
    pub const STATIC_COPY: c_uint = 0x88E6;
    pub const STATIC_COPY_ARB: c_uint = 0x88E6;
    pub const STATIC_DRAW: c_uint = 0x88E4;
    pub const STATIC_DRAW_ARB: c_uint = 0x88E4;
    pub const STATIC_READ: c_uint = 0x88E5;
    pub const STATIC_READ_ARB: c_uint = 0x88E5;
    pub const STATIC_VERTEX_ARRAY_IBM: c_uint = 103061;
    pub const STENCIL: c_uint = 0x1802;
    pub const STENCIL_ATTACHMENT: c_uint = 0x8D20;
    pub const STENCIL_ATTACHMENT_EXT: c_uint = 0x8D20;
    pub const STENCIL_BACK_FAIL: c_uint = 0x8801;
    pub const STENCIL_BACK_FAIL_ATI: c_uint = 0x8801;
    pub const STENCIL_BACK_FUNC: c_uint = 0x8800;
    pub const STENCIL_BACK_FUNC_ATI: c_uint = 0x8800;
    pub const STENCIL_BACK_OP_VALUE_AMD: c_uint = 0x874D;
    pub const STENCIL_BACK_PASS_DEPTH_FAIL: c_uint = 0x8802;
    pub const STENCIL_BACK_PASS_DEPTH_FAIL_ATI: c_uint = 0x8802;
    pub const STENCIL_BACK_PASS_DEPTH_PASS: c_uint = 0x8803;
    pub const STENCIL_BACK_PASS_DEPTH_PASS_ATI: c_uint = 0x8803;
    pub const STENCIL_BACK_REF: c_uint = 0x8CA3;
    pub const STENCIL_BACK_VALUE_MASK: c_uint = 0x8CA4;
    pub const STENCIL_BACK_WRITEMASK: c_uint = 0x8CA5;
    pub const STENCIL_BUFFER_BIT: c_uint = 0x00000400;
    pub const STENCIL_CLEAR_TAG_VALUE_EXT: c_uint = 0x88F3;
    pub const STENCIL_CLEAR_VALUE: c_uint = 0x0B91;
    pub const STENCIL_COMPONENTS: c_uint = 0x8285;
    pub const STENCIL_FAIL: c_uint = 0x0B94;
    pub const STENCIL_FUNC: c_uint = 0x0B92;
    pub const STENCIL_INDEX: c_uint = 0x1901;
    pub const STENCIL_INDEX1: c_uint = 0x8D46;
    pub const STENCIL_INDEX16: c_uint = 0x8D49;
    pub const STENCIL_INDEX16_EXT: c_uint = 0x8D49;
    pub const STENCIL_INDEX1_EXT: c_uint = 0x8D46;
    pub const STENCIL_INDEX4: c_uint = 0x8D47;
    pub const STENCIL_INDEX4_EXT: c_uint = 0x8D47;
    pub const STENCIL_INDEX8: c_uint = 0x8D48;
    pub const STENCIL_INDEX8_EXT: c_uint = 0x8D48;
    pub const STENCIL_OP_VALUE_AMD: c_uint = 0x874C;
    pub const STENCIL_PASS_DEPTH_FAIL: c_uint = 0x0B95;
    pub const STENCIL_PASS_DEPTH_PASS: c_uint = 0x0B96;
    pub const STENCIL_REF: c_uint = 0x0B97;
    pub const STENCIL_REF_COMMAND_NV: c_uint = 0x000C;
    pub const STENCIL_RENDERABLE: c_uint = 0x8288;
    pub const STENCIL_SAMPLES_NV: c_uint = 0x932E;
    pub const STENCIL_TAG_BITS_EXT: c_uint = 0x88F2;
    pub const STENCIL_TEST: c_uint = 0x0B90;
    pub const STENCIL_TEST_TWO_SIDE_EXT: c_uint = 0x8910;
    pub const STENCIL_VALUE_MASK: c_uint = 0x0B93;
    pub const STENCIL_WRITEMASK: c_uint = 0x0B98;
    pub const STEREO: c_uint = 0x0C33;
    pub const STORAGE_CACHED_APPLE: c_uint = 0x85BE;
    pub const STORAGE_CLIENT_APPLE: c_uint = 0x85B4;
    pub const STORAGE_PRIVATE_APPLE: c_uint = 0x85BD;
    pub const STORAGE_SHARED_APPLE: c_uint = 0x85BF;
    pub const STREAM_COPY: c_uint = 0x88E2;
    pub const STREAM_COPY_ARB: c_uint = 0x88E2;
    pub const STREAM_DRAW: c_uint = 0x88E0;
    pub const STREAM_DRAW_ARB: c_uint = 0x88E0;
    pub const STREAM_RASTERIZATION_AMD: c_uint = 0x91A0;
    pub const STREAM_READ: c_uint = 0x88E1;
    pub const STREAM_READ_ARB: c_uint = 0x88E1;
    pub const STRICT_DEPTHFUNC_HINT_PGI: c_uint = 0x1A216;
    pub const STRICT_LIGHTING_HINT_PGI: c_uint = 0x1A217;
    pub const STRICT_SCISSOR_HINT_PGI: c_uint = 0x1A218;
    pub const SUBGROUP_FEATURE_ARITHMETIC_BIT_KHR: c_uint = 0x00000004;
    pub const SUBGROUP_FEATURE_BALLOT_BIT_KHR: c_uint = 0x00000008;
    pub const SUBGROUP_FEATURE_BASIC_BIT_KHR: c_uint = 0x00000001;
    pub const SUBGROUP_FEATURE_CLUSTERED_BIT_KHR: c_uint = 0x00000040;
    pub const SUBGROUP_FEATURE_PARTITIONED_BIT_NV: c_uint = 0x00000100;
    pub const SUBGROUP_FEATURE_QUAD_BIT_KHR: c_uint = 0x00000080;
    pub const SUBGROUP_FEATURE_SHUFFLE_BIT_KHR: c_uint = 0x00000010;
    pub const SUBGROUP_FEATURE_SHUFFLE_RELATIVE_BIT_KHR: c_uint = 0x00000020;
    pub const SUBGROUP_FEATURE_VOTE_BIT_KHR: c_uint = 0x00000002;
    pub const SUBGROUP_QUAD_ALL_STAGES_KHR: c_uint = 0x9535;
    pub const SUBGROUP_SIZE_KHR: c_uint = 0x9532;
    pub const SUBGROUP_SUPPORTED_FEATURES_KHR: c_uint = 0x9534;
    pub const SUBGROUP_SUPPORTED_STAGES_KHR: c_uint = 0x9533;
    pub const SUBPIXEL_BITS: c_uint = 0x0D50;
    pub const SUBPIXEL_PRECISION_BIAS_X_BITS_NV: c_uint = 0x9347;
    pub const SUBPIXEL_PRECISION_BIAS_Y_BITS_NV: c_uint = 0x9348;
    pub const SUBSAMPLE_DISTANCE_AMD: c_uint = 0x883F;
    pub const SUBTRACT_ARB: c_uint = 0x84E7;
    pub const SUB_ATI: c_uint = 0x8965;
    pub const SUCCESS_NV: c_uint = 0x902F;
    pub const SUPERSAMPLE_SCALE_X_NV: c_uint = 0x9372;
    pub const SUPERSAMPLE_SCALE_Y_NV: c_uint = 0x9373;
    pub const SUPPORTED_MULTISAMPLE_MODES_AMD: c_uint = 0x91B7;
    pub const SURFACE_MAPPED_NV: c_uint = 0x8700;
    pub const SURFACE_REGISTERED_NV: c_uint = 0x86FD;
    pub const SURFACE_STATE_NV: c_uint = 0x86EB;
    pub const SWIZZLE_STQ_ATI: c_uint = 0x8977;
    pub const SWIZZLE_STQ_DQ_ATI: c_uint = 0x8979;
    pub const SWIZZLE_STRQ_ATI: c_uint = 0x897A;
    pub const SWIZZLE_STRQ_DQ_ATI: c_uint = 0x897B;
    pub const SWIZZLE_STR_ATI: c_uint = 0x8976;
    pub const SWIZZLE_STR_DR_ATI: c_uint = 0x8978;
    pub const SYNC_CL_EVENT_ARB: c_uint = 0x8240;
    pub const SYNC_CL_EVENT_COMPLETE_ARB: c_uint = 0x8241;
    pub const SYNC_CONDITION: c_uint = 0x9113;
    pub const SYNC_FENCE: c_uint = 0x9116;
    pub const SYNC_FLAGS: c_uint = 0x9115;
    pub const SYNC_FLUSH_COMMANDS_BIT: c_uint = 0x00000001;
    pub const SYNC_GPU_COMMANDS_COMPLETE: c_uint = 0x9117;
    pub const SYNC_STATUS: c_uint = 0x9114;
    pub const SYNC_X11_FENCE_EXT: c_uint = 0x90E1;
    pub const SYSTEM_FONT_NAME_NV: c_uint = 0x9073;
    pub const T2F_IUI_N3F_V2F_EXT: c_uint = 0x81B3;
    pub const T2F_IUI_N3F_V3F_EXT: c_uint = 0x81B4;
    pub const T2F_IUI_V2F_EXT: c_uint = 0x81B1;
    pub const T2F_IUI_V3F_EXT: c_uint = 0x81B2;
    pub const TABLE_TOO_LARGE_EXT: c_uint = 0x8031;
    pub const TANGENT_ARRAY_EXT: c_uint = 0x8439;
    pub const TANGENT_ARRAY_POINTER_EXT: c_uint = 0x8442;
    pub const TANGENT_ARRAY_STRIDE_EXT: c_uint = 0x843F;
    pub const TANGENT_ARRAY_TYPE_EXT: c_uint = 0x843E;
    pub const TASK_SHADER_BIT_NV: c_uint = 0x00000080;
    pub const TASK_SHADER_NV: c_uint = 0x955A;
    pub const TASK_SUBROUTINE_NV: c_uint = 0x957D;
    pub const TASK_SUBROUTINE_UNIFORM_NV: c_uint = 0x957F;
    pub const TASK_WORK_GROUP_SIZE_NV: c_uint = 0x953F;
    pub const TERMINATE_SEQUENCE_COMMAND_NV: c_uint = 0x0000;
    pub const TESSELLATION_FACTOR_AMD: c_uint = 0x9005;
    pub const TESSELLATION_MODE_AMD: c_uint = 0x9004;
    pub const TESS_CONTROL_OUTPUT_VERTICES: c_uint = 0x8E75;
    pub const TESS_CONTROL_PROGRAM_NV: c_uint = 0x891E;
    pub const TESS_CONTROL_PROGRAM_PARAMETER_BUFFER_NV: c_uint = 0x8C74;
    pub const TESS_CONTROL_SHADER: c_uint = 0x8E88;
    pub const TESS_CONTROL_SHADER_BIT: c_uint = 0x00000008;
    pub const TESS_CONTROL_SHADER_PATCHES: c_uint = 0x82F1;
    pub const TESS_CONTROL_SHADER_PATCHES_ARB: c_uint = 0x82F1;
    pub const TESS_CONTROL_SUBROUTINE: c_uint = 0x92E9;
    pub const TESS_CONTROL_SUBROUTINE_UNIFORM: c_uint = 0x92EF;
    pub const TESS_CONTROL_TEXTURE: c_uint = 0x829C;
    pub const TESS_EVALUATION_PROGRAM_NV: c_uint = 0x891F;
    pub const TESS_EVALUATION_PROGRAM_PARAMETER_BUFFER_NV: c_uint = 0x8C75;
    pub const TESS_EVALUATION_SHADER: c_uint = 0x8E87;
    pub const TESS_EVALUATION_SHADER_BIT: c_uint = 0x00000010;
    pub const TESS_EVALUATION_SHADER_INVOCATIONS: c_uint = 0x82F2;
    pub const TESS_EVALUATION_SHADER_INVOCATIONS_ARB: c_uint = 0x82F2;
    pub const TESS_EVALUATION_SUBROUTINE: c_uint = 0x92EA;
    pub const TESS_EVALUATION_SUBROUTINE_UNIFORM: c_uint = 0x92F0;
    pub const TESS_EVALUATION_TEXTURE: c_uint = 0x829D;
    pub const TESS_GEN_MODE: c_uint = 0x8E76;
    pub const TESS_GEN_POINT_MODE: c_uint = 0x8E79;
    pub const TESS_GEN_SPACING: c_uint = 0x8E77;
    pub const TESS_GEN_VERTEX_ORDER: c_uint = 0x8E78;
    pub const TEXCOORD1_BIT_PGI: c_uint = 0x10000000;
    pub const TEXCOORD2_BIT_PGI: c_uint = 0x20000000;
    pub const TEXCOORD3_BIT_PGI: c_uint = 0x40000000;
    pub const TEXCOORD4_BIT_PGI: c_uint = 0x80000000;
    pub const TEXTURE: c_uint = 0x1702;
    pub const TEXTURE0: c_uint = 0x84C0;
    pub const TEXTURE0_ARB: c_uint = 0x84C0;
    pub const TEXTURE1: c_uint = 0x84C1;
    pub const TEXTURE10: c_uint = 0x84CA;
    pub const TEXTURE10_ARB: c_uint = 0x84CA;
    pub const TEXTURE11: c_uint = 0x84CB;
    pub const TEXTURE11_ARB: c_uint = 0x84CB;
    pub const TEXTURE12: c_uint = 0x84CC;
    pub const TEXTURE12_ARB: c_uint = 0x84CC;
    pub const TEXTURE13: c_uint = 0x84CD;
    pub const TEXTURE13_ARB: c_uint = 0x84CD;
    pub const TEXTURE14: c_uint = 0x84CE;
    pub const TEXTURE14_ARB: c_uint = 0x84CE;
    pub const TEXTURE15: c_uint = 0x84CF;
    pub const TEXTURE15_ARB: c_uint = 0x84CF;
    pub const TEXTURE16: c_uint = 0x84D0;
    pub const TEXTURE16_ARB: c_uint = 0x84D0;
    pub const TEXTURE17: c_uint = 0x84D1;
    pub const TEXTURE17_ARB: c_uint = 0x84D1;
    pub const TEXTURE18: c_uint = 0x84D2;
    pub const TEXTURE18_ARB: c_uint = 0x84D2;
    pub const TEXTURE19: c_uint = 0x84D3;
    pub const TEXTURE19_ARB: c_uint = 0x84D3;
    pub const TEXTURE1_ARB: c_uint = 0x84C1;
    pub const TEXTURE2: c_uint = 0x84C2;
    pub const TEXTURE20: c_uint = 0x84D4;
    pub const TEXTURE20_ARB: c_uint = 0x84D4;
    pub const TEXTURE21: c_uint = 0x84D5;
    pub const TEXTURE21_ARB: c_uint = 0x84D5;
    pub const TEXTURE22: c_uint = 0x84D6;
    pub const TEXTURE22_ARB: c_uint = 0x84D6;
    pub const TEXTURE23: c_uint = 0x84D7;
    pub const TEXTURE23_ARB: c_uint = 0x84D7;
    pub const TEXTURE24: c_uint = 0x84D8;
    pub const TEXTURE24_ARB: c_uint = 0x84D8;
    pub const TEXTURE25: c_uint = 0x84D9;
    pub const TEXTURE25_ARB: c_uint = 0x84D9;
    pub const TEXTURE26: c_uint = 0x84DA;
    pub const TEXTURE26_ARB: c_uint = 0x84DA;
    pub const TEXTURE27: c_uint = 0x84DB;
    pub const TEXTURE27_ARB: c_uint = 0x84DB;
    pub const TEXTURE28: c_uint = 0x84DC;
    pub const TEXTURE28_ARB: c_uint = 0x84DC;
    pub const TEXTURE29: c_uint = 0x84DD;
    pub const TEXTURE29_ARB: c_uint = 0x84DD;
    pub const TEXTURE2_ARB: c_uint = 0x84C2;
    pub const TEXTURE3: c_uint = 0x84C3;
    pub const TEXTURE30: c_uint = 0x84DE;
    pub const TEXTURE30_ARB: c_uint = 0x84DE;
    pub const TEXTURE31: c_uint = 0x84DF;
    pub const TEXTURE31_ARB: c_uint = 0x84DF;
    pub const TEXTURE3_ARB: c_uint = 0x84C3;
    pub const TEXTURE4: c_uint = 0x84C4;
    pub const TEXTURE4_ARB: c_uint = 0x84C4;
    pub const TEXTURE5: c_uint = 0x84C5;
    pub const TEXTURE5_ARB: c_uint = 0x84C5;
    pub const TEXTURE6: c_uint = 0x84C6;
    pub const TEXTURE6_ARB: c_uint = 0x84C6;
    pub const TEXTURE7: c_uint = 0x84C7;
    pub const TEXTURE7_ARB: c_uint = 0x84C7;
    pub const TEXTURE8: c_uint = 0x84C8;
    pub const TEXTURE8_ARB: c_uint = 0x84C8;
    pub const TEXTURE9: c_uint = 0x84C9;
    pub const TEXTURE9_ARB: c_uint = 0x84C9;
    pub const TEXTURE_1D: c_uint = 0x0DE0;
    pub const TEXTURE_1D_ARRAY: c_uint = 0x8C18;
    pub const TEXTURE_1D_ARRAY_EXT: c_uint = 0x8C18;
    pub const TEXTURE_1D_BINDING_EXT: c_uint = 0x8068;
    pub const TEXTURE_1D_STACK_BINDING_MESAX: c_uint = 0x875D;
    pub const TEXTURE_1D_STACK_MESAX: c_uint = 0x8759;
    pub const TEXTURE_2D: c_uint = 0x0DE1;
    pub const TEXTURE_2D_ARRAY: c_uint = 0x8C1A;
    pub const TEXTURE_2D_ARRAY_EXT: c_uint = 0x8C1A;
    pub const TEXTURE_2D_BINDING_EXT: c_uint = 0x8069;
    pub const TEXTURE_2D_MULTISAMPLE: c_uint = 0x9100;
    pub const TEXTURE_2D_MULTISAMPLE_ARRAY: c_uint = 0x9102;
    pub const TEXTURE_2D_STACK_BINDING_MESAX: c_uint = 0x875E;
    pub const TEXTURE_2D_STACK_MESAX: c_uint = 0x875A;
    pub const TEXTURE_3D: c_uint = 0x806F;
    pub const TEXTURE_3D_BINDING_EXT: c_uint = 0x806A;
    pub const TEXTURE_3D_EXT: c_uint = 0x806F;
    pub const TEXTURE_4DSIZE_SGIS: c_uint = 0x8136;
    pub const TEXTURE_4D_BINDING_SGIS: c_uint = 0x814F;
    pub const TEXTURE_4D_SGIS: c_uint = 0x8134;
    pub const TEXTURE_ALPHA_SIZE: c_uint = 0x805F;
    pub const TEXTURE_ALPHA_SIZE_EXT: c_uint = 0x805F;
    pub const TEXTURE_ALPHA_TYPE: c_uint = 0x8C13;
    pub const TEXTURE_ALPHA_TYPE_ARB: c_uint = 0x8C13;
    pub const TEXTURE_APPLICATION_MODE_EXT: c_uint = 0x834F;
    pub const TEXTURE_BASE_LEVEL: c_uint = 0x813C;
    pub const TEXTURE_BASE_LEVEL_SGIS: c_uint = 0x813C;
    pub const TEXTURE_BINDING_1D: c_uint = 0x8068;
    pub const TEXTURE_BINDING_1D_ARRAY: c_uint = 0x8C1C;
    pub const TEXTURE_BINDING_1D_ARRAY_EXT: c_uint = 0x8C1C;
    pub const TEXTURE_BINDING_2D: c_uint = 0x8069;
    pub const TEXTURE_BINDING_2D_ARRAY: c_uint = 0x8C1D;
    pub const TEXTURE_BINDING_2D_ARRAY_EXT: c_uint = 0x8C1D;
    pub const TEXTURE_BINDING_2D_MULTISAMPLE: c_uint = 0x9104;
    pub const TEXTURE_BINDING_2D_MULTISAMPLE_ARRAY: c_uint = 0x9105;
    pub const TEXTURE_BINDING_3D: c_uint = 0x806A;
    pub const TEXTURE_BINDING_BUFFER: c_uint = 0x8C2C;
    pub const TEXTURE_BINDING_BUFFER_ARB: c_uint = 0x8C2C;
    pub const TEXTURE_BINDING_BUFFER_EXT: c_uint = 0x8C2C;
    pub const TEXTURE_BINDING_CUBE_MAP: c_uint = 0x8514;
    pub const TEXTURE_BINDING_CUBE_MAP_ARB: c_uint = 0x8514;
    pub const TEXTURE_BINDING_CUBE_MAP_ARRAY: c_uint = 0x900A;
    pub const TEXTURE_BINDING_CUBE_MAP_ARRAY_ARB: c_uint = 0x900A;
    pub const TEXTURE_BINDING_CUBE_MAP_EXT: c_uint = 0x8514;
    pub const TEXTURE_BINDING_RECTANGLE: c_uint = 0x84F6;
    pub const TEXTURE_BINDING_RECTANGLE_ARB: c_uint = 0x84F6;
    pub const TEXTURE_BINDING_RECTANGLE_NV: c_uint = 0x84F6;
    pub const TEXTURE_BINDING_RENDERBUFFER_NV: c_uint = 0x8E53;
    pub const TEXTURE_BLUE_SIZE: c_uint = 0x805E;
    pub const TEXTURE_BLUE_SIZE_EXT: c_uint = 0x805E;
    pub const TEXTURE_BLUE_TYPE: c_uint = 0x8C12;
    pub const TEXTURE_BLUE_TYPE_ARB: c_uint = 0x8C12;
    pub const TEXTURE_BORDER_COLOR: c_uint = 0x1004;
    pub const TEXTURE_BORDER_VALUES_NV: c_uint = 0x871A;
    pub const TEXTURE_BUFFER: c_uint = 0x8C2A;
    pub const TEXTURE_BUFFER_ARB: c_uint = 0x8C2A;
    pub const TEXTURE_BUFFER_BINDING: c_uint = 0x8C2A;
    pub const TEXTURE_BUFFER_DATA_STORE_BINDING: c_uint = 0x8C2D;
    pub const TEXTURE_BUFFER_DATA_STORE_BINDING_ARB: c_uint = 0x8C2D;
    pub const TEXTURE_BUFFER_DATA_STORE_BINDING_EXT: c_uint = 0x8C2D;
    pub const TEXTURE_BUFFER_EXT: c_uint = 0x8C2A;
    pub const TEXTURE_BUFFER_FORMAT_ARB: c_uint = 0x8C2E;
    pub const TEXTURE_BUFFER_FORMAT_EXT: c_uint = 0x8C2E;
    pub const TEXTURE_BUFFER_OFFSET: c_uint = 0x919D;
    pub const TEXTURE_BUFFER_OFFSET_ALIGNMENT: c_uint = 0x919F;
    pub const TEXTURE_BUFFER_SIZE: c_uint = 0x919E;
    pub const TEXTURE_CLIPMAP_CENTER_SGIX: c_uint = 0x8171;
    pub const TEXTURE_CLIPMAP_DEPTH_SGIX: c_uint = 0x8176;
    pub const TEXTURE_CLIPMAP_FRAME_SGIX: c_uint = 0x8172;
    pub const TEXTURE_CLIPMAP_LOD_OFFSET_SGIX: c_uint = 0x8175;
    pub const TEXTURE_CLIPMAP_OFFSET_SGIX: c_uint = 0x8173;
    pub const TEXTURE_CLIPMAP_VIRTUAL_DEPTH_SGIX: c_uint = 0x8174;
    pub const TEXTURE_COLOR_SAMPLES_NV: c_uint = 0x9046;
    pub const TEXTURE_COLOR_TABLE_SGI: c_uint = 0x80BC;
    pub const TEXTURE_COLOR_WRITEMASK_SGIS: c_uint = 0x81EF;
    pub const TEXTURE_COMPARE_FAIL_VALUE_ARB: c_uint = 0x80BF;
    pub const TEXTURE_COMPARE_FUNC: c_uint = 0x884D;
    pub const TEXTURE_COMPARE_FUNC_ARB: c_uint = 0x884D;
    pub const TEXTURE_COMPARE_MODE: c_uint = 0x884C;
    pub const TEXTURE_COMPARE_MODE_ARB: c_uint = 0x884C;
    pub const TEXTURE_COMPARE_OPERATOR_SGIX: c_uint = 0x819B;
    pub const TEXTURE_COMPARE_SGIX: c_uint = 0x819A;
    pub const TEXTURE_COMPRESSED: c_uint = 0x86A1;
    pub const TEXTURE_COMPRESSED_ARB: c_uint = 0x86A1;
    pub const TEXTURE_COMPRESSED_BLOCK_HEIGHT: c_uint = 0x82B2;
    pub const TEXTURE_COMPRESSED_BLOCK_SIZE: c_uint = 0x82B3;
    pub const TEXTURE_COMPRESSED_BLOCK_WIDTH: c_uint = 0x82B1;
    pub const TEXTURE_COMPRESSED_IMAGE_SIZE: c_uint = 0x86A0;
    pub const TEXTURE_COMPRESSED_IMAGE_SIZE_ARB: c_uint = 0x86A0;
    pub const TEXTURE_COMPRESSION_HINT: c_uint = 0x84EF;
    pub const TEXTURE_COMPRESSION_HINT_ARB: c_uint = 0x84EF;
    pub const TEXTURE_CONSTANT_DATA_SUNX: c_uint = 0x81D6;
    pub const TEXTURE_COORD_ARRAY_ADDRESS_NV: c_uint = 0x8F25;
    pub const TEXTURE_COORD_ARRAY_BUFFER_BINDING_ARB: c_uint = 0x889A;
    pub const TEXTURE_COORD_ARRAY_COUNT_EXT: c_uint = 0x808B;
    pub const TEXTURE_COORD_ARRAY_EXT: c_uint = 0x8078;
    pub const TEXTURE_COORD_ARRAY_LENGTH_NV: c_uint = 0x8F2F;
    pub const TEXTURE_COORD_ARRAY_LIST_IBM: c_uint = 103074;
    pub const TEXTURE_COORD_ARRAY_LIST_STRIDE_IBM: c_uint = 103084;
    pub const TEXTURE_COORD_ARRAY_PARALLEL_POINTERS_INTEL: c_uint = 0x83F8;
    pub const TEXTURE_COORD_ARRAY_POINTER_EXT: c_uint = 0x8092;
    pub const TEXTURE_COORD_ARRAY_SIZE_EXT: c_uint = 0x8088;
    pub const TEXTURE_COORD_ARRAY_STRIDE_EXT: c_uint = 0x808A;
    pub const TEXTURE_COORD_ARRAY_TYPE_EXT: c_uint = 0x8089;
    pub const TEXTURE_COORD_NV: c_uint = 0x8C79;
    pub const TEXTURE_COVERAGE_SAMPLES_NV: c_uint = 0x9045;
    pub const TEXTURE_CUBE_MAP: c_uint = 0x8513;
    pub const TEXTURE_CUBE_MAP_ARB: c_uint = 0x8513;
    pub const TEXTURE_CUBE_MAP_ARRAY: c_uint = 0x9009;
    pub const TEXTURE_CUBE_MAP_ARRAY_ARB: c_uint = 0x9009;
    pub const TEXTURE_CUBE_MAP_EXT: c_uint = 0x8513;
    pub const TEXTURE_CUBE_MAP_NEGATIVE_X: c_uint = 0x8516;
    pub const TEXTURE_CUBE_MAP_NEGATIVE_X_ARB: c_uint = 0x8516;
    pub const TEXTURE_CUBE_MAP_NEGATIVE_X_EXT: c_uint = 0x8516;
    pub const TEXTURE_CUBE_MAP_NEGATIVE_Y: c_uint = 0x8518;
    pub const TEXTURE_CUBE_MAP_NEGATIVE_Y_ARB: c_uint = 0x8518;
    pub const TEXTURE_CUBE_MAP_NEGATIVE_Y_EXT: c_uint = 0x8518;
    pub const TEXTURE_CUBE_MAP_NEGATIVE_Z: c_uint = 0x851A;
    pub const TEXTURE_CUBE_MAP_NEGATIVE_Z_ARB: c_uint = 0x851A;
    pub const TEXTURE_CUBE_MAP_NEGATIVE_Z_EXT: c_uint = 0x851A;
    pub const TEXTURE_CUBE_MAP_POSITIVE_X: c_uint = 0x8515;
    pub const TEXTURE_CUBE_MAP_POSITIVE_X_ARB: c_uint = 0x8515;
    pub const TEXTURE_CUBE_MAP_POSITIVE_X_EXT: c_uint = 0x8515;
    pub const TEXTURE_CUBE_MAP_POSITIVE_Y: c_uint = 0x8517;
    pub const TEXTURE_CUBE_MAP_POSITIVE_Y_ARB: c_uint = 0x8517;
    pub const TEXTURE_CUBE_MAP_POSITIVE_Y_EXT: c_uint = 0x8517;
    pub const TEXTURE_CUBE_MAP_POSITIVE_Z: c_uint = 0x8519;
    pub const TEXTURE_CUBE_MAP_POSITIVE_Z_ARB: c_uint = 0x8519;
    pub const TEXTURE_CUBE_MAP_POSITIVE_Z_EXT: c_uint = 0x8519;
    pub const TEXTURE_CUBE_MAP_SEAMLESS: c_uint = 0x884F;
    pub const TEXTURE_DEFORMATION_BIT_SGIX: c_uint = 0x00000001;
    pub const TEXTURE_DEFORMATION_SGIX: c_uint = 0x8195;
    pub const TEXTURE_DEPTH: c_uint = 0x8071;
    pub const TEXTURE_DEPTH_EXT: c_uint = 0x8071;
    pub const TEXTURE_DEPTH_SIZE: c_uint = 0x884A;
    pub const TEXTURE_DEPTH_SIZE_ARB: c_uint = 0x884A;
    pub const TEXTURE_DEPTH_TYPE: c_uint = 0x8C16;
    pub const TEXTURE_DEPTH_TYPE_ARB: c_uint = 0x8C16;
    pub const TEXTURE_DS_SIZE_NV: c_uint = 0x871D;
    pub const TEXTURE_DT_SIZE_NV: c_uint = 0x871E;
    pub const TEXTURE_ENV_BIAS_SGIX: c_uint = 0x80BE;
    pub const TEXTURE_FETCH_BARRIER_BIT: c_uint = 0x00000008;
    pub const TEXTURE_FETCH_BARRIER_BIT_EXT: c_uint = 0x00000008;
    pub const TEXTURE_FILTER4_SIZE_SGIS: c_uint = 0x8147;
    pub const TEXTURE_FILTER_CONTROL_EXT: c_uint = 0x8500;
    pub const TEXTURE_FIXED_SAMPLE_LOCATIONS: c_uint = 0x9107;
    pub const TEXTURE_FLOAT_COMPONENTS_NV: c_uint = 0x888C;
    pub const TEXTURE_FREE_MEMORY_ATI: c_uint = 0x87FC;
    pub const TEXTURE_GATHER: c_uint = 0x82A2;
    pub const TEXTURE_GATHER_SHADOW: c_uint = 0x82A3;
    pub const TEXTURE_GEQUAL_R_SGIX: c_uint = 0x819D;
    pub const TEXTURE_GREEN_SIZE: c_uint = 0x805D;
    pub const TEXTURE_GREEN_SIZE_EXT: c_uint = 0x805D;
    pub const TEXTURE_GREEN_TYPE: c_uint = 0x8C11;
    pub const TEXTURE_GREEN_TYPE_ARB: c_uint = 0x8C11;
    pub const TEXTURE_HEIGHT: c_uint = 0x1001;
    pub const TEXTURE_HI_SIZE_NV: c_uint = 0x871B;
    pub const TEXTURE_IMAGE_FORMAT: c_uint = 0x828F;
    pub const TEXTURE_IMAGE_TYPE: c_uint = 0x8290;
    pub const TEXTURE_IMMUTABLE_FORMAT: c_uint = 0x912F;
    pub const TEXTURE_IMMUTABLE_FORMAT_EXT: c_uint = 0x912F;
    pub const TEXTURE_IMMUTABLE_LEVELS: c_uint = 0x82DF;
    pub const TEXTURE_INDEX_SIZE_EXT: c_uint = 0x80ED;
    pub const TEXTURE_INTENSITY_SIZE_EXT: c_uint = 0x8061;
    pub const TEXTURE_INTENSITY_TYPE_ARB: c_uint = 0x8C15;
    pub const TEXTURE_INTERNAL_FORMAT: c_uint = 0x1003;
    pub const TEXTURE_LEQUAL_R_SGIX: c_uint = 0x819C;
    pub const TEXTURE_LIGHTING_MODE_HP: c_uint = 0x8167;
    pub const TEXTURE_LIGHT_EXT: c_uint = 0x8350;
    pub const TEXTURE_LOD_BIAS: c_uint = 0x8501;
    pub const TEXTURE_LOD_BIAS_EXT: c_uint = 0x8501;
    pub const TEXTURE_LOD_BIAS_R_SGIX: c_uint = 0x8190;
    pub const TEXTURE_LOD_BIAS_S_SGIX: c_uint = 0x818E;
    pub const TEXTURE_LOD_BIAS_T_SGIX: c_uint = 0x818F;
    pub const TEXTURE_LO_SIZE_NV: c_uint = 0x871C;
    pub const TEXTURE_LUMINANCE_SIZE_EXT: c_uint = 0x8060;
    pub const TEXTURE_LUMINANCE_TYPE_ARB: c_uint = 0x8C14;
    pub const TEXTURE_MAG_FILTER: c_uint = 0x2800;
    pub const TEXTURE_MAG_SIZE_NV: c_uint = 0x871F;
    pub const TEXTURE_MATERIAL_FACE_EXT: c_uint = 0x8351;
    pub const TEXTURE_MATERIAL_PARAMETER_EXT: c_uint = 0x8352;
    pub const TEXTURE_MAX_ANISOTROPY: c_uint = 0x84FE;
    pub const TEXTURE_MAX_ANISOTROPY_EXT: c_uint = 0x84FE;
    pub const TEXTURE_MAX_CLAMP_R_SGIX: c_uint = 0x836B;
    pub const TEXTURE_MAX_CLAMP_S_SGIX: c_uint = 0x8369;
    pub const TEXTURE_MAX_CLAMP_T_SGIX: c_uint = 0x836A;
    pub const TEXTURE_MAX_LEVEL: c_uint = 0x813D;
    pub const TEXTURE_MAX_LEVEL_SGIS: c_uint = 0x813D;
    pub const TEXTURE_MAX_LOD: c_uint = 0x813B;
    pub const TEXTURE_MAX_LOD_SGIS: c_uint = 0x813B;
    pub const TEXTURE_MEMORY_LAYOUT_INTEL: c_uint = 0x83FF;
    pub const TEXTURE_MIN_FILTER: c_uint = 0x2801;
    pub const TEXTURE_MIN_LOD: c_uint = 0x813A;
    pub const TEXTURE_MIN_LOD_SGIS: c_uint = 0x813A;
    pub const TEXTURE_MULTI_BUFFER_HINT_SGIX: c_uint = 0x812E;
    pub const TEXTURE_NORMAL_EXT: c_uint = 0x85AF;
    pub const TEXTURE_POST_SPECULAR_HP: c_uint = 0x8168;
    pub const TEXTURE_PRE_SPECULAR_HP: c_uint = 0x8169;
    pub const TEXTURE_PRIORITY_EXT: c_uint = 0x8066;
    pub const TEXTURE_RANGE_LENGTH_APPLE: c_uint = 0x85B7;
    pub const TEXTURE_RANGE_POINTER_APPLE: c_uint = 0x85B8;
    pub const TEXTURE_RECTANGLE: c_uint = 0x84F5;
    pub const TEXTURE_RECTANGLE_ARB: c_uint = 0x84F5;
    pub const TEXTURE_RECTANGLE_NV: c_uint = 0x84F5;
    pub const TEXTURE_REDUCTION_MODE_ARB: c_uint = 0x9366;
    pub const TEXTURE_REDUCTION_MODE_EXT: c_uint = 0x9366;
    pub const TEXTURE_RED_SIZE: c_uint = 0x805C;
    pub const TEXTURE_RED_SIZE_EXT: c_uint = 0x805C;
    pub const TEXTURE_RED_TYPE: c_uint = 0x8C10;
    pub const TEXTURE_RED_TYPE_ARB: c_uint = 0x8C10;
    pub const TEXTURE_RENDERBUFFER_DATA_STORE_BINDING_NV: c_uint = 0x8E54;
    pub const TEXTURE_RENDERBUFFER_NV: c_uint = 0x8E55;
    pub const TEXTURE_RESIDENT_EXT: c_uint = 0x8067;
    pub const TEXTURE_SAMPLES: c_uint = 0x9106;
    pub const TEXTURE_SHADER_NV: c_uint = 0x86DE;
    pub const TEXTURE_SHADOW: c_uint = 0x82A1;
    pub const TEXTURE_SHARED_SIZE: c_uint = 0x8C3F;
    pub const TEXTURE_SHARED_SIZE_EXT: c_uint = 0x8C3F;
    pub const TEXTURE_SPARSE_ARB: c_uint = 0x91A6;
    pub const TEXTURE_SRGB_DECODE_EXT: c_uint = 0x8A48;
    pub const TEXTURE_STENCIL_SIZE: c_uint = 0x88F1;
    pub const TEXTURE_STENCIL_SIZE_EXT: c_uint = 0x88F1;
    pub const TEXTURE_STORAGE_HINT_APPLE: c_uint = 0x85BC;
    pub const TEXTURE_STORAGE_SPARSE_BIT_AMD: c_uint = 0x00000001;
    pub const TEXTURE_SWIZZLE_A: c_uint = 0x8E45;
    pub const TEXTURE_SWIZZLE_A_EXT: c_uint = 0x8E45;
    pub const TEXTURE_SWIZZLE_B: c_uint = 0x8E44;
    pub const TEXTURE_SWIZZLE_B_EXT: c_uint = 0x8E44;
    pub const TEXTURE_SWIZZLE_G: c_uint = 0x8E43;
    pub const TEXTURE_SWIZZLE_G_EXT: c_uint = 0x8E43;
    pub const TEXTURE_SWIZZLE_R: c_uint = 0x8E42;
    pub const TEXTURE_SWIZZLE_RGBA: c_uint = 0x8E46;
    pub const TEXTURE_SWIZZLE_RGBA_EXT: c_uint = 0x8E46;
    pub const TEXTURE_SWIZZLE_R_EXT: c_uint = 0x8E42;
    pub const TEXTURE_TARGET: c_uint = 0x1006;
    pub const TEXTURE_TILING_EXT: c_uint = 0x9580;
    pub const TEXTURE_TOO_LARGE_EXT: c_uint = 0x8065;
    pub const TEXTURE_UNSIGNED_REMAP_MODE_NV: c_uint = 0x888F;
    pub const TEXTURE_UPDATE_BARRIER_BIT: c_uint = 0x00000100;
    pub const TEXTURE_UPDATE_BARRIER_BIT_EXT: c_uint = 0x00000100;
    pub const TEXTURE_VIEW: c_uint = 0x82B5;
    pub const TEXTURE_VIEW_MIN_LAYER: c_uint = 0x82DD;
    pub const TEXTURE_VIEW_MIN_LEVEL: c_uint = 0x82DB;
    pub const TEXTURE_VIEW_NUM_LAYERS: c_uint = 0x82DE;
    pub const TEXTURE_VIEW_NUM_LEVELS: c_uint = 0x82DC;
    pub const TEXTURE_WIDTH: c_uint = 0x1000;
    pub const TEXTURE_WRAP_Q_SGIS: c_uint = 0x8137;
    pub const TEXTURE_WRAP_R: c_uint = 0x8072;
    pub const TEXTURE_WRAP_R_EXT: c_uint = 0x8072;
    pub const TEXTURE_WRAP_S: c_uint = 0x2802;
    pub const TEXTURE_WRAP_T: c_uint = 0x2803;
    pub const TEXT_FRAGMENT_SHADER_ATI: c_uint = 0x8200;
    pub const TILE_RASTER_ORDER_FIXED_MESA: c_uint = 0x8BB8;
    pub const TILE_RASTER_ORDER_INCREASING_X_MESA: c_uint = 0x8BB9;
    pub const TILE_RASTER_ORDER_INCREASING_Y_MESA: c_uint = 0x8BBA;
    pub const TILING_TYPES_EXT: c_uint = 0x9583;
    pub const TIMELINE_SEMAPHORE_VALUE_NV: c_uint = 0x9595;
    pub const TIMEOUT_EXPIRED: c_uint = 0x911B;
    pub const TIMEOUT_IGNORED: u64 = 0xFFFFFFFFFFFFFFFF;
    pub const TIMESTAMP: c_uint = 0x8E28;
    pub const TIME_ELAPSED: c_uint = 0x88BF;
    pub const TIME_ELAPSED_EXT: c_uint = 0x88BF;
    pub const TOP_LEVEL_ARRAY_SIZE: c_uint = 0x930C;
    pub const TOP_LEVEL_ARRAY_STRIDE: c_uint = 0x930D;
    pub const TRACK_MATRIX_NV: c_uint = 0x8648;
    pub const TRACK_MATRIX_TRANSFORM_NV: c_uint = 0x8649;
    pub const TRANSFORM_FEEDBACK: c_uint = 0x8E22;
    pub const TRANSFORM_FEEDBACK_ACTIVE: c_uint = 0x8E24;
    pub const TRANSFORM_FEEDBACK_ATTRIBS_NV: c_uint = 0x8C7E;
    pub const TRANSFORM_FEEDBACK_BARRIER_BIT: c_uint = 0x00000800;
    pub const TRANSFORM_FEEDBACK_BARRIER_BIT_EXT: c_uint = 0x00000800;
    pub const TRANSFORM_FEEDBACK_BINDING: c_uint = 0x8E25;
    pub const TRANSFORM_FEEDBACK_BINDING_NV: c_uint = 0x8E25;
    pub const TRANSFORM_FEEDBACK_BUFFER: c_uint = 0x8C8E;
    pub const TRANSFORM_FEEDBACK_BUFFER_ACTIVE: c_uint = 0x8E24;
    pub const TRANSFORM_FEEDBACK_BUFFER_ACTIVE_NV: c_uint = 0x8E24;
    pub const TRANSFORM_FEEDBACK_BUFFER_BINDING: c_uint = 0x8C8F;
    pub const TRANSFORM_FEEDBACK_BUFFER_BINDING_EXT: c_uint = 0x8C8F;
    pub const TRANSFORM_FEEDBACK_BUFFER_BINDING_NV: c_uint = 0x8C8F;
    pub const TRANSFORM_FEEDBACK_BUFFER_EXT: c_uint = 0x8C8E;
    pub const TRANSFORM_FEEDBACK_BUFFER_INDEX: c_uint = 0x934B;
    pub const TRANSFORM_FEEDBACK_BUFFER_MODE: c_uint = 0x8C7F;
    pub const TRANSFORM_FEEDBACK_BUFFER_MODE_EXT: c_uint = 0x8C7F;
    pub const TRANSFORM_FEEDBACK_BUFFER_MODE_NV: c_uint = 0x8C7F;
    pub const TRANSFORM_FEEDBACK_BUFFER_NV: c_uint = 0x8C8E;
    pub const TRANSFORM_FEEDBACK_BUFFER_PAUSED: c_uint = 0x8E23;
    pub const TRANSFORM_FEEDBACK_BUFFER_PAUSED_NV: c_uint = 0x8E23;
    pub const TRANSFORM_FEEDBACK_BUFFER_SIZE: c_uint = 0x8C85;
    pub const TRANSFORM_FEEDBACK_BUFFER_SIZE_EXT: c_uint = 0x8C85;
    pub const TRANSFORM_FEEDBACK_BUFFER_SIZE_NV: c_uint = 0x8C85;
    pub const TRANSFORM_FEEDBACK_BUFFER_START: c_uint = 0x8C84;
    pub const TRANSFORM_FEEDBACK_BUFFER_START_EXT: c_uint = 0x8C84;
    pub const TRANSFORM_FEEDBACK_BUFFER_START_NV: c_uint = 0x8C84;
    pub const TRANSFORM_FEEDBACK_BUFFER_STRIDE: c_uint = 0x934C;
    pub const TRANSFORM_FEEDBACK_NV: c_uint = 0x8E22;
    pub const TRANSFORM_FEEDBACK_OVERFLOW: c_uint = 0x82EC;
    pub const TRANSFORM_FEEDBACK_OVERFLOW_ARB: c_uint = 0x82EC;
    pub const TRANSFORM_FEEDBACK_PAUSED: c_uint = 0x8E23;
    pub const TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN: c_uint = 0x8C88;
    pub const TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN_EXT: c_uint = 0x8C88;
    pub const TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN_NV: c_uint = 0x8C88;
    pub const TRANSFORM_FEEDBACK_RECORD_NV: c_uint = 0x8C86;
    pub const TRANSFORM_FEEDBACK_STREAM_OVERFLOW: c_uint = 0x82ED;
    pub const TRANSFORM_FEEDBACK_STREAM_OVERFLOW_ARB: c_uint = 0x82ED;
    pub const TRANSFORM_FEEDBACK_VARYING: c_uint = 0x92F4;
    pub const TRANSFORM_FEEDBACK_VARYINGS: c_uint = 0x8C83;
    pub const TRANSFORM_FEEDBACK_VARYINGS_EXT: c_uint = 0x8C83;
    pub const TRANSFORM_FEEDBACK_VARYINGS_NV: c_uint = 0x8C83;
    pub const TRANSFORM_FEEDBACK_VARYING_MAX_LENGTH: c_uint = 0x8C76;
    pub const TRANSFORM_FEEDBACK_VARYING_MAX_LENGTH_EXT: c_uint = 0x8C76;
    pub const TRANSFORM_HINT_APPLE: c_uint = 0x85B1;
    pub const TRANSLATE_2D_NV: c_uint = 0x9090;
    pub const TRANSLATE_3D_NV: c_uint = 0x9091;
    pub const TRANSLATE_X_NV: c_uint = 0x908E;
    pub const TRANSLATE_Y_NV: c_uint = 0x908F;
    pub const TRANSPOSE_AFFINE_2D_NV: c_uint = 0x9096;
    pub const TRANSPOSE_AFFINE_3D_NV: c_uint = 0x9098;
    pub const TRANSPOSE_COLOR_MATRIX_ARB: c_uint = 0x84E6;
    pub const TRANSPOSE_CURRENT_MATRIX_ARB: c_uint = 0x88B7;
    pub const TRANSPOSE_MODELVIEW_MATRIX_ARB: c_uint = 0x84E3;
    pub const TRANSPOSE_NV: c_uint = 0x862C;
    pub const TRANSPOSE_PROGRAM_MATRIX_EXT: c_uint = 0x8E2E;
    pub const TRANSPOSE_PROJECTION_MATRIX_ARB: c_uint = 0x84E4;
    pub const TRANSPOSE_TEXTURE_MATRIX_ARB: c_uint = 0x84E5;
    pub const TRIANGLES: c_uint = 0x0004;
    pub const TRIANGLES_ADJACENCY: c_uint = 0x000C;
    pub const TRIANGLES_ADJACENCY_ARB: c_uint = 0x000C;
    pub const TRIANGLES_ADJACENCY_EXT: c_uint = 0x000C;
    pub const TRIANGLE_FAN: c_uint = 0x0006;
    pub const TRIANGLE_LIST_SUN: c_uint = 0x81D7;
    pub const TRIANGLE_MESH_SUN: c_uint = 0x8615;
    pub const TRIANGLE_STRIP: c_uint = 0x0005;
    pub const TRIANGLE_STRIP_ADJACENCY: c_uint = 0x000D;
    pub const TRIANGLE_STRIP_ADJACENCY_ARB: c_uint = 0x000D;
    pub const TRIANGLE_STRIP_ADJACENCY_EXT: c_uint = 0x000D;
    pub const TRIANGULAR_NV: c_uint = 0x90A5;
    pub const TRUE: c_uchar = 1;
    pub const TYPE: c_uint = 0x92FA;
    pub const UNCORRELATED_NV: c_uint = 0x9282;
    pub const UNDEFINED_APPLE: c_uint = 0x8A1C;
    pub const UNDEFINED_VERTEX: c_uint = 0x8260;
    pub const UNIFORM: c_uint = 0x92E1;
    pub const UNIFORM_ADDRESS_COMMAND_NV: c_uint = 0x000A;
    pub const UNIFORM_ARRAY_STRIDE: c_uint = 0x8A3C;
    pub const UNIFORM_ATOMIC_COUNTER_BUFFER_INDEX: c_uint = 0x92DA;
    pub const UNIFORM_BARRIER_BIT: c_uint = 0x00000004;
    pub const UNIFORM_BARRIER_BIT_EXT: c_uint = 0x00000004;
    pub const UNIFORM_BLOCK: c_uint = 0x92E2;
    pub const UNIFORM_BLOCK_ACTIVE_UNIFORMS: c_uint = 0x8A42;
    pub const UNIFORM_BLOCK_ACTIVE_UNIFORM_INDICES: c_uint = 0x8A43;
    pub const UNIFORM_BLOCK_BINDING: c_uint = 0x8A3F;
    pub const UNIFORM_BLOCK_DATA_SIZE: c_uint = 0x8A40;
    pub const UNIFORM_BLOCK_INDEX: c_uint = 0x8A3A;
    pub const UNIFORM_BLOCK_NAME_LENGTH: c_uint = 0x8A41;
    pub const UNIFORM_BLOCK_REFERENCED_BY_COMPUTE_SHADER: c_uint = 0x90EC;
    pub const UNIFORM_BLOCK_REFERENCED_BY_FRAGMENT_SHADER: c_uint = 0x8A46;
    pub const UNIFORM_BLOCK_REFERENCED_BY_GEOMETRY_SHADER: c_uint = 0x8A45;
    pub const UNIFORM_BLOCK_REFERENCED_BY_MESH_SHADER_NV: c_uint = 0x959C;
    pub const UNIFORM_BLOCK_REFERENCED_BY_TASK_SHADER_NV: c_uint = 0x959D;
    pub const UNIFORM_BLOCK_REFERENCED_BY_TESS_CONTROL_SHADER: c_uint = 0x84F0;
    pub const UNIFORM_BLOCK_REFERENCED_BY_TESS_EVALUATION_SHADER: c_uint = 0x84F1;
    pub const UNIFORM_BLOCK_REFERENCED_BY_VERTEX_SHADER: c_uint = 0x8A44;
    pub const UNIFORM_BUFFER: c_uint = 0x8A11;
    pub const UNIFORM_BUFFER_ADDRESS_NV: c_uint = 0x936F;
    pub const UNIFORM_BUFFER_BINDING: c_uint = 0x8A28;
    pub const UNIFORM_BUFFER_BINDING_EXT: c_uint = 0x8DEF;
    pub const UNIFORM_BUFFER_EXT: c_uint = 0x8DEE;
    pub const UNIFORM_BUFFER_LENGTH_NV: c_uint = 0x9370;
    pub const UNIFORM_BUFFER_OFFSET_ALIGNMENT: c_uint = 0x8A34;
    pub const UNIFORM_BUFFER_SIZE: c_uint = 0x8A2A;
    pub const UNIFORM_BUFFER_START: c_uint = 0x8A29;
    pub const UNIFORM_BUFFER_UNIFIED_NV: c_uint = 0x936E;
    pub const UNIFORM_IS_ROW_MAJOR: c_uint = 0x8A3E;
    pub const UNIFORM_MATRIX_STRIDE: c_uint = 0x8A3D;
    pub const UNIFORM_NAME_LENGTH: c_uint = 0x8A39;
    pub const UNIFORM_OFFSET: c_uint = 0x8A3B;
    pub const UNIFORM_SIZE: c_uint = 0x8A38;
    pub const UNIFORM_TYPE: c_uint = 0x8A37;
    pub const UNKNOWN_CONTEXT_RESET: c_uint = 0x8255;
    pub const UNKNOWN_CONTEXT_RESET_ARB: c_uint = 0x8255;
    pub const UNPACK_ALIGNMENT: c_uint = 0x0CF5;
    pub const UNPACK_CLIENT_STORAGE_APPLE: c_uint = 0x85B2;
    pub const UNPACK_CMYK_HINT_EXT: c_uint = 0x800F;
    pub const UNPACK_COMPRESSED_BLOCK_DEPTH: c_uint = 0x9129;
    pub const UNPACK_COMPRESSED_BLOCK_HEIGHT: c_uint = 0x9128;
    pub const UNPACK_COMPRESSED_BLOCK_SIZE: c_uint = 0x912A;
    pub const UNPACK_COMPRESSED_BLOCK_WIDTH: c_uint = 0x9127;
    pub const UNPACK_CONSTANT_DATA_SUNX: c_uint = 0x81D5;
    pub const UNPACK_IMAGE_DEPTH_SGIS: c_uint = 0x8133;
    pub const UNPACK_IMAGE_HEIGHT: c_uint = 0x806E;
    pub const UNPACK_IMAGE_HEIGHT_EXT: c_uint = 0x806E;
    pub const UNPACK_LSB_FIRST: c_uint = 0x0CF1;
    pub const UNPACK_RESAMPLE_OML: c_uint = 0x8985;
    pub const UNPACK_RESAMPLE_SGIX: c_uint = 0x842F;
    pub const UNPACK_ROW_BYTES_APPLE: c_uint = 0x8A16;
    pub const UNPACK_ROW_LENGTH: c_uint = 0x0CF2;
    pub const UNPACK_SKIP_IMAGES: c_uint = 0x806D;
    pub const UNPACK_SKIP_IMAGES_EXT: c_uint = 0x806D;
    pub const UNPACK_SKIP_PIXELS: c_uint = 0x0CF4;
    pub const UNPACK_SKIP_ROWS: c_uint = 0x0CF3;
    pub const UNPACK_SKIP_VOLUMES_SGIS: c_uint = 0x8132;
    pub const UNPACK_SUBSAMPLE_RATE_SGIX: c_uint = 0x85A1;
    pub const UNPACK_SWAP_BYTES: c_uint = 0x0CF0;
    pub const UNSIGNALED: c_uint = 0x9118;
    pub const UNSIGNED_BYTE: c_uint = 0x1401;
    pub const UNSIGNED_BYTE_2_3_3_REV: c_uint = 0x8362;
    pub const UNSIGNED_BYTE_3_3_2: c_uint = 0x8032;
    pub const UNSIGNED_BYTE_3_3_2_EXT: c_uint = 0x8032;
    pub const UNSIGNED_IDENTITY_NV: c_uint = 0x8536;
    pub const UNSIGNED_INT: c_uint = 0x1405;
    pub const UNSIGNED_INT16_NV: c_uint = 0x8FF0;
    pub const UNSIGNED_INT16_VEC2_NV: c_uint = 0x8FF1;
    pub const UNSIGNED_INT16_VEC3_NV: c_uint = 0x8FF2;
    pub const UNSIGNED_INT16_VEC4_NV: c_uint = 0x8FF3;
    pub const UNSIGNED_INT64_AMD: c_uint = 0x8BC2;
    pub const UNSIGNED_INT64_ARB: c_uint = 0x140F;
    pub const UNSIGNED_INT64_NV: c_uint = 0x140F;
    pub const UNSIGNED_INT64_VEC2_ARB: c_uint = 0x8FF5;
    pub const UNSIGNED_INT64_VEC2_NV: c_uint = 0x8FF5;
    pub const UNSIGNED_INT64_VEC3_ARB: c_uint = 0x8FF6;
    pub const UNSIGNED_INT64_VEC3_NV: c_uint = 0x8FF6;
    pub const UNSIGNED_INT64_VEC4_ARB: c_uint = 0x8FF7;
    pub const UNSIGNED_INT64_VEC4_NV: c_uint = 0x8FF7;
    pub const UNSIGNED_INT8_NV: c_uint = 0x8FEC;
    pub const UNSIGNED_INT8_VEC2_NV: c_uint = 0x8FED;
    pub const UNSIGNED_INT8_VEC3_NV: c_uint = 0x8FEE;
    pub const UNSIGNED_INT8_VEC4_NV: c_uint = 0x8FEF;
    pub const UNSIGNED_INT_10F_11F_11F_REV: c_uint = 0x8C3B;
    pub const UNSIGNED_INT_10F_11F_11F_REV_EXT: c_uint = 0x8C3B;
    pub const UNSIGNED_INT_10_10_10_2: c_uint = 0x8036;
    pub const UNSIGNED_INT_10_10_10_2_EXT: c_uint = 0x8036;
    pub const UNSIGNED_INT_24_8: c_uint = 0x84FA;
    pub const UNSIGNED_INT_24_8_EXT: c_uint = 0x84FA;
    pub const UNSIGNED_INT_24_8_NV: c_uint = 0x84FA;
    pub const UNSIGNED_INT_2_10_10_10_REV: c_uint = 0x8368;
    pub const UNSIGNED_INT_5_9_9_9_REV: c_uint = 0x8C3E;
    pub const UNSIGNED_INT_5_9_9_9_REV_EXT: c_uint = 0x8C3E;
    pub const UNSIGNED_INT_8_8_8_8: c_uint = 0x8035;
    pub const UNSIGNED_INT_8_8_8_8_EXT: c_uint = 0x8035;
    pub const UNSIGNED_INT_8_8_8_8_REV: c_uint = 0x8367;
    pub const UNSIGNED_INT_8_8_S8_S8_REV_NV: c_uint = 0x86DB;
    pub const UNSIGNED_INT_ATOMIC_COUNTER: c_uint = 0x92DB;
    pub const UNSIGNED_INT_IMAGE_1D: c_uint = 0x9062;
    pub const UNSIGNED_INT_IMAGE_1D_ARRAY: c_uint = 0x9068;
    pub const UNSIGNED_INT_IMAGE_1D_ARRAY_EXT: c_uint = 0x9068;
    pub const UNSIGNED_INT_IMAGE_1D_EXT: c_uint = 0x9062;
    pub const UNSIGNED_INT_IMAGE_2D: c_uint = 0x9063;
    pub const UNSIGNED_INT_IMAGE_2D_ARRAY: c_uint = 0x9069;
    pub const UNSIGNED_INT_IMAGE_2D_ARRAY_EXT: c_uint = 0x9069;
    pub const UNSIGNED_INT_IMAGE_2D_EXT: c_uint = 0x9063;
    pub const UNSIGNED_INT_IMAGE_2D_MULTISAMPLE: c_uint = 0x906B;
    pub const UNSIGNED_INT_IMAGE_2D_MULTISAMPLE_ARRAY: c_uint = 0x906C;
    pub const UNSIGNED_INT_IMAGE_2D_MULTISAMPLE_ARRAY_EXT: c_uint = 0x906C;
    pub const UNSIGNED_INT_IMAGE_2D_MULTISAMPLE_EXT: c_uint = 0x906B;
    pub const UNSIGNED_INT_IMAGE_2D_RECT: c_uint = 0x9065;
    pub const UNSIGNED_INT_IMAGE_2D_RECT_EXT: c_uint = 0x9065;
    pub const UNSIGNED_INT_IMAGE_3D: c_uint = 0x9064;
    pub const UNSIGNED_INT_IMAGE_3D_EXT: c_uint = 0x9064;
    pub const UNSIGNED_INT_IMAGE_BUFFER: c_uint = 0x9067;
    pub const UNSIGNED_INT_IMAGE_BUFFER_EXT: c_uint = 0x9067;
    pub const UNSIGNED_INT_IMAGE_CUBE: c_uint = 0x9066;
    pub const UNSIGNED_INT_IMAGE_CUBE_EXT: c_uint = 0x9066;
    pub const UNSIGNED_INT_IMAGE_CUBE_MAP_ARRAY: c_uint = 0x906A;
    pub const UNSIGNED_INT_IMAGE_CUBE_MAP_ARRAY_EXT: c_uint = 0x906A;
    pub const UNSIGNED_INT_S8_S8_8_8_NV: c_uint = 0x86DA;
    pub const UNSIGNED_INT_SAMPLER_1D: c_uint = 0x8DD1;
    pub const UNSIGNED_INT_SAMPLER_1D_ARRAY: c_uint = 0x8DD6;
    pub const UNSIGNED_INT_SAMPLER_1D_ARRAY_EXT: c_uint = 0x8DD6;
    pub const UNSIGNED_INT_SAMPLER_1D_EXT: c_uint = 0x8DD1;
    pub const UNSIGNED_INT_SAMPLER_2D: c_uint = 0x8DD2;
    pub const UNSIGNED_INT_SAMPLER_2D_ARRAY: c_uint = 0x8DD7;
    pub const UNSIGNED_INT_SAMPLER_2D_ARRAY_EXT: c_uint = 0x8DD7;
    pub const UNSIGNED_INT_SAMPLER_2D_EXT: c_uint = 0x8DD2;
    pub const UNSIGNED_INT_SAMPLER_2D_MULTISAMPLE: c_uint = 0x910A;
    pub const UNSIGNED_INT_SAMPLER_2D_MULTISAMPLE_ARRAY: c_uint = 0x910D;
    pub const UNSIGNED_INT_SAMPLER_2D_RECT: c_uint = 0x8DD5;
    pub const UNSIGNED_INT_SAMPLER_2D_RECT_EXT: c_uint = 0x8DD5;
    pub const UNSIGNED_INT_SAMPLER_3D: c_uint = 0x8DD3;
    pub const UNSIGNED_INT_SAMPLER_3D_EXT: c_uint = 0x8DD3;
    pub const UNSIGNED_INT_SAMPLER_BUFFER: c_uint = 0x8DD8;
    pub const UNSIGNED_INT_SAMPLER_BUFFER_AMD: c_uint = 0x9003;
    pub const UNSIGNED_INT_SAMPLER_BUFFER_EXT: c_uint = 0x8DD8;
    pub const UNSIGNED_INT_SAMPLER_CUBE: c_uint = 0x8DD4;
    pub const UNSIGNED_INT_SAMPLER_CUBE_EXT: c_uint = 0x8DD4;
    pub const UNSIGNED_INT_SAMPLER_CUBE_MAP_ARRAY: c_uint = 0x900F;
    pub const UNSIGNED_INT_SAMPLER_CUBE_MAP_ARRAY_ARB: c_uint = 0x900F;
    pub const UNSIGNED_INT_SAMPLER_RENDERBUFFER_NV: c_uint = 0x8E58;
    pub const UNSIGNED_INT_VEC2: c_uint = 0x8DC6;
    pub const UNSIGNED_INT_VEC2_EXT: c_uint = 0x8DC6;
    pub const UNSIGNED_INT_VEC3: c_uint = 0x8DC7;
    pub const UNSIGNED_INT_VEC3_EXT: c_uint = 0x8DC7;
    pub const UNSIGNED_INT_VEC4: c_uint = 0x8DC8;
    pub const UNSIGNED_INT_VEC4_EXT: c_uint = 0x8DC8;
    pub const UNSIGNED_INVERT_NV: c_uint = 0x8537;
    pub const UNSIGNED_NORMALIZED: c_uint = 0x8C17;
    pub const UNSIGNED_NORMALIZED_ARB: c_uint = 0x8C17;
    pub const UNSIGNED_SHORT: c_uint = 0x1403;
    pub const UNSIGNED_SHORT_1_5_5_5_REV: c_uint = 0x8366;
    pub const UNSIGNED_SHORT_4_4_4_4: c_uint = 0x8033;
    pub const UNSIGNED_SHORT_4_4_4_4_EXT: c_uint = 0x8033;
    pub const UNSIGNED_SHORT_4_4_4_4_REV: c_uint = 0x8365;
    pub const UNSIGNED_SHORT_5_5_5_1: c_uint = 0x8034;
    pub const UNSIGNED_SHORT_5_5_5_1_EXT: c_uint = 0x8034;
    pub const UNSIGNED_SHORT_5_6_5: c_uint = 0x8363;
    pub const UNSIGNED_SHORT_5_6_5_REV: c_uint = 0x8364;
    pub const UNSIGNED_SHORT_8_8_APPLE: c_uint = 0x85BA;
    pub const UNSIGNED_SHORT_8_8_MESA: c_uint = 0x85BA;
    pub const UNSIGNED_SHORT_8_8_REV_APPLE: c_uint = 0x85BB;
    pub const UNSIGNED_SHORT_8_8_REV_MESA: c_uint = 0x85BB;
    pub const UPLOAD_GPU_MASK_NVX: c_uint = 0x954A;
    pub const UPPER_LEFT: c_uint = 0x8CA2;
    pub const USE_MISSING_GLYPH_NV: c_uint = 0x90AA;
    pub const UTF16_NV: c_uint = 0x909B;
    pub const UTF8_NV: c_uint = 0x909A;
    pub const UUID_SIZE_EXT: c_uint = 16;
    pub const VALIDATE_STATUS: c_uint = 0x8B83;
    pub const VARIABLE_A_NV: c_uint = 0x8523;
    pub const VARIABLE_B_NV: c_uint = 0x8524;
    pub const VARIABLE_C_NV: c_uint = 0x8525;
    pub const VARIABLE_D_NV: c_uint = 0x8526;
    pub const VARIABLE_E_NV: c_uint = 0x8527;
    pub const VARIABLE_F_NV: c_uint = 0x8528;
    pub const VARIABLE_G_NV: c_uint = 0x8529;
    pub const VARIANT_ARRAY_EXT: c_uint = 0x87E8;
    pub const VARIANT_ARRAY_POINTER_EXT: c_uint = 0x87E9;
    pub const VARIANT_ARRAY_STRIDE_EXT: c_uint = 0x87E6;
    pub const VARIANT_ARRAY_TYPE_EXT: c_uint = 0x87E7;
    pub const VARIANT_DATATYPE_EXT: c_uint = 0x87E5;
    pub const VARIANT_EXT: c_uint = 0x87C1;
    pub const VARIANT_VALUE_EXT: c_uint = 0x87E4;
    pub const VBO_FREE_MEMORY_ATI: c_uint = 0x87FB;
    pub const VECTOR_EXT: c_uint = 0x87BF;
    pub const VENDOR: c_uint = 0x1F00;
    pub const VERSION: c_uint = 0x1F02;
    pub const VERTEX23_BIT_PGI: c_uint = 0x00000004;
    pub const VERTEX4_BIT_PGI: c_uint = 0x00000008;
    pub const VERTEX_ARRAY: c_uint = 0x8074;
    pub const VERTEX_ARRAY_ADDRESS_NV: c_uint = 0x8F21;
    pub const VERTEX_ARRAY_BINDING: c_uint = 0x85B5;
    pub const VERTEX_ARRAY_BINDING_APPLE: c_uint = 0x85B5;
    pub const VERTEX_ARRAY_BUFFER_BINDING_ARB: c_uint = 0x8896;
    pub const VERTEX_ARRAY_COUNT_EXT: c_uint = 0x807D;
    pub const VERTEX_ARRAY_EXT: c_uint = 0x8074;
    pub const VERTEX_ARRAY_LENGTH_NV: c_uint = 0x8F2B;
    pub const VERTEX_ARRAY_LIST_IBM: c_uint = 103070;
    pub const VERTEX_ARRAY_LIST_STRIDE_IBM: c_uint = 103080;
    pub const VERTEX_ARRAY_OBJECT_AMD: c_uint = 0x9154;
    pub const VERTEX_ARRAY_OBJECT_EXT: c_uint = 0x9154;
    pub const VERTEX_ARRAY_PARALLEL_POINTERS_INTEL: c_uint = 0x83F5;
    pub const VERTEX_ARRAY_POINTER_EXT: c_uint = 0x808E;
    pub const VERTEX_ARRAY_RANGE_APPLE: c_uint = 0x851D;
    pub const VERTEX_ARRAY_RANGE_LENGTH_APPLE: c_uint = 0x851E;
    pub const VERTEX_ARRAY_RANGE_LENGTH_NV: c_uint = 0x851E;
    pub const VERTEX_ARRAY_RANGE_NV: c_uint = 0x851D;
    pub const VERTEX_ARRAY_RANGE_POINTER_APPLE: c_uint = 0x8521;
    pub const VERTEX_ARRAY_RANGE_POINTER_NV: c_uint = 0x8521;
    pub const VERTEX_ARRAY_RANGE_VALID_NV: c_uint = 0x851F;
    pub const VERTEX_ARRAY_RANGE_WITHOUT_FLUSH_NV: c_uint = 0x8533;
    pub const VERTEX_ARRAY_SIZE_EXT: c_uint = 0x807A;
    pub const VERTEX_ARRAY_STORAGE_HINT_APPLE: c_uint = 0x851F;
    pub const VERTEX_ARRAY_STRIDE_EXT: c_uint = 0x807C;
    pub const VERTEX_ARRAY_TYPE_EXT: c_uint = 0x807B;
    pub const VERTEX_ATTRIB_ARRAY0_NV: c_uint = 0x8650;
    pub const VERTEX_ATTRIB_ARRAY10_NV: c_uint = 0x865A;
    pub const VERTEX_ATTRIB_ARRAY11_NV: c_uint = 0x865B;
    pub const VERTEX_ATTRIB_ARRAY12_NV: c_uint = 0x865C;
    pub const VERTEX_ATTRIB_ARRAY13_NV: c_uint = 0x865D;
    pub const VERTEX_ATTRIB_ARRAY14_NV: c_uint = 0x865E;
    pub const VERTEX_ATTRIB_ARRAY15_NV: c_uint = 0x865F;
    pub const VERTEX_ATTRIB_ARRAY1_NV: c_uint = 0x8651;
    pub const VERTEX_ATTRIB_ARRAY2_NV: c_uint = 0x8652;
    pub const VERTEX_ATTRIB_ARRAY3_NV: c_uint = 0x8653;
    pub const VERTEX_ATTRIB_ARRAY4_NV: c_uint = 0x8654;
    pub const VERTEX_ATTRIB_ARRAY5_NV: c_uint = 0x8655;
    pub const VERTEX_ATTRIB_ARRAY6_NV: c_uint = 0x8656;
    pub const VERTEX_ATTRIB_ARRAY7_NV: c_uint = 0x8657;
    pub const VERTEX_ATTRIB_ARRAY8_NV: c_uint = 0x8658;
    pub const VERTEX_ATTRIB_ARRAY9_NV: c_uint = 0x8659;
    pub const VERTEX_ATTRIB_ARRAY_ADDRESS_NV: c_uint = 0x8F20;
    pub const VERTEX_ATTRIB_ARRAY_BARRIER_BIT: c_uint = 0x00000001;
    pub const VERTEX_ATTRIB_ARRAY_BARRIER_BIT_EXT: c_uint = 0x00000001;
    pub const VERTEX_ATTRIB_ARRAY_BUFFER_BINDING: c_uint = 0x889F;
    pub const VERTEX_ATTRIB_ARRAY_BUFFER_BINDING_ARB: c_uint = 0x889F;
    pub const VERTEX_ATTRIB_ARRAY_DIVISOR: c_uint = 0x88FE;
    pub const VERTEX_ATTRIB_ARRAY_DIVISOR_ARB: c_uint = 0x88FE;
    pub const VERTEX_ATTRIB_ARRAY_ENABLED: c_uint = 0x8622;
    pub const VERTEX_ATTRIB_ARRAY_ENABLED_ARB: c_uint = 0x8622;
    pub const VERTEX_ATTRIB_ARRAY_INTEGER: c_uint = 0x88FD;
    pub const VERTEX_ATTRIB_ARRAY_INTEGER_EXT: c_uint = 0x88FD;
    pub const VERTEX_ATTRIB_ARRAY_INTEGER_NV: c_uint = 0x88FD;
    pub const VERTEX_ATTRIB_ARRAY_LENGTH_NV: c_uint = 0x8F2A;
    pub const VERTEX_ATTRIB_ARRAY_LONG: c_uint = 0x874E;
    pub const VERTEX_ATTRIB_ARRAY_NORMALIZED: c_uint = 0x886A;
    pub const VERTEX_ATTRIB_ARRAY_NORMALIZED_ARB: c_uint = 0x886A;
    pub const VERTEX_ATTRIB_ARRAY_POINTER: c_uint = 0x8645;
    pub const VERTEX_ATTRIB_ARRAY_POINTER_ARB: c_uint = 0x8645;
    pub const VERTEX_ATTRIB_ARRAY_SIZE: c_uint = 0x8623;
    pub const VERTEX_ATTRIB_ARRAY_SIZE_ARB: c_uint = 0x8623;
    pub const VERTEX_ATTRIB_ARRAY_STRIDE: c_uint = 0x8624;
    pub const VERTEX_ATTRIB_ARRAY_STRIDE_ARB: c_uint = 0x8624;
    pub const VERTEX_ATTRIB_ARRAY_TYPE: c_uint = 0x8625;
    pub const VERTEX_ATTRIB_ARRAY_TYPE_ARB: c_uint = 0x8625;
    pub const VERTEX_ATTRIB_ARRAY_UNIFIED_NV: c_uint = 0x8F1E;
    pub const VERTEX_ATTRIB_BINDING: c_uint = 0x82D4;
    pub const VERTEX_ATTRIB_MAP1_APPLE: c_uint = 0x8A00;
    pub const VERTEX_ATTRIB_MAP1_COEFF_APPLE: c_uint = 0x8A03;
    pub const VERTEX_ATTRIB_MAP1_DOMAIN_APPLE: c_uint = 0x8A05;
    pub const VERTEX_ATTRIB_MAP1_ORDER_APPLE: c_uint = 0x8A04;
    pub const VERTEX_ATTRIB_MAP1_SIZE_APPLE: c_uint = 0x8A02;
    pub const VERTEX_ATTRIB_MAP2_APPLE: c_uint = 0x8A01;
    pub const VERTEX_ATTRIB_MAP2_COEFF_APPLE: c_uint = 0x8A07;
    pub const VERTEX_ATTRIB_MAP2_DOMAIN_APPLE: c_uint = 0x8A09;
    pub const VERTEX_ATTRIB_MAP2_ORDER_APPLE: c_uint = 0x8A08;
    pub const VERTEX_ATTRIB_MAP2_SIZE_APPLE: c_uint = 0x8A06;
    pub const VERTEX_ATTRIB_RELATIVE_OFFSET: c_uint = 0x82D5;
    pub const VERTEX_BINDING_BUFFER: c_uint = 0x8F4F;
    pub const VERTEX_BINDING_DIVISOR: c_uint = 0x82D6;
    pub const VERTEX_BINDING_OFFSET: c_uint = 0x82D7;
    pub const VERTEX_BINDING_STRIDE: c_uint = 0x82D8;
    pub const VERTEX_BLEND_ARB: c_uint = 0x86A7;
    pub const VERTEX_CONSISTENT_HINT_PGI: c_uint = 0x1A22B;
    pub const VERTEX_DATA_HINT_PGI: c_uint = 0x1A22A;
    pub const VERTEX_ELEMENT_SWIZZLE_AMD: c_uint = 0x91A4;
    pub const VERTEX_ID_NV: c_uint = 0x8C7B;
    pub const VERTEX_ID_SWIZZLE_AMD: c_uint = 0x91A5;
    pub const VERTEX_PRECLIP_HINT_SGIX: c_uint = 0x83EF;
    pub const VERTEX_PRECLIP_SGIX: c_uint = 0x83EE;
    pub const VERTEX_PROGRAM_ARB: c_uint = 0x8620;
    pub const VERTEX_PROGRAM_BINDING_NV: c_uint = 0x864A;
    pub const VERTEX_PROGRAM_NV: c_uint = 0x8620;
    pub const VERTEX_PROGRAM_PARAMETER_BUFFER_NV: c_uint = 0x8DA2;
    pub const VERTEX_PROGRAM_POINT_SIZE: c_uint = 0x8642;
    pub const VERTEX_PROGRAM_POINT_SIZE_ARB: c_uint = 0x8642;
    pub const VERTEX_PROGRAM_POINT_SIZE_NV: c_uint = 0x8642;
    pub const VERTEX_PROGRAM_TWO_SIDE_ARB: c_uint = 0x8643;
    pub const VERTEX_PROGRAM_TWO_SIDE_NV: c_uint = 0x8643;
    pub const VERTEX_SHADER: c_uint = 0x8B31;
    pub const VERTEX_SHADER_ARB: c_uint = 0x8B31;
    pub const VERTEX_SHADER_BINDING_EXT: c_uint = 0x8781;
    pub const VERTEX_SHADER_BIT: c_uint = 0x00000001;
    pub const VERTEX_SHADER_EXT: c_uint = 0x8780;
    pub const VERTEX_SHADER_INSTRUCTIONS_EXT: c_uint = 0x87CF;
    pub const VERTEX_SHADER_INVARIANTS_EXT: c_uint = 0x87D1;
    pub const VERTEX_SHADER_INVOCATIONS: c_uint = 0x82F0;
    pub const VERTEX_SHADER_INVOCATIONS_ARB: c_uint = 0x82F0;
    pub const VERTEX_SHADER_LOCALS_EXT: c_uint = 0x87D3;
    pub const VERTEX_SHADER_LOCAL_CONSTANTS_EXT: c_uint = 0x87D2;
    pub const VERTEX_SHADER_OPTIMIZED_EXT: c_uint = 0x87D4;
    pub const VERTEX_SHADER_VARIANTS_EXT: c_uint = 0x87D0;
    pub const VERTEX_SOURCE_ATI: c_uint = 0x8774;
    pub const VERTEX_STATE_PROGRAM_NV: c_uint = 0x8621;
    pub const VERTEX_STREAM0_ATI: c_uint = 0x876C;
    pub const VERTEX_STREAM1_ATI: c_uint = 0x876D;
    pub const VERTEX_STREAM2_ATI: c_uint = 0x876E;
    pub const VERTEX_STREAM3_ATI: c_uint = 0x876F;
    pub const VERTEX_STREAM4_ATI: c_uint = 0x8770;
    pub const VERTEX_STREAM5_ATI: c_uint = 0x8771;
    pub const VERTEX_STREAM6_ATI: c_uint = 0x8772;
    pub const VERTEX_STREAM7_ATI: c_uint = 0x8773;
    pub const VERTEX_SUBROUTINE: c_uint = 0x92E8;
    pub const VERTEX_SUBROUTINE_UNIFORM: c_uint = 0x92EE;
    pub const VERTEX_TEXTURE: c_uint = 0x829B;
    pub const VERTEX_WEIGHTING_EXT: c_uint = 0x8509;
    pub const VERTEX_WEIGHT_ARRAY_EXT: c_uint = 0x850C;
    pub const VERTEX_WEIGHT_ARRAY_POINTER_EXT: c_uint = 0x8510;
    pub const VERTEX_WEIGHT_ARRAY_SIZE_EXT: c_uint = 0x850D;
    pub const VERTEX_WEIGHT_ARRAY_STRIDE_EXT: c_uint = 0x850F;
    pub const VERTEX_WEIGHT_ARRAY_TYPE_EXT: c_uint = 0x850E;
    pub const VERTICAL_LINE_TO_NV: c_uint = 0x08;
    pub const VERTICES_SUBMITTED: c_uint = 0x82EE;
    pub const VERTICES_SUBMITTED_ARB: c_uint = 0x82EE;
    pub const VIBRANCE_BIAS_NV: c_uint = 0x8719;
    pub const VIBRANCE_SCALE_NV: c_uint = 0x8713;
    pub const VIDEO_BUFFER_BINDING_NV: c_uint = 0x9021;
    pub const VIDEO_BUFFER_INTERNAL_FORMAT_NV: c_uint = 0x902D;
    pub const VIDEO_BUFFER_NV: c_uint = 0x9020;
    pub const VIDEO_BUFFER_PITCH_NV: c_uint = 0x9028;
    pub const VIDEO_CAPTURE_FIELD_LOWER_HEIGHT_NV: c_uint = 0x903B;
    pub const VIDEO_CAPTURE_FIELD_UPPER_HEIGHT_NV: c_uint = 0x903A;
    pub const VIDEO_CAPTURE_FRAME_HEIGHT_NV: c_uint = 0x9039;
    pub const VIDEO_CAPTURE_FRAME_WIDTH_NV: c_uint = 0x9038;
    pub const VIDEO_CAPTURE_SURFACE_ORIGIN_NV: c_uint = 0x903C;
    pub const VIDEO_CAPTURE_TO_422_SUPPORTED_NV: c_uint = 0x9026;
    pub const VIDEO_COLOR_CONVERSION_MATRIX_NV: c_uint = 0x9029;
    pub const VIDEO_COLOR_CONVERSION_MAX_NV: c_uint = 0x902A;
    pub const VIDEO_COLOR_CONVERSION_MIN_NV: c_uint = 0x902B;
    pub const VIDEO_COLOR_CONVERSION_OFFSET_NV: c_uint = 0x902C;
    pub const VIEWPORT: c_uint = 0x0BA2;
    pub const VIEWPORT_BOUNDS_RANGE: c_uint = 0x825D;
    pub const VIEWPORT_COMMAND_NV: c_uint = 0x0010;
    pub const VIEWPORT_INDEX_PROVOKING_VERTEX: c_uint = 0x825F;
    pub const VIEWPORT_POSITION_W_SCALE_NV: c_uint = 0x937C;
    pub const VIEWPORT_POSITION_W_SCALE_X_COEFF_NV: c_uint = 0x937D;
    pub const VIEWPORT_POSITION_W_SCALE_Y_COEFF_NV: c_uint = 0x937E;
    pub const VIEWPORT_SUBPIXEL_BITS: c_uint = 0x825C;
    pub const VIEWPORT_SWIZZLE_NEGATIVE_W_NV: c_uint = 0x9357;
    pub const VIEWPORT_SWIZZLE_NEGATIVE_X_NV: c_uint = 0x9351;
    pub const VIEWPORT_SWIZZLE_NEGATIVE_Y_NV: c_uint = 0x9353;
    pub const VIEWPORT_SWIZZLE_NEGATIVE_Z_NV: c_uint = 0x9355;
    pub const VIEWPORT_SWIZZLE_POSITIVE_W_NV: c_uint = 0x9356;
    pub const VIEWPORT_SWIZZLE_POSITIVE_X_NV: c_uint = 0x9350;
    pub const VIEWPORT_SWIZZLE_POSITIVE_Y_NV: c_uint = 0x9352;
    pub const VIEWPORT_SWIZZLE_POSITIVE_Z_NV: c_uint = 0x9354;
    pub const VIEWPORT_SWIZZLE_W_NV: c_uint = 0x935B;
    pub const VIEWPORT_SWIZZLE_X_NV: c_uint = 0x9358;
    pub const VIEWPORT_SWIZZLE_Y_NV: c_uint = 0x9359;
    pub const VIEWPORT_SWIZZLE_Z_NV: c_uint = 0x935A;
    pub const VIEW_CLASS_128_BITS: c_uint = 0x82C4;
    pub const VIEW_CLASS_16_BITS: c_uint = 0x82CA;
    pub const VIEW_CLASS_24_BITS: c_uint = 0x82C9;
    pub const VIEW_CLASS_32_BITS: c_uint = 0x82C8;
    pub const VIEW_CLASS_48_BITS: c_uint = 0x82C7;
    pub const VIEW_CLASS_64_BITS: c_uint = 0x82C6;
    pub const VIEW_CLASS_8_BITS: c_uint = 0x82CB;
    pub const VIEW_CLASS_96_BITS: c_uint = 0x82C5;
    pub const VIEW_CLASS_ASTC_10x10_RGBA: c_uint = 0x9393;
    pub const VIEW_CLASS_ASTC_10x5_RGBA: c_uint = 0x9390;
    pub const VIEW_CLASS_ASTC_10x6_RGBA: c_uint = 0x9391;
    pub const VIEW_CLASS_ASTC_10x8_RGBA: c_uint = 0x9392;
    pub const VIEW_CLASS_ASTC_12x10_RGBA: c_uint = 0x9394;
    pub const VIEW_CLASS_ASTC_12x12_RGBA: c_uint = 0x9395;
    pub const VIEW_CLASS_ASTC_4x4_RGBA: c_uint = 0x9388;
    pub const VIEW_CLASS_ASTC_5x4_RGBA: c_uint = 0x9389;
    pub const VIEW_CLASS_ASTC_5x5_RGBA: c_uint = 0x938A;
    pub const VIEW_CLASS_ASTC_6x5_RGBA: c_uint = 0x938B;
    pub const VIEW_CLASS_ASTC_6x6_RGBA: c_uint = 0x938C;
    pub const VIEW_CLASS_ASTC_8x5_RGBA: c_uint = 0x938D;
    pub const VIEW_CLASS_ASTC_8x6_RGBA: c_uint = 0x938E;
    pub const VIEW_CLASS_ASTC_8x8_RGBA: c_uint = 0x938F;
    pub const VIEW_CLASS_BPTC_FLOAT: c_uint = 0x82D3;
    pub const VIEW_CLASS_BPTC_UNORM: c_uint = 0x82D2;
    pub const VIEW_CLASS_EAC_R11: c_uint = 0x9383;
    pub const VIEW_CLASS_EAC_RG11: c_uint = 0x9384;
    pub const VIEW_CLASS_ETC2_EAC_RGBA: c_uint = 0x9387;
    pub const VIEW_CLASS_ETC2_RGB: c_uint = 0x9385;
    pub const VIEW_CLASS_ETC2_RGBA: c_uint = 0x9386;
    pub const VIEW_CLASS_RGTC1_RED: c_uint = 0x82D0;
    pub const VIEW_CLASS_RGTC2_RG: c_uint = 0x82D1;
    pub const VIEW_CLASS_S3TC_DXT1_RGB: c_uint = 0x82CC;
    pub const VIEW_CLASS_S3TC_DXT1_RGBA: c_uint = 0x82CD;
    pub const VIEW_CLASS_S3TC_DXT3_RGBA: c_uint = 0x82CE;
    pub const VIEW_CLASS_S3TC_DXT5_RGBA: c_uint = 0x82CF;
    pub const VIEW_COMPATIBILITY_CLASS: c_uint = 0x82B6;
    pub const VIRTUAL_PAGE_SIZE_INDEX_ARB: c_uint = 0x91A7;
    pub const VIRTUAL_PAGE_SIZE_X_AMD: c_uint = 0x9195;
    pub const VIRTUAL_PAGE_SIZE_X_ARB: c_uint = 0x9195;
    pub const VIRTUAL_PAGE_SIZE_Y_AMD: c_uint = 0x9196;
    pub const VIRTUAL_PAGE_SIZE_Y_ARB: c_uint = 0x9196;
    pub const VIRTUAL_PAGE_SIZE_Z_AMD: c_uint = 0x9197;
    pub const VIRTUAL_PAGE_SIZE_Z_ARB: c_uint = 0x9197;
    pub const VIVIDLIGHT_NV: c_uint = 0x92A6;
    pub const VOLATILE_APPLE: c_uint = 0x8A1A;
    pub const WAIT_FAILED: c_uint = 0x911D;
    pub const WARPS_PER_SM_NV: c_uint = 0x933A;
    pub const WARP_SIZE_NV: c_uint = 0x9339;
    pub const WEIGHTED_AVERAGE_ARB: c_uint = 0x9367;
    pub const WEIGHTED_AVERAGE_EXT: c_uint = 0x9367;
    pub const WEIGHT_ARRAY_ARB: c_uint = 0x86AD;
    pub const WEIGHT_ARRAY_BUFFER_BINDING_ARB: c_uint = 0x889E;
    pub const WEIGHT_ARRAY_POINTER_ARB: c_uint = 0x86AC;
    pub const WEIGHT_ARRAY_SIZE_ARB: c_uint = 0x86AB;
    pub const WEIGHT_ARRAY_STRIDE_ARB: c_uint = 0x86AA;
    pub const WEIGHT_ARRAY_TYPE_ARB: c_uint = 0x86A9;
    pub const WEIGHT_SUM_UNITY_ARB: c_uint = 0x86A6;
    pub const WIDE_LINE_HINT_PGI: c_uint = 0x1A222;
    pub const WINDOW_RECTANGLE_EXT: c_uint = 0x8F12;
    pub const WINDOW_RECTANGLE_MODE_EXT: c_uint = 0x8F13;
    pub const WRAP_BORDER_SUN: c_uint = 0x81D4;
    pub const WRITE_DISCARD_NV: c_uint = 0x88BE;
    pub const WRITE_ONLY: c_uint = 0x88B9;
    pub const WRITE_ONLY_ARB: c_uint = 0x88B9;
    pub const WRITE_PIXEL_DATA_RANGE_LENGTH_NV: c_uint = 0x887A;
    pub const WRITE_PIXEL_DATA_RANGE_NV: c_uint = 0x8878;
    pub const WRITE_PIXEL_DATA_RANGE_POINTER_NV: c_uint = 0x887C;
    pub const W_EXT: c_uint = 0x87D8;
    pub const XOR: c_uint = 0x1506;
    pub const XOR_NV: c_uint = 0x1506;
    pub const X_EXT: c_uint = 0x87D5;
    pub const YCBAYCR8A_4224_NV: c_uint = 0x9032;
    pub const YCBCR_422_APPLE: c_uint = 0x85B9;
    pub const YCBCR_MESA: c_uint = 0x8757;
    pub const YCBYCR8_422_NV: c_uint = 0x9031;
    pub const YCRCBA_SGIX: c_uint = 0x8319;
    pub const YCRCB_422_SGIX: c_uint = 0x81BB;
    pub const YCRCB_444_SGIX: c_uint = 0x81BC;
    pub const YCRCB_SGIX: c_uint = 0x8318;
    pub const Y_EXT: c_uint = 0x87D6;
    pub const Z4Y12Z4CB12Z4A12Z4Y12Z4CR12Z4A12_4224_NV: c_uint = 0x9036;
    pub const Z4Y12Z4CB12Z4CR12_444_NV: c_uint = 0x9037;
    pub const Z4Y12Z4CB12Z4Y12Z4CR12_422_NV: c_uint = 0x9035;
    pub const Z6Y10Z6CB10Z6A10Z6Y10Z6CR10Z6A10_4224_NV: c_uint = 0x9034;
    pub const Z6Y10Z6CB10Z6Y10Z6CR10_422_NV: c_uint = 0x9033;
    pub const ZERO: c_uint = 0;
    pub const ZERO_EXT: c_uint = 0x87DD;
    pub const ZERO_TO_ONE: c_uint = 0x935F;
    pub const Z_EXT: c_uint = 0x87D7;
}

pub mod functions {
    #![allow(non_snake_case, unused_variables, dead_code, unused_imports)]

    use std::mem::transmute;
    use std::os::raw::*;
    use super::*;
    use super::types::*;

    macro_rules! func {
        ($fun:ident, $ret:ty, $($name:ident: $typ:ty),*) => {
            #[inline] pub fn $fun($($name: $typ),*) -> $ret {
                unsafe {
                    transmute::<_, extern "system" fn($($typ),*) -> $ret>(storage::$fun.ptr)($($name),*)
                }
            }
        }
    }


     func!(AccumxOES, (), op: GLenum, value: GLfixed);
     func!(AcquireKeyedMutexWin32EXT, GLboolean, memory: GLuint, key: GLuint64, timeout: GLuint);
     func!(ActiveProgramEXT, (), program: GLuint);
     func!(ActiveShaderProgram, (), pipeline: GLuint, program: GLuint);
     func!(ActiveStencilFaceEXT, (), face: GLenum);
     func!(ActiveTexture, (), texture: GLenum);
     func!(ActiveTextureARB, (), texture: GLenum);
     func!(ActiveVaryingNV, (), program: GLuint, name: *const GLchar);
     func!(AlphaFragmentOp1ATI, (), op: GLenum, dst: GLuint, dstMod: GLuint, arg1: GLuint, arg1Rep: GLuint, arg1Mod: GLuint);
     func!(AlphaFragmentOp2ATI, (), op: GLenum, dst: GLuint, dstMod: GLuint, arg1: GLuint, arg1Rep: GLuint, arg1Mod: GLuint, arg2: GLuint, arg2Rep: GLuint, arg2Mod: GLuint);
     func!(AlphaFragmentOp3ATI, (), op: GLenum, dst: GLuint, dstMod: GLuint, arg1: GLuint, arg1Rep: GLuint, arg1Mod: GLuint, arg2: GLuint, arg2Rep: GLuint, arg2Mod: GLuint, arg3: GLuint, arg3Rep: GLuint, arg3Mod: GLuint);
     func!(AlphaFuncxOES, (), func: GLenum, ref_: GLfixed);
     func!(AlphaToCoverageDitherControlNV, (), mode: GLenum);
     func!(ApplyFramebufferAttachmentCMAAINTEL, (), );
     func!(ApplyTextureEXT, (), mode: GLenum);
     func!(AreProgramsResidentNV, GLboolean, n: GLsizei, programs: *const GLuint, residences: *mut GLboolean);
     func!(AreTexturesResidentEXT, GLboolean, n: GLsizei, textures: *const GLuint, residences: *mut GLboolean);
     func!(ArrayElementEXT, (), i: GLint);
     func!(ArrayObjectATI, (), array: GLenum, size: GLint, type_: GLenum, stride: GLsizei, buffer: GLuint, offset: GLuint);
     func!(AsyncCopyBufferSubDataNVX, GLuint, waitSemaphoreCount: GLsizei, waitSemaphoreArray: *const GLuint, fenceValueArray: *const GLuint64, readGpu: GLuint, writeGpuMask: GLbitfield, readBuffer: GLuint, writeBuffer: GLuint, readOffset: GLintptr, writeOffset: GLintptr, size: GLsizeiptr, signalSemaphoreCount: GLsizei, signalSemaphoreArray: *const GLuint, signalValueArray: *const GLuint64);
     func!(AsyncCopyImageSubDataNVX, GLuint, waitSemaphoreCount: GLsizei, waitSemaphoreArray: *const GLuint, waitValueArray: *const GLuint64, srcGpu: GLuint, dstGpuMask: GLbitfield, srcName: GLuint, srcTarget: GLenum, srcLevel: GLint, srcX: GLint, srcY: GLint, srcZ: GLint, dstName: GLuint, dstTarget: GLenum, dstLevel: GLint, dstX: GLint, dstY: GLint, dstZ: GLint, srcWidth: GLsizei, srcHeight: GLsizei, srcDepth: GLsizei, signalSemaphoreCount: GLsizei, signalSemaphoreArray: *const GLuint, signalValueArray: *const GLuint64);
     func!(AsyncMarkerSGIX, (), marker: GLuint);
     func!(AttachObjectARB, (), containerObj: GLhandleARB, obj: GLhandleARB);
     func!(AttachShader, (), program: GLuint, shader: GLuint);
     func!(BeginConditionalRender, (), id: GLuint, mode: GLenum);
     func!(BeginConditionalRenderNV, (), id: GLuint, mode: GLenum);
     func!(BeginConditionalRenderNVX, (), id: GLuint);
     func!(BeginFragmentShaderATI, (), );
     func!(BeginOcclusionQueryNV, (), id: GLuint);
     func!(BeginPerfMonitorAMD, (), monitor: GLuint);
     func!(BeginPerfQueryINTEL, (), queryHandle: GLuint);
     func!(BeginQuery, (), target: GLenum, id: GLuint);
     func!(BeginQueryARB, (), target: GLenum, id: GLuint);
     func!(BeginQueryIndexed, (), target: GLenum, index: GLuint, id: GLuint);
     func!(BeginTransformFeedback, (), primitiveMode: GLenum);
     func!(BeginTransformFeedbackEXT, (), primitiveMode: GLenum);
     func!(BeginTransformFeedbackNV, (), primitiveMode: GLenum);
     func!(BeginVertexShaderEXT, (), );
     func!(BeginVideoCaptureNV, (), video_capture_slot: GLuint);
     func!(BindAttribLocation, (), program: GLuint, index: GLuint, name: *const GLchar);
     func!(BindAttribLocationARB, (), programObj: GLhandleARB, index: GLuint, name: *const GLcharARB);
     func!(BindBuffer, (), target: GLenum, buffer: GLuint);
     func!(BindBufferARB, (), target: GLenum, buffer: GLuint);
     func!(BindBufferBase, (), target: GLenum, index: GLuint, buffer: GLuint);
     func!(BindBufferBaseEXT, (), target: GLenum, index: GLuint, buffer: GLuint);
     func!(BindBufferBaseNV, (), target: GLenum, index: GLuint, buffer: GLuint);
     func!(BindBufferOffsetEXT, (), target: GLenum, index: GLuint, buffer: GLuint, offset: GLintptr);
     func!(BindBufferOffsetNV, (), target: GLenum, index: GLuint, buffer: GLuint, offset: GLintptr);
     func!(BindBufferRange, (), target: GLenum, index: GLuint, buffer: GLuint, offset: GLintptr, size: GLsizeiptr);
     func!(BindBufferRangeEXT, (), target: GLenum, index: GLuint, buffer: GLuint, offset: GLintptr, size: GLsizeiptr);
     func!(BindBufferRangeNV, (), target: GLenum, index: GLuint, buffer: GLuint, offset: GLintptr, size: GLsizeiptr);
     func!(BindBuffersBase, (), target: GLenum, first: GLuint, count: GLsizei, buffers: *const GLuint);
     func!(BindBuffersRange, (), target: GLenum, first: GLuint, count: GLsizei, buffers: *const GLuint, offsets: *const GLintptr, sizes: *const GLsizeiptr);
     func!(BindFragDataLocation, (), program: GLuint, color: GLuint, name: *const GLchar);
     func!(BindFragDataLocationEXT, (), program: GLuint, color: GLuint, name: *const GLchar);
     func!(BindFragDataLocationIndexed, (), program: GLuint, colorNumber: GLuint, index: GLuint, name: *const GLchar);
     func!(BindFragmentShaderATI, (), id: GLuint);
     func!(BindFramebuffer, (), target: GLenum, framebuffer: GLuint);
     func!(BindFramebufferEXT, (), target: GLenum, framebuffer: GLuint);
     func!(BindImageTexture, (), unit: GLuint, texture: GLuint, level: GLint, layered: GLboolean, layer: GLint, access: GLenum, format: GLenum);
     func!(BindImageTextureEXT, (), index: GLuint, texture: GLuint, level: GLint, layered: GLboolean, layer: GLint, access: GLenum, format: GLint);
     func!(BindImageTextures, (), first: GLuint, count: GLsizei, textures: *const GLuint);
     func!(BindLightParameterEXT, GLuint, light: GLenum, value: GLenum);
     func!(BindMaterialParameterEXT, GLuint, face: GLenum, value: GLenum);
     func!(BindMultiTextureEXT, (), texunit: GLenum, target: GLenum, texture: GLuint);
     func!(BindParameterEXT, GLuint, value: GLenum);
     func!(BindProgramARB, (), target: GLenum, program: GLuint);
     func!(BindProgramNV, (), target: GLenum, id: GLuint);
     func!(BindProgramPipeline, (), pipeline: GLuint);
     func!(BindRenderbuffer, (), target: GLenum, renderbuffer: GLuint);
     func!(BindRenderbufferEXT, (), target: GLenum, renderbuffer: GLuint);
     func!(BindSampler, (), unit: GLuint, sampler: GLuint);
     func!(BindSamplers, (), first: GLuint, count: GLsizei, samplers: *const GLuint);
     func!(BindShadingRateImageNV, (), texture: GLuint);
     func!(BindTexGenParameterEXT, GLuint, unit: GLenum, coord: GLenum, value: GLenum);
     func!(BindTexture, (), target: GLenum, texture: GLuint);
     func!(BindTextureEXT, (), target: GLenum, texture: GLuint);
     func!(BindTextureUnit, (), unit: GLuint, texture: GLuint);
     func!(BindTextureUnitParameterEXT, GLuint, unit: GLenum, value: GLenum);
     func!(BindTextures, (), first: GLuint, count: GLsizei, textures: *const GLuint);
     func!(BindTransformFeedback, (), target: GLenum, id: GLuint);
     func!(BindTransformFeedbackNV, (), target: GLenum, id: GLuint);
     func!(BindVertexArray, (), array: GLuint);
     func!(BindVertexArrayAPPLE, (), array: GLuint);
     func!(BindVertexBuffer, (), bindingindex: GLuint, buffer: GLuint, offset: GLintptr, stride: GLsizei);
     func!(BindVertexBuffers, (), first: GLuint, count: GLsizei, buffers: *const GLuint, offsets: *const GLintptr, strides: *const GLsizei);
     func!(BindVertexShaderEXT, (), id: GLuint);
     func!(BindVideoCaptureStreamBufferNV, (), video_capture_slot: GLuint, stream: GLuint, frame_region: GLenum, offset: GLintptrARB);
     func!(BindVideoCaptureStreamTextureNV, (), video_capture_slot: GLuint, stream: GLuint, frame_region: GLenum, target: GLenum, texture: GLuint);
     func!(Binormal3bEXT, (), bx: GLbyte, by: GLbyte, bz: GLbyte);
     func!(Binormal3bvEXT, (), v: *const GLbyte);
     func!(Binormal3dEXT, (), bx: GLdouble, by: GLdouble, bz: GLdouble);
     func!(Binormal3dvEXT, (), v: *const GLdouble);
     func!(Binormal3fEXT, (), bx: GLfloat, by: GLfloat, bz: GLfloat);
     func!(Binormal3fvEXT, (), v: *const GLfloat);
     func!(Binormal3iEXT, (), bx: GLint, by: GLint, bz: GLint);
     func!(Binormal3ivEXT, (), v: *const GLint);
     func!(Binormal3sEXT, (), bx: GLshort, by: GLshort, bz: GLshort);
     func!(Binormal3svEXT, (), v: *const GLshort);
     func!(BinormalPointerEXT, (), type_: GLenum, stride: GLsizei, pointer: *const c_void);
     func!(BitmapxOES, (), width: GLsizei, height: GLsizei, xorig: GLfixed, yorig: GLfixed, xmove: GLfixed, ymove: GLfixed, bitmap: *const GLubyte);
     func!(BlendBarrierKHR, (), );
     func!(BlendBarrierNV, (), );
     func!(BlendColor, (), red: GLfloat, green: GLfloat, blue: GLfloat, alpha: GLfloat);
     func!(BlendColorEXT, (), red: GLfloat, green: GLfloat, blue: GLfloat, alpha: GLfloat);
     func!(BlendColorxOES, (), red: GLfixed, green: GLfixed, blue: GLfixed, alpha: GLfixed);
     func!(BlendEquation, (), mode: GLenum);
     func!(BlendEquationEXT, (), mode: GLenum);
     func!(BlendEquationIndexedAMD, (), buf: GLuint, mode: GLenum);
     func!(BlendEquationSeparate, (), modeRGB: GLenum, modeAlpha: GLenum);
     func!(BlendEquationSeparateEXT, (), modeRGB: GLenum, modeAlpha: GLenum);
     func!(BlendEquationSeparateIndexedAMD, (), buf: GLuint, modeRGB: GLenum, modeAlpha: GLenum);
     func!(BlendEquationSeparatei, (), buf: GLuint, modeRGB: GLenum, modeAlpha: GLenum);
     func!(BlendEquationSeparateiARB, (), buf: GLuint, modeRGB: GLenum, modeAlpha: GLenum);
     func!(BlendEquationi, (), buf: GLuint, mode: GLenum);
     func!(BlendEquationiARB, (), buf: GLuint, mode: GLenum);
     func!(BlendFunc, (), sfactor: GLenum, dfactor: GLenum);
     func!(BlendFuncIndexedAMD, (), buf: GLuint, src: GLenum, dst: GLenum);
     func!(BlendFuncSeparate, (), sfactorRGB: GLenum, dfactorRGB: GLenum, sfactorAlpha: GLenum, dfactorAlpha: GLenum);
     func!(BlendFuncSeparateEXT, (), sfactorRGB: GLenum, dfactorRGB: GLenum, sfactorAlpha: GLenum, dfactorAlpha: GLenum);
     func!(BlendFuncSeparateINGR, (), sfactorRGB: GLenum, dfactorRGB: GLenum, sfactorAlpha: GLenum, dfactorAlpha: GLenum);
     func!(BlendFuncSeparateIndexedAMD, (), buf: GLuint, srcRGB: GLenum, dstRGB: GLenum, srcAlpha: GLenum, dstAlpha: GLenum);
     func!(BlendFuncSeparatei, (), buf: GLuint, srcRGB: GLenum, dstRGB: GLenum, srcAlpha: GLenum, dstAlpha: GLenum);
     func!(BlendFuncSeparateiARB, (), buf: GLuint, srcRGB: GLenum, dstRGB: GLenum, srcAlpha: GLenum, dstAlpha: GLenum);
     func!(BlendFunci, (), buf: GLuint, src: GLenum, dst: GLenum);
     func!(BlendFunciARB, (), buf: GLuint, src: GLenum, dst: GLenum);
     func!(BlendParameteriNV, (), pname: GLenum, value: GLint);
     func!(BlitFramebuffer, (), srcX0: GLint, srcY0: GLint, srcX1: GLint, srcY1: GLint, dstX0: GLint, dstY0: GLint, dstX1: GLint, dstY1: GLint, mask: GLbitfield, filter: GLenum);
     func!(BlitFramebufferEXT, (), srcX0: GLint, srcY0: GLint, srcX1: GLint, srcY1: GLint, dstX0: GLint, dstY0: GLint, dstX1: GLint, dstY1: GLint, mask: GLbitfield, filter: GLenum);
     func!(BlitNamedFramebuffer, (), readFramebuffer: GLuint, drawFramebuffer: GLuint, srcX0: GLint, srcY0: GLint, srcX1: GLint, srcY1: GLint, dstX0: GLint, dstY0: GLint, dstX1: GLint, dstY1: GLint, mask: GLbitfield, filter: GLenum);
     func!(BufferAddressRangeNV, (), pname: GLenum, index: GLuint, address: GLuint64EXT, length: GLsizeiptr);
     func!(BufferAttachMemoryNV, (), target: GLenum, memory: GLuint, offset: GLuint64);
     func!(BufferData, (), target: GLenum, size: GLsizeiptr, data: *const c_void, usage: GLenum);
     func!(BufferDataARB, (), target: GLenum, size: GLsizeiptrARB, data: *const c_void, usage: GLenum);
     func!(BufferPageCommitmentARB, (), target: GLenum, offset: GLintptr, size: GLsizeiptr, commit: GLboolean);
     func!(BufferPageCommitmentMemNV, (), target: GLenum, offset: GLintptr, size: GLsizeiptr, memory: GLuint, memOffset: GLuint64, commit: GLboolean);
     func!(BufferParameteriAPPLE, (), target: GLenum, pname: GLenum, param: GLint);
     func!(BufferStorage, (), target: GLenum, size: GLsizeiptr, data: *const c_void, flags: GLbitfield);
     func!(BufferStorageExternalEXT, (), target: GLenum, offset: GLintptr, size: GLsizeiptr, clientBuffer: GLeglClientBufferEXT, flags: GLbitfield);
     func!(BufferStorageMemEXT, (), target: GLenum, size: GLsizeiptr, memory: GLuint, offset: GLuint64);
     func!(BufferSubData, (), target: GLenum, offset: GLintptr, size: GLsizeiptr, data: *const c_void);
     func!(BufferSubDataARB, (), target: GLenum, offset: GLintptrARB, size: GLsizeiptrARB, data: *const c_void);
     func!(CallCommandListNV, (), list: GLuint);
     func!(CheckFramebufferStatus, GLenum, target: GLenum);
     func!(CheckFramebufferStatusEXT, GLenum, target: GLenum);
     func!(CheckNamedFramebufferStatus, GLenum, framebuffer: GLuint, target: GLenum);
     func!(CheckNamedFramebufferStatusEXT, GLenum, framebuffer: GLuint, target: GLenum);
     func!(ClampColor, (), target: GLenum, clamp: GLenum);
     func!(ClampColorARB, (), target: GLenum, clamp: GLenum);
     func!(Clear, (), mask: GLbitfield);
     func!(ClearAccumxOES, (), red: GLfixed, green: GLfixed, blue: GLfixed, alpha: GLfixed);
     func!(ClearBufferData, (), target: GLenum, internalformat: GLenum, format: GLenum, type_: GLenum, data: *const c_void);
     func!(ClearBufferSubData, (), target: GLenum, internalformat: GLenum, offset: GLintptr, size: GLsizeiptr, format: GLenum, type_: GLenum, data: *const c_void);
     func!(ClearBufferfi, (), buffer: GLenum, drawbuffer: GLint, depth: GLfloat, stencil: GLint);
     func!(ClearBufferfv, (), buffer: GLenum, drawbuffer: GLint, value: *const GLfloat);
     func!(ClearBufferiv, (), buffer: GLenum, drawbuffer: GLint, value: *const GLint);
     func!(ClearBufferuiv, (), buffer: GLenum, drawbuffer: GLint, value: *const GLuint);
     func!(ClearColor, (), red: GLfloat, green: GLfloat, blue: GLfloat, alpha: GLfloat);
     func!(ClearColorIiEXT, (), red: GLint, green: GLint, blue: GLint, alpha: GLint);
     func!(ClearColorIuiEXT, (), red: GLuint, green: GLuint, blue: GLuint, alpha: GLuint);
     func!(ClearColorxOES, (), red: GLfixed, green: GLfixed, blue: GLfixed, alpha: GLfixed);
     func!(ClearDepth, (), depth: GLdouble);
     func!(ClearDepthdNV, (), depth: GLdouble);
     func!(ClearDepthf, (), d: GLfloat);
     func!(ClearDepthfOES, (), depth: GLclampf);
     func!(ClearDepthxOES, (), depth: GLfixed);
     func!(ClearNamedBufferData, (), buffer: GLuint, internalformat: GLenum, format: GLenum, type_: GLenum, data: *const c_void);
     func!(ClearNamedBufferDataEXT, (), buffer: GLuint, internalformat: GLenum, format: GLenum, type_: GLenum, data: *const c_void);
     func!(ClearNamedBufferSubData, (), buffer: GLuint, internalformat: GLenum, offset: GLintptr, size: GLsizeiptr, format: GLenum, type_: GLenum, data: *const c_void);
     func!(ClearNamedBufferSubDataEXT, (), buffer: GLuint, internalformat: GLenum, offset: GLsizeiptr, size: GLsizeiptr, format: GLenum, type_: GLenum, data: *const c_void);
     func!(ClearNamedFramebufferfi, (), framebuffer: GLuint, buffer: GLenum, drawbuffer: GLint, depth: GLfloat, stencil: GLint);
     func!(ClearNamedFramebufferfv, (), framebuffer: GLuint, buffer: GLenum, drawbuffer: GLint, value: *const GLfloat);
     func!(ClearNamedFramebufferiv, (), framebuffer: GLuint, buffer: GLenum, drawbuffer: GLint, value: *const GLint);
     func!(ClearNamedFramebufferuiv, (), framebuffer: GLuint, buffer: GLenum, drawbuffer: GLint, value: *const GLuint);
     func!(ClearStencil, (), s: GLint);
     func!(ClearTexImage, (), texture: GLuint, level: GLint, format: GLenum, type_: GLenum, data: *const c_void);
     func!(ClearTexSubImage, (), texture: GLuint, level: GLint, xoffset: GLint, yoffset: GLint, zoffset: GLint, width: GLsizei, height: GLsizei, depth: GLsizei, format: GLenum, type_: GLenum, data: *const c_void);
     func!(ClientActiveTextureARB, (), texture: GLenum);
     func!(ClientActiveVertexStreamATI, (), stream: GLenum);
     func!(ClientAttribDefaultEXT, (), mask: GLbitfield);
     func!(ClientWaitSemaphoreui64NVX, (), fenceObjectCount: GLsizei, semaphoreArray: *const GLuint, fenceValueArray: *const GLuint64);
     func!(ClientWaitSync, GLenum, sync: GLsync, flags: GLbitfield, timeout: GLuint64);
     func!(ClipControl, (), origin: GLenum, depth: GLenum);
     func!(ClipPlanefOES, (), plane: GLenum, equation: *const GLfloat);
     func!(ClipPlanexOES, (), plane: GLenum, equation: *const GLfixed);
     func!(Color3fVertex3fSUN, (), r: GLfloat, g: GLfloat, b: GLfloat, x: GLfloat, y: GLfloat, z: GLfloat);
     func!(Color3fVertex3fvSUN, (), c: *const GLfloat, v: *const GLfloat);
     func!(Color3hNV, (), red: GLhalfNV, green: GLhalfNV, blue: GLhalfNV);
     func!(Color3hvNV, (), v: *const GLhalfNV);
     func!(Color3xOES, (), red: GLfixed, green: GLfixed, blue: GLfixed);
     func!(Color3xvOES, (), components: *const GLfixed);
     func!(Color4fNormal3fVertex3fSUN, (), r: GLfloat, g: GLfloat, b: GLfloat, a: GLfloat, nx: GLfloat, ny: GLfloat, nz: GLfloat, x: GLfloat, y: GLfloat, z: GLfloat);
     func!(Color4fNormal3fVertex3fvSUN, (), c: *const GLfloat, n: *const GLfloat, v: *const GLfloat);
     func!(Color4hNV, (), red: GLhalfNV, green: GLhalfNV, blue: GLhalfNV, alpha: GLhalfNV);
     func!(Color4hvNV, (), v: *const GLhalfNV);
     func!(Color4ubVertex2fSUN, (), r: GLubyte, g: GLubyte, b: GLubyte, a: GLubyte, x: GLfloat, y: GLfloat);
     func!(Color4ubVertex2fvSUN, (), c: *const GLubyte, v: *const GLfloat);
     func!(Color4ubVertex3fSUN, (), r: GLubyte, g: GLubyte, b: GLubyte, a: GLubyte, x: GLfloat, y: GLfloat, z: GLfloat);
     func!(Color4ubVertex3fvSUN, (), c: *const GLubyte, v: *const GLfloat);
     func!(Color4xOES, (), red: GLfixed, green: GLfixed, blue: GLfixed, alpha: GLfixed);
     func!(Color4xvOES, (), components: *const GLfixed);
     func!(ColorFormatNV, (), size: GLint, type_: GLenum, stride: GLsizei);
     func!(ColorFragmentOp1ATI, (), op: GLenum, dst: GLuint, dstMask: GLuint, dstMod: GLuint, arg1: GLuint, arg1Rep: GLuint, arg1Mod: GLuint);
     func!(ColorFragmentOp2ATI, (), op: GLenum, dst: GLuint, dstMask: GLuint, dstMod: GLuint, arg1: GLuint, arg1Rep: GLuint, arg1Mod: GLuint, arg2: GLuint, arg2Rep: GLuint, arg2Mod: GLuint);
     func!(ColorFragmentOp3ATI, (), op: GLenum, dst: GLuint, dstMask: GLuint, dstMod: GLuint, arg1: GLuint, arg1Rep: GLuint, arg1Mod: GLuint, arg2: GLuint, arg2Rep: GLuint, arg2Mod: GLuint, arg3: GLuint, arg3Rep: GLuint, arg3Mod: GLuint);
     func!(ColorMask, (), red: GLboolean, green: GLboolean, blue: GLboolean, alpha: GLboolean);
     func!(ColorMaskIndexedEXT, (), index: GLuint, r: GLboolean, g: GLboolean, b: GLboolean, a: GLboolean);
     func!(ColorMaski, (), index: GLuint, r: GLboolean, g: GLboolean, b: GLboolean, a: GLboolean);
     func!(ColorPointerEXT, (), size: GLint, type_: GLenum, stride: GLsizei, count: GLsizei, pointer: *const c_void);
     func!(ColorPointerListIBM, (), size: GLint, type_: GLenum, stride: GLint, pointer: *const *const c_void, ptrstride: GLint);
     func!(ColorPointervINTEL, (), size: GLint, type_: GLenum, pointer: *const *const c_void);
     func!(ColorSubTableEXT, (), target: GLenum, start: GLsizei, count: GLsizei, format: GLenum, type_: GLenum, data: *const c_void);
     func!(ColorTableEXT, (), target: GLenum, internalFormat: GLenum, width: GLsizei, format: GLenum, type_: GLenum, table: *const c_void);
     func!(ColorTableParameterfvSGI, (), target: GLenum, pname: GLenum, params: *const GLfloat);
     func!(ColorTableParameterivSGI, (), target: GLenum, pname: GLenum, params: *const GLint);
     func!(ColorTableSGI, (), target: GLenum, internalformat: GLenum, width: GLsizei, format: GLenum, type_: GLenum, table: *const c_void);
     func!(CombinerInputNV, (), stage: GLenum, portion: GLenum, variable: GLenum, input: GLenum, mapping: GLenum, componentUsage: GLenum);
     func!(CombinerOutputNV, (), stage: GLenum, portion: GLenum, abOutput: GLenum, cdOutput: GLenum, sumOutput: GLenum, scale: GLenum, bias: GLenum, abDotProduct: GLboolean, cdDotProduct: GLboolean, muxSum: GLboolean);
     func!(CombinerParameterfNV, (), pname: GLenum, param: GLfloat);
     func!(CombinerParameterfvNV, (), pname: GLenum, params: *const GLfloat);
     func!(CombinerParameteriNV, (), pname: GLenum, param: GLint);
     func!(CombinerParameterivNV, (), pname: GLenum, params: *const GLint);
     func!(CombinerStageParameterfvNV, (), stage: GLenum, pname: GLenum, params: *const GLfloat);
     func!(CommandListSegmentsNV, (), list: GLuint, segments: GLuint);
     func!(CompileCommandListNV, (), list: GLuint);
     func!(CompileShader, (), shader: GLuint);
     func!(CompileShaderARB, (), shaderObj: GLhandleARB);
     func!(CompileShaderIncludeARB, (), shader: GLuint, count: GLsizei, path: *const *const GLchar, length: *const GLint);
     func!(CompressedMultiTexImage1DEXT, (), texunit: GLenum, target: GLenum, level: GLint, internalformat: GLenum, width: GLsizei, border: GLint, imageSize: GLsizei, bits: *const c_void);
     func!(CompressedMultiTexImage2DEXT, (), texunit: GLenum, target: GLenum, level: GLint, internalformat: GLenum, width: GLsizei, height: GLsizei, border: GLint, imageSize: GLsizei, bits: *const c_void);
     func!(CompressedMultiTexImage3DEXT, (), texunit: GLenum, target: GLenum, level: GLint, internalformat: GLenum, width: GLsizei, height: GLsizei, depth: GLsizei, border: GLint, imageSize: GLsizei, bits: *const c_void);
     func!(CompressedMultiTexSubImage1DEXT, (), texunit: GLenum, target: GLenum, level: GLint, xoffset: GLint, width: GLsizei, format: GLenum, imageSize: GLsizei, bits: *const c_void);
     func!(CompressedMultiTexSubImage2DEXT, (), texunit: GLenum, target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, width: GLsizei, height: GLsizei, format: GLenum, imageSize: GLsizei, bits: *const c_void);
     func!(CompressedMultiTexSubImage3DEXT, (), texunit: GLenum, target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, zoffset: GLint, width: GLsizei, height: GLsizei, depth: GLsizei, format: GLenum, imageSize: GLsizei, bits: *const c_void);
     func!(CompressedTexImage1D, (), target: GLenum, level: GLint, internalformat: GLenum, width: GLsizei, border: GLint, imageSize: GLsizei, data: *const c_void);
     func!(CompressedTexImage1DARB, (), target: GLenum, level: GLint, internalformat: GLenum, width: GLsizei, border: GLint, imageSize: GLsizei, data: *const c_void);
     func!(CompressedTexImage2D, (), target: GLenum, level: GLint, internalformat: GLenum, width: GLsizei, height: GLsizei, border: GLint, imageSize: GLsizei, data: *const c_void);
     func!(CompressedTexImage2DARB, (), target: GLenum, level: GLint, internalformat: GLenum, width: GLsizei, height: GLsizei, border: GLint, imageSize: GLsizei, data: *const c_void);
     func!(CompressedTexImage3D, (), target: GLenum, level: GLint, internalformat: GLenum, width: GLsizei, height: GLsizei, depth: GLsizei, border: GLint, imageSize: GLsizei, data: *const c_void);
     func!(CompressedTexImage3DARB, (), target: GLenum, level: GLint, internalformat: GLenum, width: GLsizei, height: GLsizei, depth: GLsizei, border: GLint, imageSize: GLsizei, data: *const c_void);
     func!(CompressedTexSubImage1D, (), target: GLenum, level: GLint, xoffset: GLint, width: GLsizei, format: GLenum, imageSize: GLsizei, data: *const c_void);
     func!(CompressedTexSubImage1DARB, (), target: GLenum, level: GLint, xoffset: GLint, width: GLsizei, format: GLenum, imageSize: GLsizei, data: *const c_void);
     func!(CompressedTexSubImage2D, (), target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, width: GLsizei, height: GLsizei, format: GLenum, imageSize: GLsizei, data: *const c_void);
     func!(CompressedTexSubImage2DARB, (), target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, width: GLsizei, height: GLsizei, format: GLenum, imageSize: GLsizei, data: *const c_void);
     func!(CompressedTexSubImage3D, (), target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, zoffset: GLint, width: GLsizei, height: GLsizei, depth: GLsizei, format: GLenum, imageSize: GLsizei, data: *const c_void);
     func!(CompressedTexSubImage3DARB, (), target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, zoffset: GLint, width: GLsizei, height: GLsizei, depth: GLsizei, format: GLenum, imageSize: GLsizei, data: *const c_void);
     func!(CompressedTextureImage1DEXT, (), texture: GLuint, target: GLenum, level: GLint, internalformat: GLenum, width: GLsizei, border: GLint, imageSize: GLsizei, bits: *const c_void);
     func!(CompressedTextureImage2DEXT, (), texture: GLuint, target: GLenum, level: GLint, internalformat: GLenum, width: GLsizei, height: GLsizei, border: GLint, imageSize: GLsizei, bits: *const c_void);
     func!(CompressedTextureImage3DEXT, (), texture: GLuint, target: GLenum, level: GLint, internalformat: GLenum, width: GLsizei, height: GLsizei, depth: GLsizei, border: GLint, imageSize: GLsizei, bits: *const c_void);
     func!(CompressedTextureSubImage1D, (), texture: GLuint, level: GLint, xoffset: GLint, width: GLsizei, format: GLenum, imageSize: GLsizei, data: *const c_void);
     func!(CompressedTextureSubImage1DEXT, (), texture: GLuint, target: GLenum, level: GLint, xoffset: GLint, width: GLsizei, format: GLenum, imageSize: GLsizei, bits: *const c_void);
     func!(CompressedTextureSubImage2D, (), texture: GLuint, level: GLint, xoffset: GLint, yoffset: GLint, width: GLsizei, height: GLsizei, format: GLenum, imageSize: GLsizei, data: *const c_void);
     func!(CompressedTextureSubImage2DEXT, (), texture: GLuint, target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, width: GLsizei, height: GLsizei, format: GLenum, imageSize: GLsizei, bits: *const c_void);
     func!(CompressedTextureSubImage3D, (), texture: GLuint, level: GLint, xoffset: GLint, yoffset: GLint, zoffset: GLint, width: GLsizei, height: GLsizei, depth: GLsizei, format: GLenum, imageSize: GLsizei, data: *const c_void);
     func!(CompressedTextureSubImage3DEXT, (), texture: GLuint, target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, zoffset: GLint, width: GLsizei, height: GLsizei, depth: GLsizei, format: GLenum, imageSize: GLsizei, bits: *const c_void);
     func!(ConservativeRasterParameterfNV, (), pname: GLenum, value: GLfloat);
     func!(ConservativeRasterParameteriNV, (), pname: GLenum, param: GLint);
     func!(ConvolutionFilter1DEXT, (), target: GLenum, internalformat: GLenum, width: GLsizei, format: GLenum, type_: GLenum, image: *const c_void);
     func!(ConvolutionFilter2DEXT, (), target: GLenum, internalformat: GLenum, width: GLsizei, height: GLsizei, format: GLenum, type_: GLenum, image: *const c_void);
     func!(ConvolutionParameterfEXT, (), target: GLenum, pname: GLenum, params: GLfloat);
     func!(ConvolutionParameterfvEXT, (), target: GLenum, pname: GLenum, params: *const GLfloat);
     func!(ConvolutionParameteriEXT, (), target: GLenum, pname: GLenum, params: GLint);
     func!(ConvolutionParameterivEXT, (), target: GLenum, pname: GLenum, params: *const GLint);
     func!(ConvolutionParameterxOES, (), target: GLenum, pname: GLenum, param: GLfixed);
     func!(ConvolutionParameterxvOES, (), target: GLenum, pname: GLenum, params: *const GLfixed);
     func!(CopyBufferSubData, (), readTarget: GLenum, writeTarget: GLenum, readOffset: GLintptr, writeOffset: GLintptr, size: GLsizeiptr);
     func!(CopyColorSubTableEXT, (), target: GLenum, start: GLsizei, x: GLint, y: GLint, width: GLsizei);
     func!(CopyColorTableSGI, (), target: GLenum, internalformat: GLenum, x: GLint, y: GLint, width: GLsizei);
     func!(CopyConvolutionFilter1DEXT, (), target: GLenum, internalformat: GLenum, x: GLint, y: GLint, width: GLsizei);
     func!(CopyConvolutionFilter2DEXT, (), target: GLenum, internalformat: GLenum, x: GLint, y: GLint, width: GLsizei, height: GLsizei);
     func!(CopyImageSubData, (), srcName: GLuint, srcTarget: GLenum, srcLevel: GLint, srcX: GLint, srcY: GLint, srcZ: GLint, dstName: GLuint, dstTarget: GLenum, dstLevel: GLint, dstX: GLint, dstY: GLint, dstZ: GLint, srcWidth: GLsizei, srcHeight: GLsizei, srcDepth: GLsizei);
     func!(CopyImageSubDataNV, (), srcName: GLuint, srcTarget: GLenum, srcLevel: GLint, srcX: GLint, srcY: GLint, srcZ: GLint, dstName: GLuint, dstTarget: GLenum, dstLevel: GLint, dstX: GLint, dstY: GLint, dstZ: GLint, width: GLsizei, height: GLsizei, depth: GLsizei);
     func!(CopyMultiTexImage1DEXT, (), texunit: GLenum, target: GLenum, level: GLint, internalformat: GLenum, x: GLint, y: GLint, width: GLsizei, border: GLint);
     func!(CopyMultiTexImage2DEXT, (), texunit: GLenum, target: GLenum, level: GLint, internalformat: GLenum, x: GLint, y: GLint, width: GLsizei, height: GLsizei, border: GLint);
     func!(CopyMultiTexSubImage1DEXT, (), texunit: GLenum, target: GLenum, level: GLint, xoffset: GLint, x: GLint, y: GLint, width: GLsizei);
     func!(CopyMultiTexSubImage2DEXT, (), texunit: GLenum, target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, x: GLint, y: GLint, width: GLsizei, height: GLsizei);
     func!(CopyMultiTexSubImage3DEXT, (), texunit: GLenum, target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, zoffset: GLint, x: GLint, y: GLint, width: GLsizei, height: GLsizei);
     func!(CopyNamedBufferSubData, (), readBuffer: GLuint, writeBuffer: GLuint, readOffset: GLintptr, writeOffset: GLintptr, size: GLsizeiptr);
     func!(CopyPathNV, (), resultPath: GLuint, srcPath: GLuint);
     func!(CopyTexImage1D, (), target: GLenum, level: GLint, internalformat: GLenum, x: GLint, y: GLint, width: GLsizei, border: GLint);
     func!(CopyTexImage1DEXT, (), target: GLenum, level: GLint, internalformat: GLenum, x: GLint, y: GLint, width: GLsizei, border: GLint);
     func!(CopyTexImage2D, (), target: GLenum, level: GLint, internalformat: GLenum, x: GLint, y: GLint, width: GLsizei, height: GLsizei, border: GLint);
     func!(CopyTexImage2DEXT, (), target: GLenum, level: GLint, internalformat: GLenum, x: GLint, y: GLint, width: GLsizei, height: GLsizei, border: GLint);
     func!(CopyTexSubImage1D, (), target: GLenum, level: GLint, xoffset: GLint, x: GLint, y: GLint, width: GLsizei);
     func!(CopyTexSubImage1DEXT, (), target: GLenum, level: GLint, xoffset: GLint, x: GLint, y: GLint, width: GLsizei);
     func!(CopyTexSubImage2D, (), target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, x: GLint, y: GLint, width: GLsizei, height: GLsizei);
     func!(CopyTexSubImage2DEXT, (), target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, x: GLint, y: GLint, width: GLsizei, height: GLsizei);
     func!(CopyTexSubImage3D, (), target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, zoffset: GLint, x: GLint, y: GLint, width: GLsizei, height: GLsizei);
     func!(CopyTexSubImage3DEXT, (), target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, zoffset: GLint, x: GLint, y: GLint, width: GLsizei, height: GLsizei);
     func!(CopyTextureImage1DEXT, (), texture: GLuint, target: GLenum, level: GLint, internalformat: GLenum, x: GLint, y: GLint, width: GLsizei, border: GLint);
     func!(CopyTextureImage2DEXT, (), texture: GLuint, target: GLenum, level: GLint, internalformat: GLenum, x: GLint, y: GLint, width: GLsizei, height: GLsizei, border: GLint);
     func!(CopyTextureSubImage1D, (), texture: GLuint, level: GLint, xoffset: GLint, x: GLint, y: GLint, width: GLsizei);
     func!(CopyTextureSubImage1DEXT, (), texture: GLuint, target: GLenum, level: GLint, xoffset: GLint, x: GLint, y: GLint, width: GLsizei);
     func!(CopyTextureSubImage2D, (), texture: GLuint, level: GLint, xoffset: GLint, yoffset: GLint, x: GLint, y: GLint, width: GLsizei, height: GLsizei);
     func!(CopyTextureSubImage2DEXT, (), texture: GLuint, target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, x: GLint, y: GLint, width: GLsizei, height: GLsizei);
     func!(CopyTextureSubImage3D, (), texture: GLuint, level: GLint, xoffset: GLint, yoffset: GLint, zoffset: GLint, x: GLint, y: GLint, width: GLsizei, height: GLsizei);
     func!(CopyTextureSubImage3DEXT, (), texture: GLuint, target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, zoffset: GLint, x: GLint, y: GLint, width: GLsizei, height: GLsizei);
     func!(CoverFillPathInstancedNV, (), numPaths: GLsizei, pathNameType: GLenum, paths: *const c_void, pathBase: GLuint, coverMode: GLenum, transformType: GLenum, transformValues: *const GLfloat);
     func!(CoverFillPathNV, (), path: GLuint, coverMode: GLenum);
     func!(CoverStrokePathInstancedNV, (), numPaths: GLsizei, pathNameType: GLenum, paths: *const c_void, pathBase: GLuint, coverMode: GLenum, transformType: GLenum, transformValues: *const GLfloat);
     func!(CoverStrokePathNV, (), path: GLuint, coverMode: GLenum);
     func!(CoverageModulationNV, (), components: GLenum);
     func!(CoverageModulationTableNV, (), n: GLsizei, v: *const GLfloat);
     func!(CreateBuffers, (), n: GLsizei, buffers: *mut GLuint);
     func!(CreateCommandListsNV, (), n: GLsizei, lists: *mut GLuint);
     func!(CreateFramebuffers, (), n: GLsizei, framebuffers: *mut GLuint);
     func!(CreateMemoryObjectsEXT, (), n: GLsizei, memoryObjects: *mut GLuint);
     func!(CreatePerfQueryINTEL, (), queryId: GLuint, queryHandle: *mut GLuint);
     func!(CreateProgram, GLuint, );
     func!(CreateProgramObjectARB, GLhandleARB, );
     func!(CreateProgramPipelines, (), n: GLsizei, pipelines: *mut GLuint);
     func!(CreateProgressFenceNVX, GLuint, );
     func!(CreateQueries, (), target: GLenum, n: GLsizei, ids: *mut GLuint);
     func!(CreateRenderbuffers, (), n: GLsizei, renderbuffers: *mut GLuint);
     func!(CreateSamplers, (), n: GLsizei, samplers: *mut GLuint);
     func!(CreateSemaphoresNV, (), n: GLsizei, semaphores: *mut GLuint);
     func!(CreateShader, GLuint, type_: GLenum);
     func!(CreateShaderObjectARB, GLhandleARB, shaderType: GLenum);
     func!(CreateShaderProgramEXT, GLuint, type_: GLenum, string: *const GLchar);
     func!(CreateShaderProgramv, GLuint, type_: GLenum, count: GLsizei, strings: *const *const GLchar);
     func!(CreateStatesNV, (), n: GLsizei, states: *mut GLuint);
     func!(CreateSyncFromCLeventARB, GLsync, context: *mut _cl_context, event: *mut _cl_event, flags: GLbitfield);
     func!(CreateTextures, (), target: GLenum, n: GLsizei, textures: *mut GLuint);
     func!(CreateTransformFeedbacks, (), n: GLsizei, ids: *mut GLuint);
     func!(CreateVertexArrays, (), n: GLsizei, arrays: *mut GLuint);
     func!(CullFace, (), mode: GLenum);
     func!(CullParameterdvEXT, (), pname: GLenum, params: *mut GLdouble);
     func!(CullParameterfvEXT, (), pname: GLenum, params: *mut GLfloat);
     func!(CurrentPaletteMatrixARB, (), index: GLint);
     func!(DebugMessageCallback, (), callback: GLDEBUGPROC, userParam: *const c_void);
     func!(DebugMessageCallbackAMD, (), callback: GLDEBUGPROCAMD, userParam: *mut c_void);
     func!(DebugMessageCallbackARB, (), callback: GLDEBUGPROCARB, userParam: *const c_void);
     func!(DebugMessageControl, (), source: GLenum, type_: GLenum, severity: GLenum, count: GLsizei, ids: *const GLuint, enabled: GLboolean);
     func!(DebugMessageControlARB, (), source: GLenum, type_: GLenum, severity: GLenum, count: GLsizei, ids: *const GLuint, enabled: GLboolean);
     func!(DebugMessageEnableAMD, (), category: GLenum, severity: GLenum, count: GLsizei, ids: *const GLuint, enabled: GLboolean);
     func!(DebugMessageInsert, (), source: GLenum, type_: GLenum, id: GLuint, severity: GLenum, length: GLsizei, buf: *const GLchar);
     func!(DebugMessageInsertAMD, (), category: GLenum, severity: GLenum, id: GLuint, length: GLsizei, buf: *const GLchar);
     func!(DebugMessageInsertARB, (), source: GLenum, type_: GLenum, id: GLuint, severity: GLenum, length: GLsizei, buf: *const GLchar);
     func!(DeformSGIX, (), mask: GLbitfield);
     func!(DeformationMap3dSGIX, (), target: GLenum, u1: GLdouble, u2: GLdouble, ustride: GLint, uorder: GLint, v1: GLdouble, v2: GLdouble, vstride: GLint, vorder: GLint, w1: GLdouble, w2: GLdouble, wstride: GLint, worder: GLint, points: *const GLdouble);
     func!(DeformationMap3fSGIX, (), target: GLenum, u1: GLfloat, u2: GLfloat, ustride: GLint, uorder: GLint, v1: GLfloat, v2: GLfloat, vstride: GLint, vorder: GLint, w1: GLfloat, w2: GLfloat, wstride: GLint, worder: GLint, points: *const GLfloat);
     func!(DeleteAsyncMarkersSGIX, (), marker: GLuint, range: GLsizei);
     func!(DeleteBuffers, (), n: GLsizei, buffers: *const GLuint);
     func!(DeleteBuffersARB, (), n: GLsizei, buffers: *const GLuint);
     func!(DeleteCommandListsNV, (), n: GLsizei, lists: *const GLuint);
     func!(DeleteFencesAPPLE, (), n: GLsizei, fences: *const GLuint);
     func!(DeleteFencesNV, (), n: GLsizei, fences: *const GLuint);
     func!(DeleteFragmentShaderATI, (), id: GLuint);
     func!(DeleteFramebuffers, (), n: GLsizei, framebuffers: *const GLuint);
     func!(DeleteFramebuffersEXT, (), n: GLsizei, framebuffers: *const GLuint);
     func!(DeleteMemoryObjectsEXT, (), n: GLsizei, memoryObjects: *const GLuint);
     func!(DeleteNamedStringARB, (), namelen: GLint, name: *const GLchar);
     func!(DeleteNamesAMD, (), identifier: GLenum, num: GLuint, names: *const GLuint);
     func!(DeleteObjectARB, (), obj: GLhandleARB);
     func!(DeleteOcclusionQueriesNV, (), n: GLsizei, ids: *const GLuint);
     func!(DeletePathsNV, (), path: GLuint, range: GLsizei);
     func!(DeletePerfMonitorsAMD, (), n: GLsizei, monitors: *mut GLuint);
     func!(DeletePerfQueryINTEL, (), queryHandle: GLuint);
     func!(DeleteProgram, (), program: GLuint);
     func!(DeleteProgramPipelines, (), n: GLsizei, pipelines: *const GLuint);
     func!(DeleteProgramsARB, (), n: GLsizei, programs: *const GLuint);
     func!(DeleteProgramsNV, (), n: GLsizei, programs: *const GLuint);
     func!(DeleteQueries, (), n: GLsizei, ids: *const GLuint);
     func!(DeleteQueriesARB, (), n: GLsizei, ids: *const GLuint);
     func!(DeleteQueryResourceTagNV, (), n: GLsizei, tagIds: *const GLint);
     func!(DeleteRenderbuffers, (), n: GLsizei, renderbuffers: *const GLuint);
     func!(DeleteRenderbuffersEXT, (), n: GLsizei, renderbuffers: *const GLuint);
     func!(DeleteSamplers, (), count: GLsizei, samplers: *const GLuint);
     func!(DeleteSemaphoresEXT, (), n: GLsizei, semaphores: *const GLuint);
     func!(DeleteShader, (), shader: GLuint);
     func!(DeleteStatesNV, (), n: GLsizei, states: *const GLuint);
     func!(DeleteSync, (), sync: GLsync);
     func!(DeleteTextures, (), n: GLsizei, textures: *const GLuint);
     func!(DeleteTexturesEXT, (), n: GLsizei, textures: *const GLuint);
     func!(DeleteTransformFeedbacks, (), n: GLsizei, ids: *const GLuint);
     func!(DeleteTransformFeedbacksNV, (), n: GLsizei, ids: *const GLuint);
     func!(DeleteVertexArrays, (), n: GLsizei, arrays: *const GLuint);
     func!(DeleteVertexArraysAPPLE, (), n: GLsizei, arrays: *const GLuint);
     func!(DeleteVertexShaderEXT, (), id: GLuint);
     func!(DepthBoundsEXT, (), zmin: GLclampd, zmax: GLclampd);
     func!(DepthBoundsdNV, (), zmin: GLdouble, zmax: GLdouble);
     func!(DepthFunc, (), func: GLenum);
     func!(DepthMask, (), flag: GLboolean);
     func!(DepthRange, (), n: GLdouble, f: GLdouble);
     func!(DepthRangeArraydvNV, (), first: GLuint, count: GLsizei, v: *const GLdouble);
     func!(DepthRangeArrayv, (), first: GLuint, count: GLsizei, v: *const GLdouble);
     func!(DepthRangeIndexed, (), index: GLuint, n: GLdouble, f: GLdouble);
     func!(DepthRangeIndexeddNV, (), index: GLuint, n: GLdouble, f: GLdouble);
     func!(DepthRangedNV, (), zNear: GLdouble, zFar: GLdouble);
     func!(DepthRangef, (), n: GLfloat, f: GLfloat);
     func!(DepthRangefOES, (), n: GLclampf, f: GLclampf);
     func!(DepthRangexOES, (), n: GLfixed, f: GLfixed);
     func!(DetachObjectARB, (), containerObj: GLhandleARB, attachedObj: GLhandleARB);
     func!(DetachShader, (), program: GLuint, shader: GLuint);
     func!(DetailTexFuncSGIS, (), target: GLenum, n: GLsizei, points: *const GLfloat);
     func!(Disable, (), cap: GLenum);
     func!(DisableClientStateIndexedEXT, (), array: GLenum, index: GLuint);
     func!(DisableClientStateiEXT, (), array: GLenum, index: GLuint);
     func!(DisableIndexedEXT, (), target: GLenum, index: GLuint);
     func!(DisableVariantClientStateEXT, (), id: GLuint);
     func!(DisableVertexArrayAttrib, (), vaobj: GLuint, index: GLuint);
     func!(DisableVertexArrayAttribEXT, (), vaobj: GLuint, index: GLuint);
     func!(DisableVertexArrayEXT, (), vaobj: GLuint, array: GLenum);
     func!(DisableVertexAttribAPPLE, (), index: GLuint, pname: GLenum);
     func!(DisableVertexAttribArray, (), index: GLuint);
     func!(DisableVertexAttribArrayARB, (), index: GLuint);
     func!(Disablei, (), target: GLenum, index: GLuint);
     func!(DispatchCompute, (), num_groups_x: GLuint, num_groups_y: GLuint, num_groups_z: GLuint);
     func!(DispatchComputeGroupSizeARB, (), num_groups_x: GLuint, num_groups_y: GLuint, num_groups_z: GLuint, group_size_x: GLuint, group_size_y: GLuint, group_size_z: GLuint);
     func!(DispatchComputeIndirect, (), indirect: GLintptr);
     func!(DrawArrays, (), mode: GLenum, first: GLint, count: GLsizei);
     func!(DrawArraysEXT, (), mode: GLenum, first: GLint, count: GLsizei);
     func!(DrawArraysIndirect, (), mode: GLenum, indirect: *const c_void);
     func!(DrawArraysInstanced, (), mode: GLenum, first: GLint, count: GLsizei, instancecount: GLsizei);
     func!(DrawArraysInstancedARB, (), mode: GLenum, first: GLint, count: GLsizei, primcount: GLsizei);
     func!(DrawArraysInstancedBaseInstance, (), mode: GLenum, first: GLint, count: GLsizei, instancecount: GLsizei, baseinstance: GLuint);
     func!(DrawArraysInstancedEXT, (), mode: GLenum, start: GLint, count: GLsizei, primcount: GLsizei);
     func!(DrawBuffer, (), buf: GLenum);
     func!(DrawBuffers, (), n: GLsizei, bufs: *const GLenum);
     func!(DrawBuffersARB, (), n: GLsizei, bufs: *const GLenum);
     func!(DrawBuffersATI, (), n: GLsizei, bufs: *const GLenum);
     func!(DrawCommandsAddressNV, (), primitiveMode: GLenum, indirects: *const GLuint64, sizes: *const GLsizei, count: GLuint);
     func!(DrawCommandsNV, (), primitiveMode: GLenum, buffer: GLuint, indirects: *const GLintptr, sizes: *const GLsizei, count: GLuint);
     func!(DrawCommandsStatesAddressNV, (), indirects: *const GLuint64, sizes: *const GLsizei, states: *const GLuint, fbos: *const GLuint, count: GLuint);
     func!(DrawCommandsStatesNV, (), buffer: GLuint, indirects: *const GLintptr, sizes: *const GLsizei, states: *const GLuint, fbos: *const GLuint, count: GLuint);
     func!(DrawElementArrayAPPLE, (), mode: GLenum, first: GLint, count: GLsizei);
     func!(DrawElementArrayATI, (), mode: GLenum, count: GLsizei);
     func!(DrawElements, (), mode: GLenum, count: GLsizei, type_: GLenum, indices: *const c_void);
     func!(DrawElementsBaseVertex, (), mode: GLenum, count: GLsizei, type_: GLenum, indices: *const c_void, basevertex: GLint);
     func!(DrawElementsIndirect, (), mode: GLenum, type_: GLenum, indirect: *const c_void);
     func!(DrawElementsInstanced, (), mode: GLenum, count: GLsizei, type_: GLenum, indices: *const c_void, instancecount: GLsizei);
     func!(DrawElementsInstancedARB, (), mode: GLenum, count: GLsizei, type_: GLenum, indices: *const c_void, primcount: GLsizei);
     func!(DrawElementsInstancedBaseInstance, (), mode: GLenum, count: GLsizei, type_: GLenum, indices: *const c_void, instancecount: GLsizei, baseinstance: GLuint);
     func!(DrawElementsInstancedBaseVertex, (), mode: GLenum, count: GLsizei, type_: GLenum, indices: *const c_void, instancecount: GLsizei, basevertex: GLint);
     func!(DrawElementsInstancedBaseVertexBaseInstance, (), mode: GLenum, count: GLsizei, type_: GLenum, indices: *const c_void, instancecount: GLsizei, basevertex: GLint, baseinstance: GLuint);
     func!(DrawElementsInstancedEXT, (), mode: GLenum, count: GLsizei, type_: GLenum, indices: *const c_void, primcount: GLsizei);
     func!(DrawMeshArraysSUN, (), mode: GLenum, first: GLint, count: GLsizei, width: GLsizei);
     func!(DrawMeshTasksIndirectNV, (), indirect: GLintptr);
     func!(DrawMeshTasksNV, (), first: GLuint, count: GLuint);
     func!(DrawRangeElementArrayAPPLE, (), mode: GLenum, start: GLuint, end: GLuint, first: GLint, count: GLsizei);
     func!(DrawRangeElementArrayATI, (), mode: GLenum, start: GLuint, end: GLuint, count: GLsizei);
     func!(DrawRangeElements, (), mode: GLenum, start: GLuint, end: GLuint, count: GLsizei, type_: GLenum, indices: *const c_void);
     func!(DrawRangeElementsBaseVertex, (), mode: GLenum, start: GLuint, end: GLuint, count: GLsizei, type_: GLenum, indices: *const c_void, basevertex: GLint);
     func!(DrawRangeElementsEXT, (), mode: GLenum, start: GLuint, end: GLuint, count: GLsizei, type_: GLenum, indices: *const c_void);
     func!(DrawTextureNV, (), texture: GLuint, sampler: GLuint, x0: GLfloat, y0: GLfloat, x1: GLfloat, y1: GLfloat, z: GLfloat, s0: GLfloat, t0: GLfloat, s1: GLfloat, t1: GLfloat);
     func!(DrawTransformFeedback, (), mode: GLenum, id: GLuint);
     func!(DrawTransformFeedbackInstanced, (), mode: GLenum, id: GLuint, instancecount: GLsizei);
     func!(DrawTransformFeedbackNV, (), mode: GLenum, id: GLuint);
     func!(DrawTransformFeedbackStream, (), mode: GLenum, id: GLuint, stream: GLuint);
     func!(DrawTransformFeedbackStreamInstanced, (), mode: GLenum, id: GLuint, stream: GLuint, instancecount: GLsizei);
     func!(DrawVkImageNV, (), vkImage: GLuint64, sampler: GLuint, x0: GLfloat, y0: GLfloat, x1: GLfloat, y1: GLfloat, z: GLfloat, s0: GLfloat, t0: GLfloat, s1: GLfloat, t1: GLfloat);
     func!(EGLImageTargetTexStorageEXT, (), target: GLenum, image: GLeglImageOES, attrib_list: *const GLint);
     func!(EGLImageTargetTextureStorageEXT, (), texture: GLuint, image: GLeglImageOES, attrib_list: *const GLint);
     func!(EdgeFlagFormatNV, (), stride: GLsizei);
     func!(EdgeFlagPointerEXT, (), stride: GLsizei, count: GLsizei, pointer: *const GLboolean);
     func!(EdgeFlagPointerListIBM, (), stride: GLint, pointer: *const *const GLboolean, ptrstride: GLint);
     func!(ElementPointerAPPLE, (), type_: GLenum, pointer: *const c_void);
     func!(ElementPointerATI, (), type_: GLenum, pointer: *const c_void);
     func!(Enable, (), cap: GLenum);
     func!(EnableClientStateIndexedEXT, (), array: GLenum, index: GLuint);
     func!(EnableClientStateiEXT, (), array: GLenum, index: GLuint);
     func!(EnableIndexedEXT, (), target: GLenum, index: GLuint);
     func!(EnableVariantClientStateEXT, (), id: GLuint);
     func!(EnableVertexArrayAttrib, (), vaobj: GLuint, index: GLuint);
     func!(EnableVertexArrayAttribEXT, (), vaobj: GLuint, index: GLuint);
     func!(EnableVertexArrayEXT, (), vaobj: GLuint, array: GLenum);
     func!(EnableVertexAttribAPPLE, (), index: GLuint, pname: GLenum);
     func!(EnableVertexAttribArray, (), index: GLuint);
     func!(EnableVertexAttribArrayARB, (), index: GLuint);
     func!(Enablei, (), target: GLenum, index: GLuint);
     func!(EndConditionalRender, (), );
     func!(EndConditionalRenderNV, (), );
     func!(EndConditionalRenderNVX, (), );
     func!(EndFragmentShaderATI, (), );
     func!(EndOcclusionQueryNV, (), );
     func!(EndPerfMonitorAMD, (), monitor: GLuint);
     func!(EndPerfQueryINTEL, (), queryHandle: GLuint);
     func!(EndQuery, (), target: GLenum);
     func!(EndQueryARB, (), target: GLenum);
     func!(EndQueryIndexed, (), target: GLenum, index: GLuint);
     func!(EndTransformFeedback, (), );
     func!(EndTransformFeedbackEXT, (), );
     func!(EndTransformFeedbackNV, (), );
     func!(EndVertexShaderEXT, (), );
     func!(EndVideoCaptureNV, (), video_capture_slot: GLuint);
     func!(EvalCoord1xOES, (), u: GLfixed);
     func!(EvalCoord1xvOES, (), coords: *const GLfixed);
     func!(EvalCoord2xOES, (), u: GLfixed, v: GLfixed);
     func!(EvalCoord2xvOES, (), coords: *const GLfixed);
     func!(EvalMapsNV, (), target: GLenum, mode: GLenum);
     func!(EvaluateDepthValuesARB, (), );
     func!(ExecuteProgramNV, (), target: GLenum, id: GLuint, params: *const GLfloat);
     func!(ExtractComponentEXT, (), res: GLuint, src: GLuint, num: GLuint);
     func!(FeedbackBufferxOES, (), n: GLsizei, type_: GLenum, buffer: *const GLfixed);
     func!(FenceSync, GLsync, condition: GLenum, flags: GLbitfield);
     func!(FinalCombinerInputNV, (), variable: GLenum, input: GLenum, mapping: GLenum, componentUsage: GLenum);
     func!(Finish, (), );
     func!(FinishAsyncSGIX, GLint, markerp: *mut GLuint);
     func!(FinishFenceAPPLE, (), fence: GLuint);
     func!(FinishFenceNV, (), fence: GLuint);
     func!(FinishObjectAPPLE, (), object: GLenum, name: GLint);
     func!(FinishTextureSUNX, (), );
     func!(Flush, (), );
     func!(FlushMappedBufferRange, (), target: GLenum, offset: GLintptr, length: GLsizeiptr);
     func!(FlushMappedBufferRangeAPPLE, (), target: GLenum, offset: GLintptr, size: GLsizeiptr);
     func!(FlushMappedNamedBufferRange, (), buffer: GLuint, offset: GLintptr, length: GLsizeiptr);
     func!(FlushMappedNamedBufferRangeEXT, (), buffer: GLuint, offset: GLintptr, length: GLsizeiptr);
     func!(FlushPixelDataRangeNV, (), target: GLenum);
     func!(FlushRasterSGIX, (), );
     func!(FlushStaticDataIBM, (), target: GLenum);
     func!(FlushVertexArrayRangeAPPLE, (), length: GLsizei, pointer: *mut c_void);
     func!(FlushVertexArrayRangeNV, (), );
     func!(FogCoordFormatNV, (), type_: GLenum, stride: GLsizei);
     func!(FogCoordPointerEXT, (), type_: GLenum, stride: GLsizei, pointer: *const c_void);
     func!(FogCoordPointerListIBM, (), type_: GLenum, stride: GLint, pointer: *const *const c_void, ptrstride: GLint);
     func!(FogCoorddEXT, (), coord: GLdouble);
     func!(FogCoorddvEXT, (), coord: *const GLdouble);
     func!(FogCoordfEXT, (), coord: GLfloat);
     func!(FogCoordfvEXT, (), coord: *const GLfloat);
     func!(FogCoordhNV, (), fog: GLhalfNV);
     func!(FogCoordhvNV, (), fog: *const GLhalfNV);
     func!(FogFuncSGIS, (), n: GLsizei, points: *const GLfloat);
     func!(FogxOES, (), pname: GLenum, param: GLfixed);
     func!(FogxvOES, (), pname: GLenum, param: *const GLfixed);
     func!(FragmentColorMaterialSGIX, (), face: GLenum, mode: GLenum);
     func!(FragmentCoverageColorNV, (), color: GLuint);
     func!(FragmentLightModelfSGIX, (), pname: GLenum, param: GLfloat);
     func!(FragmentLightModelfvSGIX, (), pname: GLenum, params: *const GLfloat);
     func!(FragmentLightModeliSGIX, (), pname: GLenum, param: GLint);
     func!(FragmentLightModelivSGIX, (), pname: GLenum, params: *const GLint);
     func!(FragmentLightfSGIX, (), light: GLenum, pname: GLenum, param: GLfloat);
     func!(FragmentLightfvSGIX, (), light: GLenum, pname: GLenum, params: *const GLfloat);
     func!(FragmentLightiSGIX, (), light: GLenum, pname: GLenum, param: GLint);
     func!(FragmentLightivSGIX, (), light: GLenum, pname: GLenum, params: *const GLint);
     func!(FragmentMaterialfSGIX, (), face: GLenum, pname: GLenum, param: GLfloat);
     func!(FragmentMaterialfvSGIX, (), face: GLenum, pname: GLenum, params: *const GLfloat);
     func!(FragmentMaterialiSGIX, (), face: GLenum, pname: GLenum, param: GLint);
     func!(FragmentMaterialivSGIX, (), face: GLenum, pname: GLenum, params: *const GLint);
     func!(FrameTerminatorGREMEDY, (), );
     func!(FrameZoomSGIX, (), factor: GLint);
     func!(FramebufferDrawBufferEXT, (), framebuffer: GLuint, mode: GLenum);
     func!(FramebufferDrawBuffersEXT, (), framebuffer: GLuint, n: GLsizei, bufs: *const GLenum);
     func!(FramebufferFetchBarrierEXT, (), );
     func!(FramebufferParameteri, (), target: GLenum, pname: GLenum, param: GLint);
     func!(FramebufferParameteriMESA, (), target: GLenum, pname: GLenum, param: GLint);
     func!(FramebufferReadBufferEXT, (), framebuffer: GLuint, mode: GLenum);
     func!(FramebufferRenderbuffer, (), target: GLenum, attachment: GLenum, renderbuffertarget: GLenum, renderbuffer: GLuint);
     func!(FramebufferRenderbufferEXT, (), target: GLenum, attachment: GLenum, renderbuffertarget: GLenum, renderbuffer: GLuint);
     func!(FramebufferSampleLocationsfvARB, (), target: GLenum, start: GLuint, count: GLsizei, v: *const GLfloat);
     func!(FramebufferSampleLocationsfvNV, (), target: GLenum, start: GLuint, count: GLsizei, v: *const GLfloat);
     func!(FramebufferSamplePositionsfvAMD, (), target: GLenum, numsamples: GLuint, pixelindex: GLuint, values: *const GLfloat);
     func!(FramebufferTexture, (), target: GLenum, attachment: GLenum, texture: GLuint, level: GLint);
     func!(FramebufferTexture1D, (), target: GLenum, attachment: GLenum, textarget: GLenum, texture: GLuint, level: GLint);
     func!(FramebufferTexture1DEXT, (), target: GLenum, attachment: GLenum, textarget: GLenum, texture: GLuint, level: GLint);
     func!(FramebufferTexture2D, (), target: GLenum, attachment: GLenum, textarget: GLenum, texture: GLuint, level: GLint);
     func!(FramebufferTexture2DEXT, (), target: GLenum, attachment: GLenum, textarget: GLenum, texture: GLuint, level: GLint);
     func!(FramebufferTexture3D, (), target: GLenum, attachment: GLenum, textarget: GLenum, texture: GLuint, level: GLint, zoffset: GLint);
     func!(FramebufferTexture3DEXT, (), target: GLenum, attachment: GLenum, textarget: GLenum, texture: GLuint, level: GLint, zoffset: GLint);
     func!(FramebufferTextureARB, (), target: GLenum, attachment: GLenum, texture: GLuint, level: GLint);
     func!(FramebufferTextureEXT, (), target: GLenum, attachment: GLenum, texture: GLuint, level: GLint);
     func!(FramebufferTextureFaceARB, (), target: GLenum, attachment: GLenum, texture: GLuint, level: GLint, face: GLenum);
     func!(FramebufferTextureFaceEXT, (), target: GLenum, attachment: GLenum, texture: GLuint, level: GLint, face: GLenum);
     func!(FramebufferTextureLayer, (), target: GLenum, attachment: GLenum, texture: GLuint, level: GLint, layer: GLint);
     func!(FramebufferTextureLayerARB, (), target: GLenum, attachment: GLenum, texture: GLuint, level: GLint, layer: GLint);
     func!(FramebufferTextureLayerEXT, (), target: GLenum, attachment: GLenum, texture: GLuint, level: GLint, layer: GLint);
     func!(FramebufferTextureMultiviewOVR, (), target: GLenum, attachment: GLenum, texture: GLuint, level: GLint, baseViewIndex: GLint, numViews: GLsizei);
     func!(FreeObjectBufferATI, (), buffer: GLuint);
     func!(FrontFace, (), mode: GLenum);
     func!(FrustumfOES, (), l: GLfloat, r: GLfloat, b: GLfloat, t: GLfloat, n: GLfloat, f: GLfloat);
     func!(FrustumxOES, (), l: GLfixed, r: GLfixed, b: GLfixed, t: GLfixed, n: GLfixed, f: GLfixed);
     func!(GenAsyncMarkersSGIX, GLuint, range: GLsizei);
     func!(GenBuffers, (), n: GLsizei, buffers: *mut GLuint);
     func!(GenBuffersARB, (), n: GLsizei, buffers: *mut GLuint);
     func!(GenFencesAPPLE, (), n: GLsizei, fences: *mut GLuint);
     func!(GenFencesNV, (), n: GLsizei, fences: *mut GLuint);
     func!(GenFragmentShadersATI, GLuint, range: GLuint);
     func!(GenFramebuffers, (), n: GLsizei, framebuffers: *mut GLuint);
     func!(GenFramebuffersEXT, (), n: GLsizei, framebuffers: *mut GLuint);
     func!(GenNamesAMD, (), identifier: GLenum, num: GLuint, names: *mut GLuint);
     func!(GenOcclusionQueriesNV, (), n: GLsizei, ids: *mut GLuint);
     func!(GenPathsNV, GLuint, range: GLsizei);
     func!(GenPerfMonitorsAMD, (), n: GLsizei, monitors: *mut GLuint);
     func!(GenProgramPipelines, (), n: GLsizei, pipelines: *mut GLuint);
     func!(GenProgramsARB, (), n: GLsizei, programs: *mut GLuint);
     func!(GenProgramsNV, (), n: GLsizei, programs: *mut GLuint);
     func!(GenQueries, (), n: GLsizei, ids: *mut GLuint);
     func!(GenQueriesARB, (), n: GLsizei, ids: *mut GLuint);
     func!(GenQueryResourceTagNV, (), n: GLsizei, tagIds: *mut GLint);
     func!(GenRenderbuffers, (), n: GLsizei, renderbuffers: *mut GLuint);
     func!(GenRenderbuffersEXT, (), n: GLsizei, renderbuffers: *mut GLuint);
     func!(GenSamplers, (), count: GLsizei, samplers: *mut GLuint);
     func!(GenSemaphoresEXT, (), n: GLsizei, semaphores: *mut GLuint);
     func!(GenSymbolsEXT, GLuint, datatype: GLenum, storagetype: GLenum, range: GLenum, components: GLuint);
     func!(GenTextures, (), n: GLsizei, textures: *mut GLuint);
     func!(GenTexturesEXT, (), n: GLsizei, textures: *mut GLuint);
     func!(GenTransformFeedbacks, (), n: GLsizei, ids: *mut GLuint);
     func!(GenTransformFeedbacksNV, (), n: GLsizei, ids: *mut GLuint);
     func!(GenVertexArrays, (), n: GLsizei, arrays: *mut GLuint);
     func!(GenVertexArraysAPPLE, (), n: GLsizei, arrays: *mut GLuint);
     func!(GenVertexShadersEXT, GLuint, range: GLuint);
     func!(GenerateMipmap, (), target: GLenum);
     func!(GenerateMipmapEXT, (), target: GLenum);
     func!(GenerateMultiTexMipmapEXT, (), texunit: GLenum, target: GLenum);
     func!(GenerateTextureMipmap, (), texture: GLuint);
     func!(GenerateTextureMipmapEXT, (), texture: GLuint, target: GLenum);
     func!(GetActiveAtomicCounterBufferiv, (), program: GLuint, bufferIndex: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetActiveAttrib, (), program: GLuint, index: GLuint, bufSize: GLsizei, length: *mut GLsizei, size: *mut GLint, type_: *mut GLenum, name: *mut GLchar);
     func!(GetActiveAttribARB, (), programObj: GLhandleARB, index: GLuint, maxLength: GLsizei, length: *mut GLsizei, size: *mut GLint, type_: *mut GLenum, name: *mut GLcharARB);
     func!(GetActiveSubroutineName, (), program: GLuint, shadertype: GLenum, index: GLuint, bufSize: GLsizei, length: *mut GLsizei, name: *mut GLchar);
     func!(GetActiveSubroutineUniformName, (), program: GLuint, shadertype: GLenum, index: GLuint, bufSize: GLsizei, length: *mut GLsizei, name: *mut GLchar);
     func!(GetActiveSubroutineUniformiv, (), program: GLuint, shadertype: GLenum, index: GLuint, pname: GLenum, values: *mut GLint);
     func!(GetActiveUniform, (), program: GLuint, index: GLuint, bufSize: GLsizei, length: *mut GLsizei, size: *mut GLint, type_: *mut GLenum, name: *mut GLchar);
     func!(GetActiveUniformARB, (), programObj: GLhandleARB, index: GLuint, maxLength: GLsizei, length: *mut GLsizei, size: *mut GLint, type_: *mut GLenum, name: *mut GLcharARB);
     func!(GetActiveUniformBlockName, (), program: GLuint, uniformBlockIndex: GLuint, bufSize: GLsizei, length: *mut GLsizei, uniformBlockName: *mut GLchar);
     func!(GetActiveUniformBlockiv, (), program: GLuint, uniformBlockIndex: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetActiveUniformName, (), program: GLuint, uniformIndex: GLuint, bufSize: GLsizei, length: *mut GLsizei, uniformName: *mut GLchar);
     func!(GetActiveUniformsiv, (), program: GLuint, uniformCount: GLsizei, uniformIndices: *const GLuint, pname: GLenum, params: *mut GLint);
     func!(GetActiveVaryingNV, (), program: GLuint, index: GLuint, bufSize: GLsizei, length: *mut GLsizei, size: *mut GLsizei, type_: *mut GLenum, name: *mut GLchar);
     func!(GetArrayObjectfvATI, (), array: GLenum, pname: GLenum, params: *mut GLfloat);
     func!(GetArrayObjectivATI, (), array: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetAttachedObjectsARB, (), containerObj: GLhandleARB, maxCount: GLsizei, count: *mut GLsizei, obj: *mut GLhandleARB);
     func!(GetAttachedShaders, (), program: GLuint, maxCount: GLsizei, count: *mut GLsizei, shaders: *mut GLuint);
     func!(GetAttribLocation, GLint, program: GLuint, name: *const GLchar);
     func!(GetAttribLocationARB, GLint, programObj: GLhandleARB, name: *const GLcharARB);
     func!(GetBooleanIndexedvEXT, (), target: GLenum, index: GLuint, data: *mut GLboolean);
     func!(GetBooleani_v, (), target: GLenum, index: GLuint, data: *mut GLboolean);
     func!(GetBooleanv, (), pname: GLenum, data: *mut GLboolean);
     func!(GetBufferParameteri64v, (), target: GLenum, pname: GLenum, params: *mut GLint64);
     func!(GetBufferParameteriv, (), target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetBufferParameterivARB, (), target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetBufferParameterui64vNV, (), target: GLenum, pname: GLenum, params: *mut GLuint64EXT);
     func!(GetBufferPointerv, (), target: GLenum, pname: GLenum, params: *mut *mut c_void);
     func!(GetBufferPointervARB, (), target: GLenum, pname: GLenum, params: *mut *mut c_void);
     func!(GetBufferSubData, (), target: GLenum, offset: GLintptr, size: GLsizeiptr, data: *mut c_void);
     func!(GetBufferSubDataARB, (), target: GLenum, offset: GLintptrARB, size: GLsizeiptrARB, data: *mut c_void);
     func!(GetClipPlanefOES, (), plane: GLenum, equation: *mut GLfloat);
     func!(GetClipPlanexOES, (), plane: GLenum, equation: *mut GLfixed);
     func!(GetColorTableEXT, (), target: GLenum, format: GLenum, type_: GLenum, data: *mut c_void);
     func!(GetColorTableParameterfvEXT, (), target: GLenum, pname: GLenum, params: *mut GLfloat);
     func!(GetColorTableParameterfvSGI, (), target: GLenum, pname: GLenum, params: *mut GLfloat);
     func!(GetColorTableParameterivEXT, (), target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetColorTableParameterivSGI, (), target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetColorTableSGI, (), target: GLenum, format: GLenum, type_: GLenum, table: *mut c_void);
     func!(GetCombinerInputParameterfvNV, (), stage: GLenum, portion: GLenum, variable: GLenum, pname: GLenum, params: *mut GLfloat);
     func!(GetCombinerInputParameterivNV, (), stage: GLenum, portion: GLenum, variable: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetCombinerOutputParameterfvNV, (), stage: GLenum, portion: GLenum, pname: GLenum, params: *mut GLfloat);
     func!(GetCombinerOutputParameterivNV, (), stage: GLenum, portion: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetCombinerStageParameterfvNV, (), stage: GLenum, pname: GLenum, params: *mut GLfloat);
     func!(GetCommandHeaderNV, GLuint, tokenID: GLenum, size: GLuint);
     func!(GetCompressedMultiTexImageEXT, (), texunit: GLenum, target: GLenum, lod: GLint, img: *mut c_void);
     func!(GetCompressedTexImage, (), target: GLenum, level: GLint, img: *mut c_void);
     func!(GetCompressedTexImageARB, (), target: GLenum, level: GLint, img: *mut c_void);
     func!(GetCompressedTextureImage, (), texture: GLuint, level: GLint, bufSize: GLsizei, pixels: *mut c_void);
     func!(GetCompressedTextureImageEXT, (), texture: GLuint, target: GLenum, lod: GLint, img: *mut c_void);
     func!(GetCompressedTextureSubImage, (), texture: GLuint, level: GLint, xoffset: GLint, yoffset: GLint, zoffset: GLint, width: GLsizei, height: GLsizei, depth: GLsizei, bufSize: GLsizei, pixels: *mut c_void);
     func!(GetConvolutionFilterEXT, (), target: GLenum, format: GLenum, type_: GLenum, image: *mut c_void);
     func!(GetConvolutionParameterfvEXT, (), target: GLenum, pname: GLenum, params: *mut GLfloat);
     func!(GetConvolutionParameterivEXT, (), target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetConvolutionParameterxvOES, (), target: GLenum, pname: GLenum, params: *mut GLfixed);
     func!(GetCoverageModulationTableNV, (), bufSize: GLsizei, v: *mut GLfloat);
     func!(GetDebugMessageLog, GLuint, count: GLuint, bufSize: GLsizei, sources: *mut GLenum, types: *mut GLenum, ids: *mut GLuint, severities: *mut GLenum, lengths: *mut GLsizei, messageLog: *mut GLchar);
     func!(GetDebugMessageLogAMD, GLuint, count: GLuint, bufSize: GLsizei, categories: *mut GLenum, severities: *mut GLuint, ids: *mut GLuint, lengths: *mut GLsizei, message: *mut GLchar);
     func!(GetDebugMessageLogARB, GLuint, count: GLuint, bufSize: GLsizei, sources: *mut GLenum, types: *mut GLenum, ids: *mut GLuint, severities: *mut GLenum, lengths: *mut GLsizei, messageLog: *mut GLchar);
     func!(GetDetailTexFuncSGIS, (), target: GLenum, points: *mut GLfloat);
     func!(GetDoubleIndexedvEXT, (), target: GLenum, index: GLuint, data: *mut GLdouble);
     func!(GetDoublei_v, (), target: GLenum, index: GLuint, data: *mut GLdouble);
     func!(GetDoublei_vEXT, (), pname: GLenum, index: GLuint, params: *mut GLdouble);
     func!(GetDoublev, (), pname: GLenum, data: *mut GLdouble);
     func!(GetError, GLenum, );
     func!(GetFenceivNV, (), fence: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetFinalCombinerInputParameterfvNV, (), variable: GLenum, pname: GLenum, params: *mut GLfloat);
     func!(GetFinalCombinerInputParameterivNV, (), variable: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetFirstPerfQueryIdINTEL, (), queryId: *mut GLuint);
     func!(GetFixedvOES, (), pname: GLenum, params: *mut GLfixed);
     func!(GetFloatIndexedvEXT, (), target: GLenum, index: GLuint, data: *mut GLfloat);
     func!(GetFloati_v, (), target: GLenum, index: GLuint, data: *mut GLfloat);
     func!(GetFloati_vEXT, (), pname: GLenum, index: GLuint, params: *mut GLfloat);
     func!(GetFloatv, (), pname: GLenum, data: *mut GLfloat);
     func!(GetFogFuncSGIS, (), points: *mut GLfloat);
     func!(GetFragDataIndex, GLint, program: GLuint, name: *const GLchar);
     func!(GetFragDataLocation, GLint, program: GLuint, name: *const GLchar);
     func!(GetFragDataLocationEXT, GLint, program: GLuint, name: *const GLchar);
     func!(GetFragmentLightfvSGIX, (), light: GLenum, pname: GLenum, params: *mut GLfloat);
     func!(GetFragmentLightivSGIX, (), light: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetFragmentMaterialfvSGIX, (), face: GLenum, pname: GLenum, params: *mut GLfloat);
     func!(GetFragmentMaterialivSGIX, (), face: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetFramebufferAttachmentParameteriv, (), target: GLenum, attachment: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetFramebufferAttachmentParameterivEXT, (), target: GLenum, attachment: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetFramebufferParameterfvAMD, (), target: GLenum, pname: GLenum, numsamples: GLuint, pixelindex: GLuint, size: GLsizei, values: *mut GLfloat);
     func!(GetFramebufferParameteriv, (), target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetFramebufferParameterivEXT, (), framebuffer: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetFramebufferParameterivMESA, (), target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetGraphicsResetStatus, GLenum, );
     func!(GetGraphicsResetStatusARB, GLenum, );
     func!(GetHandleARB, GLhandleARB, pname: GLenum);
     func!(GetHistogramEXT, (), target: GLenum, reset: GLboolean, format: GLenum, type_: GLenum, values: *mut c_void);
     func!(GetHistogramParameterfvEXT, (), target: GLenum, pname: GLenum, params: *mut GLfloat);
     func!(GetHistogramParameterivEXT, (), target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetHistogramParameterxvOES, (), target: GLenum, pname: GLenum, params: *mut GLfixed);
     func!(GetImageHandleARB, GLuint64, texture: GLuint, level: GLint, layered: GLboolean, layer: GLint, format: GLenum);
     func!(GetImageHandleNV, GLuint64, texture: GLuint, level: GLint, layered: GLboolean, layer: GLint, format: GLenum);
     func!(GetImageTransformParameterfvHP, (), target: GLenum, pname: GLenum, params: *mut GLfloat);
     func!(GetImageTransformParameterivHP, (), target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetInfoLogARB, (), obj: GLhandleARB, maxLength: GLsizei, length: *mut GLsizei, infoLog: *mut GLcharARB);
     func!(GetInstrumentsSGIX, GLint, );
     func!(GetInteger64i_v, (), target: GLenum, index: GLuint, data: *mut GLint64);
     func!(GetInteger64v, (), pname: GLenum, data: *mut GLint64);
     func!(GetIntegerIndexedvEXT, (), target: GLenum, index: GLuint, data: *mut GLint);
     func!(GetIntegeri_v, (), target: GLenum, index: GLuint, data: *mut GLint);
     func!(GetIntegerui64i_vNV, (), value: GLenum, index: GLuint, result: *mut GLuint64EXT);
     func!(GetIntegerui64vNV, (), value: GLenum, result: *mut GLuint64EXT);
     func!(GetIntegerv, (), pname: GLenum, data: *mut GLint);
     func!(GetInternalformatSampleivNV, (), target: GLenum, internalformat: GLenum, samples: GLsizei, pname: GLenum, count: GLsizei, params: *mut GLint);
     func!(GetInternalformati64v, (), target: GLenum, internalformat: GLenum, pname: GLenum, count: GLsizei, params: *mut GLint64);
     func!(GetInternalformativ, (), target: GLenum, internalformat: GLenum, pname: GLenum, count: GLsizei, params: *mut GLint);
     func!(GetInvariantBooleanvEXT, (), id: GLuint, value: GLenum, data: *mut GLboolean);
     func!(GetInvariantFloatvEXT, (), id: GLuint, value: GLenum, data: *mut GLfloat);
     func!(GetInvariantIntegervEXT, (), id: GLuint, value: GLenum, data: *mut GLint);
     func!(GetLightxOES, (), light: GLenum, pname: GLenum, params: *mut GLfixed);
     func!(GetListParameterfvSGIX, (), list: GLuint, pname: GLenum, params: *mut GLfloat);
     func!(GetListParameterivSGIX, (), list: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetLocalConstantBooleanvEXT, (), id: GLuint, value: GLenum, data: *mut GLboolean);
     func!(GetLocalConstantFloatvEXT, (), id: GLuint, value: GLenum, data: *mut GLfloat);
     func!(GetLocalConstantIntegervEXT, (), id: GLuint, value: GLenum, data: *mut GLint);
     func!(GetMapAttribParameterfvNV, (), target: GLenum, index: GLuint, pname: GLenum, params: *mut GLfloat);
     func!(GetMapAttribParameterivNV, (), target: GLenum, index: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetMapControlPointsNV, (), target: GLenum, index: GLuint, type_: GLenum, ustride: GLsizei, vstride: GLsizei, packed: GLboolean, points: *mut c_void);
     func!(GetMapParameterfvNV, (), target: GLenum, pname: GLenum, params: *mut GLfloat);
     func!(GetMapParameterivNV, (), target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetMapxvOES, (), target: GLenum, query: GLenum, v: *mut GLfixed);
     func!(GetMaterialxOES, (), face: GLenum, pname: GLenum, param: GLfixed);
     func!(GetMemoryObjectDetachedResourcesuivNV, (), memory: GLuint, pname: GLenum, first: GLint, count: GLsizei, params: *mut GLuint);
     func!(GetMemoryObjectParameterivEXT, (), memoryObject: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetMinmaxEXT, (), target: GLenum, reset: GLboolean, format: GLenum, type_: GLenum, values: *mut c_void);
     func!(GetMinmaxParameterfvEXT, (), target: GLenum, pname: GLenum, params: *mut GLfloat);
     func!(GetMinmaxParameterivEXT, (), target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetMultiTexEnvfvEXT, (), texunit: GLenum, target: GLenum, pname: GLenum, params: *mut GLfloat);
     func!(GetMultiTexEnvivEXT, (), texunit: GLenum, target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetMultiTexGendvEXT, (), texunit: GLenum, coord: GLenum, pname: GLenum, params: *mut GLdouble);
     func!(GetMultiTexGenfvEXT, (), texunit: GLenum, coord: GLenum, pname: GLenum, params: *mut GLfloat);
     func!(GetMultiTexGenivEXT, (), texunit: GLenum, coord: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetMultiTexImageEXT, (), texunit: GLenum, target: GLenum, level: GLint, format: GLenum, type_: GLenum, pixels: *mut c_void);
     func!(GetMultiTexLevelParameterfvEXT, (), texunit: GLenum, target: GLenum, level: GLint, pname: GLenum, params: *mut GLfloat);
     func!(GetMultiTexLevelParameterivEXT, (), texunit: GLenum, target: GLenum, level: GLint, pname: GLenum, params: *mut GLint);
     func!(GetMultiTexParameterIivEXT, (), texunit: GLenum, target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetMultiTexParameterIuivEXT, (), texunit: GLenum, target: GLenum, pname: GLenum, params: *mut GLuint);
     func!(GetMultiTexParameterfvEXT, (), texunit: GLenum, target: GLenum, pname: GLenum, params: *mut GLfloat);
     func!(GetMultiTexParameterivEXT, (), texunit: GLenum, target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetMultisamplefv, (), pname: GLenum, index: GLuint, val: *mut GLfloat);
     func!(GetMultisamplefvNV, (), pname: GLenum, index: GLuint, val: *mut GLfloat);
     func!(GetNamedBufferParameteri64v, (), buffer: GLuint, pname: GLenum, params: *mut GLint64);
     func!(GetNamedBufferParameteriv, (), buffer: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetNamedBufferParameterivEXT, (), buffer: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetNamedBufferParameterui64vNV, (), buffer: GLuint, pname: GLenum, params: *mut GLuint64EXT);
     func!(GetNamedBufferPointerv, (), buffer: GLuint, pname: GLenum, params: *mut *mut c_void);
     func!(GetNamedBufferPointervEXT, (), buffer: GLuint, pname: GLenum, params: *mut *mut c_void);
     func!(GetNamedBufferSubData, (), buffer: GLuint, offset: GLintptr, size: GLsizeiptr, data: *mut c_void);
     func!(GetNamedBufferSubDataEXT, (), buffer: GLuint, offset: GLintptr, size: GLsizeiptr, data: *mut c_void);
     func!(GetNamedFramebufferAttachmentParameteriv, (), framebuffer: GLuint, attachment: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetNamedFramebufferAttachmentParameterivEXT, (), framebuffer: GLuint, attachment: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetNamedFramebufferParameterfvAMD, (), framebuffer: GLuint, pname: GLenum, numsamples: GLuint, pixelindex: GLuint, size: GLsizei, values: *mut GLfloat);
     func!(GetNamedFramebufferParameteriv, (), framebuffer: GLuint, pname: GLenum, param: *mut GLint);
     func!(GetNamedFramebufferParameterivEXT, (), framebuffer: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetNamedProgramLocalParameterIivEXT, (), program: GLuint, target: GLenum, index: GLuint, params: *mut GLint);
     func!(GetNamedProgramLocalParameterIuivEXT, (), program: GLuint, target: GLenum, index: GLuint, params: *mut GLuint);
     func!(GetNamedProgramLocalParameterdvEXT, (), program: GLuint, target: GLenum, index: GLuint, params: *mut GLdouble);
     func!(GetNamedProgramLocalParameterfvEXT, (), program: GLuint, target: GLenum, index: GLuint, params: *mut GLfloat);
     func!(GetNamedProgramStringEXT, (), program: GLuint, target: GLenum, pname: GLenum, string: *mut c_void);
     func!(GetNamedProgramivEXT, (), program: GLuint, target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetNamedRenderbufferParameteriv, (), renderbuffer: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetNamedRenderbufferParameterivEXT, (), renderbuffer: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetNamedStringARB, (), namelen: GLint, name: *const GLchar, bufSize: GLsizei, stringlen: *mut GLint, string: *mut GLchar);
     func!(GetNamedStringivARB, (), namelen: GLint, name: *const GLchar, pname: GLenum, params: *mut GLint);
     func!(GetNextPerfQueryIdINTEL, (), queryId: GLuint, nextQueryId: *mut GLuint);
     func!(GetObjectBufferfvATI, (), buffer: GLuint, pname: GLenum, params: *mut GLfloat);
     func!(GetObjectBufferivATI, (), buffer: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetObjectLabel, (), identifier: GLenum, name: GLuint, bufSize: GLsizei, length: *mut GLsizei, label: *mut GLchar);
     func!(GetObjectLabelEXT, (), type_: GLenum, object: GLuint, bufSize: GLsizei, length: *mut GLsizei, label: *mut GLchar);
     func!(GetObjectParameterfvARB, (), obj: GLhandleARB, pname: GLenum, params: *mut GLfloat);
     func!(GetObjectParameterivAPPLE, (), objectType: GLenum, name: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetObjectParameterivARB, (), obj: GLhandleARB, pname: GLenum, params: *mut GLint);
     func!(GetObjectPtrLabel, (), ptr: *const c_void, bufSize: GLsizei, length: *mut GLsizei, label: *mut GLchar);
     func!(GetOcclusionQueryivNV, (), id: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetOcclusionQueryuivNV, (), id: GLuint, pname: GLenum, params: *mut GLuint);
     func!(GetPathCommandsNV, (), path: GLuint, commands: *mut GLubyte);
     func!(GetPathCoordsNV, (), path: GLuint, coords: *mut GLfloat);
     func!(GetPathDashArrayNV, (), path: GLuint, dashArray: *mut GLfloat);
     func!(GetPathLengthNV, GLfloat, path: GLuint, startSegment: GLsizei, numSegments: GLsizei);
     func!(GetPathMetricRangeNV, (), metricQueryMask: GLbitfield, firstPathName: GLuint, numPaths: GLsizei, stride: GLsizei, metrics: *mut GLfloat);
     func!(GetPathMetricsNV, (), metricQueryMask: GLbitfield, numPaths: GLsizei, pathNameType: GLenum, paths: *const c_void, pathBase: GLuint, stride: GLsizei, metrics: *mut GLfloat);
     func!(GetPathParameterfvNV, (), path: GLuint, pname: GLenum, value: *mut GLfloat);
     func!(GetPathParameterivNV, (), path: GLuint, pname: GLenum, value: *mut GLint);
     func!(GetPathSpacingNV, (), pathListMode: GLenum, numPaths: GLsizei, pathNameType: GLenum, paths: *const c_void, pathBase: GLuint, advanceScale: GLfloat, kerningScale: GLfloat, transformType: GLenum, returnedSpacing: *mut GLfloat);
     func!(GetPerfCounterInfoINTEL, (), queryId: GLuint, counterId: GLuint, counterNameLength: GLuint, counterName: *mut GLchar, counterDescLength: GLuint, counterDesc: *mut GLchar, counterOffset: *mut GLuint, counterDataSize: *mut GLuint, counterTypeEnum: *mut GLuint, counterDataTypeEnum: *mut GLuint, rawCounterMaxValue: *mut GLuint64);
     func!(GetPerfMonitorCounterDataAMD, (), monitor: GLuint, pname: GLenum, dataSize: GLsizei, data: *mut GLuint, bytesWritten: *mut GLint);
     func!(GetPerfMonitorCounterInfoAMD, (), group: GLuint, counter: GLuint, pname: GLenum, data: *mut c_void);
     func!(GetPerfMonitorCounterStringAMD, (), group: GLuint, counter: GLuint, bufSize: GLsizei, length: *mut GLsizei, counterString: *mut GLchar);
     func!(GetPerfMonitorCountersAMD, (), group: GLuint, numCounters: *mut GLint, maxActiveCounters: *mut GLint, counterSize: GLsizei, counters: *mut GLuint);
     func!(GetPerfMonitorGroupStringAMD, (), group: GLuint, bufSize: GLsizei, length: *mut GLsizei, groupString: *mut GLchar);
     func!(GetPerfMonitorGroupsAMD, (), numGroups: *mut GLint, groupsSize: GLsizei, groups: *mut GLuint);
     func!(GetPerfQueryDataINTEL, (), queryHandle: GLuint, flags: GLuint, dataSize: GLsizei, data: *mut c_void, bytesWritten: *mut GLuint);
     func!(GetPerfQueryIdByNameINTEL, (), queryName: *mut GLchar, queryId: *mut GLuint);
     func!(GetPerfQueryInfoINTEL, (), queryId: GLuint, queryNameLength: GLuint, queryName: *mut GLchar, dataSize: *mut GLuint, noCounters: *mut GLuint, noInstances: *mut GLuint, capsMask: *mut GLuint);
     func!(GetPixelMapxv, (), map: GLenum, size: GLint, values: *mut GLfixed);
     func!(GetPixelTexGenParameterfvSGIS, (), pname: GLenum, params: *mut GLfloat);
     func!(GetPixelTexGenParameterivSGIS, (), pname: GLenum, params: *mut GLint);
     func!(GetPixelTransformParameterfvEXT, (), target: GLenum, pname: GLenum, params: *mut GLfloat);
     func!(GetPixelTransformParameterivEXT, (), target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetPointerIndexedvEXT, (), target: GLenum, index: GLuint, data: *mut *mut c_void);
     func!(GetPointeri_vEXT, (), pname: GLenum, index: GLuint, params: *mut *mut c_void);
     func!(GetPointerv, (), pname: GLenum, params: *mut *mut c_void);
     func!(GetPointervEXT, (), pname: GLenum, params: *mut *mut c_void);
     func!(GetProgramBinary, (), program: GLuint, bufSize: GLsizei, length: *mut GLsizei, binaryFormat: *mut GLenum, binary: *mut c_void);
     func!(GetProgramEnvParameterIivNV, (), target: GLenum, index: GLuint, params: *mut GLint);
     func!(GetProgramEnvParameterIuivNV, (), target: GLenum, index: GLuint, params: *mut GLuint);
     func!(GetProgramEnvParameterdvARB, (), target: GLenum, index: GLuint, params: *mut GLdouble);
     func!(GetProgramEnvParameterfvARB, (), target: GLenum, index: GLuint, params: *mut GLfloat);
     func!(GetProgramInfoLog, (), program: GLuint, bufSize: GLsizei, length: *mut GLsizei, infoLog: *mut GLchar);
     func!(GetProgramInterfaceiv, (), program: GLuint, programInterface: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetProgramLocalParameterIivNV, (), target: GLenum, index: GLuint, params: *mut GLint);
     func!(GetProgramLocalParameterIuivNV, (), target: GLenum, index: GLuint, params: *mut GLuint);
     func!(GetProgramLocalParameterdvARB, (), target: GLenum, index: GLuint, params: *mut GLdouble);
     func!(GetProgramLocalParameterfvARB, (), target: GLenum, index: GLuint, params: *mut GLfloat);
     func!(GetProgramNamedParameterdvNV, (), id: GLuint, len: GLsizei, name: *const GLubyte, params: *mut GLdouble);
     func!(GetProgramNamedParameterfvNV, (), id: GLuint, len: GLsizei, name: *const GLubyte, params: *mut GLfloat);
     func!(GetProgramParameterdvNV, (), target: GLenum, index: GLuint, pname: GLenum, params: *mut GLdouble);
     func!(GetProgramParameterfvNV, (), target: GLenum, index: GLuint, pname: GLenum, params: *mut GLfloat);
     func!(GetProgramPipelineInfoLog, (), pipeline: GLuint, bufSize: GLsizei, length: *mut GLsizei, infoLog: *mut GLchar);
     func!(GetProgramPipelineiv, (), pipeline: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetProgramResourceIndex, GLuint, program: GLuint, programInterface: GLenum, name: *const GLchar);
     func!(GetProgramResourceLocation, GLint, program: GLuint, programInterface: GLenum, name: *const GLchar);
     func!(GetProgramResourceLocationIndex, GLint, program: GLuint, programInterface: GLenum, name: *const GLchar);
     func!(GetProgramResourceName, (), program: GLuint, programInterface: GLenum, index: GLuint, bufSize: GLsizei, length: *mut GLsizei, name: *mut GLchar);
     func!(GetProgramResourcefvNV, (), program: GLuint, programInterface: GLenum, index: GLuint, propCount: GLsizei, props: *const GLenum, count: GLsizei, length: *mut GLsizei, params: *mut GLfloat);
     func!(GetProgramResourceiv, (), program: GLuint, programInterface: GLenum, index: GLuint, propCount: GLsizei, props: *const GLenum, count: GLsizei, length: *mut GLsizei, params: *mut GLint);
     func!(GetProgramStageiv, (), program: GLuint, shadertype: GLenum, pname: GLenum, values: *mut GLint);
     func!(GetProgramStringARB, (), target: GLenum, pname: GLenum, string: *mut c_void);
     func!(GetProgramStringNV, (), id: GLuint, pname: GLenum, program: *mut GLubyte);
     func!(GetProgramSubroutineParameteruivNV, (), target: GLenum, index: GLuint, param: *mut GLuint);
     func!(GetProgramiv, (), program: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetProgramivARB, (), target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetProgramivNV, (), id: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetQueryBufferObjecti64v, (), id: GLuint, buffer: GLuint, pname: GLenum, offset: GLintptr);
     func!(GetQueryBufferObjectiv, (), id: GLuint, buffer: GLuint, pname: GLenum, offset: GLintptr);
     func!(GetQueryBufferObjectui64v, (), id: GLuint, buffer: GLuint, pname: GLenum, offset: GLintptr);
     func!(GetQueryBufferObjectuiv, (), id: GLuint, buffer: GLuint, pname: GLenum, offset: GLintptr);
     func!(GetQueryIndexediv, (), target: GLenum, index: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetQueryObjecti64v, (), id: GLuint, pname: GLenum, params: *mut GLint64);
     func!(GetQueryObjecti64vEXT, (), id: GLuint, pname: GLenum, params: *mut GLint64);
     func!(GetQueryObjectiv, (), id: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetQueryObjectivARB, (), id: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetQueryObjectui64v, (), id: GLuint, pname: GLenum, params: *mut GLuint64);
     func!(GetQueryObjectui64vEXT, (), id: GLuint, pname: GLenum, params: *mut GLuint64);
     func!(GetQueryObjectuiv, (), id: GLuint, pname: GLenum, params: *mut GLuint);
     func!(GetQueryObjectuivARB, (), id: GLuint, pname: GLenum, params: *mut GLuint);
     func!(GetQueryiv, (), target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetQueryivARB, (), target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetRenderbufferParameteriv, (), target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetRenderbufferParameterivEXT, (), target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetSamplerParameterIiv, (), sampler: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetSamplerParameterIuiv, (), sampler: GLuint, pname: GLenum, params: *mut GLuint);
     func!(GetSamplerParameterfv, (), sampler: GLuint, pname: GLenum, params: *mut GLfloat);
     func!(GetSamplerParameteriv, (), sampler: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetSemaphoreParameterivNV, (), semaphore: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetSemaphoreParameterui64vEXT, (), semaphore: GLuint, pname: GLenum, params: *mut GLuint64);
     func!(GetSeparableFilterEXT, (), target: GLenum, format: GLenum, type_: GLenum, row: *mut c_void, column: *mut c_void, span: *mut c_void);
     func!(GetShaderInfoLog, (), shader: GLuint, bufSize: GLsizei, length: *mut GLsizei, infoLog: *mut GLchar);
     func!(GetShaderPrecisionFormat, (), shadertype: GLenum, precisiontype: GLenum, range: *mut GLint, precision: *mut GLint);
     func!(GetShaderSource, (), shader: GLuint, bufSize: GLsizei, length: *mut GLsizei, source: *mut GLchar);
     func!(GetShaderSourceARB, (), obj: GLhandleARB, maxLength: GLsizei, length: *mut GLsizei, source: *mut GLcharARB);
     func!(GetShaderiv, (), shader: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetShadingRateImagePaletteNV, (), viewport: GLuint, entry: GLuint, rate: *mut GLenum);
     func!(GetShadingRateSampleLocationivNV, (), rate: GLenum, samples: GLuint, index: GLuint, location: *mut GLint);
     func!(GetSharpenTexFuncSGIS, (), target: GLenum, points: *mut GLfloat);
     func!(GetStageIndexNV, GLushort, shadertype: GLenum);
     func!(GetString, *const GLubyte, name: GLenum);
     func!(GetStringi, *const GLubyte, name: GLenum, index: GLuint);
     func!(GetSubroutineIndex, GLuint, program: GLuint, shadertype: GLenum, name: *const GLchar);
     func!(GetSubroutineUniformLocation, GLint, program: GLuint, shadertype: GLenum, name: *const GLchar);
     func!(GetSynciv, (), sync: GLsync, pname: GLenum, count: GLsizei, length: *mut GLsizei, values: *mut GLint);
     func!(GetTexBumpParameterfvATI, (), pname: GLenum, param: *mut GLfloat);
     func!(GetTexBumpParameterivATI, (), pname: GLenum, param: *mut GLint);
     func!(GetTexEnvxvOES, (), target: GLenum, pname: GLenum, params: *mut GLfixed);
     func!(GetTexFilterFuncSGIS, (), target: GLenum, filter: GLenum, weights: *mut GLfloat);
     func!(GetTexGenxvOES, (), coord: GLenum, pname: GLenum, params: *mut GLfixed);
     func!(GetTexImage, (), target: GLenum, level: GLint, format: GLenum, type_: GLenum, pixels: *mut c_void);
     func!(GetTexLevelParameterfv, (), target: GLenum, level: GLint, pname: GLenum, params: *mut GLfloat);
     func!(GetTexLevelParameteriv, (), target: GLenum, level: GLint, pname: GLenum, params: *mut GLint);
     func!(GetTexLevelParameterxvOES, (), target: GLenum, level: GLint, pname: GLenum, params: *mut GLfixed);
     func!(GetTexParameterIiv, (), target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetTexParameterIivEXT, (), target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetTexParameterIuiv, (), target: GLenum, pname: GLenum, params: *mut GLuint);
     func!(GetTexParameterIuivEXT, (), target: GLenum, pname: GLenum, params: *mut GLuint);
     func!(GetTexParameterPointervAPPLE, (), target: GLenum, pname: GLenum, params: *mut *mut c_void);
     func!(GetTexParameterfv, (), target: GLenum, pname: GLenum, params: *mut GLfloat);
     func!(GetTexParameteriv, (), target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetTexParameterxvOES, (), target: GLenum, pname: GLenum, params: *mut GLfixed);
     func!(GetTextureHandleARB, GLuint64, texture: GLuint);
     func!(GetTextureHandleNV, GLuint64, texture: GLuint);
     func!(GetTextureImage, (), texture: GLuint, level: GLint, format: GLenum, type_: GLenum, bufSize: GLsizei, pixels: *mut c_void);
     func!(GetTextureImageEXT, (), texture: GLuint, target: GLenum, level: GLint, format: GLenum, type_: GLenum, pixels: *mut c_void);
     func!(GetTextureLevelParameterfv, (), texture: GLuint, level: GLint, pname: GLenum, params: *mut GLfloat);
     func!(GetTextureLevelParameterfvEXT, (), texture: GLuint, target: GLenum, level: GLint, pname: GLenum, params: *mut GLfloat);
     func!(GetTextureLevelParameteriv, (), texture: GLuint, level: GLint, pname: GLenum, params: *mut GLint);
     func!(GetTextureLevelParameterivEXT, (), texture: GLuint, target: GLenum, level: GLint, pname: GLenum, params: *mut GLint);
     func!(GetTextureParameterIiv, (), texture: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetTextureParameterIivEXT, (), texture: GLuint, target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetTextureParameterIuiv, (), texture: GLuint, pname: GLenum, params: *mut GLuint);
     func!(GetTextureParameterIuivEXT, (), texture: GLuint, target: GLenum, pname: GLenum, params: *mut GLuint);
     func!(GetTextureParameterfv, (), texture: GLuint, pname: GLenum, params: *mut GLfloat);
     func!(GetTextureParameterfvEXT, (), texture: GLuint, target: GLenum, pname: GLenum, params: *mut GLfloat);
     func!(GetTextureParameteriv, (), texture: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetTextureParameterivEXT, (), texture: GLuint, target: GLenum, pname: GLenum, params: *mut GLint);
     func!(GetTextureSamplerHandleARB, GLuint64, texture: GLuint, sampler: GLuint);
     func!(GetTextureSamplerHandleNV, GLuint64, texture: GLuint, sampler: GLuint);
     func!(GetTextureSubImage, (), texture: GLuint, level: GLint, xoffset: GLint, yoffset: GLint, zoffset: GLint, width: GLsizei, height: GLsizei, depth: GLsizei, format: GLenum, type_: GLenum, bufSize: GLsizei, pixels: *mut c_void);
     func!(GetTrackMatrixivNV, (), target: GLenum, address: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetTransformFeedbackVarying, (), program: GLuint, index: GLuint, bufSize: GLsizei, length: *mut GLsizei, size: *mut GLsizei, type_: *mut GLenum, name: *mut GLchar);
     func!(GetTransformFeedbackVaryingEXT, (), program: GLuint, index: GLuint, bufSize: GLsizei, length: *mut GLsizei, size: *mut GLsizei, type_: *mut GLenum, name: *mut GLchar);
     func!(GetTransformFeedbackVaryingNV, (), program: GLuint, index: GLuint, location: *mut GLint);
     func!(GetTransformFeedbacki64_v, (), xfb: GLuint, pname: GLenum, index: GLuint, param: *mut GLint64);
     func!(GetTransformFeedbacki_v, (), xfb: GLuint, pname: GLenum, index: GLuint, param: *mut GLint);
     func!(GetTransformFeedbackiv, (), xfb: GLuint, pname: GLenum, param: *mut GLint);
     func!(GetUniformBlockIndex, GLuint, program: GLuint, uniformBlockName: *const GLchar);
     func!(GetUniformBufferSizeEXT, GLint, program: GLuint, location: GLint);
     func!(GetUniformIndices, (), program: GLuint, uniformCount: GLsizei, uniformNames: *const *const GLchar, uniformIndices: *mut GLuint);
     func!(GetUniformLocation, GLint, program: GLuint, name: *const GLchar);
     func!(GetUniformLocationARB, GLint, programObj: GLhandleARB, name: *const GLcharARB);
     func!(GetUniformOffsetEXT, GLintptr, program: GLuint, location: GLint);
     func!(GetUniformSubroutineuiv, (), shadertype: GLenum, location: GLint, params: *mut GLuint);
     func!(GetUniformdv, (), program: GLuint, location: GLint, params: *mut GLdouble);
     func!(GetUniformfv, (), program: GLuint, location: GLint, params: *mut GLfloat);
     func!(GetUniformfvARB, (), programObj: GLhandleARB, location: GLint, params: *mut GLfloat);
     func!(GetUniformi64vARB, (), program: GLuint, location: GLint, params: *mut GLint64);
     func!(GetUniformi64vNV, (), program: GLuint, location: GLint, params: *mut GLint64EXT);
     func!(GetUniformiv, (), program: GLuint, location: GLint, params: *mut GLint);
     func!(GetUniformivARB, (), programObj: GLhandleARB, location: GLint, params: *mut GLint);
     func!(GetUniformui64vARB, (), program: GLuint, location: GLint, params: *mut GLuint64);
     func!(GetUniformui64vNV, (), program: GLuint, location: GLint, params: *mut GLuint64EXT);
     func!(GetUniformuiv, (), program: GLuint, location: GLint, params: *mut GLuint);
     func!(GetUniformuivEXT, (), program: GLuint, location: GLint, params: *mut GLuint);
     func!(GetUnsignedBytei_vEXT, (), target: GLenum, index: GLuint, data: *mut GLubyte);
     func!(GetUnsignedBytevEXT, (), pname: GLenum, data: *mut GLubyte);
     func!(GetVariantArrayObjectfvATI, (), id: GLuint, pname: GLenum, params: *mut GLfloat);
     func!(GetVariantArrayObjectivATI, (), id: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetVariantBooleanvEXT, (), id: GLuint, value: GLenum, data: *mut GLboolean);
     func!(GetVariantFloatvEXT, (), id: GLuint, value: GLenum, data: *mut GLfloat);
     func!(GetVariantIntegervEXT, (), id: GLuint, value: GLenum, data: *mut GLint);
     func!(GetVariantPointervEXT, (), id: GLuint, value: GLenum, data: *mut *mut c_void);
     func!(GetVaryingLocationNV, GLint, program: GLuint, name: *const GLchar);
     func!(GetVertexArrayIndexed64iv, (), vaobj: GLuint, index: GLuint, pname: GLenum, param: *mut GLint64);
     func!(GetVertexArrayIndexediv, (), vaobj: GLuint, index: GLuint, pname: GLenum, param: *mut GLint);
     func!(GetVertexArrayIntegeri_vEXT, (), vaobj: GLuint, index: GLuint, pname: GLenum, param: *mut GLint);
     func!(GetVertexArrayIntegervEXT, (), vaobj: GLuint, pname: GLenum, param: *mut GLint);
     func!(GetVertexArrayPointeri_vEXT, (), vaobj: GLuint, index: GLuint, pname: GLenum, param: *mut *mut c_void);
     func!(GetVertexArrayPointervEXT, (), vaobj: GLuint, pname: GLenum, param: *mut *mut c_void);
     func!(GetVertexArrayiv, (), vaobj: GLuint, pname: GLenum, param: *mut GLint);
     func!(GetVertexAttribArrayObjectfvATI, (), index: GLuint, pname: GLenum, params: *mut GLfloat);
     func!(GetVertexAttribArrayObjectivATI, (), index: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetVertexAttribIiv, (), index: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetVertexAttribIivEXT, (), index: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetVertexAttribIuiv, (), index: GLuint, pname: GLenum, params: *mut GLuint);
     func!(GetVertexAttribIuivEXT, (), index: GLuint, pname: GLenum, params: *mut GLuint);
     func!(GetVertexAttribLdv, (), index: GLuint, pname: GLenum, params: *mut GLdouble);
     func!(GetVertexAttribLdvEXT, (), index: GLuint, pname: GLenum, params: *mut GLdouble);
     func!(GetVertexAttribLi64vNV, (), index: GLuint, pname: GLenum, params: *mut GLint64EXT);
     func!(GetVertexAttribLui64vARB, (), index: GLuint, pname: GLenum, params: *mut GLuint64EXT);
     func!(GetVertexAttribLui64vNV, (), index: GLuint, pname: GLenum, params: *mut GLuint64EXT);
     func!(GetVertexAttribPointerv, (), index: GLuint, pname: GLenum, pointer: *mut *mut c_void);
     func!(GetVertexAttribPointervARB, (), index: GLuint, pname: GLenum, pointer: *mut *mut c_void);
     func!(GetVertexAttribPointervNV, (), index: GLuint, pname: GLenum, pointer: *mut *mut c_void);
     func!(GetVertexAttribdv, (), index: GLuint, pname: GLenum, params: *mut GLdouble);
     func!(GetVertexAttribdvARB, (), index: GLuint, pname: GLenum, params: *mut GLdouble);
     func!(GetVertexAttribdvNV, (), index: GLuint, pname: GLenum, params: *mut GLdouble);
     func!(GetVertexAttribfv, (), index: GLuint, pname: GLenum, params: *mut GLfloat);
     func!(GetVertexAttribfvARB, (), index: GLuint, pname: GLenum, params: *mut GLfloat);
     func!(GetVertexAttribfvNV, (), index: GLuint, pname: GLenum, params: *mut GLfloat);
     func!(GetVertexAttribiv, (), index: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetVertexAttribivARB, (), index: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetVertexAttribivNV, (), index: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetVideoCaptureStreamdvNV, (), video_capture_slot: GLuint, stream: GLuint, pname: GLenum, params: *mut GLdouble);
     func!(GetVideoCaptureStreamfvNV, (), video_capture_slot: GLuint, stream: GLuint, pname: GLenum, params: *mut GLfloat);
     func!(GetVideoCaptureStreamivNV, (), video_capture_slot: GLuint, stream: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetVideoCaptureivNV, (), video_capture_slot: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetVideoi64vNV, (), video_slot: GLuint, pname: GLenum, params: *mut GLint64EXT);
     func!(GetVideoivNV, (), video_slot: GLuint, pname: GLenum, params: *mut GLint);
     func!(GetVideoui64vNV, (), video_slot: GLuint, pname: GLenum, params: *mut GLuint64EXT);
     func!(GetVideouivNV, (), video_slot: GLuint, pname: GLenum, params: *mut GLuint);
     func!(GetVkProcAddrNV, GLVULKANPROCNV, name: *const GLchar);
     func!(GetnCompressedTexImage, (), target: GLenum, lod: GLint, bufSize: GLsizei, pixels: *mut c_void);
     func!(GetnCompressedTexImageARB, (), target: GLenum, lod: GLint, bufSize: GLsizei, img: *mut c_void);
     func!(GetnTexImage, (), target: GLenum, level: GLint, format: GLenum, type_: GLenum, bufSize: GLsizei, pixels: *mut c_void);
     func!(GetnTexImageARB, (), target: GLenum, level: GLint, format: GLenum, type_: GLenum, bufSize: GLsizei, img: *mut c_void);
     func!(GetnUniformdv, (), program: GLuint, location: GLint, bufSize: GLsizei, params: *mut GLdouble);
     func!(GetnUniformdvARB, (), program: GLuint, location: GLint, bufSize: GLsizei, params: *mut GLdouble);
     func!(GetnUniformfv, (), program: GLuint, location: GLint, bufSize: GLsizei, params: *mut GLfloat);
     func!(GetnUniformfvARB, (), program: GLuint, location: GLint, bufSize: GLsizei, params: *mut GLfloat);
     func!(GetnUniformi64vARB, (), program: GLuint, location: GLint, bufSize: GLsizei, params: *mut GLint64);
     func!(GetnUniformiv, (), program: GLuint, location: GLint, bufSize: GLsizei, params: *mut GLint);
     func!(GetnUniformivARB, (), program: GLuint, location: GLint, bufSize: GLsizei, params: *mut GLint);
     func!(GetnUniformui64vARB, (), program: GLuint, location: GLint, bufSize: GLsizei, params: *mut GLuint64);
     func!(GetnUniformuiv, (), program: GLuint, location: GLint, bufSize: GLsizei, params: *mut GLuint);
     func!(GetnUniformuivARB, (), program: GLuint, location: GLint, bufSize: GLsizei, params: *mut GLuint);
     func!(GlobalAlphaFactorbSUN, (), factor: GLbyte);
     func!(GlobalAlphaFactordSUN, (), factor: GLdouble);
     func!(GlobalAlphaFactorfSUN, (), factor: GLfloat);
     func!(GlobalAlphaFactoriSUN, (), factor: GLint);
     func!(GlobalAlphaFactorsSUN, (), factor: GLshort);
     func!(GlobalAlphaFactorubSUN, (), factor: GLubyte);
     func!(GlobalAlphaFactoruiSUN, (), factor: GLuint);
     func!(GlobalAlphaFactorusSUN, (), factor: GLushort);
     func!(Hint, (), target: GLenum, mode: GLenum);
     func!(HintPGI, (), target: GLenum, mode: GLint);
     func!(HistogramEXT, (), target: GLenum, width: GLsizei, internalformat: GLenum, sink: GLboolean);
     func!(IglooInterfaceSGIX, (), pname: GLenum, params: *const c_void);
     func!(ImageTransformParameterfHP, (), target: GLenum, pname: GLenum, param: GLfloat);
     func!(ImageTransformParameterfvHP, (), target: GLenum, pname: GLenum, params: *const GLfloat);
     func!(ImageTransformParameteriHP, (), target: GLenum, pname: GLenum, param: GLint);
     func!(ImageTransformParameterivHP, (), target: GLenum, pname: GLenum, params: *const GLint);
     func!(ImportMemoryFdEXT, (), memory: GLuint, size: GLuint64, handleType: GLenum, fd: GLint);
     func!(ImportMemoryWin32HandleEXT, (), memory: GLuint, size: GLuint64, handleType: GLenum, handle: *mut c_void);
     func!(ImportMemoryWin32NameEXT, (), memory: GLuint, size: GLuint64, handleType: GLenum, name: *const c_void);
     func!(ImportSemaphoreFdEXT, (), semaphore: GLuint, handleType: GLenum, fd: GLint);
     func!(ImportSemaphoreWin32HandleEXT, (), semaphore: GLuint, handleType: GLenum, handle: *mut c_void);
     func!(ImportSemaphoreWin32NameEXT, (), semaphore: GLuint, handleType: GLenum, name: *const c_void);
     func!(ImportSyncEXT, GLsync, external_sync_type: GLenum, external_sync: GLintptr, flags: GLbitfield);
     func!(IndexFormatNV, (), type_: GLenum, stride: GLsizei);
     func!(IndexFuncEXT, (), func: GLenum, ref_: GLclampf);
     func!(IndexMaterialEXT, (), face: GLenum, mode: GLenum);
     func!(IndexPointerEXT, (), type_: GLenum, stride: GLsizei, count: GLsizei, pointer: *const c_void);
     func!(IndexPointerListIBM, (), type_: GLenum, stride: GLint, pointer: *const *const c_void, ptrstride: GLint);
     func!(IndexxOES, (), component: GLfixed);
     func!(IndexxvOES, (), component: *const GLfixed);
     func!(InsertComponentEXT, (), res: GLuint, src: GLuint, num: GLuint);
     func!(InsertEventMarkerEXT, (), length: GLsizei, marker: *const GLchar);
     func!(InstrumentsBufferSGIX, (), size: GLsizei, buffer: *mut GLint);
     func!(InterpolatePathsNV, (), resultPath: GLuint, pathA: GLuint, pathB: GLuint, weight: GLfloat);
     func!(InvalidateBufferData, (), buffer: GLuint);
     func!(InvalidateBufferSubData, (), buffer: GLuint, offset: GLintptr, length: GLsizeiptr);
     func!(InvalidateFramebuffer, (), target: GLenum, numAttachments: GLsizei, attachments: *const GLenum);
     func!(InvalidateNamedFramebufferData, (), framebuffer: GLuint, numAttachments: GLsizei, attachments: *const GLenum);
     func!(InvalidateNamedFramebufferSubData, (), framebuffer: GLuint, numAttachments: GLsizei, attachments: *const GLenum, x: GLint, y: GLint, width: GLsizei, height: GLsizei);
     func!(InvalidateSubFramebuffer, (), target: GLenum, numAttachments: GLsizei, attachments: *const GLenum, x: GLint, y: GLint, width: GLsizei, height: GLsizei);
     func!(InvalidateTexImage, (), texture: GLuint, level: GLint);
     func!(InvalidateTexSubImage, (), texture: GLuint, level: GLint, xoffset: GLint, yoffset: GLint, zoffset: GLint, width: GLsizei, height: GLsizei, depth: GLsizei);
     func!(IsAsyncMarkerSGIX, GLboolean, marker: GLuint);
     func!(IsBuffer, GLboolean, buffer: GLuint);
     func!(IsBufferARB, GLboolean, buffer: GLuint);
     func!(IsBufferResidentNV, GLboolean, target: GLenum);
     func!(IsCommandListNV, GLboolean, list: GLuint);
     func!(IsEnabled, GLboolean, cap: GLenum);
     func!(IsEnabledIndexedEXT, GLboolean, target: GLenum, index: GLuint);
     func!(IsEnabledi, GLboolean, target: GLenum, index: GLuint);
     func!(IsFenceAPPLE, GLboolean, fence: GLuint);
     func!(IsFenceNV, GLboolean, fence: GLuint);
     func!(IsFramebuffer, GLboolean, framebuffer: GLuint);
     func!(IsFramebufferEXT, GLboolean, framebuffer: GLuint);
     func!(IsImageHandleResidentARB, GLboolean, handle: GLuint64);
     func!(IsImageHandleResidentNV, GLboolean, handle: GLuint64);
     func!(IsMemoryObjectEXT, GLboolean, memoryObject: GLuint);
     func!(IsNameAMD, GLboolean, identifier: GLenum, name: GLuint);
     func!(IsNamedBufferResidentNV, GLboolean, buffer: GLuint);
     func!(IsNamedStringARB, GLboolean, namelen: GLint, name: *const GLchar);
     func!(IsObjectBufferATI, GLboolean, buffer: GLuint);
     func!(IsOcclusionQueryNV, GLboolean, id: GLuint);
     func!(IsPathNV, GLboolean, path: GLuint);
     func!(IsPointInFillPathNV, GLboolean, path: GLuint, mask: GLuint, x: GLfloat, y: GLfloat);
     func!(IsPointInStrokePathNV, GLboolean, path: GLuint, x: GLfloat, y: GLfloat);
     func!(IsProgram, GLboolean, program: GLuint);
     func!(IsProgramARB, GLboolean, program: GLuint);
     func!(IsProgramNV, GLboolean, id: GLuint);
     func!(IsProgramPipeline, GLboolean, pipeline: GLuint);
     func!(IsQuery, GLboolean, id: GLuint);
     func!(IsQueryARB, GLboolean, id: GLuint);
     func!(IsRenderbuffer, GLboolean, renderbuffer: GLuint);
     func!(IsRenderbufferEXT, GLboolean, renderbuffer: GLuint);
     func!(IsSampler, GLboolean, sampler: GLuint);
     func!(IsSemaphoreEXT, GLboolean, semaphore: GLuint);
     func!(IsShader, GLboolean, shader: GLuint);
     func!(IsStateNV, GLboolean, state: GLuint);
     func!(IsSync, GLboolean, sync: GLsync);
     func!(IsTexture, GLboolean, texture: GLuint);
     func!(IsTextureEXT, GLboolean, texture: GLuint);
     func!(IsTextureHandleResidentARB, GLboolean, handle: GLuint64);
     func!(IsTextureHandleResidentNV, GLboolean, handle: GLuint64);
     func!(IsTransformFeedback, GLboolean, id: GLuint);
     func!(IsTransformFeedbackNV, GLboolean, id: GLuint);
     func!(IsVariantEnabledEXT, GLboolean, id: GLuint, cap: GLenum);
     func!(IsVertexArray, GLboolean, array: GLuint);
     func!(IsVertexArrayAPPLE, GLboolean, array: GLuint);
     func!(IsVertexAttribEnabledAPPLE, GLboolean, index: GLuint, pname: GLenum);
     func!(LGPUCopyImageSubDataNVX, (), sourceGpu: GLuint, destinationGpuMask: GLbitfield, srcName: GLuint, srcTarget: GLenum, srcLevel: GLint, srcX: GLint, srxY: GLint, srcZ: GLint, dstName: GLuint, dstTarget: GLenum, dstLevel: GLint, dstX: GLint, dstY: GLint, dstZ: GLint, width: GLsizei, height: GLsizei, depth: GLsizei);
     func!(LGPUInterlockNVX, (), );
     func!(LGPUNamedBufferSubDataNVX, (), gpuMask: GLbitfield, buffer: GLuint, offset: GLintptr, size: GLsizeiptr, data: *const c_void);
     func!(LabelObjectEXT, (), type_: GLenum, object: GLuint, length: GLsizei, label: *const GLchar);
     func!(LightEnviSGIX, (), pname: GLenum, param: GLint);
     func!(LightModelxOES, (), pname: GLenum, param: GLfixed);
     func!(LightModelxvOES, (), pname: GLenum, param: *const GLfixed);
     func!(LightxOES, (), light: GLenum, pname: GLenum, param: GLfixed);
     func!(LightxvOES, (), light: GLenum, pname: GLenum, params: *const GLfixed);
     func!(LineWidth, (), width: GLfloat);
     func!(LineWidthxOES, (), width: GLfixed);
     func!(LinkProgram, (), program: GLuint);
     func!(LinkProgramARB, (), programObj: GLhandleARB);
     func!(ListDrawCommandsStatesClientNV, (), list: GLuint, segment: GLuint, indirects: *const *const c_void, sizes: *const GLsizei, states: *const GLuint, fbos: *const GLuint, count: GLuint);
     func!(ListParameterfSGIX, (), list: GLuint, pname: GLenum, param: GLfloat);
     func!(ListParameterfvSGIX, (), list: GLuint, pname: GLenum, params: *const GLfloat);
     func!(ListParameteriSGIX, (), list: GLuint, pname: GLenum, param: GLint);
     func!(ListParameterivSGIX, (), list: GLuint, pname: GLenum, params: *const GLint);
     func!(LoadIdentityDeformationMapSGIX, (), mask: GLbitfield);
     func!(LoadMatrixxOES, (), m: *const GLfixed);
     func!(LoadProgramNV, (), target: GLenum, id: GLuint, len: GLsizei, program: *const GLubyte);
     func!(LoadTransposeMatrixdARB, (), m: *const GLdouble);
     func!(LoadTransposeMatrixfARB, (), m: *const GLfloat);
     func!(LoadTransposeMatrixxOES, (), m: *const GLfixed);
     func!(LockArraysEXT, (), first: GLint, count: GLsizei);
     func!(LogicOp, (), opcode: GLenum);
     func!(MakeBufferNonResidentNV, (), target: GLenum);
     func!(MakeBufferResidentNV, (), target: GLenum, access: GLenum);
     func!(MakeImageHandleNonResidentARB, (), handle: GLuint64);
     func!(MakeImageHandleNonResidentNV, (), handle: GLuint64);
     func!(MakeImageHandleResidentARB, (), handle: GLuint64, access: GLenum);
     func!(MakeImageHandleResidentNV, (), handle: GLuint64, access: GLenum);
     func!(MakeNamedBufferNonResidentNV, (), buffer: GLuint);
     func!(MakeNamedBufferResidentNV, (), buffer: GLuint, access: GLenum);
     func!(MakeTextureHandleNonResidentARB, (), handle: GLuint64);
     func!(MakeTextureHandleNonResidentNV, (), handle: GLuint64);
     func!(MakeTextureHandleResidentARB, (), handle: GLuint64);
     func!(MakeTextureHandleResidentNV, (), handle: GLuint64);
     func!(Map1xOES, (), target: GLenum, u1: GLfixed, u2: GLfixed, stride: GLint, order: GLint, points: GLfixed);
     func!(Map2xOES, (), target: GLenum, u1: GLfixed, u2: GLfixed, ustride: GLint, uorder: GLint, v1: GLfixed, v2: GLfixed, vstride: GLint, vorder: GLint, points: GLfixed);
     func!(MapBuffer, *mut c_void, target: GLenum, access: GLenum);
     func!(MapBufferARB, *mut c_void, target: GLenum, access: GLenum);
     func!(MapBufferRange, *mut c_void, target: GLenum, offset: GLintptr, length: GLsizeiptr, access: GLbitfield);
     func!(MapControlPointsNV, (), target: GLenum, index: GLuint, type_: GLenum, ustride: GLsizei, vstride: GLsizei, uorder: GLint, vorder: GLint, packed: GLboolean, points: *const c_void);
     func!(MapGrid1xOES, (), n: GLint, u1: GLfixed, u2: GLfixed);
     func!(MapGrid2xOES, (), n: GLint, u1: GLfixed, u2: GLfixed, v1: GLfixed, v2: GLfixed);
     func!(MapNamedBuffer, *mut c_void, buffer: GLuint, access: GLenum);
     func!(MapNamedBufferEXT, *mut c_void, buffer: GLuint, access: GLenum);
     func!(MapNamedBufferRange, *mut c_void, buffer: GLuint, offset: GLintptr, length: GLsizeiptr, access: GLbitfield);
     func!(MapNamedBufferRangeEXT, *mut c_void, buffer: GLuint, offset: GLintptr, length: GLsizeiptr, access: GLbitfield);
     func!(MapObjectBufferATI, *mut c_void, buffer: GLuint);
     func!(MapParameterfvNV, (), target: GLenum, pname: GLenum, params: *const GLfloat);
     func!(MapParameterivNV, (), target: GLenum, pname: GLenum, params: *const GLint);
     func!(MapTexture2DINTEL, *mut c_void, texture: GLuint, level: GLint, access: GLbitfield, stride: *mut GLint, layout: *mut GLenum);
     func!(MapVertexAttrib1dAPPLE, (), index: GLuint, size: GLuint, u1: GLdouble, u2: GLdouble, stride: GLint, order: GLint, points: *const GLdouble);
     func!(MapVertexAttrib1fAPPLE, (), index: GLuint, size: GLuint, u1: GLfloat, u2: GLfloat, stride: GLint, order: GLint, points: *const GLfloat);
     func!(MapVertexAttrib2dAPPLE, (), index: GLuint, size: GLuint, u1: GLdouble, u2: GLdouble, ustride: GLint, uorder: GLint, v1: GLdouble, v2: GLdouble, vstride: GLint, vorder: GLint, points: *const GLdouble);
     func!(MapVertexAttrib2fAPPLE, (), index: GLuint, size: GLuint, u1: GLfloat, u2: GLfloat, ustride: GLint, uorder: GLint, v1: GLfloat, v2: GLfloat, vstride: GLint, vorder: GLint, points: *const GLfloat);
     func!(MaterialxOES, (), face: GLenum, pname: GLenum, param: GLfixed);
     func!(MaterialxvOES, (), face: GLenum, pname: GLenum, param: *const GLfixed);
     func!(MatrixFrustumEXT, (), mode: GLenum, left: GLdouble, right: GLdouble, bottom: GLdouble, top: GLdouble, zNear: GLdouble, zFar: GLdouble);
     func!(MatrixIndexPointerARB, (), size: GLint, type_: GLenum, stride: GLsizei, pointer: *const c_void);
     func!(MatrixIndexubvARB, (), size: GLint, indices: *const GLubyte);
     func!(MatrixIndexuivARB, (), size: GLint, indices: *const GLuint);
     func!(MatrixIndexusvARB, (), size: GLint, indices: *const GLushort);
     func!(MatrixLoad3x2fNV, (), matrixMode: GLenum, m: *const GLfloat);
     func!(MatrixLoad3x3fNV, (), matrixMode: GLenum, m: *const GLfloat);
     func!(MatrixLoadIdentityEXT, (), mode: GLenum);
     func!(MatrixLoadTranspose3x3fNV, (), matrixMode: GLenum, m: *const GLfloat);
     func!(MatrixLoadTransposedEXT, (), mode: GLenum, m: *const GLdouble);
     func!(MatrixLoadTransposefEXT, (), mode: GLenum, m: *const GLfloat);
     func!(MatrixLoaddEXT, (), mode: GLenum, m: *const GLdouble);
     func!(MatrixLoadfEXT, (), mode: GLenum, m: *const GLfloat);
     func!(MatrixMult3x2fNV, (), matrixMode: GLenum, m: *const GLfloat);
     func!(MatrixMult3x3fNV, (), matrixMode: GLenum, m: *const GLfloat);
     func!(MatrixMultTranspose3x3fNV, (), matrixMode: GLenum, m: *const GLfloat);
     func!(MatrixMultTransposedEXT, (), mode: GLenum, m: *const GLdouble);
     func!(MatrixMultTransposefEXT, (), mode: GLenum, m: *const GLfloat);
     func!(MatrixMultdEXT, (), mode: GLenum, m: *const GLdouble);
     func!(MatrixMultfEXT, (), mode: GLenum, m: *const GLfloat);
     func!(MatrixOrthoEXT, (), mode: GLenum, left: GLdouble, right: GLdouble, bottom: GLdouble, top: GLdouble, zNear: GLdouble, zFar: GLdouble);
     func!(MatrixPopEXT, (), mode: GLenum);
     func!(MatrixPushEXT, (), mode: GLenum);
     func!(MatrixRotatedEXT, (), mode: GLenum, angle: GLdouble, x: GLdouble, y: GLdouble, z: GLdouble);
     func!(MatrixRotatefEXT, (), mode: GLenum, angle: GLfloat, x: GLfloat, y: GLfloat, z: GLfloat);
     func!(MatrixScaledEXT, (), mode: GLenum, x: GLdouble, y: GLdouble, z: GLdouble);
     func!(MatrixScalefEXT, (), mode: GLenum, x: GLfloat, y: GLfloat, z: GLfloat);
     func!(MatrixTranslatedEXT, (), mode: GLenum, x: GLdouble, y: GLdouble, z: GLdouble);
     func!(MatrixTranslatefEXT, (), mode: GLenum, x: GLfloat, y: GLfloat, z: GLfloat);
     func!(MaxShaderCompilerThreadsARB, (), count: GLuint);
     func!(MaxShaderCompilerThreadsKHR, (), count: GLuint);
     func!(MemoryBarrier, (), barriers: GLbitfield);
     func!(MemoryBarrierByRegion, (), barriers: GLbitfield);
     func!(MemoryBarrierEXT, (), barriers: GLbitfield);
     func!(MemoryObjectParameterivEXT, (), memoryObject: GLuint, pname: GLenum, params: *const GLint);
     func!(MinSampleShading, (), value: GLfloat);
     func!(MinSampleShadingARB, (), value: GLfloat);
     func!(MinmaxEXT, (), target: GLenum, internalformat: GLenum, sink: GLboolean);
     func!(MultMatrixxOES, (), m: *const GLfixed);
     func!(MultTransposeMatrixdARB, (), m: *const GLdouble);
     func!(MultTransposeMatrixfARB, (), m: *const GLfloat);
     func!(MultTransposeMatrixxOES, (), m: *const GLfixed);
     func!(MultiDrawArrays, (), mode: GLenum, first: *const GLint, count: *const GLsizei, drawcount: GLsizei);
     func!(MultiDrawArraysEXT, (), mode: GLenum, first: *const GLint, count: *const GLsizei, primcount: GLsizei);
     func!(MultiDrawArraysIndirect, (), mode: GLenum, indirect: *const c_void, drawcount: GLsizei, stride: GLsizei);
     func!(MultiDrawArraysIndirectAMD, (), mode: GLenum, indirect: *const c_void, primcount: GLsizei, stride: GLsizei);
     func!(MultiDrawArraysIndirectBindlessCountNV, (), mode: GLenum, indirect: *const c_void, drawCount: GLsizei, maxDrawCount: GLsizei, stride: GLsizei, vertexBufferCount: GLint);
     func!(MultiDrawArraysIndirectBindlessNV, (), mode: GLenum, indirect: *const c_void, drawCount: GLsizei, stride: GLsizei, vertexBufferCount: GLint);
     func!(MultiDrawArraysIndirectCount, (), mode: GLenum, indirect: *const c_void, drawcount: GLintptr, maxdrawcount: GLsizei, stride: GLsizei);
     func!(MultiDrawArraysIndirectCountARB, (), mode: GLenum, indirect: *const c_void, drawcount: GLintptr, maxdrawcount: GLsizei, stride: GLsizei);
     func!(MultiDrawElementArrayAPPLE, (), mode: GLenum, first: *const GLint, count: *const GLsizei, primcount: GLsizei);
     func!(MultiDrawElements, (), mode: GLenum, count: *const GLsizei, type_: GLenum, indices: *const *const c_void, drawcount: GLsizei);
     func!(MultiDrawElementsBaseVertex, (), mode: GLenum, count: *const GLsizei, type_: GLenum, indices: *const *const c_void, drawcount: GLsizei, basevertex: *const GLint);
     func!(MultiDrawElementsEXT, (), mode: GLenum, count: *const GLsizei, type_: GLenum, indices: *const *const c_void, primcount: GLsizei);
     func!(MultiDrawElementsIndirect, (), mode: GLenum, type_: GLenum, indirect: *const c_void, drawcount: GLsizei, stride: GLsizei);
     func!(MultiDrawElementsIndirectAMD, (), mode: GLenum, type_: GLenum, indirect: *const c_void, primcount: GLsizei, stride: GLsizei);
     func!(MultiDrawElementsIndirectBindlessCountNV, (), mode: GLenum, type_: GLenum, indirect: *const c_void, drawCount: GLsizei, maxDrawCount: GLsizei, stride: GLsizei, vertexBufferCount: GLint);
     func!(MultiDrawElementsIndirectBindlessNV, (), mode: GLenum, type_: GLenum, indirect: *const c_void, drawCount: GLsizei, stride: GLsizei, vertexBufferCount: GLint);
     func!(MultiDrawElementsIndirectCount, (), mode: GLenum, type_: GLenum, indirect: *const c_void, drawcount: GLintptr, maxdrawcount: GLsizei, stride: GLsizei);
     func!(MultiDrawElementsIndirectCountARB, (), mode: GLenum, type_: GLenum, indirect: *const c_void, drawcount: GLintptr, maxdrawcount: GLsizei, stride: GLsizei);
     func!(MultiDrawMeshTasksIndirectCountNV, (), indirect: GLintptr, drawcount: GLintptr, maxdrawcount: GLsizei, stride: GLsizei);
     func!(MultiDrawMeshTasksIndirectNV, (), indirect: GLintptr, drawcount: GLsizei, stride: GLsizei);
     func!(MultiDrawRangeElementArrayAPPLE, (), mode: GLenum, start: GLuint, end: GLuint, first: *const GLint, count: *const GLsizei, primcount: GLsizei);
     func!(MultiModeDrawArraysIBM, (), mode: *const GLenum, first: *const GLint, count: *const GLsizei, primcount: GLsizei, modestride: GLint);
     func!(MultiModeDrawElementsIBM, (), mode: *const GLenum, count: *const GLsizei, type_: GLenum, indices: *const *const c_void, primcount: GLsizei, modestride: GLint);
     func!(MultiTexBufferEXT, (), texunit: GLenum, target: GLenum, internalformat: GLenum, buffer: GLuint);
     func!(MultiTexCoord1bOES, (), texture: GLenum, s: GLbyte);
     func!(MultiTexCoord1bvOES, (), texture: GLenum, coords: *const GLbyte);
     func!(MultiTexCoord1dARB, (), target: GLenum, s: GLdouble);
     func!(MultiTexCoord1dvARB, (), target: GLenum, v: *const GLdouble);
     func!(MultiTexCoord1fARB, (), target: GLenum, s: GLfloat);
     func!(MultiTexCoord1fvARB, (), target: GLenum, v: *const GLfloat);
     func!(MultiTexCoord1hNV, (), target: GLenum, s: GLhalfNV);
     func!(MultiTexCoord1hvNV, (), target: GLenum, v: *const GLhalfNV);
     func!(MultiTexCoord1iARB, (), target: GLenum, s: GLint);
     func!(MultiTexCoord1ivARB, (), target: GLenum, v: *const GLint);
     func!(MultiTexCoord1sARB, (), target: GLenum, s: GLshort);
     func!(MultiTexCoord1svARB, (), target: GLenum, v: *const GLshort);
     func!(MultiTexCoord1xOES, (), texture: GLenum, s: GLfixed);
     func!(MultiTexCoord1xvOES, (), texture: GLenum, coords: *const GLfixed);
     func!(MultiTexCoord2bOES, (), texture: GLenum, s: GLbyte, t: GLbyte);
     func!(MultiTexCoord2bvOES, (), texture: GLenum, coords: *const GLbyte);
     func!(MultiTexCoord2dARB, (), target: GLenum, s: GLdouble, t: GLdouble);
     func!(MultiTexCoord2dvARB, (), target: GLenum, v: *const GLdouble);
     func!(MultiTexCoord2fARB, (), target: GLenum, s: GLfloat, t: GLfloat);
     func!(MultiTexCoord2fvARB, (), target: GLenum, v: *const GLfloat);
     func!(MultiTexCoord2hNV, (), target: GLenum, s: GLhalfNV, t: GLhalfNV);
     func!(MultiTexCoord2hvNV, (), target: GLenum, v: *const GLhalfNV);
     func!(MultiTexCoord2iARB, (), target: GLenum, s: GLint, t: GLint);
     func!(MultiTexCoord2ivARB, (), target: GLenum, v: *const GLint);
     func!(MultiTexCoord2sARB, (), target: GLenum, s: GLshort, t: GLshort);
     func!(MultiTexCoord2svARB, (), target: GLenum, v: *const GLshort);
     func!(MultiTexCoord2xOES, (), texture: GLenum, s: GLfixed, t: GLfixed);
     func!(MultiTexCoord2xvOES, (), texture: GLenum, coords: *const GLfixed);
     func!(MultiTexCoord3bOES, (), texture: GLenum, s: GLbyte, t: GLbyte, r: GLbyte);
     func!(MultiTexCoord3bvOES, (), texture: GLenum, coords: *const GLbyte);
     func!(MultiTexCoord3dARB, (), target: GLenum, s: GLdouble, t: GLdouble, r: GLdouble);
     func!(MultiTexCoord3dvARB, (), target: GLenum, v: *const GLdouble);
     func!(MultiTexCoord3fARB, (), target: GLenum, s: GLfloat, t: GLfloat, r: GLfloat);
     func!(MultiTexCoord3fvARB, (), target: GLenum, v: *const GLfloat);
     func!(MultiTexCoord3hNV, (), target: GLenum, s: GLhalfNV, t: GLhalfNV, r: GLhalfNV);
     func!(MultiTexCoord3hvNV, (), target: GLenum, v: *const GLhalfNV);
     func!(MultiTexCoord3iARB, (), target: GLenum, s: GLint, t: GLint, r: GLint);
     func!(MultiTexCoord3ivARB, (), target: GLenum, v: *const GLint);
     func!(MultiTexCoord3sARB, (), target: GLenum, s: GLshort, t: GLshort, r: GLshort);
     func!(MultiTexCoord3svARB, (), target: GLenum, v: *const GLshort);
     func!(MultiTexCoord3xOES, (), texture: GLenum, s: GLfixed, t: GLfixed, r: GLfixed);
     func!(MultiTexCoord3xvOES, (), texture: GLenum, coords: *const GLfixed);
     func!(MultiTexCoord4bOES, (), texture: GLenum, s: GLbyte, t: GLbyte, r: GLbyte, q: GLbyte);
     func!(MultiTexCoord4bvOES, (), texture: GLenum, coords: *const GLbyte);
     func!(MultiTexCoord4dARB, (), target: GLenum, s: GLdouble, t: GLdouble, r: GLdouble, q: GLdouble);
     func!(MultiTexCoord4dvARB, (), target: GLenum, v: *const GLdouble);
     func!(MultiTexCoord4fARB, (), target: GLenum, s: GLfloat, t: GLfloat, r: GLfloat, q: GLfloat);
     func!(MultiTexCoord4fvARB, (), target: GLenum, v: *const GLfloat);
     func!(MultiTexCoord4hNV, (), target: GLenum, s: GLhalfNV, t: GLhalfNV, r: GLhalfNV, q: GLhalfNV);
     func!(MultiTexCoord4hvNV, (), target: GLenum, v: *const GLhalfNV);
     func!(MultiTexCoord4iARB, (), target: GLenum, s: GLint, t: GLint, r: GLint, q: GLint);
     func!(MultiTexCoord4ivARB, (), target: GLenum, v: *const GLint);
     func!(MultiTexCoord4sARB, (), target: GLenum, s: GLshort, t: GLshort, r: GLshort, q: GLshort);
     func!(MultiTexCoord4svARB, (), target: GLenum, v: *const GLshort);
     func!(MultiTexCoord4xOES, (), texture: GLenum, s: GLfixed, t: GLfixed, r: GLfixed, q: GLfixed);
     func!(MultiTexCoord4xvOES, (), texture: GLenum, coords: *const GLfixed);
     func!(MultiTexCoordPointerEXT, (), texunit: GLenum, size: GLint, type_: GLenum, stride: GLsizei, pointer: *const c_void);
     func!(MultiTexEnvfEXT, (), texunit: GLenum, target: GLenum, pname: GLenum, param: GLfloat);
     func!(MultiTexEnvfvEXT, (), texunit: GLenum, target: GLenum, pname: GLenum, params: *const GLfloat);
     func!(MultiTexEnviEXT, (), texunit: GLenum, target: GLenum, pname: GLenum, param: GLint);
     func!(MultiTexEnvivEXT, (), texunit: GLenum, target: GLenum, pname: GLenum, params: *const GLint);
     func!(MultiTexGendEXT, (), texunit: GLenum, coord: GLenum, pname: GLenum, param: GLdouble);
     func!(MultiTexGendvEXT, (), texunit: GLenum, coord: GLenum, pname: GLenum, params: *const GLdouble);
     func!(MultiTexGenfEXT, (), texunit: GLenum, coord: GLenum, pname: GLenum, param: GLfloat);
     func!(MultiTexGenfvEXT, (), texunit: GLenum, coord: GLenum, pname: GLenum, params: *const GLfloat);
     func!(MultiTexGeniEXT, (), texunit: GLenum, coord: GLenum, pname: GLenum, param: GLint);
     func!(MultiTexGenivEXT, (), texunit: GLenum, coord: GLenum, pname: GLenum, params: *const GLint);
     func!(MultiTexImage1DEXT, (), texunit: GLenum, target: GLenum, level: GLint, internalformat: GLint, width: GLsizei, border: GLint, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(MultiTexImage2DEXT, (), texunit: GLenum, target: GLenum, level: GLint, internalformat: GLint, width: GLsizei, height: GLsizei, border: GLint, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(MultiTexImage3DEXT, (), texunit: GLenum, target: GLenum, level: GLint, internalformat: GLint, width: GLsizei, height: GLsizei, depth: GLsizei, border: GLint, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(MultiTexParameterIivEXT, (), texunit: GLenum, target: GLenum, pname: GLenum, params: *const GLint);
     func!(MultiTexParameterIuivEXT, (), texunit: GLenum, target: GLenum, pname: GLenum, params: *const GLuint);
     func!(MultiTexParameterfEXT, (), texunit: GLenum, target: GLenum, pname: GLenum, param: GLfloat);
     func!(MultiTexParameterfvEXT, (), texunit: GLenum, target: GLenum, pname: GLenum, params: *const GLfloat);
     func!(MultiTexParameteriEXT, (), texunit: GLenum, target: GLenum, pname: GLenum, param: GLint);
     func!(MultiTexParameterivEXT, (), texunit: GLenum, target: GLenum, pname: GLenum, params: *const GLint);
     func!(MultiTexRenderbufferEXT, (), texunit: GLenum, target: GLenum, renderbuffer: GLuint);
     func!(MultiTexSubImage1DEXT, (), texunit: GLenum, target: GLenum, level: GLint, xoffset: GLint, width: GLsizei, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(MultiTexSubImage2DEXT, (), texunit: GLenum, target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, width: GLsizei, height: GLsizei, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(MultiTexSubImage3DEXT, (), texunit: GLenum, target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, zoffset: GLint, width: GLsizei, height: GLsizei, depth: GLsizei, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(MulticastBarrierNV, (), );
     func!(MulticastBlitFramebufferNV, (), srcGpu: GLuint, dstGpu: GLuint, srcX0: GLint, srcY0: GLint, srcX1: GLint, srcY1: GLint, dstX0: GLint, dstY0: GLint, dstX1: GLint, dstY1: GLint, mask: GLbitfield, filter: GLenum);
     func!(MulticastBufferSubDataNV, (), gpuMask: GLbitfield, buffer: GLuint, offset: GLintptr, size: GLsizeiptr, data: *const c_void);
     func!(MulticastCopyBufferSubDataNV, (), readGpu: GLuint, writeGpuMask: GLbitfield, readBuffer: GLuint, writeBuffer: GLuint, readOffset: GLintptr, writeOffset: GLintptr, size: GLsizeiptr);
     func!(MulticastCopyImageSubDataNV, (), srcGpu: GLuint, dstGpuMask: GLbitfield, srcName: GLuint, srcTarget: GLenum, srcLevel: GLint, srcX: GLint, srcY: GLint, srcZ: GLint, dstName: GLuint, dstTarget: GLenum, dstLevel: GLint, dstX: GLint, dstY: GLint, dstZ: GLint, srcWidth: GLsizei, srcHeight: GLsizei, srcDepth: GLsizei);
     func!(MulticastFramebufferSampleLocationsfvNV, (), gpu: GLuint, framebuffer: GLuint, start: GLuint, count: GLsizei, v: *const GLfloat);
     func!(MulticastGetQueryObjecti64vNV, (), gpu: GLuint, id: GLuint, pname: GLenum, params: *mut GLint64);
     func!(MulticastGetQueryObjectivNV, (), gpu: GLuint, id: GLuint, pname: GLenum, params: *mut GLint);
     func!(MulticastGetQueryObjectui64vNV, (), gpu: GLuint, id: GLuint, pname: GLenum, params: *mut GLuint64);
     func!(MulticastGetQueryObjectuivNV, (), gpu: GLuint, id: GLuint, pname: GLenum, params: *mut GLuint);
     func!(MulticastScissorArrayvNVX, (), gpu: GLuint, first: GLuint, count: GLsizei, v: *const GLint);
     func!(MulticastViewportArrayvNVX, (), gpu: GLuint, first: GLuint, count: GLsizei, v: *const GLfloat);
     func!(MulticastViewportPositionWScaleNVX, (), gpu: GLuint, index: GLuint, xcoeff: GLfloat, ycoeff: GLfloat);
     func!(MulticastWaitSyncNV, (), signalGpu: GLuint, waitGpuMask: GLbitfield);
     func!(NamedBufferAttachMemoryNV, (), buffer: GLuint, memory: GLuint, offset: GLuint64);
     func!(NamedBufferData, (), buffer: GLuint, size: GLsizeiptr, data: *const c_void, usage: GLenum);
     func!(NamedBufferDataEXT, (), buffer: GLuint, size: GLsizeiptr, data: *const c_void, usage: GLenum);
     func!(NamedBufferPageCommitmentARB, (), buffer: GLuint, offset: GLintptr, size: GLsizeiptr, commit: GLboolean);
     func!(NamedBufferPageCommitmentEXT, (), buffer: GLuint, offset: GLintptr, size: GLsizeiptr, commit: GLboolean);
     func!(NamedBufferPageCommitmentMemNV, (), buffer: GLuint, offset: GLintptr, size: GLsizeiptr, memory: GLuint, memOffset: GLuint64, commit: GLboolean);
     func!(NamedBufferStorage, (), buffer: GLuint, size: GLsizeiptr, data: *const c_void, flags: GLbitfield);
     func!(NamedBufferStorageEXT, (), buffer: GLuint, size: GLsizeiptr, data: *const c_void, flags: GLbitfield);
     func!(NamedBufferStorageExternalEXT, (), buffer: GLuint, offset: GLintptr, size: GLsizeiptr, clientBuffer: GLeglClientBufferEXT, flags: GLbitfield);
     func!(NamedBufferStorageMemEXT, (), buffer: GLuint, size: GLsizeiptr, memory: GLuint, offset: GLuint64);
     func!(NamedBufferSubData, (), buffer: GLuint, offset: GLintptr, size: GLsizeiptr, data: *const c_void);
     func!(NamedBufferSubDataEXT, (), buffer: GLuint, offset: GLintptr, size: GLsizeiptr, data: *const c_void);
     func!(NamedCopyBufferSubDataEXT, (), readBuffer: GLuint, writeBuffer: GLuint, readOffset: GLintptr, writeOffset: GLintptr, size: GLsizeiptr);
     func!(NamedFramebufferDrawBuffer, (), framebuffer: GLuint, buf: GLenum);
     func!(NamedFramebufferDrawBuffers, (), framebuffer: GLuint, n: GLsizei, bufs: *const GLenum);
     func!(NamedFramebufferParameteri, (), framebuffer: GLuint, pname: GLenum, param: GLint);
     func!(NamedFramebufferParameteriEXT, (), framebuffer: GLuint, pname: GLenum, param: GLint);
     func!(NamedFramebufferReadBuffer, (), framebuffer: GLuint, src: GLenum);
     func!(NamedFramebufferRenderbuffer, (), framebuffer: GLuint, attachment: GLenum, renderbuffertarget: GLenum, renderbuffer: GLuint);
     func!(NamedFramebufferRenderbufferEXT, (), framebuffer: GLuint, attachment: GLenum, renderbuffertarget: GLenum, renderbuffer: GLuint);
     func!(NamedFramebufferSampleLocationsfvARB, (), framebuffer: GLuint, start: GLuint, count: GLsizei, v: *const GLfloat);
     func!(NamedFramebufferSampleLocationsfvNV, (), framebuffer: GLuint, start: GLuint, count: GLsizei, v: *const GLfloat);
     func!(NamedFramebufferSamplePositionsfvAMD, (), framebuffer: GLuint, numsamples: GLuint, pixelindex: GLuint, values: *const GLfloat);
     func!(NamedFramebufferTexture, (), framebuffer: GLuint, attachment: GLenum, texture: GLuint, level: GLint);
     func!(NamedFramebufferTexture1DEXT, (), framebuffer: GLuint, attachment: GLenum, textarget: GLenum, texture: GLuint, level: GLint);
     func!(NamedFramebufferTexture2DEXT, (), framebuffer: GLuint, attachment: GLenum, textarget: GLenum, texture: GLuint, level: GLint);
     func!(NamedFramebufferTexture3DEXT, (), framebuffer: GLuint, attachment: GLenum, textarget: GLenum, texture: GLuint, level: GLint, zoffset: GLint);
     func!(NamedFramebufferTextureEXT, (), framebuffer: GLuint, attachment: GLenum, texture: GLuint, level: GLint);
     func!(NamedFramebufferTextureFaceEXT, (), framebuffer: GLuint, attachment: GLenum, texture: GLuint, level: GLint, face: GLenum);
     func!(NamedFramebufferTextureLayer, (), framebuffer: GLuint, attachment: GLenum, texture: GLuint, level: GLint, layer: GLint);
     func!(NamedFramebufferTextureLayerEXT, (), framebuffer: GLuint, attachment: GLenum, texture: GLuint, level: GLint, layer: GLint);
     func!(NamedProgramLocalParameter4dEXT, (), program: GLuint, target: GLenum, index: GLuint, x: GLdouble, y: GLdouble, z: GLdouble, w: GLdouble);
     func!(NamedProgramLocalParameter4dvEXT, (), program: GLuint, target: GLenum, index: GLuint, params: *const GLdouble);
     func!(NamedProgramLocalParameter4fEXT, (), program: GLuint, target: GLenum, index: GLuint, x: GLfloat, y: GLfloat, z: GLfloat, w: GLfloat);
     func!(NamedProgramLocalParameter4fvEXT, (), program: GLuint, target: GLenum, index: GLuint, params: *const GLfloat);
     func!(NamedProgramLocalParameterI4iEXT, (), program: GLuint, target: GLenum, index: GLuint, x: GLint, y: GLint, z: GLint, w: GLint);
     func!(NamedProgramLocalParameterI4ivEXT, (), program: GLuint, target: GLenum, index: GLuint, params: *const GLint);
     func!(NamedProgramLocalParameterI4uiEXT, (), program: GLuint, target: GLenum, index: GLuint, x: GLuint, y: GLuint, z: GLuint, w: GLuint);
     func!(NamedProgramLocalParameterI4uivEXT, (), program: GLuint, target: GLenum, index: GLuint, params: *const GLuint);
     func!(NamedProgramLocalParameters4fvEXT, (), program: GLuint, target: GLenum, index: GLuint, count: GLsizei, params: *const GLfloat);
     func!(NamedProgramLocalParametersI4ivEXT, (), program: GLuint, target: GLenum, index: GLuint, count: GLsizei, params: *const GLint);
     func!(NamedProgramLocalParametersI4uivEXT, (), program: GLuint, target: GLenum, index: GLuint, count: GLsizei, params: *const GLuint);
     func!(NamedProgramStringEXT, (), program: GLuint, target: GLenum, format: GLenum, len: GLsizei, string: *const c_void);
     func!(NamedRenderbufferStorage, (), renderbuffer: GLuint, internalformat: GLenum, width: GLsizei, height: GLsizei);
     func!(NamedRenderbufferStorageEXT, (), renderbuffer: GLuint, internalformat: GLenum, width: GLsizei, height: GLsizei);
     func!(NamedRenderbufferStorageMultisample, (), renderbuffer: GLuint, samples: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei);
     func!(NamedRenderbufferStorageMultisampleAdvancedAMD, (), renderbuffer: GLuint, samples: GLsizei, storageSamples: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei);
     func!(NamedRenderbufferStorageMultisampleCoverageEXT, (), renderbuffer: GLuint, coverageSamples: GLsizei, colorSamples: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei);
     func!(NamedRenderbufferStorageMultisampleEXT, (), renderbuffer: GLuint, samples: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei);
     func!(NamedStringARB, (), type_: GLenum, namelen: GLint, name: *const GLchar, stringlen: GLint, string: *const GLchar);
     func!(NewObjectBufferATI, GLuint, size: GLsizei, pointer: *const c_void, usage: GLenum);
     func!(Normal3fVertex3fSUN, (), nx: GLfloat, ny: GLfloat, nz: GLfloat, x: GLfloat, y: GLfloat, z: GLfloat);
     func!(Normal3fVertex3fvSUN, (), n: *const GLfloat, v: *const GLfloat);
     func!(Normal3hNV, (), nx: GLhalfNV, ny: GLhalfNV, nz: GLhalfNV);
     func!(Normal3hvNV, (), v: *const GLhalfNV);
     func!(Normal3xOES, (), nx: GLfixed, ny: GLfixed, nz: GLfixed);
     func!(Normal3xvOES, (), coords: *const GLfixed);
     func!(NormalFormatNV, (), type_: GLenum, stride: GLsizei);
     func!(NormalPointerEXT, (), type_: GLenum, stride: GLsizei, count: GLsizei, pointer: *const c_void);
     func!(NormalPointerListIBM, (), type_: GLenum, stride: GLint, pointer: *const *const c_void, ptrstride: GLint);
     func!(NormalPointervINTEL, (), type_: GLenum, pointer: *const *const c_void);
     func!(NormalStream3bATI, (), stream: GLenum, nx: GLbyte, ny: GLbyte, nz: GLbyte);
     func!(NormalStream3bvATI, (), stream: GLenum, coords: *const GLbyte);
     func!(NormalStream3dATI, (), stream: GLenum, nx: GLdouble, ny: GLdouble, nz: GLdouble);
     func!(NormalStream3dvATI, (), stream: GLenum, coords: *const GLdouble);
     func!(NormalStream3fATI, (), stream: GLenum, nx: GLfloat, ny: GLfloat, nz: GLfloat);
     func!(NormalStream3fvATI, (), stream: GLenum, coords: *const GLfloat);
     func!(NormalStream3iATI, (), stream: GLenum, nx: GLint, ny: GLint, nz: GLint);
     func!(NormalStream3ivATI, (), stream: GLenum, coords: *const GLint);
     func!(NormalStream3sATI, (), stream: GLenum, nx: GLshort, ny: GLshort, nz: GLshort);
     func!(NormalStream3svATI, (), stream: GLenum, coords: *const GLshort);
     func!(ObjectLabel, (), identifier: GLenum, name: GLuint, length: GLsizei, label: *const GLchar);
     func!(ObjectPtrLabel, (), ptr: *const c_void, length: GLsizei, label: *const GLchar);
     func!(ObjectPurgeableAPPLE, GLenum, objectType: GLenum, name: GLuint, option: GLenum);
     func!(ObjectUnpurgeableAPPLE, GLenum, objectType: GLenum, name: GLuint, option: GLenum);
     func!(OrthofOES, (), l: GLfloat, r: GLfloat, b: GLfloat, t: GLfloat, n: GLfloat, f: GLfloat);
     func!(OrthoxOES, (), l: GLfixed, r: GLfixed, b: GLfixed, t: GLfixed, n: GLfixed, f: GLfixed);
     func!(PNTrianglesfATI, (), pname: GLenum, param: GLfloat);
     func!(PNTrianglesiATI, (), pname: GLenum, param: GLint);
     func!(PassTexCoordATI, (), dst: GLuint, coord: GLuint, swizzle: GLenum);
     func!(PassThroughxOES, (), token: GLfixed);
     func!(PatchParameterfv, (), pname: GLenum, values: *const GLfloat);
     func!(PatchParameteri, (), pname: GLenum, value: GLint);
     func!(PathCommandsNV, (), path: GLuint, numCommands: GLsizei, commands: *const GLubyte, numCoords: GLsizei, coordType: GLenum, coords: *const c_void);
     func!(PathCoordsNV, (), path: GLuint, numCoords: GLsizei, coordType: GLenum, coords: *const c_void);
     func!(PathCoverDepthFuncNV, (), func: GLenum);
     func!(PathDashArrayNV, (), path: GLuint, dashCount: GLsizei, dashArray: *const GLfloat);
     func!(PathGlyphIndexArrayNV, GLenum, firstPathName: GLuint, fontTarget: GLenum, fontName: *const c_void, fontStyle: GLbitfield, firstGlyphIndex: GLuint, numGlyphs: GLsizei, pathParameterTemplate: GLuint, emScale: GLfloat);
     func!(PathGlyphIndexRangeNV, GLenum, fontTarget: GLenum, fontName: *const c_void, fontStyle: GLbitfield, pathParameterTemplate: GLuint, emScale: GLfloat, baseAndCount: *mut GLuint);
     func!(PathGlyphRangeNV, (), firstPathName: GLuint, fontTarget: GLenum, fontName: *const c_void, fontStyle: GLbitfield, firstGlyph: GLuint, numGlyphs: GLsizei, handleMissingGlyphs: GLenum, pathParameterTemplate: GLuint, emScale: GLfloat);
     func!(PathGlyphsNV, (), firstPathName: GLuint, fontTarget: GLenum, fontName: *const c_void, fontStyle: GLbitfield, numGlyphs: GLsizei, type_: GLenum, charcodes: *const c_void, handleMissingGlyphs: GLenum, pathParameterTemplate: GLuint, emScale: GLfloat);
     func!(PathMemoryGlyphIndexArrayNV, GLenum, firstPathName: GLuint, fontTarget: GLenum, fontSize: GLsizeiptr, fontData: *const c_void, faceIndex: GLsizei, firstGlyphIndex: GLuint, numGlyphs: GLsizei, pathParameterTemplate: GLuint, emScale: GLfloat);
     func!(PathParameterfNV, (), path: GLuint, pname: GLenum, value: GLfloat);
     func!(PathParameterfvNV, (), path: GLuint, pname: GLenum, value: *const GLfloat);
     func!(PathParameteriNV, (), path: GLuint, pname: GLenum, value: GLint);
     func!(PathParameterivNV, (), path: GLuint, pname: GLenum, value: *const GLint);
     func!(PathStencilDepthOffsetNV, (), factor: GLfloat, units: GLfloat);
     func!(PathStencilFuncNV, (), func: GLenum, ref_: GLint, mask: GLuint);
     func!(PathStringNV, (), path: GLuint, format: GLenum, length: GLsizei, pathString: *const c_void);
     func!(PathSubCommandsNV, (), path: GLuint, commandStart: GLsizei, commandsToDelete: GLsizei, numCommands: GLsizei, commands: *const GLubyte, numCoords: GLsizei, coordType: GLenum, coords: *const c_void);
     func!(PathSubCoordsNV, (), path: GLuint, coordStart: GLsizei, numCoords: GLsizei, coordType: GLenum, coords: *const c_void);
     func!(PauseTransformFeedback, (), );
     func!(PauseTransformFeedbackNV, (), );
     func!(PixelDataRangeNV, (), target: GLenum, length: GLsizei, pointer: *const c_void);
     func!(PixelMapx, (), map: GLenum, size: GLint, values: *const GLfixed);
     func!(PixelStoref, (), pname: GLenum, param: GLfloat);
     func!(PixelStorei, (), pname: GLenum, param: GLint);
     func!(PixelStorex, (), pname: GLenum, param: GLfixed);
     func!(PixelTexGenParameterfSGIS, (), pname: GLenum, param: GLfloat);
     func!(PixelTexGenParameterfvSGIS, (), pname: GLenum, params: *const GLfloat);
     func!(PixelTexGenParameteriSGIS, (), pname: GLenum, param: GLint);
     func!(PixelTexGenParameterivSGIS, (), pname: GLenum, params: *const GLint);
     func!(PixelTexGenSGIX, (), mode: GLenum);
     func!(PixelTransferxOES, (), pname: GLenum, param: GLfixed);
     func!(PixelTransformParameterfEXT, (), target: GLenum, pname: GLenum, param: GLfloat);
     func!(PixelTransformParameterfvEXT, (), target: GLenum, pname: GLenum, params: *const GLfloat);
     func!(PixelTransformParameteriEXT, (), target: GLenum, pname: GLenum, param: GLint);
     func!(PixelTransformParameterivEXT, (), target: GLenum, pname: GLenum, params: *const GLint);
     func!(PixelZoomxOES, (), xfactor: GLfixed, yfactor: GLfixed);
     func!(PointAlongPathNV, GLboolean, path: GLuint, startSegment: GLsizei, numSegments: GLsizei, distance: GLfloat, x: *mut GLfloat, y: *mut GLfloat, tangentX: *mut GLfloat, tangentY: *mut GLfloat);
     func!(PointParameterf, (), pname: GLenum, param: GLfloat);
     func!(PointParameterfARB, (), pname: GLenum, param: GLfloat);
     func!(PointParameterfEXT, (), pname: GLenum, param: GLfloat);
     func!(PointParameterfSGIS, (), pname: GLenum, param: GLfloat);
     func!(PointParameterfv, (), pname: GLenum, params: *const GLfloat);
     func!(PointParameterfvARB, (), pname: GLenum, params: *const GLfloat);
     func!(PointParameterfvEXT, (), pname: GLenum, params: *const GLfloat);
     func!(PointParameterfvSGIS, (), pname: GLenum, params: *const GLfloat);
     func!(PointParameteri, (), pname: GLenum, param: GLint);
     func!(PointParameteriNV, (), pname: GLenum, param: GLint);
     func!(PointParameteriv, (), pname: GLenum, params: *const GLint);
     func!(PointParameterivNV, (), pname: GLenum, params: *const GLint);
     func!(PointParameterxvOES, (), pname: GLenum, params: *const GLfixed);
     func!(PointSize, (), size: GLfloat);
     func!(PointSizexOES, (), size: GLfixed);
     func!(PollAsyncSGIX, GLint, markerp: *mut GLuint);
     func!(PollInstrumentsSGIX, GLint, marker_p: *mut GLint);
     func!(PolygonMode, (), face: GLenum, mode: GLenum);
     func!(PolygonOffset, (), factor: GLfloat, units: GLfloat);
     func!(PolygonOffsetClamp, (), factor: GLfloat, units: GLfloat, clamp: GLfloat);
     func!(PolygonOffsetClampEXT, (), factor: GLfloat, units: GLfloat, clamp: GLfloat);
     func!(PolygonOffsetEXT, (), factor: GLfloat, bias: GLfloat);
     func!(PolygonOffsetxOES, (), factor: GLfixed, units: GLfixed);
     func!(PopDebugGroup, (), );
     func!(PopGroupMarkerEXT, (), );
     func!(PresentFrameDualFillNV, (), video_slot: GLuint, minPresentTime: GLuint64EXT, beginPresentTimeId: GLuint, presentDurationId: GLuint, type_: GLenum, target0: GLenum, fill0: GLuint, target1: GLenum, fill1: GLuint, target2: GLenum, fill2: GLuint, target3: GLenum, fill3: GLuint);
     func!(PresentFrameKeyedNV, (), video_slot: GLuint, minPresentTime: GLuint64EXT, beginPresentTimeId: GLuint, presentDurationId: GLuint, type_: GLenum, target0: GLenum, fill0: GLuint, key0: GLuint, target1: GLenum, fill1: GLuint, key1: GLuint);
     func!(PrimitiveBoundingBoxARB, (), minX: GLfloat, minY: GLfloat, minZ: GLfloat, minW: GLfloat, maxX: GLfloat, maxY: GLfloat, maxZ: GLfloat, maxW: GLfloat);
     func!(PrimitiveRestartIndex, (), index: GLuint);
     func!(PrimitiveRestartIndexNV, (), index: GLuint);
     func!(PrimitiveRestartNV, (), );
     func!(PrioritizeTexturesEXT, (), n: GLsizei, textures: *const GLuint, priorities: *const GLclampf);
     func!(PrioritizeTexturesxOES, (), n: GLsizei, textures: *const GLuint, priorities: *const GLfixed);
     func!(ProgramBinary, (), program: GLuint, binaryFormat: GLenum, binary: *const c_void, length: GLsizei);
     func!(ProgramBufferParametersIivNV, (), target: GLenum, bindingIndex: GLuint, wordIndex: GLuint, count: GLsizei, params: *const GLint);
     func!(ProgramBufferParametersIuivNV, (), target: GLenum, bindingIndex: GLuint, wordIndex: GLuint, count: GLsizei, params: *const GLuint);
     func!(ProgramBufferParametersfvNV, (), target: GLenum, bindingIndex: GLuint, wordIndex: GLuint, count: GLsizei, params: *const GLfloat);
     func!(ProgramEnvParameter4dARB, (), target: GLenum, index: GLuint, x: GLdouble, y: GLdouble, z: GLdouble, w: GLdouble);
     func!(ProgramEnvParameter4dvARB, (), target: GLenum, index: GLuint, params: *const GLdouble);
     func!(ProgramEnvParameter4fARB, (), target: GLenum, index: GLuint, x: GLfloat, y: GLfloat, z: GLfloat, w: GLfloat);
     func!(ProgramEnvParameter4fvARB, (), target: GLenum, index: GLuint, params: *const GLfloat);
     func!(ProgramEnvParameterI4iNV, (), target: GLenum, index: GLuint, x: GLint, y: GLint, z: GLint, w: GLint);
     func!(ProgramEnvParameterI4ivNV, (), target: GLenum, index: GLuint, params: *const GLint);
     func!(ProgramEnvParameterI4uiNV, (), target: GLenum, index: GLuint, x: GLuint, y: GLuint, z: GLuint, w: GLuint);
     func!(ProgramEnvParameterI4uivNV, (), target: GLenum, index: GLuint, params: *const GLuint);
     func!(ProgramEnvParameters4fvEXT, (), target: GLenum, index: GLuint, count: GLsizei, params: *const GLfloat);
     func!(ProgramEnvParametersI4ivNV, (), target: GLenum, index: GLuint, count: GLsizei, params: *const GLint);
     func!(ProgramEnvParametersI4uivNV, (), target: GLenum, index: GLuint, count: GLsizei, params: *const GLuint);
     func!(ProgramLocalParameter4dARB, (), target: GLenum, index: GLuint, x: GLdouble, y: GLdouble, z: GLdouble, w: GLdouble);
     func!(ProgramLocalParameter4dvARB, (), target: GLenum, index: GLuint, params: *const GLdouble);
     func!(ProgramLocalParameter4fARB, (), target: GLenum, index: GLuint, x: GLfloat, y: GLfloat, z: GLfloat, w: GLfloat);
     func!(ProgramLocalParameter4fvARB, (), target: GLenum, index: GLuint, params: *const GLfloat);
     func!(ProgramLocalParameterI4iNV, (), target: GLenum, index: GLuint, x: GLint, y: GLint, z: GLint, w: GLint);
     func!(ProgramLocalParameterI4ivNV, (), target: GLenum, index: GLuint, params: *const GLint);
     func!(ProgramLocalParameterI4uiNV, (), target: GLenum, index: GLuint, x: GLuint, y: GLuint, z: GLuint, w: GLuint);
     func!(ProgramLocalParameterI4uivNV, (), target: GLenum, index: GLuint, params: *const GLuint);
     func!(ProgramLocalParameters4fvEXT, (), target: GLenum, index: GLuint, count: GLsizei, params: *const GLfloat);
     func!(ProgramLocalParametersI4ivNV, (), target: GLenum, index: GLuint, count: GLsizei, params: *const GLint);
     func!(ProgramLocalParametersI4uivNV, (), target: GLenum, index: GLuint, count: GLsizei, params: *const GLuint);
     func!(ProgramNamedParameter4dNV, (), id: GLuint, len: GLsizei, name: *const GLubyte, x: GLdouble, y: GLdouble, z: GLdouble, w: GLdouble);
     func!(ProgramNamedParameter4dvNV, (), id: GLuint, len: GLsizei, name: *const GLubyte, v: *const GLdouble);
     func!(ProgramNamedParameter4fNV, (), id: GLuint, len: GLsizei, name: *const GLubyte, x: GLfloat, y: GLfloat, z: GLfloat, w: GLfloat);
     func!(ProgramNamedParameter4fvNV, (), id: GLuint, len: GLsizei, name: *const GLubyte, v: *const GLfloat);
     func!(ProgramParameter4dNV, (), target: GLenum, index: GLuint, x: GLdouble, y: GLdouble, z: GLdouble, w: GLdouble);
     func!(ProgramParameter4dvNV, (), target: GLenum, index: GLuint, v: *const GLdouble);
     func!(ProgramParameter4fNV, (), target: GLenum, index: GLuint, x: GLfloat, y: GLfloat, z: GLfloat, w: GLfloat);
     func!(ProgramParameter4fvNV, (), target: GLenum, index: GLuint, v: *const GLfloat);
     func!(ProgramParameteri, (), program: GLuint, pname: GLenum, value: GLint);
     func!(ProgramParameteriARB, (), program: GLuint, pname: GLenum, value: GLint);
     func!(ProgramParameteriEXT, (), program: GLuint, pname: GLenum, value: GLint);
     func!(ProgramParameters4dvNV, (), target: GLenum, index: GLuint, count: GLsizei, v: *const GLdouble);
     func!(ProgramParameters4fvNV, (), target: GLenum, index: GLuint, count: GLsizei, v: *const GLfloat);
     func!(ProgramPathFragmentInputGenNV, (), program: GLuint, location: GLint, genMode: GLenum, components: GLint, coeffs: *const GLfloat);
     func!(ProgramStringARB, (), target: GLenum, format: GLenum, len: GLsizei, string: *const c_void);
     func!(ProgramSubroutineParametersuivNV, (), target: GLenum, count: GLsizei, params: *const GLuint);
     func!(ProgramUniform1d, (), program: GLuint, location: GLint, v0: GLdouble);
     func!(ProgramUniform1dEXT, (), program: GLuint, location: GLint, x: GLdouble);
     func!(ProgramUniform1dv, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLdouble);
     func!(ProgramUniform1dvEXT, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLdouble);
     func!(ProgramUniform1f, (), program: GLuint, location: GLint, v0: GLfloat);
     func!(ProgramUniform1fEXT, (), program: GLuint, location: GLint, v0: GLfloat);
     func!(ProgramUniform1fv, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLfloat);
     func!(ProgramUniform1fvEXT, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLfloat);
     func!(ProgramUniform1i, (), program: GLuint, location: GLint, v0: GLint);
     func!(ProgramUniform1i64ARB, (), program: GLuint, location: GLint, x: GLint64);
     func!(ProgramUniform1i64NV, (), program: GLuint, location: GLint, x: GLint64EXT);
     func!(ProgramUniform1i64vARB, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLint64);
     func!(ProgramUniform1i64vNV, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLint64EXT);
     func!(ProgramUniform1iEXT, (), program: GLuint, location: GLint, v0: GLint);
     func!(ProgramUniform1iv, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLint);
     func!(ProgramUniform1ivEXT, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLint);
     func!(ProgramUniform1ui, (), program: GLuint, location: GLint, v0: GLuint);
     func!(ProgramUniform1ui64ARB, (), program: GLuint, location: GLint, x: GLuint64);
     func!(ProgramUniform1ui64NV, (), program: GLuint, location: GLint, x: GLuint64EXT);
     func!(ProgramUniform1ui64vARB, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLuint64);
     func!(ProgramUniform1ui64vNV, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLuint64EXT);
     func!(ProgramUniform1uiEXT, (), program: GLuint, location: GLint, v0: GLuint);
     func!(ProgramUniform1uiv, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLuint);
     func!(ProgramUniform1uivEXT, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLuint);
     func!(ProgramUniform2d, (), program: GLuint, location: GLint, v0: GLdouble, v1: GLdouble);
     func!(ProgramUniform2dEXT, (), program: GLuint, location: GLint, x: GLdouble, y: GLdouble);
     func!(ProgramUniform2dv, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLdouble);
     func!(ProgramUniform2dvEXT, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLdouble);
     func!(ProgramUniform2f, (), program: GLuint, location: GLint, v0: GLfloat, v1: GLfloat);
     func!(ProgramUniform2fEXT, (), program: GLuint, location: GLint, v0: GLfloat, v1: GLfloat);
     func!(ProgramUniform2fv, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLfloat);
     func!(ProgramUniform2fvEXT, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLfloat);
     func!(ProgramUniform2i, (), program: GLuint, location: GLint, v0: GLint, v1: GLint);
     func!(ProgramUniform2i64ARB, (), program: GLuint, location: GLint, x: GLint64, y: GLint64);
     func!(ProgramUniform2i64NV, (), program: GLuint, location: GLint, x: GLint64EXT, y: GLint64EXT);
     func!(ProgramUniform2i64vARB, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLint64);
     func!(ProgramUniform2i64vNV, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLint64EXT);
     func!(ProgramUniform2iEXT, (), program: GLuint, location: GLint, v0: GLint, v1: GLint);
     func!(ProgramUniform2iv, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLint);
     func!(ProgramUniform2ivEXT, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLint);
     func!(ProgramUniform2ui, (), program: GLuint, location: GLint, v0: GLuint, v1: GLuint);
     func!(ProgramUniform2ui64ARB, (), program: GLuint, location: GLint, x: GLuint64, y: GLuint64);
     func!(ProgramUniform2ui64NV, (), program: GLuint, location: GLint, x: GLuint64EXT, y: GLuint64EXT);
     func!(ProgramUniform2ui64vARB, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLuint64);
     func!(ProgramUniform2ui64vNV, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLuint64EXT);
     func!(ProgramUniform2uiEXT, (), program: GLuint, location: GLint, v0: GLuint, v1: GLuint);
     func!(ProgramUniform2uiv, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLuint);
     func!(ProgramUniform2uivEXT, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLuint);
     func!(ProgramUniform3d, (), program: GLuint, location: GLint, v0: GLdouble, v1: GLdouble, v2: GLdouble);
     func!(ProgramUniform3dEXT, (), program: GLuint, location: GLint, x: GLdouble, y: GLdouble, z: GLdouble);
     func!(ProgramUniform3dv, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLdouble);
     func!(ProgramUniform3dvEXT, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLdouble);
     func!(ProgramUniform3f, (), program: GLuint, location: GLint, v0: GLfloat, v1: GLfloat, v2: GLfloat);
     func!(ProgramUniform3fEXT, (), program: GLuint, location: GLint, v0: GLfloat, v1: GLfloat, v2: GLfloat);
     func!(ProgramUniform3fv, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLfloat);
     func!(ProgramUniform3fvEXT, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLfloat);
     func!(ProgramUniform3i, (), program: GLuint, location: GLint, v0: GLint, v1: GLint, v2: GLint);
     func!(ProgramUniform3i64ARB, (), program: GLuint, location: GLint, x: GLint64, y: GLint64, z: GLint64);
     func!(ProgramUniform3i64NV, (), program: GLuint, location: GLint, x: GLint64EXT, y: GLint64EXT, z: GLint64EXT);
     func!(ProgramUniform3i64vARB, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLint64);
     func!(ProgramUniform3i64vNV, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLint64EXT);
     func!(ProgramUniform3iEXT, (), program: GLuint, location: GLint, v0: GLint, v1: GLint, v2: GLint);
     func!(ProgramUniform3iv, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLint);
     func!(ProgramUniform3ivEXT, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLint);
     func!(ProgramUniform3ui, (), program: GLuint, location: GLint, v0: GLuint, v1: GLuint, v2: GLuint);
     func!(ProgramUniform3ui64ARB, (), program: GLuint, location: GLint, x: GLuint64, y: GLuint64, z: GLuint64);
     func!(ProgramUniform3ui64NV, (), program: GLuint, location: GLint, x: GLuint64EXT, y: GLuint64EXT, z: GLuint64EXT);
     func!(ProgramUniform3ui64vARB, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLuint64);
     func!(ProgramUniform3ui64vNV, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLuint64EXT);
     func!(ProgramUniform3uiEXT, (), program: GLuint, location: GLint, v0: GLuint, v1: GLuint, v2: GLuint);
     func!(ProgramUniform3uiv, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLuint);
     func!(ProgramUniform3uivEXT, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLuint);
     func!(ProgramUniform4d, (), program: GLuint, location: GLint, v0: GLdouble, v1: GLdouble, v2: GLdouble, v3: GLdouble);
     func!(ProgramUniform4dEXT, (), program: GLuint, location: GLint, x: GLdouble, y: GLdouble, z: GLdouble, w: GLdouble);
     func!(ProgramUniform4dv, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLdouble);
     func!(ProgramUniform4dvEXT, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLdouble);
     func!(ProgramUniform4f, (), program: GLuint, location: GLint, v0: GLfloat, v1: GLfloat, v2: GLfloat, v3: GLfloat);
     func!(ProgramUniform4fEXT, (), program: GLuint, location: GLint, v0: GLfloat, v1: GLfloat, v2: GLfloat, v3: GLfloat);
     func!(ProgramUniform4fv, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLfloat);
     func!(ProgramUniform4fvEXT, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLfloat);
     func!(ProgramUniform4i, (), program: GLuint, location: GLint, v0: GLint, v1: GLint, v2: GLint, v3: GLint);
     func!(ProgramUniform4i64ARB, (), program: GLuint, location: GLint, x: GLint64, y: GLint64, z: GLint64, w: GLint64);
     func!(ProgramUniform4i64NV, (), program: GLuint, location: GLint, x: GLint64EXT, y: GLint64EXT, z: GLint64EXT, w: GLint64EXT);
     func!(ProgramUniform4i64vARB, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLint64);
     func!(ProgramUniform4i64vNV, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLint64EXT);
     func!(ProgramUniform4iEXT, (), program: GLuint, location: GLint, v0: GLint, v1: GLint, v2: GLint, v3: GLint);
     func!(ProgramUniform4iv, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLint);
     func!(ProgramUniform4ivEXT, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLint);
     func!(ProgramUniform4ui, (), program: GLuint, location: GLint, v0: GLuint, v1: GLuint, v2: GLuint, v3: GLuint);
     func!(ProgramUniform4ui64ARB, (), program: GLuint, location: GLint, x: GLuint64, y: GLuint64, z: GLuint64, w: GLuint64);
     func!(ProgramUniform4ui64NV, (), program: GLuint, location: GLint, x: GLuint64EXT, y: GLuint64EXT, z: GLuint64EXT, w: GLuint64EXT);
     func!(ProgramUniform4ui64vARB, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLuint64);
     func!(ProgramUniform4ui64vNV, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLuint64EXT);
     func!(ProgramUniform4uiEXT, (), program: GLuint, location: GLint, v0: GLuint, v1: GLuint, v2: GLuint, v3: GLuint);
     func!(ProgramUniform4uiv, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLuint);
     func!(ProgramUniform4uivEXT, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLuint);
     func!(ProgramUniformHandleui64ARB, (), program: GLuint, location: GLint, value: GLuint64);
     func!(ProgramUniformHandleui64NV, (), program: GLuint, location: GLint, value: GLuint64);
     func!(ProgramUniformHandleui64vARB, (), program: GLuint, location: GLint, count: GLsizei, values: *const GLuint64);
     func!(ProgramUniformHandleui64vNV, (), program: GLuint, location: GLint, count: GLsizei, values: *const GLuint64);
     func!(ProgramUniformMatrix2dv, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(ProgramUniformMatrix2dvEXT, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(ProgramUniformMatrix2fv, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(ProgramUniformMatrix2fvEXT, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(ProgramUniformMatrix2x3dv, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(ProgramUniformMatrix2x3dvEXT, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(ProgramUniformMatrix2x3fv, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(ProgramUniformMatrix2x3fvEXT, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(ProgramUniformMatrix2x4dv, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(ProgramUniformMatrix2x4dvEXT, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(ProgramUniformMatrix2x4fv, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(ProgramUniformMatrix2x4fvEXT, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(ProgramUniformMatrix3dv, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(ProgramUniformMatrix3dvEXT, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(ProgramUniformMatrix3fv, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(ProgramUniformMatrix3fvEXT, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(ProgramUniformMatrix3x2dv, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(ProgramUniformMatrix3x2dvEXT, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(ProgramUniformMatrix3x2fv, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(ProgramUniformMatrix3x2fvEXT, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(ProgramUniformMatrix3x4dv, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(ProgramUniformMatrix3x4dvEXT, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(ProgramUniformMatrix3x4fv, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(ProgramUniformMatrix3x4fvEXT, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(ProgramUniformMatrix4dv, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(ProgramUniformMatrix4dvEXT, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(ProgramUniformMatrix4fv, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(ProgramUniformMatrix4fvEXT, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(ProgramUniformMatrix4x2dv, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(ProgramUniformMatrix4x2dvEXT, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(ProgramUniformMatrix4x2fv, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(ProgramUniformMatrix4x2fvEXT, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(ProgramUniformMatrix4x3dv, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(ProgramUniformMatrix4x3dvEXT, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(ProgramUniformMatrix4x3fv, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(ProgramUniformMatrix4x3fvEXT, (), program: GLuint, location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(ProgramUniformui64NV, (), program: GLuint, location: GLint, value: GLuint64EXT);
     func!(ProgramUniformui64vNV, (), program: GLuint, location: GLint, count: GLsizei, value: *const GLuint64EXT);
     func!(ProgramVertexLimitNV, (), target: GLenum, limit: GLint);
     func!(ProvokingVertex, (), mode: GLenum);
     func!(ProvokingVertexEXT, (), mode: GLenum);
     func!(PushClientAttribDefaultEXT, (), mask: GLbitfield);
     func!(PushDebugGroup, (), source: GLenum, id: GLuint, length: GLsizei, message: *const GLchar);
     func!(PushGroupMarkerEXT, (), length: GLsizei, marker: *const GLchar);
     func!(QueryCounter, (), id: GLuint, target: GLenum);
     func!(QueryMatrixxOES, GLbitfield, mantissa: *mut GLfixed, exponent: *mut GLint);
     func!(QueryObjectParameteruiAMD, (), target: GLenum, id: GLuint, pname: GLenum, param: GLuint);
     func!(QueryResourceNV, GLint, queryType: GLenum, tagId: GLint, count: GLuint, buffer: *mut GLint);
     func!(QueryResourceTagNV, (), tagId: GLint, tagString: *const GLchar);
     func!(RasterPos2xOES, (), x: GLfixed, y: GLfixed);
     func!(RasterPos2xvOES, (), coords: *const GLfixed);
     func!(RasterPos3xOES, (), x: GLfixed, y: GLfixed, z: GLfixed);
     func!(RasterPos3xvOES, (), coords: *const GLfixed);
     func!(RasterPos4xOES, (), x: GLfixed, y: GLfixed, z: GLfixed, w: GLfixed);
     func!(RasterPos4xvOES, (), coords: *const GLfixed);
     func!(RasterSamplesEXT, (), samples: GLuint, fixedsamplelocations: GLboolean);
     func!(ReadBuffer, (), src: GLenum);
     func!(ReadInstrumentsSGIX, (), marker: GLint);
     func!(ReadPixels, (), x: GLint, y: GLint, width: GLsizei, height: GLsizei, format: GLenum, type_: GLenum, pixels: *mut c_void);
     func!(ReadnPixels, (), x: GLint, y: GLint, width: GLsizei, height: GLsizei, format: GLenum, type_: GLenum, bufSize: GLsizei, data: *mut c_void);
     func!(ReadnPixelsARB, (), x: GLint, y: GLint, width: GLsizei, height: GLsizei, format: GLenum, type_: GLenum, bufSize: GLsizei, data: *mut c_void);
     func!(RectxOES, (), x1: GLfixed, y1: GLfixed, x2: GLfixed, y2: GLfixed);
     func!(RectxvOES, (), v1: *const GLfixed, v2: *const GLfixed);
     func!(ReferencePlaneSGIX, (), equation: *const GLdouble);
     func!(ReleaseKeyedMutexWin32EXT, GLboolean, memory: GLuint, key: GLuint64);
     func!(ReleaseShaderCompiler, (), );
     func!(RenderGpuMaskNV, (), mask: GLbitfield);
     func!(RenderbufferStorage, (), target: GLenum, internalformat: GLenum, width: GLsizei, height: GLsizei);
     func!(RenderbufferStorageEXT, (), target: GLenum, internalformat: GLenum, width: GLsizei, height: GLsizei);
     func!(RenderbufferStorageMultisample, (), target: GLenum, samples: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei);
     func!(RenderbufferStorageMultisampleAdvancedAMD, (), target: GLenum, samples: GLsizei, storageSamples: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei);
     func!(RenderbufferStorageMultisampleCoverageNV, (), target: GLenum, coverageSamples: GLsizei, colorSamples: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei);
     func!(RenderbufferStorageMultisampleEXT, (), target: GLenum, samples: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei);
     func!(ReplacementCodePointerSUN, (), type_: GLenum, stride: GLsizei, pointer: *const *const c_void);
     func!(ReplacementCodeubSUN, (), code: GLubyte);
     func!(ReplacementCodeubvSUN, (), code: *const GLubyte);
     func!(ReplacementCodeuiColor3fVertex3fSUN, (), rc: GLuint, r: GLfloat, g: GLfloat, b: GLfloat, x: GLfloat, y: GLfloat, z: GLfloat);
     func!(ReplacementCodeuiColor3fVertex3fvSUN, (), rc: *const GLuint, c: *const GLfloat, v: *const GLfloat);
     func!(ReplacementCodeuiColor4fNormal3fVertex3fSUN, (), rc: GLuint, r: GLfloat, g: GLfloat, b: GLfloat, a: GLfloat, nx: GLfloat, ny: GLfloat, nz: GLfloat, x: GLfloat, y: GLfloat, z: GLfloat);
     func!(ReplacementCodeuiColor4fNormal3fVertex3fvSUN, (), rc: *const GLuint, c: *const GLfloat, n: *const GLfloat, v: *const GLfloat);
     func!(ReplacementCodeuiColor4ubVertex3fSUN, (), rc: GLuint, r: GLubyte, g: GLubyte, b: GLubyte, a: GLubyte, x: GLfloat, y: GLfloat, z: GLfloat);
     func!(ReplacementCodeuiColor4ubVertex3fvSUN, (), rc: *const GLuint, c: *const GLubyte, v: *const GLfloat);
     func!(ReplacementCodeuiNormal3fVertex3fSUN, (), rc: GLuint, nx: GLfloat, ny: GLfloat, nz: GLfloat, x: GLfloat, y: GLfloat, z: GLfloat);
     func!(ReplacementCodeuiNormal3fVertex3fvSUN, (), rc: *const GLuint, n: *const GLfloat, v: *const GLfloat);
     func!(ReplacementCodeuiSUN, (), code: GLuint);
     func!(ReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fSUN, (), rc: GLuint, s: GLfloat, t: GLfloat, r: GLfloat, g: GLfloat, b: GLfloat, a: GLfloat, nx: GLfloat, ny: GLfloat, nz: GLfloat, x: GLfloat, y: GLfloat, z: GLfloat);
     func!(ReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fvSUN, (), rc: *const GLuint, tc: *const GLfloat, c: *const GLfloat, n: *const GLfloat, v: *const GLfloat);
     func!(ReplacementCodeuiTexCoord2fNormal3fVertex3fSUN, (), rc: GLuint, s: GLfloat, t: GLfloat, nx: GLfloat, ny: GLfloat, nz: GLfloat, x: GLfloat, y: GLfloat, z: GLfloat);
     func!(ReplacementCodeuiTexCoord2fNormal3fVertex3fvSUN, (), rc: *const GLuint, tc: *const GLfloat, n: *const GLfloat, v: *const GLfloat);
     func!(ReplacementCodeuiTexCoord2fVertex3fSUN, (), rc: GLuint, s: GLfloat, t: GLfloat, x: GLfloat, y: GLfloat, z: GLfloat);
     func!(ReplacementCodeuiTexCoord2fVertex3fvSUN, (), rc: *const GLuint, tc: *const GLfloat, v: *const GLfloat);
     func!(ReplacementCodeuiVertex3fSUN, (), rc: GLuint, x: GLfloat, y: GLfloat, z: GLfloat);
     func!(ReplacementCodeuiVertex3fvSUN, (), rc: *const GLuint, v: *const GLfloat);
     func!(ReplacementCodeuivSUN, (), code: *const GLuint);
     func!(ReplacementCodeusSUN, (), code: GLushort);
     func!(ReplacementCodeusvSUN, (), code: *const GLushort);
     func!(RequestResidentProgramsNV, (), n: GLsizei, programs: *const GLuint);
     func!(ResetHistogramEXT, (), target: GLenum);
     func!(ResetMemoryObjectParameterNV, (), memory: GLuint, pname: GLenum);
     func!(ResetMinmaxEXT, (), target: GLenum);
     func!(ResizeBuffersMESA, (), );
     func!(ResolveDepthValuesNV, (), );
     func!(ResumeTransformFeedback, (), );
     func!(ResumeTransformFeedbackNV, (), );
     func!(RotatexOES, (), angle: GLfixed, x: GLfixed, y: GLfixed, z: GLfixed);
     func!(SampleCoverage, (), value: GLfloat, invert: GLboolean);
     func!(SampleCoverageARB, (), value: GLfloat, invert: GLboolean);
     func!(SampleMapATI, (), dst: GLuint, interp: GLuint, swizzle: GLenum);
     func!(SampleMaskEXT, (), value: GLclampf, invert: GLboolean);
     func!(SampleMaskIndexedNV, (), index: GLuint, mask: GLbitfield);
     func!(SampleMaskSGIS, (), value: GLclampf, invert: GLboolean);
     func!(SampleMaski, (), maskNumber: GLuint, mask: GLbitfield);
     func!(SamplePatternEXT, (), pattern: GLenum);
     func!(SamplePatternSGIS, (), pattern: GLenum);
     func!(SamplerParameterIiv, (), sampler: GLuint, pname: GLenum, param: *const GLint);
     func!(SamplerParameterIuiv, (), sampler: GLuint, pname: GLenum, param: *const GLuint);
     func!(SamplerParameterf, (), sampler: GLuint, pname: GLenum, param: GLfloat);
     func!(SamplerParameterfv, (), sampler: GLuint, pname: GLenum, param: *const GLfloat);
     func!(SamplerParameteri, (), sampler: GLuint, pname: GLenum, param: GLint);
     func!(SamplerParameteriv, (), sampler: GLuint, pname: GLenum, param: *const GLint);
     func!(ScalexOES, (), x: GLfixed, y: GLfixed, z: GLfixed);
     func!(Scissor, (), x: GLint, y: GLint, width: GLsizei, height: GLsizei);
     func!(ScissorArrayv, (), first: GLuint, count: GLsizei, v: *const GLint);
     func!(ScissorExclusiveArrayvNV, (), first: GLuint, count: GLsizei, v: *const GLint);
     func!(ScissorExclusiveNV, (), x: GLint, y: GLint, width: GLsizei, height: GLsizei);
     func!(ScissorIndexed, (), index: GLuint, left: GLint, bottom: GLint, width: GLsizei, height: GLsizei);
     func!(ScissorIndexedv, (), index: GLuint, v: *const GLint);
     func!(SecondaryColor3bEXT, (), red: GLbyte, green: GLbyte, blue: GLbyte);
     func!(SecondaryColor3bvEXT, (), v: *const GLbyte);
     func!(SecondaryColor3dEXT, (), red: GLdouble, green: GLdouble, blue: GLdouble);
     func!(SecondaryColor3dvEXT, (), v: *const GLdouble);
     func!(SecondaryColor3fEXT, (), red: GLfloat, green: GLfloat, blue: GLfloat);
     func!(SecondaryColor3fvEXT, (), v: *const GLfloat);
     func!(SecondaryColor3hNV, (), red: GLhalfNV, green: GLhalfNV, blue: GLhalfNV);
     func!(SecondaryColor3hvNV, (), v: *const GLhalfNV);
     func!(SecondaryColor3iEXT, (), red: GLint, green: GLint, blue: GLint);
     func!(SecondaryColor3ivEXT, (), v: *const GLint);
     func!(SecondaryColor3sEXT, (), red: GLshort, green: GLshort, blue: GLshort);
     func!(SecondaryColor3svEXT, (), v: *const GLshort);
     func!(SecondaryColor3ubEXT, (), red: GLubyte, green: GLubyte, blue: GLubyte);
     func!(SecondaryColor3ubvEXT, (), v: *const GLubyte);
     func!(SecondaryColor3uiEXT, (), red: GLuint, green: GLuint, blue: GLuint);
     func!(SecondaryColor3uivEXT, (), v: *const GLuint);
     func!(SecondaryColor3usEXT, (), red: GLushort, green: GLushort, blue: GLushort);
     func!(SecondaryColor3usvEXT, (), v: *const GLushort);
     func!(SecondaryColorFormatNV, (), size: GLint, type_: GLenum, stride: GLsizei);
     func!(SecondaryColorPointerEXT, (), size: GLint, type_: GLenum, stride: GLsizei, pointer: *const c_void);
     func!(SecondaryColorPointerListIBM, (), size: GLint, type_: GLenum, stride: GLint, pointer: *const *const c_void, ptrstride: GLint);
     func!(SelectPerfMonitorCountersAMD, (), monitor: GLuint, enable: GLboolean, group: GLuint, numCounters: GLint, counterList: *mut GLuint);
     func!(SemaphoreParameterivNV, (), semaphore: GLuint, pname: GLenum, params: *const GLint);
     func!(SemaphoreParameterui64vEXT, (), semaphore: GLuint, pname: GLenum, params: *const GLuint64);
     func!(SeparableFilter2DEXT, (), target: GLenum, internalformat: GLenum, width: GLsizei, height: GLsizei, format: GLenum, type_: GLenum, row: *const c_void, column: *const c_void);
     func!(SetFenceAPPLE, (), fence: GLuint);
     func!(SetFenceNV, (), fence: GLuint, condition: GLenum);
     func!(SetFragmentShaderConstantATI, (), dst: GLuint, value: *const GLfloat);
     func!(SetInvariantEXT, (), id: GLuint, type_: GLenum, addr: *const c_void);
     func!(SetLocalConstantEXT, (), id: GLuint, type_: GLenum, addr: *const c_void);
     func!(SetMultisamplefvAMD, (), pname: GLenum, index: GLuint, val: *const GLfloat);
     func!(ShaderBinary, (), count: GLsizei, shaders: *const GLuint, binaryFormat: GLenum, binary: *const c_void, length: GLsizei);
     func!(ShaderOp1EXT, (), op: GLenum, res: GLuint, arg1: GLuint);
     func!(ShaderOp2EXT, (), op: GLenum, res: GLuint, arg1: GLuint, arg2: GLuint);
     func!(ShaderOp3EXT, (), op: GLenum, res: GLuint, arg1: GLuint, arg2: GLuint, arg3: GLuint);
     func!(ShaderSource, (), shader: GLuint, count: GLsizei, string: *const *const GLchar, length: *const GLint);
     func!(ShaderSourceARB, (), shaderObj: GLhandleARB, count: GLsizei, string: *const *const GLcharARB, length: *const GLint);
     func!(ShaderStorageBlockBinding, (), program: GLuint, storageBlockIndex: GLuint, storageBlockBinding: GLuint);
     func!(ShadingRateImageBarrierNV, (), synchronize: GLboolean);
     func!(ShadingRateImagePaletteNV, (), viewport: GLuint, first: GLuint, count: GLsizei, rates: *const GLenum);
     func!(ShadingRateSampleOrderCustomNV, (), rate: GLenum, samples: GLuint, locations: *const GLint);
     func!(ShadingRateSampleOrderNV, (), order: GLenum);
     func!(SharpenTexFuncSGIS, (), target: GLenum, n: GLsizei, points: *const GLfloat);
     func!(SignalSemaphoreEXT, (), semaphore: GLuint, numBufferBarriers: GLuint, buffers: *const GLuint, numTextureBarriers: GLuint, textures: *const GLuint, dstLayouts: *const GLenum);
     func!(SignalSemaphoreui64NVX, (), signalGpu: GLuint, fenceObjectCount: GLsizei, semaphoreArray: *const GLuint, fenceValueArray: *const GLuint64);
     func!(SignalVkFenceNV, (), vkFence: GLuint64);
     func!(SignalVkSemaphoreNV, (), vkSemaphore: GLuint64);
     func!(SpecializeShader, (), shader: GLuint, pEntryPoint: *const GLchar, numSpecializationConstants: GLuint, pConstantIndex: *const GLuint, pConstantValue: *const GLuint);
     func!(SpecializeShaderARB, (), shader: GLuint, pEntryPoint: *const GLchar, numSpecializationConstants: GLuint, pConstantIndex: *const GLuint, pConstantValue: *const GLuint);
     func!(SpriteParameterfSGIX, (), pname: GLenum, param: GLfloat);
     func!(SpriteParameterfvSGIX, (), pname: GLenum, params: *const GLfloat);
     func!(SpriteParameteriSGIX, (), pname: GLenum, param: GLint);
     func!(SpriteParameterivSGIX, (), pname: GLenum, params: *const GLint);
     func!(StartInstrumentsSGIX, (), );
     func!(StateCaptureNV, (), state: GLuint, mode: GLenum);
     func!(StencilClearTagEXT, (), stencilTagBits: GLsizei, stencilClearTag: GLuint);
     func!(StencilFillPathInstancedNV, (), numPaths: GLsizei, pathNameType: GLenum, paths: *const c_void, pathBase: GLuint, fillMode: GLenum, mask: GLuint, transformType: GLenum, transformValues: *const GLfloat);
     func!(StencilFillPathNV, (), path: GLuint, fillMode: GLenum, mask: GLuint);
     func!(StencilFunc, (), func: GLenum, ref_: GLint, mask: GLuint);
     func!(StencilFuncSeparate, (), face: GLenum, func: GLenum, ref_: GLint, mask: GLuint);
     func!(StencilFuncSeparateATI, (), frontfunc: GLenum, backfunc: GLenum, ref_: GLint, mask: GLuint);
     func!(StencilMask, (), mask: GLuint);
     func!(StencilMaskSeparate, (), face: GLenum, mask: GLuint);
     func!(StencilOp, (), fail: GLenum, zfail: GLenum, zpass: GLenum);
     func!(StencilOpSeparate, (), face: GLenum, sfail: GLenum, dpfail: GLenum, dppass: GLenum);
     func!(StencilOpSeparateATI, (), face: GLenum, sfail: GLenum, dpfail: GLenum, dppass: GLenum);
     func!(StencilOpValueAMD, (), face: GLenum, value: GLuint);
     func!(StencilStrokePathInstancedNV, (), numPaths: GLsizei, pathNameType: GLenum, paths: *const c_void, pathBase: GLuint, reference: GLint, mask: GLuint, transformType: GLenum, transformValues: *const GLfloat);
     func!(StencilStrokePathNV, (), path: GLuint, reference: GLint, mask: GLuint);
     func!(StencilThenCoverFillPathInstancedNV, (), numPaths: GLsizei, pathNameType: GLenum, paths: *const c_void, pathBase: GLuint, fillMode: GLenum, mask: GLuint, coverMode: GLenum, transformType: GLenum, transformValues: *const GLfloat);
     func!(StencilThenCoverFillPathNV, (), path: GLuint, fillMode: GLenum, mask: GLuint, coverMode: GLenum);
     func!(StencilThenCoverStrokePathInstancedNV, (), numPaths: GLsizei, pathNameType: GLenum, paths: *const c_void, pathBase: GLuint, reference: GLint, mask: GLuint, coverMode: GLenum, transformType: GLenum, transformValues: *const GLfloat);
     func!(StencilThenCoverStrokePathNV, (), path: GLuint, reference: GLint, mask: GLuint, coverMode: GLenum);
     func!(StopInstrumentsSGIX, (), marker: GLint);
     func!(StringMarkerGREMEDY, (), len: GLsizei, string: *const c_void);
     func!(SubpixelPrecisionBiasNV, (), xbits: GLuint, ybits: GLuint);
     func!(SwizzleEXT, (), res: GLuint, in_: GLuint, outX: GLenum, outY: GLenum, outZ: GLenum, outW: GLenum);
     func!(SyncTextureINTEL, (), texture: GLuint);
     func!(TagSampleBufferSGIX, (), );
     func!(Tangent3bEXT, (), tx: GLbyte, ty: GLbyte, tz: GLbyte);
     func!(Tangent3bvEXT, (), v: *const GLbyte);
     func!(Tangent3dEXT, (), tx: GLdouble, ty: GLdouble, tz: GLdouble);
     func!(Tangent3dvEXT, (), v: *const GLdouble);
     func!(Tangent3fEXT, (), tx: GLfloat, ty: GLfloat, tz: GLfloat);
     func!(Tangent3fvEXT, (), v: *const GLfloat);
     func!(Tangent3iEXT, (), tx: GLint, ty: GLint, tz: GLint);
     func!(Tangent3ivEXT, (), v: *const GLint);
     func!(Tangent3sEXT, (), tx: GLshort, ty: GLshort, tz: GLshort);
     func!(Tangent3svEXT, (), v: *const GLshort);
     func!(TangentPointerEXT, (), type_: GLenum, stride: GLsizei, pointer: *const c_void);
     func!(TbufferMask3DFX, (), mask: GLuint);
     func!(TessellationFactorAMD, (), factor: GLfloat);
     func!(TessellationModeAMD, (), mode: GLenum);
     func!(TestFenceAPPLE, GLboolean, fence: GLuint);
     func!(TestFenceNV, GLboolean, fence: GLuint);
     func!(TestObjectAPPLE, GLboolean, object: GLenum, name: GLuint);
     func!(TexAttachMemoryNV, (), target: GLenum, memory: GLuint, offset: GLuint64);
     func!(TexBuffer, (), target: GLenum, internalformat: GLenum, buffer: GLuint);
     func!(TexBufferARB, (), target: GLenum, internalformat: GLenum, buffer: GLuint);
     func!(TexBufferEXT, (), target: GLenum, internalformat: GLenum, buffer: GLuint);
     func!(TexBufferRange, (), target: GLenum, internalformat: GLenum, buffer: GLuint, offset: GLintptr, size: GLsizeiptr);
     func!(TexBumpParameterfvATI, (), pname: GLenum, param: *const GLfloat);
     func!(TexBumpParameterivATI, (), pname: GLenum, param: *const GLint);
     func!(TexCoord1bOES, (), s: GLbyte);
     func!(TexCoord1bvOES, (), coords: *const GLbyte);
     func!(TexCoord1hNV, (), s: GLhalfNV);
     func!(TexCoord1hvNV, (), v: *const GLhalfNV);
     func!(TexCoord1xOES, (), s: GLfixed);
     func!(TexCoord1xvOES, (), coords: *const GLfixed);
     func!(TexCoord2bOES, (), s: GLbyte, t: GLbyte);
     func!(TexCoord2bvOES, (), coords: *const GLbyte);
     func!(TexCoord2fColor3fVertex3fSUN, (), s: GLfloat, t: GLfloat, r: GLfloat, g: GLfloat, b: GLfloat, x: GLfloat, y: GLfloat, z: GLfloat);
     func!(TexCoord2fColor3fVertex3fvSUN, (), tc: *const GLfloat, c: *const GLfloat, v: *const GLfloat);
     func!(TexCoord2fColor4fNormal3fVertex3fSUN, (), s: GLfloat, t: GLfloat, r: GLfloat, g: GLfloat, b: GLfloat, a: GLfloat, nx: GLfloat, ny: GLfloat, nz: GLfloat, x: GLfloat, y: GLfloat, z: GLfloat);
     func!(TexCoord2fColor4fNormal3fVertex3fvSUN, (), tc: *const GLfloat, c: *const GLfloat, n: *const GLfloat, v: *const GLfloat);
     func!(TexCoord2fColor4ubVertex3fSUN, (), s: GLfloat, t: GLfloat, r: GLubyte, g: GLubyte, b: GLubyte, a: GLubyte, x: GLfloat, y: GLfloat, z: GLfloat);
     func!(TexCoord2fColor4ubVertex3fvSUN, (), tc: *const GLfloat, c: *const GLubyte, v: *const GLfloat);
     func!(TexCoord2fNormal3fVertex3fSUN, (), s: GLfloat, t: GLfloat, nx: GLfloat, ny: GLfloat, nz: GLfloat, x: GLfloat, y: GLfloat, z: GLfloat);
     func!(TexCoord2fNormal3fVertex3fvSUN, (), tc: *const GLfloat, n: *const GLfloat, v: *const GLfloat);
     func!(TexCoord2fVertex3fSUN, (), s: GLfloat, t: GLfloat, x: GLfloat, y: GLfloat, z: GLfloat);
     func!(TexCoord2fVertex3fvSUN, (), tc: *const GLfloat, v: *const GLfloat);
     func!(TexCoord2hNV, (), s: GLhalfNV, t: GLhalfNV);
     func!(TexCoord2hvNV, (), v: *const GLhalfNV);
     func!(TexCoord2xOES, (), s: GLfixed, t: GLfixed);
     func!(TexCoord2xvOES, (), coords: *const GLfixed);
     func!(TexCoord3bOES, (), s: GLbyte, t: GLbyte, r: GLbyte);
     func!(TexCoord3bvOES, (), coords: *const GLbyte);
     func!(TexCoord3hNV, (), s: GLhalfNV, t: GLhalfNV, r: GLhalfNV);
     func!(TexCoord3hvNV, (), v: *const GLhalfNV);
     func!(TexCoord3xOES, (), s: GLfixed, t: GLfixed, r: GLfixed);
     func!(TexCoord3xvOES, (), coords: *const GLfixed);
     func!(TexCoord4bOES, (), s: GLbyte, t: GLbyte, r: GLbyte, q: GLbyte);
     func!(TexCoord4bvOES, (), coords: *const GLbyte);
     func!(TexCoord4fColor4fNormal3fVertex4fSUN, (), s: GLfloat, t: GLfloat, p: GLfloat, q: GLfloat, r: GLfloat, g: GLfloat, b: GLfloat, a: GLfloat, nx: GLfloat, ny: GLfloat, nz: GLfloat, x: GLfloat, y: GLfloat, z: GLfloat, w: GLfloat);
     func!(TexCoord4fColor4fNormal3fVertex4fvSUN, (), tc: *const GLfloat, c: *const GLfloat, n: *const GLfloat, v: *const GLfloat);
     func!(TexCoord4fVertex4fSUN, (), s: GLfloat, t: GLfloat, p: GLfloat, q: GLfloat, x: GLfloat, y: GLfloat, z: GLfloat, w: GLfloat);
     func!(TexCoord4fVertex4fvSUN, (), tc: *const GLfloat, v: *const GLfloat);
     func!(TexCoord4hNV, (), s: GLhalfNV, t: GLhalfNV, r: GLhalfNV, q: GLhalfNV);
     func!(TexCoord4hvNV, (), v: *const GLhalfNV);
     func!(TexCoord4xOES, (), s: GLfixed, t: GLfixed, r: GLfixed, q: GLfixed);
     func!(TexCoord4xvOES, (), coords: *const GLfixed);
     func!(TexCoordFormatNV, (), size: GLint, type_: GLenum, stride: GLsizei);
     func!(TexCoordPointerEXT, (), size: GLint, type_: GLenum, stride: GLsizei, count: GLsizei, pointer: *const c_void);
     func!(TexCoordPointerListIBM, (), size: GLint, type_: GLenum, stride: GLint, pointer: *const *const c_void, ptrstride: GLint);
     func!(TexCoordPointervINTEL, (), size: GLint, type_: GLenum, pointer: *const *const c_void);
     func!(TexEnvxOES, (), target: GLenum, pname: GLenum, param: GLfixed);
     func!(TexEnvxvOES, (), target: GLenum, pname: GLenum, params: *const GLfixed);
     func!(TexFilterFuncSGIS, (), target: GLenum, filter: GLenum, n: GLsizei, weights: *const GLfloat);
     func!(TexGenxOES, (), coord: GLenum, pname: GLenum, param: GLfixed);
     func!(TexGenxvOES, (), coord: GLenum, pname: GLenum, params: *const GLfixed);
     func!(TexImage1D, (), target: GLenum, level: GLint, internalformat: GLint, width: GLsizei, border: GLint, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(TexImage2D, (), target: GLenum, level: GLint, internalformat: GLint, width: GLsizei, height: GLsizei, border: GLint, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(TexImage2DMultisample, (), target: GLenum, samples: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei, fixedsamplelocations: GLboolean);
     func!(TexImage2DMultisampleCoverageNV, (), target: GLenum, coverageSamples: GLsizei, colorSamples: GLsizei, internalFormat: GLint, width: GLsizei, height: GLsizei, fixedSampleLocations: GLboolean);
     func!(TexImage3D, (), target: GLenum, level: GLint, internalformat: GLint, width: GLsizei, height: GLsizei, depth: GLsizei, border: GLint, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(TexImage3DEXT, (), target: GLenum, level: GLint, internalformat: GLenum, width: GLsizei, height: GLsizei, depth: GLsizei, border: GLint, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(TexImage3DMultisample, (), target: GLenum, samples: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei, depth: GLsizei, fixedsamplelocations: GLboolean);
     func!(TexImage3DMultisampleCoverageNV, (), target: GLenum, coverageSamples: GLsizei, colorSamples: GLsizei, internalFormat: GLint, width: GLsizei, height: GLsizei, depth: GLsizei, fixedSampleLocations: GLboolean);
     func!(TexImage4DSGIS, (), target: GLenum, level: GLint, internalformat: GLenum, width: GLsizei, height: GLsizei, depth: GLsizei, size4d: GLsizei, border: GLint, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(TexPageCommitmentARB, (), target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, zoffset: GLint, width: GLsizei, height: GLsizei, depth: GLsizei, commit: GLboolean);
     func!(TexPageCommitmentMemNV, (), target: GLenum, layer: GLint, level: GLint, xoffset: GLint, yoffset: GLint, zoffset: GLint, width: GLsizei, height: GLsizei, depth: GLsizei, memory: GLuint, offset: GLuint64, commit: GLboolean);
     func!(TexParameterIiv, (), target: GLenum, pname: GLenum, params: *const GLint);
     func!(TexParameterIivEXT, (), target: GLenum, pname: GLenum, params: *const GLint);
     func!(TexParameterIuiv, (), target: GLenum, pname: GLenum, params: *const GLuint);
     func!(TexParameterIuivEXT, (), target: GLenum, pname: GLenum, params: *const GLuint);
     func!(TexParameterf, (), target: GLenum, pname: GLenum, param: GLfloat);
     func!(TexParameterfv, (), target: GLenum, pname: GLenum, params: *const GLfloat);
     func!(TexParameteri, (), target: GLenum, pname: GLenum, param: GLint);
     func!(TexParameteriv, (), target: GLenum, pname: GLenum, params: *const GLint);
     func!(TexParameterxOES, (), target: GLenum, pname: GLenum, param: GLfixed);
     func!(TexParameterxvOES, (), target: GLenum, pname: GLenum, params: *const GLfixed);
     func!(TexRenderbufferNV, (), target: GLenum, renderbuffer: GLuint);
     func!(TexStorage1D, (), target: GLenum, levels: GLsizei, internalformat: GLenum, width: GLsizei);
     func!(TexStorage1DEXT, (), target: GLenum, levels: GLsizei, internalformat: GLenum, width: GLsizei);
     func!(TexStorage2D, (), target: GLenum, levels: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei);
     func!(TexStorage2DEXT, (), target: GLenum, levels: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei);
     func!(TexStorage2DMultisample, (), target: GLenum, samples: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei, fixedsamplelocations: GLboolean);
     func!(TexStorage3D, (), target: GLenum, levels: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei, depth: GLsizei);
     func!(TexStorage3DEXT, (), target: GLenum, levels: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei, depth: GLsizei);
     func!(TexStorage3DMultisample, (), target: GLenum, samples: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei, depth: GLsizei, fixedsamplelocations: GLboolean);
     func!(TexStorageMem1DEXT, (), target: GLenum, levels: GLsizei, internalFormat: GLenum, width: GLsizei, memory: GLuint, offset: GLuint64);
     func!(TexStorageMem2DEXT, (), target: GLenum, levels: GLsizei, internalFormat: GLenum, width: GLsizei, height: GLsizei, memory: GLuint, offset: GLuint64);
     func!(TexStorageMem2DMultisampleEXT, (), target: GLenum, samples: GLsizei, internalFormat: GLenum, width: GLsizei, height: GLsizei, fixedSampleLocations: GLboolean, memory: GLuint, offset: GLuint64);
     func!(TexStorageMem3DEXT, (), target: GLenum, levels: GLsizei, internalFormat: GLenum, width: GLsizei, height: GLsizei, depth: GLsizei, memory: GLuint, offset: GLuint64);
     func!(TexStorageMem3DMultisampleEXT, (), target: GLenum, samples: GLsizei, internalFormat: GLenum, width: GLsizei, height: GLsizei, depth: GLsizei, fixedSampleLocations: GLboolean, memory: GLuint, offset: GLuint64);
     func!(TexStorageSparseAMD, (), target: GLenum, internalFormat: GLenum, width: GLsizei, height: GLsizei, depth: GLsizei, layers: GLsizei, flags: GLbitfield);
     func!(TexSubImage1D, (), target: GLenum, level: GLint, xoffset: GLint, width: GLsizei, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(TexSubImage1DEXT, (), target: GLenum, level: GLint, xoffset: GLint, width: GLsizei, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(TexSubImage2D, (), target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, width: GLsizei, height: GLsizei, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(TexSubImage2DEXT, (), target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, width: GLsizei, height: GLsizei, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(TexSubImage3D, (), target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, zoffset: GLint, width: GLsizei, height: GLsizei, depth: GLsizei, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(TexSubImage3DEXT, (), target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, zoffset: GLint, width: GLsizei, height: GLsizei, depth: GLsizei, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(TexSubImage4DSGIS, (), target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, zoffset: GLint, woffset: GLint, width: GLsizei, height: GLsizei, depth: GLsizei, size4d: GLsizei, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(TextureAttachMemoryNV, (), texture: GLuint, memory: GLuint, offset: GLuint64);
     func!(TextureBarrier, (), );
     func!(TextureBarrierNV, (), );
     func!(TextureBuffer, (), texture: GLuint, internalformat: GLenum, buffer: GLuint);
     func!(TextureBufferEXT, (), texture: GLuint, target: GLenum, internalformat: GLenum, buffer: GLuint);
     func!(TextureBufferRange, (), texture: GLuint, internalformat: GLenum, buffer: GLuint, offset: GLintptr, size: GLsizeiptr);
     func!(TextureBufferRangeEXT, (), texture: GLuint, target: GLenum, internalformat: GLenum, buffer: GLuint, offset: GLintptr, size: GLsizeiptr);
     func!(TextureColorMaskSGIS, (), red: GLboolean, green: GLboolean, blue: GLboolean, alpha: GLboolean);
     func!(TextureImage1DEXT, (), texture: GLuint, target: GLenum, level: GLint, internalformat: GLint, width: GLsizei, border: GLint, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(TextureImage2DEXT, (), texture: GLuint, target: GLenum, level: GLint, internalformat: GLint, width: GLsizei, height: GLsizei, border: GLint, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(TextureImage2DMultisampleCoverageNV, (), texture: GLuint, target: GLenum, coverageSamples: GLsizei, colorSamples: GLsizei, internalFormat: GLint, width: GLsizei, height: GLsizei, fixedSampleLocations: GLboolean);
     func!(TextureImage2DMultisampleNV, (), texture: GLuint, target: GLenum, samples: GLsizei, internalFormat: GLint, width: GLsizei, height: GLsizei, fixedSampleLocations: GLboolean);
     func!(TextureImage3DEXT, (), texture: GLuint, target: GLenum, level: GLint, internalformat: GLint, width: GLsizei, height: GLsizei, depth: GLsizei, border: GLint, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(TextureImage3DMultisampleCoverageNV, (), texture: GLuint, target: GLenum, coverageSamples: GLsizei, colorSamples: GLsizei, internalFormat: GLint, width: GLsizei, height: GLsizei, depth: GLsizei, fixedSampleLocations: GLboolean);
     func!(TextureImage3DMultisampleNV, (), texture: GLuint, target: GLenum, samples: GLsizei, internalFormat: GLint, width: GLsizei, height: GLsizei, depth: GLsizei, fixedSampleLocations: GLboolean);
     func!(TextureLightEXT, (), pname: GLenum);
     func!(TextureMaterialEXT, (), face: GLenum, mode: GLenum);
     func!(TextureNormalEXT, (), mode: GLenum);
     func!(TexturePageCommitmentEXT, (), texture: GLuint, level: GLint, xoffset: GLint, yoffset: GLint, zoffset: GLint, width: GLsizei, height: GLsizei, depth: GLsizei, commit: GLboolean);
     func!(TexturePageCommitmentMemNV, (), texture: GLuint, layer: GLint, level: GLint, xoffset: GLint, yoffset: GLint, zoffset: GLint, width: GLsizei, height: GLsizei, depth: GLsizei, memory: GLuint, offset: GLuint64, commit: GLboolean);
     func!(TextureParameterIiv, (), texture: GLuint, pname: GLenum, params: *const GLint);
     func!(TextureParameterIivEXT, (), texture: GLuint, target: GLenum, pname: GLenum, params: *const GLint);
     func!(TextureParameterIuiv, (), texture: GLuint, pname: GLenum, params: *const GLuint);
     func!(TextureParameterIuivEXT, (), texture: GLuint, target: GLenum, pname: GLenum, params: *const GLuint);
     func!(TextureParameterf, (), texture: GLuint, pname: GLenum, param: GLfloat);
     func!(TextureParameterfEXT, (), texture: GLuint, target: GLenum, pname: GLenum, param: GLfloat);
     func!(TextureParameterfv, (), texture: GLuint, pname: GLenum, param: *const GLfloat);
     func!(TextureParameterfvEXT, (), texture: GLuint, target: GLenum, pname: GLenum, params: *const GLfloat);
     func!(TextureParameteri, (), texture: GLuint, pname: GLenum, param: GLint);
     func!(TextureParameteriEXT, (), texture: GLuint, target: GLenum, pname: GLenum, param: GLint);
     func!(TextureParameteriv, (), texture: GLuint, pname: GLenum, param: *const GLint);
     func!(TextureParameterivEXT, (), texture: GLuint, target: GLenum, pname: GLenum, params: *const GLint);
     func!(TextureRangeAPPLE, (), target: GLenum, length: GLsizei, pointer: *const c_void);
     func!(TextureRenderbufferEXT, (), texture: GLuint, target: GLenum, renderbuffer: GLuint);
     func!(TextureStorage1D, (), texture: GLuint, levels: GLsizei, internalformat: GLenum, width: GLsizei);
     func!(TextureStorage1DEXT, (), texture: GLuint, target: GLenum, levels: GLsizei, internalformat: GLenum, width: GLsizei);
     func!(TextureStorage2D, (), texture: GLuint, levels: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei);
     func!(TextureStorage2DEXT, (), texture: GLuint, target: GLenum, levels: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei);
     func!(TextureStorage2DMultisample, (), texture: GLuint, samples: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei, fixedsamplelocations: GLboolean);
     func!(TextureStorage2DMultisampleEXT, (), texture: GLuint, target: GLenum, samples: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei, fixedsamplelocations: GLboolean);
     func!(TextureStorage3D, (), texture: GLuint, levels: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei, depth: GLsizei);
     func!(TextureStorage3DEXT, (), texture: GLuint, target: GLenum, levels: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei, depth: GLsizei);
     func!(TextureStorage3DMultisample, (), texture: GLuint, samples: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei, depth: GLsizei, fixedsamplelocations: GLboolean);
     func!(TextureStorage3DMultisampleEXT, (), texture: GLuint, target: GLenum, samples: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei, depth: GLsizei, fixedsamplelocations: GLboolean);
     func!(TextureStorageMem1DEXT, (), texture: GLuint, levels: GLsizei, internalFormat: GLenum, width: GLsizei, memory: GLuint, offset: GLuint64);
     func!(TextureStorageMem2DEXT, (), texture: GLuint, levels: GLsizei, internalFormat: GLenum, width: GLsizei, height: GLsizei, memory: GLuint, offset: GLuint64);
     func!(TextureStorageMem2DMultisampleEXT, (), texture: GLuint, samples: GLsizei, internalFormat: GLenum, width: GLsizei, height: GLsizei, fixedSampleLocations: GLboolean, memory: GLuint, offset: GLuint64);
     func!(TextureStorageMem3DEXT, (), texture: GLuint, levels: GLsizei, internalFormat: GLenum, width: GLsizei, height: GLsizei, depth: GLsizei, memory: GLuint, offset: GLuint64);
     func!(TextureStorageMem3DMultisampleEXT, (), texture: GLuint, samples: GLsizei, internalFormat: GLenum, width: GLsizei, height: GLsizei, depth: GLsizei, fixedSampleLocations: GLboolean, memory: GLuint, offset: GLuint64);
     func!(TextureStorageSparseAMD, (), texture: GLuint, target: GLenum, internalFormat: GLenum, width: GLsizei, height: GLsizei, depth: GLsizei, layers: GLsizei, flags: GLbitfield);
     func!(TextureSubImage1D, (), texture: GLuint, level: GLint, xoffset: GLint, width: GLsizei, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(TextureSubImage1DEXT, (), texture: GLuint, target: GLenum, level: GLint, xoffset: GLint, width: GLsizei, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(TextureSubImage2D, (), texture: GLuint, level: GLint, xoffset: GLint, yoffset: GLint, width: GLsizei, height: GLsizei, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(TextureSubImage2DEXT, (), texture: GLuint, target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, width: GLsizei, height: GLsizei, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(TextureSubImage3D, (), texture: GLuint, level: GLint, xoffset: GLint, yoffset: GLint, zoffset: GLint, width: GLsizei, height: GLsizei, depth: GLsizei, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(TextureSubImage3DEXT, (), texture: GLuint, target: GLenum, level: GLint, xoffset: GLint, yoffset: GLint, zoffset: GLint, width: GLsizei, height: GLsizei, depth: GLsizei, format: GLenum, type_: GLenum, pixels: *const c_void);
     func!(TextureView, (), texture: GLuint, target: GLenum, origtexture: GLuint, internalformat: GLenum, minlevel: GLuint, numlevels: GLuint, minlayer: GLuint, numlayers: GLuint);
     func!(TrackMatrixNV, (), target: GLenum, address: GLuint, matrix: GLenum, transform: GLenum);
     func!(TransformFeedbackAttribsNV, (), count: GLsizei, attribs: *const GLint, bufferMode: GLenum);
     func!(TransformFeedbackBufferBase, (), xfb: GLuint, index: GLuint, buffer: GLuint);
     func!(TransformFeedbackBufferRange, (), xfb: GLuint, index: GLuint, buffer: GLuint, offset: GLintptr, size: GLsizeiptr);
     func!(TransformFeedbackStreamAttribsNV, (), count: GLsizei, attribs: *const GLint, nbuffers: GLsizei, bufstreams: *const GLint, bufferMode: GLenum);
     func!(TransformFeedbackVaryings, (), program: GLuint, count: GLsizei, varyings: *const *const GLchar, bufferMode: GLenum);
     func!(TransformFeedbackVaryingsEXT, (), program: GLuint, count: GLsizei, varyings: *const *const GLchar, bufferMode: GLenum);
     func!(TransformFeedbackVaryingsNV, (), program: GLuint, count: GLsizei, locations: *const GLint, bufferMode: GLenum);
     func!(TransformPathNV, (), resultPath: GLuint, srcPath: GLuint, transformType: GLenum, transformValues: *const GLfloat);
     func!(TranslatexOES, (), x: GLfixed, y: GLfixed, z: GLfixed);
     func!(Uniform1d, (), location: GLint, x: GLdouble);
     func!(Uniform1dv, (), location: GLint, count: GLsizei, value: *const GLdouble);
     func!(Uniform1f, (), location: GLint, v0: GLfloat);
     func!(Uniform1fARB, (), location: GLint, v0: GLfloat);
     func!(Uniform1fv, (), location: GLint, count: GLsizei, value: *const GLfloat);
     func!(Uniform1fvARB, (), location: GLint, count: GLsizei, value: *const GLfloat);
     func!(Uniform1i, (), location: GLint, v0: GLint);
     func!(Uniform1i64ARB, (), location: GLint, x: GLint64);
     func!(Uniform1i64NV, (), location: GLint, x: GLint64EXT);
     func!(Uniform1i64vARB, (), location: GLint, count: GLsizei, value: *const GLint64);
     func!(Uniform1i64vNV, (), location: GLint, count: GLsizei, value: *const GLint64EXT);
     func!(Uniform1iARB, (), location: GLint, v0: GLint);
     func!(Uniform1iv, (), location: GLint, count: GLsizei, value: *const GLint);
     func!(Uniform1ivARB, (), location: GLint, count: GLsizei, value: *const GLint);
     func!(Uniform1ui, (), location: GLint, v0: GLuint);
     func!(Uniform1ui64ARB, (), location: GLint, x: GLuint64);
     func!(Uniform1ui64NV, (), location: GLint, x: GLuint64EXT);
     func!(Uniform1ui64vARB, (), location: GLint, count: GLsizei, value: *const GLuint64);
     func!(Uniform1ui64vNV, (), location: GLint, count: GLsizei, value: *const GLuint64EXT);
     func!(Uniform1uiEXT, (), location: GLint, v0: GLuint);
     func!(Uniform1uiv, (), location: GLint, count: GLsizei, value: *const GLuint);
     func!(Uniform1uivEXT, (), location: GLint, count: GLsizei, value: *const GLuint);
     func!(Uniform2d, (), location: GLint, x: GLdouble, y: GLdouble);
     func!(Uniform2dv, (), location: GLint, count: GLsizei, value: *const GLdouble);
     func!(Uniform2f, (), location: GLint, v0: GLfloat, v1: GLfloat);
     func!(Uniform2fARB, (), location: GLint, v0: GLfloat, v1: GLfloat);
     func!(Uniform2fv, (), location: GLint, count: GLsizei, value: *const GLfloat);
     func!(Uniform2fvARB, (), location: GLint, count: GLsizei, value: *const GLfloat);
     func!(Uniform2i, (), location: GLint, v0: GLint, v1: GLint);
     func!(Uniform2i64ARB, (), location: GLint, x: GLint64, y: GLint64);
     func!(Uniform2i64NV, (), location: GLint, x: GLint64EXT, y: GLint64EXT);
     func!(Uniform2i64vARB, (), location: GLint, count: GLsizei, value: *const GLint64);
     func!(Uniform2i64vNV, (), location: GLint, count: GLsizei, value: *const GLint64EXT);
     func!(Uniform2iARB, (), location: GLint, v0: GLint, v1: GLint);
     func!(Uniform2iv, (), location: GLint, count: GLsizei, value: *const GLint);
     func!(Uniform2ivARB, (), location: GLint, count: GLsizei, value: *const GLint);
     func!(Uniform2ui, (), location: GLint, v0: GLuint, v1: GLuint);
     func!(Uniform2ui64ARB, (), location: GLint, x: GLuint64, y: GLuint64);
     func!(Uniform2ui64NV, (), location: GLint, x: GLuint64EXT, y: GLuint64EXT);
     func!(Uniform2ui64vARB, (), location: GLint, count: GLsizei, value: *const GLuint64);
     func!(Uniform2ui64vNV, (), location: GLint, count: GLsizei, value: *const GLuint64EXT);
     func!(Uniform2uiEXT, (), location: GLint, v0: GLuint, v1: GLuint);
     func!(Uniform2uiv, (), location: GLint, count: GLsizei, value: *const GLuint);
     func!(Uniform2uivEXT, (), location: GLint, count: GLsizei, value: *const GLuint);
     func!(Uniform3d, (), location: GLint, x: GLdouble, y: GLdouble, z: GLdouble);
     func!(Uniform3dv, (), location: GLint, count: GLsizei, value: *const GLdouble);
     func!(Uniform3f, (), location: GLint, v0: GLfloat, v1: GLfloat, v2: GLfloat);
     func!(Uniform3fARB, (), location: GLint, v0: GLfloat, v1: GLfloat, v2: GLfloat);
     func!(Uniform3fv, (), location: GLint, count: GLsizei, value: *const GLfloat);
     func!(Uniform3fvARB, (), location: GLint, count: GLsizei, value: *const GLfloat);
     func!(Uniform3i, (), location: GLint, v0: GLint, v1: GLint, v2: GLint);
     func!(Uniform3i64ARB, (), location: GLint, x: GLint64, y: GLint64, z: GLint64);
     func!(Uniform3i64NV, (), location: GLint, x: GLint64EXT, y: GLint64EXT, z: GLint64EXT);
     func!(Uniform3i64vARB, (), location: GLint, count: GLsizei, value: *const GLint64);
     func!(Uniform3i64vNV, (), location: GLint, count: GLsizei, value: *const GLint64EXT);
     func!(Uniform3iARB, (), location: GLint, v0: GLint, v1: GLint, v2: GLint);
     func!(Uniform3iv, (), location: GLint, count: GLsizei, value: *const GLint);
     func!(Uniform3ivARB, (), location: GLint, count: GLsizei, value: *const GLint);
     func!(Uniform3ui, (), location: GLint, v0: GLuint, v1: GLuint, v2: GLuint);
     func!(Uniform3ui64ARB, (), location: GLint, x: GLuint64, y: GLuint64, z: GLuint64);
     func!(Uniform3ui64NV, (), location: GLint, x: GLuint64EXT, y: GLuint64EXT, z: GLuint64EXT);
     func!(Uniform3ui64vARB, (), location: GLint, count: GLsizei, value: *const GLuint64);
     func!(Uniform3ui64vNV, (), location: GLint, count: GLsizei, value: *const GLuint64EXT);
     func!(Uniform3uiEXT, (), location: GLint, v0: GLuint, v1: GLuint, v2: GLuint);
     func!(Uniform3uiv, (), location: GLint, count: GLsizei, value: *const GLuint);
     func!(Uniform3uivEXT, (), location: GLint, count: GLsizei, value: *const GLuint);
     func!(Uniform4d, (), location: GLint, x: GLdouble, y: GLdouble, z: GLdouble, w: GLdouble);
     func!(Uniform4dv, (), location: GLint, count: GLsizei, value: *const GLdouble);
     func!(Uniform4f, (), location: GLint, v0: GLfloat, v1: GLfloat, v2: GLfloat, v3: GLfloat);
     func!(Uniform4fARB, (), location: GLint, v0: GLfloat, v1: GLfloat, v2: GLfloat, v3: GLfloat);
     func!(Uniform4fv, (), location: GLint, count: GLsizei, value: *const GLfloat);
     func!(Uniform4fvARB, (), location: GLint, count: GLsizei, value: *const GLfloat);
     func!(Uniform4i, (), location: GLint, v0: GLint, v1: GLint, v2: GLint, v3: GLint);
     func!(Uniform4i64ARB, (), location: GLint, x: GLint64, y: GLint64, z: GLint64, w: GLint64);
     func!(Uniform4i64NV, (), location: GLint, x: GLint64EXT, y: GLint64EXT, z: GLint64EXT, w: GLint64EXT);
     func!(Uniform4i64vARB, (), location: GLint, count: GLsizei, value: *const GLint64);
     func!(Uniform4i64vNV, (), location: GLint, count: GLsizei, value: *const GLint64EXT);
     func!(Uniform4iARB, (), location: GLint, v0: GLint, v1: GLint, v2: GLint, v3: GLint);
     func!(Uniform4iv, (), location: GLint, count: GLsizei, value: *const GLint);
     func!(Uniform4ivARB, (), location: GLint, count: GLsizei, value: *const GLint);
     func!(Uniform4ui, (), location: GLint, v0: GLuint, v1: GLuint, v2: GLuint, v3: GLuint);
     func!(Uniform4ui64ARB, (), location: GLint, x: GLuint64, y: GLuint64, z: GLuint64, w: GLuint64);
     func!(Uniform4ui64NV, (), location: GLint, x: GLuint64EXT, y: GLuint64EXT, z: GLuint64EXT, w: GLuint64EXT);
     func!(Uniform4ui64vARB, (), location: GLint, count: GLsizei, value: *const GLuint64);
     func!(Uniform4ui64vNV, (), location: GLint, count: GLsizei, value: *const GLuint64EXT);
     func!(Uniform4uiEXT, (), location: GLint, v0: GLuint, v1: GLuint, v2: GLuint, v3: GLuint);
     func!(Uniform4uiv, (), location: GLint, count: GLsizei, value: *const GLuint);
     func!(Uniform4uivEXT, (), location: GLint, count: GLsizei, value: *const GLuint);
     func!(UniformBlockBinding, (), program: GLuint, uniformBlockIndex: GLuint, uniformBlockBinding: GLuint);
     func!(UniformBufferEXT, (), program: GLuint, location: GLint, buffer: GLuint);
     func!(UniformHandleui64ARB, (), location: GLint, value: GLuint64);
     func!(UniformHandleui64NV, (), location: GLint, value: GLuint64);
     func!(UniformHandleui64vARB, (), location: GLint, count: GLsizei, value: *const GLuint64);
     func!(UniformHandleui64vNV, (), location: GLint, count: GLsizei, value: *const GLuint64);
     func!(UniformMatrix2dv, (), location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(UniformMatrix2fv, (), location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(UniformMatrix2fvARB, (), location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(UniformMatrix2x3dv, (), location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(UniformMatrix2x3fv, (), location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(UniformMatrix2x4dv, (), location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(UniformMatrix2x4fv, (), location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(UniformMatrix3dv, (), location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(UniformMatrix3fv, (), location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(UniformMatrix3fvARB, (), location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(UniformMatrix3x2dv, (), location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(UniformMatrix3x2fv, (), location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(UniformMatrix3x4dv, (), location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(UniformMatrix3x4fv, (), location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(UniformMatrix4dv, (), location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(UniformMatrix4fv, (), location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(UniformMatrix4fvARB, (), location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(UniformMatrix4x2dv, (), location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(UniformMatrix4x2fv, (), location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(UniformMatrix4x3dv, (), location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLdouble);
     func!(UniformMatrix4x3fv, (), location: GLint, count: GLsizei, transpose: GLboolean, value: *const GLfloat);
     func!(UniformSubroutinesuiv, (), shadertype: GLenum, count: GLsizei, indices: *const GLuint);
     func!(Uniformui64NV, (), location: GLint, value: GLuint64EXT);
     func!(Uniformui64vNV, (), location: GLint, count: GLsizei, value: *const GLuint64EXT);
     func!(UnlockArraysEXT, (), );
     func!(UnmapBuffer, GLboolean, target: GLenum);
     func!(UnmapBufferARB, GLboolean, target: GLenum);
     func!(UnmapNamedBuffer, GLboolean, buffer: GLuint);
     func!(UnmapNamedBufferEXT, GLboolean, buffer: GLuint);
     func!(UnmapObjectBufferATI, (), buffer: GLuint);
     func!(UnmapTexture2DINTEL, (), texture: GLuint, level: GLint);
     func!(UpdateObjectBufferATI, (), buffer: GLuint, offset: GLuint, size: GLsizei, pointer: *const c_void, preserve: GLenum);
     func!(UploadGpuMaskNVX, (), mask: GLbitfield);
     func!(UseProgram, (), program: GLuint);
     func!(UseProgramObjectARB, (), programObj: GLhandleARB);
     func!(UseProgramStages, (), pipeline: GLuint, stages: GLbitfield, program: GLuint);
     func!(UseShaderProgramEXT, (), type_: GLenum, program: GLuint);
     func!(VDPAUFiniNV, (), );
     func!(VDPAUGetSurfaceivNV, (), surface: GLvdpauSurfaceNV, pname: GLenum, count: GLsizei, length: *mut GLsizei, values: *mut GLint);
     func!(VDPAUInitNV, (), vdpDevice: *const c_void, getProcAddress: *const c_void);
     func!(VDPAUIsSurfaceNV, GLboolean, surface: GLvdpauSurfaceNV);
     func!(VDPAUMapSurfacesNV, (), numSurfaces: GLsizei, surfaces: *const GLvdpauSurfaceNV);
     func!(VDPAURegisterOutputSurfaceNV, GLvdpauSurfaceNV, vdpSurface: *const c_void, target: GLenum, numTextureNames: GLsizei, textureNames: *const GLuint);
     func!(VDPAURegisterVideoSurfaceNV, GLvdpauSurfaceNV, vdpSurface: *const c_void, target: GLenum, numTextureNames: GLsizei, textureNames: *const GLuint);
     func!(VDPAURegisterVideoSurfaceWithPictureStructureNV, GLvdpauSurfaceNV, vdpSurface: *const c_void, target: GLenum, numTextureNames: GLsizei, textureNames: *const GLuint, isFrameStructure: GLboolean);
     func!(VDPAUSurfaceAccessNV, (), surface: GLvdpauSurfaceNV, access: GLenum);
     func!(VDPAUUnmapSurfacesNV, (), numSurface: GLsizei, surfaces: *const GLvdpauSurfaceNV);
     func!(VDPAUUnregisterSurfaceNV, (), surface: GLvdpauSurfaceNV);
     func!(ValidateProgram, (), program: GLuint);
     func!(ValidateProgramARB, (), programObj: GLhandleARB);
     func!(ValidateProgramPipeline, (), pipeline: GLuint);
     func!(VariantArrayObjectATI, (), id: GLuint, type_: GLenum, stride: GLsizei, buffer: GLuint, offset: GLuint);
     func!(VariantPointerEXT, (), id: GLuint, type_: GLenum, stride: GLuint, addr: *const c_void);
     func!(VariantbvEXT, (), id: GLuint, addr: *const GLbyte);
     func!(VariantdvEXT, (), id: GLuint, addr: *const GLdouble);
     func!(VariantfvEXT, (), id: GLuint, addr: *const GLfloat);
     func!(VariantivEXT, (), id: GLuint, addr: *const GLint);
     func!(VariantsvEXT, (), id: GLuint, addr: *const GLshort);
     func!(VariantubvEXT, (), id: GLuint, addr: *const GLubyte);
     func!(VariantuivEXT, (), id: GLuint, addr: *const GLuint);
     func!(VariantusvEXT, (), id: GLuint, addr: *const GLushort);
     func!(Vertex2bOES, (), x: GLbyte, y: GLbyte);
     func!(Vertex2bvOES, (), coords: *const GLbyte);
     func!(Vertex2hNV, (), x: GLhalfNV, y: GLhalfNV);
     func!(Vertex2hvNV, (), v: *const GLhalfNV);
     func!(Vertex2xOES, (), x: GLfixed);
     func!(Vertex2xvOES, (), coords: *const GLfixed);
     func!(Vertex3bOES, (), x: GLbyte, y: GLbyte, z: GLbyte);
     func!(Vertex3bvOES, (), coords: *const GLbyte);
     func!(Vertex3hNV, (), x: GLhalfNV, y: GLhalfNV, z: GLhalfNV);
     func!(Vertex3hvNV, (), v: *const GLhalfNV);
     func!(Vertex3xOES, (), x: GLfixed, y: GLfixed);
     func!(Vertex3xvOES, (), coords: *const GLfixed);
     func!(Vertex4bOES, (), x: GLbyte, y: GLbyte, z: GLbyte, w: GLbyte);
     func!(Vertex4bvOES, (), coords: *const GLbyte);
     func!(Vertex4hNV, (), x: GLhalfNV, y: GLhalfNV, z: GLhalfNV, w: GLhalfNV);
     func!(Vertex4hvNV, (), v: *const GLhalfNV);
     func!(Vertex4xOES, (), x: GLfixed, y: GLfixed, z: GLfixed);
     func!(Vertex4xvOES, (), coords: *const GLfixed);
     func!(VertexArrayAttribBinding, (), vaobj: GLuint, attribindex: GLuint, bindingindex: GLuint);
     func!(VertexArrayAttribFormat, (), vaobj: GLuint, attribindex: GLuint, size: GLint, type_: GLenum, normalized: GLboolean, relativeoffset: GLuint);
     func!(VertexArrayAttribIFormat, (), vaobj: GLuint, attribindex: GLuint, size: GLint, type_: GLenum, relativeoffset: GLuint);
     func!(VertexArrayAttribLFormat, (), vaobj: GLuint, attribindex: GLuint, size: GLint, type_: GLenum, relativeoffset: GLuint);
     func!(VertexArrayBindVertexBufferEXT, (), vaobj: GLuint, bindingindex: GLuint, buffer: GLuint, offset: GLintptr, stride: GLsizei);
     func!(VertexArrayBindingDivisor, (), vaobj: GLuint, bindingindex: GLuint, divisor: GLuint);
     func!(VertexArrayColorOffsetEXT, (), vaobj: GLuint, buffer: GLuint, size: GLint, type_: GLenum, stride: GLsizei, offset: GLintptr);
     func!(VertexArrayEdgeFlagOffsetEXT, (), vaobj: GLuint, buffer: GLuint, stride: GLsizei, offset: GLintptr);
     func!(VertexArrayElementBuffer, (), vaobj: GLuint, buffer: GLuint);
     func!(VertexArrayFogCoordOffsetEXT, (), vaobj: GLuint, buffer: GLuint, type_: GLenum, stride: GLsizei, offset: GLintptr);
     func!(VertexArrayIndexOffsetEXT, (), vaobj: GLuint, buffer: GLuint, type_: GLenum, stride: GLsizei, offset: GLintptr);
     func!(VertexArrayMultiTexCoordOffsetEXT, (), vaobj: GLuint, buffer: GLuint, texunit: GLenum, size: GLint, type_: GLenum, stride: GLsizei, offset: GLintptr);
     func!(VertexArrayNormalOffsetEXT, (), vaobj: GLuint, buffer: GLuint, type_: GLenum, stride: GLsizei, offset: GLintptr);
     func!(VertexArrayParameteriAPPLE, (), pname: GLenum, param: GLint);
     func!(VertexArrayRangeAPPLE, (), length: GLsizei, pointer: *mut c_void);
     func!(VertexArrayRangeNV, (), length: GLsizei, pointer: *const c_void);
     func!(VertexArraySecondaryColorOffsetEXT, (), vaobj: GLuint, buffer: GLuint, size: GLint, type_: GLenum, stride: GLsizei, offset: GLintptr);
     func!(VertexArrayTexCoordOffsetEXT, (), vaobj: GLuint, buffer: GLuint, size: GLint, type_: GLenum, stride: GLsizei, offset: GLintptr);
     func!(VertexArrayVertexAttribBindingEXT, (), vaobj: GLuint, attribindex: GLuint, bindingindex: GLuint);
     func!(VertexArrayVertexAttribDivisorEXT, (), vaobj: GLuint, index: GLuint, divisor: GLuint);
     func!(VertexArrayVertexAttribFormatEXT, (), vaobj: GLuint, attribindex: GLuint, size: GLint, type_: GLenum, normalized: GLboolean, relativeoffset: GLuint);
     func!(VertexArrayVertexAttribIFormatEXT, (), vaobj: GLuint, attribindex: GLuint, size: GLint, type_: GLenum, relativeoffset: GLuint);
     func!(VertexArrayVertexAttribIOffsetEXT, (), vaobj: GLuint, buffer: GLuint, index: GLuint, size: GLint, type_: GLenum, stride: GLsizei, offset: GLintptr);
     func!(VertexArrayVertexAttribLFormatEXT, (), vaobj: GLuint, attribindex: GLuint, size: GLint, type_: GLenum, relativeoffset: GLuint);
     func!(VertexArrayVertexAttribLOffsetEXT, (), vaobj: GLuint, buffer: GLuint, index: GLuint, size: GLint, type_: GLenum, stride: GLsizei, offset: GLintptr);
     func!(VertexArrayVertexAttribOffsetEXT, (), vaobj: GLuint, buffer: GLuint, index: GLuint, size: GLint, type_: GLenum, normalized: GLboolean, stride: GLsizei, offset: GLintptr);
     func!(VertexArrayVertexBindingDivisorEXT, (), vaobj: GLuint, bindingindex: GLuint, divisor: GLuint);
     func!(VertexArrayVertexBuffer, (), vaobj: GLuint, bindingindex: GLuint, buffer: GLuint, offset: GLintptr, stride: GLsizei);
     func!(VertexArrayVertexBuffers, (), vaobj: GLuint, first: GLuint, count: GLsizei, buffers: *const GLuint, offsets: *const GLintptr, strides: *const GLsizei);
     func!(VertexArrayVertexOffsetEXT, (), vaobj: GLuint, buffer: GLuint, size: GLint, type_: GLenum, stride: GLsizei, offset: GLintptr);
     func!(VertexAttrib1d, (), index: GLuint, x: GLdouble);
     func!(VertexAttrib1dARB, (), index: GLuint, x: GLdouble);
     func!(VertexAttrib1dNV, (), index: GLuint, x: GLdouble);
     func!(VertexAttrib1dv, (), index: GLuint, v: *const GLdouble);
     func!(VertexAttrib1dvARB, (), index: GLuint, v: *const GLdouble);
     func!(VertexAttrib1dvNV, (), index: GLuint, v: *const GLdouble);
     func!(VertexAttrib1f, (), index: GLuint, x: GLfloat);
     func!(VertexAttrib1fARB, (), index: GLuint, x: GLfloat);
     func!(VertexAttrib1fNV, (), index: GLuint, x: GLfloat);
     func!(VertexAttrib1fv, (), index: GLuint, v: *const GLfloat);
     func!(VertexAttrib1fvARB, (), index: GLuint, v: *const GLfloat);
     func!(VertexAttrib1fvNV, (), index: GLuint, v: *const GLfloat);
     func!(VertexAttrib1hNV, (), index: GLuint, x: GLhalfNV);
     func!(VertexAttrib1hvNV, (), index: GLuint, v: *const GLhalfNV);
     func!(VertexAttrib1s, (), index: GLuint, x: GLshort);
     func!(VertexAttrib1sARB, (), index: GLuint, x: GLshort);
     func!(VertexAttrib1sNV, (), index: GLuint, x: GLshort);
     func!(VertexAttrib1sv, (), index: GLuint, v: *const GLshort);
     func!(VertexAttrib1svARB, (), index: GLuint, v: *const GLshort);
     func!(VertexAttrib1svNV, (), index: GLuint, v: *const GLshort);
     func!(VertexAttrib2d, (), index: GLuint, x: GLdouble, y: GLdouble);
     func!(VertexAttrib2dARB, (), index: GLuint, x: GLdouble, y: GLdouble);
     func!(VertexAttrib2dNV, (), index: GLuint, x: GLdouble, y: GLdouble);
     func!(VertexAttrib2dv, (), index: GLuint, v: *const GLdouble);
     func!(VertexAttrib2dvARB, (), index: GLuint, v: *const GLdouble);
     func!(VertexAttrib2dvNV, (), index: GLuint, v: *const GLdouble);
     func!(VertexAttrib2f, (), index: GLuint, x: GLfloat, y: GLfloat);
     func!(VertexAttrib2fARB, (), index: GLuint, x: GLfloat, y: GLfloat);
     func!(VertexAttrib2fNV, (), index: GLuint, x: GLfloat, y: GLfloat);
     func!(VertexAttrib2fv, (), index: GLuint, v: *const GLfloat);
     func!(VertexAttrib2fvARB, (), index: GLuint, v: *const GLfloat);
     func!(VertexAttrib2fvNV, (), index: GLuint, v: *const GLfloat);
     func!(VertexAttrib2hNV, (), index: GLuint, x: GLhalfNV, y: GLhalfNV);
     func!(VertexAttrib2hvNV, (), index: GLuint, v: *const GLhalfNV);
     func!(VertexAttrib2s, (), index: GLuint, x: GLshort, y: GLshort);
     func!(VertexAttrib2sARB, (), index: GLuint, x: GLshort, y: GLshort);
     func!(VertexAttrib2sNV, (), index: GLuint, x: GLshort, y: GLshort);
     func!(VertexAttrib2sv, (), index: GLuint, v: *const GLshort);
     func!(VertexAttrib2svARB, (), index: GLuint, v: *const GLshort);
     func!(VertexAttrib2svNV, (), index: GLuint, v: *const GLshort);
     func!(VertexAttrib3d, (), index: GLuint, x: GLdouble, y: GLdouble, z: GLdouble);
     func!(VertexAttrib3dARB, (), index: GLuint, x: GLdouble, y: GLdouble, z: GLdouble);
     func!(VertexAttrib3dNV, (), index: GLuint, x: GLdouble, y: GLdouble, z: GLdouble);
     func!(VertexAttrib3dv, (), index: GLuint, v: *const GLdouble);
     func!(VertexAttrib3dvARB, (), index: GLuint, v: *const GLdouble);
     func!(VertexAttrib3dvNV, (), index: GLuint, v: *const GLdouble);
     func!(VertexAttrib3f, (), index: GLuint, x: GLfloat, y: GLfloat, z: GLfloat);
     func!(VertexAttrib3fARB, (), index: GLuint, x: GLfloat, y: GLfloat, z: GLfloat);
     func!(VertexAttrib3fNV, (), index: GLuint, x: GLfloat, y: GLfloat, z: GLfloat);
     func!(VertexAttrib3fv, (), index: GLuint, v: *const GLfloat);
     func!(VertexAttrib3fvARB, (), index: GLuint, v: *const GLfloat);
     func!(VertexAttrib3fvNV, (), index: GLuint, v: *const GLfloat);
     func!(VertexAttrib3hNV, (), index: GLuint, x: GLhalfNV, y: GLhalfNV, z: GLhalfNV);
     func!(VertexAttrib3hvNV, (), index: GLuint, v: *const GLhalfNV);
     func!(VertexAttrib3s, (), index: GLuint, x: GLshort, y: GLshort, z: GLshort);
     func!(VertexAttrib3sARB, (), index: GLuint, x: GLshort, y: GLshort, z: GLshort);
     func!(VertexAttrib3sNV, (), index: GLuint, x: GLshort, y: GLshort, z: GLshort);
     func!(VertexAttrib3sv, (), index: GLuint, v: *const GLshort);
     func!(VertexAttrib3svARB, (), index: GLuint, v: *const GLshort);
     func!(VertexAttrib3svNV, (), index: GLuint, v: *const GLshort);
     func!(VertexAttrib4Nbv, (), index: GLuint, v: *const GLbyte);
     func!(VertexAttrib4NbvARB, (), index: GLuint, v: *const GLbyte);
     func!(VertexAttrib4Niv, (), index: GLuint, v: *const GLint);
     func!(VertexAttrib4NivARB, (), index: GLuint, v: *const GLint);
     func!(VertexAttrib4Nsv, (), index: GLuint, v: *const GLshort);
     func!(VertexAttrib4NsvARB, (), index: GLuint, v: *const GLshort);
     func!(VertexAttrib4Nub, (), index: GLuint, x: GLubyte, y: GLubyte, z: GLubyte, w: GLubyte);
     func!(VertexAttrib4NubARB, (), index: GLuint, x: GLubyte, y: GLubyte, z: GLubyte, w: GLubyte);
     func!(VertexAttrib4Nubv, (), index: GLuint, v: *const GLubyte);
     func!(VertexAttrib4NubvARB, (), index: GLuint, v: *const GLubyte);
     func!(VertexAttrib4Nuiv, (), index: GLuint, v: *const GLuint);
     func!(VertexAttrib4NuivARB, (), index: GLuint, v: *const GLuint);
     func!(VertexAttrib4Nusv, (), index: GLuint, v: *const GLushort);
     func!(VertexAttrib4NusvARB, (), index: GLuint, v: *const GLushort);
     func!(VertexAttrib4bv, (), index: GLuint, v: *const GLbyte);
     func!(VertexAttrib4bvARB, (), index: GLuint, v: *const GLbyte);
     func!(VertexAttrib4d, (), index: GLuint, x: GLdouble, y: GLdouble, z: GLdouble, w: GLdouble);
     func!(VertexAttrib4dARB, (), index: GLuint, x: GLdouble, y: GLdouble, z: GLdouble, w: GLdouble);
     func!(VertexAttrib4dNV, (), index: GLuint, x: GLdouble, y: GLdouble, z: GLdouble, w: GLdouble);
     func!(VertexAttrib4dv, (), index: GLuint, v: *const GLdouble);
     func!(VertexAttrib4dvARB, (), index: GLuint, v: *const GLdouble);
     func!(VertexAttrib4dvNV, (), index: GLuint, v: *const GLdouble);
     func!(VertexAttrib4f, (), index: GLuint, x: GLfloat, y: GLfloat, z: GLfloat, w: GLfloat);
     func!(VertexAttrib4fARB, (), index: GLuint, x: GLfloat, y: GLfloat, z: GLfloat, w: GLfloat);
     func!(VertexAttrib4fNV, (), index: GLuint, x: GLfloat, y: GLfloat, z: GLfloat, w: GLfloat);
     func!(VertexAttrib4fv, (), index: GLuint, v: *const GLfloat);
     func!(VertexAttrib4fvARB, (), index: GLuint, v: *const GLfloat);
     func!(VertexAttrib4fvNV, (), index: GLuint, v: *const GLfloat);
     func!(VertexAttrib4hNV, (), index: GLuint, x: GLhalfNV, y: GLhalfNV, z: GLhalfNV, w: GLhalfNV);
     func!(VertexAttrib4hvNV, (), index: GLuint, v: *const GLhalfNV);
     func!(VertexAttrib4iv, (), index: GLuint, v: *const GLint);
     func!(VertexAttrib4ivARB, (), index: GLuint, v: *const GLint);
     func!(VertexAttrib4s, (), index: GLuint, x: GLshort, y: GLshort, z: GLshort, w: GLshort);
     func!(VertexAttrib4sARB, (), index: GLuint, x: GLshort, y: GLshort, z: GLshort, w: GLshort);
     func!(VertexAttrib4sNV, (), index: GLuint, x: GLshort, y: GLshort, z: GLshort, w: GLshort);
     func!(VertexAttrib4sv, (), index: GLuint, v: *const GLshort);
     func!(VertexAttrib4svARB, (), index: GLuint, v: *const GLshort);
     func!(VertexAttrib4svNV, (), index: GLuint, v: *const GLshort);
     func!(VertexAttrib4ubNV, (), index: GLuint, x: GLubyte, y: GLubyte, z: GLubyte, w: GLubyte);
     func!(VertexAttrib4ubv, (), index: GLuint, v: *const GLubyte);
     func!(VertexAttrib4ubvARB, (), index: GLuint, v: *const GLubyte);
     func!(VertexAttrib4ubvNV, (), index: GLuint, v: *const GLubyte);
     func!(VertexAttrib4uiv, (), index: GLuint, v: *const GLuint);
     func!(VertexAttrib4uivARB, (), index: GLuint, v: *const GLuint);
     func!(VertexAttrib4usv, (), index: GLuint, v: *const GLushort);
     func!(VertexAttrib4usvARB, (), index: GLuint, v: *const GLushort);
     func!(VertexAttribArrayObjectATI, (), index: GLuint, size: GLint, type_: GLenum, normalized: GLboolean, stride: GLsizei, buffer: GLuint, offset: GLuint);
     func!(VertexAttribBinding, (), attribindex: GLuint, bindingindex: GLuint);
     func!(VertexAttribDivisor, (), index: GLuint, divisor: GLuint);
     func!(VertexAttribDivisorARB, (), index: GLuint, divisor: GLuint);
     func!(VertexAttribFormat, (), attribindex: GLuint, size: GLint, type_: GLenum, normalized: GLboolean, relativeoffset: GLuint);
     func!(VertexAttribFormatNV, (), index: GLuint, size: GLint, type_: GLenum, normalized: GLboolean, stride: GLsizei);
     func!(VertexAttribI1i, (), index: GLuint, x: GLint);
     func!(VertexAttribI1iEXT, (), index: GLuint, x: GLint);
     func!(VertexAttribI1iv, (), index: GLuint, v: *const GLint);
     func!(VertexAttribI1ivEXT, (), index: GLuint, v: *const GLint);
     func!(VertexAttribI1ui, (), index: GLuint, x: GLuint);
     func!(VertexAttribI1uiEXT, (), index: GLuint, x: GLuint);
     func!(VertexAttribI1uiv, (), index: GLuint, v: *const GLuint);
     func!(VertexAttribI1uivEXT, (), index: GLuint, v: *const GLuint);
     func!(VertexAttribI2i, (), index: GLuint, x: GLint, y: GLint);
     func!(VertexAttribI2iEXT, (), index: GLuint, x: GLint, y: GLint);
     func!(VertexAttribI2iv, (), index: GLuint, v: *const GLint);
     func!(VertexAttribI2ivEXT, (), index: GLuint, v: *const GLint);
     func!(VertexAttribI2ui, (), index: GLuint, x: GLuint, y: GLuint);
     func!(VertexAttribI2uiEXT, (), index: GLuint, x: GLuint, y: GLuint);
     func!(VertexAttribI2uiv, (), index: GLuint, v: *const GLuint);
     func!(VertexAttribI2uivEXT, (), index: GLuint, v: *const GLuint);
     func!(VertexAttribI3i, (), index: GLuint, x: GLint, y: GLint, z: GLint);
     func!(VertexAttribI3iEXT, (), index: GLuint, x: GLint, y: GLint, z: GLint);
     func!(VertexAttribI3iv, (), index: GLuint, v: *const GLint);
     func!(VertexAttribI3ivEXT, (), index: GLuint, v: *const GLint);
     func!(VertexAttribI3ui, (), index: GLuint, x: GLuint, y: GLuint, z: GLuint);
     func!(VertexAttribI3uiEXT, (), index: GLuint, x: GLuint, y: GLuint, z: GLuint);
     func!(VertexAttribI3uiv, (), index: GLuint, v: *const GLuint);
     func!(VertexAttribI3uivEXT, (), index: GLuint, v: *const GLuint);
     func!(VertexAttribI4bv, (), index: GLuint, v: *const GLbyte);
     func!(VertexAttribI4bvEXT, (), index: GLuint, v: *const GLbyte);
     func!(VertexAttribI4i, (), index: GLuint, x: GLint, y: GLint, z: GLint, w: GLint);
     func!(VertexAttribI4iEXT, (), index: GLuint, x: GLint, y: GLint, z: GLint, w: GLint);
     func!(VertexAttribI4iv, (), index: GLuint, v: *const GLint);
     func!(VertexAttribI4ivEXT, (), index: GLuint, v: *const GLint);
     func!(VertexAttribI4sv, (), index: GLuint, v: *const GLshort);
     func!(VertexAttribI4svEXT, (), index: GLuint, v: *const GLshort);
     func!(VertexAttribI4ubv, (), index: GLuint, v: *const GLubyte);
     func!(VertexAttribI4ubvEXT, (), index: GLuint, v: *const GLubyte);
     func!(VertexAttribI4ui, (), index: GLuint, x: GLuint, y: GLuint, z: GLuint, w: GLuint);
     func!(VertexAttribI4uiEXT, (), index: GLuint, x: GLuint, y: GLuint, z: GLuint, w: GLuint);
     func!(VertexAttribI4uiv, (), index: GLuint, v: *const GLuint);
     func!(VertexAttribI4uivEXT, (), index: GLuint, v: *const GLuint);
     func!(VertexAttribI4usv, (), index: GLuint, v: *const GLushort);
     func!(VertexAttribI4usvEXT, (), index: GLuint, v: *const GLushort);
     func!(VertexAttribIFormat, (), attribindex: GLuint, size: GLint, type_: GLenum, relativeoffset: GLuint);
     func!(VertexAttribIFormatNV, (), index: GLuint, size: GLint, type_: GLenum, stride: GLsizei);
     func!(VertexAttribIPointer, (), index: GLuint, size: GLint, type_: GLenum, stride: GLsizei, pointer: *const c_void);
     func!(VertexAttribIPointerEXT, (), index: GLuint, size: GLint, type_: GLenum, stride: GLsizei, pointer: *const c_void);
     func!(VertexAttribL1d, (), index: GLuint, x: GLdouble);
     func!(VertexAttribL1dEXT, (), index: GLuint, x: GLdouble);
     func!(VertexAttribL1dv, (), index: GLuint, v: *const GLdouble);
     func!(VertexAttribL1dvEXT, (), index: GLuint, v: *const GLdouble);
     func!(VertexAttribL1i64NV, (), index: GLuint, x: GLint64EXT);
     func!(VertexAttribL1i64vNV, (), index: GLuint, v: *const GLint64EXT);
     func!(VertexAttribL1ui64ARB, (), index: GLuint, x: GLuint64EXT);
     func!(VertexAttribL1ui64NV, (), index: GLuint, x: GLuint64EXT);
     func!(VertexAttribL1ui64vARB, (), index: GLuint, v: *const GLuint64EXT);
     func!(VertexAttribL1ui64vNV, (), index: GLuint, v: *const GLuint64EXT);
     func!(VertexAttribL2d, (), index: GLuint, x: GLdouble, y: GLdouble);
     func!(VertexAttribL2dEXT, (), index: GLuint, x: GLdouble, y: GLdouble);
     func!(VertexAttribL2dv, (), index: GLuint, v: *const GLdouble);
     func!(VertexAttribL2dvEXT, (), index: GLuint, v: *const GLdouble);
     func!(VertexAttribL2i64NV, (), index: GLuint, x: GLint64EXT, y: GLint64EXT);
     func!(VertexAttribL2i64vNV, (), index: GLuint, v: *const GLint64EXT);
     func!(VertexAttribL2ui64NV, (), index: GLuint, x: GLuint64EXT, y: GLuint64EXT);
     func!(VertexAttribL2ui64vNV, (), index: GLuint, v: *const GLuint64EXT);
     func!(VertexAttribL3d, (), index: GLuint, x: GLdouble, y: GLdouble, z: GLdouble);
     func!(VertexAttribL3dEXT, (), index: GLuint, x: GLdouble, y: GLdouble, z: GLdouble);
     func!(VertexAttribL3dv, (), index: GLuint, v: *const GLdouble);
     func!(VertexAttribL3dvEXT, (), index: GLuint, v: *const GLdouble);
     func!(VertexAttribL3i64NV, (), index: GLuint, x: GLint64EXT, y: GLint64EXT, z: GLint64EXT);
     func!(VertexAttribL3i64vNV, (), index: GLuint, v: *const GLint64EXT);
     func!(VertexAttribL3ui64NV, (), index: GLuint, x: GLuint64EXT, y: GLuint64EXT, z: GLuint64EXT);
     func!(VertexAttribL3ui64vNV, (), index: GLuint, v: *const GLuint64EXT);
     func!(VertexAttribL4d, (), index: GLuint, x: GLdouble, y: GLdouble, z: GLdouble, w: GLdouble);
     func!(VertexAttribL4dEXT, (), index: GLuint, x: GLdouble, y: GLdouble, z: GLdouble, w: GLdouble);
     func!(VertexAttribL4dv, (), index: GLuint, v: *const GLdouble);
     func!(VertexAttribL4dvEXT, (), index: GLuint, v: *const GLdouble);
     func!(VertexAttribL4i64NV, (), index: GLuint, x: GLint64EXT, y: GLint64EXT, z: GLint64EXT, w: GLint64EXT);
     func!(VertexAttribL4i64vNV, (), index: GLuint, v: *const GLint64EXT);
     func!(VertexAttribL4ui64NV, (), index: GLuint, x: GLuint64EXT, y: GLuint64EXT, z: GLuint64EXT, w: GLuint64EXT);
     func!(VertexAttribL4ui64vNV, (), index: GLuint, v: *const GLuint64EXT);
     func!(VertexAttribLFormat, (), attribindex: GLuint, size: GLint, type_: GLenum, relativeoffset: GLuint);
     func!(VertexAttribLFormatNV, (), index: GLuint, size: GLint, type_: GLenum, stride: GLsizei);
     func!(VertexAttribLPointer, (), index: GLuint, size: GLint, type_: GLenum, stride: GLsizei, pointer: *const c_void);
     func!(VertexAttribLPointerEXT, (), index: GLuint, size: GLint, type_: GLenum, stride: GLsizei, pointer: *const c_void);
     func!(VertexAttribP1ui, (), index: GLuint, type_: GLenum, normalized: GLboolean, value: GLuint);
     func!(VertexAttribP1uiv, (), index: GLuint, type_: GLenum, normalized: GLboolean, value: *const GLuint);
     func!(VertexAttribP2ui, (), index: GLuint, type_: GLenum, normalized: GLboolean, value: GLuint);
     func!(VertexAttribP2uiv, (), index: GLuint, type_: GLenum, normalized: GLboolean, value: *const GLuint);
     func!(VertexAttribP3ui, (), index: GLuint, type_: GLenum, normalized: GLboolean, value: GLuint);
     func!(VertexAttribP3uiv, (), index: GLuint, type_: GLenum, normalized: GLboolean, value: *const GLuint);
     func!(VertexAttribP4ui, (), index: GLuint, type_: GLenum, normalized: GLboolean, value: GLuint);
     func!(VertexAttribP4uiv, (), index: GLuint, type_: GLenum, normalized: GLboolean, value: *const GLuint);
     func!(VertexAttribParameteriAMD, (), index: GLuint, pname: GLenum, param: GLint);
     func!(VertexAttribPointer, (), index: GLuint, size: GLint, type_: GLenum, normalized: GLboolean, stride: GLsizei, pointer: *const c_void);
     func!(VertexAttribPointerARB, (), index: GLuint, size: GLint, type_: GLenum, normalized: GLboolean, stride: GLsizei, pointer: *const c_void);
     func!(VertexAttribPointerNV, (), index: GLuint, fsize: GLint, type_: GLenum, stride: GLsizei, pointer: *const c_void);
     func!(VertexAttribs1dvNV, (), index: GLuint, count: GLsizei, v: *const GLdouble);
     func!(VertexAttribs1fvNV, (), index: GLuint, count: GLsizei, v: *const GLfloat);
     func!(VertexAttribs1hvNV, (), index: GLuint, n: GLsizei, v: *const GLhalfNV);
     func!(VertexAttribs1svNV, (), index: GLuint, count: GLsizei, v: *const GLshort);
     func!(VertexAttribs2dvNV, (), index: GLuint, count: GLsizei, v: *const GLdouble);
     func!(VertexAttribs2fvNV, (), index: GLuint, count: GLsizei, v: *const GLfloat);
     func!(VertexAttribs2hvNV, (), index: GLuint, n: GLsizei, v: *const GLhalfNV);
     func!(VertexAttribs2svNV, (), index: GLuint, count: GLsizei, v: *const GLshort);
     func!(VertexAttribs3dvNV, (), index: GLuint, count: GLsizei, v: *const GLdouble);
     func!(VertexAttribs3fvNV, (), index: GLuint, count: GLsizei, v: *const GLfloat);
     func!(VertexAttribs3hvNV, (), index: GLuint, n: GLsizei, v: *const GLhalfNV);
     func!(VertexAttribs3svNV, (), index: GLuint, count: GLsizei, v: *const GLshort);
     func!(VertexAttribs4dvNV, (), index: GLuint, count: GLsizei, v: *const GLdouble);
     func!(VertexAttribs4fvNV, (), index: GLuint, count: GLsizei, v: *const GLfloat);
     func!(VertexAttribs4hvNV, (), index: GLuint, n: GLsizei, v: *const GLhalfNV);
     func!(VertexAttribs4svNV, (), index: GLuint, count: GLsizei, v: *const GLshort);
     func!(VertexAttribs4ubvNV, (), index: GLuint, count: GLsizei, v: *const GLubyte);
     func!(VertexBindingDivisor, (), bindingindex: GLuint, divisor: GLuint);
     func!(VertexBlendARB, (), count: GLint);
     func!(VertexBlendEnvfATI, (), pname: GLenum, param: GLfloat);
     func!(VertexBlendEnviATI, (), pname: GLenum, param: GLint);
     func!(VertexFormatNV, (), size: GLint, type_: GLenum, stride: GLsizei);
     func!(VertexPointerEXT, (), size: GLint, type_: GLenum, stride: GLsizei, count: GLsizei, pointer: *const c_void);
     func!(VertexPointerListIBM, (), size: GLint, type_: GLenum, stride: GLint, pointer: *const *const c_void, ptrstride: GLint);
     func!(VertexPointervINTEL, (), size: GLint, type_: GLenum, pointer: *const *const c_void);
     func!(VertexStream1dATI, (), stream: GLenum, x: GLdouble);
     func!(VertexStream1dvATI, (), stream: GLenum, coords: *const GLdouble);
     func!(VertexStream1fATI, (), stream: GLenum, x: GLfloat);
     func!(VertexStream1fvATI, (), stream: GLenum, coords: *const GLfloat);
     func!(VertexStream1iATI, (), stream: GLenum, x: GLint);
     func!(VertexStream1ivATI, (), stream: GLenum, coords: *const GLint);
     func!(VertexStream1sATI, (), stream: GLenum, x: GLshort);
     func!(VertexStream1svATI, (), stream: GLenum, coords: *const GLshort);
     func!(VertexStream2dATI, (), stream: GLenum, x: GLdouble, y: GLdouble);
     func!(VertexStream2dvATI, (), stream: GLenum, coords: *const GLdouble);
     func!(VertexStream2fATI, (), stream: GLenum, x: GLfloat, y: GLfloat);
     func!(VertexStream2fvATI, (), stream: GLenum, coords: *const GLfloat);
     func!(VertexStream2iATI, (), stream: GLenum, x: GLint, y: GLint);
     func!(VertexStream2ivATI, (), stream: GLenum, coords: *const GLint);
     func!(VertexStream2sATI, (), stream: GLenum, x: GLshort, y: GLshort);
     func!(VertexStream2svATI, (), stream: GLenum, coords: *const GLshort);
     func!(VertexStream3dATI, (), stream: GLenum, x: GLdouble, y: GLdouble, z: GLdouble);
     func!(VertexStream3dvATI, (), stream: GLenum, coords: *const GLdouble);
     func!(VertexStream3fATI, (), stream: GLenum, x: GLfloat, y: GLfloat, z: GLfloat);
     func!(VertexStream3fvATI, (), stream: GLenum, coords: *const GLfloat);
     func!(VertexStream3iATI, (), stream: GLenum, x: GLint, y: GLint, z: GLint);
     func!(VertexStream3ivATI, (), stream: GLenum, coords: *const GLint);
     func!(VertexStream3sATI, (), stream: GLenum, x: GLshort, y: GLshort, z: GLshort);
     func!(VertexStream3svATI, (), stream: GLenum, coords: *const GLshort);
     func!(VertexStream4dATI, (), stream: GLenum, x: GLdouble, y: GLdouble, z: GLdouble, w: GLdouble);
     func!(VertexStream4dvATI, (), stream: GLenum, coords: *const GLdouble);
     func!(VertexStream4fATI, (), stream: GLenum, x: GLfloat, y: GLfloat, z: GLfloat, w: GLfloat);
     func!(VertexStream4fvATI, (), stream: GLenum, coords: *const GLfloat);
     func!(VertexStream4iATI, (), stream: GLenum, x: GLint, y: GLint, z: GLint, w: GLint);
     func!(VertexStream4ivATI, (), stream: GLenum, coords: *const GLint);
     func!(VertexStream4sATI, (), stream: GLenum, x: GLshort, y: GLshort, z: GLshort, w: GLshort);
     func!(VertexStream4svATI, (), stream: GLenum, coords: *const GLshort);
     func!(VertexWeightPointerEXT, (), size: GLint, type_: GLenum, stride: GLsizei, pointer: *const c_void);
     func!(VertexWeightfEXT, (), weight: GLfloat);
     func!(VertexWeightfvEXT, (), weight: *const GLfloat);
     func!(VertexWeighthNV, (), weight: GLhalfNV);
     func!(VertexWeighthvNV, (), weight: *const GLhalfNV);
     func!(VideoCaptureNV, GLenum, video_capture_slot: GLuint, sequence_num: *mut GLuint, capture_time: *mut GLuint64EXT);
     func!(VideoCaptureStreamParameterdvNV, (), video_capture_slot: GLuint, stream: GLuint, pname: GLenum, params: *const GLdouble);
     func!(VideoCaptureStreamParameterfvNV, (), video_capture_slot: GLuint, stream: GLuint, pname: GLenum, params: *const GLfloat);
     func!(VideoCaptureStreamParameterivNV, (), video_capture_slot: GLuint, stream: GLuint, pname: GLenum, params: *const GLint);
     func!(Viewport, (), x: GLint, y: GLint, width: GLsizei, height: GLsizei);
     func!(ViewportArrayv, (), first: GLuint, count: GLsizei, v: *const GLfloat);
     func!(ViewportIndexedf, (), index: GLuint, x: GLfloat, y: GLfloat, w: GLfloat, h: GLfloat);
     func!(ViewportIndexedfv, (), index: GLuint, v: *const GLfloat);
     func!(ViewportPositionWScaleNV, (), index: GLuint, xcoeff: GLfloat, ycoeff: GLfloat);
     func!(ViewportSwizzleNV, (), index: GLuint, swizzlex: GLenum, swizzley: GLenum, swizzlez: GLenum, swizzlew: GLenum);
     func!(WaitSemaphoreEXT, (), semaphore: GLuint, numBufferBarriers: GLuint, buffers: *const GLuint, numTextureBarriers: GLuint, textures: *const GLuint, srcLayouts: *const GLenum);
     func!(WaitSemaphoreui64NVX, (), waitGpu: GLuint, fenceObjectCount: GLsizei, semaphoreArray: *const GLuint, fenceValueArray: *const GLuint64);
     func!(WaitSync, (), sync: GLsync, flags: GLbitfield, timeout: GLuint64);
     func!(WaitVkSemaphoreNV, (), vkSemaphore: GLuint64);
     func!(WeightPathsNV, (), resultPath: GLuint, numPaths: GLsizei, paths: *const GLuint, weights: *const GLfloat);
     func!(WeightPointerARB, (), size: GLint, type_: GLenum, stride: GLsizei, pointer: *const c_void);
     func!(WeightbvARB, (), size: GLint, weights: *const GLbyte);
     func!(WeightdvARB, (), size: GLint, weights: *const GLdouble);
     func!(WeightfvARB, (), size: GLint, weights: *const GLfloat);
     func!(WeightivARB, (), size: GLint, weights: *const GLint);
     func!(WeightsvARB, (), size: GLint, weights: *const GLshort);
     func!(WeightubvARB, (), size: GLint, weights: *const GLubyte);
     func!(WeightuivARB, (), size: GLint, weights: *const GLuint);
     func!(WeightusvARB, (), size: GLint, weights: *const GLushort);
     func!(WindowPos2dARB, (), x: GLdouble, y: GLdouble);
     func!(WindowPos2dMESA, (), x: GLdouble, y: GLdouble);
     func!(WindowPos2dvARB, (), v: *const GLdouble);
     func!(WindowPos2dvMESA, (), v: *const GLdouble);
     func!(WindowPos2fARB, (), x: GLfloat, y: GLfloat);
     func!(WindowPos2fMESA, (), x: GLfloat, y: GLfloat);
     func!(WindowPos2fvARB, (), v: *const GLfloat);
     func!(WindowPos2fvMESA, (), v: *const GLfloat);
     func!(WindowPos2iARB, (), x: GLint, y: GLint);
     func!(WindowPos2iMESA, (), x: GLint, y: GLint);
     func!(WindowPos2ivARB, (), v: *const GLint);
     func!(WindowPos2ivMESA, (), v: *const GLint);
     func!(WindowPos2sARB, (), x: GLshort, y: GLshort);
     func!(WindowPos2sMESA, (), x: GLshort, y: GLshort);
     func!(WindowPos2svARB, (), v: *const GLshort);
     func!(WindowPos2svMESA, (), v: *const GLshort);
     func!(WindowPos3dARB, (), x: GLdouble, y: GLdouble, z: GLdouble);
     func!(WindowPos3dMESA, (), x: GLdouble, y: GLdouble, z: GLdouble);
     func!(WindowPos3dvARB, (), v: *const GLdouble);
     func!(WindowPos3dvMESA, (), v: *const GLdouble);
     func!(WindowPos3fARB, (), x: GLfloat, y: GLfloat, z: GLfloat);
     func!(WindowPos3fMESA, (), x: GLfloat, y: GLfloat, z: GLfloat);
     func!(WindowPos3fvARB, (), v: *const GLfloat);
     func!(WindowPos3fvMESA, (), v: *const GLfloat);
     func!(WindowPos3iARB, (), x: GLint, y: GLint, z: GLint);
     func!(WindowPos3iMESA, (), x: GLint, y: GLint, z: GLint);
     func!(WindowPos3ivARB, (), v: *const GLint);
     func!(WindowPos3ivMESA, (), v: *const GLint);
     func!(WindowPos3sARB, (), x: GLshort, y: GLshort, z: GLshort);
     func!(WindowPos3sMESA, (), x: GLshort, y: GLshort, z: GLshort);
     func!(WindowPos3svARB, (), v: *const GLshort);
     func!(WindowPos3svMESA, (), v: *const GLshort);
     func!(WindowPos4dMESA, (), x: GLdouble, y: GLdouble, z: GLdouble, w: GLdouble);
     func!(WindowPos4dvMESA, (), v: *const GLdouble);
     func!(WindowPos4fMESA, (), x: GLfloat, y: GLfloat, z: GLfloat, w: GLfloat);
     func!(WindowPos4fvMESA, (), v: *const GLfloat);
     func!(WindowPos4iMESA, (), x: GLint, y: GLint, z: GLint, w: GLint);
     func!(WindowPos4ivMESA, (), v: *const GLint);
     func!(WindowPos4sMESA, (), x: GLshort, y: GLshort, z: GLshort, w: GLshort);
     func!(WindowPos4svMESA, (), v: *const GLshort);
     func!(WindowRectanglesEXT, (), mode: GLenum, count: GLsizei, box_: *const GLint);
     func!(WriteMaskEXT, (), res: GLuint, in_: GLuint, outX: GLenum, outY: GLenum, outZ: GLenum, outW: GLenum);


}

mod storage {
    #![allow(non_snake_case, non_upper_case_globals)]

    use super::FnPtr;
    use std::os::raw::*;

    macro_rules! store {
        ($name:ident) => {
            pub(super) static mut $name: FnPtr = FnPtr { ptr: FnPtr::not_initialized as *const c_void, is_loaded: false };
        }
    }

     store!(AccumxOES);
     store!(AcquireKeyedMutexWin32EXT);
     store!(ActiveProgramEXT);
     store!(ActiveShaderProgram);
     store!(ActiveStencilFaceEXT);
     store!(ActiveTexture);
     store!(ActiveTextureARB);
     store!(ActiveVaryingNV);
     store!(AlphaFragmentOp1ATI);
     store!(AlphaFragmentOp2ATI);
     store!(AlphaFragmentOp3ATI);
     store!(AlphaFuncxOES);
     store!(AlphaToCoverageDitherControlNV);
     store!(ApplyFramebufferAttachmentCMAAINTEL);
     store!(ApplyTextureEXT);
     store!(AreProgramsResidentNV);
     store!(AreTexturesResidentEXT);
     store!(ArrayElementEXT);
     store!(ArrayObjectATI);
     store!(AsyncCopyBufferSubDataNVX);
     store!(AsyncCopyImageSubDataNVX);
     store!(AsyncMarkerSGIX);
     store!(AttachObjectARB);
     store!(AttachShader);
     store!(BeginConditionalRender);
     store!(BeginConditionalRenderNV);
     store!(BeginConditionalRenderNVX);
     store!(BeginFragmentShaderATI);
     store!(BeginOcclusionQueryNV);
     store!(BeginPerfMonitorAMD);
     store!(BeginPerfQueryINTEL);
     store!(BeginQuery);
     store!(BeginQueryARB);
     store!(BeginQueryIndexed);
     store!(BeginTransformFeedback);
     store!(BeginTransformFeedbackEXT);
     store!(BeginTransformFeedbackNV);
     store!(BeginVertexShaderEXT);
     store!(BeginVideoCaptureNV);
     store!(BindAttribLocation);
     store!(BindAttribLocationARB);
     store!(BindBuffer);
     store!(BindBufferARB);
     store!(BindBufferBase);
     store!(BindBufferBaseEXT);
     store!(BindBufferBaseNV);
     store!(BindBufferOffsetEXT);
     store!(BindBufferOffsetNV);
     store!(BindBufferRange);
     store!(BindBufferRangeEXT);
     store!(BindBufferRangeNV);
     store!(BindBuffersBase);
     store!(BindBuffersRange);
     store!(BindFragDataLocation);
     store!(BindFragDataLocationEXT);
     store!(BindFragDataLocationIndexed);
     store!(BindFragmentShaderATI);
     store!(BindFramebuffer);
     store!(BindFramebufferEXT);
     store!(BindImageTexture);
     store!(BindImageTextureEXT);
     store!(BindImageTextures);
     store!(BindLightParameterEXT);
     store!(BindMaterialParameterEXT);
     store!(BindMultiTextureEXT);
     store!(BindParameterEXT);
     store!(BindProgramARB);
     store!(BindProgramNV);
     store!(BindProgramPipeline);
     store!(BindRenderbuffer);
     store!(BindRenderbufferEXT);
     store!(BindSampler);
     store!(BindSamplers);
     store!(BindShadingRateImageNV);
     store!(BindTexGenParameterEXT);
     store!(BindTexture);
     store!(BindTextureEXT);
     store!(BindTextureUnit);
     store!(BindTextureUnitParameterEXT);
     store!(BindTextures);
     store!(BindTransformFeedback);
     store!(BindTransformFeedbackNV);
     store!(BindVertexArray);
     store!(BindVertexArrayAPPLE);
     store!(BindVertexBuffer);
     store!(BindVertexBuffers);
     store!(BindVertexShaderEXT);
     store!(BindVideoCaptureStreamBufferNV);
     store!(BindVideoCaptureStreamTextureNV);
     store!(Binormal3bEXT);
     store!(Binormal3bvEXT);
     store!(Binormal3dEXT);
     store!(Binormal3dvEXT);
     store!(Binormal3fEXT);
     store!(Binormal3fvEXT);
     store!(Binormal3iEXT);
     store!(Binormal3ivEXT);
     store!(Binormal3sEXT);
     store!(Binormal3svEXT);
     store!(BinormalPointerEXT);
     store!(BitmapxOES);
     store!(BlendBarrierKHR);
     store!(BlendBarrierNV);
     store!(BlendColor);
     store!(BlendColorEXT);
     store!(BlendColorxOES);
     store!(BlendEquation);
     store!(BlendEquationEXT);
     store!(BlendEquationIndexedAMD);
     store!(BlendEquationSeparate);
     store!(BlendEquationSeparateEXT);
     store!(BlendEquationSeparateIndexedAMD);
     store!(BlendEquationSeparatei);
     store!(BlendEquationSeparateiARB);
     store!(BlendEquationi);
     store!(BlendEquationiARB);
     store!(BlendFunc);
     store!(BlendFuncIndexedAMD);
     store!(BlendFuncSeparate);
     store!(BlendFuncSeparateEXT);
     store!(BlendFuncSeparateINGR);
     store!(BlendFuncSeparateIndexedAMD);
     store!(BlendFuncSeparatei);
     store!(BlendFuncSeparateiARB);
     store!(BlendFunci);
     store!(BlendFunciARB);
     store!(BlendParameteriNV);
     store!(BlitFramebuffer);
     store!(BlitFramebufferEXT);
     store!(BlitNamedFramebuffer);
     store!(BufferAddressRangeNV);
     store!(BufferAttachMemoryNV);
     store!(BufferData);
     store!(BufferDataARB);
     store!(BufferPageCommitmentARB);
     store!(BufferPageCommitmentMemNV);
     store!(BufferParameteriAPPLE);
     store!(BufferStorage);
     store!(BufferStorageExternalEXT);
     store!(BufferStorageMemEXT);
     store!(BufferSubData);
     store!(BufferSubDataARB);
     store!(CallCommandListNV);
     store!(CheckFramebufferStatus);
     store!(CheckFramebufferStatusEXT);
     store!(CheckNamedFramebufferStatus);
     store!(CheckNamedFramebufferStatusEXT);
     store!(ClampColor);
     store!(ClampColorARB);
     store!(Clear);
     store!(ClearAccumxOES);
     store!(ClearBufferData);
     store!(ClearBufferSubData);
     store!(ClearBufferfi);
     store!(ClearBufferfv);
     store!(ClearBufferiv);
     store!(ClearBufferuiv);
     store!(ClearColor);
     store!(ClearColorIiEXT);
     store!(ClearColorIuiEXT);
     store!(ClearColorxOES);
     store!(ClearDepth);
     store!(ClearDepthdNV);
     store!(ClearDepthf);
     store!(ClearDepthfOES);
     store!(ClearDepthxOES);
     store!(ClearNamedBufferData);
     store!(ClearNamedBufferDataEXT);
     store!(ClearNamedBufferSubData);
     store!(ClearNamedBufferSubDataEXT);
     store!(ClearNamedFramebufferfi);
     store!(ClearNamedFramebufferfv);
     store!(ClearNamedFramebufferiv);
     store!(ClearNamedFramebufferuiv);
     store!(ClearStencil);
     store!(ClearTexImage);
     store!(ClearTexSubImage);
     store!(ClientActiveTextureARB);
     store!(ClientActiveVertexStreamATI);
     store!(ClientAttribDefaultEXT);
     store!(ClientWaitSemaphoreui64NVX);
     store!(ClientWaitSync);
     store!(ClipControl);
     store!(ClipPlanefOES);
     store!(ClipPlanexOES);
     store!(Color3fVertex3fSUN);
     store!(Color3fVertex3fvSUN);
     store!(Color3hNV);
     store!(Color3hvNV);
     store!(Color3xOES);
     store!(Color3xvOES);
     store!(Color4fNormal3fVertex3fSUN);
     store!(Color4fNormal3fVertex3fvSUN);
     store!(Color4hNV);
     store!(Color4hvNV);
     store!(Color4ubVertex2fSUN);
     store!(Color4ubVertex2fvSUN);
     store!(Color4ubVertex3fSUN);
     store!(Color4ubVertex3fvSUN);
     store!(Color4xOES);
     store!(Color4xvOES);
     store!(ColorFormatNV);
     store!(ColorFragmentOp1ATI);
     store!(ColorFragmentOp2ATI);
     store!(ColorFragmentOp3ATI);
     store!(ColorMask);
     store!(ColorMaskIndexedEXT);
     store!(ColorMaski);
     store!(ColorPointerEXT);
     store!(ColorPointerListIBM);
     store!(ColorPointervINTEL);
     store!(ColorSubTableEXT);
     store!(ColorTableEXT);
     store!(ColorTableParameterfvSGI);
     store!(ColorTableParameterivSGI);
     store!(ColorTableSGI);
     store!(CombinerInputNV);
     store!(CombinerOutputNV);
     store!(CombinerParameterfNV);
     store!(CombinerParameterfvNV);
     store!(CombinerParameteriNV);
     store!(CombinerParameterivNV);
     store!(CombinerStageParameterfvNV);
     store!(CommandListSegmentsNV);
     store!(CompileCommandListNV);
     store!(CompileShader);
     store!(CompileShaderARB);
     store!(CompileShaderIncludeARB);
     store!(CompressedMultiTexImage1DEXT);
     store!(CompressedMultiTexImage2DEXT);
     store!(CompressedMultiTexImage3DEXT);
     store!(CompressedMultiTexSubImage1DEXT);
     store!(CompressedMultiTexSubImage2DEXT);
     store!(CompressedMultiTexSubImage3DEXT);
     store!(CompressedTexImage1D);
     store!(CompressedTexImage1DARB);
     store!(CompressedTexImage2D);
     store!(CompressedTexImage2DARB);
     store!(CompressedTexImage3D);
     store!(CompressedTexImage3DARB);
     store!(CompressedTexSubImage1D);
     store!(CompressedTexSubImage1DARB);
     store!(CompressedTexSubImage2D);
     store!(CompressedTexSubImage2DARB);
     store!(CompressedTexSubImage3D);
     store!(CompressedTexSubImage3DARB);
     store!(CompressedTextureImage1DEXT);
     store!(CompressedTextureImage2DEXT);
     store!(CompressedTextureImage3DEXT);
     store!(CompressedTextureSubImage1D);
     store!(CompressedTextureSubImage1DEXT);
     store!(CompressedTextureSubImage2D);
     store!(CompressedTextureSubImage2DEXT);
     store!(CompressedTextureSubImage3D);
     store!(CompressedTextureSubImage3DEXT);
     store!(ConservativeRasterParameterfNV);
     store!(ConservativeRasterParameteriNV);
     store!(ConvolutionFilter1DEXT);
     store!(ConvolutionFilter2DEXT);
     store!(ConvolutionParameterfEXT);
     store!(ConvolutionParameterfvEXT);
     store!(ConvolutionParameteriEXT);
     store!(ConvolutionParameterivEXT);
     store!(ConvolutionParameterxOES);
     store!(ConvolutionParameterxvOES);
     store!(CopyBufferSubData);
     store!(CopyColorSubTableEXT);
     store!(CopyColorTableSGI);
     store!(CopyConvolutionFilter1DEXT);
     store!(CopyConvolutionFilter2DEXT);
     store!(CopyImageSubData);
     store!(CopyImageSubDataNV);
     store!(CopyMultiTexImage1DEXT);
     store!(CopyMultiTexImage2DEXT);
     store!(CopyMultiTexSubImage1DEXT);
     store!(CopyMultiTexSubImage2DEXT);
     store!(CopyMultiTexSubImage3DEXT);
     store!(CopyNamedBufferSubData);
     store!(CopyPathNV);
     store!(CopyTexImage1D);
     store!(CopyTexImage1DEXT);
     store!(CopyTexImage2D);
     store!(CopyTexImage2DEXT);
     store!(CopyTexSubImage1D);
     store!(CopyTexSubImage1DEXT);
     store!(CopyTexSubImage2D);
     store!(CopyTexSubImage2DEXT);
     store!(CopyTexSubImage3D);
     store!(CopyTexSubImage3DEXT);
     store!(CopyTextureImage1DEXT);
     store!(CopyTextureImage2DEXT);
     store!(CopyTextureSubImage1D);
     store!(CopyTextureSubImage1DEXT);
     store!(CopyTextureSubImage2D);
     store!(CopyTextureSubImage2DEXT);
     store!(CopyTextureSubImage3D);
     store!(CopyTextureSubImage3DEXT);
     store!(CoverFillPathInstancedNV);
     store!(CoverFillPathNV);
     store!(CoverStrokePathInstancedNV);
     store!(CoverStrokePathNV);
     store!(CoverageModulationNV);
     store!(CoverageModulationTableNV);
     store!(CreateBuffers);
     store!(CreateCommandListsNV);
     store!(CreateFramebuffers);
     store!(CreateMemoryObjectsEXT);
     store!(CreatePerfQueryINTEL);
     store!(CreateProgram);
     store!(CreateProgramObjectARB);
     store!(CreateProgramPipelines);
     store!(CreateProgressFenceNVX);
     store!(CreateQueries);
     store!(CreateRenderbuffers);
     store!(CreateSamplers);
     store!(CreateSemaphoresNV);
     store!(CreateShader);
     store!(CreateShaderObjectARB);
     store!(CreateShaderProgramEXT);
     store!(CreateShaderProgramv);
     store!(CreateStatesNV);
     store!(CreateSyncFromCLeventARB);
     store!(CreateTextures);
     store!(CreateTransformFeedbacks);
     store!(CreateVertexArrays);
     store!(CullFace);
     store!(CullParameterdvEXT);
     store!(CullParameterfvEXT);
     store!(CurrentPaletteMatrixARB);
     store!(DebugMessageCallback);
     store!(DebugMessageCallbackAMD);
     store!(DebugMessageCallbackARB);
     store!(DebugMessageControl);
     store!(DebugMessageControlARB);
     store!(DebugMessageEnableAMD);
     store!(DebugMessageInsert);
     store!(DebugMessageInsertAMD);
     store!(DebugMessageInsertARB);
     store!(DeformSGIX);
     store!(DeformationMap3dSGIX);
     store!(DeformationMap3fSGIX);
     store!(DeleteAsyncMarkersSGIX);
     store!(DeleteBuffers);
     store!(DeleteBuffersARB);
     store!(DeleteCommandListsNV);
     store!(DeleteFencesAPPLE);
     store!(DeleteFencesNV);
     store!(DeleteFragmentShaderATI);
     store!(DeleteFramebuffers);
     store!(DeleteFramebuffersEXT);
     store!(DeleteMemoryObjectsEXT);
     store!(DeleteNamedStringARB);
     store!(DeleteNamesAMD);
     store!(DeleteObjectARB);
     store!(DeleteOcclusionQueriesNV);
     store!(DeletePathsNV);
     store!(DeletePerfMonitorsAMD);
     store!(DeletePerfQueryINTEL);
     store!(DeleteProgram);
     store!(DeleteProgramPipelines);
     store!(DeleteProgramsARB);
     store!(DeleteProgramsNV);
     store!(DeleteQueries);
     store!(DeleteQueriesARB);
     store!(DeleteQueryResourceTagNV);
     store!(DeleteRenderbuffers);
     store!(DeleteRenderbuffersEXT);
     store!(DeleteSamplers);
     store!(DeleteSemaphoresEXT);
     store!(DeleteShader);
     store!(DeleteStatesNV);
     store!(DeleteSync);
     store!(DeleteTextures);
     store!(DeleteTexturesEXT);
     store!(DeleteTransformFeedbacks);
     store!(DeleteTransformFeedbacksNV);
     store!(DeleteVertexArrays);
     store!(DeleteVertexArraysAPPLE);
     store!(DeleteVertexShaderEXT);
     store!(DepthBoundsEXT);
     store!(DepthBoundsdNV);
     store!(DepthFunc);
     store!(DepthMask);
     store!(DepthRange);
     store!(DepthRangeArraydvNV);
     store!(DepthRangeArrayv);
     store!(DepthRangeIndexed);
     store!(DepthRangeIndexeddNV);
     store!(DepthRangedNV);
     store!(DepthRangef);
     store!(DepthRangefOES);
     store!(DepthRangexOES);
     store!(DetachObjectARB);
     store!(DetachShader);
     store!(DetailTexFuncSGIS);
     store!(Disable);
     store!(DisableClientStateIndexedEXT);
     store!(DisableClientStateiEXT);
     store!(DisableIndexedEXT);
     store!(DisableVariantClientStateEXT);
     store!(DisableVertexArrayAttrib);
     store!(DisableVertexArrayAttribEXT);
     store!(DisableVertexArrayEXT);
     store!(DisableVertexAttribAPPLE);
     store!(DisableVertexAttribArray);
     store!(DisableVertexAttribArrayARB);
     store!(Disablei);
     store!(DispatchCompute);
     store!(DispatchComputeGroupSizeARB);
     store!(DispatchComputeIndirect);
     store!(DrawArrays);
     store!(DrawArraysEXT);
     store!(DrawArraysIndirect);
     store!(DrawArraysInstanced);
     store!(DrawArraysInstancedARB);
     store!(DrawArraysInstancedBaseInstance);
     store!(DrawArraysInstancedEXT);
     store!(DrawBuffer);
     store!(DrawBuffers);
     store!(DrawBuffersARB);
     store!(DrawBuffersATI);
     store!(DrawCommandsAddressNV);
     store!(DrawCommandsNV);
     store!(DrawCommandsStatesAddressNV);
     store!(DrawCommandsStatesNV);
     store!(DrawElementArrayAPPLE);
     store!(DrawElementArrayATI);
     store!(DrawElements);
     store!(DrawElementsBaseVertex);
     store!(DrawElementsIndirect);
     store!(DrawElementsInstanced);
     store!(DrawElementsInstancedARB);
     store!(DrawElementsInstancedBaseInstance);
     store!(DrawElementsInstancedBaseVertex);
     store!(DrawElementsInstancedBaseVertexBaseInstance);
     store!(DrawElementsInstancedEXT);
     store!(DrawMeshArraysSUN);
     store!(DrawMeshTasksIndirectNV);
     store!(DrawMeshTasksNV);
     store!(DrawRangeElementArrayAPPLE);
     store!(DrawRangeElementArrayATI);
     store!(DrawRangeElements);
     store!(DrawRangeElementsBaseVertex);
     store!(DrawRangeElementsEXT);
     store!(DrawTextureNV);
     store!(DrawTransformFeedback);
     store!(DrawTransformFeedbackInstanced);
     store!(DrawTransformFeedbackNV);
     store!(DrawTransformFeedbackStream);
     store!(DrawTransformFeedbackStreamInstanced);
     store!(DrawVkImageNV);
     store!(EGLImageTargetTexStorageEXT);
     store!(EGLImageTargetTextureStorageEXT);
     store!(EdgeFlagFormatNV);
     store!(EdgeFlagPointerEXT);
     store!(EdgeFlagPointerListIBM);
     store!(ElementPointerAPPLE);
     store!(ElementPointerATI);
     store!(Enable);
     store!(EnableClientStateIndexedEXT);
     store!(EnableClientStateiEXT);
     store!(EnableIndexedEXT);
     store!(EnableVariantClientStateEXT);
     store!(EnableVertexArrayAttrib);
     store!(EnableVertexArrayAttribEXT);
     store!(EnableVertexArrayEXT);
     store!(EnableVertexAttribAPPLE);
     store!(EnableVertexAttribArray);
     store!(EnableVertexAttribArrayARB);
     store!(Enablei);
     store!(EndConditionalRender);
     store!(EndConditionalRenderNV);
     store!(EndConditionalRenderNVX);
     store!(EndFragmentShaderATI);
     store!(EndOcclusionQueryNV);
     store!(EndPerfMonitorAMD);
     store!(EndPerfQueryINTEL);
     store!(EndQuery);
     store!(EndQueryARB);
     store!(EndQueryIndexed);
     store!(EndTransformFeedback);
     store!(EndTransformFeedbackEXT);
     store!(EndTransformFeedbackNV);
     store!(EndVertexShaderEXT);
     store!(EndVideoCaptureNV);
     store!(EvalCoord1xOES);
     store!(EvalCoord1xvOES);
     store!(EvalCoord2xOES);
     store!(EvalCoord2xvOES);
     store!(EvalMapsNV);
     store!(EvaluateDepthValuesARB);
     store!(ExecuteProgramNV);
     store!(ExtractComponentEXT);
     store!(FeedbackBufferxOES);
     store!(FenceSync);
     store!(FinalCombinerInputNV);
     store!(Finish);
     store!(FinishAsyncSGIX);
     store!(FinishFenceAPPLE);
     store!(FinishFenceNV);
     store!(FinishObjectAPPLE);
     store!(FinishTextureSUNX);
     store!(Flush);
     store!(FlushMappedBufferRange);
     store!(FlushMappedBufferRangeAPPLE);
     store!(FlushMappedNamedBufferRange);
     store!(FlushMappedNamedBufferRangeEXT);
     store!(FlushPixelDataRangeNV);
     store!(FlushRasterSGIX);
     store!(FlushStaticDataIBM);
     store!(FlushVertexArrayRangeAPPLE);
     store!(FlushVertexArrayRangeNV);
     store!(FogCoordFormatNV);
     store!(FogCoordPointerEXT);
     store!(FogCoordPointerListIBM);
     store!(FogCoorddEXT);
     store!(FogCoorddvEXT);
     store!(FogCoordfEXT);
     store!(FogCoordfvEXT);
     store!(FogCoordhNV);
     store!(FogCoordhvNV);
     store!(FogFuncSGIS);
     store!(FogxOES);
     store!(FogxvOES);
     store!(FragmentColorMaterialSGIX);
     store!(FragmentCoverageColorNV);
     store!(FragmentLightModelfSGIX);
     store!(FragmentLightModelfvSGIX);
     store!(FragmentLightModeliSGIX);
     store!(FragmentLightModelivSGIX);
     store!(FragmentLightfSGIX);
     store!(FragmentLightfvSGIX);
     store!(FragmentLightiSGIX);
     store!(FragmentLightivSGIX);
     store!(FragmentMaterialfSGIX);
     store!(FragmentMaterialfvSGIX);
     store!(FragmentMaterialiSGIX);
     store!(FragmentMaterialivSGIX);
     store!(FrameTerminatorGREMEDY);
     store!(FrameZoomSGIX);
     store!(FramebufferDrawBufferEXT);
     store!(FramebufferDrawBuffersEXT);
     store!(FramebufferFetchBarrierEXT);
     store!(FramebufferParameteri);
     store!(FramebufferParameteriMESA);
     store!(FramebufferReadBufferEXT);
     store!(FramebufferRenderbuffer);
     store!(FramebufferRenderbufferEXT);
     store!(FramebufferSampleLocationsfvARB);
     store!(FramebufferSampleLocationsfvNV);
     store!(FramebufferSamplePositionsfvAMD);
     store!(FramebufferTexture);
     store!(FramebufferTexture1D);
     store!(FramebufferTexture1DEXT);
     store!(FramebufferTexture2D);
     store!(FramebufferTexture2DEXT);
     store!(FramebufferTexture3D);
     store!(FramebufferTexture3DEXT);
     store!(FramebufferTextureARB);
     store!(FramebufferTextureEXT);
     store!(FramebufferTextureFaceARB);
     store!(FramebufferTextureFaceEXT);
     store!(FramebufferTextureLayer);
     store!(FramebufferTextureLayerARB);
     store!(FramebufferTextureLayerEXT);
     store!(FramebufferTextureMultiviewOVR);
     store!(FreeObjectBufferATI);
     store!(FrontFace);
     store!(FrustumfOES);
     store!(FrustumxOES);
     store!(GenAsyncMarkersSGIX);
     store!(GenBuffers);
     store!(GenBuffersARB);
     store!(GenFencesAPPLE);
     store!(GenFencesNV);
     store!(GenFragmentShadersATI);
     store!(GenFramebuffers);
     store!(GenFramebuffersEXT);
     store!(GenNamesAMD);
     store!(GenOcclusionQueriesNV);
     store!(GenPathsNV);
     store!(GenPerfMonitorsAMD);
     store!(GenProgramPipelines);
     store!(GenProgramsARB);
     store!(GenProgramsNV);
     store!(GenQueries);
     store!(GenQueriesARB);
     store!(GenQueryResourceTagNV);
     store!(GenRenderbuffers);
     store!(GenRenderbuffersEXT);
     store!(GenSamplers);
     store!(GenSemaphoresEXT);
     store!(GenSymbolsEXT);
     store!(GenTextures);
     store!(GenTexturesEXT);
     store!(GenTransformFeedbacks);
     store!(GenTransformFeedbacksNV);
     store!(GenVertexArrays);
     store!(GenVertexArraysAPPLE);
     store!(GenVertexShadersEXT);
     store!(GenerateMipmap);
     store!(GenerateMipmapEXT);
     store!(GenerateMultiTexMipmapEXT);
     store!(GenerateTextureMipmap);
     store!(GenerateTextureMipmapEXT);
     store!(GetActiveAtomicCounterBufferiv);
     store!(GetActiveAttrib);
     store!(GetActiveAttribARB);
     store!(GetActiveSubroutineName);
     store!(GetActiveSubroutineUniformName);
     store!(GetActiveSubroutineUniformiv);
     store!(GetActiveUniform);
     store!(GetActiveUniformARB);
     store!(GetActiveUniformBlockName);
     store!(GetActiveUniformBlockiv);
     store!(GetActiveUniformName);
     store!(GetActiveUniformsiv);
     store!(GetActiveVaryingNV);
     store!(GetArrayObjectfvATI);
     store!(GetArrayObjectivATI);
     store!(GetAttachedObjectsARB);
     store!(GetAttachedShaders);
     store!(GetAttribLocation);
     store!(GetAttribLocationARB);
     store!(GetBooleanIndexedvEXT);
     store!(GetBooleani_v);
     store!(GetBooleanv);
     store!(GetBufferParameteri64v);
     store!(GetBufferParameteriv);
     store!(GetBufferParameterivARB);
     store!(GetBufferParameterui64vNV);
     store!(GetBufferPointerv);
     store!(GetBufferPointervARB);
     store!(GetBufferSubData);
     store!(GetBufferSubDataARB);
     store!(GetClipPlanefOES);
     store!(GetClipPlanexOES);
     store!(GetColorTableEXT);
     store!(GetColorTableParameterfvEXT);
     store!(GetColorTableParameterfvSGI);
     store!(GetColorTableParameterivEXT);
     store!(GetColorTableParameterivSGI);
     store!(GetColorTableSGI);
     store!(GetCombinerInputParameterfvNV);
     store!(GetCombinerInputParameterivNV);
     store!(GetCombinerOutputParameterfvNV);
     store!(GetCombinerOutputParameterivNV);
     store!(GetCombinerStageParameterfvNV);
     store!(GetCommandHeaderNV);
     store!(GetCompressedMultiTexImageEXT);
     store!(GetCompressedTexImage);
     store!(GetCompressedTexImageARB);
     store!(GetCompressedTextureImage);
     store!(GetCompressedTextureImageEXT);
     store!(GetCompressedTextureSubImage);
     store!(GetConvolutionFilterEXT);
     store!(GetConvolutionParameterfvEXT);
     store!(GetConvolutionParameterivEXT);
     store!(GetConvolutionParameterxvOES);
     store!(GetCoverageModulationTableNV);
     store!(GetDebugMessageLog);
     store!(GetDebugMessageLogAMD);
     store!(GetDebugMessageLogARB);
     store!(GetDetailTexFuncSGIS);
     store!(GetDoubleIndexedvEXT);
     store!(GetDoublei_v);
     store!(GetDoublei_vEXT);
     store!(GetDoublev);
     store!(GetError);
     store!(GetFenceivNV);
     store!(GetFinalCombinerInputParameterfvNV);
     store!(GetFinalCombinerInputParameterivNV);
     store!(GetFirstPerfQueryIdINTEL);
     store!(GetFixedvOES);
     store!(GetFloatIndexedvEXT);
     store!(GetFloati_v);
     store!(GetFloati_vEXT);
     store!(GetFloatv);
     store!(GetFogFuncSGIS);
     store!(GetFragDataIndex);
     store!(GetFragDataLocation);
     store!(GetFragDataLocationEXT);
     store!(GetFragmentLightfvSGIX);
     store!(GetFragmentLightivSGIX);
     store!(GetFragmentMaterialfvSGIX);
     store!(GetFragmentMaterialivSGIX);
     store!(GetFramebufferAttachmentParameteriv);
     store!(GetFramebufferAttachmentParameterivEXT);
     store!(GetFramebufferParameterfvAMD);
     store!(GetFramebufferParameteriv);
     store!(GetFramebufferParameterivEXT);
     store!(GetFramebufferParameterivMESA);
     store!(GetGraphicsResetStatus);
     store!(GetGraphicsResetStatusARB);
     store!(GetHandleARB);
     store!(GetHistogramEXT);
     store!(GetHistogramParameterfvEXT);
     store!(GetHistogramParameterivEXT);
     store!(GetHistogramParameterxvOES);
     store!(GetImageHandleARB);
     store!(GetImageHandleNV);
     store!(GetImageTransformParameterfvHP);
     store!(GetImageTransformParameterivHP);
     store!(GetInfoLogARB);
     store!(GetInstrumentsSGIX);
     store!(GetInteger64i_v);
     store!(GetInteger64v);
     store!(GetIntegerIndexedvEXT);
     store!(GetIntegeri_v);
     store!(GetIntegerui64i_vNV);
     store!(GetIntegerui64vNV);
     store!(GetIntegerv);
     store!(GetInternalformatSampleivNV);
     store!(GetInternalformati64v);
     store!(GetInternalformativ);
     store!(GetInvariantBooleanvEXT);
     store!(GetInvariantFloatvEXT);
     store!(GetInvariantIntegervEXT);
     store!(GetLightxOES);
     store!(GetListParameterfvSGIX);
     store!(GetListParameterivSGIX);
     store!(GetLocalConstantBooleanvEXT);
     store!(GetLocalConstantFloatvEXT);
     store!(GetLocalConstantIntegervEXT);
     store!(GetMapAttribParameterfvNV);
     store!(GetMapAttribParameterivNV);
     store!(GetMapControlPointsNV);
     store!(GetMapParameterfvNV);
     store!(GetMapParameterivNV);
     store!(GetMapxvOES);
     store!(GetMaterialxOES);
     store!(GetMemoryObjectDetachedResourcesuivNV);
     store!(GetMemoryObjectParameterivEXT);
     store!(GetMinmaxEXT);
     store!(GetMinmaxParameterfvEXT);
     store!(GetMinmaxParameterivEXT);
     store!(GetMultiTexEnvfvEXT);
     store!(GetMultiTexEnvivEXT);
     store!(GetMultiTexGendvEXT);
     store!(GetMultiTexGenfvEXT);
     store!(GetMultiTexGenivEXT);
     store!(GetMultiTexImageEXT);
     store!(GetMultiTexLevelParameterfvEXT);
     store!(GetMultiTexLevelParameterivEXT);
     store!(GetMultiTexParameterIivEXT);
     store!(GetMultiTexParameterIuivEXT);
     store!(GetMultiTexParameterfvEXT);
     store!(GetMultiTexParameterivEXT);
     store!(GetMultisamplefv);
     store!(GetMultisamplefvNV);
     store!(GetNamedBufferParameteri64v);
     store!(GetNamedBufferParameteriv);
     store!(GetNamedBufferParameterivEXT);
     store!(GetNamedBufferParameterui64vNV);
     store!(GetNamedBufferPointerv);
     store!(GetNamedBufferPointervEXT);
     store!(GetNamedBufferSubData);
     store!(GetNamedBufferSubDataEXT);
     store!(GetNamedFramebufferAttachmentParameteriv);
     store!(GetNamedFramebufferAttachmentParameterivEXT);
     store!(GetNamedFramebufferParameterfvAMD);
     store!(GetNamedFramebufferParameteriv);
     store!(GetNamedFramebufferParameterivEXT);
     store!(GetNamedProgramLocalParameterIivEXT);
     store!(GetNamedProgramLocalParameterIuivEXT);
     store!(GetNamedProgramLocalParameterdvEXT);
     store!(GetNamedProgramLocalParameterfvEXT);
     store!(GetNamedProgramStringEXT);
     store!(GetNamedProgramivEXT);
     store!(GetNamedRenderbufferParameteriv);
     store!(GetNamedRenderbufferParameterivEXT);
     store!(GetNamedStringARB);
     store!(GetNamedStringivARB);
     store!(GetNextPerfQueryIdINTEL);
     store!(GetObjectBufferfvATI);
     store!(GetObjectBufferivATI);
     store!(GetObjectLabel);
     store!(GetObjectLabelEXT);
     store!(GetObjectParameterfvARB);
     store!(GetObjectParameterivAPPLE);
     store!(GetObjectParameterivARB);
     store!(GetObjectPtrLabel);
     store!(GetOcclusionQueryivNV);
     store!(GetOcclusionQueryuivNV);
     store!(GetPathCommandsNV);
     store!(GetPathCoordsNV);
     store!(GetPathDashArrayNV);
     store!(GetPathLengthNV);
     store!(GetPathMetricRangeNV);
     store!(GetPathMetricsNV);
     store!(GetPathParameterfvNV);
     store!(GetPathParameterivNV);
     store!(GetPathSpacingNV);
     store!(GetPerfCounterInfoINTEL);
     store!(GetPerfMonitorCounterDataAMD);
     store!(GetPerfMonitorCounterInfoAMD);
     store!(GetPerfMonitorCounterStringAMD);
     store!(GetPerfMonitorCountersAMD);
     store!(GetPerfMonitorGroupStringAMD);
     store!(GetPerfMonitorGroupsAMD);
     store!(GetPerfQueryDataINTEL);
     store!(GetPerfQueryIdByNameINTEL);
     store!(GetPerfQueryInfoINTEL);
     store!(GetPixelMapxv);
     store!(GetPixelTexGenParameterfvSGIS);
     store!(GetPixelTexGenParameterivSGIS);
     store!(GetPixelTransformParameterfvEXT);
     store!(GetPixelTransformParameterivEXT);
     store!(GetPointerIndexedvEXT);
     store!(GetPointeri_vEXT);
     store!(GetPointerv);
     store!(GetPointervEXT);
     store!(GetProgramBinary);
     store!(GetProgramEnvParameterIivNV);
     store!(GetProgramEnvParameterIuivNV);
     store!(GetProgramEnvParameterdvARB);
     store!(GetProgramEnvParameterfvARB);
     store!(GetProgramInfoLog);
     store!(GetProgramInterfaceiv);
     store!(GetProgramLocalParameterIivNV);
     store!(GetProgramLocalParameterIuivNV);
     store!(GetProgramLocalParameterdvARB);
     store!(GetProgramLocalParameterfvARB);
     store!(GetProgramNamedParameterdvNV);
     store!(GetProgramNamedParameterfvNV);
     store!(GetProgramParameterdvNV);
     store!(GetProgramParameterfvNV);
     store!(GetProgramPipelineInfoLog);
     store!(GetProgramPipelineiv);
     store!(GetProgramResourceIndex);
     store!(GetProgramResourceLocation);
     store!(GetProgramResourceLocationIndex);
     store!(GetProgramResourceName);
     store!(GetProgramResourcefvNV);
     store!(GetProgramResourceiv);
     store!(GetProgramStageiv);
     store!(GetProgramStringARB);
     store!(GetProgramStringNV);
     store!(GetProgramSubroutineParameteruivNV);
     store!(GetProgramiv);
     store!(GetProgramivARB);
     store!(GetProgramivNV);
     store!(GetQueryBufferObjecti64v);
     store!(GetQueryBufferObjectiv);
     store!(GetQueryBufferObjectui64v);
     store!(GetQueryBufferObjectuiv);
     store!(GetQueryIndexediv);
     store!(GetQueryObjecti64v);
     store!(GetQueryObjecti64vEXT);
     store!(GetQueryObjectiv);
     store!(GetQueryObjectivARB);
     store!(GetQueryObjectui64v);
     store!(GetQueryObjectui64vEXT);
     store!(GetQueryObjectuiv);
     store!(GetQueryObjectuivARB);
     store!(GetQueryiv);
     store!(GetQueryivARB);
     store!(GetRenderbufferParameteriv);
     store!(GetRenderbufferParameterivEXT);
     store!(GetSamplerParameterIiv);
     store!(GetSamplerParameterIuiv);
     store!(GetSamplerParameterfv);
     store!(GetSamplerParameteriv);
     store!(GetSemaphoreParameterivNV);
     store!(GetSemaphoreParameterui64vEXT);
     store!(GetSeparableFilterEXT);
     store!(GetShaderInfoLog);
     store!(GetShaderPrecisionFormat);
     store!(GetShaderSource);
     store!(GetShaderSourceARB);
     store!(GetShaderiv);
     store!(GetShadingRateImagePaletteNV);
     store!(GetShadingRateSampleLocationivNV);
     store!(GetSharpenTexFuncSGIS);
     store!(GetStageIndexNV);
     store!(GetString);
     store!(GetStringi);
     store!(GetSubroutineIndex);
     store!(GetSubroutineUniformLocation);
     store!(GetSynciv);
     store!(GetTexBumpParameterfvATI);
     store!(GetTexBumpParameterivATI);
     store!(GetTexEnvxvOES);
     store!(GetTexFilterFuncSGIS);
     store!(GetTexGenxvOES);
     store!(GetTexImage);
     store!(GetTexLevelParameterfv);
     store!(GetTexLevelParameteriv);
     store!(GetTexLevelParameterxvOES);
     store!(GetTexParameterIiv);
     store!(GetTexParameterIivEXT);
     store!(GetTexParameterIuiv);
     store!(GetTexParameterIuivEXT);
     store!(GetTexParameterPointervAPPLE);
     store!(GetTexParameterfv);
     store!(GetTexParameteriv);
     store!(GetTexParameterxvOES);
     store!(GetTextureHandleARB);
     store!(GetTextureHandleNV);
     store!(GetTextureImage);
     store!(GetTextureImageEXT);
     store!(GetTextureLevelParameterfv);
     store!(GetTextureLevelParameterfvEXT);
     store!(GetTextureLevelParameteriv);
     store!(GetTextureLevelParameterivEXT);
     store!(GetTextureParameterIiv);
     store!(GetTextureParameterIivEXT);
     store!(GetTextureParameterIuiv);
     store!(GetTextureParameterIuivEXT);
     store!(GetTextureParameterfv);
     store!(GetTextureParameterfvEXT);
     store!(GetTextureParameteriv);
     store!(GetTextureParameterivEXT);
     store!(GetTextureSamplerHandleARB);
     store!(GetTextureSamplerHandleNV);
     store!(GetTextureSubImage);
     store!(GetTrackMatrixivNV);
     store!(GetTransformFeedbackVarying);
     store!(GetTransformFeedbackVaryingEXT);
     store!(GetTransformFeedbackVaryingNV);
     store!(GetTransformFeedbacki64_v);
     store!(GetTransformFeedbacki_v);
     store!(GetTransformFeedbackiv);
     store!(GetUniformBlockIndex);
     store!(GetUniformBufferSizeEXT);
     store!(GetUniformIndices);
     store!(GetUniformLocation);
     store!(GetUniformLocationARB);
     store!(GetUniformOffsetEXT);
     store!(GetUniformSubroutineuiv);
     store!(GetUniformdv);
     store!(GetUniformfv);
     store!(GetUniformfvARB);
     store!(GetUniformi64vARB);
     store!(GetUniformi64vNV);
     store!(GetUniformiv);
     store!(GetUniformivARB);
     store!(GetUniformui64vARB);
     store!(GetUniformui64vNV);
     store!(GetUniformuiv);
     store!(GetUniformuivEXT);
     store!(GetUnsignedBytei_vEXT);
     store!(GetUnsignedBytevEXT);
     store!(GetVariantArrayObjectfvATI);
     store!(GetVariantArrayObjectivATI);
     store!(GetVariantBooleanvEXT);
     store!(GetVariantFloatvEXT);
     store!(GetVariantIntegervEXT);
     store!(GetVariantPointervEXT);
     store!(GetVaryingLocationNV);
     store!(GetVertexArrayIndexed64iv);
     store!(GetVertexArrayIndexediv);
     store!(GetVertexArrayIntegeri_vEXT);
     store!(GetVertexArrayIntegervEXT);
     store!(GetVertexArrayPointeri_vEXT);
     store!(GetVertexArrayPointervEXT);
     store!(GetVertexArrayiv);
     store!(GetVertexAttribArrayObjectfvATI);
     store!(GetVertexAttribArrayObjectivATI);
     store!(GetVertexAttribIiv);
     store!(GetVertexAttribIivEXT);
     store!(GetVertexAttribIuiv);
     store!(GetVertexAttribIuivEXT);
     store!(GetVertexAttribLdv);
     store!(GetVertexAttribLdvEXT);
     store!(GetVertexAttribLi64vNV);
     store!(GetVertexAttribLui64vARB);
     store!(GetVertexAttribLui64vNV);
     store!(GetVertexAttribPointerv);
     store!(GetVertexAttribPointervARB);
     store!(GetVertexAttribPointervNV);
     store!(GetVertexAttribdv);
     store!(GetVertexAttribdvARB);
     store!(GetVertexAttribdvNV);
     store!(GetVertexAttribfv);
     store!(GetVertexAttribfvARB);
     store!(GetVertexAttribfvNV);
     store!(GetVertexAttribiv);
     store!(GetVertexAttribivARB);
     store!(GetVertexAttribivNV);
     store!(GetVideoCaptureStreamdvNV);
     store!(GetVideoCaptureStreamfvNV);
     store!(GetVideoCaptureStreamivNV);
     store!(GetVideoCaptureivNV);
     store!(GetVideoi64vNV);
     store!(GetVideoivNV);
     store!(GetVideoui64vNV);
     store!(GetVideouivNV);
     store!(GetVkProcAddrNV);
     store!(GetnCompressedTexImage);
     store!(GetnCompressedTexImageARB);
     store!(GetnTexImage);
     store!(GetnTexImageARB);
     store!(GetnUniformdv);
     store!(GetnUniformdvARB);
     store!(GetnUniformfv);
     store!(GetnUniformfvARB);
     store!(GetnUniformi64vARB);
     store!(GetnUniformiv);
     store!(GetnUniformivARB);
     store!(GetnUniformui64vARB);
     store!(GetnUniformuiv);
     store!(GetnUniformuivARB);
     store!(GlobalAlphaFactorbSUN);
     store!(GlobalAlphaFactordSUN);
     store!(GlobalAlphaFactorfSUN);
     store!(GlobalAlphaFactoriSUN);
     store!(GlobalAlphaFactorsSUN);
     store!(GlobalAlphaFactorubSUN);
     store!(GlobalAlphaFactoruiSUN);
     store!(GlobalAlphaFactorusSUN);
     store!(Hint);
     store!(HintPGI);
     store!(HistogramEXT);
     store!(IglooInterfaceSGIX);
     store!(ImageTransformParameterfHP);
     store!(ImageTransformParameterfvHP);
     store!(ImageTransformParameteriHP);
     store!(ImageTransformParameterivHP);
     store!(ImportMemoryFdEXT);
     store!(ImportMemoryWin32HandleEXT);
     store!(ImportMemoryWin32NameEXT);
     store!(ImportSemaphoreFdEXT);
     store!(ImportSemaphoreWin32HandleEXT);
     store!(ImportSemaphoreWin32NameEXT);
     store!(ImportSyncEXT);
     store!(IndexFormatNV);
     store!(IndexFuncEXT);
     store!(IndexMaterialEXT);
     store!(IndexPointerEXT);
     store!(IndexPointerListIBM);
     store!(IndexxOES);
     store!(IndexxvOES);
     store!(InsertComponentEXT);
     store!(InsertEventMarkerEXT);
     store!(InstrumentsBufferSGIX);
     store!(InterpolatePathsNV);
     store!(InvalidateBufferData);
     store!(InvalidateBufferSubData);
     store!(InvalidateFramebuffer);
     store!(InvalidateNamedFramebufferData);
     store!(InvalidateNamedFramebufferSubData);
     store!(InvalidateSubFramebuffer);
     store!(InvalidateTexImage);
     store!(InvalidateTexSubImage);
     store!(IsAsyncMarkerSGIX);
     store!(IsBuffer);
     store!(IsBufferARB);
     store!(IsBufferResidentNV);
     store!(IsCommandListNV);
     store!(IsEnabled);
     store!(IsEnabledIndexedEXT);
     store!(IsEnabledi);
     store!(IsFenceAPPLE);
     store!(IsFenceNV);
     store!(IsFramebuffer);
     store!(IsFramebufferEXT);
     store!(IsImageHandleResidentARB);
     store!(IsImageHandleResidentNV);
     store!(IsMemoryObjectEXT);
     store!(IsNameAMD);
     store!(IsNamedBufferResidentNV);
     store!(IsNamedStringARB);
     store!(IsObjectBufferATI);
     store!(IsOcclusionQueryNV);
     store!(IsPathNV);
     store!(IsPointInFillPathNV);
     store!(IsPointInStrokePathNV);
     store!(IsProgram);
     store!(IsProgramARB);
     store!(IsProgramNV);
     store!(IsProgramPipeline);
     store!(IsQuery);
     store!(IsQueryARB);
     store!(IsRenderbuffer);
     store!(IsRenderbufferEXT);
     store!(IsSampler);
     store!(IsSemaphoreEXT);
     store!(IsShader);
     store!(IsStateNV);
     store!(IsSync);
     store!(IsTexture);
     store!(IsTextureEXT);
     store!(IsTextureHandleResidentARB);
     store!(IsTextureHandleResidentNV);
     store!(IsTransformFeedback);
     store!(IsTransformFeedbackNV);
     store!(IsVariantEnabledEXT);
     store!(IsVertexArray);
     store!(IsVertexArrayAPPLE);
     store!(IsVertexAttribEnabledAPPLE);
     store!(LGPUCopyImageSubDataNVX);
     store!(LGPUInterlockNVX);
     store!(LGPUNamedBufferSubDataNVX);
     store!(LabelObjectEXT);
     store!(LightEnviSGIX);
     store!(LightModelxOES);
     store!(LightModelxvOES);
     store!(LightxOES);
     store!(LightxvOES);
     store!(LineWidth);
     store!(LineWidthxOES);
     store!(LinkProgram);
     store!(LinkProgramARB);
     store!(ListDrawCommandsStatesClientNV);
     store!(ListParameterfSGIX);
     store!(ListParameterfvSGIX);
     store!(ListParameteriSGIX);
     store!(ListParameterivSGIX);
     store!(LoadIdentityDeformationMapSGIX);
     store!(LoadMatrixxOES);
     store!(LoadProgramNV);
     store!(LoadTransposeMatrixdARB);
     store!(LoadTransposeMatrixfARB);
     store!(LoadTransposeMatrixxOES);
     store!(LockArraysEXT);
     store!(LogicOp);
     store!(MakeBufferNonResidentNV);
     store!(MakeBufferResidentNV);
     store!(MakeImageHandleNonResidentARB);
     store!(MakeImageHandleNonResidentNV);
     store!(MakeImageHandleResidentARB);
     store!(MakeImageHandleResidentNV);
     store!(MakeNamedBufferNonResidentNV);
     store!(MakeNamedBufferResidentNV);
     store!(MakeTextureHandleNonResidentARB);
     store!(MakeTextureHandleNonResidentNV);
     store!(MakeTextureHandleResidentARB);
     store!(MakeTextureHandleResidentNV);
     store!(Map1xOES);
     store!(Map2xOES);
     store!(MapBuffer);
     store!(MapBufferARB);
     store!(MapBufferRange);
     store!(MapControlPointsNV);
     store!(MapGrid1xOES);
     store!(MapGrid2xOES);
     store!(MapNamedBuffer);
     store!(MapNamedBufferEXT);
     store!(MapNamedBufferRange);
     store!(MapNamedBufferRangeEXT);
     store!(MapObjectBufferATI);
     store!(MapParameterfvNV);
     store!(MapParameterivNV);
     store!(MapTexture2DINTEL);
     store!(MapVertexAttrib1dAPPLE);
     store!(MapVertexAttrib1fAPPLE);
     store!(MapVertexAttrib2dAPPLE);
     store!(MapVertexAttrib2fAPPLE);
     store!(MaterialxOES);
     store!(MaterialxvOES);
     store!(MatrixFrustumEXT);
     store!(MatrixIndexPointerARB);
     store!(MatrixIndexubvARB);
     store!(MatrixIndexuivARB);
     store!(MatrixIndexusvARB);
     store!(MatrixLoad3x2fNV);
     store!(MatrixLoad3x3fNV);
     store!(MatrixLoadIdentityEXT);
     store!(MatrixLoadTranspose3x3fNV);
     store!(MatrixLoadTransposedEXT);
     store!(MatrixLoadTransposefEXT);
     store!(MatrixLoaddEXT);
     store!(MatrixLoadfEXT);
     store!(MatrixMult3x2fNV);
     store!(MatrixMult3x3fNV);
     store!(MatrixMultTranspose3x3fNV);
     store!(MatrixMultTransposedEXT);
     store!(MatrixMultTransposefEXT);
     store!(MatrixMultdEXT);
     store!(MatrixMultfEXT);
     store!(MatrixOrthoEXT);
     store!(MatrixPopEXT);
     store!(MatrixPushEXT);
     store!(MatrixRotatedEXT);
     store!(MatrixRotatefEXT);
     store!(MatrixScaledEXT);
     store!(MatrixScalefEXT);
     store!(MatrixTranslatedEXT);
     store!(MatrixTranslatefEXT);
     store!(MaxShaderCompilerThreadsARB);
     store!(MaxShaderCompilerThreadsKHR);
     store!(MemoryBarrier);
     store!(MemoryBarrierByRegion);
     store!(MemoryBarrierEXT);
     store!(MemoryObjectParameterivEXT);
     store!(MinSampleShading);
     store!(MinSampleShadingARB);
     store!(MinmaxEXT);
     store!(MultMatrixxOES);
     store!(MultTransposeMatrixdARB);
     store!(MultTransposeMatrixfARB);
     store!(MultTransposeMatrixxOES);
     store!(MultiDrawArrays);
     store!(MultiDrawArraysEXT);
     store!(MultiDrawArraysIndirect);
     store!(MultiDrawArraysIndirectAMD);
     store!(MultiDrawArraysIndirectBindlessCountNV);
     store!(MultiDrawArraysIndirectBindlessNV);
     store!(MultiDrawArraysIndirectCount);
     store!(MultiDrawArraysIndirectCountARB);
     store!(MultiDrawElementArrayAPPLE);
     store!(MultiDrawElements);
     store!(MultiDrawElementsBaseVertex);
     store!(MultiDrawElementsEXT);
     store!(MultiDrawElementsIndirect);
     store!(MultiDrawElementsIndirectAMD);
     store!(MultiDrawElementsIndirectBindlessCountNV);
     store!(MultiDrawElementsIndirectBindlessNV);
     store!(MultiDrawElementsIndirectCount);
     store!(MultiDrawElementsIndirectCountARB);
     store!(MultiDrawMeshTasksIndirectCountNV);
     store!(MultiDrawMeshTasksIndirectNV);
     store!(MultiDrawRangeElementArrayAPPLE);
     store!(MultiModeDrawArraysIBM);
     store!(MultiModeDrawElementsIBM);
     store!(MultiTexBufferEXT);
     store!(MultiTexCoord1bOES);
     store!(MultiTexCoord1bvOES);
     store!(MultiTexCoord1dARB);
     store!(MultiTexCoord1dvARB);
     store!(MultiTexCoord1fARB);
     store!(MultiTexCoord1fvARB);
     store!(MultiTexCoord1hNV);
     store!(MultiTexCoord1hvNV);
     store!(MultiTexCoord1iARB);
     store!(MultiTexCoord1ivARB);
     store!(MultiTexCoord1sARB);
     store!(MultiTexCoord1svARB);
     store!(MultiTexCoord1xOES);
     store!(MultiTexCoord1xvOES);
     store!(MultiTexCoord2bOES);
     store!(MultiTexCoord2bvOES);
     store!(MultiTexCoord2dARB);
     store!(MultiTexCoord2dvARB);
     store!(MultiTexCoord2fARB);
     store!(MultiTexCoord2fvARB);
     store!(MultiTexCoord2hNV);
     store!(MultiTexCoord2hvNV);
     store!(MultiTexCoord2iARB);
     store!(MultiTexCoord2ivARB);
     store!(MultiTexCoord2sARB);
     store!(MultiTexCoord2svARB);
     store!(MultiTexCoord2xOES);
     store!(MultiTexCoord2xvOES);
     store!(MultiTexCoord3bOES);
     store!(MultiTexCoord3bvOES);
     store!(MultiTexCoord3dARB);
     store!(MultiTexCoord3dvARB);
     store!(MultiTexCoord3fARB);
     store!(MultiTexCoord3fvARB);
     store!(MultiTexCoord3hNV);
     store!(MultiTexCoord3hvNV);
     store!(MultiTexCoord3iARB);
     store!(MultiTexCoord3ivARB);
     store!(MultiTexCoord3sARB);
     store!(MultiTexCoord3svARB);
     store!(MultiTexCoord3xOES);
     store!(MultiTexCoord3xvOES);
     store!(MultiTexCoord4bOES);
     store!(MultiTexCoord4bvOES);
     store!(MultiTexCoord4dARB);
     store!(MultiTexCoord4dvARB);
     store!(MultiTexCoord4fARB);
     store!(MultiTexCoord4fvARB);
     store!(MultiTexCoord4hNV);
     store!(MultiTexCoord4hvNV);
     store!(MultiTexCoord4iARB);
     store!(MultiTexCoord4ivARB);
     store!(MultiTexCoord4sARB);
     store!(MultiTexCoord4svARB);
     store!(MultiTexCoord4xOES);
     store!(MultiTexCoord4xvOES);
     store!(MultiTexCoordPointerEXT);
     store!(MultiTexEnvfEXT);
     store!(MultiTexEnvfvEXT);
     store!(MultiTexEnviEXT);
     store!(MultiTexEnvivEXT);
     store!(MultiTexGendEXT);
     store!(MultiTexGendvEXT);
     store!(MultiTexGenfEXT);
     store!(MultiTexGenfvEXT);
     store!(MultiTexGeniEXT);
     store!(MultiTexGenivEXT);
     store!(MultiTexImage1DEXT);
     store!(MultiTexImage2DEXT);
     store!(MultiTexImage3DEXT);
     store!(MultiTexParameterIivEXT);
     store!(MultiTexParameterIuivEXT);
     store!(MultiTexParameterfEXT);
     store!(MultiTexParameterfvEXT);
     store!(MultiTexParameteriEXT);
     store!(MultiTexParameterivEXT);
     store!(MultiTexRenderbufferEXT);
     store!(MultiTexSubImage1DEXT);
     store!(MultiTexSubImage2DEXT);
     store!(MultiTexSubImage3DEXT);
     store!(MulticastBarrierNV);
     store!(MulticastBlitFramebufferNV);
     store!(MulticastBufferSubDataNV);
     store!(MulticastCopyBufferSubDataNV);
     store!(MulticastCopyImageSubDataNV);
     store!(MulticastFramebufferSampleLocationsfvNV);
     store!(MulticastGetQueryObjecti64vNV);
     store!(MulticastGetQueryObjectivNV);
     store!(MulticastGetQueryObjectui64vNV);
     store!(MulticastGetQueryObjectuivNV);
     store!(MulticastScissorArrayvNVX);
     store!(MulticastViewportArrayvNVX);
     store!(MulticastViewportPositionWScaleNVX);
     store!(MulticastWaitSyncNV);
     store!(NamedBufferAttachMemoryNV);
     store!(NamedBufferData);
     store!(NamedBufferDataEXT);
     store!(NamedBufferPageCommitmentARB);
     store!(NamedBufferPageCommitmentEXT);
     store!(NamedBufferPageCommitmentMemNV);
     store!(NamedBufferStorage);
     store!(NamedBufferStorageEXT);
     store!(NamedBufferStorageExternalEXT);
     store!(NamedBufferStorageMemEXT);
     store!(NamedBufferSubData);
     store!(NamedBufferSubDataEXT);
     store!(NamedCopyBufferSubDataEXT);
     store!(NamedFramebufferDrawBuffer);
     store!(NamedFramebufferDrawBuffers);
     store!(NamedFramebufferParameteri);
     store!(NamedFramebufferParameteriEXT);
     store!(NamedFramebufferReadBuffer);
     store!(NamedFramebufferRenderbuffer);
     store!(NamedFramebufferRenderbufferEXT);
     store!(NamedFramebufferSampleLocationsfvARB);
     store!(NamedFramebufferSampleLocationsfvNV);
     store!(NamedFramebufferSamplePositionsfvAMD);
     store!(NamedFramebufferTexture);
     store!(NamedFramebufferTexture1DEXT);
     store!(NamedFramebufferTexture2DEXT);
     store!(NamedFramebufferTexture3DEXT);
     store!(NamedFramebufferTextureEXT);
     store!(NamedFramebufferTextureFaceEXT);
     store!(NamedFramebufferTextureLayer);
     store!(NamedFramebufferTextureLayerEXT);
     store!(NamedProgramLocalParameter4dEXT);
     store!(NamedProgramLocalParameter4dvEXT);
     store!(NamedProgramLocalParameter4fEXT);
     store!(NamedProgramLocalParameter4fvEXT);
     store!(NamedProgramLocalParameterI4iEXT);
     store!(NamedProgramLocalParameterI4ivEXT);
     store!(NamedProgramLocalParameterI4uiEXT);
     store!(NamedProgramLocalParameterI4uivEXT);
     store!(NamedProgramLocalParameters4fvEXT);
     store!(NamedProgramLocalParametersI4ivEXT);
     store!(NamedProgramLocalParametersI4uivEXT);
     store!(NamedProgramStringEXT);
     store!(NamedRenderbufferStorage);
     store!(NamedRenderbufferStorageEXT);
     store!(NamedRenderbufferStorageMultisample);
     store!(NamedRenderbufferStorageMultisampleAdvancedAMD);
     store!(NamedRenderbufferStorageMultisampleCoverageEXT);
     store!(NamedRenderbufferStorageMultisampleEXT);
     store!(NamedStringARB);
     store!(NewObjectBufferATI);
     store!(Normal3fVertex3fSUN);
     store!(Normal3fVertex3fvSUN);
     store!(Normal3hNV);
     store!(Normal3hvNV);
     store!(Normal3xOES);
     store!(Normal3xvOES);
     store!(NormalFormatNV);
     store!(NormalPointerEXT);
     store!(NormalPointerListIBM);
     store!(NormalPointervINTEL);
     store!(NormalStream3bATI);
     store!(NormalStream3bvATI);
     store!(NormalStream3dATI);
     store!(NormalStream3dvATI);
     store!(NormalStream3fATI);
     store!(NormalStream3fvATI);
     store!(NormalStream3iATI);
     store!(NormalStream3ivATI);
     store!(NormalStream3sATI);
     store!(NormalStream3svATI);
     store!(ObjectLabel);
     store!(ObjectPtrLabel);
     store!(ObjectPurgeableAPPLE);
     store!(ObjectUnpurgeableAPPLE);
     store!(OrthofOES);
     store!(OrthoxOES);
     store!(PNTrianglesfATI);
     store!(PNTrianglesiATI);
     store!(PassTexCoordATI);
     store!(PassThroughxOES);
     store!(PatchParameterfv);
     store!(PatchParameteri);
     store!(PathCommandsNV);
     store!(PathCoordsNV);
     store!(PathCoverDepthFuncNV);
     store!(PathDashArrayNV);
     store!(PathGlyphIndexArrayNV);
     store!(PathGlyphIndexRangeNV);
     store!(PathGlyphRangeNV);
     store!(PathGlyphsNV);
     store!(PathMemoryGlyphIndexArrayNV);
     store!(PathParameterfNV);
     store!(PathParameterfvNV);
     store!(PathParameteriNV);
     store!(PathParameterivNV);
     store!(PathStencilDepthOffsetNV);
     store!(PathStencilFuncNV);
     store!(PathStringNV);
     store!(PathSubCommandsNV);
     store!(PathSubCoordsNV);
     store!(PauseTransformFeedback);
     store!(PauseTransformFeedbackNV);
     store!(PixelDataRangeNV);
     store!(PixelMapx);
     store!(PixelStoref);
     store!(PixelStorei);
     store!(PixelStorex);
     store!(PixelTexGenParameterfSGIS);
     store!(PixelTexGenParameterfvSGIS);
     store!(PixelTexGenParameteriSGIS);
     store!(PixelTexGenParameterivSGIS);
     store!(PixelTexGenSGIX);
     store!(PixelTransferxOES);
     store!(PixelTransformParameterfEXT);
     store!(PixelTransformParameterfvEXT);
     store!(PixelTransformParameteriEXT);
     store!(PixelTransformParameterivEXT);
     store!(PixelZoomxOES);
     store!(PointAlongPathNV);
     store!(PointParameterf);
     store!(PointParameterfARB);
     store!(PointParameterfEXT);
     store!(PointParameterfSGIS);
     store!(PointParameterfv);
     store!(PointParameterfvARB);
     store!(PointParameterfvEXT);
     store!(PointParameterfvSGIS);
     store!(PointParameteri);
     store!(PointParameteriNV);
     store!(PointParameteriv);
     store!(PointParameterivNV);
     store!(PointParameterxvOES);
     store!(PointSize);
     store!(PointSizexOES);
     store!(PollAsyncSGIX);
     store!(PollInstrumentsSGIX);
     store!(PolygonMode);
     store!(PolygonOffset);
     store!(PolygonOffsetClamp);
     store!(PolygonOffsetClampEXT);
     store!(PolygonOffsetEXT);
     store!(PolygonOffsetxOES);
     store!(PopDebugGroup);
     store!(PopGroupMarkerEXT);
     store!(PresentFrameDualFillNV);
     store!(PresentFrameKeyedNV);
     store!(PrimitiveBoundingBoxARB);
     store!(PrimitiveRestartIndex);
     store!(PrimitiveRestartIndexNV);
     store!(PrimitiveRestartNV);
     store!(PrioritizeTexturesEXT);
     store!(PrioritizeTexturesxOES);
     store!(ProgramBinary);
     store!(ProgramBufferParametersIivNV);
     store!(ProgramBufferParametersIuivNV);
     store!(ProgramBufferParametersfvNV);
     store!(ProgramEnvParameter4dARB);
     store!(ProgramEnvParameter4dvARB);
     store!(ProgramEnvParameter4fARB);
     store!(ProgramEnvParameter4fvARB);
     store!(ProgramEnvParameterI4iNV);
     store!(ProgramEnvParameterI4ivNV);
     store!(ProgramEnvParameterI4uiNV);
     store!(ProgramEnvParameterI4uivNV);
     store!(ProgramEnvParameters4fvEXT);
     store!(ProgramEnvParametersI4ivNV);
     store!(ProgramEnvParametersI4uivNV);
     store!(ProgramLocalParameter4dARB);
     store!(ProgramLocalParameter4dvARB);
     store!(ProgramLocalParameter4fARB);
     store!(ProgramLocalParameter4fvARB);
     store!(ProgramLocalParameterI4iNV);
     store!(ProgramLocalParameterI4ivNV);
     store!(ProgramLocalParameterI4uiNV);
     store!(ProgramLocalParameterI4uivNV);
     store!(ProgramLocalParameters4fvEXT);
     store!(ProgramLocalParametersI4ivNV);
     store!(ProgramLocalParametersI4uivNV);
     store!(ProgramNamedParameter4dNV);
     store!(ProgramNamedParameter4dvNV);
     store!(ProgramNamedParameter4fNV);
     store!(ProgramNamedParameter4fvNV);
     store!(ProgramParameter4dNV);
     store!(ProgramParameter4dvNV);
     store!(ProgramParameter4fNV);
     store!(ProgramParameter4fvNV);
     store!(ProgramParameteri);
     store!(ProgramParameteriARB);
     store!(ProgramParameteriEXT);
     store!(ProgramParameters4dvNV);
     store!(ProgramParameters4fvNV);
     store!(ProgramPathFragmentInputGenNV);
     store!(ProgramStringARB);
     store!(ProgramSubroutineParametersuivNV);
     store!(ProgramUniform1d);
     store!(ProgramUniform1dEXT);
     store!(ProgramUniform1dv);
     store!(ProgramUniform1dvEXT);
     store!(ProgramUniform1f);
     store!(ProgramUniform1fEXT);
     store!(ProgramUniform1fv);
     store!(ProgramUniform1fvEXT);
     store!(ProgramUniform1i);
     store!(ProgramUniform1i64ARB);
     store!(ProgramUniform1i64NV);
     store!(ProgramUniform1i64vARB);
     store!(ProgramUniform1i64vNV);
     store!(ProgramUniform1iEXT);
     store!(ProgramUniform1iv);
     store!(ProgramUniform1ivEXT);
     store!(ProgramUniform1ui);
     store!(ProgramUniform1ui64ARB);
     store!(ProgramUniform1ui64NV);
     store!(ProgramUniform1ui64vARB);
     store!(ProgramUniform1ui64vNV);
     store!(ProgramUniform1uiEXT);
     store!(ProgramUniform1uiv);
     store!(ProgramUniform1uivEXT);
     store!(ProgramUniform2d);
     store!(ProgramUniform2dEXT);
     store!(ProgramUniform2dv);
     store!(ProgramUniform2dvEXT);
     store!(ProgramUniform2f);
     store!(ProgramUniform2fEXT);
     store!(ProgramUniform2fv);
     store!(ProgramUniform2fvEXT);
     store!(ProgramUniform2i);
     store!(ProgramUniform2i64ARB);
     store!(ProgramUniform2i64NV);
     store!(ProgramUniform2i64vARB);
     store!(ProgramUniform2i64vNV);
     store!(ProgramUniform2iEXT);
     store!(ProgramUniform2iv);
     store!(ProgramUniform2ivEXT);
     store!(ProgramUniform2ui);
     store!(ProgramUniform2ui64ARB);
     store!(ProgramUniform2ui64NV);
     store!(ProgramUniform2ui64vARB);
     store!(ProgramUniform2ui64vNV);
     store!(ProgramUniform2uiEXT);
     store!(ProgramUniform2uiv);
     store!(ProgramUniform2uivEXT);
     store!(ProgramUniform3d);
     store!(ProgramUniform3dEXT);
     store!(ProgramUniform3dv);
     store!(ProgramUniform3dvEXT);
     store!(ProgramUniform3f);
     store!(ProgramUniform3fEXT);
     store!(ProgramUniform3fv);
     store!(ProgramUniform3fvEXT);
     store!(ProgramUniform3i);
     store!(ProgramUniform3i64ARB);
     store!(ProgramUniform3i64NV);
     store!(ProgramUniform3i64vARB);
     store!(ProgramUniform3i64vNV);
     store!(ProgramUniform3iEXT);
     store!(ProgramUniform3iv);
     store!(ProgramUniform3ivEXT);
     store!(ProgramUniform3ui);
     store!(ProgramUniform3ui64ARB);
     store!(ProgramUniform3ui64NV);
     store!(ProgramUniform3ui64vARB);
     store!(ProgramUniform3ui64vNV);
     store!(ProgramUniform3uiEXT);
     store!(ProgramUniform3uiv);
     store!(ProgramUniform3uivEXT);
     store!(ProgramUniform4d);
     store!(ProgramUniform4dEXT);
     store!(ProgramUniform4dv);
     store!(ProgramUniform4dvEXT);
     store!(ProgramUniform4f);
     store!(ProgramUniform4fEXT);
     store!(ProgramUniform4fv);
     store!(ProgramUniform4fvEXT);
     store!(ProgramUniform4i);
     store!(ProgramUniform4i64ARB);
     store!(ProgramUniform4i64NV);
     store!(ProgramUniform4i64vARB);
     store!(ProgramUniform4i64vNV);
     store!(ProgramUniform4iEXT);
     store!(ProgramUniform4iv);
     store!(ProgramUniform4ivEXT);
     store!(ProgramUniform4ui);
     store!(ProgramUniform4ui64ARB);
     store!(ProgramUniform4ui64NV);
     store!(ProgramUniform4ui64vARB);
     store!(ProgramUniform4ui64vNV);
     store!(ProgramUniform4uiEXT);
     store!(ProgramUniform4uiv);
     store!(ProgramUniform4uivEXT);
     store!(ProgramUniformHandleui64ARB);
     store!(ProgramUniformHandleui64NV);
     store!(ProgramUniformHandleui64vARB);
     store!(ProgramUniformHandleui64vNV);
     store!(ProgramUniformMatrix2dv);
     store!(ProgramUniformMatrix2dvEXT);
     store!(ProgramUniformMatrix2fv);
     store!(ProgramUniformMatrix2fvEXT);
     store!(ProgramUniformMatrix2x3dv);
     store!(ProgramUniformMatrix2x3dvEXT);
     store!(ProgramUniformMatrix2x3fv);
     store!(ProgramUniformMatrix2x3fvEXT);
     store!(ProgramUniformMatrix2x4dv);
     store!(ProgramUniformMatrix2x4dvEXT);
     store!(ProgramUniformMatrix2x4fv);
     store!(ProgramUniformMatrix2x4fvEXT);
     store!(ProgramUniformMatrix3dv);
     store!(ProgramUniformMatrix3dvEXT);
     store!(ProgramUniformMatrix3fv);
     store!(ProgramUniformMatrix3fvEXT);
     store!(ProgramUniformMatrix3x2dv);
     store!(ProgramUniformMatrix3x2dvEXT);
     store!(ProgramUniformMatrix3x2fv);
     store!(ProgramUniformMatrix3x2fvEXT);
     store!(ProgramUniformMatrix3x4dv);
     store!(ProgramUniformMatrix3x4dvEXT);
     store!(ProgramUniformMatrix3x4fv);
     store!(ProgramUniformMatrix3x4fvEXT);
     store!(ProgramUniformMatrix4dv);
     store!(ProgramUniformMatrix4dvEXT);
     store!(ProgramUniformMatrix4fv);
     store!(ProgramUniformMatrix4fvEXT);
     store!(ProgramUniformMatrix4x2dv);
     store!(ProgramUniformMatrix4x2dvEXT);
     store!(ProgramUniformMatrix4x2fv);
     store!(ProgramUniformMatrix4x2fvEXT);
     store!(ProgramUniformMatrix4x3dv);
     store!(ProgramUniformMatrix4x3dvEXT);
     store!(ProgramUniformMatrix4x3fv);
     store!(ProgramUniformMatrix4x3fvEXT);
     store!(ProgramUniformui64NV);
     store!(ProgramUniformui64vNV);
     store!(ProgramVertexLimitNV);
     store!(ProvokingVertex);
     store!(ProvokingVertexEXT);
     store!(PushClientAttribDefaultEXT);
     store!(PushDebugGroup);
     store!(PushGroupMarkerEXT);
     store!(QueryCounter);
     store!(QueryMatrixxOES);
     store!(QueryObjectParameteruiAMD);
     store!(QueryResourceNV);
     store!(QueryResourceTagNV);
     store!(RasterPos2xOES);
     store!(RasterPos2xvOES);
     store!(RasterPos3xOES);
     store!(RasterPos3xvOES);
     store!(RasterPos4xOES);
     store!(RasterPos4xvOES);
     store!(RasterSamplesEXT);
     store!(ReadBuffer);
     store!(ReadInstrumentsSGIX);
     store!(ReadPixels);
     store!(ReadnPixels);
     store!(ReadnPixelsARB);
     store!(RectxOES);
     store!(RectxvOES);
     store!(ReferencePlaneSGIX);
     store!(ReleaseKeyedMutexWin32EXT);
     store!(ReleaseShaderCompiler);
     store!(RenderGpuMaskNV);
     store!(RenderbufferStorage);
     store!(RenderbufferStorageEXT);
     store!(RenderbufferStorageMultisample);
     store!(RenderbufferStorageMultisampleAdvancedAMD);
     store!(RenderbufferStorageMultisampleCoverageNV);
     store!(RenderbufferStorageMultisampleEXT);
     store!(ReplacementCodePointerSUN);
     store!(ReplacementCodeubSUN);
     store!(ReplacementCodeubvSUN);
     store!(ReplacementCodeuiColor3fVertex3fSUN);
     store!(ReplacementCodeuiColor3fVertex3fvSUN);
     store!(ReplacementCodeuiColor4fNormal3fVertex3fSUN);
     store!(ReplacementCodeuiColor4fNormal3fVertex3fvSUN);
     store!(ReplacementCodeuiColor4ubVertex3fSUN);
     store!(ReplacementCodeuiColor4ubVertex3fvSUN);
     store!(ReplacementCodeuiNormal3fVertex3fSUN);
     store!(ReplacementCodeuiNormal3fVertex3fvSUN);
     store!(ReplacementCodeuiSUN);
     store!(ReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fSUN);
     store!(ReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fvSUN);
     store!(ReplacementCodeuiTexCoord2fNormal3fVertex3fSUN);
     store!(ReplacementCodeuiTexCoord2fNormal3fVertex3fvSUN);
     store!(ReplacementCodeuiTexCoord2fVertex3fSUN);
     store!(ReplacementCodeuiTexCoord2fVertex3fvSUN);
     store!(ReplacementCodeuiVertex3fSUN);
     store!(ReplacementCodeuiVertex3fvSUN);
     store!(ReplacementCodeuivSUN);
     store!(ReplacementCodeusSUN);
     store!(ReplacementCodeusvSUN);
     store!(RequestResidentProgramsNV);
     store!(ResetHistogramEXT);
     store!(ResetMemoryObjectParameterNV);
     store!(ResetMinmaxEXT);
     store!(ResizeBuffersMESA);
     store!(ResolveDepthValuesNV);
     store!(ResumeTransformFeedback);
     store!(ResumeTransformFeedbackNV);
     store!(RotatexOES);
     store!(SampleCoverage);
     store!(SampleCoverageARB);
     store!(SampleMapATI);
     store!(SampleMaskEXT);
     store!(SampleMaskIndexedNV);
     store!(SampleMaskSGIS);
     store!(SampleMaski);
     store!(SamplePatternEXT);
     store!(SamplePatternSGIS);
     store!(SamplerParameterIiv);
     store!(SamplerParameterIuiv);
     store!(SamplerParameterf);
     store!(SamplerParameterfv);
     store!(SamplerParameteri);
     store!(SamplerParameteriv);
     store!(ScalexOES);
     store!(Scissor);
     store!(ScissorArrayv);
     store!(ScissorExclusiveArrayvNV);
     store!(ScissorExclusiveNV);
     store!(ScissorIndexed);
     store!(ScissorIndexedv);
     store!(SecondaryColor3bEXT);
     store!(SecondaryColor3bvEXT);
     store!(SecondaryColor3dEXT);
     store!(SecondaryColor3dvEXT);
     store!(SecondaryColor3fEXT);
     store!(SecondaryColor3fvEXT);
     store!(SecondaryColor3hNV);
     store!(SecondaryColor3hvNV);
     store!(SecondaryColor3iEXT);
     store!(SecondaryColor3ivEXT);
     store!(SecondaryColor3sEXT);
     store!(SecondaryColor3svEXT);
     store!(SecondaryColor3ubEXT);
     store!(SecondaryColor3ubvEXT);
     store!(SecondaryColor3uiEXT);
     store!(SecondaryColor3uivEXT);
     store!(SecondaryColor3usEXT);
     store!(SecondaryColor3usvEXT);
     store!(SecondaryColorFormatNV);
     store!(SecondaryColorPointerEXT);
     store!(SecondaryColorPointerListIBM);
     store!(SelectPerfMonitorCountersAMD);
     store!(SemaphoreParameterivNV);
     store!(SemaphoreParameterui64vEXT);
     store!(SeparableFilter2DEXT);
     store!(SetFenceAPPLE);
     store!(SetFenceNV);
     store!(SetFragmentShaderConstantATI);
     store!(SetInvariantEXT);
     store!(SetLocalConstantEXT);
     store!(SetMultisamplefvAMD);
     store!(ShaderBinary);
     store!(ShaderOp1EXT);
     store!(ShaderOp2EXT);
     store!(ShaderOp3EXT);
     store!(ShaderSource);
     store!(ShaderSourceARB);
     store!(ShaderStorageBlockBinding);
     store!(ShadingRateImageBarrierNV);
     store!(ShadingRateImagePaletteNV);
     store!(ShadingRateSampleOrderCustomNV);
     store!(ShadingRateSampleOrderNV);
     store!(SharpenTexFuncSGIS);
     store!(SignalSemaphoreEXT);
     store!(SignalSemaphoreui64NVX);
     store!(SignalVkFenceNV);
     store!(SignalVkSemaphoreNV);
     store!(SpecializeShader);
     store!(SpecializeShaderARB);
     store!(SpriteParameterfSGIX);
     store!(SpriteParameterfvSGIX);
     store!(SpriteParameteriSGIX);
     store!(SpriteParameterivSGIX);
     store!(StartInstrumentsSGIX);
     store!(StateCaptureNV);
     store!(StencilClearTagEXT);
     store!(StencilFillPathInstancedNV);
     store!(StencilFillPathNV);
     store!(StencilFunc);
     store!(StencilFuncSeparate);
     store!(StencilFuncSeparateATI);
     store!(StencilMask);
     store!(StencilMaskSeparate);
     store!(StencilOp);
     store!(StencilOpSeparate);
     store!(StencilOpSeparateATI);
     store!(StencilOpValueAMD);
     store!(StencilStrokePathInstancedNV);
     store!(StencilStrokePathNV);
     store!(StencilThenCoverFillPathInstancedNV);
     store!(StencilThenCoverFillPathNV);
     store!(StencilThenCoverStrokePathInstancedNV);
     store!(StencilThenCoverStrokePathNV);
     store!(StopInstrumentsSGIX);
     store!(StringMarkerGREMEDY);
     store!(SubpixelPrecisionBiasNV);
     store!(SwizzleEXT);
     store!(SyncTextureINTEL);
     store!(TagSampleBufferSGIX);
     store!(Tangent3bEXT);
     store!(Tangent3bvEXT);
     store!(Tangent3dEXT);
     store!(Tangent3dvEXT);
     store!(Tangent3fEXT);
     store!(Tangent3fvEXT);
     store!(Tangent3iEXT);
     store!(Tangent3ivEXT);
     store!(Tangent3sEXT);
     store!(Tangent3svEXT);
     store!(TangentPointerEXT);
     store!(TbufferMask3DFX);
     store!(TessellationFactorAMD);
     store!(TessellationModeAMD);
     store!(TestFenceAPPLE);
     store!(TestFenceNV);
     store!(TestObjectAPPLE);
     store!(TexAttachMemoryNV);
     store!(TexBuffer);
     store!(TexBufferARB);
     store!(TexBufferEXT);
     store!(TexBufferRange);
     store!(TexBumpParameterfvATI);
     store!(TexBumpParameterivATI);
     store!(TexCoord1bOES);
     store!(TexCoord1bvOES);
     store!(TexCoord1hNV);
     store!(TexCoord1hvNV);
     store!(TexCoord1xOES);
     store!(TexCoord1xvOES);
     store!(TexCoord2bOES);
     store!(TexCoord2bvOES);
     store!(TexCoord2fColor3fVertex3fSUN);
     store!(TexCoord2fColor3fVertex3fvSUN);
     store!(TexCoord2fColor4fNormal3fVertex3fSUN);
     store!(TexCoord2fColor4fNormal3fVertex3fvSUN);
     store!(TexCoord2fColor4ubVertex3fSUN);
     store!(TexCoord2fColor4ubVertex3fvSUN);
     store!(TexCoord2fNormal3fVertex3fSUN);
     store!(TexCoord2fNormal3fVertex3fvSUN);
     store!(TexCoord2fVertex3fSUN);
     store!(TexCoord2fVertex3fvSUN);
     store!(TexCoord2hNV);
     store!(TexCoord2hvNV);
     store!(TexCoord2xOES);
     store!(TexCoord2xvOES);
     store!(TexCoord3bOES);
     store!(TexCoord3bvOES);
     store!(TexCoord3hNV);
     store!(TexCoord3hvNV);
     store!(TexCoord3xOES);
     store!(TexCoord3xvOES);
     store!(TexCoord4bOES);
     store!(TexCoord4bvOES);
     store!(TexCoord4fColor4fNormal3fVertex4fSUN);
     store!(TexCoord4fColor4fNormal3fVertex4fvSUN);
     store!(TexCoord4fVertex4fSUN);
     store!(TexCoord4fVertex4fvSUN);
     store!(TexCoord4hNV);
     store!(TexCoord4hvNV);
     store!(TexCoord4xOES);
     store!(TexCoord4xvOES);
     store!(TexCoordFormatNV);
     store!(TexCoordPointerEXT);
     store!(TexCoordPointerListIBM);
     store!(TexCoordPointervINTEL);
     store!(TexEnvxOES);
     store!(TexEnvxvOES);
     store!(TexFilterFuncSGIS);
     store!(TexGenxOES);
     store!(TexGenxvOES);
     store!(TexImage1D);
     store!(TexImage2D);
     store!(TexImage2DMultisample);
     store!(TexImage2DMultisampleCoverageNV);
     store!(TexImage3D);
     store!(TexImage3DEXT);
     store!(TexImage3DMultisample);
     store!(TexImage3DMultisampleCoverageNV);
     store!(TexImage4DSGIS);
     store!(TexPageCommitmentARB);
     store!(TexPageCommitmentMemNV);
     store!(TexParameterIiv);
     store!(TexParameterIivEXT);
     store!(TexParameterIuiv);
     store!(TexParameterIuivEXT);
     store!(TexParameterf);
     store!(TexParameterfv);
     store!(TexParameteri);
     store!(TexParameteriv);
     store!(TexParameterxOES);
     store!(TexParameterxvOES);
     store!(TexRenderbufferNV);
     store!(TexStorage1D);
     store!(TexStorage1DEXT);
     store!(TexStorage2D);
     store!(TexStorage2DEXT);
     store!(TexStorage2DMultisample);
     store!(TexStorage3D);
     store!(TexStorage3DEXT);
     store!(TexStorage3DMultisample);
     store!(TexStorageMem1DEXT);
     store!(TexStorageMem2DEXT);
     store!(TexStorageMem2DMultisampleEXT);
     store!(TexStorageMem3DEXT);
     store!(TexStorageMem3DMultisampleEXT);
     store!(TexStorageSparseAMD);
     store!(TexSubImage1D);
     store!(TexSubImage1DEXT);
     store!(TexSubImage2D);
     store!(TexSubImage2DEXT);
     store!(TexSubImage3D);
     store!(TexSubImage3DEXT);
     store!(TexSubImage4DSGIS);
     store!(TextureAttachMemoryNV);
     store!(TextureBarrier);
     store!(TextureBarrierNV);
     store!(TextureBuffer);
     store!(TextureBufferEXT);
     store!(TextureBufferRange);
     store!(TextureBufferRangeEXT);
     store!(TextureColorMaskSGIS);
     store!(TextureImage1DEXT);
     store!(TextureImage2DEXT);
     store!(TextureImage2DMultisampleCoverageNV);
     store!(TextureImage2DMultisampleNV);
     store!(TextureImage3DEXT);
     store!(TextureImage3DMultisampleCoverageNV);
     store!(TextureImage3DMultisampleNV);
     store!(TextureLightEXT);
     store!(TextureMaterialEXT);
     store!(TextureNormalEXT);
     store!(TexturePageCommitmentEXT);
     store!(TexturePageCommitmentMemNV);
     store!(TextureParameterIiv);
     store!(TextureParameterIivEXT);
     store!(TextureParameterIuiv);
     store!(TextureParameterIuivEXT);
     store!(TextureParameterf);
     store!(TextureParameterfEXT);
     store!(TextureParameterfv);
     store!(TextureParameterfvEXT);
     store!(TextureParameteri);
     store!(TextureParameteriEXT);
     store!(TextureParameteriv);
     store!(TextureParameterivEXT);
     store!(TextureRangeAPPLE);
     store!(TextureRenderbufferEXT);
     store!(TextureStorage1D);
     store!(TextureStorage1DEXT);
     store!(TextureStorage2D);
     store!(TextureStorage2DEXT);
     store!(TextureStorage2DMultisample);
     store!(TextureStorage2DMultisampleEXT);
     store!(TextureStorage3D);
     store!(TextureStorage3DEXT);
     store!(TextureStorage3DMultisample);
     store!(TextureStorage3DMultisampleEXT);
     store!(TextureStorageMem1DEXT);
     store!(TextureStorageMem2DEXT);
     store!(TextureStorageMem2DMultisampleEXT);
     store!(TextureStorageMem3DEXT);
     store!(TextureStorageMem3DMultisampleEXT);
     store!(TextureStorageSparseAMD);
     store!(TextureSubImage1D);
     store!(TextureSubImage1DEXT);
     store!(TextureSubImage2D);
     store!(TextureSubImage2DEXT);
     store!(TextureSubImage3D);
     store!(TextureSubImage3DEXT);
     store!(TextureView);
     store!(TrackMatrixNV);
     store!(TransformFeedbackAttribsNV);
     store!(TransformFeedbackBufferBase);
     store!(TransformFeedbackBufferRange);
     store!(TransformFeedbackStreamAttribsNV);
     store!(TransformFeedbackVaryings);
     store!(TransformFeedbackVaryingsEXT);
     store!(TransformFeedbackVaryingsNV);
     store!(TransformPathNV);
     store!(TranslatexOES);
     store!(Uniform1d);
     store!(Uniform1dv);
     store!(Uniform1f);
     store!(Uniform1fARB);
     store!(Uniform1fv);
     store!(Uniform1fvARB);
     store!(Uniform1i);
     store!(Uniform1i64ARB);
     store!(Uniform1i64NV);
     store!(Uniform1i64vARB);
     store!(Uniform1i64vNV);
     store!(Uniform1iARB);
     store!(Uniform1iv);
     store!(Uniform1ivARB);
     store!(Uniform1ui);
     store!(Uniform1ui64ARB);
     store!(Uniform1ui64NV);
     store!(Uniform1ui64vARB);
     store!(Uniform1ui64vNV);
     store!(Uniform1uiEXT);
     store!(Uniform1uiv);
     store!(Uniform1uivEXT);
     store!(Uniform2d);
     store!(Uniform2dv);
     store!(Uniform2f);
     store!(Uniform2fARB);
     store!(Uniform2fv);
     store!(Uniform2fvARB);
     store!(Uniform2i);
     store!(Uniform2i64ARB);
     store!(Uniform2i64NV);
     store!(Uniform2i64vARB);
     store!(Uniform2i64vNV);
     store!(Uniform2iARB);
     store!(Uniform2iv);
     store!(Uniform2ivARB);
     store!(Uniform2ui);
     store!(Uniform2ui64ARB);
     store!(Uniform2ui64NV);
     store!(Uniform2ui64vARB);
     store!(Uniform2ui64vNV);
     store!(Uniform2uiEXT);
     store!(Uniform2uiv);
     store!(Uniform2uivEXT);
     store!(Uniform3d);
     store!(Uniform3dv);
     store!(Uniform3f);
     store!(Uniform3fARB);
     store!(Uniform3fv);
     store!(Uniform3fvARB);
     store!(Uniform3i);
     store!(Uniform3i64ARB);
     store!(Uniform3i64NV);
     store!(Uniform3i64vARB);
     store!(Uniform3i64vNV);
     store!(Uniform3iARB);
     store!(Uniform3iv);
     store!(Uniform3ivARB);
     store!(Uniform3ui);
     store!(Uniform3ui64ARB);
     store!(Uniform3ui64NV);
     store!(Uniform3ui64vARB);
     store!(Uniform3ui64vNV);
     store!(Uniform3uiEXT);
     store!(Uniform3uiv);
     store!(Uniform3uivEXT);
     store!(Uniform4d);
     store!(Uniform4dv);
     store!(Uniform4f);
     store!(Uniform4fARB);
     store!(Uniform4fv);
     store!(Uniform4fvARB);
     store!(Uniform4i);
     store!(Uniform4i64ARB);
     store!(Uniform4i64NV);
     store!(Uniform4i64vARB);
     store!(Uniform4i64vNV);
     store!(Uniform4iARB);
     store!(Uniform4iv);
     store!(Uniform4ivARB);
     store!(Uniform4ui);
     store!(Uniform4ui64ARB);
     store!(Uniform4ui64NV);
     store!(Uniform4ui64vARB);
     store!(Uniform4ui64vNV);
     store!(Uniform4uiEXT);
     store!(Uniform4uiv);
     store!(Uniform4uivEXT);
     store!(UniformBlockBinding);
     store!(UniformBufferEXT);
     store!(UniformHandleui64ARB);
     store!(UniformHandleui64NV);
     store!(UniformHandleui64vARB);
     store!(UniformHandleui64vNV);
     store!(UniformMatrix2dv);
     store!(UniformMatrix2fv);
     store!(UniformMatrix2fvARB);
     store!(UniformMatrix2x3dv);
     store!(UniformMatrix2x3fv);
     store!(UniformMatrix2x4dv);
     store!(UniformMatrix2x4fv);
     store!(UniformMatrix3dv);
     store!(UniformMatrix3fv);
     store!(UniformMatrix3fvARB);
     store!(UniformMatrix3x2dv);
     store!(UniformMatrix3x2fv);
     store!(UniformMatrix3x4dv);
     store!(UniformMatrix3x4fv);
     store!(UniformMatrix4dv);
     store!(UniformMatrix4fv);
     store!(UniformMatrix4fvARB);
     store!(UniformMatrix4x2dv);
     store!(UniformMatrix4x2fv);
     store!(UniformMatrix4x3dv);
     store!(UniformMatrix4x3fv);
     store!(UniformSubroutinesuiv);
     store!(Uniformui64NV);
     store!(Uniformui64vNV);
     store!(UnlockArraysEXT);
     store!(UnmapBuffer);
     store!(UnmapBufferARB);
     store!(UnmapNamedBuffer);
     store!(UnmapNamedBufferEXT);
     store!(UnmapObjectBufferATI);
     store!(UnmapTexture2DINTEL);
     store!(UpdateObjectBufferATI);
     store!(UploadGpuMaskNVX);
     store!(UseProgram);
     store!(UseProgramObjectARB);
     store!(UseProgramStages);
     store!(UseShaderProgramEXT);
     store!(VDPAUFiniNV);
     store!(VDPAUGetSurfaceivNV);
     store!(VDPAUInitNV);
     store!(VDPAUIsSurfaceNV);
     store!(VDPAUMapSurfacesNV);
     store!(VDPAURegisterOutputSurfaceNV);
     store!(VDPAURegisterVideoSurfaceNV);
     store!(VDPAURegisterVideoSurfaceWithPictureStructureNV);
     store!(VDPAUSurfaceAccessNV);
     store!(VDPAUUnmapSurfacesNV);
     store!(VDPAUUnregisterSurfaceNV);
     store!(ValidateProgram);
     store!(ValidateProgramARB);
     store!(ValidateProgramPipeline);
     store!(VariantArrayObjectATI);
     store!(VariantPointerEXT);
     store!(VariantbvEXT);
     store!(VariantdvEXT);
     store!(VariantfvEXT);
     store!(VariantivEXT);
     store!(VariantsvEXT);
     store!(VariantubvEXT);
     store!(VariantuivEXT);
     store!(VariantusvEXT);
     store!(Vertex2bOES);
     store!(Vertex2bvOES);
     store!(Vertex2hNV);
     store!(Vertex2hvNV);
     store!(Vertex2xOES);
     store!(Vertex2xvOES);
     store!(Vertex3bOES);
     store!(Vertex3bvOES);
     store!(Vertex3hNV);
     store!(Vertex3hvNV);
     store!(Vertex3xOES);
     store!(Vertex3xvOES);
     store!(Vertex4bOES);
     store!(Vertex4bvOES);
     store!(Vertex4hNV);
     store!(Vertex4hvNV);
     store!(Vertex4xOES);
     store!(Vertex4xvOES);
     store!(VertexArrayAttribBinding);
     store!(VertexArrayAttribFormat);
     store!(VertexArrayAttribIFormat);
     store!(VertexArrayAttribLFormat);
     store!(VertexArrayBindVertexBufferEXT);
     store!(VertexArrayBindingDivisor);
     store!(VertexArrayColorOffsetEXT);
     store!(VertexArrayEdgeFlagOffsetEXT);
     store!(VertexArrayElementBuffer);
     store!(VertexArrayFogCoordOffsetEXT);
     store!(VertexArrayIndexOffsetEXT);
     store!(VertexArrayMultiTexCoordOffsetEXT);
     store!(VertexArrayNormalOffsetEXT);
     store!(VertexArrayParameteriAPPLE);
     store!(VertexArrayRangeAPPLE);
     store!(VertexArrayRangeNV);
     store!(VertexArraySecondaryColorOffsetEXT);
     store!(VertexArrayTexCoordOffsetEXT);
     store!(VertexArrayVertexAttribBindingEXT);
     store!(VertexArrayVertexAttribDivisorEXT);
     store!(VertexArrayVertexAttribFormatEXT);
     store!(VertexArrayVertexAttribIFormatEXT);
     store!(VertexArrayVertexAttribIOffsetEXT);
     store!(VertexArrayVertexAttribLFormatEXT);
     store!(VertexArrayVertexAttribLOffsetEXT);
     store!(VertexArrayVertexAttribOffsetEXT);
     store!(VertexArrayVertexBindingDivisorEXT);
     store!(VertexArrayVertexBuffer);
     store!(VertexArrayVertexBuffers);
     store!(VertexArrayVertexOffsetEXT);
     store!(VertexAttrib1d);
     store!(VertexAttrib1dARB);
     store!(VertexAttrib1dNV);
     store!(VertexAttrib1dv);
     store!(VertexAttrib1dvARB);
     store!(VertexAttrib1dvNV);
     store!(VertexAttrib1f);
     store!(VertexAttrib1fARB);
     store!(VertexAttrib1fNV);
     store!(VertexAttrib1fv);
     store!(VertexAttrib1fvARB);
     store!(VertexAttrib1fvNV);
     store!(VertexAttrib1hNV);
     store!(VertexAttrib1hvNV);
     store!(VertexAttrib1s);
     store!(VertexAttrib1sARB);
     store!(VertexAttrib1sNV);
     store!(VertexAttrib1sv);
     store!(VertexAttrib1svARB);
     store!(VertexAttrib1svNV);
     store!(VertexAttrib2d);
     store!(VertexAttrib2dARB);
     store!(VertexAttrib2dNV);
     store!(VertexAttrib2dv);
     store!(VertexAttrib2dvARB);
     store!(VertexAttrib2dvNV);
     store!(VertexAttrib2f);
     store!(VertexAttrib2fARB);
     store!(VertexAttrib2fNV);
     store!(VertexAttrib2fv);
     store!(VertexAttrib2fvARB);
     store!(VertexAttrib2fvNV);
     store!(VertexAttrib2hNV);
     store!(VertexAttrib2hvNV);
     store!(VertexAttrib2s);
     store!(VertexAttrib2sARB);
     store!(VertexAttrib2sNV);
     store!(VertexAttrib2sv);
     store!(VertexAttrib2svARB);
     store!(VertexAttrib2svNV);
     store!(VertexAttrib3d);
     store!(VertexAttrib3dARB);
     store!(VertexAttrib3dNV);
     store!(VertexAttrib3dv);
     store!(VertexAttrib3dvARB);
     store!(VertexAttrib3dvNV);
     store!(VertexAttrib3f);
     store!(VertexAttrib3fARB);
     store!(VertexAttrib3fNV);
     store!(VertexAttrib3fv);
     store!(VertexAttrib3fvARB);
     store!(VertexAttrib3fvNV);
     store!(VertexAttrib3hNV);
     store!(VertexAttrib3hvNV);
     store!(VertexAttrib3s);
     store!(VertexAttrib3sARB);
     store!(VertexAttrib3sNV);
     store!(VertexAttrib3sv);
     store!(VertexAttrib3svARB);
     store!(VertexAttrib3svNV);
     store!(VertexAttrib4Nbv);
     store!(VertexAttrib4NbvARB);
     store!(VertexAttrib4Niv);
     store!(VertexAttrib4NivARB);
     store!(VertexAttrib4Nsv);
     store!(VertexAttrib4NsvARB);
     store!(VertexAttrib4Nub);
     store!(VertexAttrib4NubARB);
     store!(VertexAttrib4Nubv);
     store!(VertexAttrib4NubvARB);
     store!(VertexAttrib4Nuiv);
     store!(VertexAttrib4NuivARB);
     store!(VertexAttrib4Nusv);
     store!(VertexAttrib4NusvARB);
     store!(VertexAttrib4bv);
     store!(VertexAttrib4bvARB);
     store!(VertexAttrib4d);
     store!(VertexAttrib4dARB);
     store!(VertexAttrib4dNV);
     store!(VertexAttrib4dv);
     store!(VertexAttrib4dvARB);
     store!(VertexAttrib4dvNV);
     store!(VertexAttrib4f);
     store!(VertexAttrib4fARB);
     store!(VertexAttrib4fNV);
     store!(VertexAttrib4fv);
     store!(VertexAttrib4fvARB);
     store!(VertexAttrib4fvNV);
     store!(VertexAttrib4hNV);
     store!(VertexAttrib4hvNV);
     store!(VertexAttrib4iv);
     store!(VertexAttrib4ivARB);
     store!(VertexAttrib4s);
     store!(VertexAttrib4sARB);
     store!(VertexAttrib4sNV);
     store!(VertexAttrib4sv);
     store!(VertexAttrib4svARB);
     store!(VertexAttrib4svNV);
     store!(VertexAttrib4ubNV);
     store!(VertexAttrib4ubv);
     store!(VertexAttrib4ubvARB);
     store!(VertexAttrib4ubvNV);
     store!(VertexAttrib4uiv);
     store!(VertexAttrib4uivARB);
     store!(VertexAttrib4usv);
     store!(VertexAttrib4usvARB);
     store!(VertexAttribArrayObjectATI);
     store!(VertexAttribBinding);
     store!(VertexAttribDivisor);
     store!(VertexAttribDivisorARB);
     store!(VertexAttribFormat);
     store!(VertexAttribFormatNV);
     store!(VertexAttribI1i);
     store!(VertexAttribI1iEXT);
     store!(VertexAttribI1iv);
     store!(VertexAttribI1ivEXT);
     store!(VertexAttribI1ui);
     store!(VertexAttribI1uiEXT);
     store!(VertexAttribI1uiv);
     store!(VertexAttribI1uivEXT);
     store!(VertexAttribI2i);
     store!(VertexAttribI2iEXT);
     store!(VertexAttribI2iv);
     store!(VertexAttribI2ivEXT);
     store!(VertexAttribI2ui);
     store!(VertexAttribI2uiEXT);
     store!(VertexAttribI2uiv);
     store!(VertexAttribI2uivEXT);
     store!(VertexAttribI3i);
     store!(VertexAttribI3iEXT);
     store!(VertexAttribI3iv);
     store!(VertexAttribI3ivEXT);
     store!(VertexAttribI3ui);
     store!(VertexAttribI3uiEXT);
     store!(VertexAttribI3uiv);
     store!(VertexAttribI3uivEXT);
     store!(VertexAttribI4bv);
     store!(VertexAttribI4bvEXT);
     store!(VertexAttribI4i);
     store!(VertexAttribI4iEXT);
     store!(VertexAttribI4iv);
     store!(VertexAttribI4ivEXT);
     store!(VertexAttribI4sv);
     store!(VertexAttribI4svEXT);
     store!(VertexAttribI4ubv);
     store!(VertexAttribI4ubvEXT);
     store!(VertexAttribI4ui);
     store!(VertexAttribI4uiEXT);
     store!(VertexAttribI4uiv);
     store!(VertexAttribI4uivEXT);
     store!(VertexAttribI4usv);
     store!(VertexAttribI4usvEXT);
     store!(VertexAttribIFormat);
     store!(VertexAttribIFormatNV);
     store!(VertexAttribIPointer);
     store!(VertexAttribIPointerEXT);
     store!(VertexAttribL1d);
     store!(VertexAttribL1dEXT);
     store!(VertexAttribL1dv);
     store!(VertexAttribL1dvEXT);
     store!(VertexAttribL1i64NV);
     store!(VertexAttribL1i64vNV);
     store!(VertexAttribL1ui64ARB);
     store!(VertexAttribL1ui64NV);
     store!(VertexAttribL1ui64vARB);
     store!(VertexAttribL1ui64vNV);
     store!(VertexAttribL2d);
     store!(VertexAttribL2dEXT);
     store!(VertexAttribL2dv);
     store!(VertexAttribL2dvEXT);
     store!(VertexAttribL2i64NV);
     store!(VertexAttribL2i64vNV);
     store!(VertexAttribL2ui64NV);
     store!(VertexAttribL2ui64vNV);
     store!(VertexAttribL3d);
     store!(VertexAttribL3dEXT);
     store!(VertexAttribL3dv);
     store!(VertexAttribL3dvEXT);
     store!(VertexAttribL3i64NV);
     store!(VertexAttribL3i64vNV);
     store!(VertexAttribL3ui64NV);
     store!(VertexAttribL3ui64vNV);
     store!(VertexAttribL4d);
     store!(VertexAttribL4dEXT);
     store!(VertexAttribL4dv);
     store!(VertexAttribL4dvEXT);
     store!(VertexAttribL4i64NV);
     store!(VertexAttribL4i64vNV);
     store!(VertexAttribL4ui64NV);
     store!(VertexAttribL4ui64vNV);
     store!(VertexAttribLFormat);
     store!(VertexAttribLFormatNV);
     store!(VertexAttribLPointer);
     store!(VertexAttribLPointerEXT);
     store!(VertexAttribP1ui);
     store!(VertexAttribP1uiv);
     store!(VertexAttribP2ui);
     store!(VertexAttribP2uiv);
     store!(VertexAttribP3ui);
     store!(VertexAttribP3uiv);
     store!(VertexAttribP4ui);
     store!(VertexAttribP4uiv);
     store!(VertexAttribParameteriAMD);
     store!(VertexAttribPointer);
     store!(VertexAttribPointerARB);
     store!(VertexAttribPointerNV);
     store!(VertexAttribs1dvNV);
     store!(VertexAttribs1fvNV);
     store!(VertexAttribs1hvNV);
     store!(VertexAttribs1svNV);
     store!(VertexAttribs2dvNV);
     store!(VertexAttribs2fvNV);
     store!(VertexAttribs2hvNV);
     store!(VertexAttribs2svNV);
     store!(VertexAttribs3dvNV);
     store!(VertexAttribs3fvNV);
     store!(VertexAttribs3hvNV);
     store!(VertexAttribs3svNV);
     store!(VertexAttribs4dvNV);
     store!(VertexAttribs4fvNV);
     store!(VertexAttribs4hvNV);
     store!(VertexAttribs4svNV);
     store!(VertexAttribs4ubvNV);
     store!(VertexBindingDivisor);
     store!(VertexBlendARB);
     store!(VertexBlendEnvfATI);
     store!(VertexBlendEnviATI);
     store!(VertexFormatNV);
     store!(VertexPointerEXT);
     store!(VertexPointerListIBM);
     store!(VertexPointervINTEL);
     store!(VertexStream1dATI);
     store!(VertexStream1dvATI);
     store!(VertexStream1fATI);
     store!(VertexStream1fvATI);
     store!(VertexStream1iATI);
     store!(VertexStream1ivATI);
     store!(VertexStream1sATI);
     store!(VertexStream1svATI);
     store!(VertexStream2dATI);
     store!(VertexStream2dvATI);
     store!(VertexStream2fATI);
     store!(VertexStream2fvATI);
     store!(VertexStream2iATI);
     store!(VertexStream2ivATI);
     store!(VertexStream2sATI);
     store!(VertexStream2svATI);
     store!(VertexStream3dATI);
     store!(VertexStream3dvATI);
     store!(VertexStream3fATI);
     store!(VertexStream3fvATI);
     store!(VertexStream3iATI);
     store!(VertexStream3ivATI);
     store!(VertexStream3sATI);
     store!(VertexStream3svATI);
     store!(VertexStream4dATI);
     store!(VertexStream4dvATI);
     store!(VertexStream4fATI);
     store!(VertexStream4fvATI);
     store!(VertexStream4iATI);
     store!(VertexStream4ivATI);
     store!(VertexStream4sATI);
     store!(VertexStream4svATI);
     store!(VertexWeightPointerEXT);
     store!(VertexWeightfEXT);
     store!(VertexWeightfvEXT);
     store!(VertexWeighthNV);
     store!(VertexWeighthvNV);
     store!(VideoCaptureNV);
     store!(VideoCaptureStreamParameterdvNV);
     store!(VideoCaptureStreamParameterfvNV);
     store!(VideoCaptureStreamParameterivNV);
     store!(Viewport);
     store!(ViewportArrayv);
     store!(ViewportIndexedf);
     store!(ViewportIndexedfv);
     store!(ViewportPositionWScaleNV);
     store!(ViewportSwizzleNV);
     store!(WaitSemaphoreEXT);
     store!(WaitSemaphoreui64NVX);
     store!(WaitSync);
     store!(WaitVkSemaphoreNV);
     store!(WeightPathsNV);
     store!(WeightPointerARB);
     store!(WeightbvARB);
     store!(WeightdvARB);
     store!(WeightfvARB);
     store!(WeightivARB);
     store!(WeightsvARB);
     store!(WeightubvARB);
     store!(WeightuivARB);
     store!(WeightusvARB);
     store!(WindowPos2dARB);
     store!(WindowPos2dMESA);
     store!(WindowPos2dvARB);
     store!(WindowPos2dvMESA);
     store!(WindowPos2fARB);
     store!(WindowPos2fMESA);
     store!(WindowPos2fvARB);
     store!(WindowPos2fvMESA);
     store!(WindowPos2iARB);
     store!(WindowPos2iMESA);
     store!(WindowPos2ivARB);
     store!(WindowPos2ivMESA);
     store!(WindowPos2sARB);
     store!(WindowPos2sMESA);
     store!(WindowPos2svARB);
     store!(WindowPos2svMESA);
     store!(WindowPos3dARB);
     store!(WindowPos3dMESA);
     store!(WindowPos3dvARB);
     store!(WindowPos3dvMESA);
     store!(WindowPos3fARB);
     store!(WindowPos3fMESA);
     store!(WindowPos3fvARB);
     store!(WindowPos3fvMESA);
     store!(WindowPos3iARB);
     store!(WindowPos3iMESA);
     store!(WindowPos3ivARB);
     store!(WindowPos3ivMESA);
     store!(WindowPos3sARB);
     store!(WindowPos3sMESA);
     store!(WindowPos3svARB);
     store!(WindowPos3svMESA);
     store!(WindowPos4dMESA);
     store!(WindowPos4dvMESA);
     store!(WindowPos4fMESA);
     store!(WindowPos4fvMESA);
     store!(WindowPos4iMESA);
     store!(WindowPos4ivMESA);
     store!(WindowPos4sMESA);
     store!(WindowPos4svMESA);
     store!(WindowRectanglesEXT);
     store!(WriteMaskEXT);
}

pub fn load<F>(mut loadfn: F) where F: FnMut(&'static str) -> *const c_void {
    unsafe {
         storage::AccumxOES.set_ptr(loadfn("glAccumxOES"));
         storage::AcquireKeyedMutexWin32EXT.set_ptr(loadfn("glAcquireKeyedMutexWin32EXT"));
         storage::ActiveProgramEXT.set_ptr(loadfn("glActiveProgramEXT"));
         storage::ActiveShaderProgram.set_ptr(loadfn("glActiveShaderProgram"));
         storage::ActiveStencilFaceEXT.set_ptr(loadfn("glActiveStencilFaceEXT"));
         storage::ActiveTexture.set_ptr(loadfn("glActiveTexture"));
         storage::ActiveTextureARB.set_ptr(loadfn("glActiveTextureARB"));
         storage::ActiveVaryingNV.set_ptr(loadfn("glActiveVaryingNV"));
         storage::AlphaFragmentOp1ATI.set_ptr(loadfn("glAlphaFragmentOp1ATI"));
         storage::AlphaFragmentOp2ATI.set_ptr(loadfn("glAlphaFragmentOp2ATI"));
         storage::AlphaFragmentOp3ATI.set_ptr(loadfn("glAlphaFragmentOp3ATI"));
         storage::AlphaFuncxOES.set_ptr(loadfn("glAlphaFuncxOES"));
         storage::AlphaToCoverageDitherControlNV.set_ptr(loadfn("glAlphaToCoverageDitherControlNV"));
         storage::ApplyFramebufferAttachmentCMAAINTEL.set_ptr(loadfn("glApplyFramebufferAttachmentCMAAINTEL"));
         storage::ApplyTextureEXT.set_ptr(loadfn("glApplyTextureEXT"));
         storage::AreProgramsResidentNV.set_ptr(loadfn("glAreProgramsResidentNV"));
         storage::AreTexturesResidentEXT.set_ptr(loadfn("glAreTexturesResidentEXT"));
         storage::ArrayElementEXT.set_ptr(loadfn("glArrayElementEXT"));
         storage::ArrayObjectATI.set_ptr(loadfn("glArrayObjectATI"));
         storage::AsyncCopyBufferSubDataNVX.set_ptr(loadfn("glAsyncCopyBufferSubDataNVX"));
         storage::AsyncCopyImageSubDataNVX.set_ptr(loadfn("glAsyncCopyImageSubDataNVX"));
         storage::AsyncMarkerSGIX.set_ptr(loadfn("glAsyncMarkerSGIX"));
         storage::AttachObjectARB.set_ptr(loadfn("glAttachObjectARB"));
         storage::AttachShader.set_ptr(loadfn("glAttachShader"));
         storage::BeginConditionalRender.set_ptr(loadfn("glBeginConditionalRender"));
         storage::BeginConditionalRenderNV.set_ptr(loadfn("glBeginConditionalRenderNV"));
         storage::BeginConditionalRenderNVX.set_ptr(loadfn("glBeginConditionalRenderNVX"));
         storage::BeginFragmentShaderATI.set_ptr(loadfn("glBeginFragmentShaderATI"));
         storage::BeginOcclusionQueryNV.set_ptr(loadfn("glBeginOcclusionQueryNV"));
         storage::BeginPerfMonitorAMD.set_ptr(loadfn("glBeginPerfMonitorAMD"));
         storage::BeginPerfQueryINTEL.set_ptr(loadfn("glBeginPerfQueryINTEL"));
         storage::BeginQuery.set_ptr(loadfn("glBeginQuery"));
         storage::BeginQueryARB.set_ptr(loadfn("glBeginQueryARB"));
         storage::BeginQueryIndexed.set_ptr(loadfn("glBeginQueryIndexed"));
         storage::BeginTransformFeedback.set_ptr(loadfn("glBeginTransformFeedback"));
         storage::BeginTransformFeedbackEXT.set_ptr(loadfn("glBeginTransformFeedbackEXT"));
         storage::BeginTransformFeedbackNV.set_ptr(loadfn("glBeginTransformFeedbackNV"));
         storage::BeginVertexShaderEXT.set_ptr(loadfn("glBeginVertexShaderEXT"));
         storage::BeginVideoCaptureNV.set_ptr(loadfn("glBeginVideoCaptureNV"));
         storage::BindAttribLocation.set_ptr(loadfn("glBindAttribLocation"));
         storage::BindAttribLocationARB.set_ptr(loadfn("glBindAttribLocationARB"));
         storage::BindBuffer.set_ptr(loadfn("glBindBuffer"));
         storage::BindBufferARB.set_ptr(loadfn("glBindBufferARB"));
         storage::BindBufferBase.set_ptr(loadfn("glBindBufferBase"));
         storage::BindBufferBaseEXT.set_ptr(loadfn("glBindBufferBaseEXT"));
         storage::BindBufferBaseNV.set_ptr(loadfn("glBindBufferBaseNV"));
         storage::BindBufferOffsetEXT.set_ptr(loadfn("glBindBufferOffsetEXT"));
         storage::BindBufferOffsetNV.set_ptr(loadfn("glBindBufferOffsetNV"));
         storage::BindBufferRange.set_ptr(loadfn("glBindBufferRange"));
         storage::BindBufferRangeEXT.set_ptr(loadfn("glBindBufferRangeEXT"));
         storage::BindBufferRangeNV.set_ptr(loadfn("glBindBufferRangeNV"));
         storage::BindBuffersBase.set_ptr(loadfn("glBindBuffersBase"));
         storage::BindBuffersRange.set_ptr(loadfn("glBindBuffersRange"));
         storage::BindFragDataLocation.set_ptr(loadfn("glBindFragDataLocation"));
         storage::BindFragDataLocationEXT.set_ptr(loadfn("glBindFragDataLocationEXT"));
         storage::BindFragDataLocationIndexed.set_ptr(loadfn("glBindFragDataLocationIndexed"));
         storage::BindFragmentShaderATI.set_ptr(loadfn("glBindFragmentShaderATI"));
         storage::BindFramebuffer.set_ptr(loadfn("glBindFramebuffer"));
         storage::BindFramebufferEXT.set_ptr(loadfn("glBindFramebufferEXT"));
         storage::BindImageTexture.set_ptr(loadfn("glBindImageTexture"));
         storage::BindImageTextureEXT.set_ptr(loadfn("glBindImageTextureEXT"));
         storage::BindImageTextures.set_ptr(loadfn("glBindImageTextures"));
         storage::BindLightParameterEXT.set_ptr(loadfn("glBindLightParameterEXT"));
         storage::BindMaterialParameterEXT.set_ptr(loadfn("glBindMaterialParameterEXT"));
         storage::BindMultiTextureEXT.set_ptr(loadfn("glBindMultiTextureEXT"));
         storage::BindParameterEXT.set_ptr(loadfn("glBindParameterEXT"));
         storage::BindProgramARB.set_ptr(loadfn("glBindProgramARB"));
         storage::BindProgramNV.set_ptr(loadfn("glBindProgramNV"));
         storage::BindProgramPipeline.set_ptr(loadfn("glBindProgramPipeline"));
         storage::BindRenderbuffer.set_ptr(loadfn("glBindRenderbuffer"));
         storage::BindRenderbufferEXT.set_ptr(loadfn("glBindRenderbufferEXT"));
         storage::BindSampler.set_ptr(loadfn("glBindSampler"));
         storage::BindSamplers.set_ptr(loadfn("glBindSamplers"));
         storage::BindShadingRateImageNV.set_ptr(loadfn("glBindShadingRateImageNV"));
         storage::BindTexGenParameterEXT.set_ptr(loadfn("glBindTexGenParameterEXT"));
         storage::BindTexture.set_ptr(loadfn("glBindTexture"));
         storage::BindTextureEXT.set_ptr(loadfn("glBindTextureEXT"));
         storage::BindTextureUnit.set_ptr(loadfn("glBindTextureUnit"));
         storage::BindTextureUnitParameterEXT.set_ptr(loadfn("glBindTextureUnitParameterEXT"));
         storage::BindTextures.set_ptr(loadfn("glBindTextures"));
         storage::BindTransformFeedback.set_ptr(loadfn("glBindTransformFeedback"));
         storage::BindTransformFeedbackNV.set_ptr(loadfn("glBindTransformFeedbackNV"));
         storage::BindVertexArray.set_ptr(loadfn("glBindVertexArray"));
         storage::BindVertexArrayAPPLE.set_ptr(loadfn("glBindVertexArrayAPPLE"));
         storage::BindVertexBuffer.set_ptr(loadfn("glBindVertexBuffer"));
         storage::BindVertexBuffers.set_ptr(loadfn("glBindVertexBuffers"));
         storage::BindVertexShaderEXT.set_ptr(loadfn("glBindVertexShaderEXT"));
         storage::BindVideoCaptureStreamBufferNV.set_ptr(loadfn("glBindVideoCaptureStreamBufferNV"));
         storage::BindVideoCaptureStreamTextureNV.set_ptr(loadfn("glBindVideoCaptureStreamTextureNV"));
         storage::Binormal3bEXT.set_ptr(loadfn("glBinormal3bEXT"));
         storage::Binormal3bvEXT.set_ptr(loadfn("glBinormal3bvEXT"));
         storage::Binormal3dEXT.set_ptr(loadfn("glBinormal3dEXT"));
         storage::Binormal3dvEXT.set_ptr(loadfn("glBinormal3dvEXT"));
         storage::Binormal3fEXT.set_ptr(loadfn("glBinormal3fEXT"));
         storage::Binormal3fvEXT.set_ptr(loadfn("glBinormal3fvEXT"));
         storage::Binormal3iEXT.set_ptr(loadfn("glBinormal3iEXT"));
         storage::Binormal3ivEXT.set_ptr(loadfn("glBinormal3ivEXT"));
         storage::Binormal3sEXT.set_ptr(loadfn("glBinormal3sEXT"));
         storage::Binormal3svEXT.set_ptr(loadfn("glBinormal3svEXT"));
         storage::BinormalPointerEXT.set_ptr(loadfn("glBinormalPointerEXT"));
         storage::BitmapxOES.set_ptr(loadfn("glBitmapxOES"));
         storage::BlendBarrierKHR.set_ptr(loadfn("glBlendBarrierKHR"));
         storage::BlendBarrierNV.set_ptr(loadfn("glBlendBarrierNV"));
         storage::BlendColor.set_ptr(loadfn("glBlendColor"));
         storage::BlendColorEXT.set_ptr(loadfn("glBlendColorEXT"));
         storage::BlendColorxOES.set_ptr(loadfn("glBlendColorxOES"));
         storage::BlendEquation.set_ptr(loadfn("glBlendEquation"));
         storage::BlendEquationEXT.set_ptr(loadfn("glBlendEquationEXT"));
         storage::BlendEquationIndexedAMD.set_ptr(loadfn("glBlendEquationIndexedAMD"));
         storage::BlendEquationSeparate.set_ptr(loadfn("glBlendEquationSeparate"));
         storage::BlendEquationSeparateEXT.set_ptr(loadfn("glBlendEquationSeparateEXT"));
         storage::BlendEquationSeparateIndexedAMD.set_ptr(loadfn("glBlendEquationSeparateIndexedAMD"));
         storage::BlendEquationSeparatei.set_ptr(loadfn("glBlendEquationSeparatei"));
         storage::BlendEquationSeparateiARB.set_ptr(loadfn("glBlendEquationSeparateiARB"));
         storage::BlendEquationi.set_ptr(loadfn("glBlendEquationi"));
         storage::BlendEquationiARB.set_ptr(loadfn("glBlendEquationiARB"));
         storage::BlendFunc.set_ptr(loadfn("glBlendFunc"));
         storage::BlendFuncIndexedAMD.set_ptr(loadfn("glBlendFuncIndexedAMD"));
         storage::BlendFuncSeparate.set_ptr(loadfn("glBlendFuncSeparate"));
         storage::BlendFuncSeparateEXT.set_ptr(loadfn("glBlendFuncSeparateEXT"));
         storage::BlendFuncSeparateINGR.set_ptr(loadfn("glBlendFuncSeparateINGR"));
         storage::BlendFuncSeparateIndexedAMD.set_ptr(loadfn("glBlendFuncSeparateIndexedAMD"));
         storage::BlendFuncSeparatei.set_ptr(loadfn("glBlendFuncSeparatei"));
         storage::BlendFuncSeparateiARB.set_ptr(loadfn("glBlendFuncSeparateiARB"));
         storage::BlendFunci.set_ptr(loadfn("glBlendFunci"));
         storage::BlendFunciARB.set_ptr(loadfn("glBlendFunciARB"));
         storage::BlendParameteriNV.set_ptr(loadfn("glBlendParameteriNV"));
         storage::BlitFramebuffer.set_ptr(loadfn("glBlitFramebuffer"));
         storage::BlitFramebufferEXT.set_ptr(loadfn("glBlitFramebufferEXT"));
         storage::BlitNamedFramebuffer.set_ptr(loadfn("glBlitNamedFramebuffer"));
         storage::BufferAddressRangeNV.set_ptr(loadfn("glBufferAddressRangeNV"));
         storage::BufferAttachMemoryNV.set_ptr(loadfn("glBufferAttachMemoryNV"));
         storage::BufferData.set_ptr(loadfn("glBufferData"));
         storage::BufferDataARB.set_ptr(loadfn("glBufferDataARB"));
         storage::BufferPageCommitmentARB.set_ptr(loadfn("glBufferPageCommitmentARB"));
         storage::BufferPageCommitmentMemNV.set_ptr(loadfn("glBufferPageCommitmentMemNV"));
         storage::BufferParameteriAPPLE.set_ptr(loadfn("glBufferParameteriAPPLE"));
         storage::BufferStorage.set_ptr(loadfn("glBufferStorage"));
         storage::BufferStorageExternalEXT.set_ptr(loadfn("glBufferStorageExternalEXT"));
         storage::BufferStorageMemEXT.set_ptr(loadfn("glBufferStorageMemEXT"));
         storage::BufferSubData.set_ptr(loadfn("glBufferSubData"));
         storage::BufferSubDataARB.set_ptr(loadfn("glBufferSubDataARB"));
         storage::CallCommandListNV.set_ptr(loadfn("glCallCommandListNV"));
         storage::CheckFramebufferStatus.set_ptr(loadfn("glCheckFramebufferStatus"));
         storage::CheckFramebufferStatusEXT.set_ptr(loadfn("glCheckFramebufferStatusEXT"));
         storage::CheckNamedFramebufferStatus.set_ptr(loadfn("glCheckNamedFramebufferStatus"));
         storage::CheckNamedFramebufferStatusEXT.set_ptr(loadfn("glCheckNamedFramebufferStatusEXT"));
         storage::ClampColor.set_ptr(loadfn("glClampColor"));
         storage::ClampColorARB.set_ptr(loadfn("glClampColorARB"));
         storage::Clear.set_ptr(loadfn("glClear"));
         storage::ClearAccumxOES.set_ptr(loadfn("glClearAccumxOES"));
         storage::ClearBufferData.set_ptr(loadfn("glClearBufferData"));
         storage::ClearBufferSubData.set_ptr(loadfn("glClearBufferSubData"));
         storage::ClearBufferfi.set_ptr(loadfn("glClearBufferfi"));
         storage::ClearBufferfv.set_ptr(loadfn("glClearBufferfv"));
         storage::ClearBufferiv.set_ptr(loadfn("glClearBufferiv"));
         storage::ClearBufferuiv.set_ptr(loadfn("glClearBufferuiv"));
         storage::ClearColor.set_ptr(loadfn("glClearColor"));
         storage::ClearColorIiEXT.set_ptr(loadfn("glClearColorIiEXT"));
         storage::ClearColorIuiEXT.set_ptr(loadfn("glClearColorIuiEXT"));
         storage::ClearColorxOES.set_ptr(loadfn("glClearColorxOES"));
         storage::ClearDepth.set_ptr(loadfn("glClearDepth"));
         storage::ClearDepthdNV.set_ptr(loadfn("glClearDepthdNV"));
         storage::ClearDepthf.set_ptr(loadfn("glClearDepthf"));
         storage::ClearDepthfOES.set_ptr(loadfn("glClearDepthfOES"));
         storage::ClearDepthxOES.set_ptr(loadfn("glClearDepthxOES"));
         storage::ClearNamedBufferData.set_ptr(loadfn("glClearNamedBufferData"));
         storage::ClearNamedBufferDataEXT.set_ptr(loadfn("glClearNamedBufferDataEXT"));
         storage::ClearNamedBufferSubData.set_ptr(loadfn("glClearNamedBufferSubData"));
         storage::ClearNamedBufferSubDataEXT.set_ptr(loadfn("glClearNamedBufferSubDataEXT"));
         storage::ClearNamedFramebufferfi.set_ptr(loadfn("glClearNamedFramebufferfi"));
         storage::ClearNamedFramebufferfv.set_ptr(loadfn("glClearNamedFramebufferfv"));
         storage::ClearNamedFramebufferiv.set_ptr(loadfn("glClearNamedFramebufferiv"));
         storage::ClearNamedFramebufferuiv.set_ptr(loadfn("glClearNamedFramebufferuiv"));
         storage::ClearStencil.set_ptr(loadfn("glClearStencil"));
         storage::ClearTexImage.set_ptr(loadfn("glClearTexImage"));
         storage::ClearTexSubImage.set_ptr(loadfn("glClearTexSubImage"));
         storage::ClientActiveTextureARB.set_ptr(loadfn("glClientActiveTextureARB"));
         storage::ClientActiveVertexStreamATI.set_ptr(loadfn("glClientActiveVertexStreamATI"));
         storage::ClientAttribDefaultEXT.set_ptr(loadfn("glClientAttribDefaultEXT"));
         storage::ClientWaitSemaphoreui64NVX.set_ptr(loadfn("glClientWaitSemaphoreui64NVX"));
         storage::ClientWaitSync.set_ptr(loadfn("glClientWaitSync"));
         storage::ClipControl.set_ptr(loadfn("glClipControl"));
         storage::ClipPlanefOES.set_ptr(loadfn("glClipPlanefOES"));
         storage::ClipPlanexOES.set_ptr(loadfn("glClipPlanexOES"));
         storage::Color3fVertex3fSUN.set_ptr(loadfn("glColor3fVertex3fSUN"));
         storage::Color3fVertex3fvSUN.set_ptr(loadfn("glColor3fVertex3fvSUN"));
         storage::Color3hNV.set_ptr(loadfn("glColor3hNV"));
         storage::Color3hvNV.set_ptr(loadfn("glColor3hvNV"));
         storage::Color3xOES.set_ptr(loadfn("glColor3xOES"));
         storage::Color3xvOES.set_ptr(loadfn("glColor3xvOES"));
         storage::Color4fNormal3fVertex3fSUN.set_ptr(loadfn("glColor4fNormal3fVertex3fSUN"));
         storage::Color4fNormal3fVertex3fvSUN.set_ptr(loadfn("glColor4fNormal3fVertex3fvSUN"));
         storage::Color4hNV.set_ptr(loadfn("glColor4hNV"));
         storage::Color4hvNV.set_ptr(loadfn("glColor4hvNV"));
         storage::Color4ubVertex2fSUN.set_ptr(loadfn("glColor4ubVertex2fSUN"));
         storage::Color4ubVertex2fvSUN.set_ptr(loadfn("glColor4ubVertex2fvSUN"));
         storage::Color4ubVertex3fSUN.set_ptr(loadfn("glColor4ubVertex3fSUN"));
         storage::Color4ubVertex3fvSUN.set_ptr(loadfn("glColor4ubVertex3fvSUN"));
         storage::Color4xOES.set_ptr(loadfn("glColor4xOES"));
         storage::Color4xvOES.set_ptr(loadfn("glColor4xvOES"));
         storage::ColorFormatNV.set_ptr(loadfn("glColorFormatNV"));
         storage::ColorFragmentOp1ATI.set_ptr(loadfn("glColorFragmentOp1ATI"));
         storage::ColorFragmentOp2ATI.set_ptr(loadfn("glColorFragmentOp2ATI"));
         storage::ColorFragmentOp3ATI.set_ptr(loadfn("glColorFragmentOp3ATI"));
         storage::ColorMask.set_ptr(loadfn("glColorMask"));
         storage::ColorMaskIndexedEXT.set_ptr(loadfn("glColorMaskIndexedEXT"));
         storage::ColorMaski.set_ptr(loadfn("glColorMaski"));
         storage::ColorPointerEXT.set_ptr(loadfn("glColorPointerEXT"));
         storage::ColorPointerListIBM.set_ptr(loadfn("glColorPointerListIBM"));
         storage::ColorPointervINTEL.set_ptr(loadfn("glColorPointervINTEL"));
         storage::ColorSubTableEXT.set_ptr(loadfn("glColorSubTableEXT"));
         storage::ColorTableEXT.set_ptr(loadfn("glColorTableEXT"));
         storage::ColorTableParameterfvSGI.set_ptr(loadfn("glColorTableParameterfvSGI"));
         storage::ColorTableParameterivSGI.set_ptr(loadfn("glColorTableParameterivSGI"));
         storage::ColorTableSGI.set_ptr(loadfn("glColorTableSGI"));
         storage::CombinerInputNV.set_ptr(loadfn("glCombinerInputNV"));
         storage::CombinerOutputNV.set_ptr(loadfn("glCombinerOutputNV"));
         storage::CombinerParameterfNV.set_ptr(loadfn("glCombinerParameterfNV"));
         storage::CombinerParameterfvNV.set_ptr(loadfn("glCombinerParameterfvNV"));
         storage::CombinerParameteriNV.set_ptr(loadfn("glCombinerParameteriNV"));
         storage::CombinerParameterivNV.set_ptr(loadfn("glCombinerParameterivNV"));
         storage::CombinerStageParameterfvNV.set_ptr(loadfn("glCombinerStageParameterfvNV"));
         storage::CommandListSegmentsNV.set_ptr(loadfn("glCommandListSegmentsNV"));
         storage::CompileCommandListNV.set_ptr(loadfn("glCompileCommandListNV"));
         storage::CompileShader.set_ptr(loadfn("glCompileShader"));
         storage::CompileShaderARB.set_ptr(loadfn("glCompileShaderARB"));
         storage::CompileShaderIncludeARB.set_ptr(loadfn("glCompileShaderIncludeARB"));
         storage::CompressedMultiTexImage1DEXT.set_ptr(loadfn("glCompressedMultiTexImage1DEXT"));
         storage::CompressedMultiTexImage2DEXT.set_ptr(loadfn("glCompressedMultiTexImage2DEXT"));
         storage::CompressedMultiTexImage3DEXT.set_ptr(loadfn("glCompressedMultiTexImage3DEXT"));
         storage::CompressedMultiTexSubImage1DEXT.set_ptr(loadfn("glCompressedMultiTexSubImage1DEXT"));
         storage::CompressedMultiTexSubImage2DEXT.set_ptr(loadfn("glCompressedMultiTexSubImage2DEXT"));
         storage::CompressedMultiTexSubImage3DEXT.set_ptr(loadfn("glCompressedMultiTexSubImage3DEXT"));
         storage::CompressedTexImage1D.set_ptr(loadfn("glCompressedTexImage1D"));
         storage::CompressedTexImage1DARB.set_ptr(loadfn("glCompressedTexImage1DARB"));
         storage::CompressedTexImage2D.set_ptr(loadfn("glCompressedTexImage2D"));
         storage::CompressedTexImage2DARB.set_ptr(loadfn("glCompressedTexImage2DARB"));
         storage::CompressedTexImage3D.set_ptr(loadfn("glCompressedTexImage3D"));
         storage::CompressedTexImage3DARB.set_ptr(loadfn("glCompressedTexImage3DARB"));
         storage::CompressedTexSubImage1D.set_ptr(loadfn("glCompressedTexSubImage1D"));
         storage::CompressedTexSubImage1DARB.set_ptr(loadfn("glCompressedTexSubImage1DARB"));
         storage::CompressedTexSubImage2D.set_ptr(loadfn("glCompressedTexSubImage2D"));
         storage::CompressedTexSubImage2DARB.set_ptr(loadfn("glCompressedTexSubImage2DARB"));
         storage::CompressedTexSubImage3D.set_ptr(loadfn("glCompressedTexSubImage3D"));
         storage::CompressedTexSubImage3DARB.set_ptr(loadfn("glCompressedTexSubImage3DARB"));
         storage::CompressedTextureImage1DEXT.set_ptr(loadfn("glCompressedTextureImage1DEXT"));
         storage::CompressedTextureImage2DEXT.set_ptr(loadfn("glCompressedTextureImage2DEXT"));
         storage::CompressedTextureImage3DEXT.set_ptr(loadfn("glCompressedTextureImage3DEXT"));
         storage::CompressedTextureSubImage1D.set_ptr(loadfn("glCompressedTextureSubImage1D"));
         storage::CompressedTextureSubImage1DEXT.set_ptr(loadfn("glCompressedTextureSubImage1DEXT"));
         storage::CompressedTextureSubImage2D.set_ptr(loadfn("glCompressedTextureSubImage2D"));
         storage::CompressedTextureSubImage2DEXT.set_ptr(loadfn("glCompressedTextureSubImage2DEXT"));
         storage::CompressedTextureSubImage3D.set_ptr(loadfn("glCompressedTextureSubImage3D"));
         storage::CompressedTextureSubImage3DEXT.set_ptr(loadfn("glCompressedTextureSubImage3DEXT"));
         storage::ConservativeRasterParameterfNV.set_ptr(loadfn("glConservativeRasterParameterfNV"));
         storage::ConservativeRasterParameteriNV.set_ptr(loadfn("glConservativeRasterParameteriNV"));
         storage::ConvolutionFilter1DEXT.set_ptr(loadfn("glConvolutionFilter1DEXT"));
         storage::ConvolutionFilter2DEXT.set_ptr(loadfn("glConvolutionFilter2DEXT"));
         storage::ConvolutionParameterfEXT.set_ptr(loadfn("glConvolutionParameterfEXT"));
         storage::ConvolutionParameterfvEXT.set_ptr(loadfn("glConvolutionParameterfvEXT"));
         storage::ConvolutionParameteriEXT.set_ptr(loadfn("glConvolutionParameteriEXT"));
         storage::ConvolutionParameterivEXT.set_ptr(loadfn("glConvolutionParameterivEXT"));
         storage::ConvolutionParameterxOES.set_ptr(loadfn("glConvolutionParameterxOES"));
         storage::ConvolutionParameterxvOES.set_ptr(loadfn("glConvolutionParameterxvOES"));
         storage::CopyBufferSubData.set_ptr(loadfn("glCopyBufferSubData"));
         storage::CopyColorSubTableEXT.set_ptr(loadfn("glCopyColorSubTableEXT"));
         storage::CopyColorTableSGI.set_ptr(loadfn("glCopyColorTableSGI"));
         storage::CopyConvolutionFilter1DEXT.set_ptr(loadfn("glCopyConvolutionFilter1DEXT"));
         storage::CopyConvolutionFilter2DEXT.set_ptr(loadfn("glCopyConvolutionFilter2DEXT"));
         storage::CopyImageSubData.set_ptr(loadfn("glCopyImageSubData"));
         storage::CopyImageSubDataNV.set_ptr(loadfn("glCopyImageSubDataNV"));
         storage::CopyMultiTexImage1DEXT.set_ptr(loadfn("glCopyMultiTexImage1DEXT"));
         storage::CopyMultiTexImage2DEXT.set_ptr(loadfn("glCopyMultiTexImage2DEXT"));
         storage::CopyMultiTexSubImage1DEXT.set_ptr(loadfn("glCopyMultiTexSubImage1DEXT"));
         storage::CopyMultiTexSubImage2DEXT.set_ptr(loadfn("glCopyMultiTexSubImage2DEXT"));
         storage::CopyMultiTexSubImage3DEXT.set_ptr(loadfn("glCopyMultiTexSubImage3DEXT"));
         storage::CopyNamedBufferSubData.set_ptr(loadfn("glCopyNamedBufferSubData"));
         storage::CopyPathNV.set_ptr(loadfn("glCopyPathNV"));
         storage::CopyTexImage1D.set_ptr(loadfn("glCopyTexImage1D"));
         storage::CopyTexImage1DEXT.set_ptr(loadfn("glCopyTexImage1DEXT"));
         storage::CopyTexImage2D.set_ptr(loadfn("glCopyTexImage2D"));
         storage::CopyTexImage2DEXT.set_ptr(loadfn("glCopyTexImage2DEXT"));
         storage::CopyTexSubImage1D.set_ptr(loadfn("glCopyTexSubImage1D"));
         storage::CopyTexSubImage1DEXT.set_ptr(loadfn("glCopyTexSubImage1DEXT"));
         storage::CopyTexSubImage2D.set_ptr(loadfn("glCopyTexSubImage2D"));
         storage::CopyTexSubImage2DEXT.set_ptr(loadfn("glCopyTexSubImage2DEXT"));
         storage::CopyTexSubImage3D.set_ptr(loadfn("glCopyTexSubImage3D"));
         storage::CopyTexSubImage3DEXT.set_ptr(loadfn("glCopyTexSubImage3DEXT"));
         storage::CopyTextureImage1DEXT.set_ptr(loadfn("glCopyTextureImage1DEXT"));
         storage::CopyTextureImage2DEXT.set_ptr(loadfn("glCopyTextureImage2DEXT"));
         storage::CopyTextureSubImage1D.set_ptr(loadfn("glCopyTextureSubImage1D"));
         storage::CopyTextureSubImage1DEXT.set_ptr(loadfn("glCopyTextureSubImage1DEXT"));
         storage::CopyTextureSubImage2D.set_ptr(loadfn("glCopyTextureSubImage2D"));
         storage::CopyTextureSubImage2DEXT.set_ptr(loadfn("glCopyTextureSubImage2DEXT"));
         storage::CopyTextureSubImage3D.set_ptr(loadfn("glCopyTextureSubImage3D"));
         storage::CopyTextureSubImage3DEXT.set_ptr(loadfn("glCopyTextureSubImage3DEXT"));
         storage::CoverFillPathInstancedNV.set_ptr(loadfn("glCoverFillPathInstancedNV"));
         storage::CoverFillPathNV.set_ptr(loadfn("glCoverFillPathNV"));
         storage::CoverStrokePathInstancedNV.set_ptr(loadfn("glCoverStrokePathInstancedNV"));
         storage::CoverStrokePathNV.set_ptr(loadfn("glCoverStrokePathNV"));
         storage::CoverageModulationNV.set_ptr(loadfn("glCoverageModulationNV"));
         storage::CoverageModulationTableNV.set_ptr(loadfn("glCoverageModulationTableNV"));
         storage::CreateBuffers.set_ptr(loadfn("glCreateBuffers"));
         storage::CreateCommandListsNV.set_ptr(loadfn("glCreateCommandListsNV"));
         storage::CreateFramebuffers.set_ptr(loadfn("glCreateFramebuffers"));
         storage::CreateMemoryObjectsEXT.set_ptr(loadfn("glCreateMemoryObjectsEXT"));
         storage::CreatePerfQueryINTEL.set_ptr(loadfn("glCreatePerfQueryINTEL"));
         storage::CreateProgram.set_ptr(loadfn("glCreateProgram"));
         storage::CreateProgramObjectARB.set_ptr(loadfn("glCreateProgramObjectARB"));
         storage::CreateProgramPipelines.set_ptr(loadfn("glCreateProgramPipelines"));
         storage::CreateProgressFenceNVX.set_ptr(loadfn("glCreateProgressFenceNVX"));
         storage::CreateQueries.set_ptr(loadfn("glCreateQueries"));
         storage::CreateRenderbuffers.set_ptr(loadfn("glCreateRenderbuffers"));
         storage::CreateSamplers.set_ptr(loadfn("glCreateSamplers"));
         storage::CreateSemaphoresNV.set_ptr(loadfn("glCreateSemaphoresNV"));
         storage::CreateShader.set_ptr(loadfn("glCreateShader"));
         storage::CreateShaderObjectARB.set_ptr(loadfn("glCreateShaderObjectARB"));
         storage::CreateShaderProgramEXT.set_ptr(loadfn("glCreateShaderProgramEXT"));
         storage::CreateShaderProgramv.set_ptr(loadfn("glCreateShaderProgramv"));
         storage::CreateStatesNV.set_ptr(loadfn("glCreateStatesNV"));
         storage::CreateSyncFromCLeventARB.set_ptr(loadfn("glCreateSyncFromCLeventARB"));
         storage::CreateTextures.set_ptr(loadfn("glCreateTextures"));
         storage::CreateTransformFeedbacks.set_ptr(loadfn("glCreateTransformFeedbacks"));
         storage::CreateVertexArrays.set_ptr(loadfn("glCreateVertexArrays"));
         storage::CullFace.set_ptr(loadfn("glCullFace"));
         storage::CullParameterdvEXT.set_ptr(loadfn("glCullParameterdvEXT"));
         storage::CullParameterfvEXT.set_ptr(loadfn("glCullParameterfvEXT"));
         storage::CurrentPaletteMatrixARB.set_ptr(loadfn("glCurrentPaletteMatrixARB"));
         storage::DebugMessageCallback.set_ptr(loadfn("glDebugMessageCallback"));
         storage::DebugMessageCallbackAMD.set_ptr(loadfn("glDebugMessageCallbackAMD"));
         storage::DebugMessageCallbackARB.set_ptr(loadfn("glDebugMessageCallbackARB"));
         storage::DebugMessageControl.set_ptr(loadfn("glDebugMessageControl"));
         storage::DebugMessageControlARB.set_ptr(loadfn("glDebugMessageControlARB"));
         storage::DebugMessageEnableAMD.set_ptr(loadfn("glDebugMessageEnableAMD"));
         storage::DebugMessageInsert.set_ptr(loadfn("glDebugMessageInsert"));
         storage::DebugMessageInsertAMD.set_ptr(loadfn("glDebugMessageInsertAMD"));
         storage::DebugMessageInsertARB.set_ptr(loadfn("glDebugMessageInsertARB"));
         storage::DeformSGIX.set_ptr(loadfn("glDeformSGIX"));
         storage::DeformationMap3dSGIX.set_ptr(loadfn("glDeformationMap3dSGIX"));
         storage::DeformationMap3fSGIX.set_ptr(loadfn("glDeformationMap3fSGIX"));
         storage::DeleteAsyncMarkersSGIX.set_ptr(loadfn("glDeleteAsyncMarkersSGIX"));
         storage::DeleteBuffers.set_ptr(loadfn("glDeleteBuffers"));
         storage::DeleteBuffersARB.set_ptr(loadfn("glDeleteBuffersARB"));
         storage::DeleteCommandListsNV.set_ptr(loadfn("glDeleteCommandListsNV"));
         storage::DeleteFencesAPPLE.set_ptr(loadfn("glDeleteFencesAPPLE"));
         storage::DeleteFencesNV.set_ptr(loadfn("glDeleteFencesNV"));
         storage::DeleteFragmentShaderATI.set_ptr(loadfn("glDeleteFragmentShaderATI"));
         storage::DeleteFramebuffers.set_ptr(loadfn("glDeleteFramebuffers"));
         storage::DeleteFramebuffersEXT.set_ptr(loadfn("glDeleteFramebuffersEXT"));
         storage::DeleteMemoryObjectsEXT.set_ptr(loadfn("glDeleteMemoryObjectsEXT"));
         storage::DeleteNamedStringARB.set_ptr(loadfn("glDeleteNamedStringARB"));
         storage::DeleteNamesAMD.set_ptr(loadfn("glDeleteNamesAMD"));
         storage::DeleteObjectARB.set_ptr(loadfn("glDeleteObjectARB"));
         storage::DeleteOcclusionQueriesNV.set_ptr(loadfn("glDeleteOcclusionQueriesNV"));
         storage::DeletePathsNV.set_ptr(loadfn("glDeletePathsNV"));
         storage::DeletePerfMonitorsAMD.set_ptr(loadfn("glDeletePerfMonitorsAMD"));
         storage::DeletePerfQueryINTEL.set_ptr(loadfn("glDeletePerfQueryINTEL"));
         storage::DeleteProgram.set_ptr(loadfn("glDeleteProgram"));
         storage::DeleteProgramPipelines.set_ptr(loadfn("glDeleteProgramPipelines"));
         storage::DeleteProgramsARB.set_ptr(loadfn("glDeleteProgramsARB"));
         storage::DeleteProgramsNV.set_ptr(loadfn("glDeleteProgramsNV"));
         storage::DeleteQueries.set_ptr(loadfn("glDeleteQueries"));
         storage::DeleteQueriesARB.set_ptr(loadfn("glDeleteQueriesARB"));
         storage::DeleteQueryResourceTagNV.set_ptr(loadfn("glDeleteQueryResourceTagNV"));
         storage::DeleteRenderbuffers.set_ptr(loadfn("glDeleteRenderbuffers"));
         storage::DeleteRenderbuffersEXT.set_ptr(loadfn("glDeleteRenderbuffersEXT"));
         storage::DeleteSamplers.set_ptr(loadfn("glDeleteSamplers"));
         storage::DeleteSemaphoresEXT.set_ptr(loadfn("glDeleteSemaphoresEXT"));
         storage::DeleteShader.set_ptr(loadfn("glDeleteShader"));
         storage::DeleteStatesNV.set_ptr(loadfn("glDeleteStatesNV"));
         storage::DeleteSync.set_ptr(loadfn("glDeleteSync"));
         storage::DeleteTextures.set_ptr(loadfn("glDeleteTextures"));
         storage::DeleteTexturesEXT.set_ptr(loadfn("glDeleteTexturesEXT"));
         storage::DeleteTransformFeedbacks.set_ptr(loadfn("glDeleteTransformFeedbacks"));
         storage::DeleteTransformFeedbacksNV.set_ptr(loadfn("glDeleteTransformFeedbacksNV"));
         storage::DeleteVertexArrays.set_ptr(loadfn("glDeleteVertexArrays"));
         storage::DeleteVertexArraysAPPLE.set_ptr(loadfn("glDeleteVertexArraysAPPLE"));
         storage::DeleteVertexShaderEXT.set_ptr(loadfn("glDeleteVertexShaderEXT"));
         storage::DepthBoundsEXT.set_ptr(loadfn("glDepthBoundsEXT"));
         storage::DepthBoundsdNV.set_ptr(loadfn("glDepthBoundsdNV"));
         storage::DepthFunc.set_ptr(loadfn("glDepthFunc"));
         storage::DepthMask.set_ptr(loadfn("glDepthMask"));
         storage::DepthRange.set_ptr(loadfn("glDepthRange"));
         storage::DepthRangeArraydvNV.set_ptr(loadfn("glDepthRangeArraydvNV"));
         storage::DepthRangeArrayv.set_ptr(loadfn("glDepthRangeArrayv"));
         storage::DepthRangeIndexed.set_ptr(loadfn("glDepthRangeIndexed"));
         storage::DepthRangeIndexeddNV.set_ptr(loadfn("glDepthRangeIndexeddNV"));
         storage::DepthRangedNV.set_ptr(loadfn("glDepthRangedNV"));
         storage::DepthRangef.set_ptr(loadfn("glDepthRangef"));
         storage::DepthRangefOES.set_ptr(loadfn("glDepthRangefOES"));
         storage::DepthRangexOES.set_ptr(loadfn("glDepthRangexOES"));
         storage::DetachObjectARB.set_ptr(loadfn("glDetachObjectARB"));
         storage::DetachShader.set_ptr(loadfn("glDetachShader"));
         storage::DetailTexFuncSGIS.set_ptr(loadfn("glDetailTexFuncSGIS"));
         storage::Disable.set_ptr(loadfn("glDisable"));
         storage::DisableClientStateIndexedEXT.set_ptr(loadfn("glDisableClientStateIndexedEXT"));
         storage::DisableClientStateiEXT.set_ptr(loadfn("glDisableClientStateiEXT"));
         storage::DisableIndexedEXT.set_ptr(loadfn("glDisableIndexedEXT"));
         storage::DisableVariantClientStateEXT.set_ptr(loadfn("glDisableVariantClientStateEXT"));
         storage::DisableVertexArrayAttrib.set_ptr(loadfn("glDisableVertexArrayAttrib"));
         storage::DisableVertexArrayAttribEXT.set_ptr(loadfn("glDisableVertexArrayAttribEXT"));
         storage::DisableVertexArrayEXT.set_ptr(loadfn("glDisableVertexArrayEXT"));
         storage::DisableVertexAttribAPPLE.set_ptr(loadfn("glDisableVertexAttribAPPLE"));
         storage::DisableVertexAttribArray.set_ptr(loadfn("glDisableVertexAttribArray"));
         storage::DisableVertexAttribArrayARB.set_ptr(loadfn("glDisableVertexAttribArrayARB"));
         storage::Disablei.set_ptr(loadfn("glDisablei"));
         storage::DispatchCompute.set_ptr(loadfn("glDispatchCompute"));
         storage::DispatchComputeGroupSizeARB.set_ptr(loadfn("glDispatchComputeGroupSizeARB"));
         storage::DispatchComputeIndirect.set_ptr(loadfn("glDispatchComputeIndirect"));
         storage::DrawArrays.set_ptr(loadfn("glDrawArrays"));
         storage::DrawArraysEXT.set_ptr(loadfn("glDrawArraysEXT"));
         storage::DrawArraysIndirect.set_ptr(loadfn("glDrawArraysIndirect"));
         storage::DrawArraysInstanced.set_ptr(loadfn("glDrawArraysInstanced"));
         storage::DrawArraysInstancedARB.set_ptr(loadfn("glDrawArraysInstancedARB"));
         storage::DrawArraysInstancedBaseInstance.set_ptr(loadfn("glDrawArraysInstancedBaseInstance"));
         storage::DrawArraysInstancedEXT.set_ptr(loadfn("glDrawArraysInstancedEXT"));
         storage::DrawBuffer.set_ptr(loadfn("glDrawBuffer"));
         storage::DrawBuffers.set_ptr(loadfn("glDrawBuffers"));
         storage::DrawBuffersARB.set_ptr(loadfn("glDrawBuffersARB"));
         storage::DrawBuffersATI.set_ptr(loadfn("glDrawBuffersATI"));
         storage::DrawCommandsAddressNV.set_ptr(loadfn("glDrawCommandsAddressNV"));
         storage::DrawCommandsNV.set_ptr(loadfn("glDrawCommandsNV"));
         storage::DrawCommandsStatesAddressNV.set_ptr(loadfn("glDrawCommandsStatesAddressNV"));
         storage::DrawCommandsStatesNV.set_ptr(loadfn("glDrawCommandsStatesNV"));
         storage::DrawElementArrayAPPLE.set_ptr(loadfn("glDrawElementArrayAPPLE"));
         storage::DrawElementArrayATI.set_ptr(loadfn("glDrawElementArrayATI"));
         storage::DrawElements.set_ptr(loadfn("glDrawElements"));
         storage::DrawElementsBaseVertex.set_ptr(loadfn("glDrawElementsBaseVertex"));
         storage::DrawElementsIndirect.set_ptr(loadfn("glDrawElementsIndirect"));
         storage::DrawElementsInstanced.set_ptr(loadfn("glDrawElementsInstanced"));
         storage::DrawElementsInstancedARB.set_ptr(loadfn("glDrawElementsInstancedARB"));
         storage::DrawElementsInstancedBaseInstance.set_ptr(loadfn("glDrawElementsInstancedBaseInstance"));
         storage::DrawElementsInstancedBaseVertex.set_ptr(loadfn("glDrawElementsInstancedBaseVertex"));
         storage::DrawElementsInstancedBaseVertexBaseInstance.set_ptr(loadfn("glDrawElementsInstancedBaseVertexBaseInstance"));
         storage::DrawElementsInstancedEXT.set_ptr(loadfn("glDrawElementsInstancedEXT"));
         storage::DrawMeshArraysSUN.set_ptr(loadfn("glDrawMeshArraysSUN"));
         storage::DrawMeshTasksIndirectNV.set_ptr(loadfn("glDrawMeshTasksIndirectNV"));
         storage::DrawMeshTasksNV.set_ptr(loadfn("glDrawMeshTasksNV"));
         storage::DrawRangeElementArrayAPPLE.set_ptr(loadfn("glDrawRangeElementArrayAPPLE"));
         storage::DrawRangeElementArrayATI.set_ptr(loadfn("glDrawRangeElementArrayATI"));
         storage::DrawRangeElements.set_ptr(loadfn("glDrawRangeElements"));
         storage::DrawRangeElementsBaseVertex.set_ptr(loadfn("glDrawRangeElementsBaseVertex"));
         storage::DrawRangeElementsEXT.set_ptr(loadfn("glDrawRangeElementsEXT"));
         storage::DrawTextureNV.set_ptr(loadfn("glDrawTextureNV"));
         storage::DrawTransformFeedback.set_ptr(loadfn("glDrawTransformFeedback"));
         storage::DrawTransformFeedbackInstanced.set_ptr(loadfn("glDrawTransformFeedbackInstanced"));
         storage::DrawTransformFeedbackNV.set_ptr(loadfn("glDrawTransformFeedbackNV"));
         storage::DrawTransformFeedbackStream.set_ptr(loadfn("glDrawTransformFeedbackStream"));
         storage::DrawTransformFeedbackStreamInstanced.set_ptr(loadfn("glDrawTransformFeedbackStreamInstanced"));
         storage::DrawVkImageNV.set_ptr(loadfn("glDrawVkImageNV"));
         storage::EGLImageTargetTexStorageEXT.set_ptr(loadfn("glEGLImageTargetTexStorageEXT"));
         storage::EGLImageTargetTextureStorageEXT.set_ptr(loadfn("glEGLImageTargetTextureStorageEXT"));
         storage::EdgeFlagFormatNV.set_ptr(loadfn("glEdgeFlagFormatNV"));
         storage::EdgeFlagPointerEXT.set_ptr(loadfn("glEdgeFlagPointerEXT"));
         storage::EdgeFlagPointerListIBM.set_ptr(loadfn("glEdgeFlagPointerListIBM"));
         storage::ElementPointerAPPLE.set_ptr(loadfn("glElementPointerAPPLE"));
         storage::ElementPointerATI.set_ptr(loadfn("glElementPointerATI"));
         storage::Enable.set_ptr(loadfn("glEnable"));
         storage::EnableClientStateIndexedEXT.set_ptr(loadfn("glEnableClientStateIndexedEXT"));
         storage::EnableClientStateiEXT.set_ptr(loadfn("glEnableClientStateiEXT"));
         storage::EnableIndexedEXT.set_ptr(loadfn("glEnableIndexedEXT"));
         storage::EnableVariantClientStateEXT.set_ptr(loadfn("glEnableVariantClientStateEXT"));
         storage::EnableVertexArrayAttrib.set_ptr(loadfn("glEnableVertexArrayAttrib"));
         storage::EnableVertexArrayAttribEXT.set_ptr(loadfn("glEnableVertexArrayAttribEXT"));
         storage::EnableVertexArrayEXT.set_ptr(loadfn("glEnableVertexArrayEXT"));
         storage::EnableVertexAttribAPPLE.set_ptr(loadfn("glEnableVertexAttribAPPLE"));
         storage::EnableVertexAttribArray.set_ptr(loadfn("glEnableVertexAttribArray"));
         storage::EnableVertexAttribArrayARB.set_ptr(loadfn("glEnableVertexAttribArrayARB"));
         storage::Enablei.set_ptr(loadfn("glEnablei"));
         storage::EndConditionalRender.set_ptr(loadfn("glEndConditionalRender"));
         storage::EndConditionalRenderNV.set_ptr(loadfn("glEndConditionalRenderNV"));
         storage::EndConditionalRenderNVX.set_ptr(loadfn("glEndConditionalRenderNVX"));
         storage::EndFragmentShaderATI.set_ptr(loadfn("glEndFragmentShaderATI"));
         storage::EndOcclusionQueryNV.set_ptr(loadfn("glEndOcclusionQueryNV"));
         storage::EndPerfMonitorAMD.set_ptr(loadfn("glEndPerfMonitorAMD"));
         storage::EndPerfQueryINTEL.set_ptr(loadfn("glEndPerfQueryINTEL"));
         storage::EndQuery.set_ptr(loadfn("glEndQuery"));
         storage::EndQueryARB.set_ptr(loadfn("glEndQueryARB"));
         storage::EndQueryIndexed.set_ptr(loadfn("glEndQueryIndexed"));
         storage::EndTransformFeedback.set_ptr(loadfn("glEndTransformFeedback"));
         storage::EndTransformFeedbackEXT.set_ptr(loadfn("glEndTransformFeedbackEXT"));
         storage::EndTransformFeedbackNV.set_ptr(loadfn("glEndTransformFeedbackNV"));
         storage::EndVertexShaderEXT.set_ptr(loadfn("glEndVertexShaderEXT"));
         storage::EndVideoCaptureNV.set_ptr(loadfn("glEndVideoCaptureNV"));
         storage::EvalCoord1xOES.set_ptr(loadfn("glEvalCoord1xOES"));
         storage::EvalCoord1xvOES.set_ptr(loadfn("glEvalCoord1xvOES"));
         storage::EvalCoord2xOES.set_ptr(loadfn("glEvalCoord2xOES"));
         storage::EvalCoord2xvOES.set_ptr(loadfn("glEvalCoord2xvOES"));
         storage::EvalMapsNV.set_ptr(loadfn("glEvalMapsNV"));
         storage::EvaluateDepthValuesARB.set_ptr(loadfn("glEvaluateDepthValuesARB"));
         storage::ExecuteProgramNV.set_ptr(loadfn("glExecuteProgramNV"));
         storage::ExtractComponentEXT.set_ptr(loadfn("glExtractComponentEXT"));
         storage::FeedbackBufferxOES.set_ptr(loadfn("glFeedbackBufferxOES"));
         storage::FenceSync.set_ptr(loadfn("glFenceSync"));
         storage::FinalCombinerInputNV.set_ptr(loadfn("glFinalCombinerInputNV"));
         storage::Finish.set_ptr(loadfn("glFinish"));
         storage::FinishAsyncSGIX.set_ptr(loadfn("glFinishAsyncSGIX"));
         storage::FinishFenceAPPLE.set_ptr(loadfn("glFinishFenceAPPLE"));
         storage::FinishFenceNV.set_ptr(loadfn("glFinishFenceNV"));
         storage::FinishObjectAPPLE.set_ptr(loadfn("glFinishObjectAPPLE"));
         storage::FinishTextureSUNX.set_ptr(loadfn("glFinishTextureSUNX"));
         storage::Flush.set_ptr(loadfn("glFlush"));
         storage::FlushMappedBufferRange.set_ptr(loadfn("glFlushMappedBufferRange"));
         storage::FlushMappedBufferRangeAPPLE.set_ptr(loadfn("glFlushMappedBufferRangeAPPLE"));
         storage::FlushMappedNamedBufferRange.set_ptr(loadfn("glFlushMappedNamedBufferRange"));
         storage::FlushMappedNamedBufferRangeEXT.set_ptr(loadfn("glFlushMappedNamedBufferRangeEXT"));
         storage::FlushPixelDataRangeNV.set_ptr(loadfn("glFlushPixelDataRangeNV"));
         storage::FlushRasterSGIX.set_ptr(loadfn("glFlushRasterSGIX"));
         storage::FlushStaticDataIBM.set_ptr(loadfn("glFlushStaticDataIBM"));
         storage::FlushVertexArrayRangeAPPLE.set_ptr(loadfn("glFlushVertexArrayRangeAPPLE"));
         storage::FlushVertexArrayRangeNV.set_ptr(loadfn("glFlushVertexArrayRangeNV"));
         storage::FogCoordFormatNV.set_ptr(loadfn("glFogCoordFormatNV"));
         storage::FogCoordPointerEXT.set_ptr(loadfn("glFogCoordPointerEXT"));
         storage::FogCoordPointerListIBM.set_ptr(loadfn("glFogCoordPointerListIBM"));
         storage::FogCoorddEXT.set_ptr(loadfn("glFogCoorddEXT"));
         storage::FogCoorddvEXT.set_ptr(loadfn("glFogCoorddvEXT"));
         storage::FogCoordfEXT.set_ptr(loadfn("glFogCoordfEXT"));
         storage::FogCoordfvEXT.set_ptr(loadfn("glFogCoordfvEXT"));
         storage::FogCoordhNV.set_ptr(loadfn("glFogCoordhNV"));
         storage::FogCoordhvNV.set_ptr(loadfn("glFogCoordhvNV"));
         storage::FogFuncSGIS.set_ptr(loadfn("glFogFuncSGIS"));
         storage::FogxOES.set_ptr(loadfn("glFogxOES"));
         storage::FogxvOES.set_ptr(loadfn("glFogxvOES"));
         storage::FragmentColorMaterialSGIX.set_ptr(loadfn("glFragmentColorMaterialSGIX"));
         storage::FragmentCoverageColorNV.set_ptr(loadfn("glFragmentCoverageColorNV"));
         storage::FragmentLightModelfSGIX.set_ptr(loadfn("glFragmentLightModelfSGIX"));
         storage::FragmentLightModelfvSGIX.set_ptr(loadfn("glFragmentLightModelfvSGIX"));
         storage::FragmentLightModeliSGIX.set_ptr(loadfn("glFragmentLightModeliSGIX"));
         storage::FragmentLightModelivSGIX.set_ptr(loadfn("glFragmentLightModelivSGIX"));
         storage::FragmentLightfSGIX.set_ptr(loadfn("glFragmentLightfSGIX"));
         storage::FragmentLightfvSGIX.set_ptr(loadfn("glFragmentLightfvSGIX"));
         storage::FragmentLightiSGIX.set_ptr(loadfn("glFragmentLightiSGIX"));
         storage::FragmentLightivSGIX.set_ptr(loadfn("glFragmentLightivSGIX"));
         storage::FragmentMaterialfSGIX.set_ptr(loadfn("glFragmentMaterialfSGIX"));
         storage::FragmentMaterialfvSGIX.set_ptr(loadfn("glFragmentMaterialfvSGIX"));
         storage::FragmentMaterialiSGIX.set_ptr(loadfn("glFragmentMaterialiSGIX"));
         storage::FragmentMaterialivSGIX.set_ptr(loadfn("glFragmentMaterialivSGIX"));
         storage::FrameTerminatorGREMEDY.set_ptr(loadfn("glFrameTerminatorGREMEDY"));
         storage::FrameZoomSGIX.set_ptr(loadfn("glFrameZoomSGIX"));
         storage::FramebufferDrawBufferEXT.set_ptr(loadfn("glFramebufferDrawBufferEXT"));
         storage::FramebufferDrawBuffersEXT.set_ptr(loadfn("glFramebufferDrawBuffersEXT"));
         storage::FramebufferFetchBarrierEXT.set_ptr(loadfn("glFramebufferFetchBarrierEXT"));
         storage::FramebufferParameteri.set_ptr(loadfn("glFramebufferParameteri"));
         storage::FramebufferParameteriMESA.set_ptr(loadfn("glFramebufferParameteriMESA"));
         storage::FramebufferReadBufferEXT.set_ptr(loadfn("glFramebufferReadBufferEXT"));
         storage::FramebufferRenderbuffer.set_ptr(loadfn("glFramebufferRenderbuffer"));
         storage::FramebufferRenderbufferEXT.set_ptr(loadfn("glFramebufferRenderbufferEXT"));
         storage::FramebufferSampleLocationsfvARB.set_ptr(loadfn("glFramebufferSampleLocationsfvARB"));
         storage::FramebufferSampleLocationsfvNV.set_ptr(loadfn("glFramebufferSampleLocationsfvNV"));
         storage::FramebufferSamplePositionsfvAMD.set_ptr(loadfn("glFramebufferSamplePositionsfvAMD"));
         storage::FramebufferTexture.set_ptr(loadfn("glFramebufferTexture"));
         storage::FramebufferTexture1D.set_ptr(loadfn("glFramebufferTexture1D"));
         storage::FramebufferTexture1DEXT.set_ptr(loadfn("glFramebufferTexture1DEXT"));
         storage::FramebufferTexture2D.set_ptr(loadfn("glFramebufferTexture2D"));
         storage::FramebufferTexture2DEXT.set_ptr(loadfn("glFramebufferTexture2DEXT"));
         storage::FramebufferTexture3D.set_ptr(loadfn("glFramebufferTexture3D"));
         storage::FramebufferTexture3DEXT.set_ptr(loadfn("glFramebufferTexture3DEXT"));
         storage::FramebufferTextureARB.set_ptr(loadfn("glFramebufferTextureARB"));
         storage::FramebufferTextureEXT.set_ptr(loadfn("glFramebufferTextureEXT"));
         storage::FramebufferTextureFaceARB.set_ptr(loadfn("glFramebufferTextureFaceARB"));
         storage::FramebufferTextureFaceEXT.set_ptr(loadfn("glFramebufferTextureFaceEXT"));
         storage::FramebufferTextureLayer.set_ptr(loadfn("glFramebufferTextureLayer"));
         storage::FramebufferTextureLayerARB.set_ptr(loadfn("glFramebufferTextureLayerARB"));
         storage::FramebufferTextureLayerEXT.set_ptr(loadfn("glFramebufferTextureLayerEXT"));
         storage::FramebufferTextureMultiviewOVR.set_ptr(loadfn("glFramebufferTextureMultiviewOVR"));
         storage::FreeObjectBufferATI.set_ptr(loadfn("glFreeObjectBufferATI"));
         storage::FrontFace.set_ptr(loadfn("glFrontFace"));
         storage::FrustumfOES.set_ptr(loadfn("glFrustumfOES"));
         storage::FrustumxOES.set_ptr(loadfn("glFrustumxOES"));
         storage::GenAsyncMarkersSGIX.set_ptr(loadfn("glGenAsyncMarkersSGIX"));
         storage::GenBuffers.set_ptr(loadfn("glGenBuffers"));
         storage::GenBuffersARB.set_ptr(loadfn("glGenBuffersARB"));
         storage::GenFencesAPPLE.set_ptr(loadfn("glGenFencesAPPLE"));
         storage::GenFencesNV.set_ptr(loadfn("glGenFencesNV"));
         storage::GenFragmentShadersATI.set_ptr(loadfn("glGenFragmentShadersATI"));
         storage::GenFramebuffers.set_ptr(loadfn("glGenFramebuffers"));
         storage::GenFramebuffersEXT.set_ptr(loadfn("glGenFramebuffersEXT"));
         storage::GenNamesAMD.set_ptr(loadfn("glGenNamesAMD"));
         storage::GenOcclusionQueriesNV.set_ptr(loadfn("glGenOcclusionQueriesNV"));
         storage::GenPathsNV.set_ptr(loadfn("glGenPathsNV"));
         storage::GenPerfMonitorsAMD.set_ptr(loadfn("glGenPerfMonitorsAMD"));
         storage::GenProgramPipelines.set_ptr(loadfn("glGenProgramPipelines"));
         storage::GenProgramsARB.set_ptr(loadfn("glGenProgramsARB"));
         storage::GenProgramsNV.set_ptr(loadfn("glGenProgramsNV"));
         storage::GenQueries.set_ptr(loadfn("glGenQueries"));
         storage::GenQueriesARB.set_ptr(loadfn("glGenQueriesARB"));
         storage::GenQueryResourceTagNV.set_ptr(loadfn("glGenQueryResourceTagNV"));
         storage::GenRenderbuffers.set_ptr(loadfn("glGenRenderbuffers"));
         storage::GenRenderbuffersEXT.set_ptr(loadfn("glGenRenderbuffersEXT"));
         storage::GenSamplers.set_ptr(loadfn("glGenSamplers"));
         storage::GenSemaphoresEXT.set_ptr(loadfn("glGenSemaphoresEXT"));
         storage::GenSymbolsEXT.set_ptr(loadfn("glGenSymbolsEXT"));
         storage::GenTextures.set_ptr(loadfn("glGenTextures"));
         storage::GenTexturesEXT.set_ptr(loadfn("glGenTexturesEXT"));
         storage::GenTransformFeedbacks.set_ptr(loadfn("glGenTransformFeedbacks"));
         storage::GenTransformFeedbacksNV.set_ptr(loadfn("glGenTransformFeedbacksNV"));
         storage::GenVertexArrays.set_ptr(loadfn("glGenVertexArrays"));
         storage::GenVertexArraysAPPLE.set_ptr(loadfn("glGenVertexArraysAPPLE"));
         storage::GenVertexShadersEXT.set_ptr(loadfn("glGenVertexShadersEXT"));
         storage::GenerateMipmap.set_ptr(loadfn("glGenerateMipmap"));
         storage::GenerateMipmapEXT.set_ptr(loadfn("glGenerateMipmapEXT"));
         storage::GenerateMultiTexMipmapEXT.set_ptr(loadfn("glGenerateMultiTexMipmapEXT"));
         storage::GenerateTextureMipmap.set_ptr(loadfn("glGenerateTextureMipmap"));
         storage::GenerateTextureMipmapEXT.set_ptr(loadfn("glGenerateTextureMipmapEXT"));
         storage::GetActiveAtomicCounterBufferiv.set_ptr(loadfn("glGetActiveAtomicCounterBufferiv"));
         storage::GetActiveAttrib.set_ptr(loadfn("glGetActiveAttrib"));
         storage::GetActiveAttribARB.set_ptr(loadfn("glGetActiveAttribARB"));
         storage::GetActiveSubroutineName.set_ptr(loadfn("glGetActiveSubroutineName"));
         storage::GetActiveSubroutineUniformName.set_ptr(loadfn("glGetActiveSubroutineUniformName"));
         storage::GetActiveSubroutineUniformiv.set_ptr(loadfn("glGetActiveSubroutineUniformiv"));
         storage::GetActiveUniform.set_ptr(loadfn("glGetActiveUniform"));
         storage::GetActiveUniformARB.set_ptr(loadfn("glGetActiveUniformARB"));
         storage::GetActiveUniformBlockName.set_ptr(loadfn("glGetActiveUniformBlockName"));
         storage::GetActiveUniformBlockiv.set_ptr(loadfn("glGetActiveUniformBlockiv"));
         storage::GetActiveUniformName.set_ptr(loadfn("glGetActiveUniformName"));
         storage::GetActiveUniformsiv.set_ptr(loadfn("glGetActiveUniformsiv"));
         storage::GetActiveVaryingNV.set_ptr(loadfn("glGetActiveVaryingNV"));
         storage::GetArrayObjectfvATI.set_ptr(loadfn("glGetArrayObjectfvATI"));
         storage::GetArrayObjectivATI.set_ptr(loadfn("glGetArrayObjectivATI"));
         storage::GetAttachedObjectsARB.set_ptr(loadfn("glGetAttachedObjectsARB"));
         storage::GetAttachedShaders.set_ptr(loadfn("glGetAttachedShaders"));
         storage::GetAttribLocation.set_ptr(loadfn("glGetAttribLocation"));
         storage::GetAttribLocationARB.set_ptr(loadfn("glGetAttribLocationARB"));
         storage::GetBooleanIndexedvEXT.set_ptr(loadfn("glGetBooleanIndexedvEXT"));
         storage::GetBooleani_v.set_ptr(loadfn("glGetBooleani_v"));
         storage::GetBooleanv.set_ptr(loadfn("glGetBooleanv"));
         storage::GetBufferParameteri64v.set_ptr(loadfn("glGetBufferParameteri64v"));
         storage::GetBufferParameteriv.set_ptr(loadfn("glGetBufferParameteriv"));
         storage::GetBufferParameterivARB.set_ptr(loadfn("glGetBufferParameterivARB"));
         storage::GetBufferParameterui64vNV.set_ptr(loadfn("glGetBufferParameterui64vNV"));
         storage::GetBufferPointerv.set_ptr(loadfn("glGetBufferPointerv"));
         storage::GetBufferPointervARB.set_ptr(loadfn("glGetBufferPointervARB"));
         storage::GetBufferSubData.set_ptr(loadfn("glGetBufferSubData"));
         storage::GetBufferSubDataARB.set_ptr(loadfn("glGetBufferSubDataARB"));
         storage::GetClipPlanefOES.set_ptr(loadfn("glGetClipPlanefOES"));
         storage::GetClipPlanexOES.set_ptr(loadfn("glGetClipPlanexOES"));
         storage::GetColorTableEXT.set_ptr(loadfn("glGetColorTableEXT"));
         storage::GetColorTableParameterfvEXT.set_ptr(loadfn("glGetColorTableParameterfvEXT"));
         storage::GetColorTableParameterfvSGI.set_ptr(loadfn("glGetColorTableParameterfvSGI"));
         storage::GetColorTableParameterivEXT.set_ptr(loadfn("glGetColorTableParameterivEXT"));
         storage::GetColorTableParameterivSGI.set_ptr(loadfn("glGetColorTableParameterivSGI"));
         storage::GetColorTableSGI.set_ptr(loadfn("glGetColorTableSGI"));
         storage::GetCombinerInputParameterfvNV.set_ptr(loadfn("glGetCombinerInputParameterfvNV"));
         storage::GetCombinerInputParameterivNV.set_ptr(loadfn("glGetCombinerInputParameterivNV"));
         storage::GetCombinerOutputParameterfvNV.set_ptr(loadfn("glGetCombinerOutputParameterfvNV"));
         storage::GetCombinerOutputParameterivNV.set_ptr(loadfn("glGetCombinerOutputParameterivNV"));
         storage::GetCombinerStageParameterfvNV.set_ptr(loadfn("glGetCombinerStageParameterfvNV"));
         storage::GetCommandHeaderNV.set_ptr(loadfn("glGetCommandHeaderNV"));
         storage::GetCompressedMultiTexImageEXT.set_ptr(loadfn("glGetCompressedMultiTexImageEXT"));
         storage::GetCompressedTexImage.set_ptr(loadfn("glGetCompressedTexImage"));
         storage::GetCompressedTexImageARB.set_ptr(loadfn("glGetCompressedTexImageARB"));
         storage::GetCompressedTextureImage.set_ptr(loadfn("glGetCompressedTextureImage"));
         storage::GetCompressedTextureImageEXT.set_ptr(loadfn("glGetCompressedTextureImageEXT"));
         storage::GetCompressedTextureSubImage.set_ptr(loadfn("glGetCompressedTextureSubImage"));
         storage::GetConvolutionFilterEXT.set_ptr(loadfn("glGetConvolutionFilterEXT"));
         storage::GetConvolutionParameterfvEXT.set_ptr(loadfn("glGetConvolutionParameterfvEXT"));
         storage::GetConvolutionParameterivEXT.set_ptr(loadfn("glGetConvolutionParameterivEXT"));
         storage::GetConvolutionParameterxvOES.set_ptr(loadfn("glGetConvolutionParameterxvOES"));
         storage::GetCoverageModulationTableNV.set_ptr(loadfn("glGetCoverageModulationTableNV"));
         storage::GetDebugMessageLog.set_ptr(loadfn("glGetDebugMessageLog"));
         storage::GetDebugMessageLogAMD.set_ptr(loadfn("glGetDebugMessageLogAMD"));
         storage::GetDebugMessageLogARB.set_ptr(loadfn("glGetDebugMessageLogARB"));
         storage::GetDetailTexFuncSGIS.set_ptr(loadfn("glGetDetailTexFuncSGIS"));
         storage::GetDoubleIndexedvEXT.set_ptr(loadfn("glGetDoubleIndexedvEXT"));
         storage::GetDoublei_v.set_ptr(loadfn("glGetDoublei_v"));
         storage::GetDoublei_vEXT.set_ptr(loadfn("glGetDoublei_vEXT"));
         storage::GetDoublev.set_ptr(loadfn("glGetDoublev"));
         storage::GetError.set_ptr(loadfn("glGetError"));
         storage::GetFenceivNV.set_ptr(loadfn("glGetFenceivNV"));
         storage::GetFinalCombinerInputParameterfvNV.set_ptr(loadfn("glGetFinalCombinerInputParameterfvNV"));
         storage::GetFinalCombinerInputParameterivNV.set_ptr(loadfn("glGetFinalCombinerInputParameterivNV"));
         storage::GetFirstPerfQueryIdINTEL.set_ptr(loadfn("glGetFirstPerfQueryIdINTEL"));
         storage::GetFixedvOES.set_ptr(loadfn("glGetFixedvOES"));
         storage::GetFloatIndexedvEXT.set_ptr(loadfn("glGetFloatIndexedvEXT"));
         storage::GetFloati_v.set_ptr(loadfn("glGetFloati_v"));
         storage::GetFloati_vEXT.set_ptr(loadfn("glGetFloati_vEXT"));
         storage::GetFloatv.set_ptr(loadfn("glGetFloatv"));
         storage::GetFogFuncSGIS.set_ptr(loadfn("glGetFogFuncSGIS"));
         storage::GetFragDataIndex.set_ptr(loadfn("glGetFragDataIndex"));
         storage::GetFragDataLocation.set_ptr(loadfn("glGetFragDataLocation"));
         storage::GetFragDataLocationEXT.set_ptr(loadfn("glGetFragDataLocationEXT"));
         storage::GetFragmentLightfvSGIX.set_ptr(loadfn("glGetFragmentLightfvSGIX"));
         storage::GetFragmentLightivSGIX.set_ptr(loadfn("glGetFragmentLightivSGIX"));
         storage::GetFragmentMaterialfvSGIX.set_ptr(loadfn("glGetFragmentMaterialfvSGIX"));
         storage::GetFragmentMaterialivSGIX.set_ptr(loadfn("glGetFragmentMaterialivSGIX"));
         storage::GetFramebufferAttachmentParameteriv.set_ptr(loadfn("glGetFramebufferAttachmentParameteriv"));
         storage::GetFramebufferAttachmentParameterivEXT.set_ptr(loadfn("glGetFramebufferAttachmentParameterivEXT"));
         storage::GetFramebufferParameterfvAMD.set_ptr(loadfn("glGetFramebufferParameterfvAMD"));
         storage::GetFramebufferParameteriv.set_ptr(loadfn("glGetFramebufferParameteriv"));
         storage::GetFramebufferParameterivEXT.set_ptr(loadfn("glGetFramebufferParameterivEXT"));
         storage::GetFramebufferParameterivMESA.set_ptr(loadfn("glGetFramebufferParameterivMESA"));
         storage::GetGraphicsResetStatus.set_ptr(loadfn("glGetGraphicsResetStatus"));
         storage::GetGraphicsResetStatusARB.set_ptr(loadfn("glGetGraphicsResetStatusARB"));
         storage::GetHandleARB.set_ptr(loadfn("glGetHandleARB"));
         storage::GetHistogramEXT.set_ptr(loadfn("glGetHistogramEXT"));
         storage::GetHistogramParameterfvEXT.set_ptr(loadfn("glGetHistogramParameterfvEXT"));
         storage::GetHistogramParameterivEXT.set_ptr(loadfn("glGetHistogramParameterivEXT"));
         storage::GetHistogramParameterxvOES.set_ptr(loadfn("glGetHistogramParameterxvOES"));
         storage::GetImageHandleARB.set_ptr(loadfn("glGetImageHandleARB"));
         storage::GetImageHandleNV.set_ptr(loadfn("glGetImageHandleNV"));
         storage::GetImageTransformParameterfvHP.set_ptr(loadfn("glGetImageTransformParameterfvHP"));
         storage::GetImageTransformParameterivHP.set_ptr(loadfn("glGetImageTransformParameterivHP"));
         storage::GetInfoLogARB.set_ptr(loadfn("glGetInfoLogARB"));
         storage::GetInstrumentsSGIX.set_ptr(loadfn("glGetInstrumentsSGIX"));
         storage::GetInteger64i_v.set_ptr(loadfn("glGetInteger64i_v"));
         storage::GetInteger64v.set_ptr(loadfn("glGetInteger64v"));
         storage::GetIntegerIndexedvEXT.set_ptr(loadfn("glGetIntegerIndexedvEXT"));
         storage::GetIntegeri_v.set_ptr(loadfn("glGetIntegeri_v"));
         storage::GetIntegerui64i_vNV.set_ptr(loadfn("glGetIntegerui64i_vNV"));
         storage::GetIntegerui64vNV.set_ptr(loadfn("glGetIntegerui64vNV"));
         storage::GetIntegerv.set_ptr(loadfn("glGetIntegerv"));
         storage::GetInternalformatSampleivNV.set_ptr(loadfn("glGetInternalformatSampleivNV"));
         storage::GetInternalformati64v.set_ptr(loadfn("glGetInternalformati64v"));
         storage::GetInternalformativ.set_ptr(loadfn("glGetInternalformativ"));
         storage::GetInvariantBooleanvEXT.set_ptr(loadfn("glGetInvariantBooleanvEXT"));
         storage::GetInvariantFloatvEXT.set_ptr(loadfn("glGetInvariantFloatvEXT"));
         storage::GetInvariantIntegervEXT.set_ptr(loadfn("glGetInvariantIntegervEXT"));
         storage::GetLightxOES.set_ptr(loadfn("glGetLightxOES"));
         storage::GetListParameterfvSGIX.set_ptr(loadfn("glGetListParameterfvSGIX"));
         storage::GetListParameterivSGIX.set_ptr(loadfn("glGetListParameterivSGIX"));
         storage::GetLocalConstantBooleanvEXT.set_ptr(loadfn("glGetLocalConstantBooleanvEXT"));
         storage::GetLocalConstantFloatvEXT.set_ptr(loadfn("glGetLocalConstantFloatvEXT"));
         storage::GetLocalConstantIntegervEXT.set_ptr(loadfn("glGetLocalConstantIntegervEXT"));
         storage::GetMapAttribParameterfvNV.set_ptr(loadfn("glGetMapAttribParameterfvNV"));
         storage::GetMapAttribParameterivNV.set_ptr(loadfn("glGetMapAttribParameterivNV"));
         storage::GetMapControlPointsNV.set_ptr(loadfn("glGetMapControlPointsNV"));
         storage::GetMapParameterfvNV.set_ptr(loadfn("glGetMapParameterfvNV"));
         storage::GetMapParameterivNV.set_ptr(loadfn("glGetMapParameterivNV"));
         storage::GetMapxvOES.set_ptr(loadfn("glGetMapxvOES"));
         storage::GetMaterialxOES.set_ptr(loadfn("glGetMaterialxOES"));
         storage::GetMemoryObjectDetachedResourcesuivNV.set_ptr(loadfn("glGetMemoryObjectDetachedResourcesuivNV"));
         storage::GetMemoryObjectParameterivEXT.set_ptr(loadfn("glGetMemoryObjectParameterivEXT"));
         storage::GetMinmaxEXT.set_ptr(loadfn("glGetMinmaxEXT"));
         storage::GetMinmaxParameterfvEXT.set_ptr(loadfn("glGetMinmaxParameterfvEXT"));
         storage::GetMinmaxParameterivEXT.set_ptr(loadfn("glGetMinmaxParameterivEXT"));
         storage::GetMultiTexEnvfvEXT.set_ptr(loadfn("glGetMultiTexEnvfvEXT"));
         storage::GetMultiTexEnvivEXT.set_ptr(loadfn("glGetMultiTexEnvivEXT"));
         storage::GetMultiTexGendvEXT.set_ptr(loadfn("glGetMultiTexGendvEXT"));
         storage::GetMultiTexGenfvEXT.set_ptr(loadfn("glGetMultiTexGenfvEXT"));
         storage::GetMultiTexGenivEXT.set_ptr(loadfn("glGetMultiTexGenivEXT"));
         storage::GetMultiTexImageEXT.set_ptr(loadfn("glGetMultiTexImageEXT"));
         storage::GetMultiTexLevelParameterfvEXT.set_ptr(loadfn("glGetMultiTexLevelParameterfvEXT"));
         storage::GetMultiTexLevelParameterivEXT.set_ptr(loadfn("glGetMultiTexLevelParameterivEXT"));
         storage::GetMultiTexParameterIivEXT.set_ptr(loadfn("glGetMultiTexParameterIivEXT"));
         storage::GetMultiTexParameterIuivEXT.set_ptr(loadfn("glGetMultiTexParameterIuivEXT"));
         storage::GetMultiTexParameterfvEXT.set_ptr(loadfn("glGetMultiTexParameterfvEXT"));
         storage::GetMultiTexParameterivEXT.set_ptr(loadfn("glGetMultiTexParameterivEXT"));
         storage::GetMultisamplefv.set_ptr(loadfn("glGetMultisamplefv"));
         storage::GetMultisamplefvNV.set_ptr(loadfn("glGetMultisamplefvNV"));
         storage::GetNamedBufferParameteri64v.set_ptr(loadfn("glGetNamedBufferParameteri64v"));
         storage::GetNamedBufferParameteriv.set_ptr(loadfn("glGetNamedBufferParameteriv"));
         storage::GetNamedBufferParameterivEXT.set_ptr(loadfn("glGetNamedBufferParameterivEXT"));
         storage::GetNamedBufferParameterui64vNV.set_ptr(loadfn("glGetNamedBufferParameterui64vNV"));
         storage::GetNamedBufferPointerv.set_ptr(loadfn("glGetNamedBufferPointerv"));
         storage::GetNamedBufferPointervEXT.set_ptr(loadfn("glGetNamedBufferPointervEXT"));
         storage::GetNamedBufferSubData.set_ptr(loadfn("glGetNamedBufferSubData"));
         storage::GetNamedBufferSubDataEXT.set_ptr(loadfn("glGetNamedBufferSubDataEXT"));
         storage::GetNamedFramebufferAttachmentParameteriv.set_ptr(loadfn("glGetNamedFramebufferAttachmentParameteriv"));
         storage::GetNamedFramebufferAttachmentParameterivEXT.set_ptr(loadfn("glGetNamedFramebufferAttachmentParameterivEXT"));
         storage::GetNamedFramebufferParameterfvAMD.set_ptr(loadfn("glGetNamedFramebufferParameterfvAMD"));
         storage::GetNamedFramebufferParameteriv.set_ptr(loadfn("glGetNamedFramebufferParameteriv"));
         storage::GetNamedFramebufferParameterivEXT.set_ptr(loadfn("glGetNamedFramebufferParameterivEXT"));
         storage::GetNamedProgramLocalParameterIivEXT.set_ptr(loadfn("glGetNamedProgramLocalParameterIivEXT"));
         storage::GetNamedProgramLocalParameterIuivEXT.set_ptr(loadfn("glGetNamedProgramLocalParameterIuivEXT"));
         storage::GetNamedProgramLocalParameterdvEXT.set_ptr(loadfn("glGetNamedProgramLocalParameterdvEXT"));
         storage::GetNamedProgramLocalParameterfvEXT.set_ptr(loadfn("glGetNamedProgramLocalParameterfvEXT"));
         storage::GetNamedProgramStringEXT.set_ptr(loadfn("glGetNamedProgramStringEXT"));
         storage::GetNamedProgramivEXT.set_ptr(loadfn("glGetNamedProgramivEXT"));
         storage::GetNamedRenderbufferParameteriv.set_ptr(loadfn("glGetNamedRenderbufferParameteriv"));
         storage::GetNamedRenderbufferParameterivEXT.set_ptr(loadfn("glGetNamedRenderbufferParameterivEXT"));
         storage::GetNamedStringARB.set_ptr(loadfn("glGetNamedStringARB"));
         storage::GetNamedStringivARB.set_ptr(loadfn("glGetNamedStringivARB"));
         storage::GetNextPerfQueryIdINTEL.set_ptr(loadfn("glGetNextPerfQueryIdINTEL"));
         storage::GetObjectBufferfvATI.set_ptr(loadfn("glGetObjectBufferfvATI"));
         storage::GetObjectBufferivATI.set_ptr(loadfn("glGetObjectBufferivATI"));
         storage::GetObjectLabel.set_ptr(loadfn("glGetObjectLabel"));
         storage::GetObjectLabelEXT.set_ptr(loadfn("glGetObjectLabelEXT"));
         storage::GetObjectParameterfvARB.set_ptr(loadfn("glGetObjectParameterfvARB"));
         storage::GetObjectParameterivAPPLE.set_ptr(loadfn("glGetObjectParameterivAPPLE"));
         storage::GetObjectParameterivARB.set_ptr(loadfn("glGetObjectParameterivARB"));
         storage::GetObjectPtrLabel.set_ptr(loadfn("glGetObjectPtrLabel"));
         storage::GetOcclusionQueryivNV.set_ptr(loadfn("glGetOcclusionQueryivNV"));
         storage::GetOcclusionQueryuivNV.set_ptr(loadfn("glGetOcclusionQueryuivNV"));
         storage::GetPathCommandsNV.set_ptr(loadfn("glGetPathCommandsNV"));
         storage::GetPathCoordsNV.set_ptr(loadfn("glGetPathCoordsNV"));
         storage::GetPathDashArrayNV.set_ptr(loadfn("glGetPathDashArrayNV"));
         storage::GetPathLengthNV.set_ptr(loadfn("glGetPathLengthNV"));
         storage::GetPathMetricRangeNV.set_ptr(loadfn("glGetPathMetricRangeNV"));
         storage::GetPathMetricsNV.set_ptr(loadfn("glGetPathMetricsNV"));
         storage::GetPathParameterfvNV.set_ptr(loadfn("glGetPathParameterfvNV"));
         storage::GetPathParameterivNV.set_ptr(loadfn("glGetPathParameterivNV"));
         storage::GetPathSpacingNV.set_ptr(loadfn("glGetPathSpacingNV"));
         storage::GetPerfCounterInfoINTEL.set_ptr(loadfn("glGetPerfCounterInfoINTEL"));
         storage::GetPerfMonitorCounterDataAMD.set_ptr(loadfn("glGetPerfMonitorCounterDataAMD"));
         storage::GetPerfMonitorCounterInfoAMD.set_ptr(loadfn("glGetPerfMonitorCounterInfoAMD"));
         storage::GetPerfMonitorCounterStringAMD.set_ptr(loadfn("glGetPerfMonitorCounterStringAMD"));
         storage::GetPerfMonitorCountersAMD.set_ptr(loadfn("glGetPerfMonitorCountersAMD"));
         storage::GetPerfMonitorGroupStringAMD.set_ptr(loadfn("glGetPerfMonitorGroupStringAMD"));
         storage::GetPerfMonitorGroupsAMD.set_ptr(loadfn("glGetPerfMonitorGroupsAMD"));
         storage::GetPerfQueryDataINTEL.set_ptr(loadfn("glGetPerfQueryDataINTEL"));
         storage::GetPerfQueryIdByNameINTEL.set_ptr(loadfn("glGetPerfQueryIdByNameINTEL"));
         storage::GetPerfQueryInfoINTEL.set_ptr(loadfn("glGetPerfQueryInfoINTEL"));
         storage::GetPixelMapxv.set_ptr(loadfn("glGetPixelMapxv"));
         storage::GetPixelTexGenParameterfvSGIS.set_ptr(loadfn("glGetPixelTexGenParameterfvSGIS"));
         storage::GetPixelTexGenParameterivSGIS.set_ptr(loadfn("glGetPixelTexGenParameterivSGIS"));
         storage::GetPixelTransformParameterfvEXT.set_ptr(loadfn("glGetPixelTransformParameterfvEXT"));
         storage::GetPixelTransformParameterivEXT.set_ptr(loadfn("glGetPixelTransformParameterivEXT"));
         storage::GetPointerIndexedvEXT.set_ptr(loadfn("glGetPointerIndexedvEXT"));
         storage::GetPointeri_vEXT.set_ptr(loadfn("glGetPointeri_vEXT"));
         storage::GetPointerv.set_ptr(loadfn("glGetPointerv"));
         storage::GetPointervEXT.set_ptr(loadfn("glGetPointervEXT"));
         storage::GetProgramBinary.set_ptr(loadfn("glGetProgramBinary"));
         storage::GetProgramEnvParameterIivNV.set_ptr(loadfn("glGetProgramEnvParameterIivNV"));
         storage::GetProgramEnvParameterIuivNV.set_ptr(loadfn("glGetProgramEnvParameterIuivNV"));
         storage::GetProgramEnvParameterdvARB.set_ptr(loadfn("glGetProgramEnvParameterdvARB"));
         storage::GetProgramEnvParameterfvARB.set_ptr(loadfn("glGetProgramEnvParameterfvARB"));
         storage::GetProgramInfoLog.set_ptr(loadfn("glGetProgramInfoLog"));
         storage::GetProgramInterfaceiv.set_ptr(loadfn("glGetProgramInterfaceiv"));
         storage::GetProgramLocalParameterIivNV.set_ptr(loadfn("glGetProgramLocalParameterIivNV"));
         storage::GetProgramLocalParameterIuivNV.set_ptr(loadfn("glGetProgramLocalParameterIuivNV"));
         storage::GetProgramLocalParameterdvARB.set_ptr(loadfn("glGetProgramLocalParameterdvARB"));
         storage::GetProgramLocalParameterfvARB.set_ptr(loadfn("glGetProgramLocalParameterfvARB"));
         storage::GetProgramNamedParameterdvNV.set_ptr(loadfn("glGetProgramNamedParameterdvNV"));
         storage::GetProgramNamedParameterfvNV.set_ptr(loadfn("glGetProgramNamedParameterfvNV"));
         storage::GetProgramParameterdvNV.set_ptr(loadfn("glGetProgramParameterdvNV"));
         storage::GetProgramParameterfvNV.set_ptr(loadfn("glGetProgramParameterfvNV"));
         storage::GetProgramPipelineInfoLog.set_ptr(loadfn("glGetProgramPipelineInfoLog"));
         storage::GetProgramPipelineiv.set_ptr(loadfn("glGetProgramPipelineiv"));
         storage::GetProgramResourceIndex.set_ptr(loadfn("glGetProgramResourceIndex"));
         storage::GetProgramResourceLocation.set_ptr(loadfn("glGetProgramResourceLocation"));
         storage::GetProgramResourceLocationIndex.set_ptr(loadfn("glGetProgramResourceLocationIndex"));
         storage::GetProgramResourceName.set_ptr(loadfn("glGetProgramResourceName"));
         storage::GetProgramResourcefvNV.set_ptr(loadfn("glGetProgramResourcefvNV"));
         storage::GetProgramResourceiv.set_ptr(loadfn("glGetProgramResourceiv"));
         storage::GetProgramStageiv.set_ptr(loadfn("glGetProgramStageiv"));
         storage::GetProgramStringARB.set_ptr(loadfn("glGetProgramStringARB"));
         storage::GetProgramStringNV.set_ptr(loadfn("glGetProgramStringNV"));
         storage::GetProgramSubroutineParameteruivNV.set_ptr(loadfn("glGetProgramSubroutineParameteruivNV"));
         storage::GetProgramiv.set_ptr(loadfn("glGetProgramiv"));
         storage::GetProgramivARB.set_ptr(loadfn("glGetProgramivARB"));
         storage::GetProgramivNV.set_ptr(loadfn("glGetProgramivNV"));
         storage::GetQueryBufferObjecti64v.set_ptr(loadfn("glGetQueryBufferObjecti64v"));
         storage::GetQueryBufferObjectiv.set_ptr(loadfn("glGetQueryBufferObjectiv"));
         storage::GetQueryBufferObjectui64v.set_ptr(loadfn("glGetQueryBufferObjectui64v"));
         storage::GetQueryBufferObjectuiv.set_ptr(loadfn("glGetQueryBufferObjectuiv"));
         storage::GetQueryIndexediv.set_ptr(loadfn("glGetQueryIndexediv"));
         storage::GetQueryObjecti64v.set_ptr(loadfn("glGetQueryObjecti64v"));
         storage::GetQueryObjecti64vEXT.set_ptr(loadfn("glGetQueryObjecti64vEXT"));
         storage::GetQueryObjectiv.set_ptr(loadfn("glGetQueryObjectiv"));
         storage::GetQueryObjectivARB.set_ptr(loadfn("glGetQueryObjectivARB"));
         storage::GetQueryObjectui64v.set_ptr(loadfn("glGetQueryObjectui64v"));
         storage::GetQueryObjectui64vEXT.set_ptr(loadfn("glGetQueryObjectui64vEXT"));
         storage::GetQueryObjectuiv.set_ptr(loadfn("glGetQueryObjectuiv"));
         storage::GetQueryObjectuivARB.set_ptr(loadfn("glGetQueryObjectuivARB"));
         storage::GetQueryiv.set_ptr(loadfn("glGetQueryiv"));
         storage::GetQueryivARB.set_ptr(loadfn("glGetQueryivARB"));
         storage::GetRenderbufferParameteriv.set_ptr(loadfn("glGetRenderbufferParameteriv"));
         storage::GetRenderbufferParameterivEXT.set_ptr(loadfn("glGetRenderbufferParameterivEXT"));
         storage::GetSamplerParameterIiv.set_ptr(loadfn("glGetSamplerParameterIiv"));
         storage::GetSamplerParameterIuiv.set_ptr(loadfn("glGetSamplerParameterIuiv"));
         storage::GetSamplerParameterfv.set_ptr(loadfn("glGetSamplerParameterfv"));
         storage::GetSamplerParameteriv.set_ptr(loadfn("glGetSamplerParameteriv"));
         storage::GetSemaphoreParameterivNV.set_ptr(loadfn("glGetSemaphoreParameterivNV"));
         storage::GetSemaphoreParameterui64vEXT.set_ptr(loadfn("glGetSemaphoreParameterui64vEXT"));
         storage::GetSeparableFilterEXT.set_ptr(loadfn("glGetSeparableFilterEXT"));
         storage::GetShaderInfoLog.set_ptr(loadfn("glGetShaderInfoLog"));
         storage::GetShaderPrecisionFormat.set_ptr(loadfn("glGetShaderPrecisionFormat"));
         storage::GetShaderSource.set_ptr(loadfn("glGetShaderSource"));
         storage::GetShaderSourceARB.set_ptr(loadfn("glGetShaderSourceARB"));
         storage::GetShaderiv.set_ptr(loadfn("glGetShaderiv"));
         storage::GetShadingRateImagePaletteNV.set_ptr(loadfn("glGetShadingRateImagePaletteNV"));
         storage::GetShadingRateSampleLocationivNV.set_ptr(loadfn("glGetShadingRateSampleLocationivNV"));
         storage::GetSharpenTexFuncSGIS.set_ptr(loadfn("glGetSharpenTexFuncSGIS"));
         storage::GetStageIndexNV.set_ptr(loadfn("glGetStageIndexNV"));
         storage::GetString.set_ptr(loadfn("glGetString"));
         storage::GetStringi.set_ptr(loadfn("glGetStringi"));
         storage::GetSubroutineIndex.set_ptr(loadfn("glGetSubroutineIndex"));
         storage::GetSubroutineUniformLocation.set_ptr(loadfn("glGetSubroutineUniformLocation"));
         storage::GetSynciv.set_ptr(loadfn("glGetSynciv"));
         storage::GetTexBumpParameterfvATI.set_ptr(loadfn("glGetTexBumpParameterfvATI"));
         storage::GetTexBumpParameterivATI.set_ptr(loadfn("glGetTexBumpParameterivATI"));
         storage::GetTexEnvxvOES.set_ptr(loadfn("glGetTexEnvxvOES"));
         storage::GetTexFilterFuncSGIS.set_ptr(loadfn("glGetTexFilterFuncSGIS"));
         storage::GetTexGenxvOES.set_ptr(loadfn("glGetTexGenxvOES"));
         storage::GetTexImage.set_ptr(loadfn("glGetTexImage"));
         storage::GetTexLevelParameterfv.set_ptr(loadfn("glGetTexLevelParameterfv"));
         storage::GetTexLevelParameteriv.set_ptr(loadfn("glGetTexLevelParameteriv"));
         storage::GetTexLevelParameterxvOES.set_ptr(loadfn("glGetTexLevelParameterxvOES"));
         storage::GetTexParameterIiv.set_ptr(loadfn("glGetTexParameterIiv"));
         storage::GetTexParameterIivEXT.set_ptr(loadfn("glGetTexParameterIivEXT"));
         storage::GetTexParameterIuiv.set_ptr(loadfn("glGetTexParameterIuiv"));
         storage::GetTexParameterIuivEXT.set_ptr(loadfn("glGetTexParameterIuivEXT"));
         storage::GetTexParameterPointervAPPLE.set_ptr(loadfn("glGetTexParameterPointervAPPLE"));
         storage::GetTexParameterfv.set_ptr(loadfn("glGetTexParameterfv"));
         storage::GetTexParameteriv.set_ptr(loadfn("glGetTexParameteriv"));
         storage::GetTexParameterxvOES.set_ptr(loadfn("glGetTexParameterxvOES"));
         storage::GetTextureHandleARB.set_ptr(loadfn("glGetTextureHandleARB"));
         storage::GetTextureHandleNV.set_ptr(loadfn("glGetTextureHandleNV"));
         storage::GetTextureImage.set_ptr(loadfn("glGetTextureImage"));
         storage::GetTextureImageEXT.set_ptr(loadfn("glGetTextureImageEXT"));
         storage::GetTextureLevelParameterfv.set_ptr(loadfn("glGetTextureLevelParameterfv"));
         storage::GetTextureLevelParameterfvEXT.set_ptr(loadfn("glGetTextureLevelParameterfvEXT"));
         storage::GetTextureLevelParameteriv.set_ptr(loadfn("glGetTextureLevelParameteriv"));
         storage::GetTextureLevelParameterivEXT.set_ptr(loadfn("glGetTextureLevelParameterivEXT"));
         storage::GetTextureParameterIiv.set_ptr(loadfn("glGetTextureParameterIiv"));
         storage::GetTextureParameterIivEXT.set_ptr(loadfn("glGetTextureParameterIivEXT"));
         storage::GetTextureParameterIuiv.set_ptr(loadfn("glGetTextureParameterIuiv"));
         storage::GetTextureParameterIuivEXT.set_ptr(loadfn("glGetTextureParameterIuivEXT"));
         storage::GetTextureParameterfv.set_ptr(loadfn("glGetTextureParameterfv"));
         storage::GetTextureParameterfvEXT.set_ptr(loadfn("glGetTextureParameterfvEXT"));
         storage::GetTextureParameteriv.set_ptr(loadfn("glGetTextureParameteriv"));
         storage::GetTextureParameterivEXT.set_ptr(loadfn("glGetTextureParameterivEXT"));
         storage::GetTextureSamplerHandleARB.set_ptr(loadfn("glGetTextureSamplerHandleARB"));
         storage::GetTextureSamplerHandleNV.set_ptr(loadfn("glGetTextureSamplerHandleNV"));
         storage::GetTextureSubImage.set_ptr(loadfn("glGetTextureSubImage"));
         storage::GetTrackMatrixivNV.set_ptr(loadfn("glGetTrackMatrixivNV"));
         storage::GetTransformFeedbackVarying.set_ptr(loadfn("glGetTransformFeedbackVarying"));
         storage::GetTransformFeedbackVaryingEXT.set_ptr(loadfn("glGetTransformFeedbackVaryingEXT"));
         storage::GetTransformFeedbackVaryingNV.set_ptr(loadfn("glGetTransformFeedbackVaryingNV"));
         storage::GetTransformFeedbacki64_v.set_ptr(loadfn("glGetTransformFeedbacki64_v"));
         storage::GetTransformFeedbacki_v.set_ptr(loadfn("glGetTransformFeedbacki_v"));
         storage::GetTransformFeedbackiv.set_ptr(loadfn("glGetTransformFeedbackiv"));
         storage::GetUniformBlockIndex.set_ptr(loadfn("glGetUniformBlockIndex"));
         storage::GetUniformBufferSizeEXT.set_ptr(loadfn("glGetUniformBufferSizeEXT"));
         storage::GetUniformIndices.set_ptr(loadfn("glGetUniformIndices"));
         storage::GetUniformLocation.set_ptr(loadfn("glGetUniformLocation"));
         storage::GetUniformLocationARB.set_ptr(loadfn("glGetUniformLocationARB"));
         storage::GetUniformOffsetEXT.set_ptr(loadfn("glGetUniformOffsetEXT"));
         storage::GetUniformSubroutineuiv.set_ptr(loadfn("glGetUniformSubroutineuiv"));
         storage::GetUniformdv.set_ptr(loadfn("glGetUniformdv"));
         storage::GetUniformfv.set_ptr(loadfn("glGetUniformfv"));
         storage::GetUniformfvARB.set_ptr(loadfn("glGetUniformfvARB"));
         storage::GetUniformi64vARB.set_ptr(loadfn("glGetUniformi64vARB"));
         storage::GetUniformi64vNV.set_ptr(loadfn("glGetUniformi64vNV"));
         storage::GetUniformiv.set_ptr(loadfn("glGetUniformiv"));
         storage::GetUniformivARB.set_ptr(loadfn("glGetUniformivARB"));
         storage::GetUniformui64vARB.set_ptr(loadfn("glGetUniformui64vARB"));
         storage::GetUniformui64vNV.set_ptr(loadfn("glGetUniformui64vNV"));
         storage::GetUniformuiv.set_ptr(loadfn("glGetUniformuiv"));
         storage::GetUniformuivEXT.set_ptr(loadfn("glGetUniformuivEXT"));
         storage::GetUnsignedBytei_vEXT.set_ptr(loadfn("glGetUnsignedBytei_vEXT"));
         storage::GetUnsignedBytevEXT.set_ptr(loadfn("glGetUnsignedBytevEXT"));
         storage::GetVariantArrayObjectfvATI.set_ptr(loadfn("glGetVariantArrayObjectfvATI"));
         storage::GetVariantArrayObjectivATI.set_ptr(loadfn("glGetVariantArrayObjectivATI"));
         storage::GetVariantBooleanvEXT.set_ptr(loadfn("glGetVariantBooleanvEXT"));
         storage::GetVariantFloatvEXT.set_ptr(loadfn("glGetVariantFloatvEXT"));
         storage::GetVariantIntegervEXT.set_ptr(loadfn("glGetVariantIntegervEXT"));
         storage::GetVariantPointervEXT.set_ptr(loadfn("glGetVariantPointervEXT"));
         storage::GetVaryingLocationNV.set_ptr(loadfn("glGetVaryingLocationNV"));
         storage::GetVertexArrayIndexed64iv.set_ptr(loadfn("glGetVertexArrayIndexed64iv"));
         storage::GetVertexArrayIndexediv.set_ptr(loadfn("glGetVertexArrayIndexediv"));
         storage::GetVertexArrayIntegeri_vEXT.set_ptr(loadfn("glGetVertexArrayIntegeri_vEXT"));
         storage::GetVertexArrayIntegervEXT.set_ptr(loadfn("glGetVertexArrayIntegervEXT"));
         storage::GetVertexArrayPointeri_vEXT.set_ptr(loadfn("glGetVertexArrayPointeri_vEXT"));
         storage::GetVertexArrayPointervEXT.set_ptr(loadfn("glGetVertexArrayPointervEXT"));
         storage::GetVertexArrayiv.set_ptr(loadfn("glGetVertexArrayiv"));
         storage::GetVertexAttribArrayObjectfvATI.set_ptr(loadfn("glGetVertexAttribArrayObjectfvATI"));
         storage::GetVertexAttribArrayObjectivATI.set_ptr(loadfn("glGetVertexAttribArrayObjectivATI"));
         storage::GetVertexAttribIiv.set_ptr(loadfn("glGetVertexAttribIiv"));
         storage::GetVertexAttribIivEXT.set_ptr(loadfn("glGetVertexAttribIivEXT"));
         storage::GetVertexAttribIuiv.set_ptr(loadfn("glGetVertexAttribIuiv"));
         storage::GetVertexAttribIuivEXT.set_ptr(loadfn("glGetVertexAttribIuivEXT"));
         storage::GetVertexAttribLdv.set_ptr(loadfn("glGetVertexAttribLdv"));
         storage::GetVertexAttribLdvEXT.set_ptr(loadfn("glGetVertexAttribLdvEXT"));
         storage::GetVertexAttribLi64vNV.set_ptr(loadfn("glGetVertexAttribLi64vNV"));
         storage::GetVertexAttribLui64vARB.set_ptr(loadfn("glGetVertexAttribLui64vARB"));
         storage::GetVertexAttribLui64vNV.set_ptr(loadfn("glGetVertexAttribLui64vNV"));
         storage::GetVertexAttribPointerv.set_ptr(loadfn("glGetVertexAttribPointerv"));
         storage::GetVertexAttribPointervARB.set_ptr(loadfn("glGetVertexAttribPointervARB"));
         storage::GetVertexAttribPointervNV.set_ptr(loadfn("glGetVertexAttribPointervNV"));
         storage::GetVertexAttribdv.set_ptr(loadfn("glGetVertexAttribdv"));
         storage::GetVertexAttribdvARB.set_ptr(loadfn("glGetVertexAttribdvARB"));
         storage::GetVertexAttribdvNV.set_ptr(loadfn("glGetVertexAttribdvNV"));
         storage::GetVertexAttribfv.set_ptr(loadfn("glGetVertexAttribfv"));
         storage::GetVertexAttribfvARB.set_ptr(loadfn("glGetVertexAttribfvARB"));
         storage::GetVertexAttribfvNV.set_ptr(loadfn("glGetVertexAttribfvNV"));
         storage::GetVertexAttribiv.set_ptr(loadfn("glGetVertexAttribiv"));
         storage::GetVertexAttribivARB.set_ptr(loadfn("glGetVertexAttribivARB"));
         storage::GetVertexAttribivNV.set_ptr(loadfn("glGetVertexAttribivNV"));
         storage::GetVideoCaptureStreamdvNV.set_ptr(loadfn("glGetVideoCaptureStreamdvNV"));
         storage::GetVideoCaptureStreamfvNV.set_ptr(loadfn("glGetVideoCaptureStreamfvNV"));
         storage::GetVideoCaptureStreamivNV.set_ptr(loadfn("glGetVideoCaptureStreamivNV"));
         storage::GetVideoCaptureivNV.set_ptr(loadfn("glGetVideoCaptureivNV"));
         storage::GetVideoi64vNV.set_ptr(loadfn("glGetVideoi64vNV"));
         storage::GetVideoivNV.set_ptr(loadfn("glGetVideoivNV"));
         storage::GetVideoui64vNV.set_ptr(loadfn("glGetVideoui64vNV"));
         storage::GetVideouivNV.set_ptr(loadfn("glGetVideouivNV"));
         storage::GetVkProcAddrNV.set_ptr(loadfn("glGetVkProcAddrNV"));
         storage::GetnCompressedTexImage.set_ptr(loadfn("glGetnCompressedTexImage"));
         storage::GetnCompressedTexImageARB.set_ptr(loadfn("glGetnCompressedTexImageARB"));
         storage::GetnTexImage.set_ptr(loadfn("glGetnTexImage"));
         storage::GetnTexImageARB.set_ptr(loadfn("glGetnTexImageARB"));
         storage::GetnUniformdv.set_ptr(loadfn("glGetnUniformdv"));
         storage::GetnUniformdvARB.set_ptr(loadfn("glGetnUniformdvARB"));
         storage::GetnUniformfv.set_ptr(loadfn("glGetnUniformfv"));
         storage::GetnUniformfvARB.set_ptr(loadfn("glGetnUniformfvARB"));
         storage::GetnUniformi64vARB.set_ptr(loadfn("glGetnUniformi64vARB"));
         storage::GetnUniformiv.set_ptr(loadfn("glGetnUniformiv"));
         storage::GetnUniformivARB.set_ptr(loadfn("glGetnUniformivARB"));
         storage::GetnUniformui64vARB.set_ptr(loadfn("glGetnUniformui64vARB"));
         storage::GetnUniformuiv.set_ptr(loadfn("glGetnUniformuiv"));
         storage::GetnUniformuivARB.set_ptr(loadfn("glGetnUniformuivARB"));
         storage::GlobalAlphaFactorbSUN.set_ptr(loadfn("glGlobalAlphaFactorbSUN"));
         storage::GlobalAlphaFactordSUN.set_ptr(loadfn("glGlobalAlphaFactordSUN"));
         storage::GlobalAlphaFactorfSUN.set_ptr(loadfn("glGlobalAlphaFactorfSUN"));
         storage::GlobalAlphaFactoriSUN.set_ptr(loadfn("glGlobalAlphaFactoriSUN"));
         storage::GlobalAlphaFactorsSUN.set_ptr(loadfn("glGlobalAlphaFactorsSUN"));
         storage::GlobalAlphaFactorubSUN.set_ptr(loadfn("glGlobalAlphaFactorubSUN"));
         storage::GlobalAlphaFactoruiSUN.set_ptr(loadfn("glGlobalAlphaFactoruiSUN"));
         storage::GlobalAlphaFactorusSUN.set_ptr(loadfn("glGlobalAlphaFactorusSUN"));
         storage::Hint.set_ptr(loadfn("glHint"));
         storage::HintPGI.set_ptr(loadfn("glHintPGI"));
         storage::HistogramEXT.set_ptr(loadfn("glHistogramEXT"));
         storage::IglooInterfaceSGIX.set_ptr(loadfn("glIglooInterfaceSGIX"));
         storage::ImageTransformParameterfHP.set_ptr(loadfn("glImageTransformParameterfHP"));
         storage::ImageTransformParameterfvHP.set_ptr(loadfn("glImageTransformParameterfvHP"));
         storage::ImageTransformParameteriHP.set_ptr(loadfn("glImageTransformParameteriHP"));
         storage::ImageTransformParameterivHP.set_ptr(loadfn("glImageTransformParameterivHP"));
         storage::ImportMemoryFdEXT.set_ptr(loadfn("glImportMemoryFdEXT"));
         storage::ImportMemoryWin32HandleEXT.set_ptr(loadfn("glImportMemoryWin32HandleEXT"));
         storage::ImportMemoryWin32NameEXT.set_ptr(loadfn("glImportMemoryWin32NameEXT"));
         storage::ImportSemaphoreFdEXT.set_ptr(loadfn("glImportSemaphoreFdEXT"));
         storage::ImportSemaphoreWin32HandleEXT.set_ptr(loadfn("glImportSemaphoreWin32HandleEXT"));
         storage::ImportSemaphoreWin32NameEXT.set_ptr(loadfn("glImportSemaphoreWin32NameEXT"));
         storage::ImportSyncEXT.set_ptr(loadfn("glImportSyncEXT"));
         storage::IndexFormatNV.set_ptr(loadfn("glIndexFormatNV"));
         storage::IndexFuncEXT.set_ptr(loadfn("glIndexFuncEXT"));
         storage::IndexMaterialEXT.set_ptr(loadfn("glIndexMaterialEXT"));
         storage::IndexPointerEXT.set_ptr(loadfn("glIndexPointerEXT"));
         storage::IndexPointerListIBM.set_ptr(loadfn("glIndexPointerListIBM"));
         storage::IndexxOES.set_ptr(loadfn("glIndexxOES"));
         storage::IndexxvOES.set_ptr(loadfn("glIndexxvOES"));
         storage::InsertComponentEXT.set_ptr(loadfn("glInsertComponentEXT"));
         storage::InsertEventMarkerEXT.set_ptr(loadfn("glInsertEventMarkerEXT"));
         storage::InstrumentsBufferSGIX.set_ptr(loadfn("glInstrumentsBufferSGIX"));
         storage::InterpolatePathsNV.set_ptr(loadfn("glInterpolatePathsNV"));
         storage::InvalidateBufferData.set_ptr(loadfn("glInvalidateBufferData"));
         storage::InvalidateBufferSubData.set_ptr(loadfn("glInvalidateBufferSubData"));
         storage::InvalidateFramebuffer.set_ptr(loadfn("glInvalidateFramebuffer"));
         storage::InvalidateNamedFramebufferData.set_ptr(loadfn("glInvalidateNamedFramebufferData"));
         storage::InvalidateNamedFramebufferSubData.set_ptr(loadfn("glInvalidateNamedFramebufferSubData"));
         storage::InvalidateSubFramebuffer.set_ptr(loadfn("glInvalidateSubFramebuffer"));
         storage::InvalidateTexImage.set_ptr(loadfn("glInvalidateTexImage"));
         storage::InvalidateTexSubImage.set_ptr(loadfn("glInvalidateTexSubImage"));
         storage::IsAsyncMarkerSGIX.set_ptr(loadfn("glIsAsyncMarkerSGIX"));
         storage::IsBuffer.set_ptr(loadfn("glIsBuffer"));
         storage::IsBufferARB.set_ptr(loadfn("glIsBufferARB"));
         storage::IsBufferResidentNV.set_ptr(loadfn("glIsBufferResidentNV"));
         storage::IsCommandListNV.set_ptr(loadfn("glIsCommandListNV"));
         storage::IsEnabled.set_ptr(loadfn("glIsEnabled"));
         storage::IsEnabledIndexedEXT.set_ptr(loadfn("glIsEnabledIndexedEXT"));
         storage::IsEnabledi.set_ptr(loadfn("glIsEnabledi"));
         storage::IsFenceAPPLE.set_ptr(loadfn("glIsFenceAPPLE"));
         storage::IsFenceNV.set_ptr(loadfn("glIsFenceNV"));
         storage::IsFramebuffer.set_ptr(loadfn("glIsFramebuffer"));
         storage::IsFramebufferEXT.set_ptr(loadfn("glIsFramebufferEXT"));
         storage::IsImageHandleResidentARB.set_ptr(loadfn("glIsImageHandleResidentARB"));
         storage::IsImageHandleResidentNV.set_ptr(loadfn("glIsImageHandleResidentNV"));
         storage::IsMemoryObjectEXT.set_ptr(loadfn("glIsMemoryObjectEXT"));
         storage::IsNameAMD.set_ptr(loadfn("glIsNameAMD"));
         storage::IsNamedBufferResidentNV.set_ptr(loadfn("glIsNamedBufferResidentNV"));
         storage::IsNamedStringARB.set_ptr(loadfn("glIsNamedStringARB"));
         storage::IsObjectBufferATI.set_ptr(loadfn("glIsObjectBufferATI"));
         storage::IsOcclusionQueryNV.set_ptr(loadfn("glIsOcclusionQueryNV"));
         storage::IsPathNV.set_ptr(loadfn("glIsPathNV"));
         storage::IsPointInFillPathNV.set_ptr(loadfn("glIsPointInFillPathNV"));
         storage::IsPointInStrokePathNV.set_ptr(loadfn("glIsPointInStrokePathNV"));
         storage::IsProgram.set_ptr(loadfn("glIsProgram"));
         storage::IsProgramARB.set_ptr(loadfn("glIsProgramARB"));
         storage::IsProgramNV.set_ptr(loadfn("glIsProgramNV"));
         storage::IsProgramPipeline.set_ptr(loadfn("glIsProgramPipeline"));
         storage::IsQuery.set_ptr(loadfn("glIsQuery"));
         storage::IsQueryARB.set_ptr(loadfn("glIsQueryARB"));
         storage::IsRenderbuffer.set_ptr(loadfn("glIsRenderbuffer"));
         storage::IsRenderbufferEXT.set_ptr(loadfn("glIsRenderbufferEXT"));
         storage::IsSampler.set_ptr(loadfn("glIsSampler"));
         storage::IsSemaphoreEXT.set_ptr(loadfn("glIsSemaphoreEXT"));
         storage::IsShader.set_ptr(loadfn("glIsShader"));
         storage::IsStateNV.set_ptr(loadfn("glIsStateNV"));
         storage::IsSync.set_ptr(loadfn("glIsSync"));
         storage::IsTexture.set_ptr(loadfn("glIsTexture"));
         storage::IsTextureEXT.set_ptr(loadfn("glIsTextureEXT"));
         storage::IsTextureHandleResidentARB.set_ptr(loadfn("glIsTextureHandleResidentARB"));
         storage::IsTextureHandleResidentNV.set_ptr(loadfn("glIsTextureHandleResidentNV"));
         storage::IsTransformFeedback.set_ptr(loadfn("glIsTransformFeedback"));
         storage::IsTransformFeedbackNV.set_ptr(loadfn("glIsTransformFeedbackNV"));
         storage::IsVariantEnabledEXT.set_ptr(loadfn("glIsVariantEnabledEXT"));
         storage::IsVertexArray.set_ptr(loadfn("glIsVertexArray"));
         storage::IsVertexArrayAPPLE.set_ptr(loadfn("glIsVertexArrayAPPLE"));
         storage::IsVertexAttribEnabledAPPLE.set_ptr(loadfn("glIsVertexAttribEnabledAPPLE"));
         storage::LGPUCopyImageSubDataNVX.set_ptr(loadfn("glLGPUCopyImageSubDataNVX"));
         storage::LGPUInterlockNVX.set_ptr(loadfn("glLGPUInterlockNVX"));
         storage::LGPUNamedBufferSubDataNVX.set_ptr(loadfn("glLGPUNamedBufferSubDataNVX"));
         storage::LabelObjectEXT.set_ptr(loadfn("glLabelObjectEXT"));
         storage::LightEnviSGIX.set_ptr(loadfn("glLightEnviSGIX"));
         storage::LightModelxOES.set_ptr(loadfn("glLightModelxOES"));
         storage::LightModelxvOES.set_ptr(loadfn("glLightModelxvOES"));
         storage::LightxOES.set_ptr(loadfn("glLightxOES"));
         storage::LightxvOES.set_ptr(loadfn("glLightxvOES"));
         storage::LineWidth.set_ptr(loadfn("glLineWidth"));
         storage::LineWidthxOES.set_ptr(loadfn("glLineWidthxOES"));
         storage::LinkProgram.set_ptr(loadfn("glLinkProgram"));
         storage::LinkProgramARB.set_ptr(loadfn("glLinkProgramARB"));
         storage::ListDrawCommandsStatesClientNV.set_ptr(loadfn("glListDrawCommandsStatesClientNV"));
         storage::ListParameterfSGIX.set_ptr(loadfn("glListParameterfSGIX"));
         storage::ListParameterfvSGIX.set_ptr(loadfn("glListParameterfvSGIX"));
         storage::ListParameteriSGIX.set_ptr(loadfn("glListParameteriSGIX"));
         storage::ListParameterivSGIX.set_ptr(loadfn("glListParameterivSGIX"));
         storage::LoadIdentityDeformationMapSGIX.set_ptr(loadfn("glLoadIdentityDeformationMapSGIX"));
         storage::LoadMatrixxOES.set_ptr(loadfn("glLoadMatrixxOES"));
         storage::LoadProgramNV.set_ptr(loadfn("glLoadProgramNV"));
         storage::LoadTransposeMatrixdARB.set_ptr(loadfn("glLoadTransposeMatrixdARB"));
         storage::LoadTransposeMatrixfARB.set_ptr(loadfn("glLoadTransposeMatrixfARB"));
         storage::LoadTransposeMatrixxOES.set_ptr(loadfn("glLoadTransposeMatrixxOES"));
         storage::LockArraysEXT.set_ptr(loadfn("glLockArraysEXT"));
         storage::LogicOp.set_ptr(loadfn("glLogicOp"));
         storage::MakeBufferNonResidentNV.set_ptr(loadfn("glMakeBufferNonResidentNV"));
         storage::MakeBufferResidentNV.set_ptr(loadfn("glMakeBufferResidentNV"));
         storage::MakeImageHandleNonResidentARB.set_ptr(loadfn("glMakeImageHandleNonResidentARB"));
         storage::MakeImageHandleNonResidentNV.set_ptr(loadfn("glMakeImageHandleNonResidentNV"));
         storage::MakeImageHandleResidentARB.set_ptr(loadfn("glMakeImageHandleResidentARB"));
         storage::MakeImageHandleResidentNV.set_ptr(loadfn("glMakeImageHandleResidentNV"));
         storage::MakeNamedBufferNonResidentNV.set_ptr(loadfn("glMakeNamedBufferNonResidentNV"));
         storage::MakeNamedBufferResidentNV.set_ptr(loadfn("glMakeNamedBufferResidentNV"));
         storage::MakeTextureHandleNonResidentARB.set_ptr(loadfn("glMakeTextureHandleNonResidentARB"));
         storage::MakeTextureHandleNonResidentNV.set_ptr(loadfn("glMakeTextureHandleNonResidentNV"));
         storage::MakeTextureHandleResidentARB.set_ptr(loadfn("glMakeTextureHandleResidentARB"));
         storage::MakeTextureHandleResidentNV.set_ptr(loadfn("glMakeTextureHandleResidentNV"));
         storage::Map1xOES.set_ptr(loadfn("glMap1xOES"));
         storage::Map2xOES.set_ptr(loadfn("glMap2xOES"));
         storage::MapBuffer.set_ptr(loadfn("glMapBuffer"));
         storage::MapBufferARB.set_ptr(loadfn("glMapBufferARB"));
         storage::MapBufferRange.set_ptr(loadfn("glMapBufferRange"));
         storage::MapControlPointsNV.set_ptr(loadfn("glMapControlPointsNV"));
         storage::MapGrid1xOES.set_ptr(loadfn("glMapGrid1xOES"));
         storage::MapGrid2xOES.set_ptr(loadfn("glMapGrid2xOES"));
         storage::MapNamedBuffer.set_ptr(loadfn("glMapNamedBuffer"));
         storage::MapNamedBufferEXT.set_ptr(loadfn("glMapNamedBufferEXT"));
         storage::MapNamedBufferRange.set_ptr(loadfn("glMapNamedBufferRange"));
         storage::MapNamedBufferRangeEXT.set_ptr(loadfn("glMapNamedBufferRangeEXT"));
         storage::MapObjectBufferATI.set_ptr(loadfn("glMapObjectBufferATI"));
         storage::MapParameterfvNV.set_ptr(loadfn("glMapParameterfvNV"));
         storage::MapParameterivNV.set_ptr(loadfn("glMapParameterivNV"));
         storage::MapTexture2DINTEL.set_ptr(loadfn("glMapTexture2DINTEL"));
         storage::MapVertexAttrib1dAPPLE.set_ptr(loadfn("glMapVertexAttrib1dAPPLE"));
         storage::MapVertexAttrib1fAPPLE.set_ptr(loadfn("glMapVertexAttrib1fAPPLE"));
         storage::MapVertexAttrib2dAPPLE.set_ptr(loadfn("glMapVertexAttrib2dAPPLE"));
         storage::MapVertexAttrib2fAPPLE.set_ptr(loadfn("glMapVertexAttrib2fAPPLE"));
         storage::MaterialxOES.set_ptr(loadfn("glMaterialxOES"));
         storage::MaterialxvOES.set_ptr(loadfn("glMaterialxvOES"));
         storage::MatrixFrustumEXT.set_ptr(loadfn("glMatrixFrustumEXT"));
         storage::MatrixIndexPointerARB.set_ptr(loadfn("glMatrixIndexPointerARB"));
         storage::MatrixIndexubvARB.set_ptr(loadfn("glMatrixIndexubvARB"));
         storage::MatrixIndexuivARB.set_ptr(loadfn("glMatrixIndexuivARB"));
         storage::MatrixIndexusvARB.set_ptr(loadfn("glMatrixIndexusvARB"));
         storage::MatrixLoad3x2fNV.set_ptr(loadfn("glMatrixLoad3x2fNV"));
         storage::MatrixLoad3x3fNV.set_ptr(loadfn("glMatrixLoad3x3fNV"));
         storage::MatrixLoadIdentityEXT.set_ptr(loadfn("glMatrixLoadIdentityEXT"));
         storage::MatrixLoadTranspose3x3fNV.set_ptr(loadfn("glMatrixLoadTranspose3x3fNV"));
         storage::MatrixLoadTransposedEXT.set_ptr(loadfn("glMatrixLoadTransposedEXT"));
         storage::MatrixLoadTransposefEXT.set_ptr(loadfn("glMatrixLoadTransposefEXT"));
         storage::MatrixLoaddEXT.set_ptr(loadfn("glMatrixLoaddEXT"));
         storage::MatrixLoadfEXT.set_ptr(loadfn("glMatrixLoadfEXT"));
         storage::MatrixMult3x2fNV.set_ptr(loadfn("glMatrixMult3x2fNV"));
         storage::MatrixMult3x3fNV.set_ptr(loadfn("glMatrixMult3x3fNV"));
         storage::MatrixMultTranspose3x3fNV.set_ptr(loadfn("glMatrixMultTranspose3x3fNV"));
         storage::MatrixMultTransposedEXT.set_ptr(loadfn("glMatrixMultTransposedEXT"));
         storage::MatrixMultTransposefEXT.set_ptr(loadfn("glMatrixMultTransposefEXT"));
         storage::MatrixMultdEXT.set_ptr(loadfn("glMatrixMultdEXT"));
         storage::MatrixMultfEXT.set_ptr(loadfn("glMatrixMultfEXT"));
         storage::MatrixOrthoEXT.set_ptr(loadfn("glMatrixOrthoEXT"));
         storage::MatrixPopEXT.set_ptr(loadfn("glMatrixPopEXT"));
         storage::MatrixPushEXT.set_ptr(loadfn("glMatrixPushEXT"));
         storage::MatrixRotatedEXT.set_ptr(loadfn("glMatrixRotatedEXT"));
         storage::MatrixRotatefEXT.set_ptr(loadfn("glMatrixRotatefEXT"));
         storage::MatrixScaledEXT.set_ptr(loadfn("glMatrixScaledEXT"));
         storage::MatrixScalefEXT.set_ptr(loadfn("glMatrixScalefEXT"));
         storage::MatrixTranslatedEXT.set_ptr(loadfn("glMatrixTranslatedEXT"));
         storage::MatrixTranslatefEXT.set_ptr(loadfn("glMatrixTranslatefEXT"));
         storage::MaxShaderCompilerThreadsARB.set_ptr(loadfn("glMaxShaderCompilerThreadsARB"));
         storage::MaxShaderCompilerThreadsKHR.set_ptr(loadfn("glMaxShaderCompilerThreadsKHR"));
         storage::MemoryBarrier.set_ptr(loadfn("glMemoryBarrier"));
         storage::MemoryBarrierByRegion.set_ptr(loadfn("glMemoryBarrierByRegion"));
         storage::MemoryBarrierEXT.set_ptr(loadfn("glMemoryBarrierEXT"));
         storage::MemoryObjectParameterivEXT.set_ptr(loadfn("glMemoryObjectParameterivEXT"));
         storage::MinSampleShading.set_ptr(loadfn("glMinSampleShading"));
         storage::MinSampleShadingARB.set_ptr(loadfn("glMinSampleShadingARB"));
         storage::MinmaxEXT.set_ptr(loadfn("glMinmaxEXT"));
         storage::MultMatrixxOES.set_ptr(loadfn("glMultMatrixxOES"));
         storage::MultTransposeMatrixdARB.set_ptr(loadfn("glMultTransposeMatrixdARB"));
         storage::MultTransposeMatrixfARB.set_ptr(loadfn("glMultTransposeMatrixfARB"));
         storage::MultTransposeMatrixxOES.set_ptr(loadfn("glMultTransposeMatrixxOES"));
         storage::MultiDrawArrays.set_ptr(loadfn("glMultiDrawArrays"));
         storage::MultiDrawArraysEXT.set_ptr(loadfn("glMultiDrawArraysEXT"));
         storage::MultiDrawArraysIndirect.set_ptr(loadfn("glMultiDrawArraysIndirect"));
         storage::MultiDrawArraysIndirectAMD.set_ptr(loadfn("glMultiDrawArraysIndirectAMD"));
         storage::MultiDrawArraysIndirectBindlessCountNV.set_ptr(loadfn("glMultiDrawArraysIndirectBindlessCountNV"));
         storage::MultiDrawArraysIndirectBindlessNV.set_ptr(loadfn("glMultiDrawArraysIndirectBindlessNV"));
         storage::MultiDrawArraysIndirectCount.set_ptr(loadfn("glMultiDrawArraysIndirectCount"));
         storage::MultiDrawArraysIndirectCountARB.set_ptr(loadfn("glMultiDrawArraysIndirectCountARB"));
         storage::MultiDrawElementArrayAPPLE.set_ptr(loadfn("glMultiDrawElementArrayAPPLE"));
         storage::MultiDrawElements.set_ptr(loadfn("glMultiDrawElements"));
         storage::MultiDrawElementsBaseVertex.set_ptr(loadfn("glMultiDrawElementsBaseVertex"));
         storage::MultiDrawElementsEXT.set_ptr(loadfn("glMultiDrawElementsEXT"));
         storage::MultiDrawElementsIndirect.set_ptr(loadfn("glMultiDrawElementsIndirect"));
         storage::MultiDrawElementsIndirectAMD.set_ptr(loadfn("glMultiDrawElementsIndirectAMD"));
         storage::MultiDrawElementsIndirectBindlessCountNV.set_ptr(loadfn("glMultiDrawElementsIndirectBindlessCountNV"));
         storage::MultiDrawElementsIndirectBindlessNV.set_ptr(loadfn("glMultiDrawElementsIndirectBindlessNV"));
         storage::MultiDrawElementsIndirectCount.set_ptr(loadfn("glMultiDrawElementsIndirectCount"));
         storage::MultiDrawElementsIndirectCountARB.set_ptr(loadfn("glMultiDrawElementsIndirectCountARB"));
         storage::MultiDrawMeshTasksIndirectCountNV.set_ptr(loadfn("glMultiDrawMeshTasksIndirectCountNV"));
         storage::MultiDrawMeshTasksIndirectNV.set_ptr(loadfn("glMultiDrawMeshTasksIndirectNV"));
         storage::MultiDrawRangeElementArrayAPPLE.set_ptr(loadfn("glMultiDrawRangeElementArrayAPPLE"));
         storage::MultiModeDrawArraysIBM.set_ptr(loadfn("glMultiModeDrawArraysIBM"));
         storage::MultiModeDrawElementsIBM.set_ptr(loadfn("glMultiModeDrawElementsIBM"));
         storage::MultiTexBufferEXT.set_ptr(loadfn("glMultiTexBufferEXT"));
         storage::MultiTexCoord1bOES.set_ptr(loadfn("glMultiTexCoord1bOES"));
         storage::MultiTexCoord1bvOES.set_ptr(loadfn("glMultiTexCoord1bvOES"));
         storage::MultiTexCoord1dARB.set_ptr(loadfn("glMultiTexCoord1dARB"));
         storage::MultiTexCoord1dvARB.set_ptr(loadfn("glMultiTexCoord1dvARB"));
         storage::MultiTexCoord1fARB.set_ptr(loadfn("glMultiTexCoord1fARB"));
         storage::MultiTexCoord1fvARB.set_ptr(loadfn("glMultiTexCoord1fvARB"));
         storage::MultiTexCoord1hNV.set_ptr(loadfn("glMultiTexCoord1hNV"));
         storage::MultiTexCoord1hvNV.set_ptr(loadfn("glMultiTexCoord1hvNV"));
         storage::MultiTexCoord1iARB.set_ptr(loadfn("glMultiTexCoord1iARB"));
         storage::MultiTexCoord1ivARB.set_ptr(loadfn("glMultiTexCoord1ivARB"));
         storage::MultiTexCoord1sARB.set_ptr(loadfn("glMultiTexCoord1sARB"));
         storage::MultiTexCoord1svARB.set_ptr(loadfn("glMultiTexCoord1svARB"));
         storage::MultiTexCoord1xOES.set_ptr(loadfn("glMultiTexCoord1xOES"));
         storage::MultiTexCoord1xvOES.set_ptr(loadfn("glMultiTexCoord1xvOES"));
         storage::MultiTexCoord2bOES.set_ptr(loadfn("glMultiTexCoord2bOES"));
         storage::MultiTexCoord2bvOES.set_ptr(loadfn("glMultiTexCoord2bvOES"));
         storage::MultiTexCoord2dARB.set_ptr(loadfn("glMultiTexCoord2dARB"));
         storage::MultiTexCoord2dvARB.set_ptr(loadfn("glMultiTexCoord2dvARB"));
         storage::MultiTexCoord2fARB.set_ptr(loadfn("glMultiTexCoord2fARB"));
         storage::MultiTexCoord2fvARB.set_ptr(loadfn("glMultiTexCoord2fvARB"));
         storage::MultiTexCoord2hNV.set_ptr(loadfn("glMultiTexCoord2hNV"));
         storage::MultiTexCoord2hvNV.set_ptr(loadfn("glMultiTexCoord2hvNV"));
         storage::MultiTexCoord2iARB.set_ptr(loadfn("glMultiTexCoord2iARB"));
         storage::MultiTexCoord2ivARB.set_ptr(loadfn("glMultiTexCoord2ivARB"));
         storage::MultiTexCoord2sARB.set_ptr(loadfn("glMultiTexCoord2sARB"));
         storage::MultiTexCoord2svARB.set_ptr(loadfn("glMultiTexCoord2svARB"));
         storage::MultiTexCoord2xOES.set_ptr(loadfn("glMultiTexCoord2xOES"));
         storage::MultiTexCoord2xvOES.set_ptr(loadfn("glMultiTexCoord2xvOES"));
         storage::MultiTexCoord3bOES.set_ptr(loadfn("glMultiTexCoord3bOES"));
         storage::MultiTexCoord3bvOES.set_ptr(loadfn("glMultiTexCoord3bvOES"));
         storage::MultiTexCoord3dARB.set_ptr(loadfn("glMultiTexCoord3dARB"));
         storage::MultiTexCoord3dvARB.set_ptr(loadfn("glMultiTexCoord3dvARB"));
         storage::MultiTexCoord3fARB.set_ptr(loadfn("glMultiTexCoord3fARB"));
         storage::MultiTexCoord3fvARB.set_ptr(loadfn("glMultiTexCoord3fvARB"));
         storage::MultiTexCoord3hNV.set_ptr(loadfn("glMultiTexCoord3hNV"));
         storage::MultiTexCoord3hvNV.set_ptr(loadfn("glMultiTexCoord3hvNV"));
         storage::MultiTexCoord3iARB.set_ptr(loadfn("glMultiTexCoord3iARB"));
         storage::MultiTexCoord3ivARB.set_ptr(loadfn("glMultiTexCoord3ivARB"));
         storage::MultiTexCoord3sARB.set_ptr(loadfn("glMultiTexCoord3sARB"));
         storage::MultiTexCoord3svARB.set_ptr(loadfn("glMultiTexCoord3svARB"));
         storage::MultiTexCoord3xOES.set_ptr(loadfn("glMultiTexCoord3xOES"));
         storage::MultiTexCoord3xvOES.set_ptr(loadfn("glMultiTexCoord3xvOES"));
         storage::MultiTexCoord4bOES.set_ptr(loadfn("glMultiTexCoord4bOES"));
         storage::MultiTexCoord4bvOES.set_ptr(loadfn("glMultiTexCoord4bvOES"));
         storage::MultiTexCoord4dARB.set_ptr(loadfn("glMultiTexCoord4dARB"));
         storage::MultiTexCoord4dvARB.set_ptr(loadfn("glMultiTexCoord4dvARB"));
         storage::MultiTexCoord4fARB.set_ptr(loadfn("glMultiTexCoord4fARB"));
         storage::MultiTexCoord4fvARB.set_ptr(loadfn("glMultiTexCoord4fvARB"));
         storage::MultiTexCoord4hNV.set_ptr(loadfn("glMultiTexCoord4hNV"));
         storage::MultiTexCoord4hvNV.set_ptr(loadfn("glMultiTexCoord4hvNV"));
         storage::MultiTexCoord4iARB.set_ptr(loadfn("glMultiTexCoord4iARB"));
         storage::MultiTexCoord4ivARB.set_ptr(loadfn("glMultiTexCoord4ivARB"));
         storage::MultiTexCoord4sARB.set_ptr(loadfn("glMultiTexCoord4sARB"));
         storage::MultiTexCoord4svARB.set_ptr(loadfn("glMultiTexCoord4svARB"));
         storage::MultiTexCoord4xOES.set_ptr(loadfn("glMultiTexCoord4xOES"));
         storage::MultiTexCoord4xvOES.set_ptr(loadfn("glMultiTexCoord4xvOES"));
         storage::MultiTexCoordPointerEXT.set_ptr(loadfn("glMultiTexCoordPointerEXT"));
         storage::MultiTexEnvfEXT.set_ptr(loadfn("glMultiTexEnvfEXT"));
         storage::MultiTexEnvfvEXT.set_ptr(loadfn("glMultiTexEnvfvEXT"));
         storage::MultiTexEnviEXT.set_ptr(loadfn("glMultiTexEnviEXT"));
         storage::MultiTexEnvivEXT.set_ptr(loadfn("glMultiTexEnvivEXT"));
         storage::MultiTexGendEXT.set_ptr(loadfn("glMultiTexGendEXT"));
         storage::MultiTexGendvEXT.set_ptr(loadfn("glMultiTexGendvEXT"));
         storage::MultiTexGenfEXT.set_ptr(loadfn("glMultiTexGenfEXT"));
         storage::MultiTexGenfvEXT.set_ptr(loadfn("glMultiTexGenfvEXT"));
         storage::MultiTexGeniEXT.set_ptr(loadfn("glMultiTexGeniEXT"));
         storage::MultiTexGenivEXT.set_ptr(loadfn("glMultiTexGenivEXT"));
         storage::MultiTexImage1DEXT.set_ptr(loadfn("glMultiTexImage1DEXT"));
         storage::MultiTexImage2DEXT.set_ptr(loadfn("glMultiTexImage2DEXT"));
         storage::MultiTexImage3DEXT.set_ptr(loadfn("glMultiTexImage3DEXT"));
         storage::MultiTexParameterIivEXT.set_ptr(loadfn("glMultiTexParameterIivEXT"));
         storage::MultiTexParameterIuivEXT.set_ptr(loadfn("glMultiTexParameterIuivEXT"));
         storage::MultiTexParameterfEXT.set_ptr(loadfn("glMultiTexParameterfEXT"));
         storage::MultiTexParameterfvEXT.set_ptr(loadfn("glMultiTexParameterfvEXT"));
         storage::MultiTexParameteriEXT.set_ptr(loadfn("glMultiTexParameteriEXT"));
         storage::MultiTexParameterivEXT.set_ptr(loadfn("glMultiTexParameterivEXT"));
         storage::MultiTexRenderbufferEXT.set_ptr(loadfn("glMultiTexRenderbufferEXT"));
         storage::MultiTexSubImage1DEXT.set_ptr(loadfn("glMultiTexSubImage1DEXT"));
         storage::MultiTexSubImage2DEXT.set_ptr(loadfn("glMultiTexSubImage2DEXT"));
         storage::MultiTexSubImage3DEXT.set_ptr(loadfn("glMultiTexSubImage3DEXT"));
         storage::MulticastBarrierNV.set_ptr(loadfn("glMulticastBarrierNV"));
         storage::MulticastBlitFramebufferNV.set_ptr(loadfn("glMulticastBlitFramebufferNV"));
         storage::MulticastBufferSubDataNV.set_ptr(loadfn("glMulticastBufferSubDataNV"));
         storage::MulticastCopyBufferSubDataNV.set_ptr(loadfn("glMulticastCopyBufferSubDataNV"));
         storage::MulticastCopyImageSubDataNV.set_ptr(loadfn("glMulticastCopyImageSubDataNV"));
         storage::MulticastFramebufferSampleLocationsfvNV.set_ptr(loadfn("glMulticastFramebufferSampleLocationsfvNV"));
         storage::MulticastGetQueryObjecti64vNV.set_ptr(loadfn("glMulticastGetQueryObjecti64vNV"));
         storage::MulticastGetQueryObjectivNV.set_ptr(loadfn("glMulticastGetQueryObjectivNV"));
         storage::MulticastGetQueryObjectui64vNV.set_ptr(loadfn("glMulticastGetQueryObjectui64vNV"));
         storage::MulticastGetQueryObjectuivNV.set_ptr(loadfn("glMulticastGetQueryObjectuivNV"));
         storage::MulticastScissorArrayvNVX.set_ptr(loadfn("glMulticastScissorArrayvNVX"));
         storage::MulticastViewportArrayvNVX.set_ptr(loadfn("glMulticastViewportArrayvNVX"));
         storage::MulticastViewportPositionWScaleNVX.set_ptr(loadfn("glMulticastViewportPositionWScaleNVX"));
         storage::MulticastWaitSyncNV.set_ptr(loadfn("glMulticastWaitSyncNV"));
         storage::NamedBufferAttachMemoryNV.set_ptr(loadfn("glNamedBufferAttachMemoryNV"));
         storage::NamedBufferData.set_ptr(loadfn("glNamedBufferData"));
         storage::NamedBufferDataEXT.set_ptr(loadfn("glNamedBufferDataEXT"));
         storage::NamedBufferPageCommitmentARB.set_ptr(loadfn("glNamedBufferPageCommitmentARB"));
         storage::NamedBufferPageCommitmentEXT.set_ptr(loadfn("glNamedBufferPageCommitmentEXT"));
         storage::NamedBufferPageCommitmentMemNV.set_ptr(loadfn("glNamedBufferPageCommitmentMemNV"));
         storage::NamedBufferStorage.set_ptr(loadfn("glNamedBufferStorage"));
         storage::NamedBufferStorageEXT.set_ptr(loadfn("glNamedBufferStorageEXT"));
         storage::NamedBufferStorageExternalEXT.set_ptr(loadfn("glNamedBufferStorageExternalEXT"));
         storage::NamedBufferStorageMemEXT.set_ptr(loadfn("glNamedBufferStorageMemEXT"));
         storage::NamedBufferSubData.set_ptr(loadfn("glNamedBufferSubData"));
         storage::NamedBufferSubDataEXT.set_ptr(loadfn("glNamedBufferSubDataEXT"));
         storage::NamedCopyBufferSubDataEXT.set_ptr(loadfn("glNamedCopyBufferSubDataEXT"));
         storage::NamedFramebufferDrawBuffer.set_ptr(loadfn("glNamedFramebufferDrawBuffer"));
         storage::NamedFramebufferDrawBuffers.set_ptr(loadfn("glNamedFramebufferDrawBuffers"));
         storage::NamedFramebufferParameteri.set_ptr(loadfn("glNamedFramebufferParameteri"));
         storage::NamedFramebufferParameteriEXT.set_ptr(loadfn("glNamedFramebufferParameteriEXT"));
         storage::NamedFramebufferReadBuffer.set_ptr(loadfn("glNamedFramebufferReadBuffer"));
         storage::NamedFramebufferRenderbuffer.set_ptr(loadfn("glNamedFramebufferRenderbuffer"));
         storage::NamedFramebufferRenderbufferEXT.set_ptr(loadfn("glNamedFramebufferRenderbufferEXT"));
         storage::NamedFramebufferSampleLocationsfvARB.set_ptr(loadfn("glNamedFramebufferSampleLocationsfvARB"));
         storage::NamedFramebufferSampleLocationsfvNV.set_ptr(loadfn("glNamedFramebufferSampleLocationsfvNV"));
         storage::NamedFramebufferSamplePositionsfvAMD.set_ptr(loadfn("glNamedFramebufferSamplePositionsfvAMD"));
         storage::NamedFramebufferTexture.set_ptr(loadfn("glNamedFramebufferTexture"));
         storage::NamedFramebufferTexture1DEXT.set_ptr(loadfn("glNamedFramebufferTexture1DEXT"));
         storage::NamedFramebufferTexture2DEXT.set_ptr(loadfn("glNamedFramebufferTexture2DEXT"));
         storage::NamedFramebufferTexture3DEXT.set_ptr(loadfn("glNamedFramebufferTexture3DEXT"));
         storage::NamedFramebufferTextureEXT.set_ptr(loadfn("glNamedFramebufferTextureEXT"));
         storage::NamedFramebufferTextureFaceEXT.set_ptr(loadfn("glNamedFramebufferTextureFaceEXT"));
         storage::NamedFramebufferTextureLayer.set_ptr(loadfn("glNamedFramebufferTextureLayer"));
         storage::NamedFramebufferTextureLayerEXT.set_ptr(loadfn("glNamedFramebufferTextureLayerEXT"));
         storage::NamedProgramLocalParameter4dEXT.set_ptr(loadfn("glNamedProgramLocalParameter4dEXT"));
         storage::NamedProgramLocalParameter4dvEXT.set_ptr(loadfn("glNamedProgramLocalParameter4dvEXT"));
         storage::NamedProgramLocalParameter4fEXT.set_ptr(loadfn("glNamedProgramLocalParameter4fEXT"));
         storage::NamedProgramLocalParameter4fvEXT.set_ptr(loadfn("glNamedProgramLocalParameter4fvEXT"));
         storage::NamedProgramLocalParameterI4iEXT.set_ptr(loadfn("glNamedProgramLocalParameterI4iEXT"));
         storage::NamedProgramLocalParameterI4ivEXT.set_ptr(loadfn("glNamedProgramLocalParameterI4ivEXT"));
         storage::NamedProgramLocalParameterI4uiEXT.set_ptr(loadfn("glNamedProgramLocalParameterI4uiEXT"));
         storage::NamedProgramLocalParameterI4uivEXT.set_ptr(loadfn("glNamedProgramLocalParameterI4uivEXT"));
         storage::NamedProgramLocalParameters4fvEXT.set_ptr(loadfn("glNamedProgramLocalParameters4fvEXT"));
         storage::NamedProgramLocalParametersI4ivEXT.set_ptr(loadfn("glNamedProgramLocalParametersI4ivEXT"));
         storage::NamedProgramLocalParametersI4uivEXT.set_ptr(loadfn("glNamedProgramLocalParametersI4uivEXT"));
         storage::NamedProgramStringEXT.set_ptr(loadfn("glNamedProgramStringEXT"));
         storage::NamedRenderbufferStorage.set_ptr(loadfn("glNamedRenderbufferStorage"));
         storage::NamedRenderbufferStorageEXT.set_ptr(loadfn("glNamedRenderbufferStorageEXT"));
         storage::NamedRenderbufferStorageMultisample.set_ptr(loadfn("glNamedRenderbufferStorageMultisample"));
         storage::NamedRenderbufferStorageMultisampleAdvancedAMD.set_ptr(loadfn("glNamedRenderbufferStorageMultisampleAdvancedAMD"));
         storage::NamedRenderbufferStorageMultisampleCoverageEXT.set_ptr(loadfn("glNamedRenderbufferStorageMultisampleCoverageEXT"));
         storage::NamedRenderbufferStorageMultisampleEXT.set_ptr(loadfn("glNamedRenderbufferStorageMultisampleEXT"));
         storage::NamedStringARB.set_ptr(loadfn("glNamedStringARB"));
         storage::NewObjectBufferATI.set_ptr(loadfn("glNewObjectBufferATI"));
         storage::Normal3fVertex3fSUN.set_ptr(loadfn("glNormal3fVertex3fSUN"));
         storage::Normal3fVertex3fvSUN.set_ptr(loadfn("glNormal3fVertex3fvSUN"));
         storage::Normal3hNV.set_ptr(loadfn("glNormal3hNV"));
         storage::Normal3hvNV.set_ptr(loadfn("glNormal3hvNV"));
         storage::Normal3xOES.set_ptr(loadfn("glNormal3xOES"));
         storage::Normal3xvOES.set_ptr(loadfn("glNormal3xvOES"));
         storage::NormalFormatNV.set_ptr(loadfn("glNormalFormatNV"));
         storage::NormalPointerEXT.set_ptr(loadfn("glNormalPointerEXT"));
         storage::NormalPointerListIBM.set_ptr(loadfn("glNormalPointerListIBM"));
         storage::NormalPointervINTEL.set_ptr(loadfn("glNormalPointervINTEL"));
         storage::NormalStream3bATI.set_ptr(loadfn("glNormalStream3bATI"));
         storage::NormalStream3bvATI.set_ptr(loadfn("glNormalStream3bvATI"));
         storage::NormalStream3dATI.set_ptr(loadfn("glNormalStream3dATI"));
         storage::NormalStream3dvATI.set_ptr(loadfn("glNormalStream3dvATI"));
         storage::NormalStream3fATI.set_ptr(loadfn("glNormalStream3fATI"));
         storage::NormalStream3fvATI.set_ptr(loadfn("glNormalStream3fvATI"));
         storage::NormalStream3iATI.set_ptr(loadfn("glNormalStream3iATI"));
         storage::NormalStream3ivATI.set_ptr(loadfn("glNormalStream3ivATI"));
         storage::NormalStream3sATI.set_ptr(loadfn("glNormalStream3sATI"));
         storage::NormalStream3svATI.set_ptr(loadfn("glNormalStream3svATI"));
         storage::ObjectLabel.set_ptr(loadfn("glObjectLabel"));
         storage::ObjectPtrLabel.set_ptr(loadfn("glObjectPtrLabel"));
         storage::ObjectPurgeableAPPLE.set_ptr(loadfn("glObjectPurgeableAPPLE"));
         storage::ObjectUnpurgeableAPPLE.set_ptr(loadfn("glObjectUnpurgeableAPPLE"));
         storage::OrthofOES.set_ptr(loadfn("glOrthofOES"));
         storage::OrthoxOES.set_ptr(loadfn("glOrthoxOES"));
         storage::PNTrianglesfATI.set_ptr(loadfn("glPNTrianglesfATI"));
         storage::PNTrianglesiATI.set_ptr(loadfn("glPNTrianglesiATI"));
         storage::PassTexCoordATI.set_ptr(loadfn("glPassTexCoordATI"));
         storage::PassThroughxOES.set_ptr(loadfn("glPassThroughxOES"));
         storage::PatchParameterfv.set_ptr(loadfn("glPatchParameterfv"));
         storage::PatchParameteri.set_ptr(loadfn("glPatchParameteri"));
         storage::PathCommandsNV.set_ptr(loadfn("glPathCommandsNV"));
         storage::PathCoordsNV.set_ptr(loadfn("glPathCoordsNV"));
         storage::PathCoverDepthFuncNV.set_ptr(loadfn("glPathCoverDepthFuncNV"));
         storage::PathDashArrayNV.set_ptr(loadfn("glPathDashArrayNV"));
         storage::PathGlyphIndexArrayNV.set_ptr(loadfn("glPathGlyphIndexArrayNV"));
         storage::PathGlyphIndexRangeNV.set_ptr(loadfn("glPathGlyphIndexRangeNV"));
         storage::PathGlyphRangeNV.set_ptr(loadfn("glPathGlyphRangeNV"));
         storage::PathGlyphsNV.set_ptr(loadfn("glPathGlyphsNV"));
         storage::PathMemoryGlyphIndexArrayNV.set_ptr(loadfn("glPathMemoryGlyphIndexArrayNV"));
         storage::PathParameterfNV.set_ptr(loadfn("glPathParameterfNV"));
         storage::PathParameterfvNV.set_ptr(loadfn("glPathParameterfvNV"));
         storage::PathParameteriNV.set_ptr(loadfn("glPathParameteriNV"));
         storage::PathParameterivNV.set_ptr(loadfn("glPathParameterivNV"));
         storage::PathStencilDepthOffsetNV.set_ptr(loadfn("glPathStencilDepthOffsetNV"));
         storage::PathStencilFuncNV.set_ptr(loadfn("glPathStencilFuncNV"));
         storage::PathStringNV.set_ptr(loadfn("glPathStringNV"));
         storage::PathSubCommandsNV.set_ptr(loadfn("glPathSubCommandsNV"));
         storage::PathSubCoordsNV.set_ptr(loadfn("glPathSubCoordsNV"));
         storage::PauseTransformFeedback.set_ptr(loadfn("glPauseTransformFeedback"));
         storage::PauseTransformFeedbackNV.set_ptr(loadfn("glPauseTransformFeedbackNV"));
         storage::PixelDataRangeNV.set_ptr(loadfn("glPixelDataRangeNV"));
         storage::PixelMapx.set_ptr(loadfn("glPixelMapx"));
         storage::PixelStoref.set_ptr(loadfn("glPixelStoref"));
         storage::PixelStorei.set_ptr(loadfn("glPixelStorei"));
         storage::PixelStorex.set_ptr(loadfn("glPixelStorex"));
         storage::PixelTexGenParameterfSGIS.set_ptr(loadfn("glPixelTexGenParameterfSGIS"));
         storage::PixelTexGenParameterfvSGIS.set_ptr(loadfn("glPixelTexGenParameterfvSGIS"));
         storage::PixelTexGenParameteriSGIS.set_ptr(loadfn("glPixelTexGenParameteriSGIS"));
         storage::PixelTexGenParameterivSGIS.set_ptr(loadfn("glPixelTexGenParameterivSGIS"));
         storage::PixelTexGenSGIX.set_ptr(loadfn("glPixelTexGenSGIX"));
         storage::PixelTransferxOES.set_ptr(loadfn("glPixelTransferxOES"));
         storage::PixelTransformParameterfEXT.set_ptr(loadfn("glPixelTransformParameterfEXT"));
         storage::PixelTransformParameterfvEXT.set_ptr(loadfn("glPixelTransformParameterfvEXT"));
         storage::PixelTransformParameteriEXT.set_ptr(loadfn("glPixelTransformParameteriEXT"));
         storage::PixelTransformParameterivEXT.set_ptr(loadfn("glPixelTransformParameterivEXT"));
         storage::PixelZoomxOES.set_ptr(loadfn("glPixelZoomxOES"));
         storage::PointAlongPathNV.set_ptr(loadfn("glPointAlongPathNV"));
         storage::PointParameterf.set_ptr(loadfn("glPointParameterf"));
         storage::PointParameterfARB.set_ptr(loadfn("glPointParameterfARB"));
         storage::PointParameterfEXT.set_ptr(loadfn("glPointParameterfEXT"));
         storage::PointParameterfSGIS.set_ptr(loadfn("glPointParameterfSGIS"));
         storage::PointParameterfv.set_ptr(loadfn("glPointParameterfv"));
         storage::PointParameterfvARB.set_ptr(loadfn("glPointParameterfvARB"));
         storage::PointParameterfvEXT.set_ptr(loadfn("glPointParameterfvEXT"));
         storage::PointParameterfvSGIS.set_ptr(loadfn("glPointParameterfvSGIS"));
         storage::PointParameteri.set_ptr(loadfn("glPointParameteri"));
         storage::PointParameteriNV.set_ptr(loadfn("glPointParameteriNV"));
         storage::PointParameteriv.set_ptr(loadfn("glPointParameteriv"));
         storage::PointParameterivNV.set_ptr(loadfn("glPointParameterivNV"));
         storage::PointParameterxvOES.set_ptr(loadfn("glPointParameterxvOES"));
         storage::PointSize.set_ptr(loadfn("glPointSize"));
         storage::PointSizexOES.set_ptr(loadfn("glPointSizexOES"));
         storage::PollAsyncSGIX.set_ptr(loadfn("glPollAsyncSGIX"));
         storage::PollInstrumentsSGIX.set_ptr(loadfn("glPollInstrumentsSGIX"));
         storage::PolygonMode.set_ptr(loadfn("glPolygonMode"));
         storage::PolygonOffset.set_ptr(loadfn("glPolygonOffset"));
         storage::PolygonOffsetClamp.set_ptr(loadfn("glPolygonOffsetClamp"));
         storage::PolygonOffsetClampEXT.set_ptr(loadfn("glPolygonOffsetClampEXT"));
         storage::PolygonOffsetEXT.set_ptr(loadfn("glPolygonOffsetEXT"));
         storage::PolygonOffsetxOES.set_ptr(loadfn("glPolygonOffsetxOES"));
         storage::PopDebugGroup.set_ptr(loadfn("glPopDebugGroup"));
         storage::PopGroupMarkerEXT.set_ptr(loadfn("glPopGroupMarkerEXT"));
         storage::PresentFrameDualFillNV.set_ptr(loadfn("glPresentFrameDualFillNV"));
         storage::PresentFrameKeyedNV.set_ptr(loadfn("glPresentFrameKeyedNV"));
         storage::PrimitiveBoundingBoxARB.set_ptr(loadfn("glPrimitiveBoundingBoxARB"));
         storage::PrimitiveRestartIndex.set_ptr(loadfn("glPrimitiveRestartIndex"));
         storage::PrimitiveRestartIndexNV.set_ptr(loadfn("glPrimitiveRestartIndexNV"));
         storage::PrimitiveRestartNV.set_ptr(loadfn("glPrimitiveRestartNV"));
         storage::PrioritizeTexturesEXT.set_ptr(loadfn("glPrioritizeTexturesEXT"));
         storage::PrioritizeTexturesxOES.set_ptr(loadfn("glPrioritizeTexturesxOES"));
         storage::ProgramBinary.set_ptr(loadfn("glProgramBinary"));
         storage::ProgramBufferParametersIivNV.set_ptr(loadfn("glProgramBufferParametersIivNV"));
         storage::ProgramBufferParametersIuivNV.set_ptr(loadfn("glProgramBufferParametersIuivNV"));
         storage::ProgramBufferParametersfvNV.set_ptr(loadfn("glProgramBufferParametersfvNV"));
         storage::ProgramEnvParameter4dARB.set_ptr(loadfn("glProgramEnvParameter4dARB"));
         storage::ProgramEnvParameter4dvARB.set_ptr(loadfn("glProgramEnvParameter4dvARB"));
         storage::ProgramEnvParameter4fARB.set_ptr(loadfn("glProgramEnvParameter4fARB"));
         storage::ProgramEnvParameter4fvARB.set_ptr(loadfn("glProgramEnvParameter4fvARB"));
         storage::ProgramEnvParameterI4iNV.set_ptr(loadfn("glProgramEnvParameterI4iNV"));
         storage::ProgramEnvParameterI4ivNV.set_ptr(loadfn("glProgramEnvParameterI4ivNV"));
         storage::ProgramEnvParameterI4uiNV.set_ptr(loadfn("glProgramEnvParameterI4uiNV"));
         storage::ProgramEnvParameterI4uivNV.set_ptr(loadfn("glProgramEnvParameterI4uivNV"));
         storage::ProgramEnvParameters4fvEXT.set_ptr(loadfn("glProgramEnvParameters4fvEXT"));
         storage::ProgramEnvParametersI4ivNV.set_ptr(loadfn("glProgramEnvParametersI4ivNV"));
         storage::ProgramEnvParametersI4uivNV.set_ptr(loadfn("glProgramEnvParametersI4uivNV"));
         storage::ProgramLocalParameter4dARB.set_ptr(loadfn("glProgramLocalParameter4dARB"));
         storage::ProgramLocalParameter4dvARB.set_ptr(loadfn("glProgramLocalParameter4dvARB"));
         storage::ProgramLocalParameter4fARB.set_ptr(loadfn("glProgramLocalParameter4fARB"));
         storage::ProgramLocalParameter4fvARB.set_ptr(loadfn("glProgramLocalParameter4fvARB"));
         storage::ProgramLocalParameterI4iNV.set_ptr(loadfn("glProgramLocalParameterI4iNV"));
         storage::ProgramLocalParameterI4ivNV.set_ptr(loadfn("glProgramLocalParameterI4ivNV"));
         storage::ProgramLocalParameterI4uiNV.set_ptr(loadfn("glProgramLocalParameterI4uiNV"));
         storage::ProgramLocalParameterI4uivNV.set_ptr(loadfn("glProgramLocalParameterI4uivNV"));
         storage::ProgramLocalParameters4fvEXT.set_ptr(loadfn("glProgramLocalParameters4fvEXT"));
         storage::ProgramLocalParametersI4ivNV.set_ptr(loadfn("glProgramLocalParametersI4ivNV"));
         storage::ProgramLocalParametersI4uivNV.set_ptr(loadfn("glProgramLocalParametersI4uivNV"));
         storage::ProgramNamedParameter4dNV.set_ptr(loadfn("glProgramNamedParameter4dNV"));
         storage::ProgramNamedParameter4dvNV.set_ptr(loadfn("glProgramNamedParameter4dvNV"));
         storage::ProgramNamedParameter4fNV.set_ptr(loadfn("glProgramNamedParameter4fNV"));
         storage::ProgramNamedParameter4fvNV.set_ptr(loadfn("glProgramNamedParameter4fvNV"));
         storage::ProgramParameter4dNV.set_ptr(loadfn("glProgramParameter4dNV"));
         storage::ProgramParameter4dvNV.set_ptr(loadfn("glProgramParameter4dvNV"));
         storage::ProgramParameter4fNV.set_ptr(loadfn("glProgramParameter4fNV"));
         storage::ProgramParameter4fvNV.set_ptr(loadfn("glProgramParameter4fvNV"));
         storage::ProgramParameteri.set_ptr(loadfn("glProgramParameteri"));
         storage::ProgramParameteriARB.set_ptr(loadfn("glProgramParameteriARB"));
         storage::ProgramParameteriEXT.set_ptr(loadfn("glProgramParameteriEXT"));
         storage::ProgramParameters4dvNV.set_ptr(loadfn("glProgramParameters4dvNV"));
         storage::ProgramParameters4fvNV.set_ptr(loadfn("glProgramParameters4fvNV"));
         storage::ProgramPathFragmentInputGenNV.set_ptr(loadfn("glProgramPathFragmentInputGenNV"));
         storage::ProgramStringARB.set_ptr(loadfn("glProgramStringARB"));
         storage::ProgramSubroutineParametersuivNV.set_ptr(loadfn("glProgramSubroutineParametersuivNV"));
         storage::ProgramUniform1d.set_ptr(loadfn("glProgramUniform1d"));
         storage::ProgramUniform1dEXT.set_ptr(loadfn("glProgramUniform1dEXT"));
         storage::ProgramUniform1dv.set_ptr(loadfn("glProgramUniform1dv"));
         storage::ProgramUniform1dvEXT.set_ptr(loadfn("glProgramUniform1dvEXT"));
         storage::ProgramUniform1f.set_ptr(loadfn("glProgramUniform1f"));
         storage::ProgramUniform1fEXT.set_ptr(loadfn("glProgramUniform1fEXT"));
         storage::ProgramUniform1fv.set_ptr(loadfn("glProgramUniform1fv"));
         storage::ProgramUniform1fvEXT.set_ptr(loadfn("glProgramUniform1fvEXT"));
         storage::ProgramUniform1i.set_ptr(loadfn("glProgramUniform1i"));
         storage::ProgramUniform1i64ARB.set_ptr(loadfn("glProgramUniform1i64ARB"));
         storage::ProgramUniform1i64NV.set_ptr(loadfn("glProgramUniform1i64NV"));
         storage::ProgramUniform1i64vARB.set_ptr(loadfn("glProgramUniform1i64vARB"));
         storage::ProgramUniform1i64vNV.set_ptr(loadfn("glProgramUniform1i64vNV"));
         storage::ProgramUniform1iEXT.set_ptr(loadfn("glProgramUniform1iEXT"));
         storage::ProgramUniform1iv.set_ptr(loadfn("glProgramUniform1iv"));
         storage::ProgramUniform1ivEXT.set_ptr(loadfn("glProgramUniform1ivEXT"));
         storage::ProgramUniform1ui.set_ptr(loadfn("glProgramUniform1ui"));
         storage::ProgramUniform1ui64ARB.set_ptr(loadfn("glProgramUniform1ui64ARB"));
         storage::ProgramUniform1ui64NV.set_ptr(loadfn("glProgramUniform1ui64NV"));
         storage::ProgramUniform1ui64vARB.set_ptr(loadfn("glProgramUniform1ui64vARB"));
         storage::ProgramUniform1ui64vNV.set_ptr(loadfn("glProgramUniform1ui64vNV"));
         storage::ProgramUniform1uiEXT.set_ptr(loadfn("glProgramUniform1uiEXT"));
         storage::ProgramUniform1uiv.set_ptr(loadfn("glProgramUniform1uiv"));
         storage::ProgramUniform1uivEXT.set_ptr(loadfn("glProgramUniform1uivEXT"));
         storage::ProgramUniform2d.set_ptr(loadfn("glProgramUniform2d"));
         storage::ProgramUniform2dEXT.set_ptr(loadfn("glProgramUniform2dEXT"));
         storage::ProgramUniform2dv.set_ptr(loadfn("glProgramUniform2dv"));
         storage::ProgramUniform2dvEXT.set_ptr(loadfn("glProgramUniform2dvEXT"));
         storage::ProgramUniform2f.set_ptr(loadfn("glProgramUniform2f"));
         storage::ProgramUniform2fEXT.set_ptr(loadfn("glProgramUniform2fEXT"));
         storage::ProgramUniform2fv.set_ptr(loadfn("glProgramUniform2fv"));
         storage::ProgramUniform2fvEXT.set_ptr(loadfn("glProgramUniform2fvEXT"));
         storage::ProgramUniform2i.set_ptr(loadfn("glProgramUniform2i"));
         storage::ProgramUniform2i64ARB.set_ptr(loadfn("glProgramUniform2i64ARB"));
         storage::ProgramUniform2i64NV.set_ptr(loadfn("glProgramUniform2i64NV"));
         storage::ProgramUniform2i64vARB.set_ptr(loadfn("glProgramUniform2i64vARB"));
         storage::ProgramUniform2i64vNV.set_ptr(loadfn("glProgramUniform2i64vNV"));
         storage::ProgramUniform2iEXT.set_ptr(loadfn("glProgramUniform2iEXT"));
         storage::ProgramUniform2iv.set_ptr(loadfn("glProgramUniform2iv"));
         storage::ProgramUniform2ivEXT.set_ptr(loadfn("glProgramUniform2ivEXT"));
         storage::ProgramUniform2ui.set_ptr(loadfn("glProgramUniform2ui"));
         storage::ProgramUniform2ui64ARB.set_ptr(loadfn("glProgramUniform2ui64ARB"));
         storage::ProgramUniform2ui64NV.set_ptr(loadfn("glProgramUniform2ui64NV"));
         storage::ProgramUniform2ui64vARB.set_ptr(loadfn("glProgramUniform2ui64vARB"));
         storage::ProgramUniform2ui64vNV.set_ptr(loadfn("glProgramUniform2ui64vNV"));
         storage::ProgramUniform2uiEXT.set_ptr(loadfn("glProgramUniform2uiEXT"));
         storage::ProgramUniform2uiv.set_ptr(loadfn("glProgramUniform2uiv"));
         storage::ProgramUniform2uivEXT.set_ptr(loadfn("glProgramUniform2uivEXT"));
         storage::ProgramUniform3d.set_ptr(loadfn("glProgramUniform3d"));
         storage::ProgramUniform3dEXT.set_ptr(loadfn("glProgramUniform3dEXT"));
         storage::ProgramUniform3dv.set_ptr(loadfn("glProgramUniform3dv"));
         storage::ProgramUniform3dvEXT.set_ptr(loadfn("glProgramUniform3dvEXT"));
         storage::ProgramUniform3f.set_ptr(loadfn("glProgramUniform3f"));
         storage::ProgramUniform3fEXT.set_ptr(loadfn("glProgramUniform3fEXT"));
         storage::ProgramUniform3fv.set_ptr(loadfn("glProgramUniform3fv"));
         storage::ProgramUniform3fvEXT.set_ptr(loadfn("glProgramUniform3fvEXT"));
         storage::ProgramUniform3i.set_ptr(loadfn("glProgramUniform3i"));
         storage::ProgramUniform3i64ARB.set_ptr(loadfn("glProgramUniform3i64ARB"));
         storage::ProgramUniform3i64NV.set_ptr(loadfn("glProgramUniform3i64NV"));
         storage::ProgramUniform3i64vARB.set_ptr(loadfn("glProgramUniform3i64vARB"));
         storage::ProgramUniform3i64vNV.set_ptr(loadfn("glProgramUniform3i64vNV"));
         storage::ProgramUniform3iEXT.set_ptr(loadfn("glProgramUniform3iEXT"));
         storage::ProgramUniform3iv.set_ptr(loadfn("glProgramUniform3iv"));
         storage::ProgramUniform3ivEXT.set_ptr(loadfn("glProgramUniform3ivEXT"));
         storage::ProgramUniform3ui.set_ptr(loadfn("glProgramUniform3ui"));
         storage::ProgramUniform3ui64ARB.set_ptr(loadfn("glProgramUniform3ui64ARB"));
         storage::ProgramUniform3ui64NV.set_ptr(loadfn("glProgramUniform3ui64NV"));
         storage::ProgramUniform3ui64vARB.set_ptr(loadfn("glProgramUniform3ui64vARB"));
         storage::ProgramUniform3ui64vNV.set_ptr(loadfn("glProgramUniform3ui64vNV"));
         storage::ProgramUniform3uiEXT.set_ptr(loadfn("glProgramUniform3uiEXT"));
         storage::ProgramUniform3uiv.set_ptr(loadfn("glProgramUniform3uiv"));
         storage::ProgramUniform3uivEXT.set_ptr(loadfn("glProgramUniform3uivEXT"));
         storage::ProgramUniform4d.set_ptr(loadfn("glProgramUniform4d"));
         storage::ProgramUniform4dEXT.set_ptr(loadfn("glProgramUniform4dEXT"));
         storage::ProgramUniform4dv.set_ptr(loadfn("glProgramUniform4dv"));
         storage::ProgramUniform4dvEXT.set_ptr(loadfn("glProgramUniform4dvEXT"));
         storage::ProgramUniform4f.set_ptr(loadfn("glProgramUniform4f"));
         storage::ProgramUniform4fEXT.set_ptr(loadfn("glProgramUniform4fEXT"));
         storage::ProgramUniform4fv.set_ptr(loadfn("glProgramUniform4fv"));
         storage::ProgramUniform4fvEXT.set_ptr(loadfn("glProgramUniform4fvEXT"));
         storage::ProgramUniform4i.set_ptr(loadfn("glProgramUniform4i"));
         storage::ProgramUniform4i64ARB.set_ptr(loadfn("glProgramUniform4i64ARB"));
         storage::ProgramUniform4i64NV.set_ptr(loadfn("glProgramUniform4i64NV"));
         storage::ProgramUniform4i64vARB.set_ptr(loadfn("glProgramUniform4i64vARB"));
         storage::ProgramUniform4i64vNV.set_ptr(loadfn("glProgramUniform4i64vNV"));
         storage::ProgramUniform4iEXT.set_ptr(loadfn("glProgramUniform4iEXT"));
         storage::ProgramUniform4iv.set_ptr(loadfn("glProgramUniform4iv"));
         storage::ProgramUniform4ivEXT.set_ptr(loadfn("glProgramUniform4ivEXT"));
         storage::ProgramUniform4ui.set_ptr(loadfn("glProgramUniform4ui"));
         storage::ProgramUniform4ui64ARB.set_ptr(loadfn("glProgramUniform4ui64ARB"));
         storage::ProgramUniform4ui64NV.set_ptr(loadfn("glProgramUniform4ui64NV"));
         storage::ProgramUniform4ui64vARB.set_ptr(loadfn("glProgramUniform4ui64vARB"));
         storage::ProgramUniform4ui64vNV.set_ptr(loadfn("glProgramUniform4ui64vNV"));
         storage::ProgramUniform4uiEXT.set_ptr(loadfn("glProgramUniform4uiEXT"));
         storage::ProgramUniform4uiv.set_ptr(loadfn("glProgramUniform4uiv"));
         storage::ProgramUniform4uivEXT.set_ptr(loadfn("glProgramUniform4uivEXT"));
         storage::ProgramUniformHandleui64ARB.set_ptr(loadfn("glProgramUniformHandleui64ARB"));
         storage::ProgramUniformHandleui64NV.set_ptr(loadfn("glProgramUniformHandleui64NV"));
         storage::ProgramUniformHandleui64vARB.set_ptr(loadfn("glProgramUniformHandleui64vARB"));
         storage::ProgramUniformHandleui64vNV.set_ptr(loadfn("glProgramUniformHandleui64vNV"));
         storage::ProgramUniformMatrix2dv.set_ptr(loadfn("glProgramUniformMatrix2dv"));
         storage::ProgramUniformMatrix2dvEXT.set_ptr(loadfn("glProgramUniformMatrix2dvEXT"));
         storage::ProgramUniformMatrix2fv.set_ptr(loadfn("glProgramUniformMatrix2fv"));
         storage::ProgramUniformMatrix2fvEXT.set_ptr(loadfn("glProgramUniformMatrix2fvEXT"));
         storage::ProgramUniformMatrix2x3dv.set_ptr(loadfn("glProgramUniformMatrix2x3dv"));
         storage::ProgramUniformMatrix2x3dvEXT.set_ptr(loadfn("glProgramUniformMatrix2x3dvEXT"));
         storage::ProgramUniformMatrix2x3fv.set_ptr(loadfn("glProgramUniformMatrix2x3fv"));
         storage::ProgramUniformMatrix2x3fvEXT.set_ptr(loadfn("glProgramUniformMatrix2x3fvEXT"));
         storage::ProgramUniformMatrix2x4dv.set_ptr(loadfn("glProgramUniformMatrix2x4dv"));
         storage::ProgramUniformMatrix2x4dvEXT.set_ptr(loadfn("glProgramUniformMatrix2x4dvEXT"));
         storage::ProgramUniformMatrix2x4fv.set_ptr(loadfn("glProgramUniformMatrix2x4fv"));
         storage::ProgramUniformMatrix2x4fvEXT.set_ptr(loadfn("glProgramUniformMatrix2x4fvEXT"));
         storage::ProgramUniformMatrix3dv.set_ptr(loadfn("glProgramUniformMatrix3dv"));
         storage::ProgramUniformMatrix3dvEXT.set_ptr(loadfn("glProgramUniformMatrix3dvEXT"));
         storage::ProgramUniformMatrix3fv.set_ptr(loadfn("glProgramUniformMatrix3fv"));
         storage::ProgramUniformMatrix3fvEXT.set_ptr(loadfn("glProgramUniformMatrix3fvEXT"));
         storage::ProgramUniformMatrix3x2dv.set_ptr(loadfn("glProgramUniformMatrix3x2dv"));
         storage::ProgramUniformMatrix3x2dvEXT.set_ptr(loadfn("glProgramUniformMatrix3x2dvEXT"));
         storage::ProgramUniformMatrix3x2fv.set_ptr(loadfn("glProgramUniformMatrix3x2fv"));
         storage::ProgramUniformMatrix3x2fvEXT.set_ptr(loadfn("glProgramUniformMatrix3x2fvEXT"));
         storage::ProgramUniformMatrix3x4dv.set_ptr(loadfn("glProgramUniformMatrix3x4dv"));
         storage::ProgramUniformMatrix3x4dvEXT.set_ptr(loadfn("glProgramUniformMatrix3x4dvEXT"));
         storage::ProgramUniformMatrix3x4fv.set_ptr(loadfn("glProgramUniformMatrix3x4fv"));
         storage::ProgramUniformMatrix3x4fvEXT.set_ptr(loadfn("glProgramUniformMatrix3x4fvEXT"));
         storage::ProgramUniformMatrix4dv.set_ptr(loadfn("glProgramUniformMatrix4dv"));
         storage::ProgramUniformMatrix4dvEXT.set_ptr(loadfn("glProgramUniformMatrix4dvEXT"));
         storage::ProgramUniformMatrix4fv.set_ptr(loadfn("glProgramUniformMatrix4fv"));
         storage::ProgramUniformMatrix4fvEXT.set_ptr(loadfn("glProgramUniformMatrix4fvEXT"));
         storage::ProgramUniformMatrix4x2dv.set_ptr(loadfn("glProgramUniformMatrix4x2dv"));
         storage::ProgramUniformMatrix4x2dvEXT.set_ptr(loadfn("glProgramUniformMatrix4x2dvEXT"));
         storage::ProgramUniformMatrix4x2fv.set_ptr(loadfn("glProgramUniformMatrix4x2fv"));
         storage::ProgramUniformMatrix4x2fvEXT.set_ptr(loadfn("glProgramUniformMatrix4x2fvEXT"));
         storage::ProgramUniformMatrix4x3dv.set_ptr(loadfn("glProgramUniformMatrix4x3dv"));
         storage::ProgramUniformMatrix4x3dvEXT.set_ptr(loadfn("glProgramUniformMatrix4x3dvEXT"));
         storage::ProgramUniformMatrix4x3fv.set_ptr(loadfn("glProgramUniformMatrix4x3fv"));
         storage::ProgramUniformMatrix4x3fvEXT.set_ptr(loadfn("glProgramUniformMatrix4x3fvEXT"));
         storage::ProgramUniformui64NV.set_ptr(loadfn("glProgramUniformui64NV"));
         storage::ProgramUniformui64vNV.set_ptr(loadfn("glProgramUniformui64vNV"));
         storage::ProgramVertexLimitNV.set_ptr(loadfn("glProgramVertexLimitNV"));
         storage::ProvokingVertex.set_ptr(loadfn("glProvokingVertex"));
         storage::ProvokingVertexEXT.set_ptr(loadfn("glProvokingVertexEXT"));
         storage::PushClientAttribDefaultEXT.set_ptr(loadfn("glPushClientAttribDefaultEXT"));
         storage::PushDebugGroup.set_ptr(loadfn("glPushDebugGroup"));
         storage::PushGroupMarkerEXT.set_ptr(loadfn("glPushGroupMarkerEXT"));
         storage::QueryCounter.set_ptr(loadfn("glQueryCounter"));
         storage::QueryMatrixxOES.set_ptr(loadfn("glQueryMatrixxOES"));
         storage::QueryObjectParameteruiAMD.set_ptr(loadfn("glQueryObjectParameteruiAMD"));
         storage::QueryResourceNV.set_ptr(loadfn("glQueryResourceNV"));
         storage::QueryResourceTagNV.set_ptr(loadfn("glQueryResourceTagNV"));
         storage::RasterPos2xOES.set_ptr(loadfn("glRasterPos2xOES"));
         storage::RasterPos2xvOES.set_ptr(loadfn("glRasterPos2xvOES"));
         storage::RasterPos3xOES.set_ptr(loadfn("glRasterPos3xOES"));
         storage::RasterPos3xvOES.set_ptr(loadfn("glRasterPos3xvOES"));
         storage::RasterPos4xOES.set_ptr(loadfn("glRasterPos4xOES"));
         storage::RasterPos4xvOES.set_ptr(loadfn("glRasterPos4xvOES"));
         storage::RasterSamplesEXT.set_ptr(loadfn("glRasterSamplesEXT"));
         storage::ReadBuffer.set_ptr(loadfn("glReadBuffer"));
         storage::ReadInstrumentsSGIX.set_ptr(loadfn("glReadInstrumentsSGIX"));
         storage::ReadPixels.set_ptr(loadfn("glReadPixels"));
         storage::ReadnPixels.set_ptr(loadfn("glReadnPixels"));
         storage::ReadnPixelsARB.set_ptr(loadfn("glReadnPixelsARB"));
         storage::RectxOES.set_ptr(loadfn("glRectxOES"));
         storage::RectxvOES.set_ptr(loadfn("glRectxvOES"));
         storage::ReferencePlaneSGIX.set_ptr(loadfn("glReferencePlaneSGIX"));
         storage::ReleaseKeyedMutexWin32EXT.set_ptr(loadfn("glReleaseKeyedMutexWin32EXT"));
         storage::ReleaseShaderCompiler.set_ptr(loadfn("glReleaseShaderCompiler"));
         storage::RenderGpuMaskNV.set_ptr(loadfn("glRenderGpuMaskNV"));
         storage::RenderbufferStorage.set_ptr(loadfn("glRenderbufferStorage"));
         storage::RenderbufferStorageEXT.set_ptr(loadfn("glRenderbufferStorageEXT"));
         storage::RenderbufferStorageMultisample.set_ptr(loadfn("glRenderbufferStorageMultisample"));
         storage::RenderbufferStorageMultisampleAdvancedAMD.set_ptr(loadfn("glRenderbufferStorageMultisampleAdvancedAMD"));
         storage::RenderbufferStorageMultisampleCoverageNV.set_ptr(loadfn("glRenderbufferStorageMultisampleCoverageNV"));
         storage::RenderbufferStorageMultisampleEXT.set_ptr(loadfn("glRenderbufferStorageMultisampleEXT"));
         storage::ReplacementCodePointerSUN.set_ptr(loadfn("glReplacementCodePointerSUN"));
         storage::ReplacementCodeubSUN.set_ptr(loadfn("glReplacementCodeubSUN"));
         storage::ReplacementCodeubvSUN.set_ptr(loadfn("glReplacementCodeubvSUN"));
         storage::ReplacementCodeuiColor3fVertex3fSUN.set_ptr(loadfn("glReplacementCodeuiColor3fVertex3fSUN"));
         storage::ReplacementCodeuiColor3fVertex3fvSUN.set_ptr(loadfn("glReplacementCodeuiColor3fVertex3fvSUN"));
         storage::ReplacementCodeuiColor4fNormal3fVertex3fSUN.set_ptr(loadfn("glReplacementCodeuiColor4fNormal3fVertex3fSUN"));
         storage::ReplacementCodeuiColor4fNormal3fVertex3fvSUN.set_ptr(loadfn("glReplacementCodeuiColor4fNormal3fVertex3fvSUN"));
         storage::ReplacementCodeuiColor4ubVertex3fSUN.set_ptr(loadfn("glReplacementCodeuiColor4ubVertex3fSUN"));
         storage::ReplacementCodeuiColor4ubVertex3fvSUN.set_ptr(loadfn("glReplacementCodeuiColor4ubVertex3fvSUN"));
         storage::ReplacementCodeuiNormal3fVertex3fSUN.set_ptr(loadfn("glReplacementCodeuiNormal3fVertex3fSUN"));
         storage::ReplacementCodeuiNormal3fVertex3fvSUN.set_ptr(loadfn("glReplacementCodeuiNormal3fVertex3fvSUN"));
         storage::ReplacementCodeuiSUN.set_ptr(loadfn("glReplacementCodeuiSUN"));
         storage::ReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fSUN.set_ptr(loadfn("glReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fSUN"));
         storage::ReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fvSUN.set_ptr(loadfn("glReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fvSUN"));
         storage::ReplacementCodeuiTexCoord2fNormal3fVertex3fSUN.set_ptr(loadfn("glReplacementCodeuiTexCoord2fNormal3fVertex3fSUN"));
         storage::ReplacementCodeuiTexCoord2fNormal3fVertex3fvSUN.set_ptr(loadfn("glReplacementCodeuiTexCoord2fNormal3fVertex3fvSUN"));
         storage::ReplacementCodeuiTexCoord2fVertex3fSUN.set_ptr(loadfn("glReplacementCodeuiTexCoord2fVertex3fSUN"));
         storage::ReplacementCodeuiTexCoord2fVertex3fvSUN.set_ptr(loadfn("glReplacementCodeuiTexCoord2fVertex3fvSUN"));
         storage::ReplacementCodeuiVertex3fSUN.set_ptr(loadfn("glReplacementCodeuiVertex3fSUN"));
         storage::ReplacementCodeuiVertex3fvSUN.set_ptr(loadfn("glReplacementCodeuiVertex3fvSUN"));
         storage::ReplacementCodeuivSUN.set_ptr(loadfn("glReplacementCodeuivSUN"));
         storage::ReplacementCodeusSUN.set_ptr(loadfn("glReplacementCodeusSUN"));
         storage::ReplacementCodeusvSUN.set_ptr(loadfn("glReplacementCodeusvSUN"));
         storage::RequestResidentProgramsNV.set_ptr(loadfn("glRequestResidentProgramsNV"));
         storage::ResetHistogramEXT.set_ptr(loadfn("glResetHistogramEXT"));
         storage::ResetMemoryObjectParameterNV.set_ptr(loadfn("glResetMemoryObjectParameterNV"));
         storage::ResetMinmaxEXT.set_ptr(loadfn("glResetMinmaxEXT"));
         storage::ResizeBuffersMESA.set_ptr(loadfn("glResizeBuffersMESA"));
         storage::ResolveDepthValuesNV.set_ptr(loadfn("glResolveDepthValuesNV"));
         storage::ResumeTransformFeedback.set_ptr(loadfn("glResumeTransformFeedback"));
         storage::ResumeTransformFeedbackNV.set_ptr(loadfn("glResumeTransformFeedbackNV"));
         storage::RotatexOES.set_ptr(loadfn("glRotatexOES"));
         storage::SampleCoverage.set_ptr(loadfn("glSampleCoverage"));
         storage::SampleCoverageARB.set_ptr(loadfn("glSampleCoverageARB"));
         storage::SampleMapATI.set_ptr(loadfn("glSampleMapATI"));
         storage::SampleMaskEXT.set_ptr(loadfn("glSampleMaskEXT"));
         storage::SampleMaskIndexedNV.set_ptr(loadfn("glSampleMaskIndexedNV"));
         storage::SampleMaskSGIS.set_ptr(loadfn("glSampleMaskSGIS"));
         storage::SampleMaski.set_ptr(loadfn("glSampleMaski"));
         storage::SamplePatternEXT.set_ptr(loadfn("glSamplePatternEXT"));
         storage::SamplePatternSGIS.set_ptr(loadfn("glSamplePatternSGIS"));
         storage::SamplerParameterIiv.set_ptr(loadfn("glSamplerParameterIiv"));
         storage::SamplerParameterIuiv.set_ptr(loadfn("glSamplerParameterIuiv"));
         storage::SamplerParameterf.set_ptr(loadfn("glSamplerParameterf"));
         storage::SamplerParameterfv.set_ptr(loadfn("glSamplerParameterfv"));
         storage::SamplerParameteri.set_ptr(loadfn("glSamplerParameteri"));
         storage::SamplerParameteriv.set_ptr(loadfn("glSamplerParameteriv"));
         storage::ScalexOES.set_ptr(loadfn("glScalexOES"));
         storage::Scissor.set_ptr(loadfn("glScissor"));
         storage::ScissorArrayv.set_ptr(loadfn("glScissorArrayv"));
         storage::ScissorExclusiveArrayvNV.set_ptr(loadfn("glScissorExclusiveArrayvNV"));
         storage::ScissorExclusiveNV.set_ptr(loadfn("glScissorExclusiveNV"));
         storage::ScissorIndexed.set_ptr(loadfn("glScissorIndexed"));
         storage::ScissorIndexedv.set_ptr(loadfn("glScissorIndexedv"));
         storage::SecondaryColor3bEXT.set_ptr(loadfn("glSecondaryColor3bEXT"));
         storage::SecondaryColor3bvEXT.set_ptr(loadfn("glSecondaryColor3bvEXT"));
         storage::SecondaryColor3dEXT.set_ptr(loadfn("glSecondaryColor3dEXT"));
         storage::SecondaryColor3dvEXT.set_ptr(loadfn("glSecondaryColor3dvEXT"));
         storage::SecondaryColor3fEXT.set_ptr(loadfn("glSecondaryColor3fEXT"));
         storage::SecondaryColor3fvEXT.set_ptr(loadfn("glSecondaryColor3fvEXT"));
         storage::SecondaryColor3hNV.set_ptr(loadfn("glSecondaryColor3hNV"));
         storage::SecondaryColor3hvNV.set_ptr(loadfn("glSecondaryColor3hvNV"));
         storage::SecondaryColor3iEXT.set_ptr(loadfn("glSecondaryColor3iEXT"));
         storage::SecondaryColor3ivEXT.set_ptr(loadfn("glSecondaryColor3ivEXT"));
         storage::SecondaryColor3sEXT.set_ptr(loadfn("glSecondaryColor3sEXT"));
         storage::SecondaryColor3svEXT.set_ptr(loadfn("glSecondaryColor3svEXT"));
         storage::SecondaryColor3ubEXT.set_ptr(loadfn("glSecondaryColor3ubEXT"));
         storage::SecondaryColor3ubvEXT.set_ptr(loadfn("glSecondaryColor3ubvEXT"));
         storage::SecondaryColor3uiEXT.set_ptr(loadfn("glSecondaryColor3uiEXT"));
         storage::SecondaryColor3uivEXT.set_ptr(loadfn("glSecondaryColor3uivEXT"));
         storage::SecondaryColor3usEXT.set_ptr(loadfn("glSecondaryColor3usEXT"));
         storage::SecondaryColor3usvEXT.set_ptr(loadfn("glSecondaryColor3usvEXT"));
         storage::SecondaryColorFormatNV.set_ptr(loadfn("glSecondaryColorFormatNV"));
         storage::SecondaryColorPointerEXT.set_ptr(loadfn("glSecondaryColorPointerEXT"));
         storage::SecondaryColorPointerListIBM.set_ptr(loadfn("glSecondaryColorPointerListIBM"));
         storage::SelectPerfMonitorCountersAMD.set_ptr(loadfn("glSelectPerfMonitorCountersAMD"));
         storage::SemaphoreParameterivNV.set_ptr(loadfn("glSemaphoreParameterivNV"));
         storage::SemaphoreParameterui64vEXT.set_ptr(loadfn("glSemaphoreParameterui64vEXT"));
         storage::SeparableFilter2DEXT.set_ptr(loadfn("glSeparableFilter2DEXT"));
         storage::SetFenceAPPLE.set_ptr(loadfn("glSetFenceAPPLE"));
         storage::SetFenceNV.set_ptr(loadfn("glSetFenceNV"));
         storage::SetFragmentShaderConstantATI.set_ptr(loadfn("glSetFragmentShaderConstantATI"));
         storage::SetInvariantEXT.set_ptr(loadfn("glSetInvariantEXT"));
         storage::SetLocalConstantEXT.set_ptr(loadfn("glSetLocalConstantEXT"));
         storage::SetMultisamplefvAMD.set_ptr(loadfn("glSetMultisamplefvAMD"));
         storage::ShaderBinary.set_ptr(loadfn("glShaderBinary"));
         storage::ShaderOp1EXT.set_ptr(loadfn("glShaderOp1EXT"));
         storage::ShaderOp2EXT.set_ptr(loadfn("glShaderOp2EXT"));
         storage::ShaderOp3EXT.set_ptr(loadfn("glShaderOp3EXT"));
         storage::ShaderSource.set_ptr(loadfn("glShaderSource"));
         storage::ShaderSourceARB.set_ptr(loadfn("glShaderSourceARB"));
         storage::ShaderStorageBlockBinding.set_ptr(loadfn("glShaderStorageBlockBinding"));
         storage::ShadingRateImageBarrierNV.set_ptr(loadfn("glShadingRateImageBarrierNV"));
         storage::ShadingRateImagePaletteNV.set_ptr(loadfn("glShadingRateImagePaletteNV"));
         storage::ShadingRateSampleOrderCustomNV.set_ptr(loadfn("glShadingRateSampleOrderCustomNV"));
         storage::ShadingRateSampleOrderNV.set_ptr(loadfn("glShadingRateSampleOrderNV"));
         storage::SharpenTexFuncSGIS.set_ptr(loadfn("glSharpenTexFuncSGIS"));
         storage::SignalSemaphoreEXT.set_ptr(loadfn("glSignalSemaphoreEXT"));
         storage::SignalSemaphoreui64NVX.set_ptr(loadfn("glSignalSemaphoreui64NVX"));
         storage::SignalVkFenceNV.set_ptr(loadfn("glSignalVkFenceNV"));
         storage::SignalVkSemaphoreNV.set_ptr(loadfn("glSignalVkSemaphoreNV"));
         storage::SpecializeShader.set_ptr(loadfn("glSpecializeShader"));
         storage::SpecializeShaderARB.set_ptr(loadfn("glSpecializeShaderARB"));
         storage::SpriteParameterfSGIX.set_ptr(loadfn("glSpriteParameterfSGIX"));
         storage::SpriteParameterfvSGIX.set_ptr(loadfn("glSpriteParameterfvSGIX"));
         storage::SpriteParameteriSGIX.set_ptr(loadfn("glSpriteParameteriSGIX"));
         storage::SpriteParameterivSGIX.set_ptr(loadfn("glSpriteParameterivSGIX"));
         storage::StartInstrumentsSGIX.set_ptr(loadfn("glStartInstrumentsSGIX"));
         storage::StateCaptureNV.set_ptr(loadfn("glStateCaptureNV"));
         storage::StencilClearTagEXT.set_ptr(loadfn("glStencilClearTagEXT"));
         storage::StencilFillPathInstancedNV.set_ptr(loadfn("glStencilFillPathInstancedNV"));
         storage::StencilFillPathNV.set_ptr(loadfn("glStencilFillPathNV"));
         storage::StencilFunc.set_ptr(loadfn("glStencilFunc"));
         storage::StencilFuncSeparate.set_ptr(loadfn("glStencilFuncSeparate"));
         storage::StencilFuncSeparateATI.set_ptr(loadfn("glStencilFuncSeparateATI"));
         storage::StencilMask.set_ptr(loadfn("glStencilMask"));
         storage::StencilMaskSeparate.set_ptr(loadfn("glStencilMaskSeparate"));
         storage::StencilOp.set_ptr(loadfn("glStencilOp"));
         storage::StencilOpSeparate.set_ptr(loadfn("glStencilOpSeparate"));
         storage::StencilOpSeparateATI.set_ptr(loadfn("glStencilOpSeparateATI"));
         storage::StencilOpValueAMD.set_ptr(loadfn("glStencilOpValueAMD"));
         storage::StencilStrokePathInstancedNV.set_ptr(loadfn("glStencilStrokePathInstancedNV"));
         storage::StencilStrokePathNV.set_ptr(loadfn("glStencilStrokePathNV"));
         storage::StencilThenCoverFillPathInstancedNV.set_ptr(loadfn("glStencilThenCoverFillPathInstancedNV"));
         storage::StencilThenCoverFillPathNV.set_ptr(loadfn("glStencilThenCoverFillPathNV"));
         storage::StencilThenCoverStrokePathInstancedNV.set_ptr(loadfn("glStencilThenCoverStrokePathInstancedNV"));
         storage::StencilThenCoverStrokePathNV.set_ptr(loadfn("glStencilThenCoverStrokePathNV"));
         storage::StopInstrumentsSGIX.set_ptr(loadfn("glStopInstrumentsSGIX"));
         storage::StringMarkerGREMEDY.set_ptr(loadfn("glStringMarkerGREMEDY"));
         storage::SubpixelPrecisionBiasNV.set_ptr(loadfn("glSubpixelPrecisionBiasNV"));
         storage::SwizzleEXT.set_ptr(loadfn("glSwizzleEXT"));
         storage::SyncTextureINTEL.set_ptr(loadfn("glSyncTextureINTEL"));
         storage::TagSampleBufferSGIX.set_ptr(loadfn("glTagSampleBufferSGIX"));
         storage::Tangent3bEXT.set_ptr(loadfn("glTangent3bEXT"));
         storage::Tangent3bvEXT.set_ptr(loadfn("glTangent3bvEXT"));
         storage::Tangent3dEXT.set_ptr(loadfn("glTangent3dEXT"));
         storage::Tangent3dvEXT.set_ptr(loadfn("glTangent3dvEXT"));
         storage::Tangent3fEXT.set_ptr(loadfn("glTangent3fEXT"));
         storage::Tangent3fvEXT.set_ptr(loadfn("glTangent3fvEXT"));
         storage::Tangent3iEXT.set_ptr(loadfn("glTangent3iEXT"));
         storage::Tangent3ivEXT.set_ptr(loadfn("glTangent3ivEXT"));
         storage::Tangent3sEXT.set_ptr(loadfn("glTangent3sEXT"));
         storage::Tangent3svEXT.set_ptr(loadfn("glTangent3svEXT"));
         storage::TangentPointerEXT.set_ptr(loadfn("glTangentPointerEXT"));
         storage::TbufferMask3DFX.set_ptr(loadfn("glTbufferMask3DFX"));
         storage::TessellationFactorAMD.set_ptr(loadfn("glTessellationFactorAMD"));
         storage::TessellationModeAMD.set_ptr(loadfn("glTessellationModeAMD"));
         storage::TestFenceAPPLE.set_ptr(loadfn("glTestFenceAPPLE"));
         storage::TestFenceNV.set_ptr(loadfn("glTestFenceNV"));
         storage::TestObjectAPPLE.set_ptr(loadfn("glTestObjectAPPLE"));
         storage::TexAttachMemoryNV.set_ptr(loadfn("glTexAttachMemoryNV"));
         storage::TexBuffer.set_ptr(loadfn("glTexBuffer"));
         storage::TexBufferARB.set_ptr(loadfn("glTexBufferARB"));
         storage::TexBufferEXT.set_ptr(loadfn("glTexBufferEXT"));
         storage::TexBufferRange.set_ptr(loadfn("glTexBufferRange"));
         storage::TexBumpParameterfvATI.set_ptr(loadfn("glTexBumpParameterfvATI"));
         storage::TexBumpParameterivATI.set_ptr(loadfn("glTexBumpParameterivATI"));
         storage::TexCoord1bOES.set_ptr(loadfn("glTexCoord1bOES"));
         storage::TexCoord1bvOES.set_ptr(loadfn("glTexCoord1bvOES"));
         storage::TexCoord1hNV.set_ptr(loadfn("glTexCoord1hNV"));
         storage::TexCoord1hvNV.set_ptr(loadfn("glTexCoord1hvNV"));
         storage::TexCoord1xOES.set_ptr(loadfn("glTexCoord1xOES"));
         storage::TexCoord1xvOES.set_ptr(loadfn("glTexCoord1xvOES"));
         storage::TexCoord2bOES.set_ptr(loadfn("glTexCoord2bOES"));
         storage::TexCoord2bvOES.set_ptr(loadfn("glTexCoord2bvOES"));
         storage::TexCoord2fColor3fVertex3fSUN.set_ptr(loadfn("glTexCoord2fColor3fVertex3fSUN"));
         storage::TexCoord2fColor3fVertex3fvSUN.set_ptr(loadfn("glTexCoord2fColor3fVertex3fvSUN"));
         storage::TexCoord2fColor4fNormal3fVertex3fSUN.set_ptr(loadfn("glTexCoord2fColor4fNormal3fVertex3fSUN"));
         storage::TexCoord2fColor4fNormal3fVertex3fvSUN.set_ptr(loadfn("glTexCoord2fColor4fNormal3fVertex3fvSUN"));
         storage::TexCoord2fColor4ubVertex3fSUN.set_ptr(loadfn("glTexCoord2fColor4ubVertex3fSUN"));
         storage::TexCoord2fColor4ubVertex3fvSUN.set_ptr(loadfn("glTexCoord2fColor4ubVertex3fvSUN"));
         storage::TexCoord2fNormal3fVertex3fSUN.set_ptr(loadfn("glTexCoord2fNormal3fVertex3fSUN"));
         storage::TexCoord2fNormal3fVertex3fvSUN.set_ptr(loadfn("glTexCoord2fNormal3fVertex3fvSUN"));
         storage::TexCoord2fVertex3fSUN.set_ptr(loadfn("glTexCoord2fVertex3fSUN"));
         storage::TexCoord2fVertex3fvSUN.set_ptr(loadfn("glTexCoord2fVertex3fvSUN"));
         storage::TexCoord2hNV.set_ptr(loadfn("glTexCoord2hNV"));
         storage::TexCoord2hvNV.set_ptr(loadfn("glTexCoord2hvNV"));
         storage::TexCoord2xOES.set_ptr(loadfn("glTexCoord2xOES"));
         storage::TexCoord2xvOES.set_ptr(loadfn("glTexCoord2xvOES"));
         storage::TexCoord3bOES.set_ptr(loadfn("glTexCoord3bOES"));
         storage::TexCoord3bvOES.set_ptr(loadfn("glTexCoord3bvOES"));
         storage::TexCoord3hNV.set_ptr(loadfn("glTexCoord3hNV"));
         storage::TexCoord3hvNV.set_ptr(loadfn("glTexCoord3hvNV"));
         storage::TexCoord3xOES.set_ptr(loadfn("glTexCoord3xOES"));
         storage::TexCoord3xvOES.set_ptr(loadfn("glTexCoord3xvOES"));
         storage::TexCoord4bOES.set_ptr(loadfn("glTexCoord4bOES"));
         storage::TexCoord4bvOES.set_ptr(loadfn("glTexCoord4bvOES"));
         storage::TexCoord4fColor4fNormal3fVertex4fSUN.set_ptr(loadfn("glTexCoord4fColor4fNormal3fVertex4fSUN"));
         storage::TexCoord4fColor4fNormal3fVertex4fvSUN.set_ptr(loadfn("glTexCoord4fColor4fNormal3fVertex4fvSUN"));
         storage::TexCoord4fVertex4fSUN.set_ptr(loadfn("glTexCoord4fVertex4fSUN"));
         storage::TexCoord4fVertex4fvSUN.set_ptr(loadfn("glTexCoord4fVertex4fvSUN"));
         storage::TexCoord4hNV.set_ptr(loadfn("glTexCoord4hNV"));
         storage::TexCoord4hvNV.set_ptr(loadfn("glTexCoord4hvNV"));
         storage::TexCoord4xOES.set_ptr(loadfn("glTexCoord4xOES"));
         storage::TexCoord4xvOES.set_ptr(loadfn("glTexCoord4xvOES"));
         storage::TexCoordFormatNV.set_ptr(loadfn("glTexCoordFormatNV"));
         storage::TexCoordPointerEXT.set_ptr(loadfn("glTexCoordPointerEXT"));
         storage::TexCoordPointerListIBM.set_ptr(loadfn("glTexCoordPointerListIBM"));
         storage::TexCoordPointervINTEL.set_ptr(loadfn("glTexCoordPointervINTEL"));
         storage::TexEnvxOES.set_ptr(loadfn("glTexEnvxOES"));
         storage::TexEnvxvOES.set_ptr(loadfn("glTexEnvxvOES"));
         storage::TexFilterFuncSGIS.set_ptr(loadfn("glTexFilterFuncSGIS"));
         storage::TexGenxOES.set_ptr(loadfn("glTexGenxOES"));
         storage::TexGenxvOES.set_ptr(loadfn("glTexGenxvOES"));
         storage::TexImage1D.set_ptr(loadfn("glTexImage1D"));
         storage::TexImage2D.set_ptr(loadfn("glTexImage2D"));
         storage::TexImage2DMultisample.set_ptr(loadfn("glTexImage2DMultisample"));
         storage::TexImage2DMultisampleCoverageNV.set_ptr(loadfn("glTexImage2DMultisampleCoverageNV"));
         storage::TexImage3D.set_ptr(loadfn("glTexImage3D"));
         storage::TexImage3DEXT.set_ptr(loadfn("glTexImage3DEXT"));
         storage::TexImage3DMultisample.set_ptr(loadfn("glTexImage3DMultisample"));
         storage::TexImage3DMultisampleCoverageNV.set_ptr(loadfn("glTexImage3DMultisampleCoverageNV"));
         storage::TexImage4DSGIS.set_ptr(loadfn("glTexImage4DSGIS"));
         storage::TexPageCommitmentARB.set_ptr(loadfn("glTexPageCommitmentARB"));
         storage::TexPageCommitmentMemNV.set_ptr(loadfn("glTexPageCommitmentMemNV"));
         storage::TexParameterIiv.set_ptr(loadfn("glTexParameterIiv"));
         storage::TexParameterIivEXT.set_ptr(loadfn("glTexParameterIivEXT"));
         storage::TexParameterIuiv.set_ptr(loadfn("glTexParameterIuiv"));
         storage::TexParameterIuivEXT.set_ptr(loadfn("glTexParameterIuivEXT"));
         storage::TexParameterf.set_ptr(loadfn("glTexParameterf"));
         storage::TexParameterfv.set_ptr(loadfn("glTexParameterfv"));
         storage::TexParameteri.set_ptr(loadfn("glTexParameteri"));
         storage::TexParameteriv.set_ptr(loadfn("glTexParameteriv"));
         storage::TexParameterxOES.set_ptr(loadfn("glTexParameterxOES"));
         storage::TexParameterxvOES.set_ptr(loadfn("glTexParameterxvOES"));
         storage::TexRenderbufferNV.set_ptr(loadfn("glTexRenderbufferNV"));
         storage::TexStorage1D.set_ptr(loadfn("glTexStorage1D"));
         storage::TexStorage1DEXT.set_ptr(loadfn("glTexStorage1DEXT"));
         storage::TexStorage2D.set_ptr(loadfn("glTexStorage2D"));
         storage::TexStorage2DEXT.set_ptr(loadfn("glTexStorage2DEXT"));
         storage::TexStorage2DMultisample.set_ptr(loadfn("glTexStorage2DMultisample"));
         storage::TexStorage3D.set_ptr(loadfn("glTexStorage3D"));
         storage::TexStorage3DEXT.set_ptr(loadfn("glTexStorage3DEXT"));
         storage::TexStorage3DMultisample.set_ptr(loadfn("glTexStorage3DMultisample"));
         storage::TexStorageMem1DEXT.set_ptr(loadfn("glTexStorageMem1DEXT"));
         storage::TexStorageMem2DEXT.set_ptr(loadfn("glTexStorageMem2DEXT"));
         storage::TexStorageMem2DMultisampleEXT.set_ptr(loadfn("glTexStorageMem2DMultisampleEXT"));
         storage::TexStorageMem3DEXT.set_ptr(loadfn("glTexStorageMem3DEXT"));
         storage::TexStorageMem3DMultisampleEXT.set_ptr(loadfn("glTexStorageMem3DMultisampleEXT"));
         storage::TexStorageSparseAMD.set_ptr(loadfn("glTexStorageSparseAMD"));
         storage::TexSubImage1D.set_ptr(loadfn("glTexSubImage1D"));
         storage::TexSubImage1DEXT.set_ptr(loadfn("glTexSubImage1DEXT"));
         storage::TexSubImage2D.set_ptr(loadfn("glTexSubImage2D"));
         storage::TexSubImage2DEXT.set_ptr(loadfn("glTexSubImage2DEXT"));
         storage::TexSubImage3D.set_ptr(loadfn("glTexSubImage3D"));
         storage::TexSubImage3DEXT.set_ptr(loadfn("glTexSubImage3DEXT"));
         storage::TexSubImage4DSGIS.set_ptr(loadfn("glTexSubImage4DSGIS"));
         storage::TextureAttachMemoryNV.set_ptr(loadfn("glTextureAttachMemoryNV"));
         storage::TextureBarrier.set_ptr(loadfn("glTextureBarrier"));
         storage::TextureBarrierNV.set_ptr(loadfn("glTextureBarrierNV"));
         storage::TextureBuffer.set_ptr(loadfn("glTextureBuffer"));
         storage::TextureBufferEXT.set_ptr(loadfn("glTextureBufferEXT"));
         storage::TextureBufferRange.set_ptr(loadfn("glTextureBufferRange"));
         storage::TextureBufferRangeEXT.set_ptr(loadfn("glTextureBufferRangeEXT"));
         storage::TextureColorMaskSGIS.set_ptr(loadfn("glTextureColorMaskSGIS"));
         storage::TextureImage1DEXT.set_ptr(loadfn("glTextureImage1DEXT"));
         storage::TextureImage2DEXT.set_ptr(loadfn("glTextureImage2DEXT"));
         storage::TextureImage2DMultisampleCoverageNV.set_ptr(loadfn("glTextureImage2DMultisampleCoverageNV"));
         storage::TextureImage2DMultisampleNV.set_ptr(loadfn("glTextureImage2DMultisampleNV"));
         storage::TextureImage3DEXT.set_ptr(loadfn("glTextureImage3DEXT"));
         storage::TextureImage3DMultisampleCoverageNV.set_ptr(loadfn("glTextureImage3DMultisampleCoverageNV"));
         storage::TextureImage3DMultisampleNV.set_ptr(loadfn("glTextureImage3DMultisampleNV"));
         storage::TextureLightEXT.set_ptr(loadfn("glTextureLightEXT"));
         storage::TextureMaterialEXT.set_ptr(loadfn("glTextureMaterialEXT"));
         storage::TextureNormalEXT.set_ptr(loadfn("glTextureNormalEXT"));
         storage::TexturePageCommitmentEXT.set_ptr(loadfn("glTexturePageCommitmentEXT"));
         storage::TexturePageCommitmentMemNV.set_ptr(loadfn("glTexturePageCommitmentMemNV"));
         storage::TextureParameterIiv.set_ptr(loadfn("glTextureParameterIiv"));
         storage::TextureParameterIivEXT.set_ptr(loadfn("glTextureParameterIivEXT"));
         storage::TextureParameterIuiv.set_ptr(loadfn("glTextureParameterIuiv"));
         storage::TextureParameterIuivEXT.set_ptr(loadfn("glTextureParameterIuivEXT"));
         storage::TextureParameterf.set_ptr(loadfn("glTextureParameterf"));
         storage::TextureParameterfEXT.set_ptr(loadfn("glTextureParameterfEXT"));
         storage::TextureParameterfv.set_ptr(loadfn("glTextureParameterfv"));
         storage::TextureParameterfvEXT.set_ptr(loadfn("glTextureParameterfvEXT"));
         storage::TextureParameteri.set_ptr(loadfn("glTextureParameteri"));
         storage::TextureParameteriEXT.set_ptr(loadfn("glTextureParameteriEXT"));
         storage::TextureParameteriv.set_ptr(loadfn("glTextureParameteriv"));
         storage::TextureParameterivEXT.set_ptr(loadfn("glTextureParameterivEXT"));
         storage::TextureRangeAPPLE.set_ptr(loadfn("glTextureRangeAPPLE"));
         storage::TextureRenderbufferEXT.set_ptr(loadfn("glTextureRenderbufferEXT"));
         storage::TextureStorage1D.set_ptr(loadfn("glTextureStorage1D"));
         storage::TextureStorage1DEXT.set_ptr(loadfn("glTextureStorage1DEXT"));
         storage::TextureStorage2D.set_ptr(loadfn("glTextureStorage2D"));
         storage::TextureStorage2DEXT.set_ptr(loadfn("glTextureStorage2DEXT"));
         storage::TextureStorage2DMultisample.set_ptr(loadfn("glTextureStorage2DMultisample"));
         storage::TextureStorage2DMultisampleEXT.set_ptr(loadfn("glTextureStorage2DMultisampleEXT"));
         storage::TextureStorage3D.set_ptr(loadfn("glTextureStorage3D"));
         storage::TextureStorage3DEXT.set_ptr(loadfn("glTextureStorage3DEXT"));
         storage::TextureStorage3DMultisample.set_ptr(loadfn("glTextureStorage3DMultisample"));
         storage::TextureStorage3DMultisampleEXT.set_ptr(loadfn("glTextureStorage3DMultisampleEXT"));
         storage::TextureStorageMem1DEXT.set_ptr(loadfn("glTextureStorageMem1DEXT"));
         storage::TextureStorageMem2DEXT.set_ptr(loadfn("glTextureStorageMem2DEXT"));
         storage::TextureStorageMem2DMultisampleEXT.set_ptr(loadfn("glTextureStorageMem2DMultisampleEXT"));
         storage::TextureStorageMem3DEXT.set_ptr(loadfn("glTextureStorageMem3DEXT"));
         storage::TextureStorageMem3DMultisampleEXT.set_ptr(loadfn("glTextureStorageMem3DMultisampleEXT"));
         storage::TextureStorageSparseAMD.set_ptr(loadfn("glTextureStorageSparseAMD"));
         storage::TextureSubImage1D.set_ptr(loadfn("glTextureSubImage1D"));
         storage::TextureSubImage1DEXT.set_ptr(loadfn("glTextureSubImage1DEXT"));
         storage::TextureSubImage2D.set_ptr(loadfn("glTextureSubImage2D"));
         storage::TextureSubImage2DEXT.set_ptr(loadfn("glTextureSubImage2DEXT"));
         storage::TextureSubImage3D.set_ptr(loadfn("glTextureSubImage3D"));
         storage::TextureSubImage3DEXT.set_ptr(loadfn("glTextureSubImage3DEXT"));
         storage::TextureView.set_ptr(loadfn("glTextureView"));
         storage::TrackMatrixNV.set_ptr(loadfn("glTrackMatrixNV"));
         storage::TransformFeedbackAttribsNV.set_ptr(loadfn("glTransformFeedbackAttribsNV"));
         storage::TransformFeedbackBufferBase.set_ptr(loadfn("glTransformFeedbackBufferBase"));
         storage::TransformFeedbackBufferRange.set_ptr(loadfn("glTransformFeedbackBufferRange"));
         storage::TransformFeedbackStreamAttribsNV.set_ptr(loadfn("glTransformFeedbackStreamAttribsNV"));
         storage::TransformFeedbackVaryings.set_ptr(loadfn("glTransformFeedbackVaryings"));
         storage::TransformFeedbackVaryingsEXT.set_ptr(loadfn("glTransformFeedbackVaryingsEXT"));
         storage::TransformFeedbackVaryingsNV.set_ptr(loadfn("glTransformFeedbackVaryingsNV"));
         storage::TransformPathNV.set_ptr(loadfn("glTransformPathNV"));
         storage::TranslatexOES.set_ptr(loadfn("glTranslatexOES"));
         storage::Uniform1d.set_ptr(loadfn("glUniform1d"));
         storage::Uniform1dv.set_ptr(loadfn("glUniform1dv"));
         storage::Uniform1f.set_ptr(loadfn("glUniform1f"));
         storage::Uniform1fARB.set_ptr(loadfn("glUniform1fARB"));
         storage::Uniform1fv.set_ptr(loadfn("glUniform1fv"));
         storage::Uniform1fvARB.set_ptr(loadfn("glUniform1fvARB"));
         storage::Uniform1i.set_ptr(loadfn("glUniform1i"));
         storage::Uniform1i64ARB.set_ptr(loadfn("glUniform1i64ARB"));
         storage::Uniform1i64NV.set_ptr(loadfn("glUniform1i64NV"));
         storage::Uniform1i64vARB.set_ptr(loadfn("glUniform1i64vARB"));
         storage::Uniform1i64vNV.set_ptr(loadfn("glUniform1i64vNV"));
         storage::Uniform1iARB.set_ptr(loadfn("glUniform1iARB"));
         storage::Uniform1iv.set_ptr(loadfn("glUniform1iv"));
         storage::Uniform1ivARB.set_ptr(loadfn("glUniform1ivARB"));
         storage::Uniform1ui.set_ptr(loadfn("glUniform1ui"));
         storage::Uniform1ui64ARB.set_ptr(loadfn("glUniform1ui64ARB"));
         storage::Uniform1ui64NV.set_ptr(loadfn("glUniform1ui64NV"));
         storage::Uniform1ui64vARB.set_ptr(loadfn("glUniform1ui64vARB"));
         storage::Uniform1ui64vNV.set_ptr(loadfn("glUniform1ui64vNV"));
         storage::Uniform1uiEXT.set_ptr(loadfn("glUniform1uiEXT"));
         storage::Uniform1uiv.set_ptr(loadfn("glUniform1uiv"));
         storage::Uniform1uivEXT.set_ptr(loadfn("glUniform1uivEXT"));
         storage::Uniform2d.set_ptr(loadfn("glUniform2d"));
         storage::Uniform2dv.set_ptr(loadfn("glUniform2dv"));
         storage::Uniform2f.set_ptr(loadfn("glUniform2f"));
         storage::Uniform2fARB.set_ptr(loadfn("glUniform2fARB"));
         storage::Uniform2fv.set_ptr(loadfn("glUniform2fv"));
         storage::Uniform2fvARB.set_ptr(loadfn("glUniform2fvARB"));
         storage::Uniform2i.set_ptr(loadfn("glUniform2i"));
         storage::Uniform2i64ARB.set_ptr(loadfn("glUniform2i64ARB"));
         storage::Uniform2i64NV.set_ptr(loadfn("glUniform2i64NV"));
         storage::Uniform2i64vARB.set_ptr(loadfn("glUniform2i64vARB"));
         storage::Uniform2i64vNV.set_ptr(loadfn("glUniform2i64vNV"));
         storage::Uniform2iARB.set_ptr(loadfn("glUniform2iARB"));
         storage::Uniform2iv.set_ptr(loadfn("glUniform2iv"));
         storage::Uniform2ivARB.set_ptr(loadfn("glUniform2ivARB"));
         storage::Uniform2ui.set_ptr(loadfn("glUniform2ui"));
         storage::Uniform2ui64ARB.set_ptr(loadfn("glUniform2ui64ARB"));
         storage::Uniform2ui64NV.set_ptr(loadfn("glUniform2ui64NV"));
         storage::Uniform2ui64vARB.set_ptr(loadfn("glUniform2ui64vARB"));
         storage::Uniform2ui64vNV.set_ptr(loadfn("glUniform2ui64vNV"));
         storage::Uniform2uiEXT.set_ptr(loadfn("glUniform2uiEXT"));
         storage::Uniform2uiv.set_ptr(loadfn("glUniform2uiv"));
         storage::Uniform2uivEXT.set_ptr(loadfn("glUniform2uivEXT"));
         storage::Uniform3d.set_ptr(loadfn("glUniform3d"));
         storage::Uniform3dv.set_ptr(loadfn("glUniform3dv"));
         storage::Uniform3f.set_ptr(loadfn("glUniform3f"));
         storage::Uniform3fARB.set_ptr(loadfn("glUniform3fARB"));
         storage::Uniform3fv.set_ptr(loadfn("glUniform3fv"));
         storage::Uniform3fvARB.set_ptr(loadfn("glUniform3fvARB"));
         storage::Uniform3i.set_ptr(loadfn("glUniform3i"));
         storage::Uniform3i64ARB.set_ptr(loadfn("glUniform3i64ARB"));
         storage::Uniform3i64NV.set_ptr(loadfn("glUniform3i64NV"));
         storage::Uniform3i64vARB.set_ptr(loadfn("glUniform3i64vARB"));
         storage::Uniform3i64vNV.set_ptr(loadfn("glUniform3i64vNV"));
         storage::Uniform3iARB.set_ptr(loadfn("glUniform3iARB"));
         storage::Uniform3iv.set_ptr(loadfn("glUniform3iv"));
         storage::Uniform3ivARB.set_ptr(loadfn("glUniform3ivARB"));
         storage::Uniform3ui.set_ptr(loadfn("glUniform3ui"));
         storage::Uniform3ui64ARB.set_ptr(loadfn("glUniform3ui64ARB"));
         storage::Uniform3ui64NV.set_ptr(loadfn("glUniform3ui64NV"));
         storage::Uniform3ui64vARB.set_ptr(loadfn("glUniform3ui64vARB"));
         storage::Uniform3ui64vNV.set_ptr(loadfn("glUniform3ui64vNV"));
         storage::Uniform3uiEXT.set_ptr(loadfn("glUniform3uiEXT"));
         storage::Uniform3uiv.set_ptr(loadfn("glUniform3uiv"));
         storage::Uniform3uivEXT.set_ptr(loadfn("glUniform3uivEXT"));
         storage::Uniform4d.set_ptr(loadfn("glUniform4d"));
         storage::Uniform4dv.set_ptr(loadfn("glUniform4dv"));
         storage::Uniform4f.set_ptr(loadfn("glUniform4f"));
         storage::Uniform4fARB.set_ptr(loadfn("glUniform4fARB"));
         storage::Uniform4fv.set_ptr(loadfn("glUniform4fv"));
         storage::Uniform4fvARB.set_ptr(loadfn("glUniform4fvARB"));
         storage::Uniform4i.set_ptr(loadfn("glUniform4i"));
         storage::Uniform4i64ARB.set_ptr(loadfn("glUniform4i64ARB"));
         storage::Uniform4i64NV.set_ptr(loadfn("glUniform4i64NV"));
         storage::Uniform4i64vARB.set_ptr(loadfn("glUniform4i64vARB"));
         storage::Uniform4i64vNV.set_ptr(loadfn("glUniform4i64vNV"));
         storage::Uniform4iARB.set_ptr(loadfn("glUniform4iARB"));
         storage::Uniform4iv.set_ptr(loadfn("glUniform4iv"));
         storage::Uniform4ivARB.set_ptr(loadfn("glUniform4ivARB"));
         storage::Uniform4ui.set_ptr(loadfn("glUniform4ui"));
         storage::Uniform4ui64ARB.set_ptr(loadfn("glUniform4ui64ARB"));
         storage::Uniform4ui64NV.set_ptr(loadfn("glUniform4ui64NV"));
         storage::Uniform4ui64vARB.set_ptr(loadfn("glUniform4ui64vARB"));
         storage::Uniform4ui64vNV.set_ptr(loadfn("glUniform4ui64vNV"));
         storage::Uniform4uiEXT.set_ptr(loadfn("glUniform4uiEXT"));
         storage::Uniform4uiv.set_ptr(loadfn("glUniform4uiv"));
         storage::Uniform4uivEXT.set_ptr(loadfn("glUniform4uivEXT"));
         storage::UniformBlockBinding.set_ptr(loadfn("glUniformBlockBinding"));
         storage::UniformBufferEXT.set_ptr(loadfn("glUniformBufferEXT"));
         storage::UniformHandleui64ARB.set_ptr(loadfn("glUniformHandleui64ARB"));
         storage::UniformHandleui64NV.set_ptr(loadfn("glUniformHandleui64NV"));
         storage::UniformHandleui64vARB.set_ptr(loadfn("glUniformHandleui64vARB"));
         storage::UniformHandleui64vNV.set_ptr(loadfn("glUniformHandleui64vNV"));
         storage::UniformMatrix2dv.set_ptr(loadfn("glUniformMatrix2dv"));
         storage::UniformMatrix2fv.set_ptr(loadfn("glUniformMatrix2fv"));
         storage::UniformMatrix2fvARB.set_ptr(loadfn("glUniformMatrix2fvARB"));
         storage::UniformMatrix2x3dv.set_ptr(loadfn("glUniformMatrix2x3dv"));
         storage::UniformMatrix2x3fv.set_ptr(loadfn("glUniformMatrix2x3fv"));
         storage::UniformMatrix2x4dv.set_ptr(loadfn("glUniformMatrix2x4dv"));
         storage::UniformMatrix2x4fv.set_ptr(loadfn("glUniformMatrix2x4fv"));
         storage::UniformMatrix3dv.set_ptr(loadfn("glUniformMatrix3dv"));
         storage::UniformMatrix3fv.set_ptr(loadfn("glUniformMatrix3fv"));
         storage::UniformMatrix3fvARB.set_ptr(loadfn("glUniformMatrix3fvARB"));
         storage::UniformMatrix3x2dv.set_ptr(loadfn("glUniformMatrix3x2dv"));
         storage::UniformMatrix3x2fv.set_ptr(loadfn("glUniformMatrix3x2fv"));
         storage::UniformMatrix3x4dv.set_ptr(loadfn("glUniformMatrix3x4dv"));
         storage::UniformMatrix3x4fv.set_ptr(loadfn("glUniformMatrix3x4fv"));
         storage::UniformMatrix4dv.set_ptr(loadfn("glUniformMatrix4dv"));
         storage::UniformMatrix4fv.set_ptr(loadfn("glUniformMatrix4fv"));
         storage::UniformMatrix4fvARB.set_ptr(loadfn("glUniformMatrix4fvARB"));
         storage::UniformMatrix4x2dv.set_ptr(loadfn("glUniformMatrix4x2dv"));
         storage::UniformMatrix4x2fv.set_ptr(loadfn("glUniformMatrix4x2fv"));
         storage::UniformMatrix4x3dv.set_ptr(loadfn("glUniformMatrix4x3dv"));
         storage::UniformMatrix4x3fv.set_ptr(loadfn("glUniformMatrix4x3fv"));
         storage::UniformSubroutinesuiv.set_ptr(loadfn("glUniformSubroutinesuiv"));
         storage::Uniformui64NV.set_ptr(loadfn("glUniformui64NV"));
         storage::Uniformui64vNV.set_ptr(loadfn("glUniformui64vNV"));
         storage::UnlockArraysEXT.set_ptr(loadfn("glUnlockArraysEXT"));
         storage::UnmapBuffer.set_ptr(loadfn("glUnmapBuffer"));
         storage::UnmapBufferARB.set_ptr(loadfn("glUnmapBufferARB"));
         storage::UnmapNamedBuffer.set_ptr(loadfn("glUnmapNamedBuffer"));
         storage::UnmapNamedBufferEXT.set_ptr(loadfn("glUnmapNamedBufferEXT"));
         storage::UnmapObjectBufferATI.set_ptr(loadfn("glUnmapObjectBufferATI"));
         storage::UnmapTexture2DINTEL.set_ptr(loadfn("glUnmapTexture2DINTEL"));
         storage::UpdateObjectBufferATI.set_ptr(loadfn("glUpdateObjectBufferATI"));
         storage::UploadGpuMaskNVX.set_ptr(loadfn("glUploadGpuMaskNVX"));
         storage::UseProgram.set_ptr(loadfn("glUseProgram"));
         storage::UseProgramObjectARB.set_ptr(loadfn("glUseProgramObjectARB"));
         storage::UseProgramStages.set_ptr(loadfn("glUseProgramStages"));
         storage::UseShaderProgramEXT.set_ptr(loadfn("glUseShaderProgramEXT"));
         storage::VDPAUFiniNV.set_ptr(loadfn("glVDPAUFiniNV"));
         storage::VDPAUGetSurfaceivNV.set_ptr(loadfn("glVDPAUGetSurfaceivNV"));
         storage::VDPAUInitNV.set_ptr(loadfn("glVDPAUInitNV"));
         storage::VDPAUIsSurfaceNV.set_ptr(loadfn("glVDPAUIsSurfaceNV"));
         storage::VDPAUMapSurfacesNV.set_ptr(loadfn("glVDPAUMapSurfacesNV"));
         storage::VDPAURegisterOutputSurfaceNV.set_ptr(loadfn("glVDPAURegisterOutputSurfaceNV"));
         storage::VDPAURegisterVideoSurfaceNV.set_ptr(loadfn("glVDPAURegisterVideoSurfaceNV"));
         storage::VDPAURegisterVideoSurfaceWithPictureStructureNV.set_ptr(loadfn("glVDPAURegisterVideoSurfaceWithPictureStructureNV"));
         storage::VDPAUSurfaceAccessNV.set_ptr(loadfn("glVDPAUSurfaceAccessNV"));
         storage::VDPAUUnmapSurfacesNV.set_ptr(loadfn("glVDPAUUnmapSurfacesNV"));
         storage::VDPAUUnregisterSurfaceNV.set_ptr(loadfn("glVDPAUUnregisterSurfaceNV"));
         storage::ValidateProgram.set_ptr(loadfn("glValidateProgram"));
         storage::ValidateProgramARB.set_ptr(loadfn("glValidateProgramARB"));
         storage::ValidateProgramPipeline.set_ptr(loadfn("glValidateProgramPipeline"));
         storage::VariantArrayObjectATI.set_ptr(loadfn("glVariantArrayObjectATI"));
         storage::VariantPointerEXT.set_ptr(loadfn("glVariantPointerEXT"));
         storage::VariantbvEXT.set_ptr(loadfn("glVariantbvEXT"));
         storage::VariantdvEXT.set_ptr(loadfn("glVariantdvEXT"));
         storage::VariantfvEXT.set_ptr(loadfn("glVariantfvEXT"));
         storage::VariantivEXT.set_ptr(loadfn("glVariantivEXT"));
         storage::VariantsvEXT.set_ptr(loadfn("glVariantsvEXT"));
         storage::VariantubvEXT.set_ptr(loadfn("glVariantubvEXT"));
         storage::VariantuivEXT.set_ptr(loadfn("glVariantuivEXT"));
         storage::VariantusvEXT.set_ptr(loadfn("glVariantusvEXT"));
         storage::Vertex2bOES.set_ptr(loadfn("glVertex2bOES"));
         storage::Vertex2bvOES.set_ptr(loadfn("glVertex2bvOES"));
         storage::Vertex2hNV.set_ptr(loadfn("glVertex2hNV"));
         storage::Vertex2hvNV.set_ptr(loadfn("glVertex2hvNV"));
         storage::Vertex2xOES.set_ptr(loadfn("glVertex2xOES"));
         storage::Vertex2xvOES.set_ptr(loadfn("glVertex2xvOES"));
         storage::Vertex3bOES.set_ptr(loadfn("glVertex3bOES"));
         storage::Vertex3bvOES.set_ptr(loadfn("glVertex3bvOES"));
         storage::Vertex3hNV.set_ptr(loadfn("glVertex3hNV"));
         storage::Vertex3hvNV.set_ptr(loadfn("glVertex3hvNV"));
         storage::Vertex3xOES.set_ptr(loadfn("glVertex3xOES"));
         storage::Vertex3xvOES.set_ptr(loadfn("glVertex3xvOES"));
         storage::Vertex4bOES.set_ptr(loadfn("glVertex4bOES"));
         storage::Vertex4bvOES.set_ptr(loadfn("glVertex4bvOES"));
         storage::Vertex4hNV.set_ptr(loadfn("glVertex4hNV"));
         storage::Vertex4hvNV.set_ptr(loadfn("glVertex4hvNV"));
         storage::Vertex4xOES.set_ptr(loadfn("glVertex4xOES"));
         storage::Vertex4xvOES.set_ptr(loadfn("glVertex4xvOES"));
         storage::VertexArrayAttribBinding.set_ptr(loadfn("glVertexArrayAttribBinding"));
         storage::VertexArrayAttribFormat.set_ptr(loadfn("glVertexArrayAttribFormat"));
         storage::VertexArrayAttribIFormat.set_ptr(loadfn("glVertexArrayAttribIFormat"));
         storage::VertexArrayAttribLFormat.set_ptr(loadfn("glVertexArrayAttribLFormat"));
         storage::VertexArrayBindVertexBufferEXT.set_ptr(loadfn("glVertexArrayBindVertexBufferEXT"));
         storage::VertexArrayBindingDivisor.set_ptr(loadfn("glVertexArrayBindingDivisor"));
         storage::VertexArrayColorOffsetEXT.set_ptr(loadfn("glVertexArrayColorOffsetEXT"));
         storage::VertexArrayEdgeFlagOffsetEXT.set_ptr(loadfn("glVertexArrayEdgeFlagOffsetEXT"));
         storage::VertexArrayElementBuffer.set_ptr(loadfn("glVertexArrayElementBuffer"));
         storage::VertexArrayFogCoordOffsetEXT.set_ptr(loadfn("glVertexArrayFogCoordOffsetEXT"));
         storage::VertexArrayIndexOffsetEXT.set_ptr(loadfn("glVertexArrayIndexOffsetEXT"));
         storage::VertexArrayMultiTexCoordOffsetEXT.set_ptr(loadfn("glVertexArrayMultiTexCoordOffsetEXT"));
         storage::VertexArrayNormalOffsetEXT.set_ptr(loadfn("glVertexArrayNormalOffsetEXT"));
         storage::VertexArrayParameteriAPPLE.set_ptr(loadfn("glVertexArrayParameteriAPPLE"));
         storage::VertexArrayRangeAPPLE.set_ptr(loadfn("glVertexArrayRangeAPPLE"));
         storage::VertexArrayRangeNV.set_ptr(loadfn("glVertexArrayRangeNV"));
         storage::VertexArraySecondaryColorOffsetEXT.set_ptr(loadfn("glVertexArraySecondaryColorOffsetEXT"));
         storage::VertexArrayTexCoordOffsetEXT.set_ptr(loadfn("glVertexArrayTexCoordOffsetEXT"));
         storage::VertexArrayVertexAttribBindingEXT.set_ptr(loadfn("glVertexArrayVertexAttribBindingEXT"));
         storage::VertexArrayVertexAttribDivisorEXT.set_ptr(loadfn("glVertexArrayVertexAttribDivisorEXT"));
         storage::VertexArrayVertexAttribFormatEXT.set_ptr(loadfn("glVertexArrayVertexAttribFormatEXT"));
         storage::VertexArrayVertexAttribIFormatEXT.set_ptr(loadfn("glVertexArrayVertexAttribIFormatEXT"));
         storage::VertexArrayVertexAttribIOffsetEXT.set_ptr(loadfn("glVertexArrayVertexAttribIOffsetEXT"));
         storage::VertexArrayVertexAttribLFormatEXT.set_ptr(loadfn("glVertexArrayVertexAttribLFormatEXT"));
         storage::VertexArrayVertexAttribLOffsetEXT.set_ptr(loadfn("glVertexArrayVertexAttribLOffsetEXT"));
         storage::VertexArrayVertexAttribOffsetEXT.set_ptr(loadfn("glVertexArrayVertexAttribOffsetEXT"));
         storage::VertexArrayVertexBindingDivisorEXT.set_ptr(loadfn("glVertexArrayVertexBindingDivisorEXT"));
         storage::VertexArrayVertexBuffer.set_ptr(loadfn("glVertexArrayVertexBuffer"));
         storage::VertexArrayVertexBuffers.set_ptr(loadfn("glVertexArrayVertexBuffers"));
         storage::VertexArrayVertexOffsetEXT.set_ptr(loadfn("glVertexArrayVertexOffsetEXT"));
         storage::VertexAttrib1d.set_ptr(loadfn("glVertexAttrib1d"));
         storage::VertexAttrib1dARB.set_ptr(loadfn("glVertexAttrib1dARB"));
         storage::VertexAttrib1dNV.set_ptr(loadfn("glVertexAttrib1dNV"));
         storage::VertexAttrib1dv.set_ptr(loadfn("glVertexAttrib1dv"));
         storage::VertexAttrib1dvARB.set_ptr(loadfn("glVertexAttrib1dvARB"));
         storage::VertexAttrib1dvNV.set_ptr(loadfn("glVertexAttrib1dvNV"));
         storage::VertexAttrib1f.set_ptr(loadfn("glVertexAttrib1f"));
         storage::VertexAttrib1fARB.set_ptr(loadfn("glVertexAttrib1fARB"));
         storage::VertexAttrib1fNV.set_ptr(loadfn("glVertexAttrib1fNV"));
         storage::VertexAttrib1fv.set_ptr(loadfn("glVertexAttrib1fv"));
         storage::VertexAttrib1fvARB.set_ptr(loadfn("glVertexAttrib1fvARB"));
         storage::VertexAttrib1fvNV.set_ptr(loadfn("glVertexAttrib1fvNV"));
         storage::VertexAttrib1hNV.set_ptr(loadfn("glVertexAttrib1hNV"));
         storage::VertexAttrib1hvNV.set_ptr(loadfn("glVertexAttrib1hvNV"));
         storage::VertexAttrib1s.set_ptr(loadfn("glVertexAttrib1s"));
         storage::VertexAttrib1sARB.set_ptr(loadfn("glVertexAttrib1sARB"));
         storage::VertexAttrib1sNV.set_ptr(loadfn("glVertexAttrib1sNV"));
         storage::VertexAttrib1sv.set_ptr(loadfn("glVertexAttrib1sv"));
         storage::VertexAttrib1svARB.set_ptr(loadfn("glVertexAttrib1svARB"));
         storage::VertexAttrib1svNV.set_ptr(loadfn("glVertexAttrib1svNV"));
         storage::VertexAttrib2d.set_ptr(loadfn("glVertexAttrib2d"));
         storage::VertexAttrib2dARB.set_ptr(loadfn("glVertexAttrib2dARB"));
         storage::VertexAttrib2dNV.set_ptr(loadfn("glVertexAttrib2dNV"));
         storage::VertexAttrib2dv.set_ptr(loadfn("glVertexAttrib2dv"));
         storage::VertexAttrib2dvARB.set_ptr(loadfn("glVertexAttrib2dvARB"));
         storage::VertexAttrib2dvNV.set_ptr(loadfn("glVertexAttrib2dvNV"));
         storage::VertexAttrib2f.set_ptr(loadfn("glVertexAttrib2f"));
         storage::VertexAttrib2fARB.set_ptr(loadfn("glVertexAttrib2fARB"));
         storage::VertexAttrib2fNV.set_ptr(loadfn("glVertexAttrib2fNV"));
         storage::VertexAttrib2fv.set_ptr(loadfn("glVertexAttrib2fv"));
         storage::VertexAttrib2fvARB.set_ptr(loadfn("glVertexAttrib2fvARB"));
         storage::VertexAttrib2fvNV.set_ptr(loadfn("glVertexAttrib2fvNV"));
         storage::VertexAttrib2hNV.set_ptr(loadfn("glVertexAttrib2hNV"));
         storage::VertexAttrib2hvNV.set_ptr(loadfn("glVertexAttrib2hvNV"));
         storage::VertexAttrib2s.set_ptr(loadfn("glVertexAttrib2s"));
         storage::VertexAttrib2sARB.set_ptr(loadfn("glVertexAttrib2sARB"));
         storage::VertexAttrib2sNV.set_ptr(loadfn("glVertexAttrib2sNV"));
         storage::VertexAttrib2sv.set_ptr(loadfn("glVertexAttrib2sv"));
         storage::VertexAttrib2svARB.set_ptr(loadfn("glVertexAttrib2svARB"));
         storage::VertexAttrib2svNV.set_ptr(loadfn("glVertexAttrib2svNV"));
         storage::VertexAttrib3d.set_ptr(loadfn("glVertexAttrib3d"));
         storage::VertexAttrib3dARB.set_ptr(loadfn("glVertexAttrib3dARB"));
         storage::VertexAttrib3dNV.set_ptr(loadfn("glVertexAttrib3dNV"));
         storage::VertexAttrib3dv.set_ptr(loadfn("glVertexAttrib3dv"));
         storage::VertexAttrib3dvARB.set_ptr(loadfn("glVertexAttrib3dvARB"));
         storage::VertexAttrib3dvNV.set_ptr(loadfn("glVertexAttrib3dvNV"));
         storage::VertexAttrib3f.set_ptr(loadfn("glVertexAttrib3f"));
         storage::VertexAttrib3fARB.set_ptr(loadfn("glVertexAttrib3fARB"));
         storage::VertexAttrib3fNV.set_ptr(loadfn("glVertexAttrib3fNV"));
         storage::VertexAttrib3fv.set_ptr(loadfn("glVertexAttrib3fv"));
         storage::VertexAttrib3fvARB.set_ptr(loadfn("glVertexAttrib3fvARB"));
         storage::VertexAttrib3fvNV.set_ptr(loadfn("glVertexAttrib3fvNV"));
         storage::VertexAttrib3hNV.set_ptr(loadfn("glVertexAttrib3hNV"));
         storage::VertexAttrib3hvNV.set_ptr(loadfn("glVertexAttrib3hvNV"));
         storage::VertexAttrib3s.set_ptr(loadfn("glVertexAttrib3s"));
         storage::VertexAttrib3sARB.set_ptr(loadfn("glVertexAttrib3sARB"));
         storage::VertexAttrib3sNV.set_ptr(loadfn("glVertexAttrib3sNV"));
         storage::VertexAttrib3sv.set_ptr(loadfn("glVertexAttrib3sv"));
         storage::VertexAttrib3svARB.set_ptr(loadfn("glVertexAttrib3svARB"));
         storage::VertexAttrib3svNV.set_ptr(loadfn("glVertexAttrib3svNV"));
         storage::VertexAttrib4Nbv.set_ptr(loadfn("glVertexAttrib4Nbv"));
         storage::VertexAttrib4NbvARB.set_ptr(loadfn("glVertexAttrib4NbvARB"));
         storage::VertexAttrib4Niv.set_ptr(loadfn("glVertexAttrib4Niv"));
         storage::VertexAttrib4NivARB.set_ptr(loadfn("glVertexAttrib4NivARB"));
         storage::VertexAttrib4Nsv.set_ptr(loadfn("glVertexAttrib4Nsv"));
         storage::VertexAttrib4NsvARB.set_ptr(loadfn("glVertexAttrib4NsvARB"));
         storage::VertexAttrib4Nub.set_ptr(loadfn("glVertexAttrib4Nub"));
         storage::VertexAttrib4NubARB.set_ptr(loadfn("glVertexAttrib4NubARB"));
         storage::VertexAttrib4Nubv.set_ptr(loadfn("glVertexAttrib4Nubv"));
         storage::VertexAttrib4NubvARB.set_ptr(loadfn("glVertexAttrib4NubvARB"));
         storage::VertexAttrib4Nuiv.set_ptr(loadfn("glVertexAttrib4Nuiv"));
         storage::VertexAttrib4NuivARB.set_ptr(loadfn("glVertexAttrib4NuivARB"));
         storage::VertexAttrib4Nusv.set_ptr(loadfn("glVertexAttrib4Nusv"));
         storage::VertexAttrib4NusvARB.set_ptr(loadfn("glVertexAttrib4NusvARB"));
         storage::VertexAttrib4bv.set_ptr(loadfn("glVertexAttrib4bv"));
         storage::VertexAttrib4bvARB.set_ptr(loadfn("glVertexAttrib4bvARB"));
         storage::VertexAttrib4d.set_ptr(loadfn("glVertexAttrib4d"));
         storage::VertexAttrib4dARB.set_ptr(loadfn("glVertexAttrib4dARB"));
         storage::VertexAttrib4dNV.set_ptr(loadfn("glVertexAttrib4dNV"));
         storage::VertexAttrib4dv.set_ptr(loadfn("glVertexAttrib4dv"));
         storage::VertexAttrib4dvARB.set_ptr(loadfn("glVertexAttrib4dvARB"));
         storage::VertexAttrib4dvNV.set_ptr(loadfn("glVertexAttrib4dvNV"));
         storage::VertexAttrib4f.set_ptr(loadfn("glVertexAttrib4f"));
         storage::VertexAttrib4fARB.set_ptr(loadfn("glVertexAttrib4fARB"));
         storage::VertexAttrib4fNV.set_ptr(loadfn("glVertexAttrib4fNV"));
         storage::VertexAttrib4fv.set_ptr(loadfn("glVertexAttrib4fv"));
         storage::VertexAttrib4fvARB.set_ptr(loadfn("glVertexAttrib4fvARB"));
         storage::VertexAttrib4fvNV.set_ptr(loadfn("glVertexAttrib4fvNV"));
         storage::VertexAttrib4hNV.set_ptr(loadfn("glVertexAttrib4hNV"));
         storage::VertexAttrib4hvNV.set_ptr(loadfn("glVertexAttrib4hvNV"));
         storage::VertexAttrib4iv.set_ptr(loadfn("glVertexAttrib4iv"));
         storage::VertexAttrib4ivARB.set_ptr(loadfn("glVertexAttrib4ivARB"));
         storage::VertexAttrib4s.set_ptr(loadfn("glVertexAttrib4s"));
         storage::VertexAttrib4sARB.set_ptr(loadfn("glVertexAttrib4sARB"));
         storage::VertexAttrib4sNV.set_ptr(loadfn("glVertexAttrib4sNV"));
         storage::VertexAttrib4sv.set_ptr(loadfn("glVertexAttrib4sv"));
         storage::VertexAttrib4svARB.set_ptr(loadfn("glVertexAttrib4svARB"));
         storage::VertexAttrib4svNV.set_ptr(loadfn("glVertexAttrib4svNV"));
         storage::VertexAttrib4ubNV.set_ptr(loadfn("glVertexAttrib4ubNV"));
         storage::VertexAttrib4ubv.set_ptr(loadfn("glVertexAttrib4ubv"));
         storage::VertexAttrib4ubvARB.set_ptr(loadfn("glVertexAttrib4ubvARB"));
         storage::VertexAttrib4ubvNV.set_ptr(loadfn("glVertexAttrib4ubvNV"));
         storage::VertexAttrib4uiv.set_ptr(loadfn("glVertexAttrib4uiv"));
         storage::VertexAttrib4uivARB.set_ptr(loadfn("glVertexAttrib4uivARB"));
         storage::VertexAttrib4usv.set_ptr(loadfn("glVertexAttrib4usv"));
         storage::VertexAttrib4usvARB.set_ptr(loadfn("glVertexAttrib4usvARB"));
         storage::VertexAttribArrayObjectATI.set_ptr(loadfn("glVertexAttribArrayObjectATI"));
         storage::VertexAttribBinding.set_ptr(loadfn("glVertexAttribBinding"));
         storage::VertexAttribDivisor.set_ptr(loadfn("glVertexAttribDivisor"));
         storage::VertexAttribDivisorARB.set_ptr(loadfn("glVertexAttribDivisorARB"));
         storage::VertexAttribFormat.set_ptr(loadfn("glVertexAttribFormat"));
         storage::VertexAttribFormatNV.set_ptr(loadfn("glVertexAttribFormatNV"));
         storage::VertexAttribI1i.set_ptr(loadfn("glVertexAttribI1i"));
         storage::VertexAttribI1iEXT.set_ptr(loadfn("glVertexAttribI1iEXT"));
         storage::VertexAttribI1iv.set_ptr(loadfn("glVertexAttribI1iv"));
         storage::VertexAttribI1ivEXT.set_ptr(loadfn("glVertexAttribI1ivEXT"));
         storage::VertexAttribI1ui.set_ptr(loadfn("glVertexAttribI1ui"));
         storage::VertexAttribI1uiEXT.set_ptr(loadfn("glVertexAttribI1uiEXT"));
         storage::VertexAttribI1uiv.set_ptr(loadfn("glVertexAttribI1uiv"));
         storage::VertexAttribI1uivEXT.set_ptr(loadfn("glVertexAttribI1uivEXT"));
         storage::VertexAttribI2i.set_ptr(loadfn("glVertexAttribI2i"));
         storage::VertexAttribI2iEXT.set_ptr(loadfn("glVertexAttribI2iEXT"));
         storage::VertexAttribI2iv.set_ptr(loadfn("glVertexAttribI2iv"));
         storage::VertexAttribI2ivEXT.set_ptr(loadfn("glVertexAttribI2ivEXT"));
         storage::VertexAttribI2ui.set_ptr(loadfn("glVertexAttribI2ui"));
         storage::VertexAttribI2uiEXT.set_ptr(loadfn("glVertexAttribI2uiEXT"));
         storage::VertexAttribI2uiv.set_ptr(loadfn("glVertexAttribI2uiv"));
         storage::VertexAttribI2uivEXT.set_ptr(loadfn("glVertexAttribI2uivEXT"));
         storage::VertexAttribI3i.set_ptr(loadfn("glVertexAttribI3i"));
         storage::VertexAttribI3iEXT.set_ptr(loadfn("glVertexAttribI3iEXT"));
         storage::VertexAttribI3iv.set_ptr(loadfn("glVertexAttribI3iv"));
         storage::VertexAttribI3ivEXT.set_ptr(loadfn("glVertexAttribI3ivEXT"));
         storage::VertexAttribI3ui.set_ptr(loadfn("glVertexAttribI3ui"));
         storage::VertexAttribI3uiEXT.set_ptr(loadfn("glVertexAttribI3uiEXT"));
         storage::VertexAttribI3uiv.set_ptr(loadfn("glVertexAttribI3uiv"));
         storage::VertexAttribI3uivEXT.set_ptr(loadfn("glVertexAttribI3uivEXT"));
         storage::VertexAttribI4bv.set_ptr(loadfn("glVertexAttribI4bv"));
         storage::VertexAttribI4bvEXT.set_ptr(loadfn("glVertexAttribI4bvEXT"));
         storage::VertexAttribI4i.set_ptr(loadfn("glVertexAttribI4i"));
         storage::VertexAttribI4iEXT.set_ptr(loadfn("glVertexAttribI4iEXT"));
         storage::VertexAttribI4iv.set_ptr(loadfn("glVertexAttribI4iv"));
         storage::VertexAttribI4ivEXT.set_ptr(loadfn("glVertexAttribI4ivEXT"));
         storage::VertexAttribI4sv.set_ptr(loadfn("glVertexAttribI4sv"));
         storage::VertexAttribI4svEXT.set_ptr(loadfn("glVertexAttribI4svEXT"));
         storage::VertexAttribI4ubv.set_ptr(loadfn("glVertexAttribI4ubv"));
         storage::VertexAttribI4ubvEXT.set_ptr(loadfn("glVertexAttribI4ubvEXT"));
         storage::VertexAttribI4ui.set_ptr(loadfn("glVertexAttribI4ui"));
         storage::VertexAttribI4uiEXT.set_ptr(loadfn("glVertexAttribI4uiEXT"));
         storage::VertexAttribI4uiv.set_ptr(loadfn("glVertexAttribI4uiv"));
         storage::VertexAttribI4uivEXT.set_ptr(loadfn("glVertexAttribI4uivEXT"));
         storage::VertexAttribI4usv.set_ptr(loadfn("glVertexAttribI4usv"));
         storage::VertexAttribI4usvEXT.set_ptr(loadfn("glVertexAttribI4usvEXT"));
         storage::VertexAttribIFormat.set_ptr(loadfn("glVertexAttribIFormat"));
         storage::VertexAttribIFormatNV.set_ptr(loadfn("glVertexAttribIFormatNV"));
         storage::VertexAttribIPointer.set_ptr(loadfn("glVertexAttribIPointer"));
         storage::VertexAttribIPointerEXT.set_ptr(loadfn("glVertexAttribIPointerEXT"));
         storage::VertexAttribL1d.set_ptr(loadfn("glVertexAttribL1d"));
         storage::VertexAttribL1dEXT.set_ptr(loadfn("glVertexAttribL1dEXT"));
         storage::VertexAttribL1dv.set_ptr(loadfn("glVertexAttribL1dv"));
         storage::VertexAttribL1dvEXT.set_ptr(loadfn("glVertexAttribL1dvEXT"));
         storage::VertexAttribL1i64NV.set_ptr(loadfn("glVertexAttribL1i64NV"));
         storage::VertexAttribL1i64vNV.set_ptr(loadfn("glVertexAttribL1i64vNV"));
         storage::VertexAttribL1ui64ARB.set_ptr(loadfn("glVertexAttribL1ui64ARB"));
         storage::VertexAttribL1ui64NV.set_ptr(loadfn("glVertexAttribL1ui64NV"));
         storage::VertexAttribL1ui64vARB.set_ptr(loadfn("glVertexAttribL1ui64vARB"));
         storage::VertexAttribL1ui64vNV.set_ptr(loadfn("glVertexAttribL1ui64vNV"));
         storage::VertexAttribL2d.set_ptr(loadfn("glVertexAttribL2d"));
         storage::VertexAttribL2dEXT.set_ptr(loadfn("glVertexAttribL2dEXT"));
         storage::VertexAttribL2dv.set_ptr(loadfn("glVertexAttribL2dv"));
         storage::VertexAttribL2dvEXT.set_ptr(loadfn("glVertexAttribL2dvEXT"));
         storage::VertexAttribL2i64NV.set_ptr(loadfn("glVertexAttribL2i64NV"));
         storage::VertexAttribL2i64vNV.set_ptr(loadfn("glVertexAttribL2i64vNV"));
         storage::VertexAttribL2ui64NV.set_ptr(loadfn("glVertexAttribL2ui64NV"));
         storage::VertexAttribL2ui64vNV.set_ptr(loadfn("glVertexAttribL2ui64vNV"));
         storage::VertexAttribL3d.set_ptr(loadfn("glVertexAttribL3d"));
         storage::VertexAttribL3dEXT.set_ptr(loadfn("glVertexAttribL3dEXT"));
         storage::VertexAttribL3dv.set_ptr(loadfn("glVertexAttribL3dv"));
         storage::VertexAttribL3dvEXT.set_ptr(loadfn("glVertexAttribL3dvEXT"));
         storage::VertexAttribL3i64NV.set_ptr(loadfn("glVertexAttribL3i64NV"));
         storage::VertexAttribL3i64vNV.set_ptr(loadfn("glVertexAttribL3i64vNV"));
         storage::VertexAttribL3ui64NV.set_ptr(loadfn("glVertexAttribL3ui64NV"));
         storage::VertexAttribL3ui64vNV.set_ptr(loadfn("glVertexAttribL3ui64vNV"));
         storage::VertexAttribL4d.set_ptr(loadfn("glVertexAttribL4d"));
         storage::VertexAttribL4dEXT.set_ptr(loadfn("glVertexAttribL4dEXT"));
         storage::VertexAttribL4dv.set_ptr(loadfn("glVertexAttribL4dv"));
         storage::VertexAttribL4dvEXT.set_ptr(loadfn("glVertexAttribL4dvEXT"));
         storage::VertexAttribL4i64NV.set_ptr(loadfn("glVertexAttribL4i64NV"));
         storage::VertexAttribL4i64vNV.set_ptr(loadfn("glVertexAttribL4i64vNV"));
         storage::VertexAttribL4ui64NV.set_ptr(loadfn("glVertexAttribL4ui64NV"));
         storage::VertexAttribL4ui64vNV.set_ptr(loadfn("glVertexAttribL4ui64vNV"));
         storage::VertexAttribLFormat.set_ptr(loadfn("glVertexAttribLFormat"));
         storage::VertexAttribLFormatNV.set_ptr(loadfn("glVertexAttribLFormatNV"));
         storage::VertexAttribLPointer.set_ptr(loadfn("glVertexAttribLPointer"));
         storage::VertexAttribLPointerEXT.set_ptr(loadfn("glVertexAttribLPointerEXT"));
         storage::VertexAttribP1ui.set_ptr(loadfn("glVertexAttribP1ui"));
         storage::VertexAttribP1uiv.set_ptr(loadfn("glVertexAttribP1uiv"));
         storage::VertexAttribP2ui.set_ptr(loadfn("glVertexAttribP2ui"));
         storage::VertexAttribP2uiv.set_ptr(loadfn("glVertexAttribP2uiv"));
         storage::VertexAttribP3ui.set_ptr(loadfn("glVertexAttribP3ui"));
         storage::VertexAttribP3uiv.set_ptr(loadfn("glVertexAttribP3uiv"));
         storage::VertexAttribP4ui.set_ptr(loadfn("glVertexAttribP4ui"));
         storage::VertexAttribP4uiv.set_ptr(loadfn("glVertexAttribP4uiv"));
         storage::VertexAttribParameteriAMD.set_ptr(loadfn("glVertexAttribParameteriAMD"));
         storage::VertexAttribPointer.set_ptr(loadfn("glVertexAttribPointer"));
         storage::VertexAttribPointerARB.set_ptr(loadfn("glVertexAttribPointerARB"));
         storage::VertexAttribPointerNV.set_ptr(loadfn("glVertexAttribPointerNV"));
         storage::VertexAttribs1dvNV.set_ptr(loadfn("glVertexAttribs1dvNV"));
         storage::VertexAttribs1fvNV.set_ptr(loadfn("glVertexAttribs1fvNV"));
         storage::VertexAttribs1hvNV.set_ptr(loadfn("glVertexAttribs1hvNV"));
         storage::VertexAttribs1svNV.set_ptr(loadfn("glVertexAttribs1svNV"));
         storage::VertexAttribs2dvNV.set_ptr(loadfn("glVertexAttribs2dvNV"));
         storage::VertexAttribs2fvNV.set_ptr(loadfn("glVertexAttribs2fvNV"));
         storage::VertexAttribs2hvNV.set_ptr(loadfn("glVertexAttribs2hvNV"));
         storage::VertexAttribs2svNV.set_ptr(loadfn("glVertexAttribs2svNV"));
         storage::VertexAttribs3dvNV.set_ptr(loadfn("glVertexAttribs3dvNV"));
         storage::VertexAttribs3fvNV.set_ptr(loadfn("glVertexAttribs3fvNV"));
         storage::VertexAttribs3hvNV.set_ptr(loadfn("glVertexAttribs3hvNV"));
         storage::VertexAttribs3svNV.set_ptr(loadfn("glVertexAttribs3svNV"));
         storage::VertexAttribs4dvNV.set_ptr(loadfn("glVertexAttribs4dvNV"));
         storage::VertexAttribs4fvNV.set_ptr(loadfn("glVertexAttribs4fvNV"));
         storage::VertexAttribs4hvNV.set_ptr(loadfn("glVertexAttribs4hvNV"));
         storage::VertexAttribs4svNV.set_ptr(loadfn("glVertexAttribs4svNV"));
         storage::VertexAttribs4ubvNV.set_ptr(loadfn("glVertexAttribs4ubvNV"));
         storage::VertexBindingDivisor.set_ptr(loadfn("glVertexBindingDivisor"));
         storage::VertexBlendARB.set_ptr(loadfn("glVertexBlendARB"));
         storage::VertexBlendEnvfATI.set_ptr(loadfn("glVertexBlendEnvfATI"));
         storage::VertexBlendEnviATI.set_ptr(loadfn("glVertexBlendEnviATI"));
         storage::VertexFormatNV.set_ptr(loadfn("glVertexFormatNV"));
         storage::VertexPointerEXT.set_ptr(loadfn("glVertexPointerEXT"));
         storage::VertexPointerListIBM.set_ptr(loadfn("glVertexPointerListIBM"));
         storage::VertexPointervINTEL.set_ptr(loadfn("glVertexPointervINTEL"));
         storage::VertexStream1dATI.set_ptr(loadfn("glVertexStream1dATI"));
         storage::VertexStream1dvATI.set_ptr(loadfn("glVertexStream1dvATI"));
         storage::VertexStream1fATI.set_ptr(loadfn("glVertexStream1fATI"));
         storage::VertexStream1fvATI.set_ptr(loadfn("glVertexStream1fvATI"));
         storage::VertexStream1iATI.set_ptr(loadfn("glVertexStream1iATI"));
         storage::VertexStream1ivATI.set_ptr(loadfn("glVertexStream1ivATI"));
         storage::VertexStream1sATI.set_ptr(loadfn("glVertexStream1sATI"));
         storage::VertexStream1svATI.set_ptr(loadfn("glVertexStream1svATI"));
         storage::VertexStream2dATI.set_ptr(loadfn("glVertexStream2dATI"));
         storage::VertexStream2dvATI.set_ptr(loadfn("glVertexStream2dvATI"));
         storage::VertexStream2fATI.set_ptr(loadfn("glVertexStream2fATI"));
         storage::VertexStream2fvATI.set_ptr(loadfn("glVertexStream2fvATI"));
         storage::VertexStream2iATI.set_ptr(loadfn("glVertexStream2iATI"));
         storage::VertexStream2ivATI.set_ptr(loadfn("glVertexStream2ivATI"));
         storage::VertexStream2sATI.set_ptr(loadfn("glVertexStream2sATI"));
         storage::VertexStream2svATI.set_ptr(loadfn("glVertexStream2svATI"));
         storage::VertexStream3dATI.set_ptr(loadfn("glVertexStream3dATI"));
         storage::VertexStream3dvATI.set_ptr(loadfn("glVertexStream3dvATI"));
         storage::VertexStream3fATI.set_ptr(loadfn("glVertexStream3fATI"));
         storage::VertexStream3fvATI.set_ptr(loadfn("glVertexStream3fvATI"));
         storage::VertexStream3iATI.set_ptr(loadfn("glVertexStream3iATI"));
         storage::VertexStream3ivATI.set_ptr(loadfn("glVertexStream3ivATI"));
         storage::VertexStream3sATI.set_ptr(loadfn("glVertexStream3sATI"));
         storage::VertexStream3svATI.set_ptr(loadfn("glVertexStream3svATI"));
         storage::VertexStream4dATI.set_ptr(loadfn("glVertexStream4dATI"));
         storage::VertexStream4dvATI.set_ptr(loadfn("glVertexStream4dvATI"));
         storage::VertexStream4fATI.set_ptr(loadfn("glVertexStream4fATI"));
         storage::VertexStream4fvATI.set_ptr(loadfn("glVertexStream4fvATI"));
         storage::VertexStream4iATI.set_ptr(loadfn("glVertexStream4iATI"));
         storage::VertexStream4ivATI.set_ptr(loadfn("glVertexStream4ivATI"));
         storage::VertexStream4sATI.set_ptr(loadfn("glVertexStream4sATI"));
         storage::VertexStream4svATI.set_ptr(loadfn("glVertexStream4svATI"));
         storage::VertexWeightPointerEXT.set_ptr(loadfn("glVertexWeightPointerEXT"));
         storage::VertexWeightfEXT.set_ptr(loadfn("glVertexWeightfEXT"));
         storage::VertexWeightfvEXT.set_ptr(loadfn("glVertexWeightfvEXT"));
         storage::VertexWeighthNV.set_ptr(loadfn("glVertexWeighthNV"));
         storage::VertexWeighthvNV.set_ptr(loadfn("glVertexWeighthvNV"));
         storage::VideoCaptureNV.set_ptr(loadfn("glVideoCaptureNV"));
         storage::VideoCaptureStreamParameterdvNV.set_ptr(loadfn("glVideoCaptureStreamParameterdvNV"));
         storage::VideoCaptureStreamParameterfvNV.set_ptr(loadfn("glVideoCaptureStreamParameterfvNV"));
         storage::VideoCaptureStreamParameterivNV.set_ptr(loadfn("glVideoCaptureStreamParameterivNV"));
         storage::Viewport.set_ptr(loadfn("glViewport"));
         storage::ViewportArrayv.set_ptr(loadfn("glViewportArrayv"));
         storage::ViewportIndexedf.set_ptr(loadfn("glViewportIndexedf"));
         storage::ViewportIndexedfv.set_ptr(loadfn("glViewportIndexedfv"));
         storage::ViewportPositionWScaleNV.set_ptr(loadfn("glViewportPositionWScaleNV"));
         storage::ViewportSwizzleNV.set_ptr(loadfn("glViewportSwizzleNV"));
         storage::WaitSemaphoreEXT.set_ptr(loadfn("glWaitSemaphoreEXT"));
         storage::WaitSemaphoreui64NVX.set_ptr(loadfn("glWaitSemaphoreui64NVX"));
         storage::WaitSync.set_ptr(loadfn("glWaitSync"));
         storage::WaitVkSemaphoreNV.set_ptr(loadfn("glWaitVkSemaphoreNV"));
         storage::WeightPathsNV.set_ptr(loadfn("glWeightPathsNV"));
         storage::WeightPointerARB.set_ptr(loadfn("glWeightPointerARB"));
         storage::WeightbvARB.set_ptr(loadfn("glWeightbvARB"));
         storage::WeightdvARB.set_ptr(loadfn("glWeightdvARB"));
         storage::WeightfvARB.set_ptr(loadfn("glWeightfvARB"));
         storage::WeightivARB.set_ptr(loadfn("glWeightivARB"));
         storage::WeightsvARB.set_ptr(loadfn("glWeightsvARB"));
         storage::WeightubvARB.set_ptr(loadfn("glWeightubvARB"));
         storage::WeightuivARB.set_ptr(loadfn("glWeightuivARB"));
         storage::WeightusvARB.set_ptr(loadfn("glWeightusvARB"));
         storage::WindowPos2dARB.set_ptr(loadfn("glWindowPos2dARB"));
         storage::WindowPos2dMESA.set_ptr(loadfn("glWindowPos2dMESA"));
         storage::WindowPos2dvARB.set_ptr(loadfn("glWindowPos2dvARB"));
         storage::WindowPos2dvMESA.set_ptr(loadfn("glWindowPos2dvMESA"));
         storage::WindowPos2fARB.set_ptr(loadfn("glWindowPos2fARB"));
         storage::WindowPos2fMESA.set_ptr(loadfn("glWindowPos2fMESA"));
         storage::WindowPos2fvARB.set_ptr(loadfn("glWindowPos2fvARB"));
         storage::WindowPos2fvMESA.set_ptr(loadfn("glWindowPos2fvMESA"));
         storage::WindowPos2iARB.set_ptr(loadfn("glWindowPos2iARB"));
         storage::WindowPos2iMESA.set_ptr(loadfn("glWindowPos2iMESA"));
         storage::WindowPos2ivARB.set_ptr(loadfn("glWindowPos2ivARB"));
         storage::WindowPos2ivMESA.set_ptr(loadfn("glWindowPos2ivMESA"));
         storage::WindowPos2sARB.set_ptr(loadfn("glWindowPos2sARB"));
         storage::WindowPos2sMESA.set_ptr(loadfn("glWindowPos2sMESA"));
         storage::WindowPos2svARB.set_ptr(loadfn("glWindowPos2svARB"));
         storage::WindowPos2svMESA.set_ptr(loadfn("glWindowPos2svMESA"));
         storage::WindowPos3dARB.set_ptr(loadfn("glWindowPos3dARB"));
         storage::WindowPos3dMESA.set_ptr(loadfn("glWindowPos3dMESA"));
         storage::WindowPos3dvARB.set_ptr(loadfn("glWindowPos3dvARB"));
         storage::WindowPos3dvMESA.set_ptr(loadfn("glWindowPos3dvMESA"));
         storage::WindowPos3fARB.set_ptr(loadfn("glWindowPos3fARB"));
         storage::WindowPos3fMESA.set_ptr(loadfn("glWindowPos3fMESA"));
         storage::WindowPos3fvARB.set_ptr(loadfn("glWindowPos3fvARB"));
         storage::WindowPos3fvMESA.set_ptr(loadfn("glWindowPos3fvMESA"));
         storage::WindowPos3iARB.set_ptr(loadfn("glWindowPos3iARB"));
         storage::WindowPos3iMESA.set_ptr(loadfn("glWindowPos3iMESA"));
         storage::WindowPos3ivARB.set_ptr(loadfn("glWindowPos3ivARB"));
         storage::WindowPos3ivMESA.set_ptr(loadfn("glWindowPos3ivMESA"));
         storage::WindowPos3sARB.set_ptr(loadfn("glWindowPos3sARB"));
         storage::WindowPos3sMESA.set_ptr(loadfn("glWindowPos3sMESA"));
         storage::WindowPos3svARB.set_ptr(loadfn("glWindowPos3svARB"));
         storage::WindowPos3svMESA.set_ptr(loadfn("glWindowPos3svMESA"));
         storage::WindowPos4dMESA.set_ptr(loadfn("glWindowPos4dMESA"));
         storage::WindowPos4dvMESA.set_ptr(loadfn("glWindowPos4dvMESA"));
         storage::WindowPos4fMESA.set_ptr(loadfn("glWindowPos4fMESA"));
         storage::WindowPos4fvMESA.set_ptr(loadfn("glWindowPos4fvMESA"));
         storage::WindowPos4iMESA.set_ptr(loadfn("glWindowPos4iMESA"));
         storage::WindowPos4ivMESA.set_ptr(loadfn("glWindowPos4ivMESA"));
         storage::WindowPos4sMESA.set_ptr(loadfn("glWindowPos4sMESA"));
         storage::WindowPos4svMESA.set_ptr(loadfn("glWindowPos4svMESA"));
         storage::WindowRectanglesEXT.set_ptr(loadfn("glWindowRectanglesEXT"));
         storage::WriteMaskEXT.set_ptr(loadfn("glWriteMaskEXT"));

         storage::ActiveTexture.aliased(&storage::ActiveTextureARB);
         storage::ActiveTextureARB.aliased(&storage::ActiveTexture);
         storage::AttachObjectARB.aliased(&storage::AttachShader);
         storage::AttachShader.aliased(&storage::AttachObjectARB);
         storage::BeginConditionalRender.aliased(&storage::BeginConditionalRenderNV);
         storage::BeginConditionalRenderNV.aliased(&storage::BeginConditionalRender);
         storage::BeginQuery.aliased(&storage::BeginQueryARB);
         storage::BeginQueryARB.aliased(&storage::BeginQuery);
         storage::BeginTransformFeedback.aliased(&storage::BeginTransformFeedbackEXT);
         storage::BeginTransformFeedback.aliased(&storage::BeginTransformFeedbackNV);
         storage::BeginTransformFeedbackEXT.aliased(&storage::BeginTransformFeedback);
         storage::BeginTransformFeedbackEXT.aliased(&storage::BeginTransformFeedbackNV);
         storage::BeginTransformFeedbackNV.aliased(&storage::BeginTransformFeedback);
         storage::BeginTransformFeedbackNV.aliased(&storage::BeginTransformFeedbackEXT);
         storage::BindAttribLocation.aliased(&storage::BindAttribLocationARB);
         storage::BindAttribLocationARB.aliased(&storage::BindAttribLocation);
         storage::BindBuffer.aliased(&storage::BindBufferARB);
         storage::BindBufferARB.aliased(&storage::BindBuffer);
         storage::BindBufferBase.aliased(&storage::BindBufferBaseEXT);
         storage::BindBufferBase.aliased(&storage::BindBufferBaseNV);
         storage::BindBufferBaseEXT.aliased(&storage::BindBufferBase);
         storage::BindBufferBaseEXT.aliased(&storage::BindBufferBaseNV);
         storage::BindBufferBaseNV.aliased(&storage::BindBufferBase);
         storage::BindBufferBaseNV.aliased(&storage::BindBufferBaseEXT);
         storage::BindBufferOffsetEXT.aliased(&storage::BindBufferOffsetNV);
         storage::BindBufferOffsetNV.aliased(&storage::BindBufferOffsetEXT);
         storage::BindBufferRange.aliased(&storage::BindBufferRangeEXT);
         storage::BindBufferRange.aliased(&storage::BindBufferRangeNV);
         storage::BindBufferRangeEXT.aliased(&storage::BindBufferRange);
         storage::BindBufferRangeEXT.aliased(&storage::BindBufferRangeNV);
         storage::BindBufferRangeNV.aliased(&storage::BindBufferRange);
         storage::BindBufferRangeNV.aliased(&storage::BindBufferRangeEXT);
         storage::BindFragDataLocation.aliased(&storage::BindFragDataLocationEXT);
         storage::BindFragDataLocationEXT.aliased(&storage::BindFragDataLocation);
         storage::BindProgramARB.aliased(&storage::BindProgramNV);
         storage::BindProgramNV.aliased(&storage::BindProgramARB);
         storage::BindTexture.aliased(&storage::BindTextureEXT);
         storage::BindTextureEXT.aliased(&storage::BindTexture);
         storage::BlendColor.aliased(&storage::BlendColorEXT);
         storage::BlendColorEXT.aliased(&storage::BlendColor);
         storage::BlendEquation.aliased(&storage::BlendEquationEXT);
         storage::BlendEquationEXT.aliased(&storage::BlendEquation);
         storage::BlendEquationi.aliased(&storage::BlendEquationIndexedAMD);
         storage::BlendEquationi.aliased(&storage::BlendEquationiARB);
         storage::BlendEquationiARB.aliased(&storage::BlendEquationIndexedAMD);
         storage::BlendEquationiARB.aliased(&storage::BlendEquationi);
         storage::BlendEquationIndexedAMD.aliased(&storage::BlendEquationi);
         storage::BlendEquationIndexedAMD.aliased(&storage::BlendEquationiARB);
         storage::BlendEquationSeparate.aliased(&storage::BlendEquationSeparateEXT);
         storage::BlendEquationSeparateEXT.aliased(&storage::BlendEquationSeparate);
         storage::BlendEquationSeparatei.aliased(&storage::BlendEquationSeparateIndexedAMD);
         storage::BlendEquationSeparatei.aliased(&storage::BlendEquationSeparateiARB);
         storage::BlendEquationSeparateiARB.aliased(&storage::BlendEquationSeparateIndexedAMD);
         storage::BlendEquationSeparateiARB.aliased(&storage::BlendEquationSeparatei);
         storage::BlendEquationSeparateIndexedAMD.aliased(&storage::BlendEquationSeparatei);
         storage::BlendEquationSeparateIndexedAMD.aliased(&storage::BlendEquationSeparateiARB);
         storage::BlendFunci.aliased(&storage::BlendFuncIndexedAMD);
         storage::BlendFunci.aliased(&storage::BlendFunciARB);
         storage::BlendFunciARB.aliased(&storage::BlendFuncIndexedAMD);
         storage::BlendFunciARB.aliased(&storage::BlendFunci);
         storage::BlendFuncIndexedAMD.aliased(&storage::BlendFunci);
         storage::BlendFuncIndexedAMD.aliased(&storage::BlendFunciARB);
         storage::BlendFuncSeparate.aliased(&storage::BlendFuncSeparateEXT);
         storage::BlendFuncSeparate.aliased(&storage::BlendFuncSeparateINGR);
         storage::BlendFuncSeparateEXT.aliased(&storage::BlendFuncSeparate);
         storage::BlendFuncSeparateEXT.aliased(&storage::BlendFuncSeparateINGR);
         storage::BlendFuncSeparatei.aliased(&storage::BlendFuncSeparateIndexedAMD);
         storage::BlendFuncSeparatei.aliased(&storage::BlendFuncSeparateiARB);
         storage::BlendFuncSeparateiARB.aliased(&storage::BlendFuncSeparateIndexedAMD);
         storage::BlendFuncSeparateiARB.aliased(&storage::BlendFuncSeparatei);
         storage::BlendFuncSeparateIndexedAMD.aliased(&storage::BlendFuncSeparatei);
         storage::BlendFuncSeparateIndexedAMD.aliased(&storage::BlendFuncSeparateiARB);
         storage::BlendFuncSeparateINGR.aliased(&storage::BlendFuncSeparate);
         storage::BlendFuncSeparateINGR.aliased(&storage::BlendFuncSeparateEXT);
         storage::BlitFramebuffer.aliased(&storage::BlitFramebufferEXT);
         storage::BlitFramebufferEXT.aliased(&storage::BlitFramebuffer);
         storage::BufferData.aliased(&storage::BufferDataARB);
         storage::BufferDataARB.aliased(&storage::BufferData);
         storage::BufferSubData.aliased(&storage::BufferSubDataARB);
         storage::BufferSubDataARB.aliased(&storage::BufferSubData);
         storage::CheckFramebufferStatus.aliased(&storage::CheckFramebufferStatusEXT);
         storage::CheckFramebufferStatusEXT.aliased(&storage::CheckFramebufferStatus);
         storage::ClampColor.aliased(&storage::ClampColorARB);
         storage::ClampColorARB.aliased(&storage::ClampColor);
         storage::ClearDepthf.aliased(&storage::ClearDepthfOES);
         storage::ClearDepthfOES.aliased(&storage::ClearDepthf);
         storage::ColorMaski.aliased(&storage::ColorMaskIndexedEXT);
         storage::ColorMaskIndexedEXT.aliased(&storage::ColorMaski);
         storage::CompileShader.aliased(&storage::CompileShaderARB);
         storage::CompileShaderARB.aliased(&storage::CompileShader);
         storage::CompressedTexImage1D.aliased(&storage::CompressedTexImage1DARB);
         storage::CompressedTexImage1DARB.aliased(&storage::CompressedTexImage1D);
         storage::CompressedTexImage2D.aliased(&storage::CompressedTexImage2DARB);
         storage::CompressedTexImage2DARB.aliased(&storage::CompressedTexImage2D);
         storage::CompressedTexImage3D.aliased(&storage::CompressedTexImage3DARB);
         storage::CompressedTexImage3DARB.aliased(&storage::CompressedTexImage3D);
         storage::CompressedTexSubImage1D.aliased(&storage::CompressedTexSubImage1DARB);
         storage::CompressedTexSubImage1DARB.aliased(&storage::CompressedTexSubImage1D);
         storage::CompressedTexSubImage2D.aliased(&storage::CompressedTexSubImage2DARB);
         storage::CompressedTexSubImage2DARB.aliased(&storage::CompressedTexSubImage2D);
         storage::CompressedTexSubImage3D.aliased(&storage::CompressedTexSubImage3DARB);
         storage::CompressedTexSubImage3DARB.aliased(&storage::CompressedTexSubImage3D);
         storage::CopyTexImage1D.aliased(&storage::CopyTexImage1DEXT);
         storage::CopyTexImage1DEXT.aliased(&storage::CopyTexImage1D);
         storage::CopyTexImage2D.aliased(&storage::CopyTexImage2DEXT);
         storage::CopyTexImage2DEXT.aliased(&storage::CopyTexImage2D);
         storage::CopyTexSubImage1D.aliased(&storage::CopyTexSubImage1DEXT);
         storage::CopyTexSubImage1DEXT.aliased(&storage::CopyTexSubImage1D);
         storage::CopyTexSubImage2D.aliased(&storage::CopyTexSubImage2DEXT);
         storage::CopyTexSubImage2DEXT.aliased(&storage::CopyTexSubImage2D);
         storage::CopyTexSubImage3D.aliased(&storage::CopyTexSubImage3DEXT);
         storage::CopyTexSubImage3DEXT.aliased(&storage::CopyTexSubImage3D);
         storage::CreateProgram.aliased(&storage::CreateProgramObjectARB);
         storage::CreateProgramObjectARB.aliased(&storage::CreateProgram);
         storage::CreateShader.aliased(&storage::CreateShaderObjectARB);
         storage::CreateShaderObjectARB.aliased(&storage::CreateShader);
         storage::DebugMessageCallback.aliased(&storage::DebugMessageCallbackARB);
         storage::DebugMessageCallbackARB.aliased(&storage::DebugMessageCallback);
         storage::DebugMessageControl.aliased(&storage::DebugMessageControlARB);
         storage::DebugMessageControlARB.aliased(&storage::DebugMessageControl);
         storage::DebugMessageInsert.aliased(&storage::DebugMessageInsertARB);
         storage::DebugMessageInsertARB.aliased(&storage::DebugMessageInsert);
         storage::DeleteBuffers.aliased(&storage::DeleteBuffersARB);
         storage::DeleteBuffersARB.aliased(&storage::DeleteBuffers);
         storage::DeleteFramebuffers.aliased(&storage::DeleteFramebuffersEXT);
         storage::DeleteFramebuffersEXT.aliased(&storage::DeleteFramebuffers);
         storage::DeleteProgramsARB.aliased(&storage::DeleteProgramsNV);
         storage::DeleteProgramsNV.aliased(&storage::DeleteProgramsARB);
         storage::DeleteQueries.aliased(&storage::DeleteQueriesARB);
         storage::DeleteQueriesARB.aliased(&storage::DeleteQueries);
         storage::DeleteRenderbuffers.aliased(&storage::DeleteRenderbuffersEXT);
         storage::DeleteRenderbuffersEXT.aliased(&storage::DeleteRenderbuffers);
         storage::DeleteTransformFeedbacks.aliased(&storage::DeleteTransformFeedbacksNV);
         storage::DeleteTransformFeedbacksNV.aliased(&storage::DeleteTransformFeedbacks);
         storage::DeleteVertexArrays.aliased(&storage::DeleteVertexArraysAPPLE);
         storage::DeleteVertexArraysAPPLE.aliased(&storage::DeleteVertexArrays);
         storage::DepthRangef.aliased(&storage::DepthRangefOES);
         storage::DepthRangefOES.aliased(&storage::DepthRangef);
         storage::DetachObjectARB.aliased(&storage::DetachShader);
         storage::DetachShader.aliased(&storage::DetachObjectARB);
         storage::Disablei.aliased(&storage::DisableIndexedEXT);
         storage::DisableIndexedEXT.aliased(&storage::Disablei);
         storage::DisableVertexAttribArray.aliased(&storage::DisableVertexAttribArrayARB);
         storage::DisableVertexAttribArrayARB.aliased(&storage::DisableVertexAttribArray);
         storage::DrawArrays.aliased(&storage::DrawArraysEXT);
         storage::DrawArraysEXT.aliased(&storage::DrawArrays);
         storage::DrawArraysInstanced.aliased(&storage::DrawArraysInstancedARB);
         storage::DrawArraysInstanced.aliased(&storage::DrawArraysInstancedEXT);
         storage::DrawArraysInstancedARB.aliased(&storage::DrawArraysInstanced);
         storage::DrawArraysInstancedARB.aliased(&storage::DrawArraysInstancedEXT);
         storage::DrawArraysInstancedEXT.aliased(&storage::DrawArraysInstanced);
         storage::DrawArraysInstancedEXT.aliased(&storage::DrawArraysInstancedARB);
         storage::DrawBuffers.aliased(&storage::DrawBuffersARB);
         storage::DrawBuffers.aliased(&storage::DrawBuffersATI);
         storage::DrawBuffersARB.aliased(&storage::DrawBuffers);
         storage::DrawBuffersARB.aliased(&storage::DrawBuffersATI);
         storage::DrawBuffersATI.aliased(&storage::DrawBuffers);
         storage::DrawBuffersATI.aliased(&storage::DrawBuffersARB);
         storage::DrawElementsInstanced.aliased(&storage::DrawElementsInstancedARB);
         storage::DrawElementsInstanced.aliased(&storage::DrawElementsInstancedEXT);
         storage::DrawElementsInstancedARB.aliased(&storage::DrawElementsInstanced);
         storage::DrawElementsInstancedARB.aliased(&storage::DrawElementsInstancedEXT);
         storage::DrawElementsInstancedEXT.aliased(&storage::DrawElementsInstanced);
         storage::DrawElementsInstancedEXT.aliased(&storage::DrawElementsInstancedARB);
         storage::DrawRangeElements.aliased(&storage::DrawRangeElementsEXT);
         storage::DrawRangeElementsEXT.aliased(&storage::DrawRangeElements);
         storage::DrawTransformFeedback.aliased(&storage::DrawTransformFeedbackNV);
         storage::DrawTransformFeedbackNV.aliased(&storage::DrawTransformFeedback);
         storage::Enablei.aliased(&storage::EnableIndexedEXT);
         storage::EnableIndexedEXT.aliased(&storage::Enablei);
         storage::EnableVertexAttribArray.aliased(&storage::EnableVertexAttribArrayARB);
         storage::EnableVertexAttribArrayARB.aliased(&storage::EnableVertexAttribArray);
         storage::EndConditionalRender.aliased(&storage::EndConditionalRenderNV);
         storage::EndConditionalRender.aliased(&storage::EndConditionalRenderNVX);
         storage::EndConditionalRenderNV.aliased(&storage::EndConditionalRender);
         storage::EndConditionalRenderNV.aliased(&storage::EndConditionalRenderNVX);
         storage::EndConditionalRenderNVX.aliased(&storage::EndConditionalRender);
         storage::EndConditionalRenderNVX.aliased(&storage::EndConditionalRenderNV);
         storage::EndQuery.aliased(&storage::EndQueryARB);
         storage::EndQueryARB.aliased(&storage::EndQuery);
         storage::EndTransformFeedback.aliased(&storage::EndTransformFeedbackEXT);
         storage::EndTransformFeedback.aliased(&storage::EndTransformFeedbackNV);
         storage::EndTransformFeedbackEXT.aliased(&storage::EndTransformFeedback);
         storage::EndTransformFeedbackEXT.aliased(&storage::EndTransformFeedbackNV);
         storage::EndTransformFeedbackNV.aliased(&storage::EndTransformFeedback);
         storage::EndTransformFeedbackNV.aliased(&storage::EndTransformFeedbackEXT);
         storage::FlushMappedBufferRange.aliased(&storage::FlushMappedBufferRangeAPPLE);
         storage::FlushMappedBufferRangeAPPLE.aliased(&storage::FlushMappedBufferRange);
         storage::FramebufferRenderbuffer.aliased(&storage::FramebufferRenderbufferEXT);
         storage::FramebufferRenderbufferEXT.aliased(&storage::FramebufferRenderbuffer);
         storage::FramebufferTexture.aliased(&storage::FramebufferTextureARB);
         storage::FramebufferTexture.aliased(&storage::FramebufferTextureEXT);
         storage::FramebufferTexture1D.aliased(&storage::FramebufferTexture1DEXT);
         storage::FramebufferTexture1DEXT.aliased(&storage::FramebufferTexture1D);
         storage::FramebufferTexture2D.aliased(&storage::FramebufferTexture2DEXT);
         storage::FramebufferTexture2DEXT.aliased(&storage::FramebufferTexture2D);
         storage::FramebufferTexture3D.aliased(&storage::FramebufferTexture3DEXT);
         storage::FramebufferTexture3DEXT.aliased(&storage::FramebufferTexture3D);
         storage::FramebufferTextureARB.aliased(&storage::FramebufferTexture);
         storage::FramebufferTextureARB.aliased(&storage::FramebufferTextureEXT);
         storage::FramebufferTextureEXT.aliased(&storage::FramebufferTexture);
         storage::FramebufferTextureEXT.aliased(&storage::FramebufferTextureARB);
         storage::FramebufferTextureFaceARB.aliased(&storage::FramebufferTextureFaceEXT);
         storage::FramebufferTextureFaceEXT.aliased(&storage::FramebufferTextureFaceARB);
         storage::FramebufferTextureLayer.aliased(&storage::FramebufferTextureLayerARB);
         storage::FramebufferTextureLayer.aliased(&storage::FramebufferTextureLayerEXT);
         storage::FramebufferTextureLayerARB.aliased(&storage::FramebufferTextureLayer);
         storage::FramebufferTextureLayerARB.aliased(&storage::FramebufferTextureLayerEXT);
         storage::FramebufferTextureLayerEXT.aliased(&storage::FramebufferTextureLayer);
         storage::FramebufferTextureLayerEXT.aliased(&storage::FramebufferTextureLayerARB);
         storage::GenBuffers.aliased(&storage::GenBuffersARB);
         storage::GenBuffersARB.aliased(&storage::GenBuffers);
         storage::GenerateMipmap.aliased(&storage::GenerateMipmapEXT);
         storage::GenerateMipmapEXT.aliased(&storage::GenerateMipmap);
         storage::GenFramebuffers.aliased(&storage::GenFramebuffersEXT);
         storage::GenFramebuffersEXT.aliased(&storage::GenFramebuffers);
         storage::GenProgramsARB.aliased(&storage::GenProgramsNV);
         storage::GenProgramsNV.aliased(&storage::GenProgramsARB);
         storage::GenQueries.aliased(&storage::GenQueriesARB);
         storage::GenQueriesARB.aliased(&storage::GenQueries);
         storage::GenRenderbuffers.aliased(&storage::GenRenderbuffersEXT);
         storage::GenRenderbuffersEXT.aliased(&storage::GenRenderbuffers);
         storage::GenTransformFeedbacks.aliased(&storage::GenTransformFeedbacksNV);
         storage::GenTransformFeedbacksNV.aliased(&storage::GenTransformFeedbacks);
         storage::GenVertexArrays.aliased(&storage::GenVertexArraysAPPLE);
         storage::GenVertexArraysAPPLE.aliased(&storage::GenVertexArrays);
         storage::GetActiveAttrib.aliased(&storage::GetActiveAttribARB);
         storage::GetActiveAttribARB.aliased(&storage::GetActiveAttrib);
         storage::GetActiveUniform.aliased(&storage::GetActiveUniformARB);
         storage::GetActiveUniformARB.aliased(&storage::GetActiveUniform);
         storage::GetAttribLocation.aliased(&storage::GetAttribLocationARB);
         storage::GetAttribLocationARB.aliased(&storage::GetAttribLocation);
         storage::GetBooleani_v.aliased(&storage::GetBooleanIndexedvEXT);
         storage::GetBooleanIndexedvEXT.aliased(&storage::GetBooleani_v);
         storage::GetBufferParameteriv.aliased(&storage::GetBufferParameterivARB);
         storage::GetBufferParameterivARB.aliased(&storage::GetBufferParameteriv);
         storage::GetBufferPointerv.aliased(&storage::GetBufferPointervARB);
         storage::GetBufferPointervARB.aliased(&storage::GetBufferPointerv);
         storage::GetBufferSubData.aliased(&storage::GetBufferSubDataARB);
         storage::GetBufferSubDataARB.aliased(&storage::GetBufferSubData);
         storage::GetCompressedTexImage.aliased(&storage::GetCompressedTexImageARB);
         storage::GetCompressedTexImageARB.aliased(&storage::GetCompressedTexImage);
         storage::GetDebugMessageLog.aliased(&storage::GetDebugMessageLogARB);
         storage::GetDebugMessageLogARB.aliased(&storage::GetDebugMessageLog);
         storage::GetDoublei_v.aliased(&storage::GetDoubleIndexedvEXT);
         storage::GetDoublei_v.aliased(&storage::GetDoublei_vEXT);
         storage::GetDoublei_vEXT.aliased(&storage::GetDoubleIndexedvEXT);
         storage::GetDoublei_vEXT.aliased(&storage::GetDoublei_v);
         storage::GetDoubleIndexedvEXT.aliased(&storage::GetDoublei_v);
         storage::GetDoubleIndexedvEXT.aliased(&storage::GetDoublei_vEXT);
         storage::GetFloati_v.aliased(&storage::GetFloatIndexedvEXT);
         storage::GetFloati_v.aliased(&storage::GetFloati_vEXT);
         storage::GetFloati_vEXT.aliased(&storage::GetFloatIndexedvEXT);
         storage::GetFloati_vEXT.aliased(&storage::GetFloati_v);
         storage::GetFloatIndexedvEXT.aliased(&storage::GetFloati_v);
         storage::GetFloatIndexedvEXT.aliased(&storage::GetFloati_vEXT);
         storage::GetFragDataLocation.aliased(&storage::GetFragDataLocationEXT);
         storage::GetFragDataLocationEXT.aliased(&storage::GetFragDataLocation);
         storage::GetFramebufferAttachmentParameteriv.aliased(&storage::GetFramebufferAttachmentParameterivEXT);
         storage::GetFramebufferAttachmentParameterivEXT.aliased(&storage::GetFramebufferAttachmentParameteriv);
         storage::GetIntegeri_v.aliased(&storage::GetIntegerIndexedvEXT);
         storage::GetIntegerIndexedvEXT.aliased(&storage::GetIntegeri_v);
         storage::GetMultisamplefv.aliased(&storage::GetMultisamplefvNV);
         storage::GetMultisamplefvNV.aliased(&storage::GetMultisamplefv);
         storage::GetPointerv.aliased(&storage::GetPointervEXT);
         storage::GetPointervEXT.aliased(&storage::GetPointerv);
         storage::GetQueryiv.aliased(&storage::GetQueryivARB);
         storage::GetQueryivARB.aliased(&storage::GetQueryiv);
         storage::GetQueryObjecti64v.aliased(&storage::GetQueryObjecti64vEXT);
         storage::GetQueryObjecti64vEXT.aliased(&storage::GetQueryObjecti64v);
         storage::GetQueryObjectiv.aliased(&storage::GetQueryObjectivARB);
         storage::GetQueryObjectivARB.aliased(&storage::GetQueryObjectiv);
         storage::GetQueryObjectui64v.aliased(&storage::GetQueryObjectui64vEXT);
         storage::GetQueryObjectui64vEXT.aliased(&storage::GetQueryObjectui64v);
         storage::GetQueryObjectuiv.aliased(&storage::GetQueryObjectuivARB);
         storage::GetQueryObjectuivARB.aliased(&storage::GetQueryObjectuiv);
         storage::GetRenderbufferParameteriv.aliased(&storage::GetRenderbufferParameterivEXT);
         storage::GetRenderbufferParameterivEXT.aliased(&storage::GetRenderbufferParameteriv);
         storage::GetShaderSource.aliased(&storage::GetShaderSourceARB);
         storage::GetShaderSourceARB.aliased(&storage::GetShaderSource);
         storage::GetTexParameterIiv.aliased(&storage::GetTexParameterIivEXT);
         storage::GetTexParameterIivEXT.aliased(&storage::GetTexParameterIiv);
         storage::GetTexParameterIuiv.aliased(&storage::GetTexParameterIuivEXT);
         storage::GetTexParameterIuivEXT.aliased(&storage::GetTexParameterIuiv);
         storage::GetTransformFeedbackVarying.aliased(&storage::GetTransformFeedbackVaryingEXT);
         storage::GetTransformFeedbackVaryingEXT.aliased(&storage::GetTransformFeedbackVarying);
         storage::GetUniformfv.aliased(&storage::GetUniformfvARB);
         storage::GetUniformfvARB.aliased(&storage::GetUniformfv);
         storage::GetUniformiv.aliased(&storage::GetUniformivARB);
         storage::GetUniformivARB.aliased(&storage::GetUniformiv);
         storage::GetUniformLocation.aliased(&storage::GetUniformLocationARB);
         storage::GetUniformLocationARB.aliased(&storage::GetUniformLocation);
         storage::GetUniformuiv.aliased(&storage::GetUniformuivEXT);
         storage::GetUniformuivEXT.aliased(&storage::GetUniformuiv);
         storage::GetVertexAttribdv.aliased(&storage::GetVertexAttribdvARB);
         storage::GetVertexAttribdv.aliased(&storage::GetVertexAttribdvNV);
         storage::GetVertexAttribdvARB.aliased(&storage::GetVertexAttribdv);
         storage::GetVertexAttribdvARB.aliased(&storage::GetVertexAttribdvNV);
         storage::GetVertexAttribdvNV.aliased(&storage::GetVertexAttribdv);
         storage::GetVertexAttribdvNV.aliased(&storage::GetVertexAttribdvARB);
         storage::GetVertexAttribfv.aliased(&storage::GetVertexAttribfvARB);
         storage::GetVertexAttribfv.aliased(&storage::GetVertexAttribfvNV);
         storage::GetVertexAttribfvARB.aliased(&storage::GetVertexAttribfv);
         storage::GetVertexAttribfvARB.aliased(&storage::GetVertexAttribfvNV);
         storage::GetVertexAttribfvNV.aliased(&storage::GetVertexAttribfv);
         storage::GetVertexAttribfvNV.aliased(&storage::GetVertexAttribfvARB);
         storage::GetVertexAttribIiv.aliased(&storage::GetVertexAttribIivEXT);
         storage::GetVertexAttribIivEXT.aliased(&storage::GetVertexAttribIiv);
         storage::GetVertexAttribIuiv.aliased(&storage::GetVertexAttribIuivEXT);
         storage::GetVertexAttribIuivEXT.aliased(&storage::GetVertexAttribIuiv);
         storage::GetVertexAttribiv.aliased(&storage::GetVertexAttribivARB);
         storage::GetVertexAttribiv.aliased(&storage::GetVertexAttribivNV);
         storage::GetVertexAttribivARB.aliased(&storage::GetVertexAttribiv);
         storage::GetVertexAttribivARB.aliased(&storage::GetVertexAttribivNV);
         storage::GetVertexAttribivNV.aliased(&storage::GetVertexAttribiv);
         storage::GetVertexAttribivNV.aliased(&storage::GetVertexAttribivARB);
         storage::GetVertexAttribLdv.aliased(&storage::GetVertexAttribLdvEXT);
         storage::GetVertexAttribLdvEXT.aliased(&storage::GetVertexAttribLdv);
         storage::GetVertexAttribPointerv.aliased(&storage::GetVertexAttribPointervARB);
         storage::GetVertexAttribPointerv.aliased(&storage::GetVertexAttribPointervNV);
         storage::GetVertexAttribPointervARB.aliased(&storage::GetVertexAttribPointerv);
         storage::GetVertexAttribPointervARB.aliased(&storage::GetVertexAttribPointervNV);
         storage::GetVertexAttribPointervNV.aliased(&storage::GetVertexAttribPointerv);
         storage::GetVertexAttribPointervNV.aliased(&storage::GetVertexAttribPointervARB);
         storage::IsBuffer.aliased(&storage::IsBufferARB);
         storage::IsBufferARB.aliased(&storage::IsBuffer);
         storage::IsEnabledi.aliased(&storage::IsEnabledIndexedEXT);
         storage::IsEnabledIndexedEXT.aliased(&storage::IsEnabledi);
         storage::IsFramebuffer.aliased(&storage::IsFramebufferEXT);
         storage::IsFramebufferEXT.aliased(&storage::IsFramebuffer);
         storage::IsProgramARB.aliased(&storage::IsProgramNV);
         storage::IsProgramNV.aliased(&storage::IsProgramARB);
         storage::IsQuery.aliased(&storage::IsQueryARB);
         storage::IsQueryARB.aliased(&storage::IsQuery);
         storage::IsRenderbuffer.aliased(&storage::IsRenderbufferEXT);
         storage::IsRenderbufferEXT.aliased(&storage::IsRenderbuffer);
         storage::IsTransformFeedback.aliased(&storage::IsTransformFeedbackNV);
         storage::IsTransformFeedbackNV.aliased(&storage::IsTransformFeedback);
         storage::IsVertexArray.aliased(&storage::IsVertexArrayAPPLE);
         storage::IsVertexArrayAPPLE.aliased(&storage::IsVertexArray);
         storage::LinkProgram.aliased(&storage::LinkProgramARB);
         storage::LinkProgramARB.aliased(&storage::LinkProgram);
         storage::MapBuffer.aliased(&storage::MapBufferARB);
         storage::MapBufferARB.aliased(&storage::MapBuffer);
         storage::MaxShaderCompilerThreadsARB.aliased(&storage::MaxShaderCompilerThreadsKHR);
         storage::MaxShaderCompilerThreadsKHR.aliased(&storage::MaxShaderCompilerThreadsARB);
         storage::MemoryBarrier.aliased(&storage::MemoryBarrierEXT);
         storage::MemoryBarrierEXT.aliased(&storage::MemoryBarrier);
         storage::MinSampleShading.aliased(&storage::MinSampleShadingARB);
         storage::MinSampleShadingARB.aliased(&storage::MinSampleShading);
         storage::MultiDrawArrays.aliased(&storage::MultiDrawArraysEXT);
         storage::MultiDrawArraysEXT.aliased(&storage::MultiDrawArrays);
         storage::MultiDrawArraysIndirect.aliased(&storage::MultiDrawArraysIndirectAMD);
         storage::MultiDrawArraysIndirectAMD.aliased(&storage::MultiDrawArraysIndirect);
         storage::MultiDrawArraysIndirectCount.aliased(&storage::MultiDrawArraysIndirectCountARB);
         storage::MultiDrawArraysIndirectCountARB.aliased(&storage::MultiDrawArraysIndirectCount);
         storage::MultiDrawElements.aliased(&storage::MultiDrawElementsEXT);
         storage::MultiDrawElementsEXT.aliased(&storage::MultiDrawElements);
         storage::MultiDrawElementsIndirect.aliased(&storage::MultiDrawElementsIndirectAMD);
         storage::MultiDrawElementsIndirectAMD.aliased(&storage::MultiDrawElementsIndirect);
         storage::MultiDrawElementsIndirectCount.aliased(&storage::MultiDrawElementsIndirectCountARB);
         storage::MultiDrawElementsIndirectCountARB.aliased(&storage::MultiDrawElementsIndirectCount);
         storage::NamedBufferStorage.aliased(&storage::NamedBufferStorageEXT);
         storage::NamedBufferStorageEXT.aliased(&storage::NamedBufferStorage);
         storage::NamedBufferSubData.aliased(&storage::NamedBufferSubDataEXT);
         storage::NamedBufferSubDataEXT.aliased(&storage::NamedBufferSubData);
         storage::PauseTransformFeedback.aliased(&storage::PauseTransformFeedbackNV);
         storage::PauseTransformFeedbackNV.aliased(&storage::PauseTransformFeedback);
         storage::PointParameterf.aliased(&storage::PointParameterfARB);
         storage::PointParameterf.aliased(&storage::PointParameterfEXT);
         storage::PointParameterf.aliased(&storage::PointParameterfSGIS);
         storage::PointParameterfARB.aliased(&storage::PointParameterf);
         storage::PointParameterfARB.aliased(&storage::PointParameterfEXT);
         storage::PointParameterfARB.aliased(&storage::PointParameterfSGIS);
         storage::PointParameterfEXT.aliased(&storage::PointParameterf);
         storage::PointParameterfEXT.aliased(&storage::PointParameterfARB);
         storage::PointParameterfEXT.aliased(&storage::PointParameterfSGIS);
         storage::PointParameterfSGIS.aliased(&storage::PointParameterf);
         storage::PointParameterfSGIS.aliased(&storage::PointParameterfARB);
         storage::PointParameterfSGIS.aliased(&storage::PointParameterfEXT);
         storage::PointParameterfv.aliased(&storage::PointParameterfvARB);
         storage::PointParameterfv.aliased(&storage::PointParameterfvEXT);
         storage::PointParameterfv.aliased(&storage::PointParameterfvSGIS);
         storage::PointParameterfvARB.aliased(&storage::PointParameterfv);
         storage::PointParameterfvARB.aliased(&storage::PointParameterfvEXT);
         storage::PointParameterfvARB.aliased(&storage::PointParameterfvSGIS);
         storage::PointParameterfvEXT.aliased(&storage::PointParameterfv);
         storage::PointParameterfvEXT.aliased(&storage::PointParameterfvARB);
         storage::PointParameterfvEXT.aliased(&storage::PointParameterfvSGIS);
         storage::PointParameterfvSGIS.aliased(&storage::PointParameterfv);
         storage::PointParameterfvSGIS.aliased(&storage::PointParameterfvARB);
         storage::PointParameterfvSGIS.aliased(&storage::PointParameterfvEXT);
         storage::PointParameteri.aliased(&storage::PointParameteriNV);
         storage::PointParameteriNV.aliased(&storage::PointParameteri);
         storage::PointParameteriv.aliased(&storage::PointParameterivNV);
         storage::PointParameterivNV.aliased(&storage::PointParameteriv);
         storage::PolygonOffsetClamp.aliased(&storage::PolygonOffsetClampEXT);
         storage::PolygonOffsetClampEXT.aliased(&storage::PolygonOffsetClamp);
         storage::ProgramParameteri.aliased(&storage::ProgramParameteriARB);
         storage::ProgramParameteri.aliased(&storage::ProgramParameteriEXT);
         storage::ProgramParameteriARB.aliased(&storage::ProgramParameteri);
         storage::ProgramParameteriARB.aliased(&storage::ProgramParameteriEXT);
         storage::ProgramParameteriEXT.aliased(&storage::ProgramParameteri);
         storage::ProgramParameteriEXT.aliased(&storage::ProgramParameteriARB);
         storage::ProgramUniform1f.aliased(&storage::ProgramUniform1fEXT);
         storage::ProgramUniform1fEXT.aliased(&storage::ProgramUniform1f);
         storage::ProgramUniform1fv.aliased(&storage::ProgramUniform1fvEXT);
         storage::ProgramUniform1fvEXT.aliased(&storage::ProgramUniform1fv);
         storage::ProgramUniform1i.aliased(&storage::ProgramUniform1iEXT);
         storage::ProgramUniform1iEXT.aliased(&storage::ProgramUniform1i);
         storage::ProgramUniform1iv.aliased(&storage::ProgramUniform1ivEXT);
         storage::ProgramUniform1ivEXT.aliased(&storage::ProgramUniform1iv);
         storage::ProgramUniform1ui.aliased(&storage::ProgramUniform1uiEXT);
         storage::ProgramUniform1uiEXT.aliased(&storage::ProgramUniform1ui);
         storage::ProgramUniform1uiv.aliased(&storage::ProgramUniform1uivEXT);
         storage::ProgramUniform1uivEXT.aliased(&storage::ProgramUniform1uiv);
         storage::ProgramUniform2f.aliased(&storage::ProgramUniform2fEXT);
         storage::ProgramUniform2fEXT.aliased(&storage::ProgramUniform2f);
         storage::ProgramUniform2fv.aliased(&storage::ProgramUniform2fvEXT);
         storage::ProgramUniform2fvEXT.aliased(&storage::ProgramUniform2fv);
         storage::ProgramUniform2i.aliased(&storage::ProgramUniform2iEXT);
         storage::ProgramUniform2iEXT.aliased(&storage::ProgramUniform2i);
         storage::ProgramUniform2iv.aliased(&storage::ProgramUniform2ivEXT);
         storage::ProgramUniform2ivEXT.aliased(&storage::ProgramUniform2iv);
         storage::ProgramUniform2ui.aliased(&storage::ProgramUniform2uiEXT);
         storage::ProgramUniform2uiEXT.aliased(&storage::ProgramUniform2ui);
         storage::ProgramUniform2uiv.aliased(&storage::ProgramUniform2uivEXT);
         storage::ProgramUniform2uivEXT.aliased(&storage::ProgramUniform2uiv);
         storage::ProgramUniform3f.aliased(&storage::ProgramUniform3fEXT);
         storage::ProgramUniform3fEXT.aliased(&storage::ProgramUniform3f);
         storage::ProgramUniform3fv.aliased(&storage::ProgramUniform3fvEXT);
         storage::ProgramUniform3fvEXT.aliased(&storage::ProgramUniform3fv);
         storage::ProgramUniform3i.aliased(&storage::ProgramUniform3iEXT);
         storage::ProgramUniform3iEXT.aliased(&storage::ProgramUniform3i);
         storage::ProgramUniform3iv.aliased(&storage::ProgramUniform3ivEXT);
         storage::ProgramUniform3ivEXT.aliased(&storage::ProgramUniform3iv);
         storage::ProgramUniform3ui.aliased(&storage::ProgramUniform3uiEXT);
         storage::ProgramUniform3uiEXT.aliased(&storage::ProgramUniform3ui);
         storage::ProgramUniform3uiv.aliased(&storage::ProgramUniform3uivEXT);
         storage::ProgramUniform3uivEXT.aliased(&storage::ProgramUniform3uiv);
         storage::ProgramUniform4f.aliased(&storage::ProgramUniform4fEXT);
         storage::ProgramUniform4fEXT.aliased(&storage::ProgramUniform4f);
         storage::ProgramUniform4fv.aliased(&storage::ProgramUniform4fvEXT);
         storage::ProgramUniform4fvEXT.aliased(&storage::ProgramUniform4fv);
         storage::ProgramUniform4i.aliased(&storage::ProgramUniform4iEXT);
         storage::ProgramUniform4iEXT.aliased(&storage::ProgramUniform4i);
         storage::ProgramUniform4iv.aliased(&storage::ProgramUniform4ivEXT);
         storage::ProgramUniform4ivEXT.aliased(&storage::ProgramUniform4iv);
         storage::ProgramUniform4ui.aliased(&storage::ProgramUniform4uiEXT);
         storage::ProgramUniform4uiEXT.aliased(&storage::ProgramUniform4ui);
         storage::ProgramUniform4uiv.aliased(&storage::ProgramUniform4uivEXT);
         storage::ProgramUniform4uivEXT.aliased(&storage::ProgramUniform4uiv);
         storage::ProgramUniformMatrix2fv.aliased(&storage::ProgramUniformMatrix2fvEXT);
         storage::ProgramUniformMatrix2fvEXT.aliased(&storage::ProgramUniformMatrix2fv);
         storage::ProgramUniformMatrix2x3fv.aliased(&storage::ProgramUniformMatrix2x3fvEXT);
         storage::ProgramUniformMatrix2x3fvEXT.aliased(&storage::ProgramUniformMatrix2x3fv);
         storage::ProgramUniformMatrix2x4fv.aliased(&storage::ProgramUniformMatrix2x4fvEXT);
         storage::ProgramUniformMatrix2x4fvEXT.aliased(&storage::ProgramUniformMatrix2x4fv);
         storage::ProgramUniformMatrix3fv.aliased(&storage::ProgramUniformMatrix3fvEXT);
         storage::ProgramUniformMatrix3fvEXT.aliased(&storage::ProgramUniformMatrix3fv);
         storage::ProgramUniformMatrix3x2fv.aliased(&storage::ProgramUniformMatrix3x2fvEXT);
         storage::ProgramUniformMatrix3x2fvEXT.aliased(&storage::ProgramUniformMatrix3x2fv);
         storage::ProgramUniformMatrix3x4fv.aliased(&storage::ProgramUniformMatrix3x4fvEXT);
         storage::ProgramUniformMatrix3x4fvEXT.aliased(&storage::ProgramUniformMatrix3x4fv);
         storage::ProgramUniformMatrix4fv.aliased(&storage::ProgramUniformMatrix4fvEXT);
         storage::ProgramUniformMatrix4fvEXT.aliased(&storage::ProgramUniformMatrix4fv);
         storage::ProgramUniformMatrix4x2fv.aliased(&storage::ProgramUniformMatrix4x2fvEXT);
         storage::ProgramUniformMatrix4x2fvEXT.aliased(&storage::ProgramUniformMatrix4x2fv);
         storage::ProgramUniformMatrix4x3fv.aliased(&storage::ProgramUniformMatrix4x3fvEXT);
         storage::ProgramUniformMatrix4x3fvEXT.aliased(&storage::ProgramUniformMatrix4x3fv);
         storage::ProvokingVertex.aliased(&storage::ProvokingVertexEXT);
         storage::ProvokingVertexEXT.aliased(&storage::ProvokingVertex);
         storage::ReadnPixels.aliased(&storage::ReadnPixelsARB);
         storage::ReadnPixelsARB.aliased(&storage::ReadnPixels);
         storage::RenderbufferStorage.aliased(&storage::RenderbufferStorageEXT);
         storage::RenderbufferStorageEXT.aliased(&storage::RenderbufferStorage);
         storage::RenderbufferStorageMultisample.aliased(&storage::RenderbufferStorageMultisampleEXT);
         storage::RenderbufferStorageMultisampleEXT.aliased(&storage::RenderbufferStorageMultisample);
         storage::ResumeTransformFeedback.aliased(&storage::ResumeTransformFeedbackNV);
         storage::ResumeTransformFeedbackNV.aliased(&storage::ResumeTransformFeedback);
         storage::SampleCoverage.aliased(&storage::SampleCoverageARB);
         storage::SampleCoverageARB.aliased(&storage::SampleCoverage);
         storage::SampleMaskEXT.aliased(&storage::SampleMaskSGIS);
         storage::SampleMaskSGIS.aliased(&storage::SampleMaskEXT);
         storage::SamplePatternEXT.aliased(&storage::SamplePatternSGIS);
         storage::SamplePatternSGIS.aliased(&storage::SamplePatternEXT);
         storage::ShaderSource.aliased(&storage::ShaderSourceARB);
         storage::ShaderSourceARB.aliased(&storage::ShaderSource);
         storage::SpecializeShader.aliased(&storage::SpecializeShaderARB);
         storage::SpecializeShaderARB.aliased(&storage::SpecializeShader);
         storage::StencilOpSeparate.aliased(&storage::StencilOpSeparateATI);
         storage::StencilOpSeparateATI.aliased(&storage::StencilOpSeparate);
         storage::TexBuffer.aliased(&storage::TexBufferARB);
         storage::TexBuffer.aliased(&storage::TexBufferEXT);
         storage::TexBufferARB.aliased(&storage::TexBuffer);
         storage::TexBufferARB.aliased(&storage::TexBufferEXT);
         storage::TexBufferEXT.aliased(&storage::TexBuffer);
         storage::TexBufferEXT.aliased(&storage::TexBufferARB);
         storage::TexImage3D.aliased(&storage::TexImage3DEXT);
         storage::TexImage3DEXT.aliased(&storage::TexImage3D);
         storage::TexParameterIiv.aliased(&storage::TexParameterIivEXT);
         storage::TexParameterIivEXT.aliased(&storage::TexParameterIiv);
         storage::TexParameterIuiv.aliased(&storage::TexParameterIuivEXT);
         storage::TexParameterIuivEXT.aliased(&storage::TexParameterIuiv);
         storage::TexStorage1D.aliased(&storage::TexStorage1DEXT);
         storage::TexStorage1DEXT.aliased(&storage::TexStorage1D);
         storage::TexStorage2D.aliased(&storage::TexStorage2DEXT);
         storage::TexStorage2DEXT.aliased(&storage::TexStorage2D);
         storage::TexStorage3D.aliased(&storage::TexStorage3DEXT);
         storage::TexStorage3DEXT.aliased(&storage::TexStorage3D);
         storage::TexSubImage1D.aliased(&storage::TexSubImage1DEXT);
         storage::TexSubImage1DEXT.aliased(&storage::TexSubImage1D);
         storage::TexSubImage2D.aliased(&storage::TexSubImage2DEXT);
         storage::TexSubImage2DEXT.aliased(&storage::TexSubImage2D);
         storage::TexSubImage3D.aliased(&storage::TexSubImage3DEXT);
         storage::TexSubImage3DEXT.aliased(&storage::TexSubImage3D);
         storage::TransformFeedbackVaryings.aliased(&storage::TransformFeedbackVaryingsEXT);
         storage::TransformFeedbackVaryingsEXT.aliased(&storage::TransformFeedbackVaryings);
         storage::Uniform1f.aliased(&storage::Uniform1fARB);
         storage::Uniform1fARB.aliased(&storage::Uniform1f);
         storage::Uniform1fv.aliased(&storage::Uniform1fvARB);
         storage::Uniform1fvARB.aliased(&storage::Uniform1fv);
         storage::Uniform1i.aliased(&storage::Uniform1iARB);
         storage::Uniform1iARB.aliased(&storage::Uniform1i);
         storage::Uniform1iv.aliased(&storage::Uniform1ivARB);
         storage::Uniform1ivARB.aliased(&storage::Uniform1iv);
         storage::Uniform1ui.aliased(&storage::Uniform1uiEXT);
         storage::Uniform1uiEXT.aliased(&storage::Uniform1ui);
         storage::Uniform1uiv.aliased(&storage::Uniform1uivEXT);
         storage::Uniform1uivEXT.aliased(&storage::Uniform1uiv);
         storage::Uniform2f.aliased(&storage::Uniform2fARB);
         storage::Uniform2fARB.aliased(&storage::Uniform2f);
         storage::Uniform2fv.aliased(&storage::Uniform2fvARB);
         storage::Uniform2fvARB.aliased(&storage::Uniform2fv);
         storage::Uniform2i.aliased(&storage::Uniform2iARB);
         storage::Uniform2iARB.aliased(&storage::Uniform2i);
         storage::Uniform2iv.aliased(&storage::Uniform2ivARB);
         storage::Uniform2ivARB.aliased(&storage::Uniform2iv);
         storage::Uniform2ui.aliased(&storage::Uniform2uiEXT);
         storage::Uniform2uiEXT.aliased(&storage::Uniform2ui);
         storage::Uniform2uiv.aliased(&storage::Uniform2uivEXT);
         storage::Uniform2uivEXT.aliased(&storage::Uniform2uiv);
         storage::Uniform3f.aliased(&storage::Uniform3fARB);
         storage::Uniform3fARB.aliased(&storage::Uniform3f);
         storage::Uniform3fv.aliased(&storage::Uniform3fvARB);
         storage::Uniform3fvARB.aliased(&storage::Uniform3fv);
         storage::Uniform3i.aliased(&storage::Uniform3iARB);
         storage::Uniform3iARB.aliased(&storage::Uniform3i);
         storage::Uniform3iv.aliased(&storage::Uniform3ivARB);
         storage::Uniform3ivARB.aliased(&storage::Uniform3iv);
         storage::Uniform3ui.aliased(&storage::Uniform3uiEXT);
         storage::Uniform3uiEXT.aliased(&storage::Uniform3ui);
         storage::Uniform3uiv.aliased(&storage::Uniform3uivEXT);
         storage::Uniform3uivEXT.aliased(&storage::Uniform3uiv);
         storage::Uniform4f.aliased(&storage::Uniform4fARB);
         storage::Uniform4fARB.aliased(&storage::Uniform4f);
         storage::Uniform4fv.aliased(&storage::Uniform4fvARB);
         storage::Uniform4fvARB.aliased(&storage::Uniform4fv);
         storage::Uniform4i.aliased(&storage::Uniform4iARB);
         storage::Uniform4iARB.aliased(&storage::Uniform4i);
         storage::Uniform4iv.aliased(&storage::Uniform4ivARB);
         storage::Uniform4ivARB.aliased(&storage::Uniform4iv);
         storage::Uniform4ui.aliased(&storage::Uniform4uiEXT);
         storage::Uniform4uiEXT.aliased(&storage::Uniform4ui);
         storage::Uniform4uiv.aliased(&storage::Uniform4uivEXT);
         storage::Uniform4uivEXT.aliased(&storage::Uniform4uiv);
         storage::UniformMatrix2fv.aliased(&storage::UniformMatrix2fvARB);
         storage::UniformMatrix2fvARB.aliased(&storage::UniformMatrix2fv);
         storage::UniformMatrix3fv.aliased(&storage::UniformMatrix3fvARB);
         storage::UniformMatrix3fvARB.aliased(&storage::UniformMatrix3fv);
         storage::UniformMatrix4fv.aliased(&storage::UniformMatrix4fvARB);
         storage::UniformMatrix4fvARB.aliased(&storage::UniformMatrix4fv);
         storage::UnmapBuffer.aliased(&storage::UnmapBufferARB);
         storage::UnmapBufferARB.aliased(&storage::UnmapBuffer);
         storage::UseProgram.aliased(&storage::UseProgramObjectARB);
         storage::UseProgramObjectARB.aliased(&storage::UseProgram);
         storage::ValidateProgram.aliased(&storage::ValidateProgramARB);
         storage::ValidateProgramARB.aliased(&storage::ValidateProgram);
         storage::VertexAttrib1d.aliased(&storage::VertexAttrib1dARB);
         storage::VertexAttrib1d.aliased(&storage::VertexAttrib1dNV);
         storage::VertexAttrib1dARB.aliased(&storage::VertexAttrib1d);
         storage::VertexAttrib1dARB.aliased(&storage::VertexAttrib1dNV);
         storage::VertexAttrib1dNV.aliased(&storage::VertexAttrib1d);
         storage::VertexAttrib1dNV.aliased(&storage::VertexAttrib1dARB);
         storage::VertexAttrib1dv.aliased(&storage::VertexAttrib1dvARB);
         storage::VertexAttrib1dv.aliased(&storage::VertexAttrib1dvNV);
         storage::VertexAttrib1dvARB.aliased(&storage::VertexAttrib1dv);
         storage::VertexAttrib1dvARB.aliased(&storage::VertexAttrib1dvNV);
         storage::VertexAttrib1dvNV.aliased(&storage::VertexAttrib1dv);
         storage::VertexAttrib1dvNV.aliased(&storage::VertexAttrib1dvARB);
         storage::VertexAttrib1f.aliased(&storage::VertexAttrib1fARB);
         storage::VertexAttrib1f.aliased(&storage::VertexAttrib1fNV);
         storage::VertexAttrib1fARB.aliased(&storage::VertexAttrib1f);
         storage::VertexAttrib1fARB.aliased(&storage::VertexAttrib1fNV);
         storage::VertexAttrib1fNV.aliased(&storage::VertexAttrib1f);
         storage::VertexAttrib1fNV.aliased(&storage::VertexAttrib1fARB);
         storage::VertexAttrib1fv.aliased(&storage::VertexAttrib1fvARB);
         storage::VertexAttrib1fv.aliased(&storage::VertexAttrib1fvNV);
         storage::VertexAttrib1fvARB.aliased(&storage::VertexAttrib1fv);
         storage::VertexAttrib1fvARB.aliased(&storage::VertexAttrib1fvNV);
         storage::VertexAttrib1fvNV.aliased(&storage::VertexAttrib1fv);
         storage::VertexAttrib1fvNV.aliased(&storage::VertexAttrib1fvARB);
         storage::VertexAttrib1s.aliased(&storage::VertexAttrib1sARB);
         storage::VertexAttrib1s.aliased(&storage::VertexAttrib1sNV);
         storage::VertexAttrib1sARB.aliased(&storage::VertexAttrib1s);
         storage::VertexAttrib1sARB.aliased(&storage::VertexAttrib1sNV);
         storage::VertexAttrib1sNV.aliased(&storage::VertexAttrib1s);
         storage::VertexAttrib1sNV.aliased(&storage::VertexAttrib1sARB);
         storage::VertexAttrib1sv.aliased(&storage::VertexAttrib1svARB);
         storage::VertexAttrib1sv.aliased(&storage::VertexAttrib1svNV);
         storage::VertexAttrib1svARB.aliased(&storage::VertexAttrib1sv);
         storage::VertexAttrib1svARB.aliased(&storage::VertexAttrib1svNV);
         storage::VertexAttrib1svNV.aliased(&storage::VertexAttrib1sv);
         storage::VertexAttrib1svNV.aliased(&storage::VertexAttrib1svARB);
         storage::VertexAttrib2d.aliased(&storage::VertexAttrib2dARB);
         storage::VertexAttrib2d.aliased(&storage::VertexAttrib2dNV);
         storage::VertexAttrib2dARB.aliased(&storage::VertexAttrib2d);
         storage::VertexAttrib2dARB.aliased(&storage::VertexAttrib2dNV);
         storage::VertexAttrib2dNV.aliased(&storage::VertexAttrib2d);
         storage::VertexAttrib2dNV.aliased(&storage::VertexAttrib2dARB);
         storage::VertexAttrib2dv.aliased(&storage::VertexAttrib2dvARB);
         storage::VertexAttrib2dv.aliased(&storage::VertexAttrib2dvNV);
         storage::VertexAttrib2dvARB.aliased(&storage::VertexAttrib2dv);
         storage::VertexAttrib2dvARB.aliased(&storage::VertexAttrib2dvNV);
         storage::VertexAttrib2dvNV.aliased(&storage::VertexAttrib2dv);
         storage::VertexAttrib2dvNV.aliased(&storage::VertexAttrib2dvARB);
         storage::VertexAttrib2f.aliased(&storage::VertexAttrib2fARB);
         storage::VertexAttrib2f.aliased(&storage::VertexAttrib2fNV);
         storage::VertexAttrib2fARB.aliased(&storage::VertexAttrib2f);
         storage::VertexAttrib2fARB.aliased(&storage::VertexAttrib2fNV);
         storage::VertexAttrib2fNV.aliased(&storage::VertexAttrib2f);
         storage::VertexAttrib2fNV.aliased(&storage::VertexAttrib2fARB);
         storage::VertexAttrib2fv.aliased(&storage::VertexAttrib2fvARB);
         storage::VertexAttrib2fv.aliased(&storage::VertexAttrib2fvNV);
         storage::VertexAttrib2fvARB.aliased(&storage::VertexAttrib2fv);
         storage::VertexAttrib2fvARB.aliased(&storage::VertexAttrib2fvNV);
         storage::VertexAttrib2fvNV.aliased(&storage::VertexAttrib2fv);
         storage::VertexAttrib2fvNV.aliased(&storage::VertexAttrib2fvARB);
         storage::VertexAttrib2s.aliased(&storage::VertexAttrib2sARB);
         storage::VertexAttrib2s.aliased(&storage::VertexAttrib2sNV);
         storage::VertexAttrib2sARB.aliased(&storage::VertexAttrib2s);
         storage::VertexAttrib2sARB.aliased(&storage::VertexAttrib2sNV);
         storage::VertexAttrib2sNV.aliased(&storage::VertexAttrib2s);
         storage::VertexAttrib2sNV.aliased(&storage::VertexAttrib2sARB);
         storage::VertexAttrib2sv.aliased(&storage::VertexAttrib2svARB);
         storage::VertexAttrib2sv.aliased(&storage::VertexAttrib2svNV);
         storage::VertexAttrib2svARB.aliased(&storage::VertexAttrib2sv);
         storage::VertexAttrib2svARB.aliased(&storage::VertexAttrib2svNV);
         storage::VertexAttrib2svNV.aliased(&storage::VertexAttrib2sv);
         storage::VertexAttrib2svNV.aliased(&storage::VertexAttrib2svARB);
         storage::VertexAttrib3d.aliased(&storage::VertexAttrib3dARB);
         storage::VertexAttrib3d.aliased(&storage::VertexAttrib3dNV);
         storage::VertexAttrib3dARB.aliased(&storage::VertexAttrib3d);
         storage::VertexAttrib3dARB.aliased(&storage::VertexAttrib3dNV);
         storage::VertexAttrib3dNV.aliased(&storage::VertexAttrib3d);
         storage::VertexAttrib3dNV.aliased(&storage::VertexAttrib3dARB);
         storage::VertexAttrib3dv.aliased(&storage::VertexAttrib3dvARB);
         storage::VertexAttrib3dv.aliased(&storage::VertexAttrib3dvNV);
         storage::VertexAttrib3dvARB.aliased(&storage::VertexAttrib3dv);
         storage::VertexAttrib3dvARB.aliased(&storage::VertexAttrib3dvNV);
         storage::VertexAttrib3dvNV.aliased(&storage::VertexAttrib3dv);
         storage::VertexAttrib3dvNV.aliased(&storage::VertexAttrib3dvARB);
         storage::VertexAttrib3f.aliased(&storage::VertexAttrib3fARB);
         storage::VertexAttrib3f.aliased(&storage::VertexAttrib3fNV);
         storage::VertexAttrib3fARB.aliased(&storage::VertexAttrib3f);
         storage::VertexAttrib3fARB.aliased(&storage::VertexAttrib3fNV);
         storage::VertexAttrib3fNV.aliased(&storage::VertexAttrib3f);
         storage::VertexAttrib3fNV.aliased(&storage::VertexAttrib3fARB);
         storage::VertexAttrib3fv.aliased(&storage::VertexAttrib3fvARB);
         storage::VertexAttrib3fv.aliased(&storage::VertexAttrib3fvNV);
         storage::VertexAttrib3fvARB.aliased(&storage::VertexAttrib3fv);
         storage::VertexAttrib3fvARB.aliased(&storage::VertexAttrib3fvNV);
         storage::VertexAttrib3fvNV.aliased(&storage::VertexAttrib3fv);
         storage::VertexAttrib3fvNV.aliased(&storage::VertexAttrib3fvARB);
         storage::VertexAttrib3s.aliased(&storage::VertexAttrib3sARB);
         storage::VertexAttrib3s.aliased(&storage::VertexAttrib3sNV);
         storage::VertexAttrib3sARB.aliased(&storage::VertexAttrib3s);
         storage::VertexAttrib3sARB.aliased(&storage::VertexAttrib3sNV);
         storage::VertexAttrib3sNV.aliased(&storage::VertexAttrib3s);
         storage::VertexAttrib3sNV.aliased(&storage::VertexAttrib3sARB);
         storage::VertexAttrib3sv.aliased(&storage::VertexAttrib3svARB);
         storage::VertexAttrib3sv.aliased(&storage::VertexAttrib3svNV);
         storage::VertexAttrib3svARB.aliased(&storage::VertexAttrib3sv);
         storage::VertexAttrib3svARB.aliased(&storage::VertexAttrib3svNV);
         storage::VertexAttrib3svNV.aliased(&storage::VertexAttrib3sv);
         storage::VertexAttrib3svNV.aliased(&storage::VertexAttrib3svARB);
         storage::VertexAttrib4bv.aliased(&storage::VertexAttrib4bvARB);
         storage::VertexAttrib4bvARB.aliased(&storage::VertexAttrib4bv);
         storage::VertexAttrib4d.aliased(&storage::VertexAttrib4dARB);
         storage::VertexAttrib4d.aliased(&storage::VertexAttrib4dNV);
         storage::VertexAttrib4dARB.aliased(&storage::VertexAttrib4d);
         storage::VertexAttrib4dARB.aliased(&storage::VertexAttrib4dNV);
         storage::VertexAttrib4dNV.aliased(&storage::VertexAttrib4d);
         storage::VertexAttrib4dNV.aliased(&storage::VertexAttrib4dARB);
         storage::VertexAttrib4dv.aliased(&storage::VertexAttrib4dvARB);
         storage::VertexAttrib4dv.aliased(&storage::VertexAttrib4dvNV);
         storage::VertexAttrib4dvARB.aliased(&storage::VertexAttrib4dv);
         storage::VertexAttrib4dvARB.aliased(&storage::VertexAttrib4dvNV);
         storage::VertexAttrib4dvNV.aliased(&storage::VertexAttrib4dv);
         storage::VertexAttrib4dvNV.aliased(&storage::VertexAttrib4dvARB);
         storage::VertexAttrib4f.aliased(&storage::VertexAttrib4fARB);
         storage::VertexAttrib4f.aliased(&storage::VertexAttrib4fNV);
         storage::VertexAttrib4fARB.aliased(&storage::VertexAttrib4f);
         storage::VertexAttrib4fARB.aliased(&storage::VertexAttrib4fNV);
         storage::VertexAttrib4fNV.aliased(&storage::VertexAttrib4f);
         storage::VertexAttrib4fNV.aliased(&storage::VertexAttrib4fARB);
         storage::VertexAttrib4fv.aliased(&storage::VertexAttrib4fvARB);
         storage::VertexAttrib4fv.aliased(&storage::VertexAttrib4fvNV);
         storage::VertexAttrib4fvARB.aliased(&storage::VertexAttrib4fv);
         storage::VertexAttrib4fvARB.aliased(&storage::VertexAttrib4fvNV);
         storage::VertexAttrib4fvNV.aliased(&storage::VertexAttrib4fv);
         storage::VertexAttrib4fvNV.aliased(&storage::VertexAttrib4fvARB);
         storage::VertexAttrib4iv.aliased(&storage::VertexAttrib4ivARB);
         storage::VertexAttrib4ivARB.aliased(&storage::VertexAttrib4iv);
         storage::VertexAttrib4Nbv.aliased(&storage::VertexAttrib4NbvARB);
         storage::VertexAttrib4NbvARB.aliased(&storage::VertexAttrib4Nbv);
         storage::VertexAttrib4Niv.aliased(&storage::VertexAttrib4NivARB);
         storage::VertexAttrib4NivARB.aliased(&storage::VertexAttrib4Niv);
         storage::VertexAttrib4Nsv.aliased(&storage::VertexAttrib4NsvARB);
         storage::VertexAttrib4NsvARB.aliased(&storage::VertexAttrib4Nsv);
         storage::VertexAttrib4Nub.aliased(&storage::VertexAttrib4NubARB);
         storage::VertexAttrib4Nub.aliased(&storage::VertexAttrib4ubNV);
         storage::VertexAttrib4NubARB.aliased(&storage::VertexAttrib4Nub);
         storage::VertexAttrib4NubARB.aliased(&storage::VertexAttrib4ubNV);
         storage::VertexAttrib4Nubv.aliased(&storage::VertexAttrib4NubvARB);
         storage::VertexAttrib4Nubv.aliased(&storage::VertexAttrib4ubvNV);
         storage::VertexAttrib4NubvARB.aliased(&storage::VertexAttrib4Nubv);
         storage::VertexAttrib4NubvARB.aliased(&storage::VertexAttrib4ubvNV);
         storage::VertexAttrib4Nuiv.aliased(&storage::VertexAttrib4NuivARB);
         storage::VertexAttrib4NuivARB.aliased(&storage::VertexAttrib4Nuiv);
         storage::VertexAttrib4Nusv.aliased(&storage::VertexAttrib4NusvARB);
         storage::VertexAttrib4NusvARB.aliased(&storage::VertexAttrib4Nusv);
         storage::VertexAttrib4s.aliased(&storage::VertexAttrib4sARB);
         storage::VertexAttrib4s.aliased(&storage::VertexAttrib4sNV);
         storage::VertexAttrib4sARB.aliased(&storage::VertexAttrib4s);
         storage::VertexAttrib4sARB.aliased(&storage::VertexAttrib4sNV);
         storage::VertexAttrib4sNV.aliased(&storage::VertexAttrib4s);
         storage::VertexAttrib4sNV.aliased(&storage::VertexAttrib4sARB);
         storage::VertexAttrib4sv.aliased(&storage::VertexAttrib4svARB);
         storage::VertexAttrib4sv.aliased(&storage::VertexAttrib4svNV);
         storage::VertexAttrib4svARB.aliased(&storage::VertexAttrib4sv);
         storage::VertexAttrib4svARB.aliased(&storage::VertexAttrib4svNV);
         storage::VertexAttrib4svNV.aliased(&storage::VertexAttrib4sv);
         storage::VertexAttrib4svNV.aliased(&storage::VertexAttrib4svARB);
         storage::VertexAttrib4ubNV.aliased(&storage::VertexAttrib4Nub);
         storage::VertexAttrib4ubNV.aliased(&storage::VertexAttrib4NubARB);
         storage::VertexAttrib4ubv.aliased(&storage::VertexAttrib4ubvARB);
         storage::VertexAttrib4ubvARB.aliased(&storage::VertexAttrib4ubv);
         storage::VertexAttrib4ubvNV.aliased(&storage::VertexAttrib4Nubv);
         storage::VertexAttrib4ubvNV.aliased(&storage::VertexAttrib4NubvARB);
         storage::VertexAttrib4uiv.aliased(&storage::VertexAttrib4uivARB);
         storage::VertexAttrib4uivARB.aliased(&storage::VertexAttrib4uiv);
         storage::VertexAttrib4usv.aliased(&storage::VertexAttrib4usvARB);
         storage::VertexAttrib4usvARB.aliased(&storage::VertexAttrib4usv);
         storage::VertexAttribDivisor.aliased(&storage::VertexAttribDivisorARB);
         storage::VertexAttribDivisorARB.aliased(&storage::VertexAttribDivisor);
         storage::VertexAttribI1i.aliased(&storage::VertexAttribI1iEXT);
         storage::VertexAttribI1iEXT.aliased(&storage::VertexAttribI1i);
         storage::VertexAttribI1iv.aliased(&storage::VertexAttribI1ivEXT);
         storage::VertexAttribI1ivEXT.aliased(&storage::VertexAttribI1iv);
         storage::VertexAttribI1ui.aliased(&storage::VertexAttribI1uiEXT);
         storage::VertexAttribI1uiEXT.aliased(&storage::VertexAttribI1ui);
         storage::VertexAttribI1uiv.aliased(&storage::VertexAttribI1uivEXT);
         storage::VertexAttribI1uivEXT.aliased(&storage::VertexAttribI1uiv);
         storage::VertexAttribI2i.aliased(&storage::VertexAttribI2iEXT);
         storage::VertexAttribI2iEXT.aliased(&storage::VertexAttribI2i);
         storage::VertexAttribI2iv.aliased(&storage::VertexAttribI2ivEXT);
         storage::VertexAttribI2ivEXT.aliased(&storage::VertexAttribI2iv);
         storage::VertexAttribI2ui.aliased(&storage::VertexAttribI2uiEXT);
         storage::VertexAttribI2uiEXT.aliased(&storage::VertexAttribI2ui);
         storage::VertexAttribI2uiv.aliased(&storage::VertexAttribI2uivEXT);
         storage::VertexAttribI2uivEXT.aliased(&storage::VertexAttribI2uiv);
         storage::VertexAttribI3i.aliased(&storage::VertexAttribI3iEXT);
         storage::VertexAttribI3iEXT.aliased(&storage::VertexAttribI3i);
         storage::VertexAttribI3iv.aliased(&storage::VertexAttribI3ivEXT);
         storage::VertexAttribI3ivEXT.aliased(&storage::VertexAttribI3iv);
         storage::VertexAttribI3ui.aliased(&storage::VertexAttribI3uiEXT);
         storage::VertexAttribI3uiEXT.aliased(&storage::VertexAttribI3ui);
         storage::VertexAttribI3uiv.aliased(&storage::VertexAttribI3uivEXT);
         storage::VertexAttribI3uivEXT.aliased(&storage::VertexAttribI3uiv);
         storage::VertexAttribI4bv.aliased(&storage::VertexAttribI4bvEXT);
         storage::VertexAttribI4bvEXT.aliased(&storage::VertexAttribI4bv);
         storage::VertexAttribI4i.aliased(&storage::VertexAttribI4iEXT);
         storage::VertexAttribI4iEXT.aliased(&storage::VertexAttribI4i);
         storage::VertexAttribI4iv.aliased(&storage::VertexAttribI4ivEXT);
         storage::VertexAttribI4ivEXT.aliased(&storage::VertexAttribI4iv);
         storage::VertexAttribI4sv.aliased(&storage::VertexAttribI4svEXT);
         storage::VertexAttribI4svEXT.aliased(&storage::VertexAttribI4sv);
         storage::VertexAttribI4ubv.aliased(&storage::VertexAttribI4ubvEXT);
         storage::VertexAttribI4ubvEXT.aliased(&storage::VertexAttribI4ubv);
         storage::VertexAttribI4ui.aliased(&storage::VertexAttribI4uiEXT);
         storage::VertexAttribI4uiEXT.aliased(&storage::VertexAttribI4ui);
         storage::VertexAttribI4uiv.aliased(&storage::VertexAttribI4uivEXT);
         storage::VertexAttribI4uivEXT.aliased(&storage::VertexAttribI4uiv);
         storage::VertexAttribI4usv.aliased(&storage::VertexAttribI4usvEXT);
         storage::VertexAttribI4usvEXT.aliased(&storage::VertexAttribI4usv);
         storage::VertexAttribIPointer.aliased(&storage::VertexAttribIPointerEXT);
         storage::VertexAttribIPointerEXT.aliased(&storage::VertexAttribIPointer);
         storage::VertexAttribL1d.aliased(&storage::VertexAttribL1dEXT);
         storage::VertexAttribL1dEXT.aliased(&storage::VertexAttribL1d);
         storage::VertexAttribL1dv.aliased(&storage::VertexAttribL1dvEXT);
         storage::VertexAttribL1dvEXT.aliased(&storage::VertexAttribL1dv);
         storage::VertexAttribL2d.aliased(&storage::VertexAttribL2dEXT);
         storage::VertexAttribL2dEXT.aliased(&storage::VertexAttribL2d);
         storage::VertexAttribL2dv.aliased(&storage::VertexAttribL2dvEXT);
         storage::VertexAttribL2dvEXT.aliased(&storage::VertexAttribL2dv);
         storage::VertexAttribL3d.aliased(&storage::VertexAttribL3dEXT);
         storage::VertexAttribL3dEXT.aliased(&storage::VertexAttribL3d);
         storage::VertexAttribL3dv.aliased(&storage::VertexAttribL3dvEXT);
         storage::VertexAttribL3dvEXT.aliased(&storage::VertexAttribL3dv);
         storage::VertexAttribL4d.aliased(&storage::VertexAttribL4dEXT);
         storage::VertexAttribL4dEXT.aliased(&storage::VertexAttribL4d);
         storage::VertexAttribL4dv.aliased(&storage::VertexAttribL4dvEXT);
         storage::VertexAttribL4dvEXT.aliased(&storage::VertexAttribL4dv);
         storage::VertexAttribLPointer.aliased(&storage::VertexAttribLPointerEXT);
         storage::VertexAttribLPointerEXT.aliased(&storage::VertexAttribLPointer);
         storage::VertexAttribPointer.aliased(&storage::VertexAttribPointerARB);
         storage::VertexAttribPointerARB.aliased(&storage::VertexAttribPointer);
    }
}
